;;;; -*- Mode: Lisp -*- 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; 
;;;; File            : spa-trace.lisp
;;;; Author          : Frank Ritter
;;;; Created On      : Fri May 22 15:18:00 1992
;;;; Last Modified By: Frank Ritter
;;;; Last Modified On: Thu Jun 18 19:03:23 1992
;;;; Update Count    : 14
;;;; 
;;;; PURPOSE
;;;; 	Provides a more compact and manipulable trace necessary for protocol
;;;; analysis at least.  You can control (1) how much to indent at each 
;;;; impasse level, (2) what to put in front of goals (e.g., ==> or =>), 
;;;; (3) whether to put out a real tab at the front of the line, (4) whether 
;;;; to print the id of the traced item, and (5) show the level of impasse by
;;;; putting up dots.  loading this file now sets up the most modifed version,
;;;; (put with id's loaded back in) because presumably you wanted something 
;;;; that looked like that, or else you wouldn't have loaded this.
;;;;
;;;; BUGS: 
;;;; There is one TAQL interaction noted below.  If you use TAQL, you 
;;;; should load this after TAQL, or put the patch noted below in 
;;;; your .taql-init.lisp file (useful with TAQL 3.1.5 and later versions).
;;;;
;;;; Plain Soar trace:
;;;;  0   G: G1 
;;;;  1   P: P2 (TOP-PS)
;;;;  2   S: S4 (TOP-STATE)
;;;;  3   O: O6 (WAIT)
;;;;  4   ==> G: G2 (operator no-change)
;;;;  5       P: P3 (some-space)
;;;;  
;;;; Trace with TAQL:
;;;;  0   G: G1 
;;;;  1   P: P2 (TOP-SPACE)
;;;;  2   S: S4 
;;;;  3   O: O7 (HALT)
;;;;  4   ==> G: G2 (operator no-change)
;;;;  5       P: P3 (some-space)
;;;;  
;;;; Most modified spa-trace:
;;;;  0   G: G1 ()
;;;;  1   P: TOP-SPACE ()
;;;;  2   S: S4 ()
;;;;  3   O: HALT ()
;;;;  4   => G: (operator no-change)
;;;;  5   .  P: SOME-SPACE ()
;;;;
;;;; TABLE OF CONTENTS
;;;;	i.	Controling variables
;;;;	I.	print-pgs-context
;;;; 
;;;; Copyright 1992, Frank Ritter.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; Status          : Unknown, Use with caution!
;;;; HISTORY
;;;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(eval-when (load eval compile)
  (soarresetsyntax)
  (in-package "SOAR"))

(eval-when (load eval compile)
  (export '(pgs-tab-size
            pgs-goal-leader
            pgs-pscm-id
            pgs-real-tab
            pgs-show-depth
            )
	(find-package "SOAR")))


;;;
;;;	i.	Controling variables
;;;
;;; The current Soar defaults, if they exist, are noted here.
;;; You can change them in your .soar-init.lisp file, or upon
;;; installing this file.

(push :spa-trace1.0 *features*)

(push :spa-trace *features*)

;;; TAQL when it loads calls trace-attributes on name for operators
;;; (Erik and I can't figure out why).  So if you load TAQL after loading 
;;; this file, you must either type this in by hand, or when running TAQL 3.1.5
;;;  put it in your .taql-init.lisp file.  If you load this after taql, then
;;; this line will protect you.

#+taql(untrace-attributes '((operator name)))

(defvar pgs-tab-size 3
  "*Number of leading spaces to indent each new goal.")
;; Soar5.2 default is 3

(defvar pgs-goal-leader "==>"
  "*String to put on the front of a new goal")
;; this must be the same length as pgs-tab-size
;; Soar5.2 default is ==>

(defvar pgs-real-tab t
  "*If not nil (default it t), insert a real tab after dc.")
;; Soar5.2 default is nil

(defvar pgs-pscm-id t
  "*If t (the default) print PSCM objects in trace or pgs with id.")
;; Soar5.2 default is t

(defvar pgs-show-depth t
  "*If t (default t) print periods in the trace for each level down.")
;; Soar5.2 default is nil


;;;
;;;	I.	print-pgs-context
;;;

(defun format-tab-or-depth (tab goal-depth)
 (declare (fixnum tab goal-depth))
 ;; Added test for zero to avoid Allegro bug. -BGM 9/21/89
 ;; note that absolute tabbing refers to local buffer.
 (if *subgoal-tabs*
     (if (zerop tab)
         ""
         (if pgs-show-depth
             (let ((result ""))
               ;(format t "[~s ~s]" tab goal-depth)
               (setq tab (1- tab))
               (dotimes (i goal-depth)
                 (setq result (concatenate 'string result "."))
                 (dotimes (i tab)
                    (setq result (concatenate 'string result " "))))
               result)
             (format NIL "~vT" tab)))
     (format NIL "(~D)~6T" goal-depth)))


(eval-when (compile eval load)
  (proclaim '(ftype (function (fixnum tme-object gme-value gme-value gme-value)
                                true) print-pgs-context)))

(defun print-pgs-context (depth goal space state operator)                 
 (declare (fixnum depth)
          (type tme-object goal)
          (type gme-value space state operator))
 ;;;; let ((tab (* pgs-tab-size depth)))
 ;; goal.
 (cond ((= depth 0)
        (soar-format *trace-file*
                     "~%~4T~AG: "
                     (format-tab-or-depth pgs-tab-size depth)))
       (T
        (soar-format *trace-file*
                     "~%~4T~A~AG: "
                     (format-tab-or-depth pgs-tab-size (1- depth))
                     pgs-goal-leader) ))
 (soar-format *trace-file* "~A ~A~a~a"
              (context-object-name goal)
              (format-trace-attributes goal)
              (if pgs-pscm-id " " "")
              (if pgs-pscm-id goal ""))
 ;; problem space.
 (if space
     (soar-format *trace-file*
                     "~%~4T~AP: ~A ~A~A~A"
                     (format-tab-or-depth pgs-tab-size depth)
                     (context-object-name space)
                     (format-trace-attributes space)
                     (if pgs-pscm-id " " "")
                     (if pgs-pscm-id space "")))
 ;; state.
 (if state
        (soar-format *trace-file*
                     "~%~4T~AS: ~A ~a~a~A"
                     (format-tab-or-depth pgs-tab-size depth)
                     (context-object-name state)
                     (format-trace-attributes state)
                     (if pgs-pscm-id " " "")
                     (if pgs-pscm-id state "")) )
 ;; operators.                                      
 (if operator
        (soar-format *trace-file*
                     "~%~4T~AO: ~A ~a~a~A"
                     (format-tab-or-depth pgs-tab-size depth)
                     (context-object-name operator)
                     (format-trace-attributes operator)
                     (if pgs-pscm-id " " "")
                     (if pgs-pscm-id operator "")) )
 T)

(defun goal-creation-report 
       (count goal attribute impasse items supergoal depth)
 (declare (fixnum depth)  (type tme-object goal)
          (ignore count attribute impasse items supergoal))
  (cond ((= depth 0)
         ;; first goal.
         (soar-format *trace-file*
                     (if pgs-real-tab
                         "~%~D	 ~AG: ~A ~A~A~a" ;; real tab & spc in this line
                         "~%~D~4T~AG: ~A ~A~A~a")
                      (1- (cycle-count 'quiet))
                      (format-tab-or-depth 0 (1- depth))
                      (context-object-name goal)
                      (format-trace-attributes goal)
                      (if pgs-pscm-id " " "")
                      (if pgs-pscm-id goal "")))
        (T    
         (soar-format *trace-file*
                     (if pgs-real-tab
                         "~%~D	 ~A~aG: ~A ~a~A~a" ;;real tab&spc in this line
                         "~%~D~4T~A~aG: ~A ~a~A~a")
                      (1- (cycle-count 'quiet))
                      (format-tab-or-depth pgs-tab-size (1- depth))
                      pgs-goal-leader
                      (context-object-name goal)
                      (format-trace-attributes goal)
                      (if pgs-pscm-id " " "")
                      (if pgs-pscm-id goal "")) ))
 T)

(eval-when (compile eval load) (proclaim '(ftype (function 
                     (fixnum tme-object gme-attribute gme-value)
                     true) context-installation-short-report))) 
(defun context-installation-short-report (count goal attribute value)
 (declare (type tme-object goal)
          (type gme-attribute attribute)
          (type gme-value value)
          (ignore count))
 (let ((depth (goal-depth goal)))
  (declare (fixnum depth))
  (soar-format *trace-file*
               (if pgs-real-tab
                   "~%~D	 ~A~A~A ~A~A~a" ;; real tab & spc in this line
                   "~%~D~4T~A~A~A ~A~A~a")
               (1- (cycle-count 'quiet))
               (format-tab-or-depth pgs-tab-size depth)  ;(* pgs-tab-size depth)
               (case attribute
                  (problem-space "P: ")
                  (state "S: ")
                  (operator "O: "))
               (context-object-name value)
               (format-trace-attributes value)
               (if pgs-pscm-id " " "")
               (if pgs-pscm-id value ""))
 T))

;; 
;; have to make name untracable
;; 

;; removed (cons 'name attributes)
;; which should also be an efficiency win

(eval-when (compile eval load) (proclaim '(ftype (function () list) trace-attributes-by-class)))
(defun trace-attributes-by-class ()
 (let ((class-attributes NIL))
  (declare (list class-attributes))
  (maphash 
   #'(lambda (class attributes)
      (declare (type tme-class class)
               (list attributes))
      (setf class-attributes 
            (cons (list class attributes) class-attributes)))
   *trace-attributes*)         
 class-attributes))

;; removed (cons 'name (gethash etc...)
;; which is also an efficiency win
(defdsmmacro class-trace-attributes (class)                      
                                    (tme-class) list
 (declare (type tme-class class))       
 `(remove NIL
         (gethash ,class *trace-attributes*)))


;; cleaned out the use of soar-format, not necc here
;; made it return "()" if no value there.
;; 29-Apr-92 -FER
(eval-when (compile eval load) (proclaim '(ftype (function (tme-object) true) format-trace-attributes)))
(defun format-trace-attributes (object)                  
 (declare (type tme-object object))

 ;; might remove outer parens here.
 (let ((expansion (nested-trace-attribute-values object)))
  (declare (list expansion))
  (format NIL "~A"
               (if expansion
                   expansion
                   "()")))
)

;; now put name back in for subparts
(eval-when (compile eval load) (proclaim '(ftype (function (list list) list) nested-trace-attribute-values-aide)))
(defun nested-trace-attribute-values-aide (visitees visited)
 (declare (list visitees visited))                    
  (mapcar
   #'(lambda (child)
      (declare (type tme-value child))
      (cond ((member child visited :test #'tme-value=)
             ;; cycle break. 
             child)
            (T
             (let ((child-onode (object-onode child)))
              (declare (type onode-or-NIL child-onode))
              (cond ((not child-onode)
                     ;; leaf.
                     child)
                    (T     
                     ;; leaf or branch.
                     (let ((grandchildren
                             (nested-trace-attribute-values-aide
                               (onode-trace-attribute-children-with-name child-onode)
                               (push child visited))))
                      (declare (list grandchildren))
                      (if grandchildren    
                          grandchildren
                          child) ) )) ) )) )
   visitees))

(defun onode-trace-attribute-children-with-name (onode)                   
 (declare (type onode onode))
 ;; returns list of values of object's name and (other) trace attributes.
 ;; assumes object is a valid object-id.

 (let ((children NIL))
  (declare (list children))
  (dolist (anode
           (mapcar #'(lambda (attribute)
                       (declare (type tme-attribute attribute))
                       (onode-to-anode onode attribute))
                   (cons 'name (class-trace-attributes (onode-class onode))))
           children)
    (declare (type anode-or-NIL anode))                     
    (if anode
        (dolist (aug (anode-augs anode))
         (declare (type aug aug))                                           
         (pushnew (tme-value (aug-ame aug)) children :test #'tme-value=)) ))
))                                   


;; now put name back on, but not this way, by putting into trace above.
;; 
;; (trace-attributes '((goal name)))
;; (trace-attributes '((space name)))
;; (trace-attributes '((state name)))
;; (trace-attributes '((operator name)))

(trace-attributes '((boolean name)))
(trace-attributes '((list name)))
(trace-attributes '((column name)))
(trace-attributes '((set name)))
(trace-attributes '((real name)))
(trace-attributes '((term name)))
(trace-attributes '((text  name)))
(trace-attributes '((space name)))
(trace-attributes '((integer name)))
(trace-attributes '((goal name)))
(trace-attributes '((fraction name)))
