#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#


;==========================================================================
; File:   output.lisp       Version: 1.31                 Created: 
;
; Locked by:  nobody.                                    Modified: 4/6/88
; 
; Purpose:    Output functions...
;
; NOTE: I explicitly used terpri instead of ~% in the format statements 
; because the format statements would do really funny things with 
; indenting.  For example, if I had a newline and then wanted to indent
; the next 2 lines with 3 spaces each, it would indent the first line
; with 3 spaces and the second line with 6 spaces.  Using terpri took care
; of this problem.
;==========================================================================
 

(proclaim '(special *NODES* *ALL-NODES* *TRACE-TEXT-FLAG* *DOMAIN-GRAPHICS* 
                    *PRINT-ALTS* *START-STATE* *PRINT-PREFS* *PRINT-ON-DECK* 
		    *START-TIME* *STOP-TIME* *OP-SEQ* *STEP-NUM*
		    *STOP*  *DOMAIN-WINDOW* *NODE-CUTOFF*))

;-------------------------------------------------------------------------- 

(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types"))

;-------------------------------------------------------------------------- 

(defun print-success ()
    (cond (*TRACE-TEXT-FLAG* (format t "~%~%Completed Success") t)))
 

(defun print-failure ()
    (cond  (*TRACE-TEXT-FLAG* (format t "~%~%Completed Failure") nil)))
 

(defun print-node-cutoff-reached ()
  (format t "~%~%Node cutoff reached: ~A. EXECUTION TERMINATED!" *NODE-CUTOFF*))

;-------------------------------------------------------------------------- 

;; PRINT-INITIAL-INFO prints out the initial goal and start states.  If 
;; graphics are turned on it will also display the world.
 
(defun print-initial-info ()
  (declare (special *STATE-TEXT-FLAG*))
  (when *STATE-TEXT-FLAG*
     (if (not (boundp '*START-STATE*)) (load-problem))
     (terpri)  (terpri)
     (format t "* * * * * P R O D I G Y ~A * * * * *" (version))
     (format t "~%~%Goal State: ~A" (op-preconds '*FINISH*))
     (if *TRACE-TEXT-FLAG* (print-list "Start State: " *START-STATE*))
     (finish-output)
  )
  (if *DOMAIN-GRAPHICS*
      (display-domain-graphics-start-state)))

;; This code will simply output a version message so that all
;; versions messages printed out by PRODIGY will be the same.
;; The number is returned as a string.

(defun version () "2.11")



;--------------------------------------------------------------------------  

 
(defun print-final-stats (final-node)
  (declare (special *STATE-TEXT-FLAG*))
  "Prints the statistics out after the problem has been solved."
  (when *STATE-TEXT-FLAG*
     (terpri) (terpri)

     (format t "CPU time: ~,1F seconds"
		(/ (- *STOP-TIME* *START-TIME*)
	        	  internal-time-units-per-second))
     (terpri)
     (format t "Number of Nodes: ~A" (commas (length *ALL-NODES*)))
     (if final-node
      (let ((op-seq (find-op-seq (node-parent final-node))))
	(terpri)
	(format t "Solution Length: ~A" (length op-seq))
	(print-op-list "Operator Sequence: " op-seq)))
     (terpri)
  )
  (if *DOMAIN-GRAPHICS* 
      (reset-domain-graphics-parameters)))
	  


(defun commas (n)
  (and (integerp n)
       (let* ((s (format nil "~A" n)) (ls (length s)) (n  "") (ps 0))
	 (dotimes (p ls n)
	   (setq ps (- ls p 1))
	   (setq n (if (and (> p 0) (zerop (mod  p 3)))
		       (concatenate 'string (subseq s ps (1+ ps)) "," n)
		       (concatenate 'string (subseq s ps (1+ ps)) n)))))))

;--------------------------------------------------------------------------  

;; PRINT-OP-LIST prints out the list of operators, ignoring the inference 
;; rules.  It stays in this function until the first operator is found.  This 
;; way the heading is only printed once.
 
(defun print-op-list (heading ops &optional (everything nil))
    (cond ((null ops) nil) ; could be that there are no ops cause of failure
	  ((or everything (operator-p (caar ops)))
	   (terpri)
	   (format t "~A~%" heading)
;;;	   (print-op (caar ops) (cadar ops))
	   (print-rest-op-list 	
	    (floor (length (princ-to-string heading)) 3)
	    ops
	    everything))
	  (t (print-op-list heading (cdr ops) everything))))

;--------------------------------------------------------------------------  
 

(defun print-rest-op-list (indent ops &optional (everything nil))
  "Prints out the remaining operators, if any."
    (cond ((null ops) nil)
	  ((or everything (operator-p (caar ops)))
	   (terpri)
	   (format t "~VT" indent)
	   (print-op (caar ops) (cadar ops))
	   (print-rest-op-list indent (cdr ops) everything))
	  (t (print-rest-op-list indent (cdr ops) everything))))
    
;-------------------------------------------------------------------------- 

;; PRINT-NODE prints out a brief summary of a node.  This function should be 
;; passed a node to be printed, a mode which indicates which alts will be 
;; printed ('nil' for none, 't' for the current alt, and 'all' for all the 
;; alts), and a level which forces the info to be printed at the given 
;; indentation level ('nil' prints the level based on the size of the goal 
;; stack).

 
(defun print-node (node mode level)
    (if *TRACE-TEXT-FLAG*
	   (cond (level (display-node node mode (* level 3)))
		 (t (display-node node mode (get-indent node)))))
    (cond (*DOMAIN-GRAPHICS*
	   (display-domain-graphics-state-change node)
	   (cond ((node-failure-reason node)
		; (display-failure-at-node node)
		;  (print-failure-reason node)
                )))))

;-------------------------------------------------------------------------- 

 
(defun display-node (node mode indent)
  "Prints out the node number, goal, and alts (if mode is non-nil)."
    (terpri)
    (terpri)
    (format t "~VT~A~A" indent (node-name node) " ")
    (cond ((not (null (node-children node)))
	   (print-current-alt node mode indent))
	  (t (format t "~A" 
		     (cond ((node-current-goal node))
			   (t " . . .")))))
    (cond ((node-failure-reason node)
	   (print-failure-reason node)
	   (terpri)
	   (format t "~A"
		   "-------------------------------------------------"))))
 
 
;--------------------------------------------------------------------------  

;; PRINT-CURRENT-ALT prints the current alt of a node, which is the alt that 
;; a node is currently pursuing. It will print DONE, and the operator name, 
;; if it is applicable, and then the alternatives associated with the node.

 
(defun print-current-alt (node mode indent)
    (let ((children (node-children node))
	  (generating-alt nil))
	 (cond (children
		   (setq generating-alt (node-generating-alt (car children)))
		   (format t "~A" (alt-goal generating-alt))
		   (cond (mode (print-alts (cons generating-alt 
						 (node-alternatives node))
				   mode indent (1- (length (node-children 
							       node))))))
		   (cond ((node-applied-node (car children))
			  (print-doing (car children) indent)))))))


 
;-------------------------------------------------------------------------- 

;; PRINT-ALTS prints out a list of alts based on the setting of mode.  In 
;; addition to the usual arguments, num-tried is passed to indicate how many
;; alts have already been tried.  This number is determined by the number
;; of childeren at a node.
;; Note that 'remaining is used to supress "Alts:" after the first alt 
;; has been printed.
 
(defun print-alts (alts mode indent num-tried)
    (let ((alt (car alts)))
	 (cond ((null alts))
	       ((eq mode 'all)
		(terpri)
		(format t "~VT~A" indent "Alts: ")
		(print-op (alt-op alt) (alt-vars alt))
		(print-alts (cdr alts) 'remaining (+ 6 indent) num-tried))
	       ((not (eq mode 'remaining))
		(terpri)
		(format t "~VT~A" indent "Alts: ")
		(print-alts-tried num-tried)
		(print-op (alt-op alt) (alt-vars alt))
		(print-alts-left num-tried (1- (length alts))))
	       (t (terpri)
		  (format t "~VT" indent)
		  (print-op (alt-op alt)(alt-vars alt))
		  (print-alts (cdr alts) mode indent num-tried)))))
 
 
;--------------------------------------------------------------------------  

 
(defun print-alts-tried (num-tried)
  "Prints the number of the alts already tried in square-brackets."
    (cond ((zerop num-tried) nil)
	  ((eql 1 num-tried) (format t "[1] "))
	  (t (format t "[1-~D] " num-tried))))

;-------------------------------------------------------------------------- 

(defun print-alts-left (num-tried num-left)
  "Prints the numbers of the alts left in square brackets."
    (cond ((zerop num-left) nil)
	  ((eql 1 num-left) (format t " [~D]" (+ 2 num-tried)))
	  (t (format t " [~D-~D]" (+ 2 num-tried) 
		  (1+ (+ num-tried num-left))))))
 
;-------------------------------------------------------------------------- 

 
(defun get-indent (node)
  "Calculates the indentation for a node 
   based on the size of the goal stack."
    (* 3 (length (node-goal-stack node)))) 
 

 
(defun print-doing (node indent)
  "Prints the operation or inference that was done at a node."
    (terpri)
    (terpri)
    (format t "~VT~A" (- indent 3) "...Done: ")
    (print-op (alt-op (node-generating-alt node))
	(alt-vars (node-generating-alt node))))
 
 

 
(defun print-done (node)
  "Prints the op or inference without a header for the explanation stuff."
    (print-op (alt-op (node-generating-alt node))
	(alt-vars (node-generating-alt node))))
 

 
(defun print-failure-reason (node &optional (indent-p t))
  "Prints out the failure reason for a node when backtracking occurs."
    (let ((failure-reason (car (node-failure-reason node))))
	 (cond ((eq failure-reason 'repeat-world)
		(print-repeat-world node 
		    (cadr (node-failure-reason node)) indent-p))
	       ((eq failure-reason 'goal-already-achieved)
		(print-goal-already-achieved node
		    (cadr (node-failure-reason node)) indent-p))
	       ((eq failure-reason 'goal-repeat)
		(print-goal-repeat node
		    (cadr (node-failure-reason node)) indent-p))
	       ((eq failure-reason 'no-relevant-alts)
		(print-no-relevant-alts node indent-p)))))
 
 
 

 
(defun print-no-relevant-alts (node &optional (indent-p t))
  "Failure message when alts have been exhausted."
    (and indent-p (indent-msg node))
    (princ "There are no relevant alts for this node!"))
 
 

 
(defun indent-msg (node)
  "Does indentation for backtracking message"
;   (cond  (*TRACE-TEXT-FLAG*
	   (format t "~%~VT" (get-indent node)))
;))



(defun print-goal-already-achieved (node goal &optional (indent-p t))
  "Failure message when goal has already been
   achieved unbeknowst to the problem solver."
    (and indent-p (indent-msg node))
    (format t "Goal has already been achieved! Goal: ~A" goal))
  
 

 
(defun print-goal-repeat (node goal &optional (indent-p t))
  "Failure message when a goal is found repeated on the stack."
    (and indent-p (indent-msg node))
    (format t "Goal Stack loop detected!")
    (and indent-p (indent-msg node))
    (format t "Goal on Stack: ~A" goal))
 

 
(defun print-repeat-world (node state &optional (indent-p t))
  "Failure message when there is a repeated state."
    (and indent-p (indent-msg node))
    (format t "World repeat detected with state: ~A!" (state-name state)))
 

 
(defun print-op (op ivars)
  "Prints out an operator with the variables instantiated"
   (format t "~A" op)
   (print-params (op-vars op) ivars (op-params op)))



 
(defun print-params (vars ivars params)
  "Prints out the parameter list associated with an operator."
    (cond ((null vars) nil)
	  ((member (car vars) params)
	   (write-char #\space)
	   (princ (car ivars))
	   (print-params (cdr vars) (cdr ivars) params))
	  (t (print-params (cdr vars) (cdr ivars) params))))
 


; detailed-alt in commands.lisp called performs the same function as show-alt.

; ======================================================================
;                     Printing  Alternatives
; ======================================================================

(defun show-alt (onealt)
"This function shows one alternative.  It is called from engine.lisp,
 by the function pop-best-alt.lisp when *take-advice* is turned on."
  (format t "~3,2T :goal ~S ~%" (alt-goal onealt))
  (format t "~3,2T :op ~S ~%" (alt-op onealt))
  (format t "~3,2T :unmatched-conds ~S ~%" (alt-unmatched-conds onealt))
  (format t "~3,2T :failed-cond ~S ~%" (alt-failed-cond onealt))
  (format t "~3,2T :vars ~S ~%" (alt-vars onealt))
  (format t "~3,2T :post-cond ~S ~%" (alt-post-cond onealt)))

; ======================================================================
;                 End    of    Output.lisp
; ======================================================================
