#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#


; ======================================================================
; File:  load-domain.lisp	 Version: 1-20 	     Created:  -------
; Locked by: nobody                                 Modified:  6/29/88
;
; Purpose:   This file contains functions used in loading a domain....
; ======================================================================


; (provide 'prodigy-load-domain)
; (require 'whatever)
; (in-package 'prodigy-load-domain)
; (export)
; (import)

; ----------------------------------------------------------------------

; NOTE: Before the function LOAD-DOMAIN is called (eg. in domain.l)
; the file functions.l (if it exists) should be loaded into lisp.

(proclaim '(special *TRUIFY-RELEVANCE-TABLE* *FALSIFY-RELEVANCE-TABLE*
          	    *OPERATORS* *INFERENCE-RULES* *PREDICATES* *STATIC-PREDS*
	            *FUNCTION-PREDS* *CLOSED-PREDS* *ACHIEVABLE-PREDS* 
  		    *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
  	     	    *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES*
	       	    *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
		    *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
   	            *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
	     	    *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*
		    *META-FUNCTIONS* *WORLD-PATH*
		    *NEGATABLE-PREDS* *START-STATE* *QUIET-LOAD* 
		    *LAST-LOAD-DOMAIN-TIME* *FINISH* 
		    *DOMAINS-PATH* *CURRENT-PROBLEM* *CURRENT-DOMAIN*
		    *PROBLEM-GOAL* *AUTODEFINED-FUNCTION-PREDS*
		    *AUTODEFINE-STATIC-FUNCTIONS* *PREDICATE-ARITIES*
		    *STATIC-STATE-PREDS*  *INITIAL-STATE-PREDS* *EBL-FLAG*))


; ----------------------------------------------------------------------

(eval-when (compile) 
  (load-path *PLANNER-PATH* "g-loop")
  (load-path *PLANNER-PATH* "g-map")
  (load-path *PLANNER-PATH* "data-types"))


; ----------------------------------------------------------------------

;; The following code initializes operators, inference rules, and tables.  
;; The attributes of each operator or inference rule are stored on the property
;; list of that object. Also creates *TRUIFY-RELEVANCE-TABLE* and
;; *FALSIFY-RELEVANCE-TABLE* which contain predicates and the corresponding
;; operators or inference rules that add or delete those predicates.  The goal
;; is stored in an operator called *FINISH* so that there is only one top
;; level goal.  POTENTIAL BUG: Make sure no predicate in the domain has the
;; same name as any pre-defined lisp function (eg. TYPE) since this will cause
;; load-domain to think that the predicate is implmented as a function. - SNM

(defun load-goal (goal)
  (declare (special *FINAL-STATE* *TRACE-TEXT-FLAG*))
  (setq *FINAL-STATE* goal)
  (setq *PROBLEM-GOAL* goal)
  (if *TRACE-TEXT-FLAG*
     (format t "~%  Goal State:  ~A" *PROBLEM-GOAL*))
  (setf (op-preconds '*FINISH*) *PROBLEM-GOAL*)
  (setf (op-effects '*FINISH*) '((add (done))))
  (setf (op-add-list '*FINISH*) '((done)))
  (add-lpreconds '(*FINISH*))
  (check-goal *PROBLEM-GOAL*))


(defun load-start-state (start-lits)
  (declare (special *INITIAL-STATE* *END-TIME*))
  (setq *START-STATE* start-lits)
  ;; Hack for the scheduling domain.
  (cond ((assoc 'last-time *START-STATE*)
	 (setq *END-TIME* (cadr (assoc 'last-time *START-STATE*)))))
  (if *REMOVE-UNUSED-PREDS-FLAG*
      (setf *START-STATE* (remove-unused-state-preds *START-STATE*)))
  (unbind-static-pred-functions)
  (check-start-state *START-STATE* *AUTODEFINE-STATIC-FUNCTIONS*)
  (find-static/initial-state-preds)
  (when *AUTODEFINE-STATIC-FUNCTIONS*
   (functionalize-static-predicates)
    (setq *START-STATE* (delete-fsps-from-start-state)))
  (setq *INITIAL-STATE* *START-STATE*))

;added makunbound  to eliminate false errors when changing
;domains.--dkahn

(defun load-domain ()
  (declare (special *EXPL-NODE*))
  "Load-domain checks the syntax of the operators, inference rules and
   control rules.  It generates a list of valid predicates and produces
   some warnings about such things as meta-functions in unusual places
   or conflicts between meta-functions and Common Lisp functions."
  (setf *EXPL-NODE* nil) ;So new graphics don't crash.
  (setq *TRUIFY-RELEVANCE-TABLE* nil *FALSIFY-RELEVANCE-TABLE* nil)

  (if *EBL-FLAG* (ebl-uniquefy-rules))
  (dolist (op *OPERATORS*) (add-operator op))
  (dolist (inference-rule *INFERENCE-RULES*) (add-inference inference-rule))
  (makunbound '*PREDICATES*) ;unbind to avoid errors in syntax check
  (dolist (rule (append *OPERATORS* *INFERENCE-RULES*))
    (add-rule-info rule)
    (check-rule-syntax rule))
  (push (list 'done '*FINISH*) *TRUIFY-RELEVANCE-TABLE*)
  (load-search-control-rules)
  (build-predicate-lists)
  (check-search-control-rules-syntax)
  (dolist (rule (append *OPERATORS* *INFERENCE-RULES*))
    (check-rule-syntax rule)); check again now that predicate lists etc, built
  (if *EBL-FLAG* (ebl-domain-processing))
  (setq *LAST-LOAD-DOMAIN-TIME* (get-internal-run-time))
  (fix-auto-scntrl-rules)
  'domain-is-loaded)


; ----------------------------------------------------------------------

; only allowed if operator does not mention any "new" predicates.
; Otherwise, whole new domain must be loaded.

(defun load-new-operator (op type)
  (cond ((check-rule-syntax op)
	 (format t "~% Operator not added")
	 nil)
	((eq type 'operator)
	 (push op *OPERATORS*)
	 (add-operator op))
	((eq type 'inference-rule)
	 (push op *INFERENCE-RULES*)
	 (add-inference op))
	(t (error "bad type")))
  (add-rule-info op)
  (build-predicate-lists)		; rebuild predicate lists in case some
					; predicates are now achievable.
  (setq *LAST-LOAD-DOMAIN-TIME* (get-internal-run-time))
  'domain-is-loaded)


; ----------------------------------------------------------------------

;; ADD-OPERATOR adds an operator to ??? and checks for invalid syntax.



(defun add-operator (op)
  (cond
   ((not (listp op))
    (format t "~% ERROR: ~A is not a valid operator:" op))
   (t (let ((name (car op)))
	(setf (op-operator name) t)
	(add-properties name (cdr op))
	))))


; ----------------------------------------------------------------------

 
(defun add-inference (inference-rule)
; Loads an inference rule
  (let ((name (car inference-rule)))
    (setf (op-inference-rule 'name) t)
    (add-properties name (cdr inference-rule))
    ))

; ----------------------------------------------------------------------

(defun add-rule-info (rule)
  (let ((name (car rule)))
    (make-add-and-delete-lists name (op-effects name))
    (setf (op-all-vars name) (find-all-vars (cdr rule)))
    (setf (op-all-vars-in-effects name)
	  (find-all-vars (op-effects name)))
    (cond ((op-operator name)
	   (setf (op-wildcard-vars name)
		 (get-wildcard-vars (op-preconds name) 
				    (op-del-list name))))
	  (t (setf (op-wildcard-vars name) 
		   nil))) ; wildcards only in deletes
; may want to change vars so only outside existentials are used
; sometimes people are lazy and dont define vars they know will
; be bound from the rhs....					 
    (setf (op-vars name)
	  (make-same-order-as-params 
	   name
	   (remove-duplicates 
	    (append (get-outside-existentials (op-preconds name))
		    (lst-difference (op-all-vars-in-effects name)
				    (op-wildcard-vars name)))
	    :test #'equal)))
  ;; need to add-lpreconds after setting all-vars-in-effects...
  (add-lpreconds rule)
  ;; should probably get rid of wildcards...
  (dolist (effect (extract-effects (op-effects name))) 	 
	  (cond ((eq (car effect) 'add)
		 (add-to-relevance-table name (cadr effect)))
 		((eq (car effect) 'del) 	
		 (add-to-relevance-table name (list '~ (cadr effect))))))))




; ----------------------------------------------------------------------

;; This function takes the name and attributes for an operator or inference
;; rule and create properties for each attribute.  

(defun add-properties (name attributes)
  (do* ((attrs attributes (cdr attrs))
	(attr (car attrs)(car attrs)))
       ((null attrs))
    (cond ((not (member (car attr) '(params preconds effects)))
	   (format t "~2% Invalid property, `~A', in ~A." (car attr) name))
	  ((not (listp (cadr attr)))
	   (format t "~2% Syntax error in ~A of ~A.~% Invalid expression: ~A"
		   (car attr) name (cadr attr)))
	  ((cddr attr)
	   (format t "~2% Syntax error in ~A of ~A.~% Extra expression~P: ~
	   ~{~A~%                    ~}~%"
		   (car attr) name (length (cddr attr)) (cddr attr))))
;    (setf (get name (car attr)) (cadr attr)))
    (case (car attr)
      (params
       (setf (op-params name) (cadr attr)))
      (preconds
       (setf (op-preconds name) (cadr attr)))
      (effects
       (setf (op-effects name) (cadr attr))))
))



; ----------------------------------------------------------------------

; orders vars so that params are in front, in same order as they
; occur in params list.
; If there is a param that is not in vars list, then do nothing
; error will be discovered in check-rule-syntax

(defun make-same-order-as-params (op-name vars)
  (g-loop (init params (op-params op-name) ret-val nil param nil)
	  (while (setq param (pop params)))
	  (do (cond ((member param vars)
		     (setq vars (del-eq param vars))
		     (push param ret-val))
		    (t nil)))
	  (result (append (nreverse ret-val) vars))))
				

; ----------------------------------------------------------------------

(defun get-wildcard-vars (preconds del-list)
  (g-loop (init del nil ret-val nil precond-vars (find-all-vars preconds))
	  (while (setq del (pop del-list)))
	  (do (cond ((eq (car del) 'if)
		     (setq ret-val 
			   (append (lst-difference 
				     (find-all-vars (cddr del))
				     (append (find-all-vars (cadr del))
					     precond-vars))
				   ret-val)))
		    ((setq ret-val
			   (append (lst-difference (find-all-vars del)
						   precond-vars)
				   ret-val)))))
	  (result ret-val)))

; ----------------------------------------------------------------------

;; GET-OUTSIDE-EXISTENTIALS returns the existentially quantified variables
;; that are not within the scope of a negated exists or a forall


(defun get-outside-existentials (exp)
  (cond ((atom exp)
	 (and (is-variable exp) (list exp)))
	((or (member (car exp) '(or and)) ;  should I include Or?
	     (atomic-formula-p exp))
	 (g-map (sub in (cdr exp))
		(splice (get-outside-existentials sub))))
	((eq (car exp) 'exists)
	 (g-map (sub in (cddr exp))
		(splice (get-outside-existentials sub))))
	((eq (car exp) 'forall) nil)
	((negated-p exp) nil)))

; ----------------------------------------------------------------------

;; MAKE-ADD-AND-DELETE-LISTS constructs an add-list and delete-list from the
;; effect list for the operator or inference rule called "name".

;; Basically, the add list contains the add effects from the effects-list,
;; and the delete-list contains the deletes.  Conditionals with
;; several effects are divided up into separate adds and deletes,
;; as in (if (foo) (add (bar1)) (add (bar2))), which goes on the
;; add-list as (if (foo) (bar1)) (if (foo) (bar2)). This is slightly
;; wasteful when the operator is actually applied (see adjust.lisp), but 
;; makes backchaining on effects easier (see engine.lisp).

;; del-list-for-matching is used in engine.lisp to match del-list
;; against the goal.

(defun make-add-and-delete-lists (name effects-list)
    (g-loop (init add-list nil del-list nil effect nil dels-for-matching nil)
	    (while (setq effect (pop effects-list)))
	    (do (cond ((eq (car effect) 'add) 
		       (push (cadr effect) add-list))
		      ((eq (car effect) 'del)
		       (push (cadr effect) del-list)
		       (push (list '~ (cadr effect)) dels-for-matching))
		      ((eq (car effect) 'if)
		       (dolist (then-part (cddr effect))
			       (cond ((eq (car then-part) 'add)
				      (push `(if ,(cadr effect)
						 ,(cadr then-part))
					    add-list))
				     (t 
					(push `(if ,(cadr effect)
						   ,(cadr then-part))
					      del-list)
					(push `(if ,(cadr effect)
						  ,(list '~ (cadr then-part)))
					      dels-for-matching)))))))
	    ; Both add-list and dels-for-matching are reversed to provide
	    ; the same default ordering as an older version
	    (result (progn (setf (op-add-list name) (reverse add-list))
			   (setf (op-dels-for-matching name)
				  (reverse dels-for-matching))
			   (setf (op-del-list name) del-list)))))

; ----------------------------------------------------------------------

(defun get-adds-from-conditional-effect (effect)
  (mapcan #'(lambda (sub-eff)
	      (cond ((eq (car sub-eff) 'add)
		     (list (cadr sub-eff)))
		    (t nil)))
	  (cddr effect)))


(defun get-dels-from-conditional-effect (effect)
  (mapcan #'(lambda (sub-eff)
	      (cond ((eq (car sub-eff) 'del)
		     (list (cadr sub-eff)))
		    (t nil)))
	  (cddr effect)))

; ----------------------------------------------------------------------

;; ADD-TO-RELEVANCE-TABLE adds an given atomic-formula and operator to the
;; corresponding relevence table.
;;
;; eg. (add-to-relevance-table 'move '(at RMG locx))
;; The tables each look like: 
;; ((predicate-name operator-name operator-name ...)
;;  (predicate-name inference-rule-name inference-rule-name) 
;;  (predicate-name inference-rule-name inference-rule-name) 
;;   ....)


(defun add-to-relevance-table (name formula)
  (prog (table table-entry pred)
     (cond ((eq '~ (car formula))
	    (setq pred (caadr formula))
	    (setq table '*FALSIFY-RELEVANCE-TABLE*))
	   (t (setq pred (car formula))
	      (setq table '*TRUIFY-RELEVANCE-TABLE*)))
     (setq table-entry (assoc pred (eval table)))
     (cond ((null table-entry)
	    (set table (cons (list pred name) (eval table))))
	   ((member name table-entry))
	   (t (rplaca (member table-entry (eval table) :test #'equal)
		      (cons pred (cons name (cdr table-entry))))))))

; ----------------------------------------------------------------------

;; ADD-LPRECONDS takes the preconds of an op in PDL, and them converts to
;; "old pdl", i.e. explicility adds generators for existially quantified
;; variables. Thus creates lpreconds, "longer" preconditions.  takes preconds,
;; and creates conj-lists, for each pred the list of other conjuncts in the
;; same conjunctive expression.


(defun add-lpreconds (op)
  (setf (op-lpreconds (car op))
	(existify (car op) (op-preconds (car op))  nil))
  (setf (op-conj-lists (car op))
	(get-conj-lists (op-preconds (car op)) nil)))

; ----------------------------------------------------------------------

;; For each literal GET-CONJ-LISTS gives the conjunctively related literals
;; that are completely instantiated with literal is reached in the
;; left-to-right match carried out by the matcher.

(defun get-conj-lists (precond-exp dvars)
  (cond ((member (car precond-exp) '(forall exists))
	 (and (get-exp precond-exp)	; watching out for exists without sub-exp
	      (get-conj-lists 
		(get-exp precond-exp) 
		(nconc (make-new-dvars (get-gen-exp precond-exp) dvars)
		       dvars))))
	((atomic-formula-p precond-exp) nil)
	((negated-p precond-exp)
	 nil)				; punt on negated exists
	((eq (car precond-exp) 'or)
	 (g-map (sub in (cdr precond-exp))
		(splice (get-conj-lists sub dvars))))
	((eq (car precond-exp) 'and)
	 (g-loop (init ret-val nil)
		 (next precond-exp (cdr precond-exp))
		 (while precond-exp)
		 (do (cond
		       ((is-literal (car precond-exp))
			(cond ((atomic-formula-p (car precond-exp))
			       (setq dvars
				     (nconc (make-new-dvars 
					      (car precond-exp) dvars) 
					    dvars))))
			(push (list (car precond-exp)
				    (get-defined-lits
				      (cdr precond-exp) dvars))
			      ret-val))
		       (t
			(setq ret-val
			      (append (get-conj-lists 
					(car precond-exp) dvars)
				      ret-val)))))
		 (result ret-val)))
	((null precond-exp) nil)
	((error "GET-CONJ-LISTS: bad expression"))))


; ----------------------------------------------------------------------

; OR's which generate are going to be a problem....
; so I'm disallowing them.
; ie. (and (or (p x) (q x)) (d x)...) is illegal, cause
; there's no way to make it into an old-syntax exp without
; having two (d x)'s

; I should change the syntax checker so that it checks generators
; for this problem...


(defun existify (op exp dvars)
  (cond ((eq (car exp) 'exists)
	 (cond ((get-exp exp)
		(list 'exists
		      (get-vars-lst exp)
		      (get-gen-exp exp)
		      (existify op (get-exp exp) 
				(append (get-vars-lst exp) dvars))))
	       (exp)))
	((eq (car exp) 'and)
	 (and-existify op (cdr exp) dvars))
	((eq (car exp) 'forall)
	 (list 'forall (get-vars-lst exp)
	       (get-gen-exp exp)
	       (existify op (get-exp exp)
			 (append (get-gen-exp exp) dvars))))
	((negated-p exp)
	 (list (car exp)
	       (existify op (cadr exp) dvars)))
	((eq (car exp) 'or)
	 (cons 'or (g-map (sub in (cdr exp))
			  (save (existify op sub dvars)))))
	((and (atomic-formula-p exp)
	      (has-new-dvars exp dvars))
	 (list 'exists (make-new-dvars exp dvars) exp))
	(exp)))

; ----------------------------------------------------------------------


(defun get-defined-lits (subs dvars)
  (g-map (sub in subs)
	 (when (and (is-literal sub)
		    (not (has-new-dvars sub dvars))))
	 (save sub)))
			 


(defun make-new-dvars (generator dvars)
  ; makes new set of defined-vars from values in generator
  (g-map (v in generator)
	 (when (and (is-variable v)
		    (not (member v dvars))))
	 (save v)))



(defun has-new-dvars (exp dvars)
  ; any new defined-vars in expression?
  (g-loop (init vars (find-all-vars exp))
	  (while vars)
	  (do (and (not (member (car vars) dvars))
		   (return t)))
	  (next vars (cdr vars))))

; ----------------------------------------------------------------------

;; AND-EXISTIFY processes a conjunctive expression, bringing out any 
;; generators.


(defun and-existify (op exps dvars)
  (g-loop (init next nil ret-val nil exps-partition nil new-dvars nil)
	  (while (setq next (pop exps)))
	  (do (cond ((atomic-formula-p next)
		     (cond ((has-new-dvars next dvars)
			    (setq new-dvars (make-new-dvars next dvars))
			    (setq exps-partition 
				  (partition-dependents op new-dvars
							dvars exps))
			    (setq exps (cadr exps-partition))
			    (cond ((null (car exps-partition))
				   (push (list 'exists new-dvars next)
					 ret-val))
;;;; ((null (cdr (car exps-partition))) ; only one
;;;; (push (list 'exists new-dvars next
;;;; (caar exps-partition))
;;;; ret-val))
				  (t (push
				       (list 'exists new-dvars next
					     (existify
					       op
					       (cons 'and (car exps-partition))
					       (append
						 (make-new-dvars next dvars)
						 dvars)))
				       ret-val))))
			   ((push next ret-val))))
		    ((push (existify op next dvars) ret-val))))
	  (result (cond ((null (cdr ret-val)) (car ret-val))
			((cons 'and (nreverse ret-val)))))))


; ----------------------------------------------------------------------

;; PARTITION-DEPENDENTS partitions the exps in a conjunction into those that
;; need to be in the exists exp, and those that can be outside.  Keeps exps in
;; order.  (see /usr/snm/logic/ps/old/load-domain) for a better version, but
;; changes order.  added all-vars-in-effects test because we have to make sure
;; all such vars are defined "linearly" for ebl stuff.

(defun partition-dependents (op-nm new-dvars old-dvars orig-exps)
  (g-loop (init exp nil exps orig-exps independents nil dependents nil)
	  (while (setq exp (pop exps)))
	  (do (cond ((or (intersectq-p new-dvars (find-all-vars exp))
			 (intersectq-p (op-all-vars-in-effects op-nm) 
				       (lst-difference (find-all-vars exp)
						       old-dvars)))
		     (setq new-dvars (append (find-all-vars
					       (cons exp independents))
					     new-dvars))
		     (setq dependents (cons exp
					    (append independents dependents)))
		     (setq independents nil))
		    ((push exp independents))))
	  (result (list (g-map (e in orig-exps)
		               (when (member e dependents))
			       (save e))
			(g-map (e in orig-exps)
			       (when (member e independents))
			       (save e))))))

; ----------------------------------------------------------------------
				      
;; CHECK-RULE-SYNTAX is used for checking the syntax of expressions and
;; rules (including operators). I assume the predicate lists have not been
;; built yet, so that some atomic expressions which illegal predicates
;; may slip by. (These problems should be caught later, when the predicate
;; lists are built). Note that once a syntactic error is caught in a rule
;; or in an expression, error checking stops, so you have to fix one error 
;; at a time.

(defun check-rule-syntax (rule)
  (prog (err)
	(cond 
	 ((not (atom (car rule)))
	  (setq err (list " missing name" (car rule)))
	  (format t " ~%~% Rule is missing name ~A" rule))
	 ((and (op-all-vars (car rule)) ; makes sure vars list has been built
	       (mapcan #'(lambda (v) 
			   (cond ((or (not (is-variable v))
				      (not (member v 
						   (op-all-vars (car rule)))))
				  (list v))
				 (t nil)))
		       (op-params (car rule))))
	  (terpri) (terpri) (format t " Bad params-list in: ~A" (car rule))
	  (terpri) (format t " ~A" (op-params (car rule))))
	 ((setq err (check-effects-list-syntax (car rule)))
	  (terpri) (terpri) 
	  (format t " Syntax error detected in effects of ~A :" (car rule))
	  (terpri) (format t " ~A ~A" (car err) (cadr err)))
	 ((setq err (check-exp-syntax (op-preconds (car rule))))
	  (terpri) (terpri)
	  (format t " Syntax error detected in preconditions of ~A :" 
		  (car rule))
	  (terpri) (format t " ~A ~A" (car err) (cadr err)))
	 ((setq err (check-all-variables rule))
	  (terpri) (terpri)
	  (format t " Syntax error detected in ~A :" (car rule))
	  (terpri) (format t " ~A" (car err))))
	(return err)))


; ----------------------------------------------------------------------


;; CHECK-EXP-SYNTAX returns nil if EXP is syntactically correct, otherwise
;; returns a list = (ERR-MSG SUB-EXP), where ERR-MSG is an error message,
;; and SUB-EXP is the sub-expression containing the error.

(defun check-exp-syntax (exp)
  (cond ((eq (car exp) 'forall)
	 (cond ((not (or (equal 4 (length exp)) (equal 3 (length exp))))
		(list "Expression is wrong length" exp))
	       ((check-vars-lst exp))
	       ((check-gen-exp exp))
	       (t (check-exp-syntax (get-exp exp)))))
	((eq (car exp) 'exists)	       
	 (cond ((not (or (equal 4 (length exp)) (equal 3 (length exp))))
		(list "Expression is wrong length" exp))
	       ((check-vars-lst exp))
	       ((check-gen-exp exp))
	       (t (check-exp-syntax (get-exp exp)))))
	((eq (car exp) '~)
	 (cond ((equal (caadr exp) 'exists)
		(check-exp-syntax (cadr exp)))
	       (t (check-atomic-formula (cadr exp)))))
	((eq (car exp) 'and)
	 (cond ((< (length exp) 3)
		(list "Not enough conjuncts in expression" exp))
	       (t (car (mapcan #'(lambda (x) 
				   (let ((val (check-exp-syntax x)))
				     (cond (val (list val))
					   (t nil))))
			       (cdr exp))))))
	((eq (car exp) 'or)
	 (cond ((< (length exp) 3)
		(list "Not enough disjuncts in expression" exp))
	       (t (car (mapcan #'(lambda (x) 
				   (let ((val (check-exp-syntax x)))
				     (cond (val (list val))
					   (t nil))))
			       (cdr exp))))))
	((null exp) nil)
	(t  (check-atomic-formula exp))))

; ----------------------------------------------------------------------
;; CHECK-EFFECTS-LIST-SYNTAX returns nil if the effects-list is 
;; syntactically correct, otherwise it
;; returns a list = (ERR-MSG SUB-EXP), where ERR-MSG is an error message, and
;; SUB-EXP is the sub-expression containing the error.
 
(defun check-effects-list-syntax (rule-nm)
  (g-loop (init err nil eff-list (op-effects rule-nm))
	  (while eff-list)
   	  (do (setq err (check-effect-syntax rule-nm (pop eff-list))))
	  (until err)
  	  (result err)))



;; CHECK-EFFECT-SYNTAX returns nil if EFF is syntactically correct, otherwise
;; returns a list = (ERR-MSG SUB-EXP), where ERR-MSG is an error message, and
;; SUB-EXP is the sub-expression containing the error.


(defun check-effect-syntax (rule-nm eff)
  (cond ((not (listp eff))
 	 (list "Effect must be a list " eff))
	((eq (car eff) 'if)
	 (cond ((< (length eff) 3)
		(list "IF expression is too short" eff))
	       ((check-exp-syntax (cadr eff)))
	       (t
		(car (mapcan #'(lambda (sub-eff)
				 (let ((temp (check-effect-syntax
					      rule-nm sub-eff)))
				   (cond (temp (list temp))
					 (t nil))))
			     (cddr eff))))))
	(t
	 (cond ((not (or (eq (car eff) 'add) (eq (car eff) 'del)))
		(list "An effect must be either add, del, or if" eff))
	       ((not (eq 2 (length eff)))
		(list "An effect must add or delete a single literal" eff))
	       ((and (op-inference-rule rule-nm)
		     (eq (car eff) 'del))
		(list "Inference rules cannot delete" eff))
	       (t
		(check-atomic-formula (cadr eff)))))))

; ----------------------------------------------------------------------

;; CHECK-VARS-LST checks the variables-list of a quantified expression.
;; Returns nil if OK.

(defun check-vars-lst (exp)
  (cond ((not (listp (get-vars-lst exp)))
	 (list "Need list of variables" (get-vars-lst exp)))
	((mapcan #'(lambda (v) (cond ((not (is-variable v))
				      (list v))
				     (t nil)))
		 (get-vars-lst exp))
	 (list "The list of variables must contain only variables" exp))
	(t nil)))

; ----------------------------------------------------------------------

;; CHECK-GEN-EXP checks the generator for a quantified expression.
;; Returns nil if OK.


(defun check-gen-exp (exp)
  (cond ((check-atomic-formula (get-gen-exp exp))
	 (list "Generator must be an atomic formula" (get-gen-exp exp)))
	(t nil)))



(defun check-atomic-formula (exp)
  (cond ((listp (car exp))
	 (list "Not a correct atomic formula: " exp))
	((not (boundp '*PREDICATES*)) nil) ;  default exit
	((member (car exp) *PREDICATES*) 
	 ;;
	 ;; arity check
	 ;;
	 (let ((expected-arity (cadr (assoc (car exp) *PREDICATE-ARITIES*)))
	       (actual-arity (length exp)))
	   (cond ((< actual-arity expected-arity)
		  (list "Too few terms in predicate" exp))
		 ((> actual-arity expected-arity)
		  (list "Too many terms in predicate" exp)))))
	(t (list "Unknown predicate in atomic formula: " exp))))

; ----------------------------------------------------------------------

; Checks whether list l1 has the same members as list l2. Assumes list l1
; has no duplicates.


(defun same-members (l1 l2)
  (and (eq (length l1) (length l2))
       (g-loop (while l1)
	       (do (cond ((not (member (car l1) l2 :test #'equal))
			  (return))))
	       (next l1 (cdr l1))
	       (result t))))

; ----------------------------------------------------------------------

;; CHECK-GOAL checks the syntax of a problem description. At this point, I
;; assume the predicate lists have been built. (Ie. You must load the domain
;; before you load a problem).
 
(defun check-goal (goal)
  (cond ((check-exp-syntax goal)
	 (terpri) (format t " Syntax error detected in goal statement")
	 (let ((error (check-exp-syntax goal)))
	   (terpri) (format t " Error type: ~A" (car error))
	   (terpri) (format t " ~A" (cadr error))))
	((set-difference (get-preds-from-exp goal) *PREDICATES*
			 :test #'equal)
	 (terpri) 
	 (format t " There are unknown predicates in the goal statement:")
	 (terpri)
	 (format t " ~A" (set-difference (get-preds-from-exp goal)
				*PREDICATES* :test #'equal)))))

(defun remove-unused-state-preds (start-state)
     (remove-if-not #'(lambda (x) (member x *PREDICATES*))
		    start-state
		    :key #'car)
)

(defun check-start-state (start-state auto-define-p)
  (let ((start-preds (mapcar #'car start-state))
	(closed-world-preds (append *CLOSED-PREDS* *FUNCTION-PREDS*)))
    (cond ((set-difference start-preds *PREDICATES*
			:test #'equal)
	   (terpri) 
	   (format t " There are unknown predicates in *START-STATE*:")
	   (terpri)
	   (format t " ~A" (remove-duplicates
			    (set-difference start-preds
					    *PREDICATES* :test #'equal)))))
    (cond ((not (subsetp start-preds closed-world-preds))
	   (terpri)
	   (format t " There are open world predicates in the start state:")
	   (terpri)
	   (format t " ~A" (remove-duplicates
			    (set-difference start-preds closed-world-preds)))))
    (cond ((find-if #'fboundp start-preds)
	   (warn-or-err (find-if #'fboundp start-preds)
		        auto-define-p)))
	  ))

(defun warn-or-err (bad-pred error-p)
  "Generates an error if error-p else causes a warning to happen."
  (cond (error-p (error "The initial state predicate ~S has a function definition.
this cannot be used with DSPF or else it would overwrite the pre-existing
function.  Remove the predicate from the start state or turn off DSPF."
		        bad-pred))
	
	(t (format t " ~%WARNING: The initial state predicate ~S has a
         function definition.~%" bad-pred))))


  

; ----------------------------------------------------------------------

(defun check-all-variables (x)
  (cond ((null x) nil)
	((listp x)
	 (or (check-all-variables (car x))
	     (check-all-variables (cdr x))))
	((symbolp x)
	 (let ((b (char-equal #\< (elt (symbol-name x) 0)))
	       (e (char-equal #\> (elt (symbol-name x)
				       (1- (length (symbol-name x))))))
	       (lx (length (symbol-name x))))
	   (cond ((and (> lx 1) (or (and b (not e)) (and e (not b))))
		  (list (format nil "The atom ~A is not a well formed ~
		  variable." x))))))))


; ======================================================================
;                       Build   Predicate  Lists
; ======================================================================

;; BUILD-PREDICATE-LISTS builds the predicate lists. If the lists are already
;; defined (ie. the corresponding global variables are already bound), then we
;; check to make sure that the predicate list calculated by the code below is
;; the same as list given by the user, and if not, the user is warned. The
;; user is also notified of any staticc predicates not implmented as functions.
;; (I assume functions.lisp has already been loaded.) If the global 
;; *QUIET-LOAD* is set to nil, then the user is told all the calculated 
;; predicate lists.
;;
;; The meanings of the various predicate lists are given below: (These lists
;; are used by the matcher, and presumably, byany analysis routines built on
;; top of PRODIGY).
;;
;;  *PREDICATES*     -  All predicates
;;
;;  *STATIC-PREDS*   -  All predicates that don't change their value. 
;;                      (ie. Relations that are unaffected by operations 
;;                      in the domain).
;;
;;  *FUNCTION-PREDS* -  Static preds that are implemented as functions. See
;; 		        the file functions.l for their definitions.
;;
;;  *CLOSED-PREDS*   -  All predicates that, if they are not known to be 
;; 		        true can be assumed to be false. Found in the add 
;;		        and delete lists of operators, and including all 
;;		        static preds not implmented as functions.	
;;
;;*ACHIEVABLE-PREDS* -  All predicates that are added by some operator or
;;		        inference rule. (Not including any negated 
;;		        predicates that are added by an inference rule).
;;
;; *NEGATABLE-PREDS* -  All predicates whose negation is added by some
;; 		        inference rule, and all closed world predicates that
;;		        are deleted.



(defun build-predicate-lists ()
  (setq *ACHIEVABLE-PREDS* (get-achievable-preds))
  (setq *NEGATABLE-PREDS* (get-negatable-preds))
  (setq *PREDICATES*
	(remove-duplicates 
	  (append *META-FUNCTIONS* 
		  (remove-duplicates (get-all-rule-preds) :test #'equal))
	  :test #'equal))
 ; test for any predicate being a lisp function which could screw up
 ; the works later -mpm 9/14/88
  (dolist (pred *PREDICATES*) (test-whether-bad-pred-name pred))
  (setq *FUNCTION-PREDS* (extract-functions *PREDICATES*))
  (setq *STATIC-PREDS*
	(remove-duplicates
	  (intersect (set-difference *PREDICATES* *ACHIEVABLE-PREDS*
				     :test #'equal)
		     (set-difference *PREDICATES* *NEGATABLE-PREDS* 
				     :test #'equal)) :test #'equal))
  (warn-about-non-function-static-preds *STATIC-PREDS* *FUNCTION-PREDS*)
  (setq *CLOSED-PREDS*
	(remove-duplicates 
	  (append (mapcan #'(lambda (op) (extract-add-preds op)) 
			  *OPERATORS*)
		  (mapcan #'(lambda (op) (extract-del-preds op))
			  *OPERATORS*)
		  (set-difference *STATIC-PREDS* *FUNCTION-PREDS*
				  :test #'equal))))
  (setq *PREDICATE-ARITIES* nil)
  (create-pred-arity-list
   (append *OPERATORS* *INFERENCE-RULES* *SCR-NODE-SELECT-RULES*
	   *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
	   *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES*
	   *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
	   *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
	   *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
	   *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*))

				  
  (if (not *QUIET-LOAD*) (print-predicate-lists-for-user)))

; ----------------------------------------------------------------------

; for user-defined meta-functions. Must be used before domain is loaded.

(defun add-meta-fn (fn-nm)
  (pushnew fn-nm *meta-functions*)) 


; ----------------------------------------------------------------------

(defun print-predicate-lists-for-user ()
  (g-loop (init predicate-lists 
		'(*ACHIEVABLE-PREDS* *NEGATABLE-PREDS*
		  *STATIC-PREDS* *PREDICATES*
		  *FUNCTION-PREDS* *CLOSED-PREDS*)
		list-name nil)
	  (while (setq list-name (pop predicate-lists)))
	  (do (terpri) (terpri)
	    (format t " Calculated ~A to be: ~A" 
		    list-name (eval list-name)))))

; ----------------------------------------------------------------------

(defun warn-about-non-function-static-preds (static-preds func-preds)
  (declare (special *TRACE-TEXT-FLAG*))
  (let ((nf-static-preds
	 (set-difference static-preds func-preds
			 :test #'equal)))
    (cond ((and nf-static-preds *TRACE-TEXT-FLAG*)
	   (format t "~% NOTE: The following static predicates have")
	   (format t " not been implemented ~% as functions: ")
	   (format t " ~A~%" nf-static-preds)))))

; ----------------------------------------------------------------------
 
(defun get-achievable-preds ()
  (remove-duplicates (append (mapcan #'(lambda (rule)
					 (extract-add-preds rule))
				     *INFERENCE-RULES*)
			     (mapcan #'(lambda (op)
					 (extract-add-preds op))
				     *OPERATORS*))
		     :test #'equal))

; ----------------------------------------------------------------------

(defun get-negatable-preds ()
  (remove-duplicates (append (mapcan #'(lambda (rule)
					 (extract-neg-add-preds rule))
				     *INFERENCE-RULES*)
			     (mapcan #'(lambda (op)
					 (extract-del-preds op))
				     *OPERATORS*))
		     :test #'equal))


; ----------------------------------------------------------------------

;; EXTRACT-ADD-PREDS returns all non-negated predicates that are added in
;; an effects list. Includes those in conditionals.


(defun extract-add-preds (rule-or-op)
  (mapcan #'(lambda (effect)
	      (cond ((and (eq (car effect) 'add) 
			  (not (negated-p (cadr effect))))
		     (list (car (cadr effect))))
		    (t nil)))
	  (extract-effects (op-effects (car rule-or-op)))))

; ----------------------------------------------------------------------

;; EXTRACE-NEG-PREDS Returns all negated predicates that are added in an
;; effects list. Includes those in conditionals.

(defun extract-neg-add-preds (rule)
  (mapcan #'(lambda (effect)
	      (cond ((and (eq (car effect) 'add) 
			  (negated-p (cadr effect)))
		     (list (caadr (cadr effect))))
		    (t nil)))
	  (extract-effects (op-effects (car rule)))))

; ----------------------------------------------------------------------

;; EXTRACT-DEL-PREDS returns all predicates that are deleted in an 
;; effects list. Includes those in conditionals.

(defun extract-del-preds (op)
  (mapcan #'(lambda (effect)
	      (cond ((eq (car effect) 'del)
		     (list (car (cadr effect))))
		    (t nil)))
	  (extract-effects (op-effects (car op)))))


; ----------------------------------------------------------------------

;; EXTRACT-EFFECTS returns all effects [eg. (add (literal)) or 
;; (del (literal))] in an effects list.  Included are 
;; effects that are within IFs (i.e. conditionals). Thus, if the list 
;; includes something like (if (foobar) (add (p a)) (del (p q)))
;; the returned value will include (add (p a)) and (del (p q)).


(defun extract-effects (eff-list)
  (g-loop (init ret-val nil eff nil)
	  (while (setq eff (pop eff-list)))
          (do (cond ((equal (car eff) 'if)
		     (setq ret-val (append (cddr eff) ret-val)))
		    (t (push eff ret-val))))
          (result ret-val)))



    

; ----------------------------------------------------------------------

;; GET-ALL-RULE-PREDS returns all the predicates mentioned in the
;; preconditions of the rules (and operators).
;; NOTE: should also test the rhs of restriction rules!

(defun get-all-rule-preds ()
  (append (mapcan 
	    #'(lambda (rule)
		(get-preds-from-exp (scr-lhs (car rule))))
	    (append *SCR-NODE-SELECT-RULES* *SCR-GOAL-SELECT-RULES*
		    *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES* 
		    *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
		    *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
		    *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES*
		    *SCR-OP-PREFERENCE-RULES* 
		    *SCR-BINDINGS-PREFERENCE-RULES*))
	  (mapcan 
	    #'(lambda (rule)
		(append (get-preds-from-effects (op-effects (car rule)))
			(get-preds-from-exp (op-preconds (car rule)))))
	    (append *OPERATORS* *INFERENCE-RULES*))))

; ----------------------------------------------------------------------
 
(defun get-preds-from-effects (effects)
  (g-loop (init eff nil ret-val nil)
	  (while (setq eff (pop effects)))
	  (do (cond ((eq (car eff) 'if)
		     (setq ret-val
			   (append (get-preds-from-exp (cadr eff))
				   (g-map (sub-eff in (cddr eff))
					  (save (caadr sub-eff)))
				   ret-val)))
		    ((setq ret-val 
			   (cons (caadr eff) ret-val)))))
	  (result ret-val)))


; ----------------------------------------------------------------------

(defun get-preds-from-exp (exp)
  ;; returns all the predicates mentioned in an expression.
  (cond ((null exp) nil)
	((member (car exp) '(known achievable provable))
	 (get-preds-from-exp (caddr exp)))
	((or (eq (car exp) 'forall) 
	     (eq (car exp) 'exists))
	 (nconc (get-preds-from-exp (get-gen-exp exp))
		(get-preds-from-exp (get-exp exp))))
	((eq (car exp) '~)
	 (cond ((equal (caadr exp) 'exists)
		(get-preds-from-exp (cadr exp)))
	       (t (list (caadr exp)))))
	((or (eq (car exp) 'and)
	     (eq (car exp) 'or))
	 (mapcan #'(lambda (subexp)
		     (get-preds-from-exp subexp))
		 (cdr exp)))
	(t (list (car exp)))))

; ----------------------------------------------------------------------

;; EXTRACT-FUNCTIONS returns all predicates with function definitions. Assumes
;; functions.l has been loaded.

(defun extract-functions (preds)
  (mapcan #'(lambda (p) (cond ((fboundp p) (list p)) ; has function definition
			      (t nil)))
	  preds))

; ----------------------------------------------------------------------

(defun lst-difference (l1 l2)
  ;; returns all the members of list l1 that aren't members of list l2
  (cond ((null l1) nil)
	((member (car l1) l2) (lst-difference (cdr l1) l2))
	(t (cons (car l1) (lst-difference (cdr l1) l2)))))

; ----------------------------------------------------------------------

(defun create-pred-arity-list (rule-list)
  (dolist (rule rule-list) (r-create-pred-arity-list (cdr rule))))

(defun r-create-pred-arity-list (x)
  (cond ((or (null x) (atom x))  nil)
	((and (symbolp (car x))
	      (member (car x) *PREDICATES*)
	      (not (assoc (car x) *PREDICATE-ARITIES*)))
	 (push (list (car x) (length x)) *PREDICATE-ARITIES*)
	 (dolist (subexp (cdr x)) 
	   (r-create-pred-arity-list subexp)))
	(t 
	 (r-create-pred-arity-list (car x))
	 (r-create-pred-arity-list (cdr x)))))

; ----------------------------------------------------------------------

(defun test-whether-bad-pred-name (pred)
  (cond ((lisp-function-p pred)
	 (format t "~2% WARNING: the predicate name  \" ~:@(~A~) \"  is a~
	 ~% predefined lisp function.   Conflicts will occur.~%" 
		 pred))))



; ======================================================================
;                   Reading in Search Control Rules 
; ======================================================================

(defun load-search-control-rules ()
  (dolist (scr-rule-type  
	    '(*SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
	      *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES*
	      *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
	      *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
	      *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
	      *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*))
    (if (not (boundp scr-rule-type)) (set scr-rule-type nil)))
  (dolist (rule *SCR-NODE-SELECT-RULES*)
    (setf (scr-rule-type (car rule)) 'node-select))
  (dolist (rule *SCR-GOAL-SELECT-RULES*)
    (setf (scr-rule-type (car rule)) 'goal-select))
  (dolist (rule *SCR-OP-SELECT-RULES*)
    (setf (scr-rule-type (car rule)) 'op-select))
  (dolist (rule *SCR-BINDINGS-SELECT-RULES*)
    (setf (scr-rule-type (car rule)) 'bindings-select))
  (dolist (rule *SCR-NODE-REJECT-RULES*)
    (setf (scr-rule-type (car rule)) 'node-reject))
  (dolist (rule *SCR-GOAL-REJECT-RULES*)
    (setf (scr-rule-type (car rule)) 'goal-reject))
  (dolist (rule *SCR-OP-REJECT-RULES*)
    (setf (scr-rule-type (car rule)) 'op-reject))
  (dolist (rule *SCR-BINDINGS-REJECT-RULES*)
    (setf (scr-rule-type (car rule)) 'bindings-reject))
  (dolist (rule *SCR-NODE-PREFERENCE-RULES*)
    (remprop (car rule) 'priority)
    (setf (scr-rule-type (car rule)) 'node-pref))
  (dolist (rule *SCR-GOAL-PREFERENCE-RULES*)
    (remprop (car rule) 'priority)
    (setf (scr-rule-type (car rule)) 'goal-pref))
  (dolist (rule *SCR-OP-PREFERENCE-RULES*)
    (remprop (car rule) 'priority)
    (setf (scr-rule-type (car rule)) 'op-pref))
  (dolist (rule *SCR-BINDINGS-PREFERENCE-RULES*)
    (remprop (car rule) 'priority)
    (setf (scr-rule-type (car rule)) 'bindings-pref))
   
  (dolist (rule (append 
		  *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
		  *SCR-OP-SELECT-RULES*    *SCR-BINDINGS-SELECT-RULES*
		  *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
		  *SCR-OP-REJECT-RULES*    *SCR-BINDINGS-REJECT-RULES* 
		  *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
		  *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*))
    (set-scr-rule-attributes rule)))
     
(defun set-scr-rule-attributes (rule)
  (let ((nm (car rule)))
    (setf (scr-match-time nm) 0)
    (dolist (attr (cdr rule))
	    (cond ((not (member (car attr) '(lhs rhs priority)))
		   (format t "~2% Invalid property, `~A', in ~A."
			   (car attr) nm))
		  ((and (not (listp (cadr attr)))(not (numberp (cadr attr))))
		   (format t "~2% Syntax error in ~A of ~A.~% Invalid expression: ~A"
			   (car attr) nm (cadr attr)))
		  ((cddr attr)
		   (format t "~2% Syntax error in ~A of ~A.~% Extra expression~P: ~
	   ~{~A~%                    ~}~%"
			   (car attr) nm (length (cddr attr)) (cddr attr))))
	    (case (car attr)
	      (lhs
	       (setf (scr-lhs nm) (cadr attr)))
	      (rhs
	       (setf (scr-rhs nm) (cadr attr)))
	      (priority
	       (setf (scr-priority nm) (cadr attr)))))))
;	    (setf (get nm (car attr)) (cadr attr)))))

; ----------------------------------------------------------------------


(defun load-new-scntrl-rule (rule rule-type)
  (let ((name (car rule)))
    (or (member rule-type '(node-select goal-select op-select bindings-select
			    node-reject goal-reject op-reject bindings-reject
			    node-pref goal-pref op-pref bindings-pref))
	(error "LOAD-NEW-SCNTRL-RULE: bad rule-type" rule-type))
    (or name (error "LOAD-NEW-SCNTRL-RULE: must have whole rule"))
    (setf (scr-rule-type name) rule-type)
    (g-map (rule-list
	     in 
	     (list *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
		   *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES* 
		   *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
		   *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES* 
		   *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
		   *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*))
	   (when (member name rule-list))
	   (do (error "LOAD-NEW-SCNTRL-RULE: rule-nm already exists")))
    (set-scr-rule-attributes rule)
    (cond ((and (member rule-type 
			'(node-select goal-select op-select bindings-select))
		(check-scr-rule-syntax name 'select-rule))
	   nil)
	  ((and (member rule-type 
			'(node-reject goal-reject op-reject bindings-reject))
		(check-scr-rule-syntax name 'reject-rule))
	   nil)
	  ((and (member rule-type '(node-pref goal-pref op-pref bindings-pref))
		(or (check-scr-rule-syntax name 'preference-rule)
		    (check-pref-priorities (list rule))))
						;dkahn--something of a hack
	   nil)
	  ((push-rule-onto-rule-list rule rule-type)))))

; ----------------------------------------------------------------------


(defun push-rule-onto-rule-list (rule rule-type)
  (cond ((eq rule-type 'node-select)
	 (push rule *SCR-NODE-SELECT-RULES*))
	((eq rule-type 'goal-select)
	 (push rule *SCR-GOAL-SELECT-RULES*))
	((eq rule-type 'op-select)
	 (push rule *SCR-OP-SELECT-RULES*))	  
	((eq rule-type 'bindings-select)
	 (push rule *SCR-BINDINGS-SELECT-RULES*))    
	((eq rule-type 'node-reject)
	 (push rule *SCR-NODE-REJECT-RULES*))
	((eq rule-type 'goal-reject)
	 (push rule *SCR-GOAL-REJECT-RULES*))
	((eq rule-type 'op-reject)
	 (push rule *SCR-OP-REJECT-RULES*))	  
	((eq rule-type 'bindings-reject)
	 (push rule *SCR-BINDINGS-REJECT-RULES*))    
	((eq rule-type 'node-pref)
	 (push rule *SCR-NODE-PREFERENCE-RULES*))
	((eq rule-type 'goal-pref)
	 (push rule *SCR-GOAL-PREFERENCE-RULES*))
	((eq rule-type 'op-pref)
	 (push rule *SCR-OP-PREFERENCE-RULES*))	  
	((eq rule-type 'bindings-pref)
	 (push rule *SCR-BINDINGS-PREFERENCE-RULES*))
	((error "bad rule type?"))))
    
; ----------------------------------------------------------------------
(defun delete-control-rule (rule rule-type &aux (rule-nm (car rule)))
   (declare (ignore rule-type))
   "This function removes the control rule from the system."
   (case (scr-rule-type rule-nm)
 
	('node-select
         (setf *SCR-NODE-SELECT-RULES* 
		(delete-rule rule-nm *SCR-NODE-SELECT-RULES*)))
        ('goal-select
	 (setf *SCR-GOAL-SELECT-RULES*
          (delete-rule rule-nm *SCR-GOAL-SELECT-RULES*)))
        ('op-select
	 (setf *SCR-OP-SELECT-RULES*
          (delete-rule rule-nm *SCR-OP-SELECT-RULES*)))
        ('bindings-select
	 (setf *SCR-BINDINGS-SELECT-RULES*
          (delete-rule rule-nm *SCR-BINDINGS-SELECT-RULES*)))
        ('node-reject
	 (setf *SCR-NODE-REJECT-RULES*
          (delete-rule rule-nm *SCR-NODE-REJECT-RULES*)))
        ('goal-reject
	 (setf *SCR-GOAL-REJECT-RULES*
          (delete-rule rule-nm *SCR-GOAL-REJECT-RULES*)))
        ('op-reject
	 (setf *SCR-OP-REJECT-RULES*
          (delete-rule rule-nm *SCR-OP-REJECT-RULES*)))
        ('bindings-reject
	 (setf *SCR-BINDINGS-REJECT-RULES*
          (delete-rule rule-nm *SCR-BINDINGS-REJECT-RULES*)))
        ('node-pref
	 (setf *SCR-NODE-PREFERENCE-RULES*
          (delete-rule rule-nm *SCR-NODE-PREFERENCE-RULES*)))
        ('goal-pref
	 (setf *SCR-GOAL-PREFERENCE-RULES*
          (delete-rule rule-nm *SCR-GOAL-PREFERENCE-RULES*)))
        ('op-pref
	 (setf *SCR-OP-PREFERENCE-RULES*
          (delete-rule rule-nm *SCR-OP-PREFERENCE-RULES*)))
        ('bindings-pref
	 (setf *SCR-BINDINGS-PREFERENCE-RULES*
          (delete-rule rule-nm *SCR-BINDINGS-PREFERENCE-RULES*)))
	)


    (remove-list-of-properties  rule-nm '(match-time lhs rhs
						priority rule-type))
)


(defun delete-rule (rule rule-list)
   (remove-if #'(lambda (x) (eq rule (car x))) rule-list))

; ======================================================================
;              Syntax checking for search control rules 
; ======================================================================

(defun check-search-control-rules-syntax ()
  (dolist (rule  (append *SCR-NODE-SELECT-RULES*
			 *SCR-GOAL-SELECT-RULES*
			 *SCR-OP-SELECT-RULES*
			 *SCR-BINDINGS-SELECT-RULES*))
    (check-scr-rule-syntax (car rule) 'select-rule))
  (dolist (rule  (append *SCR-NODE-REJECT-RULES*
			 *SCR-GOAL-REJECT-RULES*
			 *SCR-OP-REJECT-RULES*
			 *SCR-BINDINGS-REJECT-RULES*))
    (check-scr-rule-syntax (car rule) 'reject-rule))
  (dolist (rule  (append *SCR-NODE-PREFERENCE-RULES*
			 *SCR-GOAL-PREFERENCE-RULES*
			 *SCR-OP-PREFERENCE-RULES*
			 *SCR-BINDINGS-PREFERENCE-RULES*))
    (check-scr-rule-syntax (car rule) 'preference-rule))
  (check-pref-priorities *SCR-NODE-PREFERENCE-RULES*)
  (check-pref-priorities *SCR-GOAL-PREFERENCE-RULES*)
  (check-pref-priorities *SCR-OP-PREFERENCE-RULES*)
  (check-pref-priorities *SCR-BINDINGS-PREFERENCE-RULES*))


; ----------------------------------------------------------------------

(defun check-pref-priorities (rules)
  (g-loop
    (init rule nil rule-nm nil old-priority 0)
    (while (setq rule (pop rules)))
    (do (setq rule-nm (car rule))
	(cond 
	  ((not (scr-priority rule-nm))
	   (format t "~% WARNING: No priority given for preference rule ~A"
		   rule-nm)
	   (format t "~%          Assigning this rule priority ~D" old-priority)
	   (setf (scr-priority rule-nm) old-priority))
	  ((not (numberp (scr-priority rule-nm)))
	   (format t
		   "~% ERROR: Non numeric priority given for preference rule ~A"
		   rule-nm))
	  ((< (scr-priority rule-nm) old-priority)
	   (format t "~% ERROR: while reading ~A" rule-nm)
	   (format t "~% Preference rules must be listed in ascending priority")))
      (setq old-priority (scr-priority rule-nm))))
  nil) ;dkahn must return nil for load-new-scr-rule hack to work.

; ----------------------------------------------------------------------


(defun check-scr-rule-syntax (rule-nm rule-type)
  (cond 
    ((null (scr-lhs rule-nm))
     (terpri)
     (format t " ERROR: no Left-hand-side in control rule ~A" rule-nm)
     t)
    ((null (scr-rhs rule-nm))
     (terpri)
     (format t " ERROR: no right-hand-side in control rule ~A" rule-nm)
     t)
    ((check-scr-rhs (scr-rhs rule-nm) rule-type rule-nm)
     (terpri) (terpri)
     (format t " Syntax error detected in rhs of control rule: ~A" rule-nm)
     t)
    ((check-scr-exp (scr-lhs rule-nm))
     (terpri) (terpri)
     (format t " Syntax error detected in lhs of control rule: ~A" rule-nm)
     (let ((error (check-scr-exp (scr-lhs rule-nm))))
       (terpri) (format t " Error type: ~A" (car error))
       (terpri) (format t " ~A" (cadr error)))
     t)
    ((check-for-illegal-generators rule-nm)
	(let ((error (check-for-illegal-generators rule-nm)))
	    (princ error)
	)
    t)
    (t nil)))

; ----------------------------------------------------------------------

    
(defun get-pred-from-exp (pred-nm exp)
  (cond ((atomic-formula-p exp)
	 (cond ((eq pred-nm (car exp)) exp)
	       (t nil)))
	((member (car exp) '(forall exists)) 	 (or
(get-pred-from-exp pred-nm (get-gen-exp exp)) 	 (get-pred-from-exp
pred-nm (get-exp exp)))) 	((member (car exp) '(and or)) 	 (car
(mapcan #'(lambda (sub) 			 (let ((val
(get-pred-from-exp pred-nm sub))) 			 (cond (val
(list val)) 				 (t nil)))) 		 (cdr
exp)))) 	((negated-p exp) 	 (get-pred-from-exp pred-nm
(cadr exp)))))


;----------------------------------------------------------------------

; A few helping macros for check-scr-rhs

(defmacro rhs-type (rhs1) `(first ,rhs1))

(defmacro rhs-type-p (rhs1 type) "Determins the type (e.g. select,
reject, prefer)" `(eq (first ,rhs1) ,type))

(defmacro rhs-level (rhs1) "Returns level (e.i. bindings, goal,
operator, node) or rhs" `(second ,rhs1))

(defmacro rhs-arg (rhs1) "This is the value that will be either
selected or rejected" `(third ,rhs1))

(defmacro rhs-pref (rhs1) "The prefered bindings on the RHS of a
preference rule."  `(third ,rhs1))

(defmacro rhs-def (rhs1) "The defered (not prefered) bindings on the
RHS of a preference rule."  `(fourth ,rhs1))

;There is a nasty trick in the routine.  It checks the arity of
;bindings against the operator argument in the current-op meta-
;predicate.  This is yuky because CURRENT-OP is not defined as
;part of the prodigy syntax, but as a feature of the system.  The
;syntax checking mechanism should have not knowledge of what the
;meta predicates are or their arguments.

;This function has been fixed a couple of times and so is pretty
;messy.  It probably should be re-written.

(defun check-scr-rhs (rhs rule-type rule-nm)
  "Checks syntax and arity of many parts of the RHS of a SCR.
I also verifies that node and operator SCR rules only have atoms
as args and that binding and goal rules have vars or lists as args."

  (cond ((and (eq rule-type 'preference-rule)
	      (= 4 (length rhs))
	      (rhs-type-p rhs 'prefer) 	 ;The case checks the arguments to the RHS of control 	 ;rules.  	 ; For bindings and goals they must be	      lists or vars.
					; For operators or nodes they must be atoms or vars.
	      (case (rhs-level rhs)
		(bindings
		 (and (or (listp (rhs-pref rhs))
			  (is-variable (rhs-pref rhs)))
		      (or (listp (rhs-def rhs))
			  (is-variable (rhs-def rhs)))
				 ;Here we do arity check against the op chosen
				 ; in the rule.
		      (let ((op (third (get-pred-from-exp 'current-op (scr-lhs rule-nm)))))
			(or (null op)
			    (= (length (rhs-pref rhs))
			       (length (rhs-def rhs))
			       (length (op-params op)))))))
		(goal
		 (and (or (listp (rhs-pref rhs))
			  (is-variable (rhs-pref rhs)))
		      (or (listp (rhs-def rhs))
			  (is-variable (rhs-def rhs)))))
		((op operator node)
		 (and (atom (rhs-pref rhs))
		       (atom (rhs-def rhs))))))

	 nil)
	((and (eq rule-type 'select-rule)
	      (rhs-type-p rhs 'select)
	      (= 3 (length rhs))
	      (case (rhs-level rhs)
		  (bindings
		    (and (or (listp (rhs-arg rhs))
			     (is-variable (rhs-arg rhs)))
		         (let ((op (caddr (get-pred-from-exp 
					  'current-op 
					  (scr-lhs rule-nm)))))
			    (or (null op)
			        (= (length (rhs-arg rhs))
				   (length (op-params op)))))))
		  (goal (or (listp (rhs-arg rhs))
			    (is-variable (rhs-arg rhs))))
		  ((op operator node) (atom (rhs-arg rhs)))))
	 nil)
;; I fixed the reject bindings so that it doesn't try an arity
;; check when current-op is used to generate a binding for the op.
;; In this case it really should look for a candidate-binding mete-
;; predicate and use it to do the arity check, but I didn't add that.
;; --dkahn
	((and (eq rule-type 'reject-rule)
	      (rhs-type-p rhs 'reject)
	      (= 3 (length rhs))
	      (case (rhs-level rhs)
		(bindings
		  (and (or (listp (rhs-arg rhs))
			   (is-variable (rhs-arg rhs)))
		       (let ((op (caddr (get-pred-from-exp
					  'current-op
					  (scr-lhs rule-nm)))))
			 (or (null op)
			     (is-variable op)
			     (=  (length (rhs-arg rhs))
				 (length (op-params op)))))))
		(goal (or (listp (rhs-arg rhs))
			  (is-variable (rhs-arg rhs))))
		((op operator node) (atom (rhs-arg rhs)))))
	 nil)
	(t 'found-error)))

; ----------------------------------------------------------------------
  
(defun check-scr-exp (exp)
  (cond ((eq (car exp) 'forall)
	 (cond ((not (or (equal 4 (length exp)) (equal 3 (length exp))))
		(list "Expression is wrong length" exp))
	       ((check-meta-pred (get-gen-exp exp)))
	       (t (check-scr-exp (get-exp exp)))))
	((eq (car exp) 'exists)	       
	 (cond ((not (or (equal 4 (length exp)) (equal 3 (length exp))))
		(list "Expression is wrong length" exp))
	       ((check-meta-pred (get-gen-exp exp)))
	       (t (check-scr-exp (get-exp exp)))))
	((eq (car exp) '~)
	 (cond ((equal (caadr exp) 'exists)
		(check-scr-exp (cadr exp)))
	       (t (check-meta-pred (cadr exp)))))
	((eq (car exp) 'and)
	 (cond ((< (length exp) 3)		 
		(list "Not enough conjuncts in expression" exp))
	       (t (car (mapcan #'(lambda (x) (let ((val (check-scr-exp x)))
					       (cond (val (list val))
						     (t nil))))
			       (cdr exp))))))
	((eq (car exp) 'or)
	 (cond ((< (length exp) 3)
		(list "Not enough disjuncts in expression" exp))
	       (t (car (mapcan #'(lambda (x) (let ((val (check-scr-exp x)))
					       (cond (val (list val))
						     (t nil))))
			       (cdr exp))))))
	((null exp) nil)
	(t  (check-meta-pred exp))))

; ----------------------------------------------------------------------

(defun check-meta-pred (exp)
  (cond ((not (member (car exp) *META-FUNCTIONS*))
	 (terpri) 
	 (format t "~% WARNING: Undefined META-FUNCTION ``~A'' in control ~
	 rule."  (car exp))))
  (cond ((member (car exp) '(known provable achievable))
	 (check-exp-syntax (caddr exp)))))

;;;; more here later
;-----------------------------------------------------------------------

; These routines provide some checking for illegal meta functions in 
; control rules.  Check-for-illegal-generators calls a checking function 
; that is appropriate to the type of rule being examined.

; The checking function tests each of the meta-fns in the lhs of the rule
; to see if is a member of the set of illegal meta-fns for that type
; of rule.  If it is an error message is added to the variable error-output.
; In this way all of the meta-fns are checked with one call to the proper
; checking funcition.  The variable error-output is returned and the 
; message is printed by check-scr-rule-syntax.

; Of course a user could write a meta function that will be as bad as the
; the ones we check for, but remain unchecked because the name of the 
; function is different.  too bad...

(defun check-for-illegal-generators (rule-nm)

	(case (rule-operand-type rule-nm)
	      ('bindings (do-binding-check rule-nm))
	      ('operator (do-operator-check rule-nm))
	      ('goal     (do-goal-check rule-nm))
	      (t nil))
)

; does the binding rule have a candidate-op, candidate-goal
; or a candidate-node meta-fn in it.  Then it is bad.
; If the binding rule is a select rule with a 
; candidate-bindings meta-fn then it too is bad.

(defun do-binding-check (rule-nm)
  (let ((error-output nil))
    (dolist (meta-fn-nm (condition-list rule-nm))
	(cond ((member-if #'(lambda (x) (eq meta-fn-nm x))
			    '(candidate-op candidate-goal 
			      candidate-node))

		(setq error-output (concatenate 'string error-output
	       		(format nil "~%WARNING:  ~A not allowed on LHS of Binding Rule ~A." meta-fn-nm rule-nm))
		))
		
	           ((and (select-rule-p rule-nm) 
                           (eq meta-fn-nm 'candidate-bindings))

			(setq error-output (concatenate 'string error-output
	    		(format nil "~%WARNING: candidate-bindings not allowed on LHS of Select Bindings Rule ~A." rule-nm))
			))

	     (t nil)
	     )
	)
    error-output
  )
)


; Does the operator rule have a candidate-goal or candidate-node meta-fn
; in it?  Then it is bad.
(defun do-operator-check (rule-nm)
  (let ((error-output nil))
    (dolist (meta-fn-nm (condition-list rule-nm) error-output)
        (when (member-if #'(lambda (x) (eq meta-fn-nm x))
                            '(candidate-goal
                             candidate-node))
             (setq error-output (concatenate 'string error-output
               (format nil "~%WARNING:  ~A not allowed on LHS of Op Rule ~A." meta-fn-nm rule-nm))
	     )
	)
    )
  )
)

; Does the operator have a candidate-node?  Then it is bad.
(defun do-goal-check (rule-nm)
    (dolist (condition (condition-list rule-nm))
       (when (eq condition 'candidate-node)
               (format nil "~%WARNING:  candidate-node not allowed on LHS of Goal Rule ~A." rule-nm))
    )
)

; rule-operand-type returns binding, operator, goal or node
(defun rule-operand-type (rule-nm)
    (second (scr-rhs rule-nm))
)

; condition-list returns a list of the names of the conditions
; (meta functions) on the lhs of the control rule.

(defun condition-list (rule-nm)
    (let ((lhs (scr-lhs rule-nm)))
	(cond ((not (eq 'and (car lhs))) (list (car lhs)))
	      (t (mapcar #'car (cdr lhs)))
	)
    )
)

;This test is need because there is the special case of bindings
;select.

(defun select-rule-p (rule-nm)
    (eq (scr-rule-type rule-nm) 'bindings-select)
)


; ======================================================================
;               Making Functions From Static Predicates
; ======================================================================


(defun unbind-static-pred-functions ()
  (dolist (p *AUTODEFINED-FUNCTION-PREDS*)
    (fmakunbound p)
    (setf *FUNCTION-PREDS* (delete p *FUNCTION-PREDS*)))
  (setq *AUTODEFINED-FUNCTION-PREDS* nil))



(defun  functionalize-static-predicates ()
  (declare (special *TRACE-TEXT-FLAG*))
  "For each static predicate in the start state,create a match function.
   The match function performs two types of test depending upon whether 
   variables are arguments to be matched: ie. the predicate is
   a test or generator.  A simple equality test is made if there are no
   variables, otherwise the bindings for the variables are generated."
  (setq *AUTODEFINED-FUNCTION-PREDS* nil)
  (let ((initial-static-pred-values  (make-hash-table :test #'equal)))
    ;;
    ;; get static pred values from start state. 
    ;;
    (dolist (p (sort (copy-list *START-STATE*) 'alphalessp-f))
      (cond ((member (car p) *STATIC-PREDS*)
	     (if (some #'(lambda (x) (is-variable x)) p)
		 (format t " WARNING: the predicate ~A contains a variable." p))
	     (push (cdr p) (gethash (car p) initial-static-pred-values))
	     (pushnew (car p) *AUTODEFINED-FUNCTION-PREDS*))))
    ;;
    ;; create a match function for each static predicate.
    ;;
    (maphash 
     #'(lambda (name values)
	 ;;;(format t "~% NOTE: Defining a static pred function for ~A." name)
	 (eval
	  `(defun ,name (&rest terms)
	     ;;;(format t "Evaluating function ~A" ',name)
	     (let ((values ',values)  (bindings nil) (binding nil))
	       (if (notany #'(lambda (x) (is-variable x)) terms)
		   (loop (let ((value (pop values)))
			   (cond ((equal value terms) (return t))
				 ((null values) (return nil)))))
		   (dolist (value values bindings)
		     (if (every #'equal-terms value terms)
			 (and (setq binding (delete nil (mapcar #'term-binding
							       value terms)))
			      (push binding bindings))))))))
         (if (not (compiled-function-p (symbol-function name)))
	     (compile name))
         )
     initial-static-pred-values)
    (setq *FUNCTION-PREDS* 
	  (delete-duplicates 
	   (append *FUNCTION-PREDS* *AUTODEFINED-FUNCTION-PREDS*)))
    (when *TRACE-TEXT-FLAG*
      (format t "~2% NOTE: The following static predicates were defined as ")
      (format t "functions: ~% ~A" *AUTODEFINED-FUNCTION-PREDS*))))



(defun equal-terms (t1 t2)
  "determine whether two terms are equal.  If either
   term is a variable then it is it is bound to the other term."
  (cond ((and (is-variable t1) (is-variable t2)) nil)
	((is-variable t1))
	((is-variable t2))
	((equal t1 t2))))



(defun term-binding (t1 t2)
  (cond ((and (is-variable t1) (not (is-variable t2))) (list t1 t2))
	((and (is-variable t2) (not (is-variable t1))) (list t2 t1))))



(defun delete-fsps-from-start-state ()
  (let ((new-start-state nil))
    (dolist (p *START-STATE* (nreverse new-start-state))
      (unless (member (car p) *AUTODEFINED-FUNCTION-PREDS*)
	(push p new-start-state)))))


(defun find-static/initial-state-preds ()
  "This function categorizes static and non-static initial state predicates
  for use with the-static-command and the-initial-command (see commands.lisp)"
  (setq *STATIC-STATE-PREDS* nil
	*INITIAL-STATE-PREDS* nil)
  (dolist (p *START-STATE*)
    (if (member (car p) *STATIC-PREDS*)
      (push p *STATIC-STATE-PREDS*)
      (push p *INITIAL-STATE-PREDS*))))

   
; ======================================================================
;                     Determining the type of a function. 
; ======================================================================

(defun lisp-function-p (symbol)
  (and (symbolp symbol)
       (fboundp symbol) 
       (eq (symbol-package symbol)
	   (find-package "LISP"))))

; ======================================================================
; Code to reset domain.  It should be called at the beginning of the domain
; startup file.
; ======================================================================


(defun reset-last-domain ()
        (declare (special; *OP-NAMES* *INFER-NAMES* *SCR-RULES* *DOMAIN-META-FNS*
			 *STANDARD-META-FNS*))
; Reset all operators

   (dolist (op (mapcar #'car *OPERATORS*))
	(remprop op 'operator)
	(remprop op 'preconds)
	(remprop op 'effects)
	(remprop op 'all-vars)
	(remprop op 'all-vars-in-effects)
	(remprop op 'wildcard-vars)
	(remprop op 'del-list)
	(remprop op 'vars)
	(remprop op 'params)
;	(remprop op 'justifs)
	(remprop op 'add-list)
	(remprop op 'dels-for-matching)
	(remprop op 'del-list)
	(remprop op 'lpreconds)
	(remprop op 'conj-lists)
   )

; Reset all inference rules

   (dolist (infer (mapcar #'car *INFERENCE-RULES*))
	(remprop infer 'inference-rule)
	(remprop infer 'preconds)
	(remprop infer 'effects)
	(remprop infer 'all-vars)
	(remprop infer 'all-vars-in-effects)
	(remprop infer 'wild-card-vars)
	(remprop infer 'vars)
	(remprop infer 'params)
;	(remprop infer 'justifs)
	(remprop infer 'add-list)
	(remprop infer 'dels-for-matching)
	(remprop infer 'del-list)
	(remprop infer 'lpreconds)
	(remprop infer 'conj-lists)
   )

; Reset control rules

   (dolist (scr (mapcar #'car (append
    		    *SCR-NODE-SELECT-RULES*  *SCR-GOAL-SELECT-RULES*  
  	     	    *SCR-OP-SELECT-RULES* *SCR-BINDINGS-SELECT-RULES*
	       	    *SCR-NODE-REJECT-RULES*  *SCR-GOAL-REJECT-RULES* 
		    *SCR-OP-REJECT-RULES* *SCR-BINDINGS-REJECT-RULES*
   	            *SCR-NODE-PREFERENCE-RULES* *SCR-GOAL-PREFERENCE-RULES* 
	     	    *SCR-OP-PREFERENCE-RULES* *SCR-BINDINGS-PREFERENCE-RULES*)))

	(remprop scr 'rule-type)
	(remprop scr 'lhs)
	(remprop scr 'rhs)
	(remprop scr 'priority)
	(remprop scr 'match-time)
   )

; unbind the extra functions, not the standard meta-preds!!
 (makunbound-func-list (remove-if-not #'fboundp
              (set-difference *FUNCTION-PREDS* *STANDARD-META-FNS*)))

; Now reset the list of meta-preds
  (setf *META-FUNCTIONS* (copy-list *STANDARD-META-FNS*))

; Reset all the control rules in case Oren forgets to.

(setf  *SCR-NODE-SELECT-RULES* nil
       *SCR-GOAL-SELECT-RULES* nil
       *SCR-OP-SELECT-RULES* nil
       *SCR-BINDINGS-SELECT-RULES* nil
       *SCR-NODE-REJECT-RULES* nil
       *SCR-GOAL-REJECT-RULES* nil
       *SCR-OP-REJECT-RULES* nil
       *SCR-BINDINGS-REJECT-RULES* nil
       *SCR-NODE-PREFERENCE-RULES* nil
       *SCR-GOAL-PREFERENCE-RULES* nil
       *SCR-OP-PREFERENCE-RULES* nil
       *SCR-BINDINGS-PREFERENCE-RULES* nil)

)


(defun makunbound-func-list (function-list)
   "Unbinds all functions that are in the function-list and the USER package"
   (let ((package (find-package "USER")))
     (declare (package package))
   (dolist (func function-list)
	(declare (symbol func))
; Only unbind functions in the USER package, we don't want to lose lisp!!
        (if (eq package (symbol-package func))
            (fmakunbound func)))))

(defun fix-auto-scntrl-rules ()
    (declare (special *AUTO-BREADTH-FIRST-FLAG* *AUTO-BREADTH-FIRST*
		      *AUTO-DFID-FLAG* *AUTO-DEPTH-FIRST-INTERATIVE-DEEPENING*
		      *TRACE-TEXT-FLAG*))
    (cond (*AUTO-BREADTH-FIRST-FLAG*
	   (apply 'load-new-scntrl-rule *AUTO-BREADTH-FIRST*)
           (if *TRACE-TEXT-FLAG*
	    (format t "~%  NOTE:  Breadth first search control rule loaded.")))
	  (*AUTO-DFID-FLAG*
	   (apply 'load-new-scntrl-rule *AUTO-DEPTH-FIRST-INTERATIVE-DEEPENING*)
	   (if *TRACE-TEXT-FLAG*
	    (format t "~%  NOTE:  Depth first iterative deepening turned on.")))
	  (t nil)))
; ======================================================================
;                    End  of  load-domain.lisp
; ======================================================================
