#|
*******************************************************************************
PRODIGY Version 2.0  
Copyright 1989 by Steven Minton, Craig Knoblock, Dan Kuokka and Jaime Carbonell

The PRODIGY System was designed and built by Steven Minton, Craig Knoblock,
Dan Kuokka and Jaime Carbonell.  Additional contributors include Henrik Nordin,
Yolanda Gil, Manuela Veloso, Robert Joseph, Santiago Rementeria, Alicia Perez, 
Ellen Riloff, Michael Miller, and Dan Kahn.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#



(proclaim '(special *ALL-NODES* *DOMAIN-GRAPHICS* *PRINT-ALTS*  
                    *NODE-PREF-GRAPH* *ACTIVATE-EBL* *PROB-NM*
		    *FAILURE-RECORD* *NODE-NUM* *INCR-LEARNING* 
		    *TRUIFY-RELEVANCE-TABLE* *FALSIFY-RELEVANCE-TABLE* 
		    *TIME-COUNTER* *STOP-TIME* *START-TIME* *INFERENCES-ONLY*
		    *START-STATE* *STATE-NUM* *SUB-STATE-NUM* *WHAT-IF-ADVICE*
		    *TRACING-TURNED-ON* *PRINT-TRACING* *TRACE-TEXT-FLAG* 
		    *EXPLAIN-MATCH-FAILURES* *MATCHER-TRACE* 
		    *MATCH-EXPLANATION* *DOMAIN-NODE* *WM-FLAG*
		    *CLOSED-PREDS* *HALT-PROBLEM-SOLVER* *EBL-FLAG* 
		    *DOMAIN-STATE* *NODE-CUTOFF* *TAKE-ADVICE*
		    *PRODIGY-TIME-BOUND* *ALLOW-REPEATED-STATES* 
                    *ALLOW-ACHIEVED-GOALS* *MAX-TIME* *ABSTRACTION-LEVEL*))

(eval-when (compile) 
  (load-path *PLANNER-PATH* "g-loop")
  (load-path *PLANNER-PATH* "g-map")
  (load-path *PLANNER-PATH* "data-types"))



; This file contains the INFERENCE ENGINE for the PRODIGY problem solver.

; Notes: Operators and inference rules look pretty much the same to PRODIGY,
; at least as far as this file is concerned. For the most part,
; we will simply use the term "operator" as a generic term.
; Therefore, when we
; refer to an operator, we generally mean an operator or inference rule.


; Main Function. Initializes globals and calls CNTRL.

(defun run ()
  (declare (special *OP-TRACING* *SCR-TRACING* *VAR-COUNTER* *GC-BEFORE-RUN*)
	   (fixnum *VAR-COUNTER*))
  (if (and (boundp '*ACTIVATE-EBL*) *ACTIVATE-EBL*)
      (if *EBL-FLAG*
	  (reset-obs)
	(error "ERROR: *EBL-FLAG* is NIL.  Reload EBL system")))
  (if (and *CURRENT-PROBLEM-SET* (boundp '*PROB-NM*)) 
		(format t "PROBLEM: ~a" *PROB-NM*))
  (reclaim *GC-BEFORE-RUN*)
  (print-initial-info)
  (setq *START-TIME* (get-internal-run-time))
  (setf *MAX-TIME* (+ (* internal-time-units-per-second
			 *PRODIGY-TIME-BOUND*)
		      *START-TIME*))
  (setq *EXPLAIN-MATCH-FAILURES* nil *MATCH-EXPLANATION* nil)
  ; For explanation
  (setq *HALT-PROBLEM-SOLVER* nil)
  (setq *NODE-PREF-GRAPH* nil)
  (setq *ALL-NODES* nil)
  (setf *VAR-COUNTER* 0)
  (setq *NODE-NUM* 0)
  (setq *STATE-NUM* 0)
  (setq *SUB-STATE-NUM* 0)
  (setq *MATCHER-TRACE* nil)
  (setq *TRACING-TURNED-ON* (or *OP-TRACING* *SCR-TRACING*))
  (setq *INFERENCES-ONLY* nil) 
  ; for search control rules, tested in matcher
  (let ((final-node (cntrl (initialize-start-node) '(done))))
    (cond (final-node (print-success))
	  (t (print-failure)))
    (setq *STOP-TIME* (get-internal-run-time))
    (if (null *ABSTRACTION-LEVEL*)(print-final-stats final-node))
    (if (and (boundp '*ACTIVATE-EBL*) *ACTIVATE-EBL*)
	(invoke-learning))
    final-node))
 

(defun initialize-start-node ()
  (let ((start-node (get-new-node))
	(state-nm (make-new-state-nm))
	(sorted-state (sort (copy-list *START-STATE*) 'alphalessp-f)))
    (and *TRACING-TURNED-ON* (push start-node *MATCHER-TRACE*))
    (setq *DOMAIN-STATE* state-nm)               ; for graphics interface
    (setq *DOMAIN-NODE* start-node)
    (setf (node-generating-alt start-node)
	  (make-alternative :goal 'start-goal 
			    :op 'start-op 
			    :unmatched-conds '((done)) 
			    :failed-cond nil 
			    :vars nil))
    (set-closed-world state-nm sorted-state)
    (set-true-assertions state-nm sorted-state)
    (setf (node-state start-node) state-nm)
    (expand-node start-node)
    start-node))


; Top level loop that picks a node, prints it out, and expands a child
; node.  If there are no nodes left then there is not a solution.
; The problem is solved if the goal literal is true
; at a node. If a solution is found, this node is returned, else nil.
; Note: Prodigy is run recursively by calling CNTRL from RECUR-RUN 
; (See the file containing the search control meta-level functions).

(defun cntrl (start-node goal)
  (declare (special *DISCARD-FAILURE-BRANCHES*))
  (g-loop (init parent (node-parent start-node)  current-alt nil
		child nil nodes (list start-node))
	  (while (multiple-value-setq (parent nodes)
		       (scr-choose-nodes nodes)))
	  (do 
	    (setq *TIME-COUNTER* (get-internal-run-time))
	    (cond ((< *MAX-TIME* *TIME-COUNTER*)
		   (format t "~%~%*PRODIGY-TIME-BOUND* exceeded!")
		   (setq *HALT-PROBLEM-SOLVER* t)))
	    (cond (*TRACING-TURNED-ON*
		   (if (eq (node-name parent) *PRINT-TRACING*)
		       (setq *PRINT-TRACING* t))
		   (push parent *MATCHER-TRACE*)))
	    (cond ((not (node-expanded parent))
		   (expand-node parent)))
	    (setq current-alt (pop-best-alt parent))
	    (if (node-alternatives parent) (push parent nodes))
	    (cond (current-alt
		   (setq child (generate-child-node parent current-alt))
		   (print-node parent *PRINT-ALTS* nil)
		   (cond ((not (node-failure-reason child))
			  (push child nodes))
			 (t  (print-node child *PRINT-ALTS* nil))))
		  (t (print-node parent *PRINT-ALTS* nil)))

;  If true first see if child failed (because of repeat state) and then
;  see if parent failed (because of no alts to expand).
	    (if  *DISCARD-FAILURE-BRANCHES*
		(cond ((node-failure-reason child)
			 (remove-nodes-until-backtrack child))
		      ((node-failure-reason parent)
			 (remove-nodes-until-backtrack child))))
	    (cond (*EBL-FLAG*
		   (record-node-time parent *TIME-COUNTER* 
				     (get-internal-run-time))
		   (if *INCR-LEARNING*
		       (setq nodes (incremental-ebl-learning 
					  parent child current-alt nodes)))))
	    (and *WM-FLAG* 
		 (applied-operator child)
		 (contrast-WM-state child))
	    (check-for-interrupt parent))
	  (until (or *HALT-PROBLEM-SOLVER*
		     (null child) ; No alts to expand
		     (cond ((eq (node-name child) *NODE-CUTOFF*)
			    (print-node-cutoff-reached)
			    t))
		     (goal-achieved goal child)))
	  (result (cond ((and child 
			      (null (node-failure-reason child))
			      (goal-achieved goal child))
			 (setf (node-succeeded child) t)
			 (reorder-nodes child nil)
			 child)
			(t nil)))))

(defun goal-achieved (goal child) 
  (cond ((eq '~ (car goal))
	 (not (member (cadr goal)
		      (state-true-assertions (node-state child))
		      :test #'equal)))
	(t (member goal (state-true-assertions (node-state child))
		   :test #'equal))))

;;; Used to reorder the solution after performing a best-first search.
(defun reorder-nodes (node prev-node)
  (cond ((null node) nil)
	((null prev-node)(reorder-nodes (node-parent node) node))
	((eq prev-node (car (node-children node)))
	 (reorder-nodes (node-parent node) node))
	(t (setf (node-children node)
		 (cons prev-node (delete prev-node (node-children node))))
	   (reorder-nodes (node-parent node) node))))


(defun record-node-time (n start stop)
  (let ((time (node-time n)))
    (cond ((null time)
	   (setf (node-time n) (- stop start)))
	  (t (setf (node-time n) (+ time (- stop start)))))))

 

; Checks to see if the interrupt key has been pressed and if so it
; calls the interaction facility with the current node.
 
(defun check-for-interrupt (node)
  (declare (type node node))
  "Check for a 'q or 'a character at the input.  Quit on 'q and enter
   analyze on 'a."
  (let ((input (check-for-interesting-input)))
    (cond ((null input) nil)
	  ((eq input 'analyze)
	   (clear-input)
	   (terpri) (terpri) (terpri)
	   (let ((analyze-result (analyze node "the problem solver")))
	     (cond ((eq analyze-result 'STOP)
		    (setq *HALT-PROBLEM-SOLVER* t))
		   (*DOMAIN-GRAPHICS*
		    (refresh-domain-graphics node)))))
	  ((eq input 'quit)
	   (setq *HALT-PROBLEM-SOLVER* t)))))

(defun check-for-interesting-input ()
  (let ((char (read-char-no-hang t nil)))
    (cond ((null char) nil)
	  ((member char '(#\a #\A)) 'analyze)
	  ((member char '(#\q #\Q)) 'quit)
	  (t (check-for-interesting-input)))))


; Finds alternatives to try by expanding possible operators.  If reset-alt
; is set then it means that an operator was
; just applied and now it will work on the alt-on-deck.  
 

(defun expand-node (node)
  (declare (type node node))
  (setf (node-expanded node) t)
  (cond ((node-reset-alt node)
	 (setf (node-alternatives node) (default-order (reset-alts node))))
	(t (setf (node-alternatives node)
		 (default-order (get-node-alts node)))))
  (if (null (node-alternatives node))
      (setf (node-failure-reason node) '(no-relevant-alts))))
 

; moves alts with no unmatched-conds to the front
; should change this, since we reorder anyway in reset-alts


(defun default-order (alts)
  (g-loop (init firsts nil lasts nil alt nil)
	  (while (setq alt (pop alts)))
	  (do (cond ((null (alt-unmatched-conds alt))
		     (push alt firsts))
		    (t (push alt lasts))))
	  (result (nconc (nreverse firsts) (nreverse lasts)))))
    

; Reset-alt will contain an alternative originally tried at an ancestor node.
; At this point, we are retrying the node since we have achieved one
; of the subgoals.
; To reset the alt, we must first update the bindings, and
; this results in a set of new alts (ie. because we further instantiate
; the alt).
 

(defun reset-alts (node) 
  (declare (type node node))
  (prog (old-alt op goal bindings)
	(setq old-alt (node-reset-alt node))
	(setq op (alt-op old-alt))
	(setq goal (alt-goal old-alt))
	(setq bindings 
	      (make-binding-list (op-vars op) (alt-vars old-alt)))
	(return (make-alts bindings (alt-post-cond old-alt) op goal node))))


; Generates all the alts for a node.
; For each of the goals at the node, we find all the alts.
; Note that the set of candidate goals is trimmed by calling
; the search-control mechanism.

(defun get-node-alts (node)
  (declare (type node node))
  (g-loop (init goals (reverse 
		       (scr-choose-goals 
			(alt-unmatched-conds (node-generating-alt node)) 
			node))
		node-alts nil)
	  (while goals)
	  (do (setq node-alts (nconc (get-goal-alts (car goals) node)
				     node-alts)))
	  (next goals (cdr goals))
	  (result node-alts)))

; Generates all the alts for a goal (at a node).
; For each of the operators relevent to that goal, we find all the alts.
; Note that the set of candidate operators is trimmed by calling
; the search-control mechanism.

(defun get-goal-alts (goal node)
  (g-loop (init ops (scr-choose-ops (calc-poss-ops goal) goal node)
		goal-alts nil)
	  (while ops)
	  (do (setq goal-alts (nconc (get-op-alts (car ops) goal node)
				     goal-alts)))
	  (next ops (cdr ops))
	  (result goal-alts)))

		
; Returns the operators relevent to the goal.

(defun calc-poss-ops (goal)
  (cond ((negated-p goal)
	 (if (not *EBL-FLAG*) 	;  with ebl, cant solve negated goals
             (cdr (assoc (caadr goal) *FALSIFY-RELEVANCE-TABLE*))))
	((cdr (assoc (car goal) *TRUIFY-RELEVANCE-TABLE*)))))



; Calculates candidate sets of bindings for the right hand side (effects) of
; an operator or inference rule (such that the current goal is matched).
; Calls make-alts to match the left hand side (preconditions), and
; generate the appropriate alts.
; Note that the search-control mechanism is called to select (ie. generate)
; candidate bindings before we call make-alts.
;
; First finds the post conditions of an operator that achieves a
; particular goal (there may be more than one) and then finds the
; bindings by matching each post condition against the goal.  Next
; it calls generate-op-alts which runs the binding select rules to
; generate all the alts.  If there are none, then the semantics of
; selection rules dictate that we return the original set, thus the
; set of alts are created simply from the rhs-bindings.
;
(defun get-op-alts (op goal node)
  (let ((post-conditions (cond ((and (negated-p goal)
				     (closed-predicate (cadr goal)))
				(op-dels-for-matching op))
			       (t (op-add-list op)))))
    (cond ((generate-op-alts op goal node post-conditions))
	  (t (make-rhs-alts post-conditions op goal node)))))


;
;
; Recurs throught the set of post-conditions and corresponding
; bindings, runs the select binding rules and creates the alts from
; the selected bindings.
;
(defun generate-op-alts (op goal node post-conditions)
  (cond ((null post-conditions) nil)
	(t (let ((rhs-bindings (find-rhs-bindings 
				(car post-conditions) goal)))
	     (cond ((null rhs-bindings)
		    (generate-op-alts op goal node (cdr post-conditions)))
		   (t (nconc (make-selected-alts 
			      (nreverse 
			       	(scr-generate-bindings rhs-bindings op 
						       goal node))
			      (car post-conditions) op goal node)
			     (generate-op-alts op goal node 
					       (cdr post-conditions)))))))))


;
;
; Creates the set of alts from the post-conditions and their
; corresponding bindings.
;
(defun make-rhs-alts (post-conditions op goal node)
    (cond ((null post-conditions) nil)
	  (t (let ((rhs-bindings (find-rhs-bindings (car post-conditions) 
						    goal)))
	       (cond ((null rhs-bindings)
		      (make-rhs-alts (cdr post-conditions) op goal node))
		     (t (nconc (make-alts rhs-bindings (car post-conditions) 
					  op goal node)
			       (make-rhs-alts (cdr post-conditions) 
					      op goal node))))))))

;
;
; Creates the set of alts for a set of binding lists.
;
(defun make-selected-alts (binding-lsts post-condition op goal node)
  (cond ((null binding-lsts) nil)
	(t (nconc (make-alts (car binding-lsts) post-condition op goal node)
		  (make-selected-alts (cdr binding-lsts)
				      post-condition op goal node)))))
;
;
; Takes a post-condition and goal and returns the bindings.
;
(defun find-rhs-bindings (post-condition goal)
  (cond ((null post-condition) nil)
	((eq 'if (car post-condition))
	 (rhs-match (caddr post-condition) goal))
	(t (rhs-match post-condition goal))))


; MAKE-ALTS builds the alts for a partly instantiated operator (or
; inference rule).
; Here's where we call the matcher to find the appropriate bindings
; given the preconditions of the operator. Afterwards,
; the search control mechanism is called to trim this set
; of candidate-bindings.


(defun make-alts (partial-bindings post op goal node)
  (declare (type node node))
  (let ((preconds (op-preconds op)) 
	(match-results nil)
	(match-failure-record nil))
    (cond ((eq (car post) 'if) ; append test for backchaining
	   (setq preconds (list 'and preconds (cadr post)))))
    (cond (*EBL-FLAG*
	   (setq *MATCH-EXPLANATION* nil)
	   (setq *EXPLAIN-MATCH-FAILURES* t)))
    (setq match-results
	  (exp-match preconds partial-bindings (node-state node)))
    (setq match-failure-record *FAILURE-RECORD*)
    (cond (*EBL-FLAG*
	   (setf (node-history-for-ebl node)
		 (cons (list op partial-bindings goal post 
			     *MATCH-EXPLANATION*)
		       (node-history-for-ebl node)))
	   (setq *MATCH-EXPLANATION* nil)
	   (setq *EXPLAIN-MATCH-FAILURES* nil)))
    (cond (match-results 
	   (nconc (scr-reject-bindings op goal node
				       (mapcar #'(lambda (bindings)
						   (make-alternative 
						    :goal goal 
						    :op op
						    :unmatched-conds nil 
						    :failed-cond nil 
						    :vars (subst-bindings 
							   (op-vars op)
							   bindings)
						    :post-cond post))
					       match-results))
		  (scr-reject-bindings op goal node
				       (get-failed-alts goal op post node 
							match-failure-record))))
	  (t (scr-reject-bindings op goal node 
				  (get-failed-alts goal op post node
						   match-failure-record))))))
 


; If the preconditions of an operator failed to match, this is called by
; make-alts to return the appropriate alts. We look at the failure-record
; generated by the matcher to order to find the bindings that were
; attempted. Note that unmatched conditions are returned as part
; of each alt, but we're not guaranteed to get ALL unmatched conditions
; of the partially instantiated preconditions that the matcher was working
; on. For each alt, you're just guaranteed to get the first unmatched
; condition. All others are extras that are found by FIND-UNMATCHED-LITS.
; By the way, if we can determine that a set of unmatched-condions are
; unachievable, we don't bother making an alt out of them.

; Returns a set of alts.

(defun get-failed-alts (goal op post-cond node match-failure-record)
  (g-loop (init unmatched-conds nil new-alts nil num nil conjs nil
		failed-exp nil failed-bindings nil
		all-variables (op-all-vars op))
	  ;; Failures will be considered in the opposite order as encountered
	  ;; in the matcher (they're reversed in match-failure-record) but
	  ;; re-reversed by pushing them onto alts. important - 
	  ;; you want to work on the alts which match the farthest!
	  ;; means that function generators have to produce reverse orders.

	  (while match-failure-record)
	  (next failed-exp (caar match-failure-record) 
		failed-bindings (cadar match-failure-record)
		match-failure-record (cdr match-failure-record))
	  ;;  have to do substition first cause of possible var-var pairs
	  (do (setq conjs (subst-bindings
			   (cadr (assoc failed-exp (op-conj-lists op)))
			   failed-bindings))
	      (setq unmatched-conds 
		    (g-map (conj in conjs)
			   (when (lit-not-true conj (node-state node)))
			   (save conj)))
	      (push (subst-bindings failed-exp failed-bindings) unmatched-conds)
	      (cond ((impossible-to-achieve-lits unmatched-conds))
		    (t (setq num (length (get-unbound-vars all-variables 
							   failed-bindings)))
		       (if (> num 0)
			   (setq unmatched-conds 
				 (subpair (get-unbound-vars 
					   all-variables failed-bindings)
					  (make-n-new-vars num)
					  unmatched-conds)))
		       (push (make-alternative 
			      :goal goal 
			      :op op
			      :unmatched-conds unmatched-conds
			      :failed-cond failed-exp
			      :vars (subst-bindings 
				     (op-vars op) failed-bindings)
			      :post-cond post-cond)
			     new-alts))))
	  (result (nreverse new-alts))))
 

		    

; used to reorder a set of alts. All alts that contain ENTRY are
; placed first in the new list. This will probably go away soon,
; after I change POP-BEST-ALT. - SNM

; entry type assumed to be op or goal (bindings never re-ordered).
(defun re-order (alts entry entry-type)
  (g-loop (init firsts nil lasts nil alt nil
		goal (cond ((eq entry-type 'op)
			    (alt-goal (car alts)))
			   ((eq entry-type 'goal) nil)
			   ((error "re-order: bad entry-type" entry-type))))
	  (while alts)
	  (until (and goal (not (equal goal (alt-goal (car alts))))))
	  (do (setq alt (pop alts))
	      (cond ((member-alt entry alt entry-type)
		     (push alt firsts))
		    (t (push alt lasts))))
	  (result (nconc (nreverse firsts) 
			 (nreverse lasts)
			 alts))))          ; hack to maintain default order

(defun member-alt (entry alt entry-type)
  (cond ((eq entry-type 'goal)
	 (equal entry (alt-goal alt)))
	((eq entry-type 'op)
	 (equal entry (alt-op alt)))
	(t (error "bad entry-type in member-alt" entry-type))))


;; POP-BEST-ALT is pretty inefficient. I'm going to change this soon so that
;; it incrementally generates alts. - SNM by setting *take-advice* to t or nil
;; will turn on or off the interactive alt selection If *take-advice* is
;; turned on, the alts are shown and numbered and the user is asked to select
;; one of these alts by entering its number.  A number not within the range of
;; the numbers of the alts shown, is considered a wrong answer and the user is
;; asked to try again.  NOTE: If you type anything different from a number (by
;; mistake or intentionally), the function turns off *take-advice* and returns
;; the first alternative shown.

(defun pop-best-alt (node)
  (declare (type node node))
  (prog (alts re-ordered best-goal best-op best-alt best-bindings)
	(setq alts (node-alternatives node))
	(cond 
	 ((and *TAKE-ADVICE* (cdr alts))
	  (setq best-alt (ask-for-best-alt node alts))
	  (setf (node-alternatives node) (delete best-alt alts))
	  (return best-alt))
	 (*WHAT-IF-ADVICE*
	  (setq *WHAT-IF-ADVICE* nil)  ;; ask for alternative only once. 
	  (when (cdr alts)
	        (setf best-alt (ask-for-best-alt node alts))
	  	(setf (node-alternatives node) (delete best-alt alts))
	  	(return best-alt))))
	(cond 
	 ((null alts)
	  (return nil)))
	(cond 
	 ((not (equal (node-last-goal node)
		      (alt-goal (car alts))))
	  (setq best-goal 
		(scr-order-goals 
		 node
		 (preserve-order-remove-dups
		  (mapcar 
		   #'(lambda (a) (alt-goal a))
		   alts))))
	  (setq re-ordered t)
	  (setf (node-last-goal node) best-goal)
	  (setq alts (re-order alts best-goal 'goal))))
	(cond 
	 ((or re-ordered (not (eq (node-last-op node)
				  (alt-op (car alts)))))
	  (setq best-op 
		(scr-order-ops 
		 (node-last-goal node) 
		 node
		 (preserve-order-remove-dups
		  (mapcan 
		   #'(lambda (a) 
		       (cond ((equal (alt-goal a)
				     (node-last-goal node))
			      (list (alt-op a)))
			     (t nil)))
		   alts))))
	  (setf (node-last-op node) best-op)
	  (setq alts (re-order alts best-op 'op))))
	(setq best-bindings 
	      (scr-order-bindings 
	       (node-last-op node) 
	       (node-last-goal node) 
	       node
	       (mapcan 
		#'(lambda (a)
		    (cond ((and (equal (alt-goal a)
				       (node-last-goal
					node))
				(equal (alt-op a) 
				       (node-last-op node)))
			   (list (alt-vars a)))
			  (t nil)))
		alts)))
	(setq best-alt 
	      (g-loop (init temp-alts alts)
		      (until (or (equal best-bindings 
					(alt-vars (car temp-alts)))
				 (and (null temp-alts) 
				      (error "no alts in pop-best-alt")))) 
; debugging
		      (next temp-alts (cdr temp-alts))
		      (result (car temp-alts))))
	(setf (node-alternatives node) (del-eq best-alt alts))
	(return best-alt)))



(defun alt-prompt-context (node alts)
  (format t "~2% You are at node  ~:(~A~)" node)
  (format t  "~% Working on goal: ~A" (node-current-goal node))
  (format t "~2% Alternative~P Remaining...~2%" (length alts))
  (let ((n 0))
    (declare (fixnum n))
    (dolist (alt alts)  (detailed-alt alt (incf n)))))



(defun ask-for-best-alt (node orig-alts)
  (let ((n 0))  (declare (fixnum n))
    (setq n (length orig-alts))
    (alt-prompt-context node orig-alts)
    (cond ((= n 1)
	   (format t "~% Selecting the only alternative...")
	   (car orig-alts))
	  (t
	   (loop 
	     (format t "~% Try which alternative: ")
	     (let ((reply (car (read-atoms))))
	       (cond ((member reply '(s p))
		      (setq *TAKE-ADVICE* nil)
		      (format t "~2%Ok, turning off advise facility.~%")
		      (return (car orig-alts)))
		     ((eq reply 'a)
		      (analyze node "the problem solver")
		      (alt-prompt-context node orig-alts))
		     ((eq reply 'q)
		      (setf *HALT-PROBLEM-SOLVER* t)
		      (return (car orig-alts)))
		     ((not (and (numberp reply) (<= 0 reply n)))
		      (format
		       t
		       "~% Enter a number between 0 and ~D, or type p to let~
		       ~% prodigy select alternatives.  Type a to enter the~
		       ~% analysis facility. Type q to return to the Prodigy:~
		       ~% prompt.  Type zero (0) to redisplay alts.~%"
		       n))
		     ((zerop reply) (alt-prompt-context node orig-alts))
		     (t (return (nth (1- reply) orig-alts))))))))))
    
    

(defun preserve-order-remove-dups (l) 
  (g-loop (init ret-val nil)
	  (while l)
	  (do (cond ((member (car l) ret-val :test #'equal))
		    ((push (car l) ret-val))))
	  (next l (cdr l))
	  (result (nreverse ret-val))))


; Generates a child node from the current-alt.
 
(defun generate-child-node (parent-node current-alt)
  (declare (type node parent-node))
  (let* ((child-node (get-new-node)))
    (declare (type node child-node))
    ; parent-child bookkeeping
    (setf (node-parent child-node) parent-node)
    (setf (node-children  parent-node)
	  (cons child-node (node-children parent-node)))
    (setf (node-generating-alt child-node) current-alt)
	  
        	  ; if null subgoals then pop goal-stack & execute operator
          	  ; otherwise continue operator subgoaling.  Checks that
            	  ; the subgoal hasn't already been achieved or is already
           	  ; in the goal stack.
    (cond ((null (alt-unmatched-conds current-alt))
	   (execute-operator current-alt parent-node child-node))
	  (t (establish-subgoals current-alt parent-node child-node)))))

; The current alt has no unmatched conditions, so the corresponding
; operator can be executed.  This is done with adjust state.  
; If we find that the state is repeating, we indicate that
; the child has failed using the property 'failure-reason.
; If one of the goals in the stack has already been achieved we
; also fail since we don't know how it was achieved.  Finally,
; we get the reset-alt from the ancestor-node which generated the 
; subgoal we just achieved.


(defun execute-operator (alt parent-node child-node)
  (declare (type node parent-node child-node))
  (setf (node-state child-node) (adjust-state alt (node-state parent-node)))
  (setf (node-goal-stack child-node) (cdr (node-goal-stack parent-node)))
  (setf (node-applied-node child-node) t)
  (if (operator-p (alt-op alt)) ;inc only if operator, not inference rule.
      (setf (node-depth child-node) (1+ (node-depth parent-node)))
      (setf (node-depth child-node) (node-depth parent-node))
  )
  (cond ((operator-p (alt-op alt))
	 (let ((repeat-state
		(check-for-repeat-world 
		 parent-node 
		 (node-state child-node)))
	       (achieved-goal
		(check-whether-goals-achieved 
		 (node-goal-stack parent-node)
		 (node-state child-node))))
	   (cond ((and repeat-state (not *ALLOW-REPEATED-STATES*))
		  (setf (node-failure-reason child-node)
			`(repeat-world ,repeat-state)))
		      ; We used to recalibrate, now we fail
		      ; note no check needed for inferences I think
		 ((and achieved-goal (not *ALLOW-ACHIEVED-GOALS*))
		  (setf (node-failure-reason child-node)
			`(goal-already-achieved ,achieved-goal)))))))
  
    ; set up clone pointers for child node, and reset-alt

  (let ((l-clone (v-pre-parent parent-node)))
    (setf (node-left-clone child-node) l-clone)
    (cond (l-clone (setf (node-right-clones l-clone)
			 (cons child-node (node-right-clones l-clone)))))
    (setf (node-right-clones parent-node)
	  (cons nil (node-right-clones parent-node))))
  (setf (node-reset-alt child-node)
	(node-generating-alt (leftmost-clone parent-node)))
  child-node)



; We check for a goal loop, then we push the new goal onto the
; goal stack.
 

(defun establish-subgoals (alt parent-node child-node)
  (declare (type node parent-node child-node))
  (let ((goal-stack (cons (alt-goal alt) (node-goal-stack parent-node)))
	repeated-goal)
    (setf (node-goal-stack child-node) goal-stack)
    (setf (node-state child-node) (node-state parent-node))
    (setf (node-depth child-node) (node-depth parent-node))
    (setq repeated-goal (check-for-goal-loop 
			 (alt-unmatched-conds alt) goal-stack))
    (cond (repeated-goal
	   (setf (node-failure-reason child-node)
		 `(goal-repeat ,repeated-goal))))
    child-node))



; Generates a new node.

(defun get-new-node ()
  (declare (special *NODE-NUM*)
	   (fixnum *NODE-NUM*))
  (incf *NODE-NUM*)
  (let ((name (intern (concatenate 'string "N" 
				   (princ-to-string *NODE-NUM*)))))
    (push (set name (make-node :name name)) *ALL-NODES*))
  (car *ALL-NODES*))
	 

; This function reclaims the space from the last problem run.  It sets
; the state symbol, the property list, and the node symbol at each node to
; NIL so that the storage can be garbage collected.
; NOTE:  The destruction is the plist is EVIL and should be removed
; buts its horribly convenient because applications on top of 
; PRODIGY might want to store info on the plist and we wouldn't want 
; to make those authors write their own plist delete code.

(defun reclaim (&optional (gc-func nil))
  (declare (special *EXPL-NODE*))
  (cond ((boundp '*ALL-NODES*)
	 (dolist (node *ALL-NODES*)
	   (delete-node node))
	 (setq *ALL-NODES* nil)
	 (setf *EXPL-NODE* nil)
	 (if gc-func (funcall gc-func)))))
  

; Remove all nodes returns true if the node was deleted and nil
; if not.  This helps when watching a trace.

(defun remove-nodes-until-backtrack (node)
    (declare (type node node) (special *ALL-NODES*))
   "Recusively removes nodes until the node has children."
    (let ((parent (node-parent node)))
       (cond ((or (node-children node); don't delete if it has children.
		  (node-alternatives node))  nil);or other posiblities.
	     (t (delete-node node)
		(setf *ALL-NODES* (delete node *ALL-NODES*))
		(setf (node-children parent) 
			(delete node (node-children parent)))
		(if (node-parent parent) ;dont delete n1
		     (remove-nodes-until-backtrack parent))
	        t)
       )
    )
)


; I think that delete node doesn't need to delete the inside information
;  just the node-name

(defun delete-node (node)
    (declare (type node node))
    "This deletes the self-reference of a node and its name and also some 
     extra info to help gc."

   ; Get rid of values for node and state names.  Note that the state
   ; still remains in the node, and will until a gc after delete-node.

     (setf (symbol-plist (node-name node)) nil)

     (set (node-name node) nil)
     (when (node-state node) ; otherwise a break in the right place
			   ; will cause an obscure error.
        (set (state-name (node-state node)) nil)
	(unintern (state-name (node-state node)))
     )

    ;The following stuff shouldn't matter if the gc works correctly
    ;but might help things along

;     (setf (node-state node) nil)

     (setf (node-parent node) nil)
     (setf (node-children node) nil)
     (setf (node-left-clone node) nil)
     (setf (node-right-clones node) '(nil))
)

(defun find-op-seq (node)
  (declare (type node node))
  "Returns the operator sequence responsible for a node."
  (nreverse (recur-find-op-seq node)))
    
(defun recur-find-op-seq (node)
  (cond ((not (node-p node)) nil)
	((node-applied-node node)
	 (cons (list (alt-op (node-generating-alt node))
		     (alt-vars (node-generating-alt node)))
	       (recur-find-op-seq (node-parent node))))
	(t (recur-find-op-seq (node-parent node)))))



(defun dfid-run (&optional (start 0) (end nil))
  (declare (integer start) (special  *DISCARD-FAILURE-BRANCHES*))
  "Performs depth first iterative depening search"
  (if (not *DISCARD-FAILURE-BRANCHES*)
      (format t "~%  WARNING: Prodigy will not discard failure branches."))
  (do ((*DEPTH-LIMIT*  start (1+ *DEPTH-LIMIT*))
       (terminate-p nil))
       (terminate-p nil)
       (declare (integer *DEPTH-LIMIT*) (special *DEPTH-LIMIT*))
       (format t "DFID:  Starting run at depth ~D~%" *DEPTH-LIMIT*)
       (setf terminate-p (or (run) end (and (numberp end)
					    (>= *DEPTH-LIMIT* end))))
  )
)


; That's all folks!!!
