#|
*******************************************************************************
PRODIGY/EBL Module Version 2.0  
Copyright 1989 by Steven Minton.

The PRODIGY/EBL module was designed and built by Steven Minton. Thanks
to Jaime Carbonell and Craig Knoblock for their helpful advice. Andy
Philips contributed to the version 2.0 modifications.

The PRODIGY system is experimental software for research purposes only.
This software is made available under the following conditions:
1) PRODIGY will only be used for internal, noncommercial research purposes.
2) The code will not be distributed to other sites without the explicit 
   permission of the designers.  PRODIGY is available by request.
3) Any bugs, bug fixes, or extensions will be forwarded to the designers. 

Send comments or requests to: prodigy@cs.cmu.edu or The PRODIGY PROJECT,
School of Computer Science, Carnegie Mellon University, Pittsburgh, PA 15213.
*******************************************************************************|#


; GEN-EX.LISP  snm A:While:Ago  ?MM-?DD-?YY created
;              abp 17:00:00  05-11-89 modified & neatened
;
; Functions within this file assist in the creation of problem sets
; for three domains in PRODIGY; BW (blocksworld), SCHED (scheduler),
; and STRIPS (strips).
;
; The functions most useful to the user are:
;   CREATE-PROBSET:	Will create and store to disk a problem set
;   GETPROB:		Will retrieve a problem by name
;   LOAD-NEXT:		Will queue up the next problem to be run
;   RUN-ALL:	        Will RUN the entire problem set
;			Note: RUN-ALL is located elsewhere.
;
;  Note:  Once created, the problem set must be loaded into memory.
;         This requires (load "filename"), where filename is the file
;         under which the problem set was stored.
(proclaim '(special *PROB-NUM* *OBJECT-NAMES* *END-TIME* *PROMPT-DATA*
            SW1 SW2 SW3 *BOLT-SIZES* *SCHED-GOAL-ATTRIBUTES* 
	    *SCHED-INIT-ATTRIBUTES*))

(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types")
	(load-path *EBL-PATH*     "ebl-data-types"))




; CREATE-PROBSET allows the user to create a set of problems for a
; particular domain.  It queries the user for important information for
; creating a problem, creates that problem, then offers to run and learn
; the problem within the PRODIGY system.  NOTE: It is important that that
; particular domain has been loaded, so that the RUN-AND-LEARN facility
; will work.
;
; It will then offer to save the problem to a file.  In this way random
; trial problems may be created and the most interesting ones can be
; saved.
; 
; The type may be BW, STRIPS, or SCHED.

(defun create-probset (type &key ((:seed seed) t))
  (setq *PROB-NUM* 0)
  (setq *TEST-PROBS* nil)
  (do* ((stop nil (null (y-or-n-p "Do you wish to continue? ")))
	(test-prob (create-probset-help type seed)
		   (if (null stop) (create-probset-help type seed)))
	(count 0 count))
       (stop nil)
       (display-problem test-prob)
       (if (y-or-n-p "Do you wish to see a run of this problem? ")
	   (progn
	     (load-goal (cadr test-prob))
	     (load-start-state (caddr test-prob))
	     (setq *PROB-NM* (car test-prob))
	     (run)
	     (terpri)))
       (if (y-or-n-p "Do you wish to keep this problem? ")
	   (progn
	     (push test-prob *TEST-PROBS*)
	     (setq count (1+ count))
	     (format t "~D problem~:P accumulated~%" count))))
  (setq *TEST-PROBS* (reverse *TEST-PROBS*))
  (if (and *TEST-PROBS*
	   (yes-or-no-p "Do you wish to save this set to a file? "))
      (with-open-file
       (ofile (get-file-name-for-prob-set) :direction :output)
       (format ofile "(setq *TEST-PROBS* '~a)" *TEST-PROBS*)
       (format t "...Data stored~%")))
  t)

#|
; GETPROB will select a problem by problem name.  It will destroy *TEST-PROBS*.

(defun getprob (p) 
    (g-loop (while *TEST-PROBS*) 
	  (do (load-next))
	  (until (equal p *PROB-NM*))))




; LOAD-NEXT will take the next problem residing in *TEST-PROBS* and queue
; it up for a problem solving session in PRODIGY.

(defun load-next ()
    (or *TEST-PROBS* (error "no more probs"))
    (setq *PROB-NM* (caar *TEST-PROBS*))
    (load-goal (cadar *TEST-PROBS*))
    (load-start-state (caddar *TEST-PROBS*))
    (cond ((assq 'last-time *START-STATE*)
	   (setq *END-TIME* (cadr (assq 'last-time *START-STATE*)))))
    (pop *TEST-PROBS*))
|#
; Functions and data used by the creation functions for IO



; Initialize some objects

(setq *OBJECT-NAMES* '(A B C D E F G H I J K L M N O P Q R S T U V W X Y Z))



; CREATE-PROBSET-HELP takes the type of problem set that is to be created
; and calls the relevant problem set generator.  The seed is for the random
; number generator and will either reseed the generator (T) or not (NIL).
; It, also, prompts the user for information needed by the generators.

(defun create-probset-help (type seed)
  (let ((messages (cdr (assoc type *PROMPT-DATA*))))
  (cond ((eql type 'bw)
	 (gen-bw (ask-num (nth 0 messages) :min 2)
		 (ask-num (nth 1 messages) :min 1)
		 t))
	((eql type 'strips)
	 (gen-strips
	  (eval (intern (format nil "SW~a" (ask-num (nth 0 messages)
						    :min 1 :max 3))))
	  (ask-num (nth 1 messages) :min 2)
	  (ask-num (nth 2 messages) :min 1)
	  t))
	((eql type 'sched)
	 (gen-sched (list (ask-num (nth 0 messages) :min 0))
		    (ask-num (nth 1 messages) :min 1)
		    (ask-num (nth 2 messages) :min 1)
		    (ask-num (nth 3 messages) :min 1)
		    t))
	(t (list '(Unknown type.  Try bw#\, sched#\, or strips)
		 nil nil)))))

(setq *PROMPT-DATA*
      '((BW
	 "Enter the maximum number of blocks: "
	 "Enter the maximum number of goals:  ")

	(STRIPS
	 "Choose a Strips World:~%~T1: SW1~%~T2: SW2~%~T3: SW3~%~%Which one? "
	 "Enter the maximum number of objects:  "
	 "Enter the maximum number of goals:  ")

	(SCHED
	 "Enter the number of time slots: "
	 "Enter the maximum number of objects: "
	 "Enter the maximum number of bolts: "
	 "Enter the maximum number of goals: ")))



; DISPLAY-PROBLEM this will display the problem for the user

(defun display-problem (tp)
  (format t "~%Problem Name: ~a~%Goal State: ~a"
	  (car tp) (cadr tp))
  (print-list "Start State:  " (caddr tp))
  (terpri))




; GET-FILE-NAME-FOR-PROB-SET will do just that.

(defun get-file-name-for-prob-set ()
  (format t "~%Please enter a filename in which to store the problem set: ")
  (string-trim '(#\SPACE #\TAB #\") (read-line)))




; ASK-NUM requires a string for output and queries the user for a number.

(defun ask-num (str &key ((:min min) nil) ((:max max) nil))
  (clear-input)
  (do* ((num nil (progn
		  (format t str)
		  (parse-integer (read-line) :junk-allowed t)))
	(less-than nil (and min num (< num min)))
	(out-of-bounds nil (or less-than (and max num (> num max)))))
       ((and (numberp num) (null out-of-bounds)) num)
       (if out-of-bounds
	   (progn
	     (format t "Enter a number that is ")
	     (if min (format t "a minimum of ~a~a" min
			     (if max " and " "")))
	     (if max (format t "a maximum of ~a" max))
	     (terpri)))))




; Utility functions used by the problem generators.



; MK-PROB-NM will create a new problem name 

(defun mk-prob-nm (nm-part)
  (intern (format nil "~a-~a" nm-part (setq *PROB-NUM* (1+ *PROB-NUM*)))))  




; GET-A-RANDOM will randomly choose a member of a list supplied.

(defun get-a-random (x)
  (if (and (listp x) x) (nth (random (length x)) x)))



; RANDOM-SEED will re-randomize the random function by
; re-initializing the *RANDOM-STATE* global variable.  Get it?

(defun random-seed (seed)
  (setq *random-state* (make-random-state seed)))



; CHECK-WORLD-PROB will examine the goals in the world problem and see
; if any are already stated in the start state.  If so, it will return
; NIL, otherwise T.

(defun check-world-prob (goal-exp start)
  (if (not (atom goal-exp))
      (do ((goals (if (eq (car goal-exp) 'and) (cdr goal-exp) (list goal-exp))
		  (cdr goals))
	   (test nil (member (car goals) start :test 'equal)))
	  ((or test (null goals)) (null test)))))



; SCRAMBLE-OBS will take a list and return the shuffled contents

(defun scramble-obs (obs)
    (g-loop (init new-obs nil guy nil)
	  (while obs)
	  (do (setq guy (get-a-random obs))
	      (setq obs (del-eq guy obs))
	      (push guy new-obs))
	  (result new-obs)))

; BW - blocksworld


; GEN-BW takes a maximum number of blocks and goals and a T/NIL value for
; random seeding, and creates a random goal and start state in the blocksworld
; domain.

(defun gen-bw (max-blocks max-goals seed)
  (random-seed seed)
  (let* ((num-blocks (+ 2 (random (1- max-blocks))))
	 (start-state (make-bw-state (make-init-bw-desc num-blocks)))
	 (goal-tmp-state (make-bw-state (make-init-bw-desc num-blocks)))
	 (goal-state (subpair (header num-blocks *OBJECT-NAMES*)
			      (scramble-obs (header num-blocks *OBJECT-NAMES*))
			      goal-tmp-state))
	 (goal-exp (make-bw-goal-exp max-goals start-state goal-state)))
    (if (check-world-prob goal-exp start-state)
	(list (mk-prob-nm 'BW) goal-exp start-state)
      (gen-bw max-blocks max-goals seed))))



; MAKE-INIT-BW-DESC will create an initial blocks world description
; given the number of blocks that are expected to be present in the
; world.  Block A is expected to always be present in any problem
; description.  Altough the resulting initial description may have had A
; removed the next function that will be called, MAKE-BW-STATE, will
; utilize A.
;
; This function will create a world description by creating lists of
; lists, with each inner list representing a pile of blocks.  A new pile
; is created with probability 1/3, then a new block is inserted in that
; pile.  If no pile was created (probability 2/3), then an existing pile
; is randomly chosen and the new block is inserted into it.

(defun make-init-bw-desc (num-blocks)
  (g-loop (init names (header num-blocks *OBJECT-NAMES*)
		init-desc nil)
	  (before-starting
	   (progn
	     (if (eq (random 3) 1) (pop names))
	     (push (list (pop names)) init-desc)))
	  (while names)
	  (do (cond ((eq (random 3) 1) (push (list (pop names)) init-desc))
		    (t (rplacd (last (get-a-random init-desc))
			       (list (pop names))))))
	  (result init-desc)))



; MAKE-BW-STATE takes the output of MAKE-INIT-BW-DESC and expands it
; into usable blocksworld data.  It passes each sublist to the helping
; function ADD-PILE.  And appends the results together to form one large
; BW description.  Note that if Block A is not included in the world
; decription, then it is assumed to be held in the arm.

(defun make-bw-state (desc)
  (apply 'append
	 (cons (if (r-memq 'A desc) '((arm-empty)) '((object A) (holding A)))
	       (mapcar #'add-pile desc))))

; MAKE-BW-GOAL-EXP takes the number of goals, the start state and the
; unrefined goal state and returns a goal expression.

(defun make-bw-goal-exp (max-goals istate state)
    (setq state 
	  (scramble-obs
	      (g-map (exp in state)
		    (when (and (null (eq (car exp) 'object))
			       (or (not (member exp istate))
				   (eq (random 3) 1))))
		    (save exp))))
    (setq state (header (1+ (random max-goals)) state))
    (cond ((null state) t)
	  ((null (cdr state)) (car state))
	  (t  (cons 'and state))))



; ADD-PILE will take a list of blocks and create a blocks world description
; that represents that list of blocks in a single stack.

(defun add-pile (blocks)
    (g-loop (init ret-val `((clear ,(car blocks))))
	  (while blocks)
	  (do (cond ((null (cdr blocks))
		     (setq ret-val
			   (append `((object ,(car blocks))
				     (on-table ,(car blocks)))
				   ret-val)))
		    ((setq ret-val
			   (append `((object ,(car blocks))
				     (on ,(car blocks) ,(cadr blocks)))
				   ret-val)))))
	  (next blocks (cdr blocks))
	  (result ret-val)))

; SW - Strips World

; There are three strips world descriptions which may be used for building
; a strips world problem set: SW1 SW2 SW3.  They are in the following format:

; SW_ = (room-names room-connects door-keys)
; room-names = list of room names
; room-connects = ((door room-name room-name) ...)
; door-keys ((door key)..)

(setq SW1 '((rm1 rm2)
	    ((dr12 rm1 rm2))
	    ((dr12 key12))))
    


(setq SW2 '((rm1 rm2 rm3 rm4)
	    ((dr12 rm1 rm2)
	     (dr23 rm2 rm3)
	     (dr34 rm3 rm4))
            ((dr12 key12))))

(setq SW3 '((den bath lroom droom kitch hall)
	    ((dr-den-bath den bath)
	     (dr-den-kitch den kitch)
	     (dr-den-droom den lroom)
	     (dr-lroom-droom lroom droom)
	     (dr-hall-kitch hall kitch)
	     (dr-kitch-droom kitch droom))
	    ((dr-den-bath bath-key)
	     (dr-den-kitch den-key))))






; GEN-STRIPS takes an empty strips world description the maximum number
; of objects allowable, the maximum number of goals allowable, and a
; seed (T/NIL).  It will fill this strips world description with objects
; and also determine a set of goals to be completed.

(defun gen-strips (sw max-objs max-goals seed)
  (random-seed seed)
  (let* ((num-objs (+ 2 (random (1- max-objs))))
	 (holding (equal 1 (random 6)))
	 (start-state (make-strips-state
		       sw (make-strips-desc sw num-objs nil) nil))
	 (goal-state  (make-strips-state
		       sw (make-strips-desc sw num-objs holding) holding))
	 (goal-exp    (make-strips-goal-exp max-goals goal-state start-state)))
	(if (and (check-world-prob goal-exp start-state)
		 (check-strips-prob sw goal-exp start-state))
	    (list (mk-prob-nm 'ST) goal-exp start-state)
	  (gen-strips sw max-objs max-goals seed))))

; MAKE-STRIPS-DESC takes the empty strips world description and does a
; number of things.  it creates objects and puts them in rooms.  it
; decides whether any objects should be next to each other.  it
; determines whether a door is open, closed, or locked.  then it returns
; the data in a list to be utilized by make-strips-state.


; obj-nms = (obj-nm obj-nms)
; dr-nms = (door...)
; obj-rms = ((obj-nm rm-nm)..)
; next-to-lsts ((obj obj)..) a obj is max next to one other obj or dr or
; robot
; rm-nms = list of room names
; rm-connects = ((dr rm-nm rm-nm) ...)
; dr-stats = ((dr status)) , where status is open, closed, or locked


(defun make-strips-desc (sw num-objs holding)
  (let* ((rm-nms (car sw))
	 (rm-connects (cadr sw))
	 (dr-keys (caddr sw))
	 (dr-nms (mapcar #'car rm-connects))
	 (obj-nms (if holding
		      (cdr (header num-objs *object-names*))
		    (header num-objs *object-names*)))
         (dr-stats nil)
	 (next-to-lsts nil)
	 (all-objects (cons 'robot (append (mapcar #'cadr dr-keys) obj-nms)))
	 (obj-rms (g-map (obj in all-objects)
			 (save (list obj (get-a-random rm-nms))))))
    (g-loop (init tmp-objs obj-nms ob nil other nil ob-rm nil)
	    (while (setq ob (pop tmp-objs)))
	    (do (cond ((and (> (random 5) 2)
			    (not (r-memq ob next-to-lsts)))
		       (setq ob-rm (cadr (assq ob obj-rms)))
		       (setq other 
			     (get-a-random 
			      (g-map (aob in obj-nms)
				     (when (eq ob-rm 
					       (cadr (assq aob obj-rms))))
				     (save aob))))
		       (and (not (r-memq other next-to-lsts))
			    (not (equal other ob))
			    (push (list ob other) next-to-lsts))))))
    (setq dr-stats (g-map (d in dr-nms)
			  (save (list d (cond ((and (> (random 3) 1)
						    (assq d dr-keys))
					       'locked)
					      ((> (random 4) 1)
					       'open)
					      (t 'closed))))))
    (list obj-nms dr-nms obj-rms next-to-lsts dr-stats)))

 
; MAKE-STRIPS-STATE takes the partial description created by
; make-strips-desc and turns it into a complete strips world state.

(defun make-strips-state (sw desc holding)
    (let ((rm-nms (car sw))
	  (rm-connects (cadr sw))
	  (dr-keys (caddr sw))
	  (obj-nms (car desc))
	  (dr-nms (cadr desc))
	  (obj-rms (caddr desc))
	  (next-to-lsts (cadddr desc))
	  (dr-stats (nth 4 desc))
	  ret-val)
	 (g-map (dk in dr-keys)
	       (do (and (push `(is-key ,(car dk) ,(cadr dk)) ret-val)
			(push `(carriable ,(cadr dk)) ret-val))))
	 (g-map (ob-rm in obj-rms)
	       (do (push `(inroom ,(car ob-rm) ,(cadr ob-rm))
			 ret-val)))
	 (g-map (ob in obj-nms)
	       (do (push (list 'is-object ob) ret-val)))
	 (g-map (entry in dr-keys)
	       (do (push (list 'is-object (cadr entry)) ret-val)))
	 (g-map (ob in obj-nms)
	       (do (and (cond ((eq (random 3) 0)
			       (push (list 'pushable ob) ret-val))
			      ((> (random 4) 1)
			       (push (list 'carriable ob) ret-val))))))			       
	 (g-map (ob in dr-nms)
	       (do (push (list 'is-door ob) ret-val)))
	 (g-map (rm in rm-nms)
	       (do (push (list 'is-room rm) ret-val)))
	 (g-map (entry in dr-stats)
	       (do (cond ((eq (cadr entry) 'open)
			  (push (list 'open (car entry)) ret-val)
			  (push (list 'unlocked (car entry)) ret-val))
			 ((eq (cadr entry) 'closed)
			  (push (list 'closed (car entry)) ret-val)
			  (push (list 'unlocked (car entry)) ret-val))
			 ((eq (cadr entry) 'locked)
			  (push (list 'locked (car entry)) ret-val)
			  (push (list 'closed (car entry)) ret-val)))))
	 (g-map (pair in next-to-lsts)
	       (do (cond ((member 'robot pair)
			  (cond ((push `(next-to robot ,(car pair))
				       ret-val))))
			 (t (and (not (member (car pair) dr-nms))
				 (push `(next-to ,(car pair) ,(cadr pair))
				       ret-val))
			    (and (not (member (cadr pair) dr-nms))
				 (push `(next-to ,(cadr pair) ,(car pair))
				       ret-val))))))
	(g-map (entry in rm-connects)
	      (do (and (push (cons 'connects entry) ret-val)
		       (push `(connects ,(car entry) ,(caddr entry) ,(cadr entry))
			     ret-val)
		       (push `(dr-to-rm ,(car entry) ,(cadr entry)) ret-val)
		       (push `(dr-to-rm ,(car entry) ,(caddr entry)) ret-val)
		  )))
	(append
	 (if holding '((holding A) (is-object A) (carriable A)) '((arm-empty)))
	 ret-val)))
 
; MAKE-STRIPS-GOAL-EXP takes the number of goals, the start state and the
; unrefined goal state and returns a goal expression.

(defun make-strips-goal-exp (max-goals state init-state)
  (let ((new-state 
	 (g-map (exp in state)
		(when (and (not (member (car exp) 
					'(is-object is-door connects 
						    pushable is-room is-key 
						    dr-to-rm carriable)))
			   (or (not (eq (car exp) 'inroom))
			       (not (member (list 'is-door (cadr exp)) 
					    state)))
			   (cond ((not (member exp init-state))
				  (> (random 6) 1)) ; 2/3 chance
				 (t (> (random 3) 1))) ; 1/3 chance
			   (cond ((eq 'next-to (car exp))
				  (or (member `(carriable ,(cadr exp))
					      init-state)
				      (member `(carriable ,(caddr exp)) 
					      init-state)
				      (member `(pushable ,(cadr exp))
					      init-state)
				      (member `(pushable ,(caddr exp))
					      init-state)))
				 ((and (eq 'inroom (car exp))
				       (null (eq 'robot (cadr exp))))
				  (or (member `(carriable ,(cadr exp)) 
					      init-state)
				      (member `(pushable ,(cadr exp))
					      init-state)))
				 (t))))
		(save exp))))
    (setq new-state (scramble-obs new-state))
    (if (> (length new-state) max-goals)
	(setq new-state (header max-goals new-state)))
    (cond ((null new-state))
	  ((null (cdr new-state)) (car new-state))
	  (t (cons 'and new-state)))))
 
; CHECK-STRIPS-PROB utilizes check-SW1 and -SW2 to determine if the
; initial configuration of the start state is impossible to solve (ie:
; robot locked in a room with no key, etc.)


(defun check-strips-prob (rm-config goal-exp istate)
  (cond ((equal rm-config SW1) (check-sw1 istate goal-exp))
	((equal rm-config SW2) (check-sw2 istate goal-exp))
	(t)))


(defun check-sw1 (istate goal-exp)
   (let ((robot-room (get-room 'robot istate))
	 (keys-room (get-room 'key12 istate)))
	(or (eq robot-room keys-room)
	    (not (member '(locked dr12) istate)))))

(defun check-sw2 (istate goal-exp)
    (let ((robot-room (get-room 'robot istate))
	  (keys-room (get-room 'key12 istate)))
	 (cond ((and (null (eq robot-room keys-room))
		     (member '(locked dr12) istate))
		(format t "~%warning robot not near keys")))
	 (cond ((or (member '(locked dr23) istate)
		    (member '(locked dr34) istate))
		(format t "~%warning, drs locked")))
	 t))





; GET-ROOM will grab a room from the state description

(defun get-room (ob state)
    (g-loop (init s nil)
	  (while (setq s (pop state)))
	  (do (and (equal 'inroom (car s))
		   (equal ob (cadr s))
		   (return (caddr s))))))
 
; SC - Scheduling World




(setq *BOLT-SIZES* '((2 mm)(4 mm)(6 mm)(8 mm)(1 cm)(1.2 cm)(1.4 cm)))


(setq *SCHED-GOAL-ATTRIBUTES*
   `((shape (CYLINDRICAL))
     (painted ((WATER-RES WHITE) (WATER-RES RED) 
	       (REGULAR RED) (REGULAR WHITE)))
     (surface-condition (POLISHED SMOOTH))
     (has-hole ,*BOLT-SIZES*
	       (ORIENTATION-1 ORIENTATION-2 ORIENTATION-3 ORIENTATION-4))
     (joined (ORIENTATION-1 ORIENTATION-2 ORIENTATION-3 ORIENTATION-4))))


(setq *SCHED-INIT-ATTRIBUTES*
   `((shape (RECTANGULAR CYLINDRICAL IRREGULAR UNDETERMINED))
     (temperature (COLD))
     (surface-condition (POLISHED SMOOTH ROUGH))
     (painted ((WATER-RES WHITE) (WATER-RES RED) 
	       (REGULAR RED) (REGULAR WHITE)))
     (has-hole ,*BOLT-SIZES*
	       (ORIENTATION-1 ORIENTATION-2 ORIENTATION-3 ORIENTATION-4))))






; GEN-SCHED takes the number of objects, bolts and goals and randomize flag
; and will create a a random start and goal state for the schedule world.


(defun gen-sched (sc num-i-objs num-bolts max-goals seed)
  (random-seed seed)  
  (let* ((i-obj-nms (header num-i-objs *OBJECT-NAMES*))
	 (start-state (make-init-sched-state i-obj-nms num-bolts))
	 (goal-exp (make-goal-sched-state i-obj-nms max-goals)))
    (push (list 'last-time (car sc)) start-state)
    (if (check-world-prob goal-exp start-state)
	(list (mk-prob-nm 'SC) goal-exp start-state)
      (gen-sched sc num-i-objs num-bolts max-goals seed))))
 
; MAKE-INIT-SCHED-STATE will create a random start state for a scheduling
; problem.  It calls MAKE-INIT-ATTS to create the initial attributes for
; items in the scheduling world.

; have initial-obs + comp-objects = final-objects
; ops = ((obj operation))

(defun make-init-sched-state (i-obj-nms num-bolts)
    (prog (i-atts)
	  (g-map (ob in i-obj-nms)
		(do (setq i-atts (make-init-atts  ob i-atts))))     
	  (g-map (ob in i-obj-nms)
		(do (push `(last-scheduled ,ob 0) i-atts)))
	  (g-loop (until (eq 0 num-bolts))
		(do (push `(is-bolt (,(concat "B" num-bolts) 
				     ,(get-a-random *BOLT-SIZES*)))
			  i-atts))
		(next num-bolts (1- num-bolts)))
	  (return i-atts)))



; MAKE-INIT-ATTS will compile a start state for the scheduling world.
; It uses the data supplied by the global SCHED-INIT-ATTRIBUTES.

; only init-objs, no composites

(defun make-init-atts (obj s-atts)
    (push `(is-object ,obj) s-atts)
    (g-loop (init att nil att-lst *SCHED-INIT-ATTRIBUTES*)
	  (while (setq att (pop att-lst)))
	  (do (cond ((member (car att) '(shape temperature))
		     (push (list (car att) obj (get-a-random (cadr att)))
			   s-atts))
		    ((null (eq 1 (random 3)))) ; 2/3 chance
		    ((push (cons (car att)
				 (cons obj 
				       (g-map (vals in (cdr att))
					     (save (get-a-random vals)))))
			   s-atts))))
	  (result s-atts)))
 
; MAKE-GOAL-SCHED-STATE will create a random goal expression for a scheduling
; problem.  It calls MAKE-GOAL-ATTS to create the initial attributes for
; items in the scheduling world.

(defun make-goal-sched-state (i-obj-nms max-goals)
    (let ((goal-exp (g-map (ob in i-obj-nms)
			  (splice (make-goal-atts ob i-obj-nms)))))
	 (g-loop (init ret-val nil)
	       (while goal-exp)
	       (do (cond ((null (eq 'joined (caar goal-exp)))
			  (push (car goal-exp) ret-val))
			 ((already-joined (cadar goal-exp)
			      (caddar goal-exp)
			      ret-val))
			 (t (push (car goal-exp) ret-val))))
	       (next goal-exp (cdr goal-exp))
	       (result (setq goal-exp ret-val)))
	 (setq goal-exp (scramble-obs goal-exp))
	 (setq goal-exp (header (1+ (random max-goals)) goal-exp))
	 (cond ((null goal-exp) nil)
	       ((null (cdr goal-exp)) (car goal-exp))
	       (t (cons 'and goal-exp)))))
		


; MAKE-GOAL-ATTS will compile a goal expression for the scheduling world.
; It uses the data supplied by the global SCHED-GOAL-ATTRIBUTES.

; all objects

(defun make-goal-atts (obj objs)
    (g-loop (init att nil att-lst *SCHED-GOAL-ATTRIBUTES* s-atts nil)
	  (while (setq att (pop att-lst)))
	  (do (cond ; ((eq (random 3) 1)) ; 1/3 chance
		    ((eq 'joined (car att))
		     (push `(joined ,obj ,(get-a-random (del-eq obj objs))
				    ,(get-a-random (cadr att)))
			   s-atts))
		    ((push (cons (car att)
				 (cons obj 
				       (g-map (vals in (cdr att))
					     (save (get-a-random vals)))))
			   s-atts))))
	  (result s-atts)))



; ALREADY-JOINED determines if two attributes have already been joined.
; It will return NIL if not and T if they have.

(defun already-joined (a b atts)
    (g-loop (init att nil)
	  (while (setq att (pop atts)))
	  (do (cond ((null (eq (car att) 'joined)))
		    ((or (equal (cadr att) a)
			 (equal (caddr att) a)
			 (equal (cadr att) b)
			 (equal (caddr att) b))
		     (return t))))))
