;;;
;;;
(proclaim '(special *ABS-PRINT-FLAG* *NONSUBGOALABLE-GENERATORS* *STATIC*))
;;;
;;; A formula is a ground level predicate that occurs as a precondition or 
;;; effect of an operator.

;;;(defstruct (formula (:print-function (lambda (formula stream level)
;;;				       (declare (ignore stream level))
;;;				       (format stream "#S~a" 
;;;                                            (formula-type formula)))))
;;;  instance    ; e.g., (in-room block1 room5)
;;;  type        ; e.g., (in-room block room)
;;;  predeladd   ; one of 'precond, 'add, 'del
;;;  status      ; one of 'primary or 'secondary
;;;  operator)   ; name of the operator

;;;=============================================================================


;;;
;;;
;;;
;;; The constraints based on the goal have already been added.  This function
;;; takes the fully constrained partial order and turns it
;;; into a total order.  Where there is a choice about which level to place
;;; before another, it prefers levels for which there are goals to be solved.
;;; When there is no preference, the choice is arbitrary.
;;;
;;;
;;; This function takes the partial order of constraints that are
;;; sufficient to guarantee the ordered monotonicity property.  The
;;; function then reduces the number of nodes in the partial order by
;;; combining certain nodes together.  This reduces the total number
;;; of levels in the abstraction hierarchy to reduce the overhead of
;;; hierarchical problem solving.  The idea is to keep only the most
;;; useful abstraction levels.  The levels are combined based on the
;;; follow criteria:
;;;
;;; 1) First literals that are generators, but can't be subgoaled on,
;;; thus there is no way to achieve them, are combined with any
;;; literals that are constrained below them in the partial order.
;;; The only exception is if any of those literals are goal literals.
;;; The motivation for this is that nonsubgoalable generators usually
;;; play a construction role in a domain and the literals below them
;;; are filters that constrain the generators.  In the scheduling
;;; domain this is how the actual scheduling is accomplished.
;;;
;;; 2) Second, literals that are not goals and are not generators are
;;; combined with other literals with the same characteristics.  Most
;;; of the action of the abstraction hierarchy comes from resolving
;;; the goal interactions, so this reduces the number of levels in
;;; which there are few interactions for the abstraction to sort out. 
;;;
;;; The levels in the hierarchy are ordered before creating the
;;; hierarchy so that when levels are combined they are combined 
;;; in order such that no loops in the hierarchy are introduced.
;;;
(defun order-hierarchy (hierarchy goal-list)
  (if *NONSUBGOALABLE-GENERATORS* 
      (combine-nonsubgoalable-generators hierarchy goal-list
					 *NONSUBGOALABLE-GENERATORS*))
  (order-levels hierarchy 
		goal-list
		(1- (array-dimension hierarchy 0))
		(find-bottom-partial-order 
		 hierarchy (1- (array-dimension hierarchy 0))))
  (create-hierarchy hierarchy 
		    goal-list
		    (find-recursive-goals *CONSTRAINTS*)
		    (1- (array-dimension hierarchy 0))
		    (find-bottom-partial-order 
		     hierarchy (1- (array-dimension hierarchy 0)))))


;;;
;;;
;;; Searches for case 1 above and then combines the appropriate groups
;;; of literals.
;;;
(defun combine-nonsubgoalable-generators (hierarchy goal-list gen-list)
  (let ((combine (find-rows-to-combine hierarchy
				       (find-goals goal-list hierarchy)
				       (find-goals *NONSUBGOALABLE-GENERATORS*
						   hierarchy))))
    (cond (combine (combine-interacting-groups (first combine)
					       (second combine)
					        hierarchy)
		   (combine-nonsubgoalable-generators hierarchy
						      goal-list
						      gen-list)))))
;;;
;;;
(defun find-rows-to-combine (hierarchy goal-rows gen-rows)
  (cond ((null gen-rows) nil)
	((member (car gen-rows) goal-rows :test #'eql)
	 (find-rows-to-combine hierarchy goal-rows (cdr gen-rows)))
	((find-row-to-combine hierarchy goal-rows (car gen-rows) 
			      (1- (array-dimension hierarchy 0))))
	(t (find-rows-to-combine hierarchy 
				 goal-rows
				 (cdr gen-rows)))))
;;;
;;;
(defun find-row-to-combine (hierarchy goal-rows gen-row index)
  (cond ((zerop index) nil)
	((and (aref hierarchy gen-row index)
	      (not (member index goal-rows :test #'eql)))
	 (list gen-row index))
	(t (find-row-to-combine hierarchy goal-rows gen-row (1- index)))))


;;;
;;;
;;;
;;; This function creates the actual total order.  If first checks to
;;; see if any of the levels on the bottom can be combined with other
;;; levels to reduce the total number of abstraction levels.  If not
;;; it selects a level to put on the bottom of the abstraction
;;; hierarchy and then recurses.
;;;
;;;
(defun create-hierarchy (hierarchy goal-list recursive-goals dimension bottom)
; To debug: (trace refine-partial-order) (setq *print-array* nil) 
;  (print-column hierarchy 0)  ;Use this for debugging
  (cond ((= dimension 1) nil)
	((combine-next-level hierarchy 
			     dimension
			     (find-goals goal-list hierarchy dimension) 
			     (find-goals recursive-goals hierarchy dimension)
			     (find-goals *NONSUBGOALABLE-GENERATORS* 
					 hierarchy dimension)
			     bottom)
	 (create-hierarchy hierarchy goal-list recursive-goals
			   (1- dimension) 
			   (find-bottom-partial-order 
			    hierarchy (1- dimension))))
	(t
	 (select-next-level hierarchy (find-goals goal-list hierarchy dimension)
			    dimension bottom)
	 (create-hierarchy hierarchy goal-list recursive-goals (1- dimension) 
			   (find-bottom-partial-order 
			    hierarchy (1- dimension))))))

;;;
;;;
;;; This function simply orders the abstraction levels without
;;; attempting to combine any levels.
;;;
(defun order-levels (hierarchy goal-list dimension bottom)
; To debug: (trace refine-partial-order) (setq *print-array* nil) 
; (print-column hierarchy 0)  ;Use this for debugging
  (cond ((= dimension 1) nil)
	(t
	 (select-next-level hierarchy (find-goals goal-list hierarchy dimension)
			    dimension bottom)
	 (order-levels hierarchy goal-list (1- dimension) 
		       (find-bottom-partial-order 
			hierarchy (1- dimension))))))

;;;
;;;
;;;
;;; There are three cases where two levels can be combined:
;;; 1) If there are two levels on the bottom of the partial order
;;; and neither is a goal or a generator. 
;;; 2) If there is a level on the bottom of the partial order that is
;;; not a goal and there is a level constrained above it that is not
;;; also constrained above one of the goals.  
;;; 3) If there is only one goal left and it is the only thing on the
;;; bottom of the hierarchy, then it should be combined with all the
;;; levels above it.  There should always be a goal on the top of the
;;; hierarchy. 
;;; The idea is always to deal with the goal literals as high as
;;; possible in the abstraction hierarchy and combine anything but the
;;; goal and generator literals.
;;;
(defun combine-next-level (hierarchy dimension goals recur gens bottom)
  (let ((levels (remaining bottom goals recur gens)))
    (cond ((> (length levels) 1)
	   (combine-interacting-groups (first levels)(second levels) hierarchy))
	  ((and (null levels) ; Nothing left but the ...
		(eql 1 (length goals)) ; last goal
		(null gens))
	   (combine-goal-with-above hierarchy dimension (car goals) dimension))
	  ((or (intersect bottom goals)
	       (intersect bottom recur))
	   nil)
	  (t (find-next-level hierarchy dimension levels
			      (append goals recur))))))

;;;
;;;
;;; Combine the goal with the level above if the level is the last goal.
;;;
(defun combine-goal-with-above (hierarchy dimension level index)
  (cond ((zerop index) nil)
	((and (aref hierarchy index level)
	      (not (eql index level)))
	 (combine-interacting-groups level index hierarchy)
	 t)
	(t (combine-goal-with-above hierarchy dimension level (1- index)))))

;;;
;;; Determine what's on the bottom of the hierarchy besides the goals
;;; and generators.
;;;
(defun remaining (bottom goals recur gens)
  (cond ((null bottom) nil)
	((or (member (car bottom) goals)
	     (member (car bottom) recur)
	     (member (car bottom) gens))
	 (remaining (cdr bottom) goals recur gens))
	(t (cons (car bottom)
		 (remaining (cdr bottom) goals recur gens)))))

;;;
;;; Recurse through each of the elements on the bottom of the hierarchy.
;;;
(defun find-next-level (hierarchy dimension levels goals)
  (cond ((null levels) nil)
	((combine-level-above hierarchy dimension (car levels) goals dimension))
	(t (find-next-level hierarchy dimension (cdr levels) goals))))
;;;
;;;
;;; Combine the level with the level above if it is not a goal and it
;;; is not constrained above another goal.
;;;
(defun combine-level-above (hierarchy dimension level goals index)
  (cond ((zerop index) nil)
	((and (aref hierarchy index level)
	      (not (member index goals :test #'eql))
	      (not (goal-constraint hierarchy dimension goals index dimension)))
	 (combine-interacting-groups level index hierarchy)
	 t)
	(t (combine-level-above hierarchy dimension level goals (1- index)))))
;;;	 
;;;
;;; Determine whether the given row is constrained above any goal
;;; literal.
;;;
(defun goal-constraint (hierarchy dimension goals row index)
  (cond ((zerop index) nil)
	((and (not (eql row index))
	      (constraint-p hierarchy row index)
	      (or (member index goals :test #'eql)
		  (goal-constraint hierarchy dimension goals index dimension))))
	(t (goal-constraint hierarchy dimension goals row (1- index)))))

;;;
;;;
;;;
;;; Selects the next level of the abstraction hierarchy working from the
;;; bottom up.
;;;
(defun select-next-level (hierarchy goal-rows dimension bottom)
  (let ((preferred-bot (find-preferred hierarchy bottom goal-rows)))
    (swap-column hierarchy preferred-bot dimension)
    (swap-row hierarchy preferred-bot dimension)))


(defun find-preferred (hierarchy bottom goal-rows)
  (let ((preferred (not-in-goal-rows bottom goal-rows)))
    (cond ((null preferred)(fewest-constraints hierarchy bottom))
	  (t (fewest-constraints hierarchy preferred)))))

;;;
;;;
;;;
;;; Returns the next level preferring levels that do not contain any goals.
;;; Thus, levels that do contain goals will be placed higher than those that
;;; don't.
;;;
(defun not-in-goal-rows (bottom goal-rows)
  (cond ((null bottom) nil)
	((member (car bottom) goal-rows :test #'eql)
	 (not-in-goal-rows (cdr bottom) goal-rows))
	(t (cons (car bottom)
		 (not-in-goal-rows (cdr bottom) goal-rows)))))

;;;
;;;
;;; Finds the bottom of the partial order by searching for a column
;;; in the array this is completely null.  A non-null column indicates 
;;; that the group in that column must be above the group in the 
;;; row containing the non-null entry.  If this fails to produce 
;;; an element of the array then we search of a column that doesn't
;;; contain any hard constraints ((t)=hard constrant, (p)=soft constraint).
;;; This is guaranteed to succeed because we combined all the strongly
;;; connected components based on the hard constraints.
;;;
(defun find-bottom-partial-order (group-graph dimension)
  (cond ;((find-bottom group-graph dimension dimension
	;	      #'no-hard-or-soft-constraints))
	((find-bottom group-graph dimension dimension #'no-hard-constraints))
	(t (error "Can find an element in the array to abstract."))))
;;;
;;;
;;;
(defun find-bottom (group-graph x y test)
  (cond ((zerop x) nil)
	((apply test (list group-graph x y))
	 (cons x (find-bottom group-graph (1- x) y test)))
	(t (find-bottom group-graph (1- x) y test))))
;;;
;;;
;;;
;;; Tests whether column x is null.
;;;
(defun no-hard-or-soft-constraints (group-graph x y)
  (cond ((zerop y))
	((aref group-graph x y) nil)
	(t (no-hard-or-soft-constraints group-graph x (1- y)))))
;;;
;;;
;;;
;;; Tests whether column x is null.
;;;
(defun no-hard-constraints (group-graph x y)
  (cond ((zerop y))
	((constraint-p group-graph x y) nil)
	(t (no-hard-constraints group-graph x (1- y)))))
;;;
;;;
;;;
;;; Determines number of constraints in row
;;;
(defun fewest-constraints (group-graph bottom)
  (car (sort bottom #'<
	     :key #'(lambda (x)
		      (number-of-constraints
		       group-graph 
		       (1- (array-dimension group-graph 0))
		       x)))))

(defun number-of-constraints (group-graph x y)
  (cond ((zerop x) 0)
	((aref group-graph y x) 
	 (1+ (number-of-constraints group-graph (1- x) y)))
	(t (number-of-constraints group-graph (1- x) y))))


;;;
;;;
;;;
;;; Takes the goal list and the current abstraction hierarchy and returns
;;; a list of the levels in which the goals occur in.  There are two 
;;; optional arguments.  The first is the dimension of the array.  It
;;; defaults to the entire array, but can be used to restrict the
;;; array to a smaller part of it.  The second optional argument is 
;;; used to build the list of levels and avoid duplicates.
;;;
(defun find-goals (goal-list hierarchy &optional 
			     (dimension (1- (array-dimension hierarchy 0)))
			     (levels nil))
  (cond ((null goal-list) levels)
	((member (extract (car goal-list)) *STATIC* :test #'equal)
	 (find-goals (cdr goal-list) hierarchy dimension levels))
	(t 
	 (let ((level (find-row hierarchy (extract-instance (car goal-list)))))
	   (cond ((or (> level dimension)
		      (member level levels))
		  (find-goals (cdr goal-list) hierarchy dimension levels))
		 (t (find-goals (cdr goal-list) hierarchy 
				 dimension (cons level levels))))))))

;;;
;;;
;;;
;;;
(defun find-recursive-goals (hierarchy &optional 
			     (dimension (1- (array-dimension hierarchy 0))))
  (cond ((zerop dimension) nil)
	((member 'p (aref hierarchy dimension dimension))
	 (cons (car (aref hierarchy 0 dimension))
	       (find-recursive-goals hierarchy (1- dimension))))
	(t (find-recursive-goals hierarchy (1- dimension)))))

