
(eval-when (compile) 
	(load-path *PLANNER-PATH* "g-loop")
	(load-path *PLANNER-PATH* "g-map")
	(load-path *PLANNER-PATH* "data-types"))

(proclaim 
    '(special *FAILURE-RECORD* *FAILURE-RECORDING* *STATE*
	      *PREDICATES* *ABSTRACTION-LEVEL*))

 
; --------  Syntax of Expressions, data-structures ------------
; 		(See file data-types.l also)

;  A bindings-list looks like ((var val)(var val)...(nil nil))
;  The last pair in the list is usually (nil nil), by convention,
;  but this is optional.
; 
; Conventions: a bindings list will never be nconced, but
; a list of binding lists may. (Functions return a
; list of binding lists, which should be mungable on the top level).
; 
 
; EXP-MATCH is the top-level call to the matcher. After
; matching an expression, *FAILURE-RECORD* contains a list of unmatched 
; literals that the matcher failed on.
; If a match is found, EXP-MATCH returns non-nil value - a
; list of bindings-lists that the expression matches
; with. (NOTE: only bindings for existentially quantified variables are 
; returned). 

(defun exp-match-failures (exp in-bindings)
  (setq *STATE* (create-state 0 0)
	*FAILURE-RECORD* nil
	*FAILURE-RECORDING* t)
  (setf (state-false-assertions *STATE*) '((t)))
  (setf (state-true-assertions *STATE*) '((t)))
  (setf (state-closed-world *STATE*) '((t)))
  (setf (state-justification-table *STATE*) '((t)))
  (descend-match-failures exp (or in-bindings (list '(nil nil))) 'all
			  (list in-bindings))
  (remove-duplicates *FAILURE-RECORD* :test #'equal))


; The *FAILURE-RECORD* contains a list of predicates that caused
; and expression to fail to match. For example, if the
; expression is: (exists (<sq>)(is-square <sq>)(empty <sq>)
; and there is no empty square, then (empty <sq>) will
; *FAILURE-RECORD* contain (empty <sq>) as a member.
; Note: Sometimes an expression will fail to match because 
; a predicate is true - In this case, FAILURE record
; contains the negated predicate.

;  RECORD-FAILURES updates the *FAILURE-RECORD*, assuming the flag
;  *FAILURE-RECORDING* is set and the lit is possible to achieve

(defun record-failures (lit bindings) 
    (cond (*MATCHER-TRACING* 
	      (cond ((eq *PRINT-TRACING* t)
		     (terpri) 
		     (format t "  Matcher Failed on: ~A" lit)
		     (terpri)
		     (format t "  bindings: ~A" bindings)))
	      (push (list lit bindings) *MATCHER-TRACE*)))
    (cond (*FAILURE-RECORDING* 
	      (cond ((impossible-to-achieve-lit lit))
		    ((push (subst-bindings lit bindings) *FAILURE-RECORD*)))))
    nil)


;  This is the recursive match routine that does all the work. 
;  (Can make more efficient by remmoving atomic formula test & moving to rear)


(defun descend-match-failures (exp bindings mode unknown)
    (cond ((eq exp t) (list bindings))
	  ((eq (car exp) 'and)
	   (and-match-failures (cdr exp) bindings mode unknown))
	  ((eq (car exp) 'exists)	       
	   (exists-match-failures bindings (get-gen-exp exp) (get-exp exp) mode unknown))
	  ((eq (car exp) 'forall)
	   (forall-match-failures bindings (get-gen-exp exp) (get-exp exp) unknown))
	  ((eq (car exp) '~)
	   (negation-match-failures (cadr exp) bindings unknown))
	  ((eq (car exp) 'or)
	   (or-match-failures (cdr exp) bindings mode unknown))
	  ((null exp) (list bindings))
	  ; must be an atomic formula
	  ((let ((vals (gen-values-failures exp bindings unknown)))
		(cond ((null vals)
		       (cond ((impossible-to-achieve-lit exp)
			      nil)
			     (t (record-failures exp bindings)
				unknown)))
		      ((eq vals 'no-match-attempted)
		       (error "Unable to match expression: ~a" exp))
		      ((eq vals t)
		       (list bindings))
		      ((mapcar #'(lambda (b) (append b bindings))
			       vals)))))))


; at present time, exp must be an non-negated literal with all vars bound
; or an expression of the form (exists ...)
; Don't need mode flage, since we know we only have to look for a single
; bindings

(defun negation-match-failures (exp bindings unknown)
    (cond ((eq exp t) nil)
	  ((eq (car exp) 'exists)
	   (cond ((negated-exists-match-failures exp bindings unknown))
		 (t (record-failures exp bindings)
		    unknown)))
	  ((member (car exp) *PREDICATES*)
	   (negated-pred-match-failures exp bindings unknown))
	  (t (error "NEGATION-MATCH: bad expression ~A" exp))))


(defun negated-exists-match-failures (exp bindings unknown)
    (prog (new-bindings matched-lits)
	  (setq *FAILURE-RECORDING* nil)
	  (setq new-bindings (car (descend-match-failures exp bindings 'one
						 (list bindings))))
	  (setq *FAILURE-RECORDING* t)
	  (cond ((not new-bindings) (return unknown))
		((setq matched-lits 
		       (mapcan #'(lambda (lit) 
				   (cond ((not (impossible-to-achieve-lit 
						   `(~  ,lit)))
					  (list lit))
					 (t nil)))
			       (get-lits (subst-bindings (get-exp exp) 
					     new-bindings))))
		 (dolist (matched-lit matched-lits nil)
		     (record-failures `(~ ,matched-lit) new-bindings))
		 (return (list bindings)))
		(t (record-failures 
		       `(~ ,(subst-bindings (get-gen-exp exp) new-bindings))
		       new-bindings)
		   (return (list bindings))))))
								 
		 
; exp = (P x y...)
; Returns bindings if exp is known to be  False.

; should do a test to make sure all vars are bound!

(defun negated-pred-match-failures (exp bindings unknown)
    (cond ((function-p exp)
	   (and (not (get-func-vals exp bindings 'no-match-attempted))
		(list bindings)))
	  ((member (car exp) *CLOSED-PREDS*)
	   (record-failures exp bindings)
	   unknown)
;	   (cond ((has-vars exp) 
;		  (and (not (member-state 
;			     exp (state-closed-world *STATE*) bindings nil))
;		       (list bindings)))
;		 ((not (member-state-inst exp (state-closed-world *STATE*) nil))
;		  (list bindings)))
	  ((member-state exp (state-false-assertions *STATE*) bindings t)
	   (list bindings))
	  (t (record-failures exp bindings)
	     unknown)))



; should make the independence test in and-match
; more sophisticated, ack-- cant do independent cause
; record-failures loses the bindings from the other clause.
; leads to funny behavior....

; Exps = (sub-exp sub-exp ....)
; if all sub-exps in exps match, then their bindings are combined and 
; returned. Better not be two variables with the same name in exp.


(defun and-match-failures (exps bindings mode unknown)
    (cond ((null exps) (list bindings))
	  ((null (cdr exps))
	   (nreverse (descend-match-failures (car exps) bindings mode unknown)))
	  ((g-loop (init new-bls (descend-match-failures (car exps) bindings
						mode (list bindings))
			  ret-val nil)
	            (while new-bls)
		    (do (setq ret-val 
			      (nconc (and-match-failures (cdr exps) 
							 (pop new-bls) 
							 mode (list bindings))
				     ret-val)))
		    (until (and ret-val (eq mode 'one)))
		    (result (if unknown ret-val nil))))))
 

; OR-MATCH returns the bindings of the 1st sub-exp to match.

(defun or-match-failures (exps bindings mode unknown)
    (g-loop (init subresult nil exp nil)
	     (while (setq exp (pop exps)))
	     (do (setq subresult (nconc (descend-match-failures exp bindings
						       mode nil)
					subresult)))
	     (until (and (eq mode 'one) subresult))
	     (result (cond (subresult)
			   (unknown (list bindings))
			   (t unknown)))))

; Matches a universally quantified expression. Returns start-bindings
; on success (unless no variables are bound,in which case lower-level bindings
; are also returned.

(defun forall-match-failures (start-bindings gen-exp exp unknown)
    (g-loop (init values (gen-values-failures gen-exp start-bindings unknown) bindings nil)
	     (before-starting           ; make sure values are good
		 (cond ((null values) (return (list start-bindings)))
		       ((eq values 'no-match-attempted) (return nil))
		       ((eq values t)
			(return (descend-match-failures exp start-bindings 'one unknown)))))
	     (while values)
	     (next bindings (append (pop values) start-bindings))
	     (do (cond ((not (descend-match-failures exp bindings 'one bindings))
			(or (static-p gen-exp)
			    (record-failures (list '~ gen-exp) bindings))
			(return (list start-bindings)))))
	     (result (list start-bindings))))

; Matches an existentially quantified expression. Returns start-bindings,
; with the new var-value pairs appended on front. 

(defun exists-match-failures (start-bindings gen-exp exp mode unknown)
    (g-loop (init values (gen-values-failures gen-exp start-bindings unknown)
		   subresults nil bindings nil)
	     (before-starting 
		 (cond ((null values)
			(record-failures gen-exp start-bindings) 
			(error "Unable to match generator ~a" gen-exp))
		       ((eq values 'no-match-attempted) (return))
		       ((eq values t)
			(return (descend-match-failures exp start-bindings mode unknown)))))
	     (while values)
	     (next bindings (append (pop values) start-bindings))
	     (do (setq subresults (nconc (descend-match-failures exp bindings
							mode bindings)
					 subresults)))
	     (until (and subresults (eq mode 'one)))
	     (result subresults)))



;  GEN-VALUES generates the values for a generator expression
;  If there are variables in the expression, a list of bindings
;  lists for the new variables is returned. Otherwise, t or nil
;  is returned, depending on whether the expression matches.

; For now, we'll make some explicit assumptions about gen-exp,
; if we expect the failure recording stuff to work right then
; it should be a non-negated literal. (Only important when
; matcher is called by engine).

; Note: If var is actually a constant, then gen-exp has already been
; at least partially instantiated
; Note: If no-match-attempted on a function-pred, then this 
; returns 'no-match-attempted

;  ASSUMEs only (meta-level) functions can have internal variables....

(defun gen-values-failures (gen-exp bindings unknown)
    (cond ((function-p gen-exp)
	   (get-func-vals gen-exp bindings 'no-match-attempted))
	  ((not (atomic-formula-p gen-exp))
	   (remove-extra-ts
	       (mapcar #'(lambda (b) (diff-bindings b bindings))
		       (descend-match-failures gen-exp bindings 'all unknown))))
	  ((and *ABSTRACTION-LEVEL*
		(not (member (extract (strip-negation gen-exp))
			     *ABSTRACTION-LEVEL* 
			     :test #'equal))))
	  ((has-unbound-vars gen-exp bindings)
	   (get-pred-values-failures gen-exp bindings))
	  ((member-state-inst (subst-bindings gen-exp bindings)
			      (state-true-assertions *STATE*)
			      t)
	   t)
	  (t nil)))




(defun get-pred-values-failures (literal bindings)
; assumes that there are unbound vars in literal.
    (setq literal (subst-bindings literal bindings))
    (if (member (car literal) *ACHIEVABLE-PREDS*)
	(let ((gen-bindings (gen-each-type (extract-all-vars (cdr literal)))))
	  (mapcar #'(lambda (x)(record-failures literal x)) gen-bindings)
	  gen-bindings)
      (mapcan #'(lambda (assertion)
		  (let ((binding (lit-match3 literal assertion)))
		    (cond (binding (list binding))
			  (t nil))))
	      (state-true-assertions *STATE*))))


(defun extract-all-vars (args)
  (cond ((null args) nil)
	((not (is-variable (car args)))
	 (extract-vars (cdr args)))
	(t (cons (car args)(extract-vars (cdr args))))))
;;;
(defun gen-each-type (types)
  (cond ((null types) nil)
	((null (cdr types))(mapcar #'(lambda (x)(list (list (car types) x)))
				   (generate-type-descendents
				    (arg-type (car types)) t)))
	(t (mapcan #'(lambda (x) (mapcar #'(lambda (y) 
					     (cons (list (car types) x) y))
					 (gen-each-type (cdr types))))
		   (generate-type-descendents (arg-type (car types)) t)))))
