  
(proclaim '(special *CONSTANT-LIST* *TYPE-LIST* *PREDICATE-LIST* *PRINT-TYPE*
		    *OPERATORS* *INFERENCE-RULES* *OPS* *GROUP-LIST*
		    *CONNECTIONS* *GROUP-GRAPH* *TRANSFORMATIONS* *DROPPED*
		    *GROUP-INTERACTION* *VARIABLE-TYPING*))
  
;;;
;;;==============================================================================
;;;
;;; Need both types and ins-types.  No negations in either of these.
;;;
;;; This function groups the predicates based on strong interactions.
;;;
(defun group-predicates (ops types ins-types)
  (let ((effects (find-effects ops))
	(connections (make-array (list (1+ (length ops)) (1+ (length types)))
				 :initial-element nil))
	(group-list (create-group-list 
		     (make-array (list (1+ (length ins-types)))
				 :adjustable t)
		     ins-types 1)))
    (setq *CONNECTIONS* connections)
    (setq *GROUP-LIST* group-list)
    (put-ops-in-array ops connections 1)
    (put-types-in-array types connections 1)
    (put-lits-in-array effects connections ops types)
    (combine-strongly-interacting (cross-product ins-types) 
				  connections group-list)
    group-list))
				  

(defun create-group-list (group-list types index)
  (cond ((null types) group-list)
	(t (setf (aref group-list index) (list (car types)))
	   (create-group-list group-list (cdr types) (1+ index)))))


;;;
;;;
;;;
;;; If two literals are strongly interacting then they are combined
;;; into the same group.
;;;
(defun combine-strongly-interacting (literals connections group-list)
  (cond ((null literals) group-list)
	(t (let* ((lit1 (first (car literals)))
		  (lit2 (second (car literals)))
		  (group1 (find-group group-list lit1))
		  (group2 (find-group group-list lit2)))
	     (cond ((and (not (eql group1 group2))
			 (or (and *VARIABLE-EXPANSION*
				  (equal (extract lit1)(extract lit2))
				  (not-subgoalable lit1)
				  (not-subgoalable lit2))
			     (strongly-interacting connections lit1 lit2)))
		    (format t "~%~% ~a strongly interacts with ~a"
			    lit1 lit2)
		    (combine-strongly-interacting
		     (cdr literals) connections 
		     (combine-literals group-list lit1 lit2)))
		   (t (combine-strongly-interacting (cdr literals) 
						    connections 
						    group-list)))))))
;;;
;;;
;;; Returns the cross product of all the elements in the list.
;;;
(defun cross-product (types)
  (mapcan #'(lambda (x)(mapcar #'(lambda (y)(list x y))
			       types))
	  types))
;;;
;;;
;;;
;;; Two literals are combined by combining the corresponding rows and columns.
;;;
(defun combine-literals (group-list lit1 lit2)
  (let ((group1 (find-group group-list lit1))
	(group2 (find-group group-list lit2)))
    (combine-elements group-list group1 group2)
    group-list))
;;;
;;;
;;;
;;; Find the element that the literal occurs in.
;;;
(defun find-group (group-list lit-type)
  (do ((element (1- (array-dimension group-list 0))(1- element)))
      ((member lit-type (aref group-list element) :test #'equal) element)
    (if (= 1 element)(error "Literal: ~a not found in array" lit-type))))

;;;
;;;
;;;
;;; Two literals are strongly interacting if they are weakly interacting in both
;;; directions.
;;;
(defun strongly-interacting (connections lit-type1 lit-type2)
  (cond ((equal lit-type1 lit-type2) nil)
	((and (weakly-interacting connections lit-type1 lit-type2)
	      (weakly-interacting connections lit-type2 lit-type1)))))
;;;
;;;
;;;
;;; The first literal weakly interacts with the second literal if there is
;;; an operator that contains the first literal as a primary add and either
;;; adds or deletes the second literal.
;;;
(defun weakly-interacting (connections lit1 lit2)
  (let ((lit1-row (find-row connections (extract lit1)))
	(lit2-row (find-row connections (extract lit2))))
    (check-ops-for-primary connections lit1 lit1-row lit2 lit2-row
			   (1- (array-dimension connections 0)))))
;;;
;;;
;;;
;;; Check each of the operators that have lit1 as a primary add to see if they
;;; interact with lit2.
;;;
(defun check-ops-for-primary (connections lit1 lit1-row lit2 lit2-row col)
  (cond ((zerop col) nil)
	((and (primary-adds-p (aref connections col lit1-row))
	      (aref connections col lit2-row)
	      (weak-interaction (find-primary-adds 
				 (aref connections col lit1-row))
				(aref connections col lit2-row)
				lit1 lit2)))
	(t (check-ops-for-primary connections lit1 lit1-row lit2 lit2-row 
				  (1- col)))))


(defun weak-interaction (primary secondary lit1 lit2)
  (cond ((null primary) nil)
	(t (let ((blsts (match (list (strip-negation 
				      (formula-for-comparison (car primary))))
			       (list lit1)
			       '(((nil))))))
	     (cond ((and (not (null blsts))
			 (interacts-with (car primary) secondary lit2 blsts)))
		   (t (weak-interaction (cdr primary) secondary lit1
					lit2)))))))

(defun interacts-with (primary secondary lit2 blsts)
  (cond ((null secondary) nil)
	((eq primary (car secondary)) ; avoids a lit interacting with itself
	 (interacts-with primary (cdr secondary) lit2 blsts))
	((match (list (strip-negation (formula-for-comparison (car secondary))))
		(list lit2) blsts))
	(t (interacts-with primary (cdr secondary) lit2 blsts))))

(defun formula-for-comparison (formula)
  (if *VARIABLE-EXPANSION* 
      (formula-instance formula)
    (formula-type formula)))
;;;
;;;
;;;
;;; Find the row that the literal occurs in.
;;;
(defun find-row (connections lit-type)
  (do ((row (1- (array-dimension connections 1))(1- row)))
      ((member lit-type (aref connections 0 row) :test #'equal) row)
    (if (= 1 row)(error "Literal: ~a not found in array" lit-type))))
;;;
;;;
;;;
;;; Find the column that the literal occurs in.
;;;
(defun find-column (connections lit-type)
  (do ((col (1- (array-dimension connections 1))(1- col)))
      ((member lit-type (aref connections 0 col) :test #'equal) col)
    (if (= 1 col)(error "Literal: ~a not found in array" lit-type))))
;;;
;;;
;;;
;;; Find one of the primary adds in the given row.
;;;
(defun find-primary-adds (literals)
  (cond ((null literals) nil)
	((primary-add-p (car literals))
	 (cons (car literals)(find-primary-adds (cdr literals))))
	(t (find-primary-adds (cdr literals)))))
;;;
;;;
;;;
;;; Determine if a particular instance of a literal is the primary add of some 
;;; operator.
;;;
(defun primary-adds-p (literals)
  (cond ((null literals) nil)
	((primary-add-p (car literals)))
	(t (primary-adds-p (cdr literals)))))

(defun primary-add-p (literal)
  (eq 'primary (formula-status literal)))


;
;==============================================================================
;

(defun order-groups (types group-list connections)
  (let ((group-graph (make-array (list (array-dimension group-list 0)
				       (array-dimension group-list 0))
				       :initial-element nil
				       :adjustable t)))
    (setq *GROUP-GRAPH* group-graph)
    (put-groups-in-group-graph group-list group-graph 
			       (1- (array-dimension group-list 0)))
    (graph-weak-interactions connections types group-graph)
    (add-generator-constraints group-graph)
    (print-weak-interactions group-graph)
    group-graph))

(defun put-groups-in-group-graph (group-list group-graph dimension)
  (cond ((zerop dimension))
	(t (setf (aref group-graph 0 dimension)(aref group-list dimension))
	   (setf (aref group-graph dimension 0)(aref group-list dimension))
	   (put-groups-in-group-graph group-list group-graph (1- dimension)))))


(defun graph-weak-interactions (connections types group-graph)
  (dolist (literals (cross-product types))
	  (if (weakly-interacting connections (first literals)
				  (second literals))
	      (add-constraint (first literals)(second literals) 
			      group-graph 'e))))


(defun add-constraint (lit1 lit2 group-graph constraint)
  (let* ((x (find-row group-graph lit1))
	 (y (find-row group-graph lit2))
	 (constraints (aref group-graph x y)))
    (cond ((member constraint constraints))
	  (t (setf (aref group-graph x y)
		   (cons constraint constraints))))))


(defun print-weak-interactions (group-graph)
  (do ((x 1 (1+ x)))
      ((= x (array-dimension group-graph 0)))
    (do ((y 1 (1+ y)))
	((= y (array-dimension group-graph 1)))
      (if (aref group-graph x y)
	  (format t "~%~%~a weakly interacts with ~%     ~a" (aref group-graph 0 x)
		  (aref group-graph 0 y))))))


;;;
;;;
;;;
;;; There may be effects which don't appear in the list of types because they
;;; is no way to acheive them if they don't start out true.  They are simply a 
;;; irreversable side-effect of an operator.
;;;
;(defun remove-side-effects (effects types)
;  (cond ((null effects) nil)
;	((position (formula-type (car effects)) types :test #'equal)
;	 (cons (car effects)
;	       (remove-side-effects (cdr effects) types)))
;	(t (remove-side-effects (cdr effects) types))))
;;;
;;;
;;;
;;; If two literals are negations of each other then they are combined into the same 
;;; group.
;;;
;(defun combine-negations (connections types)
;  (dolist (literals (cross-product types))
;    (cond ((or (and (eq '~ (car (first literals)))
;		    (not (eq '~ (car (second literals))))
;		    (equal (cadr (first literals))(second literals)))
;	       (and (not (eq '~ (car (first literals))))
;		    (eq '~ (car (second literals)))
;		    (equal (first literals)(cadr (second literals)))))
;	   (combine-literals connections (first literals)(second literals))))))
;;;
;;;
;;; Lit1 is a type descendent of lit2 if it falls below it based on
;;; the type hierarchy.
;;;
;(defun type-descendents (lit1 lit2)
;  (cond ((and (eq (car lit1)(car lit2))
;	      (eql (length lit1)(length lit2)))
;	 (compare-types (cdr lit1) (cdr lit2)))))
;
;(defun compare-types (lit1 lit2)
;  (cond ((null lit1))
;	((or (eq (car lit1)(car lit2))
;	     (descendent (car lit1)(type-children (get (car lit2) 'structure))))
;	 (compare-types (cdr lit1)(cdr lit2)))))
;
;(defun descendent (type list)
;  (cond ((null list) nil)
;	((member type list :test #'same-type))
;	(t (descendent type 
;		       (append (type-children (car list))(cdr list))))))
;
;(defun same-type (type type-struct)
;    (eq type (type-of type-struct)))
;
