
(proclaim '(special *AB-START-TIME* *TOP-LEVEL* *ABSTRACTION-LEVEL* *DBTWL* n1
	            *START-TIME* *RUN-TIME* *STATIC* *PRINT-NODE-NUMBERS*
		    *INITIAL-STATE* *GOAL-STATE* *ABS-PRINT-FLAG* *FINAL-STATE*
		    *TOP-LEVEL-GOALS* *ABSTRACTION-LEVEL-GOALS*
		    *ABS-HIERARCHY* *ABS-LEVEL-FLAG* *UNIQUE-NODES*
		    *NODE-DIST* *SOL-LENGTH* *EXPERIMENT-TIME-BOUND*
		    *SOLUTION-DIST* *SINGLE-HIERARCHY* *ABS-SOLUTION-FLAG*
		    *AUTODEFINE-STATIC-FUNCTIONS* *REMOVE-UNUSED-PREDS-FLAG*
		    *PREV-ABS-LEVEL* *TRACE-TEXT-FLAG* *DOMAIN-GRAPHICS*
		    *PRODIGY-TIME-BOUND* *START-STATE* *ALL-NODES*
		    *SCR-BINDINGS-SELECT-RULES* *SCR-OP-SELECT-RULES*
		    *PROBLEM-GOAL* *OPERATORS* *INFERENCE-RULES*
		    *HIERARCHY* *SCR-OP-REJECT-RULES* *ABS-OPERATORS*
		    *OPERATORS* *ABS-INFERENCE-RULES* *INFERENCE-RULES*
		    *ABS-SCR-GOAL-SELECT-RULES* *SCR-GOAL-SELECT-RULES*
		    *EBL-TRACE* *LEARNED-RULES-IN-SYS* *REV-PROBSET*
		    *SOLUTION*))

(setq *DBTWL* nil) ; backtrack within levels for completeness
(setq *AUTODEFINE-STATIC-FUNCTIONS* t) ; turn on dspf
(setq *ABS-PRINT-FLAG* nil) ; turn off tracing
(setq *TRACE-TEXT-FLAG* nil) ; ...
(setq *PRINT-NODE-NUMBERS* nil)
(setq *ABS-LEVEL-FLAG* nil)
(setq *ABS-SOLUTION-FLAG* nil)
(setq *EXPERIMENT-TIME-BOUND* 300)
(setq *PRODIGY-TIME-BOUND* *EXPERIMENT-TIME-BOUND*)
(setq *REMOVE-UNUSED-PREDS-FLAG* t)
(setq *ABS-OPERATORS* nil)
(setq *ABS-INFERENCE-RULES* nil)
(setq *ABS-SCR-GOAL-SELECT-RULES* nil)
(setq *EBL-TRACE* nil)
(setq *REV-PROBSET* nil)
(setq *LEARNED-RULES-IN-SYS* nil)

(eval-when (compile) 
	(load-path *PLANNER-PATH* "data-types"))
;
; Code for running prodigy using abstraction.
;
;
; Basic Design:
;
; Prodigy first produces a skeletal plan using only the preconditions at the
; highest abstraction level.  Each of the operators in this skeletal plan will
; then be expanded at the next level of abstraction.  These operators 
; correspond to the applied nodes, so each applied node will contain a pointer
; to the abstract-child and applied nodes that were generated from a higher 
; level with have an pointer to the abstract-parent.  
;
; In addition to the pointers between applied node, the first node at each
; abstraction level is linked to the first node of the levels above and 
; below it.  This makes it easy to traverse between abstract levels for 
; backtracking and expanding levels.  
;
; Since each applied node at an abstract level is expanded at the more 
; detailed levels, there can be a number of independent search trees at
; an abstraction level.  Each applied node will have its own corresponding
; tree that expands out the details at the next level.  To simplify the
; backtracking and to check for state loops, we link these independent
; search trees together using the parent and children fields of the node
; data structure.  The last node in one tree is linked to the first node
; in the next.  
;
; Backtracking is performed simply by searching backwards within an abstraction
; level for an node that has an unexplored alternative.  If one is found, 
; prodigy is called to to explore it.  The failing subtree is saved as
; one of the children of the parent node, where the first child is on the 
; current path.  Backtracking across an abstraction level works similarly,
; except we jump up a level before attempting to backtrack.  
;
; After backtracking across a level it may be possible to reuse some
; of the work already performed at the lower level.  Currently we simply
; search forward for the last node that is expanded an corresponds to 
; an operator in the abstract space.  Then the system starts expanding
; from that point.  There is an incompleteness here.  We may have already
; backtracked within the part of the abstraction level that is being reused.
; In that case, if we backtrack again, these already explored path should
; be explored again since the abstract plan has changed.  They will not be.
; A similar problem is that we may have successfully solved a subgoal and
; then in the process of backtracking the last attempt may fail.  In this case
; attempting to reuse the previous work will cause confusion.  Analysis:
; if we allow backtracking with abstraction levels, trying to reuse the
; previous work may be difficult. 
;
;
;

; Resets the abstraction stuff so that you can run prodigy without abstraction.
(defun no-abstraction ()
  (setq *ABSTRACTION-LEVEL* nil)
  (setq *PREV-ABS-LEVEL* nil)
  (load-domain)
  (load-goal *FINAL-STATE*)
  (setq *START-STATE* *INITIAL-STATE*)
  'domain-reset)


(defun toplevel-run-abstraction ()
  (cond ((or (not (boundp '*operators*))
	     (not (boundp '*inference-rules*)))
	 (format t "~%No domain loaded!"))
	(t (if (and (or (not (boundp '*ebl-flag*))(null *ebl-flag*))
		    (or (not (eq *ABS-OPERATORS* *OPERATORS*))
			(not (eq *ABS-INFERENCE-RULES* *INFERENCE-RULES*))
			(not (eq *ABS-SCR-GOAL-SELECT-RULES*
				 *SCR-GOAL-SELECT-RULES*))))
	       (create-abstraction))
	   (load-abstract-domain)
	   (run-abstraction)
	   (no-abstraction))))

;
; Top level function called initially with the highest criticality for the 
; particular domain.  The domain will get loaded with only the preconditions
; of that criticality or higher and then the system will run prodigy to 
; produce a skeletal plan.
;
; Note that *GOAL-STATE* and *INITIAL-STATE* must be set before this is
; called.  They are usually set by load-goal and load-start-state.  
;
(defun run-abstraction ()
  (setq *ABSTRACTION-LEVEL* nil)
  (setq *PREV-ABS-LEVEL* nil)
  (setq *AB-START-TIME* (get-internal-run-time))
  (if *SINGLE-HIERARCHY* 
      (setq *GOAL-STATE* (augment-goal-exp *FINAL-STATE* t))
    (setq *GOAL-STATE* (build-hierarchy *FINAL-STATE*)))
  (setf (get '*FINISH* 'params)
	(setf (get '*FINISH* 'vars)(find-all-vars *GOAL-STATE*)))
  (let ((level (1- (array-dimension *ABS-HIERARCHY* 1))))
    (setq *TOP-LEVEL* level)
    (set-abstraction-level level)
    (load-abstract-goal *GOAL-STATE*)
    (setq *TOP-LEVEL-GOALS* (abstract-goals *GOAL-STATE*))
    (load-abstract-start-state *INITIAL-STATE*)
;    (load-new-abstraction-level level)
    (cond (*EBL-TRACE* 
	   (push `((*subproblem-name* ,(make-subproblem-name))
		   (*abstraction-level* ,*abstraction-level*)
		   (load-goal ,*problem-goal*)
		   (load-start-state ,*start-state*)
		   (*op-reject-rule* nil)
		   (*binding-select-rule* nil))
		 *rev-probset*)))
    (setq *START-TIME* (get-internal-run-time))
    (cond ((run)
	   (if (or *abs-print-flag* *abs-solution-flag*)(print-solution n1))
   (let ((result 
		  (catch 'omv (change-abstraction-level (1- level) n1 nil))))
	     (cond ((null result) nil)
		   ((eq t result) t)
		   (t (handle-ordered-monotonicity-violation result)))))
	  (t (print-no-solution-stats) nil))))


(defun handle-ordered-monotonicity-violation (abs-lit)
  (let ((generating-lit (node-current-goal (third *all-nodes*)))
	(generating-goal (find-generating-goal (third *all-nodes*))))
    (format t "~%Ordered Monotonicity Violation!")
    (format t "~%Achieving ~a changes ~a" generating-lit abs-lit)
    (add-constraint (extract generating-lit)(extract abs-lit) *HIERARCHY* 'v)
    (format t "~%Generating goal of ~a: ~a" generating-lit generating-goal)))
;    (run-abstraction)))

(defun find-generating-goal (node)
  (let ((abstract-node (find-next-abstract-node node)))
    (cond ((or (null abstract-node)(null (node-current-goal abstract-node)))
	   (cond ((equal '(done) (node-current-goal node))
		  (alt-goal (node-generating-alt node)))
		 (t (node-current-goal node))))
	  (t (find-generating-goal abstract-node)))))

;
;
;
; This function moves among the various abstraction levels, expanding
; out the details of higher abstraction levels and backtracking across
; levels where necessary.  The function is called with three arguments:
; the criticality level, the initial node at the level, and a flag indicating
; whether the system has gone up or down an abstraction level.  
;
; In the top level conditional the function checks to see if the criticality
; is zero, in which the the plan has been refined to the most detailed level
; and the solution is printed.  It then checks to see if it has backtracked
; across the highest abstraction level, in which case there is no solution.
; Finally, the third condition adjusts the criticality according to the 
; backtrack flag.
;
; If the backtrack flag is set to then we are in the process of backtracking.
; In this case the backtracking function is called on the last node within
; the given abstraction level.  This function returns nil if it is unsuccessful
; or the last node in the subtree expanded if it sucessfully expands a back-
; track point.  If it fails we simply backtrack up another level.  Sucess
; means we need to expand the rest of the abstraction level, print the 
; solution, and drop down to the next level.  If the expansion fails then
; we have exhausted all backtracking possibilities within this level and 
; we need to backtrack across another level.
;
; If the backtrack flag is nil, then we simply need to start expanding the
; abstraction level.  If the level can be successfully expanded, then the 
; solution is printed and it drops down to the next level.  Note that 
; before we start expanding this level we call find-last-expanded because
; we may have already partially expanded this level and we can reuse the 
; work that has not been backtracked over at a higher level.  If this level
; cannot be expanded then we backtrack to the next level.
;
(defun change-abstraction-level (level node backtrack)
  (if (and backtrack *ABS-PRINT-FLAG*)
      (format t "~%~%Backtracking across abstraction levels..."))
  (if (or *ABS-PRINT-FLAG* *ABS-LEVEL-FLAG*)
      (format t "~%~%~%==================>>>>>> Abstraction Level ~d <<<<<<===================" level))
  (cond ((zerop level) 
	 (print-final-abs-stats node (car *all-nodes*))
	 t)
 	((or (> level *TOP-LEVEL*)
	     (> (- (get-internal-run-time) *AB-START-TIME*)
		(* *PRODIGY-TIME-BOUND* internal-time-units-per-second)))
	 (print-no-solution-stats)
	 (setq *ABSTRACTION-LEVEL* nil)
	 (setq *PREV-ABS-LEVEL* nil)
	 nil)
	(t 
	 (set-abstraction-level level)
;	 (load-new-abstraction-level level)
	 (load-abstract-goal *GOAL-STATE*)
	 (setq *TOP-LEVEL-GOALS* (abstract-goals *GOAL-STATE*))
	 (load-abstract-start-state *INITIAL-STATE*)
	 (let* ((new-node (if backtrack (node-abstract-parent node)
			                (next-abstract-level node)))
		(last-node 
		 (if backtrack
		     (backtrack-within-abstraction-level backtrack
							 (= level *TOP-LEVEL*))
		   (find-last-expanded new-node))))
	   (cond ((null last-node)
		  (change-abstraction-level (1+ level) new-node
					    (find-next-abstract-parent 
					     backtrack)))
		 (t (let ((result (expand-abstraction-level last-node)))
		      (cond ((eq 'success result)
			     (if (or *ABS-PRINT-FLAG*
				     *ABS-SOLUTION-FLAG*)
				     (print-solution new-node))
			     (change-abstraction-level (1- level)
						  new-node
						  nil))     
			    (t 
			     (change-abstraction-level (1+ level)
						       new-node result))))))))))
	   

(defun find-next-abstract-parent (node)
  (cond ((null node) t) 
	((node-abstract-parent node))
	(t (find-next-abstract-parent (car (node-children node))))))

;	   (if backtrack
;	       (let* ((new-node (node-abstract-parent node))
;		      (last-node (backtrack-within-abstraction-level
;				  (find-last-node new-node)
;				  (= level *TOP-LEVEL*))))
;		 (cond ((and last-node
;			     (expand-abstraction-level last-node))
;			(if *ABS-PRINT-FLAG* (print-solution new-node))
;			(change-abstraction-level (1- level)
;						  new-node
;						  nil))
;		       (t (change-abstraction-level (1+ level) 
;						    new-node
;						    t))))
;	       (let ((new-node (next-abstract-level node)))
;		 (cond ((expand-abstraction-level (find-last-expanded new-node))
;			(if *ABS-PRINT-FLAG* (print-solution new-node))
;			(change-abstraction-level (1- level)
;						  new-node
;						  nil))
;		       (t 
;			(change-abstraction-level (1+ level)
;						  new-node
;						  t))))))))

;
;
;
; This function takes a node within an level and starts expanding
; the level from that point.  It finds the next operator application
; at the higher level and then it tests for one of four possible cases.
; First if the given node is nil, it was backtracking and all possible 
; backtrack points were exhausted.  Second, there was no abstract node
; that needs to be expanded, then the abstraction level is completely
; expanded.  This only occurs after backtracking at the top level since 
; at other levels the third condition finished the expansion.  Third,
; the abstract operator has no children, which means it is the finish
; Operator that is being expanded.  Since this corresponds to the top level 
; goals, if there are any unmatched precondition then it should backtrack
; instead of trying to achieve them.  This is done by calling expand-subproblem
; with 'top-level' flag set.  Fourth, it expands the abstract operator.  
; If it is successful, the function is called recursively,  Otherwise it
; attempts to backtrack within the abstraction level.
;
(defun expand-abstraction-level (node)
  (let ((abstract-node (find-next-abstract-node node)))
          ; Backtrack points have been exhausted
    (cond ((null node)
	   (error "Cannot expand abstraction level with nil node."))
	  ; The top level has been re-expanded.
	  ((null abstract-node) 'success)
	  ; The last node is the top-level done
	  ((and (null (node-children abstract-node))
                (or (null *TOP-LEVEL-GOALS*)
		    (already-expanded-top-level-goals node)))
	   (cond ((expand-subproblem node abstract-node 'final) 'success)
		 (t (error "~%~%Failure...top-level ops not applicable"))))
          (t (let ((last-node 
		    (expand-subproblem node abstract-node 
				       (if (node-children abstract-node)
					   nil
					 'new-goals))))
	       (cond ((not (null last-node))
		      (expand-abstraction-level last-node))
		     (t (if *DBTWL* abstract-node
			  (let ((new-node
				 (backtrack-within-abstraction-level node nil)))
			    (cond ((null new-node) abstract-node)
				  (t (expand-abstraction-level new-node)))))))))
	  )))

;
; This is used to determine whether the top level goals for this abstraction
; level have already been expanded.  If so, then simply expand the finish 
; operator.
;
(defun already-expanded-top-level-goals (node)
  (cond ((null node) nil)
        ((node-top-level-goals node))
        (t (already-expanded-top-level-goals (node-parent node)))))
;
;
;
; This function finds the last decision point within an abstraction level and
; explores one of the other alternatives.  The function is given a node within
; the abstraction level and it backtracks from that point.  If there is a
; successful alternative, the function returns the last node expanded for 
; that alternative.  If all possible backtrack points are exhausted, it returns
; nil.  Note that if the criticality level is at the highest level, we want
; the system to expand the entire solution and not just solve the goal at 
; the node.  This is because there is no level above this to use to guide
; the expansion of this level.  Remember, that the top level is created 
; simply by calling Prodigy, while the other levels are expanded using the
; level above it.
;
(defun backtrack-within-abstraction-level (node top-level)
  (if *ABS-PRINT-FLAG* (format t "~%~%Backtracking within abstraction level..."))
  (cond ((null node) nil)
	((node-alternatives node)
	 (let ((last-node (cntrl node (find-problem-subgoal node))))
	   (cond ((and top-level
		       (not (null last-node)))
		  last-node)
		 ((not (null last-node))
		  (let ((abstract-node (find-next-abstract-node last-node)))
		    (setf (node-abstract-parent last-node) abstract-node)
		    (setf (node-abstract-child abstract-node) last-node)
		    last-node))
		 (t (backtrack-within-abstraction-level node top-level)))))
	((and *DBTWL* 
	      (node-parent node)
	      (node-abstract-parent (node-parent node)))
	 nil)
	(t (backtrack-within-abstraction-level (node-parent node) top-level))))


(defun find-problem-subgoal (node)
  (cond ((null node)(error "Problem subgoal not found."))
	((eq 'start-goal (alt-goal (node-generating-alt node)))
	 (node-current-goal node))
	(t (find-problem-subgoal (node-parent node)))))
;
;
;
; To expand a subproblem we first generate a new start node, determine
; the goal to be solved at this node and the initial state from which 
; to solve it.  The goal is taken from the abstract parent, or determined
; using abstract-goals if these are new-goals at this level.  Then it
; sets up the start node using setup-subproblem or setup-new-subproblem.
; Finally, if calls solve-subproblem to invoke prodigy in the appropriate
; manner.  Before doing so, it checks to see if this is the final node,
; and if so it first verifies that there are no unmatched conditions.
; If there are unmatched conditions, then the goals may need to be
; reordered, but this should be done at a higher level, not at this
; level since we should only have to expand the individual subproblems.
;

(defun expand-subproblem (last-node abstract-node top-level)
  (if *ABS-PRINT-FLAG* 
      (format t "~%~%~%Refining operator ... ~a (~a)" 
	      (abstract-operator abstract-node) top-level))
  (let* ((subgoal (cond ((eq 'new-goals top-level)
			 (load-abstract-goal (abstract-goals *GOAL-STATE*))
			 '(done))
			((eq 'final top-level)
			 (load-abstract-goal *GOAL-STATE*)
			 (abstract-goal abstract-node))
			(t (abstract-goal abstract-node))))
	 (node (if (eq 'new-goals top-level)
		   (setup-new-subproblem (node-state last-node) subgoal
					 abstract-node)
		   (setup-subproblem (node-state last-node) subgoal 
				     abstract-node))))
    (cond ((or (not (eq 'final top-level))
	       (null (alt-unmatched-conds (car (node-alternatives node)))))
	   (solve-subproblem node last-node abstract-node subgoal top-level)))))

;
;
; Sets up a new-subproblem that consists of the top-level goals that
; have not been expanded at this level.  Load-abstract-goal is used to
; store these conditions at preconditions of the *finish* operator.  The
; start node then contains done as the final goal.  This allows the system
; to correctly execute conjunctions of goals and do the appropriate top
; level reording.
;
(defun setup-new-subproblem (initial-state goal abstract-node)
  (let* ((new-node (get-new-node))
	 (bindings (abstract-bindings abstract-node))
	 (bindings-select (create-bindings-select-rule bindings)))
    (cond (*EBL-TRACE* 
	   (push `((*subproblem-name* ,(make-subproblem-name))
		   (*abstraction-level* ,*abstraction-level*)
		   (load-goal ,(get '*finish* 'preconds))
		   (load-start-state ,(state-closed-world initial-state))
		   (*op-reject-rule* nil)
		   (*binding-select-rule* ,(if bindings bindings-select)))
		 *rev-probset*)))
    (setf (node-generating-alt new-node)
	  (make-alternative :goal 'start-goal 
			    :op 'start-op 
			    :unmatched-conds (list goal)
			    :failed-cond nil 
			    :vars nil))
    (setf (node-state new-node) initial-state)
    (if bindings 
	(let ((bindings-select-rules *SCR-BINDINGS-SELECT-RULES*))
	  (unwind-protect
	      (progn (load-new-scntrl-rule bindings-select 'bindings-select)
		     (expand-node new-node))
	    (setq *SCR-BINDINGS-SELECT-RULES* bindings-select-rules)))
      (expand-node new-node))
;    (load-new-scntrl-rule bindings-select 'bindings-select)
;    (expand-node new-node)
;    (delete-control-rule bindings-select 'bindings-select)
    new-node))
;
;
;
; Sets up a subproblem based on an intermediate state in a more abstract
; space.  Operator selection and binding selection rules are created.
; The goal is stored in the unmatched conditions of the generating alt.
; And the initial state is stored in the new node.  Then the control rules
; are added to the system, expand-node then creates the set of alternatives,
; and the contorl rules are removed.
;
(defun setup-subproblem (initial-state goal abstract-node)
  (let ((operator-reject (create-operator-reject-rule
			  (abstract-operator abstract-node)))
	(bindings-select (create-bindings-select-rule
			  (abstract-bindings abstract-node)))
	(new-node (get-new-node)))
    (cond ((and *EBL-TRACE* (not (equal goal '(done))))
	   (push `((*subproblem-name* ,(make-subproblem-name))
		   (*abstraction-level* ,*abstraction-level*)
		   (load-goal ,goal)
		   (load-start-state ,(state-closed-world initial-state))
		   (*op-reject-rule* ,operator-reject)
		   (*binding-select-rule* ,bindings-select))
		 *rev-probset*)))
    (setf (node-generating-alt new-node)
	  (make-alternative :goal 'start-goal 
			    :op 'start-op 
			    :unmatched-conds (list goal)
			    :failed-cond nil 
			    :vars nil))
    (setf (node-state new-node) initial-state)
    (let ((op-reject-rules *SCR-OP-REJECT-RULES*)
	  (bindings-select-rules *SCR-BINDINGS-SELECT-RULES*))
      (unwind-protect
	  (progn (load-new-scntrl-rule operator-reject 'op-reject)
		 (load-new-scntrl-rule bindings-select 'bindings-select)
		 (expand-node new-node))
	(progn (setq *SCR-OP-REJECT-RULES* op-reject-rules)
	       (setq *SCR-BINDINGS-SELECT-RULES* bindings-select-rules))))
;    (delete-control-rule operator-reject 'op-reject)
;    (delete-control-rule bindings-select 'bindings-select)
    new-node))

;
;
;
; Calls prodigy on the start-node and then sets up all the appropriate
; links.
;
(defun solve-subproblem (node last-node abstract-node subgoal top-level)
  (setf (node-parent node) last-node)
  (setf (node-children last-node)(cons node (node-children last-node)))
  (if (eq 'new-goals top-level)(setf (node-top-level-goals node) t))
  (let ((result (run-prodigy node subgoal)))
    (cond ((null result))
	  ((not (eq 'new-goals top-level)) 
	   (setf (node-abstract-parent result) abstract-node)
	   (setf (node-abstract-child abstract-node) result)))
    result))

;
;
;
; Invokes prodigy by either running cntrl or calling generate-child-node.
; Generate-child-node can only be called if there is exactly one alternative
; with no unmatched conditions.  This effectively executes the operator at
; this node.  This is an efficiency hack that can be replaced by calling
; cntrl on these nodes.
;
(defun run-prodigy (node subgoal)
        ; This will usually be 1, but a variable can become bound at an
        ; intermediate level and generate a bunch of alternatives.
  (cond ;((> (length (node-alternatives node)) 1)
	; (error "More than one alternative at a top-level subproblem node."))
	((and (null *PRINT-NODE-NUMBERS*)
	      (= 1 (length (node-alternatives node)))
	      (null (alt-unmatched-conds (car (node-alternatives node)))))
	 (let ((child (generate-child-node node (car (node-alternatives node)))))
	   (setf (node-alternatives node) nil)
	   child))
	(t (cntrl node subgoal))))
;
;
;
; This creates a control rule to use the operator used in the abstract plan.
(defun create-operator-reject-rule (op)
  `(ABS-REJECT-OPERATOR ;The name of this rule appears in commands.lisp
    (lhs (and (current-node <node>)
	      (is-start-node <node>)
	      (candidate-op <node> <op>)
	      (not-equal ,op <op>)))
    (rhs (reject operator <op>))))

;
;
; This creates a control rule to use the bindings used in the abstract plan.
(defun create-bindings-select-rule (bindings)
  `(ABS-SELECT-BINDINGS ; The name of this rule appears in commands.lisp
    (lhs (and (current-node <node>)
	      (is-start-node <node>)))
    (rhs (select bindings ,bindings))))

;
;
; Finds the goal used in the abstract parent.
(defun abstract-goal (node)
  (alt-goal (node-generating-alt node)))

;
;
; Finds the operator used in the abstract parent.
(defun abstract-operator (node)
  (alt-op (node-generating-alt node)))

;
;
; Finds the bindings used in the abstract parent.
(defun abstract-bindings (node)
  (alt-vars (node-generating-alt node)))
;
;
;
; This function takes the first node in an abstraction level and returns the
; first node in the next abstraction level.  This node may already exist 
; because the level was expanded earlier and then the system had to backtrack
; across levels.  Otherwise we create a new node and copy the initial state
; for the problem from the abstract node.
;
(defun next-abstract-level (abstract-node)
  (cond ((node-abstract-child abstract-node))
	(t (if *ABS-PRINT-FLAG* 
	       (format t "~%~%~%Creating new abstraction level..."))
	   (let ((node (get-new-node))
		 (state (make-new-state-nm))
		 (sorted-state (sort (copy-list *START-STATE*) 'alphalessp-f)))
	     (setf (node-abstract-parent node) abstract-node)
	     (setf (node-abstract-child abstract-node) node)
;	     (setf (node-top-level-goals node) nil)
	     (set-closed-world state sorted-state)
	     (set-true-assertions state sorted-state)
	     (setf (node-state node) state)
	     (setf (node-generating-alt node)
		   (make-alternative :goal nil
				     :op nil
				     :unmatched-conds nil
				     :failed-cond nil
				     :vars nil))
	     node))))
;
;
;
; Takes a node in the abstraction level and goes to the last node in that level.
;
(defun find-last-node (node)
  (cond ((null (node-children node)) node)
	(t (find-last-node (car (node-children node))))))
;
;
;
; Takes a node in an abstraction level and finds the last node that was expanded
; AND corresponds to the the abstract solution in the level above.  Note that
; due to backtracking, operators may have been expanded that do not correspond
; to the current path at the level above.
;
(defun find-last-expanded (node)
  (cond ((null (node-children node)) node)
	((null (node-parent node))
	 (find-last-expanded (car (node-children node))))
	((node-abstract-parent node)
	 (cond ((eq (node-abstract-parent node)
		    (find-next-abstract-node (node-parent node)))
		(find-last-expanded (car (node-children node))))
	       (t (find-abstract-parent (node-parent node)))))
	(t (find-last-expanded (car (node-children node))))))
;
(defun find-abstract-parent (node)
  (cond ((null node)(error "Abstract parent not found!"))
	((node-abstract-parent node) node)
	(t (find-abstract-parent (node-parent node)))))
;
;
; This function is given a node at one level and it returns the next applied
; node at the level above.  It does this by first following the node-parent
; until it finds a node within the level that has an abstract parent.  Then
; it goes up a level and uses get-next-abstract-node to find the next
; applied node at the abstract level.
;
(defun find-next-abstract-node (node)
  (cond ((null node) nil)
	((node-abstract-parent node)
	 (get-next-abstract-node (car (node-children (node-abstract-parent node)))))
	(t (find-next-abstract-node (node-parent node)))))
;
; The funny test in the middle eliminates *finish* nodes that were inserted
; to expand new-goals at a given level.  However, at the next level there is
; no more work to be done on the "finish" node.  
(defun get-next-abstract-node (abstract-node)
  (cond ((null abstract-node) nil)
	((and (node-applied-node abstract-node) 
	      (not (and (node-children abstract-node)
			(equal '(done)(abstract-goal abstract-node)))))
	      abstract-node)
	(t (get-next-abstract-node (car (node-children abstract-node))))))
;
;
;
; Set the current criticality level.
;
(defun set-abstraction-level (level)
  (if (= *TOP-LEVEL* level)
      (setq *PREV-ABS-LEVEL* nil)
    (setq *PREV-ABS-LEVEL* (aref *ABS-HIERARCHY* 0 (1+ level))))
  (setq *ABSTRACTION-LEVEL* (aref *ABS-HIERARCHY* 0 level))
  (setq *ABSTRACTION-LEVEL-GOALS* (aref *ABS-HIERARCHY* 1 level))
  (if *DOMAIN-GRAPHICS*
      (display-domain-graphics-start-state)))



(defun print-no-solution-stats ()
  (terpri)
  (format t "~%Completed Failure~%")
  (cpu-time)
  (setq *SOL-LENGTH* 0)
  (setq *SOLUTION-DIST* nil)
  (setq *UNIQUE-NODES* (length *ALL-NODES*))
  (setq *NODE-DIST* nil))

;
;
;
; Prints out the final stats after abstraction planning.
;
;
(defun print-final-abs-stats (level-node final-node)
  "Prints the statistics out after the problem has been solved."
  (terpri) 
  (cpu-time)
  (unique-nodes level-node)
  (unique-nodes-in-each-level level-node)
  (format t "~%Total Nodes: ~A" (commas (length *ALL-NODES*)))
  (if final-node
      (let ((op-seq (find-op-seq (node-parent final-node))))
	(terpri)
	(format t "Solution Length: ~A" (setq *SOL-LENGTH* (sol-length op-seq)))
	(solution-in-each-level final-node)
	(print-op-list "Operator Sequence: " (setq *SOLUTION* op-seq))))
  (terpri)
  (if *DOMAIN-GRAPHICS* 
      (reset-domain-graphics-parameters)))
;
; Calculates the solution length ignoring *finish* operators.	  
(defun sol-length (solution)
  (cond ((null solution) 0)
	((not (eq '*finish* (caar solution)))
	 (1+ (sol-length (cdr solution))))
	(t (sol-length (cdr solution)))))

; Calculates the cpu time.
;
(defun cpu-time ()
  (format t "~%~%CPU time: ~,2F seconds"
    (coerce (setq *RUN-TIME* 
		  (/ (- (get-internal-run-time) *AB-START-TIME*)
		     internal-time-units-per-second)) 'float)))

(defun solution-in-each-level (node)
  (format t "~%Solution Distribution: ~a" 
	  (setq *SOLUTION-DIST* 
		(compute-solution-in-level 0
		 (reverse (solution-lengths node))))))

(defun solution-lengths (node)
  (cond ((null node) nil)
	(t (cons (sol-length (find-op-seq node))
		 (solution-lengths (node-abstract-parent node))))))


(defun compute-solution-in-level (last-node nodes)
  (cond ((null nodes) nil)
	(t (cons (- (car nodes) last-node)
		 (compute-solution-in-level (car nodes) (cdr nodes))))))


;
; Determines the number of unique nodes searched.  That is it does't
; count nodes that are brought done from the previous level. 
(defun unique-nodes (node)
  (format t "~%Unique Nodes ~a" 
	  (setq *UNIQUE-NODES* 
		(apply #'+ (new-nodes-in-each-level node)))))

(defun unique-nodes-in-each-level (node)
  (format t "~%Node Distribution: ~a" 
	  (setq *NODE-DIST* (reverse (new-nodes-in-each-level node)))))

;
;
; Determines the number of nodes at each level.
; Subtracts one for the beginnng abstract level.
;
(defun new-nodes-in-each-level (node)
  (cond ((null node) nil)
	(t (cons (count-new-nodes-in-level node)
		 (new-nodes-in-each-level (node-abstract-parent node))))))

;
;
; Counts all the new nodes in a particular level.
;
(defun count-new-nodes-in-level (node)
  (cond ((null node) 0)
	((node-abstract-parent node)
	 (cond ((node-parent node) ; Not the level node -- first node at a level
		(1- (count-new-nodes (node-children node))))
	       (t (count-new-nodes (node-children node))))) ; Just ignore
	(t (1+ (count-new-nodes (node-children node))))))

(defun count-new-nodes (children)
  (cond ((null children) 0)
	(t (+ (count-new-nodes-in-level (car children))
	      (count-new-nodes (cdr children))))))
;
;
;
; Print out the solution at the given abstraction level.  The node is the
; first node at an abstraction level.
;
(defun print-solution (node)
  (cond ((null node) nil)
	((not (node-applied-node node))
	 (print-solution (car (node-children node))))
	(t (terpri)
	   (print-op
	    (alt-op (node-generating-alt node))
	    (alt-vars (node-generating-alt node)))
	   (print-solution (car (node-children node))))))


;
; Loads an abstract goals.
(defun load-abstract-goal (goal)
  (setq *PROBLEM-GOAL* (abstract-exp goal))
  (if *abs-print-flag* (format t "~%  Goal State:  ~A" *PROBLEM-GOAL*))
  (setf (get '*FINISH* 'preconds) *PROBLEM-GOAL*)
  (setf (get '*FINISH* 'renamed-vars)(extract-vars *PROBLEM-GOAL*))
  (add-lpreconds '(*FINISH*)))


;
; Loads an abstract initial state.
(defun load-abstract-start-state (start-lits)
  (setq *START-STATE* (abstract-lits start-lits)))

;
; Finds the goals in the exp that are specific to this particular abstraction
; level.
;
(defun abstract-goals (exp)
  (let ((*ABSTRACTION-LEVEL* *ABSTRACTION-LEVEL-GOALS*))
     (cond ((null *ABSTRACTION-LEVEL*) exp)
	   (t (abstract-exp exp)))))

; ----------------------------------------------------------------------


(defun load-new-abstraction-level (level)
  (dolist (op *OPERATORS*) 
    (if (member (car op)(aref *ABS-HIERARCHY* 2 level))
	(add-abstract-operator op)))
  (dolist (inference-rule *INFERENCE-RULES*) 
    (if (member (car inference-rule)(aref *ABS-HIERARCHY* 2 level))
	(add-abstract-inference inference-rule))))


(defun add-abstract-operator (op)
  (let ((name (car op)))
    (setf (get name 'preconds)
	  (abstract-exp (get name 'abs-preconds)))
    (setf (get name 'effects)
	  (abstract-effects (get name 'abs-effects)))
    (setf (get name 'vars)(get name 'abs-vars))
    (setf (get name 'all-vars)(get name 'abs-all-vars))
    (setf (get name 'params)(get name 'abs-vars))
    (make-add-and-delete-lists name (get name 'effects))
    (add-lpreconds op)
    ))


(defun add-abstract-inference (inference-rule)
  (let ((name (car inference-rule)))
    (setf (get name 'preconds)
	  (abstract-exp (get name 'abs-preconds)))
    (setf (get name 'effects)
	  (abstract-effects (get name 'abs-effects)))
    (setf (get name 'vars)(get name 'abs-vars))
    (setf (get name 'all-vars)(get name 'abs-all-vars))
    (setf (get name 'params)(get name 'abs-vars))
    (make-add-and-delete-lists name (get name 'effects))
    (add-lpreconds inference-rule)
    ))





;
; ----------------------------------------------------------------------
; This code is used for loading problems in order to
; do abstraction planning.  The variable *ABSTRACTION-LEVEL* must be set to the 
; conditions in the abstract level.  If it is null then there is no 
; abstraction.  
;
(defun abstract-exp (exp)
  (cond ((null *ABSTRACTION-LEVEL*) exp)
	((eq 'and (car exp))
	 (abstract-and (cdr exp)))
	((eq 'or (car exp))
	 (abstract-or (cdr exp)))
	((eq 'exists (car exp))
	 (abstract-and (cddr exp)))
	((eq 'forall (car exp))
	 (abstract-forall (cdr exp)))
	(t (car (abstract-lits (list exp))))))

(defun abstract-and (exp)
  (let ((new-exp (abstract-lits exp)))
    (cond ((null new-exp) nil)
	  ((= 1 (length new-exp))
	   (car new-exp))
	  (t (cons 'and new-exp)))))

(defun abstract-or (exp)
  (let ((new-exp (abstract-lits exp)))
    (cond ((null new-exp) nil)
	  ((equal exp new-exp)
	   (cons 'or new-exp))
	  ((no-possible-subgoals new-exp) nil)
	  (t (error "Attempt to abstract 1 or more disjunctions.  Must
abstract all or none.")))))

(defun no-possible-subgoals (exp)
   (cond ((null exp))
	 ((or (is-static (car exp))
	      (not-subgoalable (car exp)))
	  (no-possible-subgoals (cdr exp)))))


(defun abstract-forall (exp)
  (let ((vars (car exp))
	(new-exp (abstract-lits (cdr exp))))
    (cond ((null new-exp) nil)
	  ((equal (cadr exp) (car new-exp))
	   (cons 'forall (cons vars new-exp)))
	  ((only-static new-exp) nil)
	  (t (error "Attempt to abstract the  generator in a forall.")))))

(defun only-static (exp)
  (cond ((null exp) t)
	((is-static (car exp))
	 (only-static (cdr exp)))))

(defun is-static (lit)
  (member (extract lit) *static* :test #'equal))


; ----------------------------------------------------------------------

(defun abstract-lits (lits)
  (cond ((null *ABSTRACTION-LEVEL*) lits)
	((null lits) nil)
	((member (caar lits) '(and or exists forall))
	 (let ((exp (abstract-exp (car lits))))
	   (if (null exp)
	       (abstract-lits (cdr lits))
	       (cons exp (abstract-lits (cdr lits))))))
	((or (member (extract-instance (strip-negation (car lits))) 
		     *ABSTRACTION-LEVEL* :test #'equal)
	     (is-static (strip-negation (car lits))))
	 (cons (car lits)(abstract-lits (cdr lits))))
	(t (abstract-lits (cdr lits)))))

(defun abstract-effects (effects)
  (cond ((null *ABSTRACTION-LEVEL*) effects)
	((null effects) nil)
	((member (extract (strip-negation (second (car effects)))) 
		 *ABSTRACTION-LEVEL* :test #'equal)
	 (cons (car effects)(abstract-effects (cdr effects))))
	(t (abstract-effects (cdr effects)))))

;;
;; Stuff for saving ebl problems
;;
(defun make-subproblem-name ()
    (declare 
	(special *prob-string*)
	(special *subprob-counter*)
        (fixnum  *subprob-counter*))
    (if (not (boundp '*subprob-counter*))
	(setq *subprob-counter* 1)
	(incf *subprob-counter*))
    (if (not (boundp '*prob-string*))
	(setq *prob-string* "SP"))
   (let ((name  (intern (concatenate 'string *prob-string*
		 (princ-to-string *subprob-counter*)))))
	name))

(defun write-abs-probset (file-nm &optional dontclear)
  (with-open-file (prt file-nm :direction :output :if-exists :append
		       :if-does-not-exist :create)
        (princ "(setq *probs-for-ebl* (quote " prt)
        (pprint (reverse *rev-probset*) prt)
        (princ "))" prt))
   (if (not dontclear)
       (setq *rev-probset* nil)))


(defun write-solutions (file-nm)
  (with-open-file (prt file-nm :direction :output :if-exists :append
		       :if-does-not-exist :create)
        (pprint (reverse *solutions*) prt)))
