;==========================================================================
; File:   graphics.lisp     Version: 0.0                  Created: 3/2/88
;
; Locked by:  none.                                      Modified: 4/4/88
; 
; Purpose:    To standardize domain graphics for Prodigy.
;
;==========================================================================


(eval-when (compile) 
  (load-path *planner-path* "pg-x11.lisp"))


(require 'pg-system)
(use-package (symbol-name 'pg))


(proclaim
 '(special *DOMAIN-WINDOW* *FINISH*   *NODE-MSG-X*    *NODE-MSG-Y*
	   *STATE-MSG-X*   *STATE-MSG-Y*   *START-STATE*   *WIDTH*
	   *DOMAIN-STATE*  *DOMAIN-NODE*  *DOMAIN-GRAPHICS*))
   ; *CHAR-WIDTH* and *OBJECT-SCALE* must be init'ed here dummy value to 
   ; keep first config event from crashing.
(defvar *CHAR-WIDTH* 1 "Width in pixels of a character")
(defvar *OBJECT-SCALE* 1 "Scale of object relative to grid size.")


(proclaim
 '(special *GRAPHICS-INFO* *ROOMS* *X-DIM* *Y-DIM* *Z-DIM* *CHAR-HEIGHT*
	   *CHAR-WIDTH* *DOMAIN-HEIGHT* *DOMAIN-WIDTH* *ORGIN-Y*  *ORGIN-X* 
	   *SUB-HEIGHT* *SUB-WIDTH* 
	   *RESOL-X* *RESOL-Y* 	; number of pixels between grid points
	   *HALF-RES-X* *HALF-RES-Y* ; half of *RESOL-X* *RESOL-Y*
           *DOOR-INFO* 		; a-list made from *DOORS*
	   *DOORS* ; data file created by the users in functions.lisp
	   *VECTOR-X* *VECTOR-Y* ; These vectors map grid points to pixels
	   *ROBOT*		; The location at which the robot is.
	   *LOC-MAP* 		; The map of all grid points &  their contents
	   *ROOM-INFO*		; An a-list representatoin of *ROOMS*
	   *OBJECT-INFO*	; An a-list of objects and their locations
	   *CHARS-IN-OBJECT*	; number of characters that can label an ob
           *HOLDING*		; is the graphic print name for the object
				; that robot is holding.  It is a subset of
				; the lisp print name that will fit into 
				; the object square.  nil if not holding.
	   *OBJECT-SCALE*	; The scale of the sides of the object
				; relative to the grid size.
	   *DOOR-STATUS*	; is door open or closed, locked or unlocked
  ))

;==========================================================================
;                      Domain  Dependent  Functions
;==========================================================================

;; RESET-DOMAIN-GRAPHICS-PARAMETERS should be called when a new problem is 
;; loaded. This function sets graphics variables to null. In particular, 
;; the variables  *NODE-MSG-X*, *NODE-MSG-Y*, which determine the location
;; of the node message in the graphics window and *STATE-MSG-X*, and 
;; *STATE-MSG-Y* which determine the location of the state message in the
;; graphics window should be set to null in this function.
;;

(defun reset-domain-graphics-parameters ()
  (psetq *NODE-MSG-X*  NIL 
	 *NODE-MSG-Y*  NIL
	 *STATE-MSG-X* NIL 
	 *STATE-MSG-Y* NIL
	 *WIDTH*       NIL
	 *HOLDING*     NIL
))

;--------------------------------------------------------------------------

;; DETERMINE-DOMAIN-GRAPHICS-PARAMETERS sets the variables used in drawing 
;; the domain graphics for a given problem. This routine should also 
;; determine the location of the node and state messages: i.e., set the
;; variables (*NODE-MSG-Y*, *NODE-MSG-X*, *STATE-MSG-X*, *STATE-MSG-Y*).
;;

(defun determine-domain-graphics-parameters (problem)
  (setq *WIDTH* 0
	*CHAR-HEIGHT* (pg-text-height *DOMAIN-WINDOW* "A")
	*CHAR-WIDTH*  (pg-text-width *DOMAIN-WINDOW* "A")
	*NODE-MSG-X*  10
	*NODE-MSG-Y*  (+ *CHAR-HEIGHT* 2)
	*STATE-MSG-X* 10
	*STATE-MSG-Y* (+ *NODE-MSG-Y* *CHAR-HEIGHT*)
				; this make-array leaves the old ones 
				; lying around.  inefficient.
	*LOC-MAP* (make-array (list *X-DIM* *Y-DIM*) :initial-element nil)
	*DOOR-INFO* (build-doors-info *DOORS*)
	*ROOM-INFO* (build-room-info *ROOMS*)
	*OBJECT-INFO* nil
	*OBJECT-SCALE* .6
	*ROBOT* (make-object-prodigy)
	*HOLDING* nil
	*DOOR-STATUS* (build-door-status *DOORS*)
  )
  (calc-window-dependents	*DOMAIN-DIMENSION-X*
				*DOMAIN-DIMENSION-Y*
				*CHAR-WIDTH*
				*OBJECT-SCALE*)
)

(defun calc-window-dependents (width height char-width scale)
   "Calculates all of the window size dependent factors of the domain
    window."
(setf	*DOMAIN-HEIGHT* height
	*DOMAIN-WIDTH* width
	*ORGIN-Y* (truncate *DOMAIN-HEIGHT* (/ .9))
	*ORGIN-X* (truncate *DOMAIN-WIDTH* (/ .1))
	*SUB-HEIGHT* (truncate *DOMAIN-HEIGHT* (/ .8))
	*SUB-WIDTH* (truncate *DOMAIN-WIDTH* (/ .9))
	*RESOL-X* (truncate *SUB-WIDTH* *X-DIM*)
	*RESOL-Y* (truncate *SUB-HEIGHT* *Y-DIM*)
	*HALF-RES-X* (truncate *RESOL-X* 2)
	*HALF-RES-Y* (truncate *RESOL-Y* 2)
	*VECTOR-X* (initialize-vector *X-DIM* *RESOL-X* *ORGIN-X*)
	*VECTOR-Y* (initialize-vector *Y-DIM* (- *RESOL-Y*) *ORGIN-Y*)
	*CHARS-IN-OBJECT* (truncate (1- (* *HALF-RES-X* scale 2)) 
								char-width)
))    


(defstruct object-prodigy
	(loc nil)
	(next-to nil)
	(at nil)
	(in-room nil)
	(drawnp nil)
	(resolvedp nil)
)

; All the macros will compatible with setf
(defmacro get-location (ob)
    `(object-prodigy-loc (cdr (assoc ,ob *OBJECT-INFO*)))
)

(defmacro get-next-to-list (ob)
    `(object-prodigy-next-to (cdr (assoc ,ob *OBJECT-INFO*)))
)

(defmacro get-at (ob)
     `(object-prodigy-at (cdr (assoc ,ob *OBJECT-INFO*)))
)

(defmacro get-in-room (ob)
     `(object-prodigy-in-room (cdr (assoc ,ob *OBJECT-INFO*)))
)

(defmacro get-drawnp (ob)
     `(object-prodigy-drawnp (cdr (assoc ,ob *OBJECT-INFO*)))
)

(defmacro get-resolvedp (ob)
      `(object-prodigy-resolvedp (cdr (assoc ,ob *OBJECT-INFO*)))
)

(defmacro get-grid-object (loc)
     `(aref *LOC-MAP* (location-x ,loc) (location-y ,loc))
)

(defstruct door-status
	(open nil)
	(locked nil)
)

; ADD-OBJECT adds an object to the a-list *OBJECT-INFO* if
; it is not already added.

(defun add-object (object-name)
	(if (not (assoc object-name *OBJECT-INFO*))
	    (setf *OBJECT-INFO* (acons object-name
					(make-object-prodigy)
					*OBJECT-INFO*))))

; There are two routines to initialize door stuff.  BUILD-DOOR-INFO builds
; an a-list that associates a door name with a gird location.
; BUILD-DOOR-STATUS builds an a-list that associates each door name with
; a door-status data structure.

;  The following routines store the door and room data in a-lists for easy 
;  access later in the program.

(defun build-doors-info (lst &optional a-list)
    (cond ((null lst) a-list)
	  (t (setq a-list 
		(acons (door-name (car lst))
		       (car lst)
		    	a-list))
	       (build-doors-info (cdr lst) a-list))
    )
)

(defun build-door-status (lst &optional a-list)
    (cond ((null lst) a-list)
	  (t (setq a-list 
		(acons (door-name (car lst))
		       (make-door-status)
		    	a-list))
	       (build-door-status (cdr lst) a-LIST))
    )
)

(defun get-door (door)
	(cdr (assoc door *DOOR-INFO*)))

(defun is-door-p (door)	(get-door door))

(defun build-room-info (lst &optional a-list)
   (cond ((null lst) a-list)
         (t (setq a-list
   		(acons (location-room (caar lst))
			(car lst) a-list))
	    (build-room-info (cdr lst) a-list))))

;--------------------------------------------------------------------------

; calc-point functions return a pixel point given a grid-point.

(defmacro calc-point-x (i)
	`(svref *VECTOR-X* ,i))

(defmacro calc-point-y (j)
	`(svref *VECTOR-Y* ,j))

;--------------------------------------------------------------------------
;; DRAW-DOMAIN-BACKGROUND uses domain graphics parameters to draw the
;; rear plane for the given domain -- drawn before any domain objects
;; are added to the domain graphics window. 
;;

(defun draw-domain-background ()
;This section draws the grid


	(dotimes (i *X-DIM*)
	   (dotimes (j *Y-DIM*)
		  (draw-grid-marker (calc-point-x i) (calc-point-y j))))

;This section draws a box around the whole domain picture.
#|
(princ "Paused") (read-char)
; This draws a box around the whole grid.
		(pg-frame-rect *DOMAIN-WINDOW*
			*ORGIN-X*
		 	(- *ORGIN-Y* *SUB-HEIGHT*)
			(+ *ORGIN-X* *SUB-WIDTH*)
			*ORGIN-Y*)
|#

	(draw-all-rooms *ROOMS*)
)

; DRAW-ALL-ROOMS draws all of the rooms in the domain.

(defun draw-all-rooms (room-list)
	(cond ((null room-list) t)
	      (t (draw-room (car room-list))
		 (draw-all-rooms (cdr room-list)))
	)
)

; DRAW-ROOM draw a room in the window

(defun draw-room (room)

; write the label of the room.
   (pg-write-text *DOMAIN-WINDOW* 
		(+ 2 (- (calc-point-x (location-x (car room))) *HALF-RES-X*))
		(- (+ *HALF-RES-Y* (calc-point-y (location-y (car room)))) 2)
		(symbol-name (location-room (car room))))


	(pg-frame-rect *DOMAIN-WINDOW*
		(- (calc-point-x (location-x (car room))) *HALF-RES-X*)
	        (- (calc-point-y (location-y (cadr room))) *HALF-RES-Y*)
		(+ *HALF-RES-X* (calc-point-x (location-x (cadr room))))
		(+ *HALF-RES-Y* (calc-point-y (location-y (car room))))
   )
)


; DRAW-GRID-MARKER draws the symbol that marks a grid-point
; always in the *DOMAIN-WINDOW*
(defun draw-grid-marker (x y)
;	(pg-draw-line *DOMAIN-WINDOW* x y x y)
	(pg-draw-line *DOMAIN-WINDOW* x (1+ y) x (1- y))
	(pg-draw-line *DOMAIN-WINDOW* (1+ x) y (1- x) y)
)



;--------------------------------------------------------------------------

;; DELETE-DOMAIN-GRAPHIC-OBJECTS erases objects from the domain window. 
;; The argument to this function is a list containing predicates to delete.
;; The function must parse the relevant graphics predicates and devise some
;; method for removing them from the domain window.
;;

(defun delete-domain-graphic-objects (state-preds)
	(cond ((null state-preds) nil)
	      (t (case (caar state-preds)
		   ('at (erase-a-thing-at (cdar state-preds)))

		   ('holding (erase-holding))

	           ('at (unset-a-thing-at (cdar state-preds)))

		   ('in-room (unset-thing-in-room (cdar state-preds)))

		   ('inroom (unset-thing-in-room (cdar state-preds)))

		   ('next-to (unset-thing-next-to (cdar state-preds)))

		   ('door-open nil)

		   ('door-closed nil)

		   ('statis nil)

		   (otherwise (format t "~%delete:Unknown graphic object predicate: ~A" (car state-preds))))

	      (delete-domain-graphic-objects (cdr state-preds)))
	)
)

(defun unset-a-thing-at (thing)
      (let ((ob (car thing)))
	(if (eq ob 'robot)
	  (progn
	    (setf (object-prodigy-at *ROBOT*) nil
		  (object-prodigy-resolvedp *ROBOT*) nil)
	    (erase-robot (object-prodigy-loc *ROBOT*))
	  )
          (progn
	    (setf (get-at ob) nil
		  (get-resolvedp ob) nil)
	    (erase-object (list 'dummy (get-location ob)))
	  )
	)
      )
)

(defun unset-thing-in-room (thing)
      (let ((ob (car thing)))
        (if (eq ob 'robot)
          (progn
            (setf (object-prodigy-in-room *ROBOT*) nil
                  (object-prodigy-resolvedp *ROBOT*) nil)
            (erase-robot (object-prodigy-loc *ROBOT*))
	  )
          (progn
            (setf (get-in-room ob) nil
                  (get-resolvedp ob) nil)
            (erase-object (list 'dummy (get-location ob)))
	  )
	)
      )
)

(defun unset-thing-next-to (thing)
      (let ((ob1 (first thing))
	    (ob2 (second thing)))
        (if (eq ob1 'robot)
          (progn
            (setf (object-prodigy-next-to *ROBOT*) nil
                  (object-prodigy-resolvedp *ROBOT*) nil)
            (erase-robot (object-prodigy-loc *ROBOT*))
	  )
          (progn
            (setf (get-next-to-list ob1) (remove ob2 (get-next-to-list ob1))
                  (get-resolvedp ob1) nil)
            (erase-object (list 'dummy (get-location ob1)))
	  )
	)
      )
)





; --------The following delete routines are for the realworld domain

; ERASE-HOLDING turns of the holding switch, redraws the robot so that the 
; object is erased and then erases the words at the top of the screen.
; The business with the words ought to be removed eventually.

(defun erase-holding ()
  (setq *HOLDING* nil)

  (erase-robot (object-prodigy-loc *ROBOT*))
  (draw-robot (object-prodigy-loc *ROBOT*))

	(pg-write-text *DOMAIN-WINDOW*
			*STATE-MSG-X*
			(+ *STATE-MSG-Y* *CHAR-HEIGHT*)
			"                        ")
)

(defun erase-a-thing-at (thing)
	(case (car thing)
	    ('robot (erase-robot (cadr thing)))
	    (otherwise (erase-object thing))
	)
)
; --------- The following functions erase stripsworld objects
(defun erase-thing-in-room (thing)

	(cond 
	      ((eq (car thing) 'robot) (erase-robot *ROBOT*))
	      (t (format t "Cannot erase ~A" (car thing)))
	)
)

;ERASE-THING-NEXT-TO
;In this function we must erase a robot or an object that is next to something
;else.  If the next-to predicate has robot in it then the robot should be 
; erased because it is the only thing that can move (unless a pick-up was
; done, but in this case the robot can be erased also because I redraw the
; robot in pick up.
(defun erase-thing-next-to (thing)
	(cond 
	     ((or (eq (car thing) 'robot) 
	          (eq (cadr thing) 'robot)) (erase-robot-next-to (cadr thing)))
	     ((assoc (car thing) *OBJECT-INFO*) (erase-object-next-to thing))
	     (t (format t "~%Unable to draw thing next to thing."))
	)
)

; ERASE-ROBOT-NEXT-TO is much simpler then the the corresponding drawing
; function because the location of the robot is already known and 
; in the *ROBOT* global variable.

(defun erase-robot-next-to (object)
    (erase-robot *ROBOT*)
)

(defun erase-object-next-to (object)
     (erase-object (cdr (assoc (car object) *OBJECT-INFO*)))
)

(defun erase-robot (robot-loc)
	(erase-small-square robot-loc .8)
)

(defun erase-object (object)
	(erase-small-square (cadr object) *OBJECT-SCALE*)
)

(defun erase-small-square (loc scale)
	(let* ((half-side-x (truncate (* *HALF-RES-X* scale)))
	      (x (- (calc-point-x (location-x loc)) half-side-x))
	      (half-side-y (truncate (* *HALF-RES-Y* scale)))
	      (y (+ (calc-point-y (location-y loc)) half-side-y)))

		(pg-erase-rect *DOMAIN-WINDOW*
			x
			(- y (* half-side-y 2))
			(+ x (* half-side-x 2))
			y)
	)
)
;--------------------------------------------------------------------------
;This is a  function to get rid of useless predicates so I won't
; have to look at them in the trace of add-domain-graphic-objects-ii
; It is primarily used in stripsworld which produces lots of non-graphical
; predicates.  It also removes the in-room predicate if an at predicate 
; exists for the same object and the same room.



(defun add-domain-graphic-objects (state-preds)
  (add-domain-graphic-objects-II 
        (convert-ats-to-location
	   (remove-useless 
		(process-objects state-preds)))))

; REMOVE-USELESS removes predicates from the state predicate list 
; that have not use in graphics.

(defun remove-useless (lst)
  (let ((non-graphical-preds '(carriable is-key is-door object is-object
				arm-empty dr-to-rm is-room connects  
					is-type	pushable)
				   ))

	(remove-if #'(lambda (x) (member (car x) non-graphical-preds)) lst)
  )
)     

; PROCESS-OBJECTS puts any new objects into the a-list *OBJECT-INFO*.
; In the ideal case this would be called only once, but there is not a
; good way to determine if a call to add-domain-graphic-objects is the
; first call of a problem solving run.  It must return the list it is
; passed

(defun process-objects (lst)
	(cond ((null lst) nil)
	      (t (if (object-pred-p (car lst))
		     (add-object (cadar lst)))
	      	 (process-objects (cdr lst)))
	)
	lst
)

; Convert-ats-to-location will return a list of all of the predicates passed
; to it, applying convert-pred on those predicates that are AT predicates 
; and returning that result in place of the original at predicate.
; This converts the non-location-p locations of stripsworld to a location-p 
; location like those used in realworld.

(defun convert-ats-to-location (preds)
	(cond ((null preds) nil)
	      ((not (eq (caar preds) 'at)) (append (list (car preds))
						(convert-ats-to-location (cdr preds))))
	      (t (append (convert-pred (car preds)) (convert-ats-to-location (cdr preds))))
	)
)
; If the location is of type location-p then the predicate is returned.
; If the location isnot of type location-p then the predicate is converted
; and that result is returned.
(defun convert-pred (pred)
   (if (not (location-p (third pred)))
        (list	(list (first pred)
	          (second pred)
	          (make-location :room (convert-loc-to-room (third pred) (fourth pred))
			     :x (third pred)
			     :y (fourth pred)
	          )
		)
	)
        (list pred)
   )
)

(defun object-pred-p (predicate)
	(or (eq (car predicate) 'object)
	    (eq (car predicate) 'is-object)
	    (and (eq (car predicate) 'is-type) ; stripsworld
		 (eq (third predicate) 'object))
	)
)

;; ADD-DOMAIN-GRAPHIC-OBJECTS adds objects to the domain window. This 
;; function is the complement of (delete-domain-graphic-objects). The 
;; argument to this function may contain predicates irrelevant to graphics, 
;; therefore the function must devise a method to parse relevant predicates
;; and add the appropriate objects to the domain window. 
;; 
;;  The 
;; 

(defun add-domain-graphic-objects-II (state-preds)
	(cond ((null state-preds) (resolve-robot)
				  (resolve-conflicts)
				  (draw-objects *OBJECT-INFO*)
				  (new-draw-robot))
	      (t (case (caar state-preds)

			('door-open (set-open-door (cadar state-preds)))

			('dr-open (set-open-door (cadar state-preds)))

			('door-closed (set-closed-door (cadar state-preds)))

			('unlocked (set-unlocked-door (cadar state-preds)))

			('locked (set-locked-door (cadar state-preds)))

			('light-off (turn-off-light (cadar state-preds)))

			('light-switch (draw-switch (cadar state-preds)))

			('at (set-a-thing-at (cdar state-preds)))

			('holding (make-holding (cadar state-preds)))

			('statis (draw-door-state (cdar state-preds)))

			('in-room (set-thing-in-room (cdar state-preds)))

			('inroom (set-thing-in-room (cdar state-preds)))

			('next-to (set-thing-next-to (cdar state-preds)))

			(otherwise (format t "~%Unknown graphic predicate:  ~A" (car state-preds)))
	       )
              (add-domain-graphic-objects-II (cdr state-preds)))
	)
)

(defun set-a-thing-at (ob-loc)
     (let ((object (car ob-loc)))
       (if (eq (car ob-loc) 'robot)
	   (setf (object-prodigy-at *ROBOT*) (cadr ob-loc))

           (setf (get-at object) (cadr ob-loc)
	         (get-drawnp object) nil
	         (get-resolvedp object) nil
	   )
	)
     )
)

(defun set-thing-in-room (ob-rm)
     (let ((object (car ob-rm)))
       (if (eq (car ob-rm) 'robot)
	   (setf (object-prodigy-in-room *ROBOT*) (cadr ob-rm))
	   (setf (get-in-room object) (cadr ob-rm)
	         (get-drawnp object) nil
	         (get-resolvedp object) nil
	   )
       )
     )
)
; NOTE:  The next to list isn't a list for the robot.
(defun set-thing-next-to (ob-ob)
     (let ((object (car ob-ob)))

        (if (eq object 'robot)
	   (progn
 	    (setf (object-prodigy-next-to *ROBOT*) (cadr ob-ob))
	    (erase-robot (object-prodigy-loc *ROBOT*))
	   )

	   (progn
	      (push (cadr ob-ob) (get-next-to-list object))
	      (setf (get-drawnp object) nil
	            (get-resolvedp object) nil)
	      (erase-small-square (get-location object) 
					*OBJECT-SCALE*)
	   )
	)
     )
)

; MAKE-HOLDING must do a number of things.  It sets the HOLDING variable
; to the name of the object being held.  It draws the object in the position
; at which it is being held and erases the old object (because not predicate
; will appear in the erase list to eliminate it.)

; This will fail if you can pick-up and move with the same operators
; I don't believe this is a problem, but if it is the new location must
; be added before make-holding is called.

; Also, only one object may be held at a time.

(defun make-holding (object)

 (unless *HOLDING*
	(let ((sym-nam (symbol-name object)))
            (setf *HOLDING* (cons object
			(subseq sym-nam (max 0 
				(- (length sym-nam) *CHARS-IN-OBJECT*))))
	)
	
	(erase-object (list object (get-location object)))
;	(draw-object (list  object *ROBOT*))
;	(draw-robot (object-prodigy-loc *ROBOT*)); because next-to in delete-domain
 				; has erased it.
	(pg-write-text *DOMAIN-WINDOW*
			*STATE-MSG-X*
			(+ *STATE-MSG-Y* *CHAR-HEIGHT*)
			(format nil "Holding: ~A"  *HOLDING*)
	)
	)
 )
)

; ------------------------------------------------------------------------
; The following draw predicates are for the realworld domain

;DRAW-DOOR-STATE will draw a door in the state determined by the 
; predicate.  This function is not used by realworld, but by
; extend-strips and stripsworld
(defun draw-door-state (state)
	(ecase (cadr state)
		('closed (set-closed-door (car state)))
		('open (set-open-door (car state)))
	)
)

; CLEAR-POINT-NEXT-TO is copied from the draw-next-to code but it doesn't 
; do any drawing.
   

(defun clear-point-next-to (loc)




   (let* ((room (location-room loc))
	  (x-pos (location-x loc))
	  (y-pos (location-y loc)))

	(cond
		 ((good-point-p (1+ x-pos) y-pos room)
			(make-location :room room
				:x (1+ x-pos)
				:y y-pos
				:z (location-z loc)))
                ((good-point-p (1- x-pos) y-pos room)
                        (make-location :room room
                                :x (1- x-pos)
                                :y y-pos
                                :z (location-z loc)))

                ((good-point-p x-pos (1+ y-pos) room)
                        (make-location :room room
                                :x x-pos
                                :y (1+ y-pos)
                                :z (location-z loc)))

                ((good-point-p x-pos (1- y-pos) room)
                        (make-location :room room
                                :x x-pos
                                :y (1- y-pos)
                                :z (location-z loc)))

		(t (format t "~%Graphics error:  Cannot find point next to ~A"
							 loc))
	)
   )
)    
(defun good-point-p (x y room)
	(let ((room-data (cdr (assoc room *ROOM-INFO*))))
		(and (>= x (location-x (first room-data)))
		     (>= y (location-y (first room-data)))
		     (<= x (location-x (second room-data)))
		     (<= y (location-y (second room-data)))
		     (clear-p x y 'robot))))

(defun find-position (room name)
	(cond ((hueristic-1 room name))
	      ((hueristic-2 room name))
	      (t (format t "~%find-position has failed."))
	)
)

(defun hueristic-1 (room name)
     (do ((i (1+ (location-x (room-loc1 room))) (1+ i)) ;  where to start in x
	  (end-x (1- (location-x (room-loc2 room)))) ;  where to end in x
	  (x-done nil)			; used to stop before count is up.
	  (result nil))
	  ((or x-done (> i end-x)) result)
          (do ((j (1+ (location-y (room-loc1 room))) (1+ j))
	       (end-y (1- (location-y (room-loc2 room))))
	       (y-done nil))
	       ((or y-done (> j end-y)))
;    Now start body of loop

		   (when (clear-p i j name)
			 (setq result (make-location :room (location-room (car room)) 
						     :x i :y j :z 0))
			 (setq x-done t y-done t)
		   )
	  )			; do in y
     )				; do in x
) 				;defun

(defun hueristic-2 (room name)
     (do ((i (location-x (room-loc1 room)) (1+ i)) ;   where to start in x
          (end-x (location-x (room-loc2 room))) ;   where to end in x
          (x-done nil)                  ;  used to stop before count is up.
          (result nil))
          ((or x-done (> i end-x)) result)
          (do ((j (location-y (room-loc1 room)) (location-y (room-loc2 room)) )
               (end-y 0 (1+ end-y))
               (y-done nil))
               ((or y-done (>  end-y 2)))
                                ;     Now start body of loop

                   (when (clear-p i j name)
                         (setq result (make-location 
					:room (location-room (room-loc1 room))
					:x i :y j :z 0))
                         (setq x-done t y-done t)
		   )
	  )                     ;  do in y
     )                          ;  do in x

; This do loop is the same as the previous except that the 
; location-x and location-y calls have been switched and also
; the i j indices have been switched in clear-p and make-location

     (do ((i (location-y (room-loc1 room)) (1+ i)) ;   where to start in x
          (end-x (location-y (room-loc2 room))) ;   where to end in x
          (x-done nil)                  ;  used to stop before count is up.
          (result nil))
          ((or x-done (> i end-x)) result)
          (do ((j (location-x (room-loc1 room)) (location-x (room-loc2 room)) )
               (end-y 0 (1+ end-y))
               (y-done nil))
               ((or y-done (> end-y 2)))
			;     Now start body of loop

                   (when (clear-p j i name)
                         (setq result (make-location 
					:room (location-room (room-loc1 room))
					    :x j :y i :z 0))
                         (setq x-done t y-done t)
		   )
	  )                     ;  do in x
     )                          ;  do in y
)                               ; defun


;  this access functions should be roomdata.lisp or functions.lisp
(defun room-loc1 (room)
   (first room))

(defun room-loc2 (room)
   (second room))

; CLEAR-P is a predicate that returns non-nil a location in the grid is 
; clear.  Nil if it is occupied.  The name is the name of an object.  If
; the object is already at that point then the point is considered clear.

(defun clear-p (x y name)
	(or (null (aref *LOC-MAP* x y)) (eq name (aref *LOC-MAP* x y)))
)



; SET-LOCATION will set a location in the grid to the object
; this indicates that the location is occupied.
; It also updates the a-list of objects so that it can be accessed in
; reverse.

(defun set-location (loc object)
	(setf (aref *LOC-MAP* (location-x loc) (location-y loc)) object)
;	(if (assoc object *OBJECT-INFO*)
;	    (rplacd (assoc object *OBJECT-INFO*) loc)
;	    (setq *OBJECT-INFO* (acons object loc *OBJECT-INFO*))
;	)
)



; CLEAR-LOCATION will set a location in the grid to nil
; to indicate that the location is clear
(defun clear-location (loc)
     (set-location loc nil))

(defun set-open-door (door)
	(setf (door-status-open (cdr (assoc door *DOOR-STATUS*))) t)
)

(defun set-closed-door (door)
	 (setf (door-status-open (cdr (assoc door *DOOR-STATUS*))) nil)
)

(defun set-locked-door (door)
	 (setf (door-status-locked (cdr (assoc door *DOOR-STATUS*))) t)
)

(defun set-unlocked-door (door)
 (setf (door-status-open (cdr (assoc door *DOOR-STATUS*))) nil)
)

;  ---------------------------------------------------------------------
; The following drawing routines are generic.

; DRAW-DOOR-SYMBOL will draw the door symbol (string in door-string)
; It will center it in the wall and between two grid points.
(defun draw-door-symbol (door-string door-var)
       (let* ((str-pix-len (* *CHAR-WIDTH* (length door-string)))
	      (x (truncate (+ (calc-point-x (location-x (door-loc1 door-var)))
		     (calc-point-x (location-x (door-loc2 door-var)))
						(- str-pix-len)) 2))
	      (y (truncate (+ (calc-point-y (location-y (door-loc1 door-var)))
		       (calc-point-y (location-y (door-loc2 door-var)))
							*CHAR-HEIGHT*) 2)))

		(pg-write-text *DOMAIN-WINDOW* x y door-string))
)

; End generic routines
; ---------------------------------------------------------------------
(defun turn-off-light (arg)
	(declare (ignore arg)))


#|
; DRAW-THING-WITH-LABEL will draw a thing with a label
(defun draw-thing-with-label (thing)
	(draw-a-thing-at (thing))
	(write-label (thing))
)

(defun write-label (loc label)
	(let*  ((x (- (calc-point-x (location-x loc)) half-side-x))
		(y (+ (calc-point-y (location-y loc)) half-side-y)))

		(pg-write-text
|#
(defun draw-switch (arg)
   (declare (ignore arg)))

; DRAW-A-THING-AT draws an object at a particular grid coord.
; robots are big boxes, objects are small boxes.

(defun draw-a-thing-at (thing)
    (case (car thing)
	  ('robot (draw-robot (cadr thing)))
	  (otherwise (draw-object thing))
    )
)

; DRAW-ROBOT draws the robot square with an R inside of it or, if the object  
; is holding an object, the object is drawn inside the robot.
; The existance of set-location AND *ROBOT* is strange.  I am not sure
; what all of the posible interactions could be between the different
; incantations of strips.

(defun draw-robot (robot-loc)
    (set-location robot-loc 'robot) ; this is for strips
;    (setq *ROBOT* robot-loc) 	; this is for realworld and extended-strips
    (cond ( *HOLDING*
              (draw-small-square robot-loc *OBJECT-SCALE* (cdr *HOLDING*))
              (draw-small-square robot-loc .8)); not label here
	  (t (draw-small-square robot-loc .8 "R")) ; .8 is the scale factor

    )
)

; DRAW-OBJECT accepts and argument in the format of
; (object-label object-location) where object-location is of
; location type.

(defun draw-object (object)
  (unless (eq (car object) (car *holding*))
   (let* ((label (symbol-name (car object)))
	  (label-len (length label)))

;	  (set-location (cadr object) (car object))
	  (and (< *CHARS-IN-OBJECT* label-len)
	     (setq label (subseq label (- label-len *CHARS-IN-OBJECT*))))

	(draw-small-square (cadr object) *OBJECT-SCALE* label)
   )
 )
)


; DRAW-SMALL-SQUARE draw a small square centered about the grid location
; specified by loc [where (location-p loc)].  The sides of the square
; will be equal to the grid size times scale. If label is non-nill then
; the number of characters 
(defun draw-small-square (loc scale &optional label)
	(let* ((half-side-x (truncate (* *HALF-RES-X* scale)))
	      (x (- (calc-point-x (location-x loc)) half-side-x))
	      (half-side-y (truncate (* *HALF-RES-Y* scale)))
	      (y (+ (calc-point-y (location-y loc)) half-side-y)))

		(pg-frame-rect *DOMAIN-WINDOW*
			x
			(- y (* half-side-y 2))
			(+ x (* half-side-x 2))
			y)

	     (when label
		(pg-write-text *DOMAIN-WINDOW* (1+ x) (- y 3) label))
	)
)

;--------------------------------------------------------------------------

;; DRAW-DOMAIN-FOREGROUND uses domain graphics parameters to draw the 
;; foremost plane of graphics -- drawn after the domain objects.
;;

 (defun draw-domain-foreground ()
     (draw-all-doors *DOOR-STATUS* *DOOR-INFO*)
 )

(defun draw-all-doors (status info-a-list)
   (cond ((null status) nil)
     (t 
	(let ((d-stat (cdar status))
	      (info   (cdr (assoc (caar status) info-a-list))))

 				; locked and open
	   (cond 
		 ((and (door-status-open d-stat)
		       (door-status-locked d-stat))
		  (draw-door-symbol "OL" info))
 				; locked and closed
		 ((and (not (door-status-open d-stat))
                       (door-status-locked d-stat))
                  (draw-door-symbol "CL" info))
 				; open and unlocked
		 ((and (door-status-open d-stat)
                       (not (door-status-locked d-stat)))
                  (draw-door-symbol "OU" info))
				; closed and unlocked
		 ((not (or (door-status-open d-stat)
                           (door-status-locked d-stat)))
                  (draw-door-symbol "CU" info))
	   )
	)
       (draw-all-doors (cdr status) info-a-list)
     )
   )
)



; RESOVE-CONFLICTS moves through the object structure resolving the 
; ambiguous terms.  The first resovle the at predicates, then the
; next-to's and then the in-room's.  In this way an order or precidence,
; at--next-to--in-room is implemented.

(defun resolve-conflicts ()
     (resolve-robot)
     (resolve-ats)
     (resolve-in-rooms)
     (resolve-next-tos)
)

(defun resolve-ats ()
     (dolist (object-cons *OBJECT-INFO*)
       (let ((object (car object-cons)))
	 (if (and (not (get-resolvedp object))
	          (get-at object))
             (setf (get-location object) (get-at object)
		   (get-grid-object (get-location object)) object
		   (get-resolvedp object) t
	     )
	 )
       )
     )
)


(defun resolve-in-rooms ()
     (dolist (object-cons *OBJECT-INFO*)
	(let ((object (car object-cons)))
	(if (and (not (get-resolvedp object))
	         (get-in-room object))
	    (setf (get-location object)
		  (find-position (cdr (assoc (get-in-room object) *ROOM-INFO*))
				 object)
		  (get-grid-object (get-location object)) object
		  (get-resolvedp object) t
	    )
	)
	)
     )
)

; when to objects are next-to each-other and one is a door the this 
; routine chooses the correct side of the door.  If the other is
; not a door then this routine chooses a grid point around the other.

(defun resolve-next-tos ()
     (dolist (object-cons *OBJECT-INFO*)
	(let ((object (car object-cons)))
           (when  (and (not (get-resolvedp object))
                     (get-next-to-list object))

	     
	       (if (is-door-p (car (get-next-to-list object)))
	 		; If it's a door then choose correct side
		  (setf (get-location object)
		    (let ((door (get-door (car (get-next-to-list object)))))
			(if (eq (get-in-room object)
				(location-room
				  (door-loc1 door)))
			    (door-loc1 door)
			    (door-loc2 door))))
			; else it's just another object, choose point
	        (setf (get-location object) (clear-point-next-to 
		      (get-next-to-location object))
		      (get-grid-object (get-location object)) object
	    	      (get-resolvedp object) t
		)
	       )
		
	   )
	)
     )
)


; REMEMBER:  object-next-to isn't a list for the *ROBOT*
(defun resolve-robot ()
	(cond ((object-prodigy-resolvedp *ROBOT*) nil)
	      ((object-prodigy-at *ROBOT*) (setf (object-prodigy-loc *ROBOT*) 
				         (object-prodigy-at *ROBOT*)))

	      ((object-prodigy-next-to *ROBOT*) 

 				; Object next-to begins
	  (if (is-door-p (object-prodigy-next-to *ROBOT*))
    			; If object is a door
		(setf (object-prodigy-loc *ROBOT*)
                    (let ((door (get-door (object-prodigy-next-to *ROBOT*))))
                        (if (eq (object-prodigy-in-room *ROBOT*)
                                (location-room
                                  (door-loc1 door)))
                            (door-loc1 door)
                            (door-loc2 door))))
			; else just another object, choose point

				(setf (object-prodigy-loc *ROBOT*)
				  (clear-point-next-to
				    (get-location  
					(object-next-to *ROBOT*))))))
		; Object next-to ends

	      ((object-prodigy-in-room *ROBOT*) (setf (object-prodigy-loc *ROBOT*)
					      (robot-location 
						   (object-prodigy-in-room *ROBOT*))))

	)
	      
)


; GET-NEXT-TO-LOCATION returns a location next-to-which you must
; be.  The first clause in the cond is not correct.
(defun get-next-to-location (ob)
    (cond ((get-location ob) (get-location ob))
	  (t (try-each-element (get-next-to-list ob)))
    )
)

(defun try-each-element (ob-list)
   (cond ((null ob-list) nil)
	 ((get-next-to-location ob-list))
	 (t nil)
   )
)


; DRAW-OBJECTS will draw recursively all of the objects in the 
(defun draw-objects (ob-list)
	(cond ((null ob-list) nil)
	      (t (draw-object (list (caar ob-list)
				    (object-prodigy-loc (cdar ob-list))))
		 (draw-objects (cdr ob-list)))
	)
)

; NEW-DRAW-ROBOT draw the robot using the object data structure stored
; in *ROBOT*

(defun new-draw-robot ()
	(draw-robot (object-prodigy-loc *ROBOT*)))

(defun initialize-vector (dim res org)
	(make-array (list dim) :element-type (type-of 8)
		:initial-contents 
		(create-vector (- dim 1) res org)
	)
)

(defun create-vector (count res org)
      (cond ((minusp count) nil)
	    (t (append	(create-vector (- count 1) res org)
					(list (+ org (* count res))) 
))
      )
)

	
#|

(defun calc-point-x (i)
	(+ *ORGIN-X* (* i *RESOL-X*))
)

(defun calc-point-y (j)
	(- *ORGIN-Y* (* j *RESOL-Y*))
)

|# 


;; DOMAIN-CONFIG is called every time the window is resized.  It normally
;; sets the variables *DOMAIN-DIMENSION-X* and *DOMAIN-DIMENSION-Y* only, but
;; the domain author can added other calculations based on these.

(defun domain-dependent-configure (new-x new-y)
	(declare (special *DOMAIN-DIMENSION-X* *DOMAIN-DIMENSION-Y*))

	(setf *DOMAIN-DIMENSION-X* new-x *DOMAIN-DIMENSION-Y* new-y)
        (calc-window-dependents new-x
				new-y
				*CHAR-WIDTH*
                                *OBJECT-SCALE*)
)
