(proclaim
   '(special *room-map* *doors* *z-dim* *y-dim* *x-dim*))

(defstruct door
  name
  loc1
  loc2
  state)

;(defstruct location
;  room
;  x
;  y
;  z)
(defun make-location (&key room x y z)
  (list 'location room x y z))

(defun location-p (loc)
  (eq (car loc) 'location))

(defun location-room (loc)
  (second loc))

(defun location-x (loc)
  (third loc))

(defun location-y (loc)
  (fourth loc))

(defun location-z (loc)
  (fifth loc))

;
; Called with a list of rooms in the form:
;   ((loc1a loc1b)
;    (loc2a loc2b) ...)
; where the locna indicates the corner of a room and the pair
; of coordinates defines a rectangle which is the room.  Note 
; that there is no check to make sure you don't have overlapping rooms.
; This function calls initialize-room to update the array called 
; *room-map* to contain the corresponding room in each element of the
; array.
;
(defun initialize-room-map (rooms)
  (cond ((null rooms))
	(t (let ((room (car rooms)))
	     (initialize-room (location-room (first room))
			      (location-x (first room))
			      (location-y (first room))
			      (location-x (second room))
			      (location-y (second room)))
	     (initialize-room-map (cdr rooms))))))
;
;
; Called with the room name and the coordinates to update *room-map*.
;
(defun initialize-room (room x1 y1 x2 y2)
  (do ((x x1 (+ x 1)))
      ((> x x2))
      (do ((y y1 (+ y 1)))
	  ((> y y2))
	  (setf (aref *ROOM-MAP* x y) room))))
;
;
; 
(defun loc-in-room (loc room)
  (cond ((is-variable loc) 'no-match-attempted)
	((is-variable room)
	 (list (list (list room (location-room loc)))))
	(t (eq room (location-room loc)))))
#|
; TYPE determines the type of a thing
; 
(defun is-type (thing thing-type)
    (cond ((or (is-variable thing) (is-variable thing-type)) 
	   'no-match-attempted)
	  (t (eq (car thing) thing-type))))
|#

;
;
(defun connects (door room1 room2)
  (cond ((and (is-variable room1)(is-variable door)) 'no-match-attempted)
	((and (not (is-variable door))
	      (is-variable room1)
	      (is-variable room2))
	 (determine-rooms door room1 room2 *DOORS*))
	((and (is-variable door)(is-variable room2))
	 (determine-door door room1 room2 *DOORS*))
	(t (verify-connects door room1 room2 *DOORS*))))


(defun determine-rooms (door room1 room2 doors)
  (cond ((null doors) nil)
	((eq door (door-name (car doors)))
	 (list (list (list room1 (location-room (door-loc1 (car doors))))
		     (list room2 (location-room (door-loc2 (car doors)))))
	       (list (list room1 (location-room (door-loc2 (car doors))))
		     (list room2 (location-room (door-loc1 (car doors)))))))
	(t (determine-rooms door room1 room2 (cdr doors)))))

(defun determine-door (door room1 room2 doors)
  (cond ((null doors) nil)
	((eq room1 (location-room (door-loc1 (car doors))))
	 (cons (list (list door (door-name (car doors)))
		     (list room2 (location-room (door-loc2 (car doors)))))
	       (determine-door door room1 room2 (cdr doors))))
	((eq room1 (location-room (door-loc2 (car doors))))
	 (cons (list (list door (door-name (car doors)))
		     (list room2 (location-room (door-loc1 (car doors)))))
	       (determine-door door room1 room2 (cdr doors))))
	(t (determine-door door room1 room2 (cdr doors)))))


(defun verify-connects (door room1 room2 doors)
  (cond ((null doors) nil)
	((and (eq room1 (location-room (door-loc1 (car doors))))
	      (eq door (door-name (car doors)))
	      (eq room2 (location-room (door-loc2 (car doors)))))
	 t)
	((and (eq room1 (location-room (door-loc2 (car doors))))
	      (eq door (door-name (car doors)))
	      (eq room2 (location-room (door-loc1 (car doors)))))
	 t)
	(t (verify-connects door room1 room2 (cdr doors)))))

(defun loc-next-to-door (door room loc)
  (cond ((or (is-variable door)(is-variable room)) 'no-match-attempted)
	((is-variable loc)
	 (binding-list loc (list (find-loc-next-to-door door room *DOORS*))))
	((is-location loc)
	 (equalp loc (find-loc-next-to-door door room *DOORS*)))
	(t (type-error 'loc-next-to-door))))

(defun find-loc-next-to-door (dname room doors)
  (cond ((null doors) nil)
	((eq dname (door-name (car doors)))
	 (cond ((eq room (location-room (door-loc1 (car doors))))
		(door-loc1 (car doors)))
	       ((eq room (location-room (door-loc2 (car doors))))
		(door-loc2 (car doors)))
	       (t (error "Invalid door room combination in loc-next-to-door"))))
        (t (find-loc-next-to-door dname room (cdr doors)))))
	 


; NEXT-TO generates all the locations next to a given location.
;
(defun next-to (loc adj-loc)
    (cond ((is-variable loc) 'no-match-attempted)
	  ((and (is-variable adj-loc) (is-location loc))
	   (binding-list adj-loc (next-to-gen loc)))
	  ((and (is-location loc) (is-location adj-loc))
	   (not (null (member adj-loc (next-to-gen loc) :test #'equalp))))
	  (t (type-error 'next-to))))


(defun next-to-gen (loc)
  (nconc (gen-adj-loc (location-room loc)
		      (location-x loc)
		      (1- (location-y loc))
		      (location-z loc)
		      (location-x loc)
		      (1- (location-y loc))
		      (location-z loc))
         (gen-adj-loc (location-room loc)
		      (location-x loc)
		      (1+ (location-y loc))
		      (location-z loc)
		      (location-x loc)
		      (1+ (location-y loc))
		      (location-z loc))
         (gen-adj-loc (location-room loc)
		      (1- (location-x loc))
		      (location-y loc)
		      (location-z loc)
		      (1- (location-x loc))
		      (location-y loc)
		      (location-z loc))
         (gen-adj-loc (location-room loc)
		      (1+ (location-x loc))
		      (location-y loc)
		      (location-z loc)
		      (1+ (location-x loc))
		      (location-y loc)
		      (location-z loc))))



; GEN-ADJ-LOC generates all the adj-loc within a specified region.
; HACK ALERT --> It is carfully designed to place lower locations before
; higher locations so PRODIGY doesn't try to build stairways to heaven.
;
(defun gen-adj-loc (room min-x min-y min-z max-x max-y max-z)
    (do ((z (min max-z (1- *Z-DIM*))(1- z))
         (loc-list nil))
        ((< z (max min-z 0)) loc-list)
      (do ((x (max min-x 0)(1+ x)))
          ((> x (min max-x (1- *X-DIM*))))
        (do ((y (max min-y 0)(1+ y)))
	    ((> y (min max-y (1- *Y-DIM*))))
	    (let ((new-loc (make-location :room room :x x :y y :z z)))
	      (cond ((is-location new-loc)
		     (setq loc-list (cons new-loc loc-list)))
		    (t loc-list)))))))



; ADJACENT-TO generates all the traversable locations from a given loc.
; The traversable locs consist of any square one unit away in either the
;  x or y direction and can have an offset of 0 or 1 in the z direction.
;
; The location must be one square big -- this should be changed.
; 
(defun adjacent-to (loc-a loc-b)
    (cond ((and (is-variable loc-a) (is-variable loc-b)) 'no-match-attempted)
	  ((and (is-variable loc-a) (is-location loc-b))
	   (binding-list loc-a (adj-gen loc-b)))
	  ((and (is-variable loc-b) (is-location loc-a))
	   (binding-list loc-b (adj-gen loc-a)))
	  ((and (is-location loc-a) (is-location loc-b))
	   (not (null (member loc-a (adj-gen loc-b) :test #'equalp))))
	  (t (type-error 'adjacent-to))))


(defun adj-gen (loc)
  (nconc (gen-adj-loc (location-room loc)
		      (location-x loc)
		      (1- (location-y loc))
		      (1- (location-z loc))
		      (location-x loc)
		      (1- (location-y loc))
		      (1+ (location-z loc)))
         (gen-adj-loc (location-room loc)
		      (location-x loc)
		      (1+ (location-y loc))
		      (1- (location-z loc))
		      (location-x loc)
		      (1+ (location-y loc))
		      (1+ (location-z loc)))
         (gen-adj-loc (location-room loc)
		      (1- (location-x loc))
		      (location-y loc)
		      (1- (location-z loc))
		      (1- (location-x loc))
		      (location-y loc)
		      (1+ (location-z loc)))
         (gen-adj-loc (location-room loc)
		      (1+ (location-x loc))
		      (location-y loc)
		      (1- (location-z loc))
		      (1+ (location-x loc))
		      (location-y loc)
		      (1+ (location-z loc)))))



; EQUALP determines if the two arguments are equal.  If one of the arguments
; is a variable it will get bound to the other argument.
; 
(defun equal-p (first-arg second-arg)
    (cond ((and (is-variable first-arg) (is-variable second-arg))
	   'no-match-attempted)
	  ((is-variable first-arg) (equalp-bind first-arg second-arg))
	  ((is-variable second-arg) (equalp-bind second-arg first-arg))
	  (t (equal first-arg second-arg))))


(defun equalp-bind (var value)
    (binding-list var (list value)))



; GROUND-LOC determines if a location is at ground level (i.e. has an location-z
;  of zero).
; 
(defun ground-loc (loc)
    (cond ((is-variable loc) 'no-match-attempted)
	  ((not (is-location loc))
	   (type-error 'ground-sq))
	  (t (zerop (location-z loc)))))




; TOWARDS is given three locations and verifies that the second one is towards
; the third location from the first location.
; 
; The locations must be one square big -- this should be changed.
;
(defun towards (f-loc a-loc t-loc)
    (cond ((or (is-variable f-loc)
	       (is-variable a-loc)
	       (is-variable t-loc))
	   'no-match-attempted)
	  ((or (not (is-location f-loc))
	       (not (is-location a-loc))
	       (not (is-location t-loc)))
	   (type-error 'towards))
	  (t
	   (= (distance-calc f-loc t-loc)
	      (+ (distance-calc f-loc a-loc) (distance-calc a-loc t-loc))))))


			       
; Calculates the distance from a to b.
; 
(defun distance (a-loc b-loc dist)
    (cond ((is-variable a-loc) 'no-match-attempted)
	  ((is-variable b-loc) 'no-match-attempted)
	  ((not (is-location a-loc))
	   (type-error 'distance))
	  ((not (is-location b-loc)) (type-error 'distance))
	  (t (let ((result (distance-calc a-loc b-loc)))
		  (cond ((is-variable dist)
			 (list (list (list dist result))))
			((equal dist result) t))))))

(defun distance-calc (a-loc b-loc)
    (+ (abs (- (location-x a-loc) (location-x b-loc)))
       (abs (- (location-y a-loc) (location-y b-loc)))
       (abs (- (location-z a-loc) (location-z b-loc)))))

; LESS-THAN 

(defun less-than (a b)
    (cond ((is-variable a) 'no-match-attempted)
	  ((is-variable b) 'no-match-attempted)
	  (t (< a b))))


; ABOVE-LOC checks or generates a-loc above b-loc.
;
(defun above-loc (a-loc b-loc)
    (cond ((and (is-variable a-loc) (is-variable b-loc)) 'no-match-attempted)
          ((and (is-variable a-loc) (is-location b-loc))
	   (let ((above-loc
		   (make-location :room (location-room b-loc)
				  :x (location-x b-loc)
				  :y (location-y b-loc)
				  :z (1+ (location-z b-loc)))))
	     (cond ((is-location above-loc)
		    (binding-list a-loc (list above-loc))))))
	  ((and (is-variable b-loc) (is-location a-loc))
	   (let ((below-loc
		  (make-location :room (location-room a-loc)
				 :x (location-x a-loc)
				 :y (location-y a-loc)
				 :z (1- (location-z a-loc)))))
	     (cond ((is-location below-loc)
		    (binding-list b-loc (list below-loc))))))
          ((and (is-location a-loc) (is-location b-loc))
	   (and (= (location-x a-loc)
		   (location-x b-loc))
		(= (location-y a-loc)
		   (location-y b-loc))
		(= (location-z a-loc)
		   (1+ (location-z b-loc)))))
          (t (type-error 'above-loc))))



; UNDER-LOC returns t if loc-b is under loc-a.
; 
(defun under-loc (loc-a loc-b)
    (cond ((or (is-variable loc-a) (is-variable loc-b)) 'no-match-attempted)
	  ((not (and (is-location loc-a) (is-location loc-b)))
	   (type-error 'under-loc))
	  (t (and (< (location-z loc-b) (location-z loc-a))
		  (= (location-x loc-a) (location-x loc-b))
		  (= (location-y loc-a) (location-y loc-b))))))


;  BINDING-LIST returns a binding list for a single variable, only
;
(defun binding-list (var val-list)
  (cond ((null val-list) nil)
        ((null (car val-list)) (binding-list var (cdr val-list)))
        (t (append (list (list (list var (car val-list))))
                   (binding-list var (cdr val-list))))))


; 
;  T Y P E     C H E C K I N G
; 

(defun type-error (function)
    (error "~A" (append '(Illegal or incorrect type encountered in -)
			(list function))))

(defun is-location (loc)
  (cond ((and (location-p loc)
	      (eq (location-room loc)
		  (aref *ROOM-MAP* (location-x loc)(location-y loc)))
	      (>= (location-z loc) 0)
	      (< (location-z loc) *Z-DIM*))
	 t)))

(defun is-in-zero-plane (loc)
   (if (not (location-p loc)) 
	(type-error 'is-in-zero-plane)

        (zerop (location-z loc))
   )
)

(add-meta-fn 'is-in-zero-plane)
