;;;
;;; MATCH - compare two arguments for equality; this can be used as a
;;; generator
;;;
(defun match (arg1 arg2)
  (cond ((and (is-variable arg1)
              (is-variable arg2)) 'no-match-attempted)
	((is-variable arg1) (list (list (list arg1 arg2))))
	((is-variable arg2) (list (list (list arg2 arg1))))
        (t (equal arg1 arg2))))

;;;
;;; DIFF - subtract two arguments; this can be used as a generator
;;;
(defun diff (arg1 arg2 res)
  (cond ((and (is-variable arg1)
	      (not (is-variable arg2))
	      (not (is-variable res))) (list (list (list arg1 (+ arg2 res)))))
        ((and (not (is-variable arg1))
	      (is-variable arg2)
	      (not (is-variable res))) (list (list (list arg2 (- arg1 res)))))
        ((and (not (is-variable arg1))
	      (not (is-variable arg2))
	      (is-variable res)) (list (list (list res (- arg1 arg2)))))
        ((and (not (is-variable arg1))
	      (not (is-variable arg2))
	      (not (is-variable res))) (= (- arg1 arg2) res))
	(t 'no-match-attempted)))

;;;
;;; VOICE - Check to see if the argument is soprano, alto, tenor, or
;;; bass.  This is also a static generator of voices.
;;;
(defun voice (voice-in)
  (cond ((is-variable voice-in) (list (list (list voice-in 'soprano))
                                      (list (list voice-in 'alto))
                                      (list (list voice-in 'tenor))
                                      (list (list voice-in 'bass))))
        (t (not (null (member voice-in
                              '(soprano alto tenor bass) :test #'equal))))))

;;;
;;; OCTAVE - Verify that the numerical argument is a valid octave
;;; (range 1-5).  It can also be used as a generator.
;;;
(defun octave (oct)
  (cond ((is-variable oct) (list (list (list oct 1))
                                 (list (list oct 2))
                                 (list (list oct 3))
                                 (list (list oct 4))
                                 (list (list oct 5))))
        (t (and (>= oct 1) (<= oct 5)))))

;;;
;;; TONE - Verify that a tone is a valid member of a chord (ie. 1, 3
;;; or 5).  It can also be used as a generator.
;;;
(defun tone (chord-loc)
  (cond ((is-variable chord-loc) (list (list (list chord-loc 1))
                                       (list (list chord-loc 3))
                                       (list (list chord-loc 5))))
        (t (or (= chord-loc 1) (= chord-loc 3) (= chord-loc 5)))))

;;;
;;; SCALE - Given a key, return a list of notes in the major scale
;;; of that key.  This is an auxiliary function used only in this
;;; file.
;;;
(defun scale (key)
  (assoc key
	 '((C D E F G A B)
	   (F G A B-flat C D E)
	   (B-flat C D E-flat F G A)
	   (E-flat F G A-flat B-flat C D)
	   (A-flat B-flat C D-flat E-flat F G)
	   (D-flat E-flat F G-flat A-flat B-flat C)
	   (G-flat A-flat B-flat C-flat D-flat E-flat F)
	   (C-flat D-flat E-flat F-flat G-flat A-flat B-flat)
	   (G A B C D E F-sharp)
	   (D E F-sharp G A B C-sharp)
	   (A B C-sharp D E F-sharp G-sharp)
	   (E F-sharp G-sharp A B C-sharp D-sharp)
	   (B C-sharp D-sharp E F-sharp G-sharp A-sharp)
	   (F-sharp G-sharp A-sharp B C-sharp D-sharp E-sharp)
	   (C-sharp D-sharp E-sharp F-sharp G-sharp A-sharp B-sharp))
	 :test #'equal))

;;;
;;; CHORD-NOTES - Given a chord (e.g. III) and a key (e.g. C), return
;;; a list of the notes in that chord.  This is an auxiliary function
;;; used only in this file.
;;;
(defun chord-notes (chord key)
  (remove-if #'null (mapcar #'(lambda (x y) (when x y))
			    (cdr (assoc chord
					'((I   t nil t nil t nil nil)
					  (II  nil t nil t nil t nil)
					  (III nil nil t nil t nil t)
					  (IV  t nil nil t nil t nil)
					  (V   nil t nil nil t nil t)
					  (VI  t nil t nil nil t nil)
					  (VII nil t nil t nil nil t))
					:test #'equal))
			    (scale key))))

;;;
;;; NOTE-IN-CHORD - Given a note, chord, and key, verify that the note
;;; is in the chord for the given key.  This can also be used for a
;;; generator of notes given the chord and key.
;;;
(defun note-in-chord (note chord key)
  (cond ((is-variable chord) 'no-match-attempted)
        ((is-variable key) 'no-match-attempted)
	((is-variable note) (mapcar #'(lambda (x) (list (list note x)))
				    (chord-notes chord key)))
        (t (not (null (member note (chord-notes chord key) :test #'equal))))))

;;;
;;; TONE-VALUE - Given a tone, return the number of semitones between it and
;;; C.  This is an auxiliary function used only in this file.
;;;
(defun tone-value (note)
  (cadr (assoc note '((C-flat -1)
		      (C 0)
		      (C-sharp 1)
		      (D-flat 1)
		      (D 2)
		      (D-sharp 3)
		      (E-flat 3)
		      (E 4)
		      (E-sharp 5)
		      (F-flat 4)
		      (F 5)
		      (F-sharp 6)
		      (G-flat 6)
		      (G 7)
		      (G-sharp 8)
		      (A-flat 8)
		      (A 9)
		      (A-sharp 10)
		      (B-flat 10)
		      (B 11)
		      (B-sharp 12))
	       :test #'equal)))

;;;
;;; NOTE-ABOVE - Given two note/octave pairs, return true if the pitch
;;; of the first is greater than or equal to the pitch of the second.
;;;
(defun note-above (note octave low-note low-octave)
  (cond ((or (is-variable note)
             (is-variable octave)
             (is-variable low-note)
             (is-variable low-octave)) 'no-match-attempted)
	(t (>= (+ (* 12 octave) (tone-value note))
	       (+ (* 12 low-octave) (tone-value low-note))))))

;;;
;;; NOTE-BETWEEN - Given three note/octave pairs, return true if the
;;; pitch of the first is greater than or equal to the pitch of the
;;; second and less than or equal to the pitch of the third.  This
;;; can be used as a generator for the first octave.
;;;
(defun note-between (note octave low-note low-octave high-note high-octave)
  (cond ((or (is-variable note)
	     (is-variable low-note)
	     (is-variable low-octave)
	     (is-variable high-note)
	     (is-variable high-octave)) 'no-match-attempted)
	((is-variable octave)
	 (append (when (note-above note low-octave low-note low-octave)
		   (list (list (list octave low-octave)))) ; low octave valid?
		 (do ((bindings nil)                       ; list middle octaves
		      (index (1- high-octave)))
		     ((<= index low-octave) bindings)
		   (setq bindings
			 (cons (list (list octave index)) bindings))
		   (setq index (1- index)))
		 (when (note-above high-note high-octave note high-octave)
		   (list (list (list octave high-octave)))))) ; hi octave valid?
	(t (and (note-above note octave low-note low-octave)
		(note-above high-note high-octave note octave)))))

;;;
;;; ROMAN-TO-DEC - Convert a Roman numeral to Arabic numeral.  This is
;;; an auxiliary function used only in this file.
;;;
(defun roman-to-dec (roman)
  (cadr (assoc roman '((I   1)
                       (II  2)
                       (III 3)
                       (IV  4)
                       (V   5)
                       (VI  6)
                       (VII 7)))))

;;;
;;; FIND-PART - Given an part (e.g. 3rd) of a chord (e.g. III) in a 
;;; key (e.g. C), return the actual pitch.  This is an auxiliary
;;; function used only in this file.
;;;
(defun find-part (part chord key)
  (nth (rem (- (+ part (roman-to-dec chord)) 2) 7) (scale key)))

;;;
;;; IS-TONE - Given a note (e.g. E-flat), a part (e.g. 3), a chord
;;; (e.g. VI), and key (e.g. A-flat), return true if the note is that
;;; part of the chord in that key.  This can be used as a generator
;;; of the note given the part, chord, and key.
;;;
(defun is-tone (note part chord key)
  (cond ((or (is-variable part)
	     (is-variable chord)
	     (is-variable key)) 'no-match-attempted)
	((is-variable note)
	 (list (list (list note (find-part part chord key)))))
	(t (equal note (find-part part chord key)))))

;;;
;;; NOTE-SPACE - Given two notes in a key, find the number of scale
;;; steps from the first to the second.  This will be a number in
;;; the range -3 to +3.  This is an auxiliary function used only in
;;; this file.
;;;
(defun note-space (note1 note2 key)
  (do ((index 0)
       (notes (member note1 (append (scale key) (scale key)))))
      ((equal note2 (car notes)) (- (rem (+ index 3) 7) 3))
    (setq notes (cdr notes))
    (setq index (1+ index))))

;;;
;;; COMMON-NOTE - Return true if the note is common between the two
;;; chords in the given key.  This can be used to generate notes
;;; given the chords and the key.
;;;
(defun common-note (note chord1 chord2 key)
  (cond ((or (is-variable chord1)
	     (is-variable chord2)
	     (is-variable key)) 'no-match-attempted)
        ((is-variable note)
	 (do ((bindings nil)
	      (notes1 (chord-notes chord1 key))
	      (notes2 (chord-notes chord2 key)))
	     ((endp notes1) bindings)
	   (when (member (car notes1) notes2)
	     (setq bindings (cons (list (list note (car notes1)))
				  bindings)))
	   (setq notes1 (cdr notes1))))
	(t (not (null (and (member note (chord-notes chord1 key))
                           (member note (chord-notes chord2 key))))))))

;;;
;;; OCTAVE-CHANGE - Given two notes and a key, return the number of 
;;; octaves changed in going from the first to the second.  The meaning
;;; of octave here is the interval from C to C.  A value of -1, 0, or
;;; +1 is returned.  This is an auxiliary function used only in this file.
;;;
(defun octave-change (note1 note2 key)
  (let ((pitch1 (tone-value note1))
        (pitch2 (tone-value note2))
	(diff (note-space note1 note2 key)))
    (cond ((and (< pitch1 pitch2)
		(< diff 0) -1))
	  ((and (> pitch1 pitch2)
		(> diff 0) 1))
	  (t 0))))

;;;
;;; NEAREST-NOTE - Given a note and octave (presumably in one chord),
;;; a second chord, and a key, generate a binding list with the nearest
;;; note/octave in the chord to the given note.
;;;
(defun nearest-note (note1 octave1 note2 octave2 pchord2 key)
  (cond ((or (is-variable note1)
             (is-variable octave1)
             (not (is-variable note2))
             (not (is-variable octave2))
	     (is-variable pchord2)
             (is-variable key)) 'no-match-attempted)
        (t (do ((note '(4 nil))          ; interval/note pair
		(distance 0)             ; current interval distance
		(notes2 (chord-notes pchord2 key))) ; list of chord notes
	       ((endp notes2)
		(list (list (list note2 (cadr note))
			    (list octave2
				  (+ octave1
				     (octave-change note1 (cadr note) key))))))
	     (setq distance (note-space note1 (car notes2) key))
	     (when (< (abs distance) (car note))
	       (setq note (list distance (car notes2))))
	     (setq notes2 (cdr notes2))))))

;;;
;;; SMALLER-INTERVAL - Return true if the interval between the two note/
;;; octave pairs is smaller than or equal to the given interval.
;;;
(defun smaller-interval (note1 octave1 note2 octave2 interval)
  (cond ((or (is-variable note1)
             (is-variable octave1)
             (is-variable note2)
             (is-variable octave2)
	     (is-variable interval)) 'no-match-attempted)
	(t (<= (abs (- (+ (* 12 octave1) (tone-value note1))
		       (+ (* 12 octave2) (tone-value note2))))
	       interval))))

;;;
;;; HIGHER-PITCH - Given two note/octave pairs, this is a generator for
;;; the higher pitch of the two.
;;;
(defun higher-pitch (note1 octave1 note2 octave2 hi-note hi-octave)
  (cond ((or (is-variable note1)
             (is-variable octave1)
             (is-variable note2)
             (is-variable octave2)) 'no-match-attempted)
        ((and (is-variable hi-note)
	      (is-variable hi-octave))
	 (if (note-above note1 octave1 note2 octave2)
	     (list (list (list hi-note note1)
			 (list hi-octave octave1)))
	     (list (list (list hi-note note2)
			 (list hi-octave octave2)))))
	((and (not (is-variable hi-note))
	      (not (is-variable hi-octave)))
	 (if (note-above note1 octave1 note2 octave2)
	     (and (equal note1 hi-note) (equal octave1 hi-octave))
	     (and (equal note2 hi-note) (equal octave2 hi-octave))))
	(t 'no-match-attempted)))

;;;
;;; LOWER-PITCH - Given two note/octave pairs, this is a generator for
;;; the lower pitch of the two.
;;;
(defun lower-pitch (note1 octave1 note2 octave2 lo-note lo-octave)
  (cond ((or (is-variable note1)
             (is-variable octave1)
             (is-variable note2)
             (is-variable octave2)) 'no-match-attempted)
        ((and (is-variable lo-note)
	      (is-variable lo-octave))
	 (if (note-above note1 octave1 note2 octave2)
	     (list (list (list lo-note note2)
			 (list lo-octave octave2)))
	     (list (list (list lo-note note1)
			 (list lo-octave octave1)))))
	((and (not (is-variable lo-note))
	      (not (is-variable lo-octave)))
	 (if (note-above note1 octave1 note2 octave2)
	     (and (equal note2 lo-note) (equal octave2 lo-octave))
	     (and (equal note1 lo-note) (equal octave1 lo-octave))))
	(t 'no-match-attempted)))
