
;;; fewer-grades-can-be-taught is a meta-function that is to be called from the control rule
;;;                            PREFER-MOST-RESTRICTED-TIME-SLOTS.  It is used to determine which of two
;;;                            time slots has the least number of grades that can be taught by a given
;;;                            specialist.  Grades at events or recess during a time slot cannot be taught.  
;;;                            This function takes two expressions which are expected to create bindings
;;;                            of grades for the two time slots being compared.  The grades must be bound
;;;                            with the variables listed in get-grades below.
;;;                            It also takes the gradelist of a specialist so the restriction on the time
;;;                            slots can be determined.

(defun fewer-grades-can-be-taught (exp1 exp2 original-gradelist node)
  (let* ((activity-list1 (exp-match exp1 '((nil nil)) (node-state node)))
	 (activity-list2 (exp-match exp2 '((nil nil)) (node-state node)))
	 (grades1 (difference original-gradelist (get-grades activity-list1)))
	 (grades2 (difference original-gradelist (get-grades activity-list2))))
    (< (length grades1) (length grades2))))
	
(add-meta-fn 'fewer-grades-can-be-taught)





;;; get-grades accepts a binding list and returns a list of all the grades bound to the
;;;            listed variables.  VERY dependent on the calling search control rule.
;;;

(defun get-grades (b-list)
  (cond ((null b-list) nil)
	((listp (first b-list)) (append (get-grades (first b-list))
					(get-grades (rest b-list))))
	((or (equal (first b-list) '<e-gradelist1>)
	     (equal (first b-list) '<e-gradelist2>)
	     (equal (first b-list) '<r-l-gradelist1>)
	     (equal (first b-list) '<r-l-gradelist2>)) (first (last b-list)))
	(t nil)))





;;; difference returns the difference of two sets (lists)
;;;            there may possibly be a standard lisp function that already does this

(defun difference (seta setb)
  (cond ((null setb) seta)
	((member (first setb) seta) (difference (remove (first setb) seta) (rest setb)))
	(t (difference seta (rest setb)))))





;;; not-enough-free-slots is a meta-function called from 
;;; THERE-HAS-TO-BE-AS-MANY-TIME-SLOTS-AS-THERE-ARE-TEACHERS-TO-BE-SCHEDULED.  This
;;; finds the number of time-slots teachers can go in by taking
;;; num-teachable = num-of-time-slots - (num-of-breaks-for-the-specialist + num-of-reserved-events-for-the-specialist)
;;; We are checking for there NOT being enough empty time slots.  So, for this to be true we check
;;;  if num-days-specialist-teaches * num-teachable < num-of-teachers-to-be-scheduled-with-specialist * specalist-num-times


(defun not-enough-free-slots (time-slot-exp 
                              break-exp
                              reserved-exp
                              meetings-exp
                              day-list num-meetings node)

   (let* ((time-slot-list (exp-match time-slot-exp '((nil nil)) (node-state node)))

         (break-list (exp-match break-exp '((nil nil)) (node-state node)))
         (reserved-list (exp-match reserved-exp '((nil nil)) (node-state node)))
         (meetings-list (exp-match meetings-exp '((nil nil)) (node-state node))))

     (<  (* (length day-list) (- (length time-slot-list) (+ (length break-list) (length reserved-list))))
         (* num-meetings (length meetings-list)))))

(add-meta-fn 'not-enough-free-slots)





;; This creates a binding lists for the number of teachers scheduled on 
;; day1 for a specialist and the number of teachers shceduled on a day2
;; for the same specialist.  Day1 is prefered over day2 if it has more
;; empty slots.  Another way of saying this is that Day1 is prefered over Day2
;; if less people are scheuled on that Day1 than Day2.

(defun day1-has-more-empty-slots (exp1 exp2 node)
   (let ((day-list1 (exp-match exp1 '((nil nil)) (node-state node)))
         (day-list2 (exp-match exp2 '((nil nil)) (node-state node))))
 
        (< (length day-list1) (length day-list2))))


(add-meta-fn 'day1-has-more-empty-slots)





;;; day-to-num associates a number with each day in order to tell if they are adjacent
;;; 

(defun day-to-num (day)
  (cond ((equal day 'Monday) '1)
        ((equal day 'Tuesday) '2)
        ((equal day 'Wednesday) '3)
        ((equal day 'Thursday) '4)
        ((equal day 'Friday) '5)
        (t (nil))))





;; This prefers day1 over day2 if
;;   (day1 is 2 or more days away from day) AND (day2 is only 1 day away from day)
;; i.e. in terms of scheduling, prefer alternate days rather than consecutive days

(defun day1-is-2-or-more-away-from-day-and-better-than-day2 (day day1 day2)
  (cond ((is-variable day) nil)
        ((is-variable day1) nil)
        ((is-variable day2) nil)
        (t (and (< '1 (abs (- (day-to-num day) (day-to-num day1))))
                (> '2  (abs (- (day-to-num day) (day-to-num day2))))))))





;;; add-time adds military time and minutes.  If all parameters are bound, returns true if the addition is
;;;          valid.  If solution (end) is unbound, returns a binding list with the solution.
;;;

(defun add-time (start length end)
  (cond ((is-variable start) nil) 
        ((is-variable end) (binding-list end (list (add-minutes-to-time start length))))
        (t (= (add-minutes-to-time start length) end))))

(defun add-minutes-to-time (start-time minutes-to-add)           ; add-minutes-to-time does the dirty work
  (multiple-value-bind
	(hours minutes)
        (truncate start-time 100)                                ; isolate hours and minutes
    (multiple-value-bind
	  (hours-to-add new-minutes)                     
	  (truncate (+ minutes minutes-to-add) 60)               ; add minutes to minutes
      (+ (* (rem (+ hours hours-to-add) 24) 100) new-minutes)))) ; add any hours left over and recombine





;;; time-will-fit takes three parameters: a starting time, a time length (in minutes), and an end time
;;;               if the time length will fit between the start and end times, returns true,
;;;               otherwise returns false

(defun time-will-fit (start length end)
  (cond ((is-variable start) nil)
	((is-variable length) nil)
	((is-variable end) nil)
	(t (<= (add-minutes-to-time start length) end))))





;;; less-than doesn't return any binding lists
;;;

(defun less-than (a b)
    (cond ((is-variable a) 'no-match-attempted)
          ((is-variable b) 'no-match-attempted)
          (t (< a b))))

(add-meta-fn 'less-than)





;;; overlap returns true if the two intervals overlap
;;;         doesn't return a binding list
;;;

(defun overlap (start1 end1 start2 end2)
  (cond ((is-variable start1) nil)
        ((is-variable end1) nil)
        ((is-variable start2) nil)
        ((is-variable end2) nil)
        (t (or (and (< start2 end1) (<= start1 start2))
               (and (< start1 end2) (<= start2 start1))))))





;;; within returns true if the first interval is inside the second
;;;        doesn't return a binding list
;;;

(defun within (start1 end1 start2 end2)
  (cond ((is-variable start1) nil)
        ((is-variable end1) nil)
        ((is-variable start2) nil)
        ((is-variable end2) nil)
	(t (and (<= start2 start1) (>= end2 end1)))))





;;; my-member returns t (instead of the list returned by member)
;;;

(defun my-member (element the-list)
  (if (member element the-list) t nil))





;;; disjoint returns t if two lists contain no common elements
;;;          unbound parameters yield t also
;;;

(defun disjoint (setA setB)
  (cond ((or (is-variable setA) (is-variable setB)) t)
	(t (null (intersection setA setB)))))





;;; time-bind returns a binding list if start is unbound.  The values for start
;;;           are such that start + length is in the interval specified, and are
;;;           in increments of 5 minutes.  If start is bound, performs a time
;;;           computation.  Adapted from prodigy manual.

(defun time-bind (start earliest latest length)
  (if (is-variable start)
      (do  ((begin earliest (add-minutes-to-time begin 5))
	(lst-of-binding-lsts nil (cons (list (list start begin))
				       lst-of-binding-lsts)))
	((> (add-minutes-to-time begin length) latest) lst-of-binding-lsts))
      (<= (add-minutes-to-time start length) latest)))





;;; day-bind takes a list of days and returns a binding list of days
;;;          

(defun day-bind (day day-list)
  (if (is-variable day)
      (do  ((d-list day-list (cdr d-list))
            (lst-of-binding-lsts nil (cons (list (list day (car d-list)))
				       lst-of-binding-lsts)))
	((null d-list) lst-of-binding-lsts))
      (member day day-list)))





;;; one-more returns true if the first parameter = second + 1
;;;          If just one of the parameters is bound, returns a binding list with
;;;          the appropriate value (other param +- 1)
;;;

(defun one-more (bigger smaller)
  (cond ((is-variable smaller) (binding-list smaller (list (1- bigger))))
	((is-variable bigger)  (binding-list bigger  (list (1+ smaller))))
	(t (= bigger (1+ smaller)))))





;;; five-more does not return any binding lists
;;;           It returns true if the first parameter = 5 + first parameter

(defun five-more (bigger smaller)
  (= bigger (+ 5 smaller)))





;;; binding-list has been snagged from schedworld.  Given a variable and a list,
;;;              it creates a list of binding lists.
;;;              

(defun binding-list (var val-list)
  (cond ((null val-list) nil)
	((null (car val-list)) (binding-list var (cdr val-list)))
	(t (append (list (list (list var (car val-list))))
		   (binding-list var (cdr val-list))))))




