(setq *MAX-CREDITS*         35)
(setq *MAX-CREDIT-CLASS*    5)
(setq *CREDIT-OPTIONS*      '(3 4 5))
(setq *CLASS-COUNT-OPTIONS* '(0 1 2))

;;
;; QUARTER-FULL:  Used to determines if the number of courses scheduled
;;   in the quarter has exceeded the maximum available.
;;

(defun Quarter-full (count max)
  (>= count max))


;;
;; INCREMENT: Simply returns bindings for incrementing old by 1.
;;   If either one is a variable, it will generate possible bindings
;;   for successfull unification
;;

(defun Increment (old-count new-count)
  (cond
    ((and (is-variable old-count) (is-variable new-count))
     (append
      (binding-list old-count *CLASS-COUNT-OPTIONS*)
      (binding-list new-count (rest *CLASS-COUNT-OPTIONS*))))

    ((is-variable old-count)
     (if (>= (- new-count 1) 0)
	 (binding-list old-count (list (- new-count 1)))
	 'no-match-attempted))

    ((is-variable new-count)
     (binding-list new-count (list (+ old-count 1))))

    (t (= (+ old-count 1) new-count))))


;;
;; NO-TIME-CONFLICT:  Used to determine if the start and end times of
;;   of two candidate in the same quarter (assumed) conflict.  
;;

(defun No-time-conflict (start1 end1 start2 end2)
  (cond
    ((= start1 0) t)
    ((= start2 0) t)
    ((AND (< start1 start2) (< end1 start2)) t)
    ((AND (> start1 start2) (> start1 end2)) t)
    (t nil)))


;;
;; QUARTER-LESS-THAN:  Determines if, given a pair of quarter and
;;   years if the first is less than the latter.
;;
 
(defun Quarter-less-than (quarter1 year1 quarter2 year2)
  (cond
    ( (< year1 year2) t)
    ( (AND (= year1 year2) (< quarter1 quarter2)) t)
    ( t nil)))


;;
;; CORE-COMPLETED: Given a count of credits already scheduled and the
;;   goal, this determines if completion.  If variables are presented
;;   then it will generate possible bindings for successful unification
;;

(defun Core-completed (count goal)
  (cond 
    ((and (is-variable count) (is-variable goal))
     'no-match-attempted)
    ((is-variable count)
     ;; generate all numbers possible past goal
     (binding-list count (Core-generate-count goal)))
    ((is-variable goal)
     'no-match-attempted)
    (t (>= count goal))))


;;
;; CORE-GENERATE-COUNT:  Used by Core-completed to generate possible
;;   bindings for the variable "count".  Generates all numbers
;;   possible past the goal up to *MAX-CREDIT-CLASS*.
;;
;; For *MAX-CREDIT-CLASS* = 5:
;; (Core-generate-count 20) returns: (20 21 22 23 24)
;;

(defun Core-generate-count (goal &optional (n 0))
  (cond
    ((= n *MAX-CREDIT-CLASS*) nil)
    (t (cons (+ goal n) (Core-generate-count goal (+ n 1))))))


;;
;; GENERATE-CREDITS: An incrementor function that increments "old" to 
;;   "new" by a step value "cred".  This function will handle ALL 
;;   cases if a ANY of the parameters are variables.. ie. generate
;;   bindings for possible successful unification.
;;

(defun Generate-credits (old new cred)
  (cond ((and (is-variable old) (is-variable new) (is-variable cred))
	  'no-match-attempted)
	((and (is-variable old) (is-variable new))
	 (append
	  (binding-list old        (generate-old-new-credits cred))
	  (binding-list new  (rest (generate-old-new-credits cred)))))

	((and (is-variable old) (is-variable cred))
	 (append
	  (binding-list cred *CREDIT-OPTIONS*)
	  (binding-list old  (generate-old-credits new *CREDIT-OPTIONS*))))

	((and (is-variable new) (is-variable cred))
	 (append
	  (binding-list cred *CREDIT-OPTIONS*)
	  (binding-list new  (generate-new-credits old *CREDIT-OPTIONS*))))

	((is-variable old)
	 (if (>= (- new cred) 0)
	     (binding-list old (list (- new cred)))
	     'no-match-attempted))

	((is-variable new)
	 (binding-list new  (list (+ old cred))))

	((is-variable cred)
	 (if (>= (- new old) 0)
	     (binding-list cred (list (- new old)))
	     'no-match-attempted))

	(t (= new (+ old cred)))))

;;
;; GENERATE-OLD-NEW-CREDITS:  Used by Generate-Credits when ONLY
;;   credits is known and possible bindings for OLD and NEW
;;   must be generated.  Generates numbers from 0 to *MAX-CREDITS*
;;   stepping by CRED.
;;
;; For *MAX-CREDITS* = 20:
;;   (generate-old-new-credits 3)  returns: (3 6 9 12 15 18 21)
;;   (generate-old-new-credits 4)  returns: (4 8 12 16 20)
;;   (generate-old-new-credits 5)  returns: (5 10 15 20)

(defun generate-old-new-credits (cred &optional (n 0))
  (cond
    ((>= n *MAX-CREDITS*) nil)
    ((cons (+ cred n) (generate-old-new-credits cred (+ n cred))))))


;;
;; GENERATE-NEW-CREDITS:  Used by Generate-Credits when we have
;;   ONLY OLD bound and we must generate possible bindings
;;   for NEW.  Generates numbers by adding each CRED candidate
;;   to OLD.
;;
;; cred = *CREDIT-OPTIONS* = '(3 4 5):
;;   (generate-new-credits 9 cred)  returns: (12 13 14)
;;   (generate-new-credits 0 cred)  returns: (3 4 5)
;;

(defun generate-new-credits (old cred)
  (cond
    ((null cred) nil)
    (t (cons (+ (first cred) old) (generate-new-credits old (rest cred))))))


;;
;; GENERATE-OLD-CREDITS:  Used by Generate-Credits when we have
;;   ONLY NEW bound and we must generate possible bindings
;;   for OLD.  Generates numbers by subtracting each CRED candidate
;;   from NEW.
;;
;; For cred = *CREDIT-OPTIONS* = '(3 4 5):
;;   (generate-old-credits 9 cred)  returns: (6 5 4)
;;   (generate-old-credits 5 cred)  returns: (2 1 0)
;;   (generate-old-credits 4 cred)  returns: (1 0)
;;

(defun generate-old-credits (new cred)
  (cond
    ((null cred) nil)
    ((>= (- new (first cred)))
     (cons (- new (first cred)) (generate-old-credits new (rest cred))))
    (t nil)))


(defun binding-list (var val-list)
  (cond ((null val-list) nil)
        ((null (car val-list)) (binding-list var (cdr val-list)))
        (t (append (list (list (list var (car val-list))))
                   (binding-list var (cdr val-list))))))

