;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Campaign world
;;
;;       There are three different non-static generators used throughout this domain.  They are:
;;  TIME, FUNDS, and, TOTAL-VOTES.  The reason that these are used is that the values they
;;  represent are constantly changing.  Time is constantly decreasing, funds changed depending
;;  on weather the operator earns you money, or costs you money, and total-votes increases by an
;;  ammount dependant upon the number of votes won in each state by the operator.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(setq *OPERATORS* '(

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The FLY operator will allow the candidate to fly from one location to another.  It is most
;; useful when the location isn't reachable by driving (such as Hawaii, or Alaska).  It costs
;; more than driving, and takes less time, which is shown in a difference in the number of votes
;; gained at the destination.
;;

(FLY
 (PARAMS (<start-loc> <end-loc> <start-orientation> <end-orientation> <money> <old-time>
         <current-votes>))
 (PRECONDS
  (AND
   (POS-LOC <start-loc> <start-orientation>)
   (AT-LOC <start-loc>)
   (POS-LOC <end-loc> <end-orientation>)
   (FUNDS <money>)
   (GREATER-THAN <money> 899)
   (TIME-LEFT <old-time>)
   (DECRIMENT-FUNDS <new-money> <money> 900)   ;; decriment funds by 900 and assign to new-money.
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 3)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <old-time> 0)
   (DECRIMENT-TIME <new-time> <old-time>))) ;; decriment time by 1 and assign to new-time.

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <old-time>))
  (DEL (AT-LOC <start-loc>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (TIME-LEFT <new-time>))
  (ADD (FUNDS <new-money>))
  (ADD (AT-LOC <end-loc>))

  (IF (AND                                     ;; change the vote counts in various locations
       (POS-LOC <loc1> <orient1>)              ;; where the candidate isn't actively campaigning.
       (NOT-EQUAL <loc1> <end-loc>)            ;; Assumes the home-state advantage by not changing
       (CAND-ORIENT <orient1>)                 ;; the home state count while the candidate is away.
       (~ (HOME-STATE <loc1>))                 ;; Also assumes the candidate will lose more ground
       (STATE <loc1> <pos1> <neg1> <neut1>)    ;; in a location whose political orientation doesn't
       (GREATER-THAN <neut1> 3)                ;; correspond to the candidate's orientation.
       (DECRIMENT-VOTES <new-neut1> <neut1> 1)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>)))

  (IF (AND
       (POS-LOC <loc2> <orient2>)
       (NOT-EQUAL <loc2> <end-loc>)
       (~ (CAND-ORIENT <orient2>))
       (~ (HOME-STATE <loc2>))
       (STATE <loc2> <pos2> <neg2> <neut2>)
       (GREATER-THAN <neut2> 3)
       (DECRIMENT-VOTES <new-neut2> <neut2> 2)
       (GENERATE-VOTES <new-neg2> <neg2> 2))

      (ADD (STATE <loc2> <pos2> <new-neg2> <new-neut2>))
      (DEL (STATE <loc2> <pos2> <neg2> <neut2>)))

  (IF (AND                                         ;; Change the vote count at the location where
       (STATE <end-loc> <pos3> <neg3> <neut3>)     ;; the candidate arrives, assumes minor 
       (GREATER-THAN <neut3> 3)                    ;; campaigning.
       (DECRIMENT-VOTES <new-neut3> <neut3> 3)
       (GENERATE-VOTES <new-pos3> <pos3> 3))

      (ADD (STATE <end-loc> <new-pos3> <neg3> <new-neut3>))
      (DEL (STATE <end-loc> <pos3> <neg3> <neut3>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The DRIVE operator allows the candidate to drive to any location that isn't unreachable by
;; car.  The usefulness of the drive operator is that it costs less money than FLY, on the other
;; hand, drive takes more time, and thus allows less time for campaigning once the destination is
;; reached.
;;

(DRIVE
 (PARAMS (<start-loc> <end-loc>))
 (PRECONDS
  (AND
   (POS-LOC <start-loc> <start-orientation>)
   (AT-LOC <start-loc>)
   (POS-LOC <end-loc> <end-orientation>)
   (DECRIMENT-TIME <new-time> <old-time>)             ;; decriment old-time by time taken.
   (DECRIMENT-FUNDS <new-money> <money> 200)           ;; decriment money by the cost to drive.
   (FUNDS <money>)
   (GREATER-THAN <money> 299)
   (TIME-LEFT <old-time>)
   (GREATER-THAN <old-time> 0)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 1)
   (TOTAL-VOTES <current-votes>)
   (~ (ISOLATED-LOC <start-loc>))
   (~ (ISOLATED-LOC <end-loc>))))

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (AT-LOC <start-loc>))
  (DEL (TIME-LEFT <old-time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (AT-LOC <end-loc>))
  (ADD (TIME-LEFT <new-time>))
  (ADD (FUNDS <new-money>))

  (IF (AND                                     ;; change the vote counts in various locations
       (POS-LOC <loc1> <orient1>)              ;; where the candidate isn't actively campaiging.
       (NOT-EQUAL <loc1> <end-loc>)            ;; Takes into account both differences in the 
       (CAND-ORIENT <orient1>)                 ;; orientation of the candidate vs. the general
       (~ (HOME-STATE <loc1>))                 ;; orientation of the location, as well as the home
       (STATE <loc1> <pos1> <neg1> <neut1>)    ;; state advantage.
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 1)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>)))

  (IF (AND
       (POS-LOC <loc2> <orient2>)
       (NOT-EQUAL <loc2> <end-loc>)
       (~ (CAND-ORIENT <orient2>))
       (~ (HOME-STATE <loc2>))
       (STATE <loc2> <pos2> <neg2> <neut2>)
       (GREATER-THAN <neut2> 3)
       (DECRIMENT-VOTES <new-neut2> <neut2> 2)
       (GENERATE-VOTES <new-neg2> <neg2> 2))

      (ADD (STATE <loc2> <pos2> <new-neg2> <new-neut2>))
      (DEL (STATE <loc2> <pos2> <neg2> <neut2>)))

  (IF (AND                                        ;; increase the positive vote count in the 
       (STATE <end-loc> <pos3> <neg3> <neut3>)    ;; location by an ammount representing the 
       (GREATER-THAN <neut3> 2)                   ;; time given for minor campaigning.
       (DECRIMENT-VOTES <new-neut3> <neut3> 1)
       (GENERATE-VOTES <new-pos3> <pos3> 1))

      (ADD (STATE <end-loc> <new-pos3> <neg3> <new-neut3>))
      (DEL (STATE <end-loc> <pos3> <neg3> <neut3>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The STAY operator allows the candidate to stay in a location, giving the person a chance to 
;; do some minor campaigning and gain constituency at a fairly low cost.  
;;

(STAY
 (PARAMS (<loc> <time> <money>))
 (PRECONDS
  (AND
   (POS-LOC <end-loc> <end-orientation>)
   (AT-LOC <end-loc>)
   (FUNDS <money>)
   (TIME-LEFT <old-time>)
   (GREATER-THAN <old-time> 0)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 3)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <money> 199)
   (DECRIMENT-TIME <new-time> <old-time>)       ;; decriment old-time by time stayed.
   (DECRIMENT-FUNDS <new-money> <money> 200)))  ;; decriment money by the cost of staying without
                                                ;;  major expenses.
 (EFFECTS (

  (DEL (TIME-LEFT <old-time>))
  (DEL (FUNDS <money>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                     ;; change the vote counts in various locations
       (POS-LOC <loc1> <orient1>)              ;; to take into account that the candidate isn't 
       (NOT-EQUAL <loc1> <end-loc>)            ;; actively campaigning in those locations.  Also
       (CAND-ORIENT <orient1>)                 ;; takes into account differences in political
       (~ (HOME-STATE <loc1>))                 ;; orientation and the home-state advantage.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 1)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>)))

  (IF (AND
       (POS-LOC <loc2> <orient2>)
       (NOT-EQUAL <loc2> <end-loc>)
       (~ (CAND-ORIENT <orient2>))
       (~ (HOME-STATE <loc2>))
       (STATE <loc2> <pos2> <neg2> <neut2>)
       (GREATER-THAN <neut2> 3)
       (DECRIMENT-VOTES <new-neut2> <neut2> 2)
       (GENERATE-VOTES <new-neg2> <neg2> 2))

      (ADD (STATE <loc2> <pos2> <new-neg2> <new-neut2>))
      (DEL (STATE <loc2> <pos2> <neg2> <neut2>)))

  (IF (AND 
       (STATE <end-loc> <pos3> <neg3> <neut3>)   ;; Add to the vote count in the place where you 
       (GREATER-THAN <neut3> 3)                  ;; are.
       (DECRIMENT-VOTES <new-neut3> <neut3> 3)
       (GENERATE-VOTES <new-pos3> <pos3> 3))

      (ADD (STATE <end-loc> <new-pos3> <neg3> <new-neut3>))
      (DEL (STATE <end-loc> <pos3> <neg3> <neut3>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The SPONSOR-BANQUET operator allows the candidate to throw exclusive "parties" for rich
;; supporters at the cost of $200 a plate, or something like that.  The requirement that the 
;; candidate have money is to show the setup cost for the banquet.  It is designed as a way to 
;; get money without having to worry about campaign contribution limitations.
;;

(SPONSOR-BANQUET
 (PARAMS (<money> <end-loc> <end-orientation>))
 (PRECONDS
  (AND
   (POS-LOC <end-loc> <end-orientation>)
   (AT-LOC <end-loc>)
   (GOOD-SUPPORT <end-loc>)
   (FUNDS <money>)
   (GREATER-THAN <money> 499)
   (TIME-LEFT <time>)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 2)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 2)
   (DECRIMENT-TIME <new-time> <time>)          ;; decriment time by time taken to prepare banquet.
   (INCRIMENT-FUNDS <new-money> <money> 600))) ;; incriment money by ammount earned after expences.

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                     ;; change the vote counts in various locations
       (POS-LOC <loc1> <orient1>)              ;; to show the lack of active campaigning at the 
       (NOT-EQUAL <loc1> <end-loc>)            ;; various locations.  It takes into account both
       (CAND-ORIENT <orient1>)                 ;; differences in political orientation, and the
       (~ (HOME-STATE <loc1>))                 ;; home state advantage.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 1)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>)))

  (IF (AND
       (POS-LOC <loc2> <orient2>)
       (NOT-EQUAL <loc2> <end-loc>)
       (~ (CAND-ORIENT <orient2>))
       (~ (HOME-STATE <loc2>))
       (STATE <loc2> <pos2> <neg2> <neut2>)
       (GREATER-THAN <neut2> 3)
       (DECRIMENT-VOTES <new-neut2> <neut2> 2)
       (GENERATE-VOTES <new-neg2> <neg2> 2))

      (ADD (STATE <loc2> <pos2> <new-neg2> <new-neut2>))
      (DEL (STATE <loc2> <pos2> <neg2> <neut2>)))

  (IF (AND 
       (STATE <end-loc> <pos3> <neg3> <neut3>)      ;; assumes some campaigning while working on
       (GREATER-THAN <neut3> 3)                     ;; the banquet.
       (DECRIMENT-VOTES <new-neut3> <neut3> 2)
       (GENERATE-VOTES <new-pos3> <pos3> 2))

      (ADD (STATE <end-loc> <new-pos3> <neg3> <new-neut3>))
      (DEL (STATE <end-loc> <pos3> <neg3> <neut3>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The SPEAK-WITH-BUISNESS operator alows the candidate to lobby buisnesses for contributions.
;; It takes into account a certain hesitation by buisnesses to give money to candidates who
;; don't entirely represent their views.
;;

(SPEAK-WITH-BUISNESS
 (PARAMS (<end-loc> <end-orientation> <name> <time> <money>))
 (PRECONDS
  (AND
   (POS-LOC <end-loc> <end-orientation>)
   (AT-LOC <end-loc>)
   (TIME-LEFT <time>)
   (FUNDS <money>)
   (GREATER-THAN <time> 0)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 3)
   (TOTAL-VOTES <current-votes>)
   (BUISNESS <name> <end-loc> <buis-orientation>)
   (DECRIMENT-TIME <new-time> <time>)))            ;; adjust time by time taken to lobby buisness.

 (EFFECTS (

  (DEL (TIME-LEFT <time>))
  (DEL (FUNDS <money>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND
       (CAND-ORIENT <buis-orientation>)
       (INCRIMENT-FUNDS <new-money> <money> 2000))  ;; adjust money by ammount given in support

      (ADD (FUNDS <new-money>)))

  (IF (AND
       (~ (CAND-ORIENT <buis-orientation>))
       (INCRIMENT-FUNDS <new-money> <money> 800))   ;; adjust money by ammount given as show of
                                                    ;; support.
      (ADD (FUNDS <money>)))

  (IF (AND                                     ;; change the vote counts in various locations
       (POS-LOC <loc1> <orient1>)              ;; where the candidate isn't actively campaigning.
       (NOT-EQUAL <loc1> <end-loc>)            ;; Also take into account diffences in political
       (CAND-ORIENT <orient1>)                 ;; orientation, and the home state advantage.
       (~ (HOME-STATE <loc1>))
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 1)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>)))

  (IF (AND
       (POS-LOC <loc2> <orient2>)
       (NOT-EQUAL <loc2> <end-loc>)
       (~ (CAND-ORIENT <orient2>))
       (~ (HOME-STATE <loc2>))
       (STATE <loc2> <pos2> <neg2> <neut2>)
       (GREATER-THAN <neut2> 3)
       (DECRIMENT-VOTES <new-neut2> <neut2> 2)
       (GENERATE-VOTES <new-neg2> <neg2> 2))

      (ADD (STATE <loc2> <pos2> <new-neg2> <new-neut2>))
      (DEL (STATE <loc2> <pos2> <neg2> <neut2>)))

  (IF (AND                                       ;; assumes some local campaigning while lobbying
       (STATE <end-loc> <pos3> <neg3> <neut3>)   ;; the buisness.
       (GREATER-THAN <neut3> 3)
       (DECRIMENT-VOTES <new-neut3> <neut3> 3)
       (GENERATE-VOTES <new-pos3> <pos3> 3))

      (ADD (STATE <end-loc> <new-pos3> <neg3> <new-neut3>))
      (DEL (STATE <end-loc> <pos3> <neg3> <neut3>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The MAKE-CHEAP-POSITIVE-COMMERCIAL allows the candidate to make some cheap positive commercials
;; for national viewing.  It reflects a fairly low cost of production, and one time-slots worth of
;; airing.
;;

(MAKE-CHEAP-POSITIVE-COMMERCIAL
 (PARAMS (<money> <time>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <money> 849)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 27)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 0)
   (DECRIMENT-TIME <new-time> <time>)          ;; adjust time to account for filming, and ideas.
   (DECRIMENT-FUNDS <new-money> <money> 800)))    ;; adjust funds to account for cost of "cheap"
                                                  ;; commercial.
 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                   ;; adjust votes to account for national playing and
       (POS-LOC <loc1> <orientation1>)       ;; differring political attitudes, also takes into
       (STATE <loc1> <pos1> <neg1> <neut1>)  ;; account the low quality of the commercial.
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 6)
       (GENERATE-VOTES <new-pos1> <pos1> 3)
       (GENERATE-VOTES <new-neg1> <neg1> 3))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The MAKE-MODERATE-POSITIVE-COMMERCIAL allows the candidate to make a moderately expensive
;; positive commercial.  Again the cost takes into account the moderate cost of production and
;; only one time slot of national air time.
;;

(MAKE-MODERATE-POSITIVE-COMMERCIAL
 (PARAMS (<money> <time>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <money> 1099)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 36)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 0)
   (DECRIMENT-TIME <new-time> <time>)       ;; adjust time and funds to account for expences of add
   (DECRIMENT-FUNDS <new-money> <money> 1100)))

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                  ;; adjust votes to account for quality and national
       (POS-LOC <loc1> <orientation1>)      ;; play of add.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 4)
       (DECRIMENT-VOTES <new-neut1> <neut1> 6)
       (GENERATE-VOTES <new-pos1> <pos1> 4)
       (GENERATE-VOTES <new-neg1> <neg1> 2))
      
      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The MAKE-EXPENSIVE-POSITIVE-COMMERCIAL operator allows the candidate to make expencively 
;; produced commercials for national airing.  The cost takes into account the higher cost of
;; production, and the cost of one time slot's worth of airing.
;;

(MAKE-EXPENSIVE-POSITIVE-COMMERCIAL
 (PARAMS (<money> <time>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <money> 1499)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 45)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 1)
   (DECRIMENT-TIME <new-time> <time>)      ;; adjust time and money to account for expences of add
   (DECRIMENT-FUNDS <new-money> <money> 1500)))

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                 ;; adjust votes to account for nation play, and quality,
       (POS-LOC <loc1> <orientation1>)     ;; of add.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 5)
       (DECRIMENT-VOTES <new-neut1> <neut1> 6)
       (GENERATE-VOTES <new-pos1> <pos1> 5)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The MAKE-CHEAP-NEGATIVE-COMMERCIAL operator allows the candidate to make low cost dirt add 
;; would damage an opponents reputation.  Again the cost represents the cost of production and
;; air time.  The changes in the vote count represent reactions of those who don't like slam adds,
;; as well as those who would be influenced by it.
;;

(MAKE-CHEAP-NEGATIVE-COMMERCIAL
 (PARAMS (<money> <time>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <money> 849)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 27)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 0)
   (DECRIMENT-TIME <new-time> <time>)      ;; adjust time and money for cost of add.
   (DECRIMENT-FUNDS <new-money> <money> 800)))

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                ;; adjust votes taking into account quality of add, and
       (POS-LOC <loc1> <orientation1>)    ;; negative orientation of add.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 7)
       (GENERATE-VOTES <new-pos1> <pos1> 3)
       (GENERATE-VOTES <new-neg1> <neg1> 4))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The MAKE-MODERATE-NEGATIVE-COMMERCIAL operator allows the candidate to make a moderately 
;; expensive negative commercial.  The cost takes into account production and air time, and the 
;; vote count changes take into account the improved effects of the quality of the add, as well as
;; the adverse effects of it being a negative add.
;;

(MAKE-MODERATE-NEGATIVE-COMMERCIAL
 (PARAMS (<money> <time>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <money> 1099)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 36)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 0)
   (DECRIMENT-TIME <new-time> <time>)        ;; adjust time and money for cost of add.
   (DECRIMENT-FUNDS <new-money> <money> 1100)))

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                ;; adjust votes, taking into account the quality and
       (POS-LOC <loc1> <orientation1>)    ;; negative orientation of the add.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 7)
       (GENERATE-VOTES <new-pos1> <pos1> 4)
       (GENERATE-VOTES <new-neg1> <neg1> 3))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The MAKE-EXPENSIVE-NEGATIVE-COMMERCIAL operator allows the candidate to make expensive
;; negative commercials that are very good at hurting the repuatation of the opponent.  The cost
;; takes into account the cost of production, and the vote count takes into account the quality
;; of the add as well as the fact that it is negative in orientation.
;;

(MAKE-EXPENSIVE-NEGATIVE-COMMERCIAL
 (PARAMS (<money> <time>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <money> 1499)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 45)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 1)
   (DECRIMENT-TIME <new-time> <time>)       ;; adjust time and money to reflect cost of add.
   (DECRIMENT-FUNDS <new-money> <money> 1500)))

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                               ;; adjust votes taking into account the quality and 
       (POS-LOC <loc1> <orientation1>)   ;; negative nature of the add.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 4)
       (DECRIMENT-VOTES <new-neut1> <neut1> 7)
       (GENERATE-VOTES <new-pos1> <pos1> 5)
       (GENERATE-VOTES <new-neg1> <neg1> 2))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The SPEAK-TO-PUBLIC operator allows the candidate to speak to a group of people at a specific
;; location.  It assumes the speach is, at least partly, covered by the local television, thus 
;; providing a reasonable audience for the speach.  The vote count changes take into account the 
;; fact that the speach isn't addressed to the concerns of the whole nation, but to just one
;; location.  Thus we get into the situation of the candidate not actively influencing a vast 
;; majority of the nation.
;;

(SPEAK-TO-PUBLIC
 (PARAMS (<money> <time> <end-loc> <end-orientation>))
 (PRECONDS
  (AND
   (FUNDS <money>)
   (GREATER-THAN <money> 399)
   (POS-LOC <end-loc> <end-orientation>)
   (AT-LOC <end-loc>)
   (TIME-LEFT <time>)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 4)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 0)
   (DECRIMENT-TIME <new-time> <time>)       ;; adjust time and money for the cost of organising the
   (DECRIMENT-FUNDS <new-money> <money> 400))) ;; speach.

 (EFFECTS (

  (DEL (FUNDS <money>))
  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (FUNDS <new-money>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                                     ;; change the vote counts in various locations
       (POS-LOC <loc1> <orient1>)              ;; assuming the candidate isn't actively campaigning
       (NOT-EQUAL <loc1> <end-loc>)            ;; in those locations.  It takes into account both
       (~ (HOME-STATE <loc1>))                 ;; differences in politica orientation, and the
       (CAND-ORIENT <orient1>)                 ;; home state advantage.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 3)
       (DECRIMENT-VOTES <new-neut1> <neut1> 1)
       (GENERATE-VOTES <new-neg1> <neg1> 1))

      (ADD (STATE <loc1> <pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>)))

  (IF (AND
       (POS-LOC <loc2> <orient2>)
       (NOT-EQUAL <loc2> <end-loc>)
       (~ (CAND-ORIENT <orient2>))
       (~ (HOME-STATE <loc2>))
       (STATE <loc2> <pos2> <neg2> <neut2>)
       (GREATER-THAN <neut2> 3)
       (DECRIMENT-VOTES <new-neut2> <neut2> 2)
       (GENERATE-VOTES <new-neg2> <neg2> 2))

      (ADD (STATE <loc2> <pos2> <new-neg2> <new-neut2>))
      (DEL (STATE <loc2> <pos2> <neg2> <neut2>)))

  (IF (AND 
       (STATE <end-loc> <pos3> <neg3> <neut3>)  ;; adjust the local vote counts to take into 
       (GREATER-THAN <neut3> 3)                 ;; account the positive effect of the speach.
       (DECRIMENT-VOTES <new-neut3> <neut3> 3)
       (GENERATE-VOTES <new-pos3> <pos3> 3))

      (ADD (STATE <end-loc> <new-pos3> <neg3> <new-neut3>))
      (DEL (STATE <end-loc> <pos3> <neg3> <neut3>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The SPEAK-WITH-SPECIAL-INTEREST operator allows the candidate to speak with a special interest
;; group in an attempt to gain the support of that group.  If given support, then you get money
;; as well as a boost in your constituency.  You may only use special interest groups once, given
;; a regulation as to the ammount of money any one SI group may donate to the candidate.
;;

(SPEAK-WITH-SPECIAL-INTEREST
 (PARAMS (<time> <money> <end-loc> <si-orientation>))
 (PRECONDS
  (AND
   (POS-SI-GROUP <si-group> <si-orientation> <end-loc>)
   (AT-LOC <end-loc>)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 54)
   (TOTAL-VOTES <current-votes>)
   (FUNDS <money>)
   (TIME-LEFT <time>)
   (GREATER-THAN <time> 1)
   (DECRIMENT-TIME <new-time> <time>)))   ;; adjust for time taken to lobby the special interest
                                          ;; group.
 (EFFECTS (

  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND
       (CAND-ORIENT <si-orientation>)
       (INCRIMENT-FUNDS <new-money> <money> 2000))  ;; if group likes position, add to funds.

      (ADD (FUNDS <new-money>))
      (DEL (FUNDS <money>)))

  (DEL (POS-SI-GROUP <si-group> <si-orientation> <end-loc>))
  (ADD (VISITED-SI-GROUP <si-group>))

  (IF (AND                                     ;; adjust votes according to how well liked the
       (POS-LOC <loc1> <si-orientation>)       ;; the candidate is by the SI group, and how well
       (STATE <loc1> <pos1> <neg1> <neut1>)    ;; the SI group is liked by each specific location.
       (GREATER-THAN <netu1> 7)
       (DECRIMENT-VOTES <new-neut1> <neut1> 8)
       (GENERATE-VOTES <new-pos1> <pos1> 6)
       (GENERATE-VOTES <new-neg1> <neg1> 2))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The TAKE-CONTROVERCIAL-STANCE operator represents the often last-ditch method of taking a
;; stance on a pollitically senceative issue.  There is generally both a strong positive, and
;; a strong negative reaction to the stance taken.  This is represented by the fairly large
;; change in both the positive and negative vote counts.
;;

(TAKE-CONTROVERCIAL-STANCE
 (PARAMS (<time> <orientation>))
 (PRECONDS
  (AND 
   (CAND-ORIENT <orientation>)
   (TIME-LEFT <time>)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 45)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 0)
   (DECRIMENT-TIME <new-time> <time>)))  ;; adjust time for time taken to get stance to the public
                                         ;; ear.
 (EFFECTS (

  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                            ;; adjust the vote counts for the impact the stance will have
       (POS-LOC <loc1> <orientation>) ;; on the voters.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 10)
       (DECRIMENT-VOTES <new-neut1> <neut1> 10)
       (GENERATE-VOTES <new-pos1> <pos1> 5)
       (GENERATE-VOTES <new-neg1> <neg1> 5))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  The DEBATE operator represents a nationally televised political debate.  The results of these
;; things seem to be, in general, determined by the political orientation of the viewer (barring
;; such people as Dan Quayle).  Thus the operator assumes the candidate is a generally competent
;; debator.
;;

(DEBATE
 (PARAMS (<location> <time>))
 (PRECONDS
  (AND
   (POS-LOC <location> <orientation>)
   (AT-LOC <location>)
   (TIME-LEFT <time>)
   (INCRIMENT-TOTAL-VOTES <new-votes> <current-votes> 27)
   (TOTAL-VOTES <current-votes>)
   (GREATER-THAN <time> 1)
   (DECRIMENT-TIME <new-time> <time>))) ;; adjust time for time taken to prepare and participate in
                                        ;; debate.
 (EFFECTS (

  (DEL (TIME-LEFT <time>))
  (DEL (TOTAL-VOTES <current-votes>))
  (ADD (TOTAL-VOTES <new-votes>))
  (ADD (TIME-LEFT <new-time>))

  (IF (AND                             ;; adjust votes taking into account how each state would be
       (POS-LOC <loc1> <orient1>)      ;; likely to view the outcome, dependant on candidiat vs.
       (STATE <loc1> <pos1> <neg1> <neut1>)
       (GREATER-THAN <neut1> 4)
       (DECRIMENT-VOTES <new-neut1> <neut1> 6)
       (GENERATE-VOTES <new-pos1> <pos1> 3)
       (GENERATE-VOTES <new-neg1> <neg1> 3))

      (ADD (STATE <loc1> <new-pos1> <new-neg1> <new-neut1>))
      (DEL (STATE <loc1> <pos1> <neg1> <neut1>))))))

(ADD-HAS-WON
 (PRECONDS
  (AND
   (MIN-TO-WIN <votes-needed>)
   (TOTAL-VOTES <votes-needed>)))
 (EFFECTS (
  (ADD (HAS-WON)))))

))
 