;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Domain specific information
;;
;; Author: Oren Etzioni
;; Modified:  Julie Roomy, Rob Spiger
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



;;------------
;; read-domain
;; read the domain from the global set by prodigy
;; if adding new domain format is:
;; ((equal *current-domain* "<domain name in prodigy>")   '<abbreviation>)
(defun read-domain ()
  (cond
   ((equal *current-domain* "blocksworld")    'bw)
   ((equal *current-domain* "bw")    'bw)
   ((equal *current-domain* "frozenblocksworld")    'mbw)
;;"frozenblocksworld" was "mod-bw"  
   ((equal *current-domain* "schedworld")    'sched)
   ((or (equal *current-domain* "strips")
		(equal *current-domain* "extended-strips"))    'strips)
   ((or (equal *current-domain* "kites")
		(equal *current-domain* "kite-building"))    'kites)
   ((equal *current-domain* "test")    'test)
   (T *current-domain*)))




;****************************************************************
; Domain dependent axioms.

; Part of specifying a domain:
; non-goals--these predicates will never be goals.
;;  used in generating achievables
; non-gratis-list--these predicates are assumed to always be true
;;  labeled p-holds for predicate holds
; bad-ops---don't try these operators for these goals.
;;  usually encoded from previous control rules, used by relevant-operators
;;
;;  Note:  It is not guaranteed that putting an entry in the bad-ops
;;  list will stop static from generating the same rule.  This is
;;  because the bad-ops list is only referenced when looking for
;;  relevant operators for a goal.  The bad-ops list does not block
;;  substitutions later on which result in something that would have
;;  been orginally removed by the bad-ops list if the substitution had
;;  taken place from the start.  The problem comes from using
;;  constants in the bad-ops list.  For example, look at this example
;;  from the extended-strips world:
;;
;;  (((next-to robot <x>) push-box)
;;   ((next-to <x> <y>) push-thru-dr))
;;
;;  When you ask for the operators for the goal (next-to <x> <y>) the
;;  operator push-box is okay to use because <x> does not necessarily
;;  equal robot so you can't rule out the operator push-box.
;;  So the PSG is created using the operator push-box.  But somewhere 
;;  farther down in the PSG, maybe in another operator, <x> may be 
;;  constrained to equal robot.  This will result in effectively
;;  having used the operator push-box to achieve (next-to robot <x>).
;;
; invariant-list---facts that have to hold.
;;  used by holds?
; false-list---facts that are always false.
;;  these predicates would never occur in prodigy
; wild-cards---wild card variables inferred from the operators.
; dont-expand-list---predicates which are too expensive to expand.
;;  labeled unknown unless hold in current state.


; For goal interactions:
; negation-list---facts that negate each other.
; rarely-unique-list---static generators which rarely distinguish
;   something as unique.  For example, is-object in
;   extended-stripsworld, because there are usually many objects.



; This is my way of encoding the ck in default crules. 
(defun make-bad-ops (domain)
  (setq *bad-ops*
    (cond ((eq domain 'mbw) nil)
	  ((eq domain 'processp) nil)

	  ;;original domains
	  ((eq domain 'test) nil)	;this is the original test
	  ((eq domain 'bw) nil)
	  ((eq domain 'sched)
	   '(((is-object <x>) weld bolt))
	   )
	  ((eq domain 'strips)
	   '(((next-to robot <x>) push-box)
	     ((next-to <x> <y>) push-thru-dr)
	     ((inroom robot <x>) carry-thru-dr push-thru-dr)
	     ((arm-empty) putdown-next-to)
	     ((inroom <x> <y>) putdown-next-to)))

	  ((eq domain 'kites)
	   ;; the only way of changing a stick-type is to change from 
	   ;; bamboo to bamboo-strip
	   '(((stick-properties <stick> type plastic) cut-stick-with-saw)
	     ((stick-properties <stick> type wooden) cut-stick-with-saw)
	     ((stick-properties <stick> type bamboo-strip) cut-stick-with-saw)
	     ((stick-properties <stick> type tubing) cut-stick-with-saw)
	     
	     ;; striping bamboo does not change it's size
	     ((stick-properties <stick> size <stick-size>) 
	      strip-bamboo-with-knife)
	     
	     ;; there is no way to unnotch a stick, these operators delete
	     ;; all the properties of a stick from the state, when they
	     ;; delete the stick because it has been transformed into
	     ;; something else.
	     ;; Ex: build-*-frame adds a frame to the state and deletes
	     ;;  those sticks (and their properties) from the state that
	     ;;  were used to build the frame.
	     ((~ (stick-properties <stick> notched <point>))
	      tube-two-sticks-with-tubing      
	      cut-stick-with-saw
	      strip-bamboo-with-knife
	      build-four-circle-frame
	      build-semicircle-frame
	      build-rectangle-frame
	      build-diamond-frame
	      build-stretch-hexagon-frame)

	     ;; there is no way to unlash a stick, (see discussion on unnotch
	     ;; above).
	     ((~ (stick-properties <stick> lashed <point>))
	      tube-two-sticks-with-tubing
	      cut-stick-with-saw
	      strip-bamboo-with-knife
	      build-four-circle-frame
	      build-semicircle-frame
	      build-rectangle-frame
	      build-diamond-frame
	      build-stretch-hexagon-frame)))

	  ;;other domains
	  (t nil))))


; is-width is necessary because negating it (with vars unbound) causes
; the matcher to crash.  I can't put idle in because it's defined
; using a forall.  regular-shape is here so that I don't have to
; reason about bindings.  It's bindings are generated by shape, so I'd
; have to include shape in the lhs if I check regular-shape, but
; static doesn't try to figure that out.  (it would need to check that
; it has the generator for each variable).  It seems that
; regular-shape is unachievable, but in fact subgoaling on shape can
; make it true.  I think this is actually an incompleteness for
; prodigy.

(defun make-non-gratis-list (domain)
  (setq *non-gratis-list*
	(cond ((eq domain 'mbw) '(object))
	      ((eq domain 'processp) nil)

	      ;;original domains
	      ((member domain '(bw mbb paper-bw min-bad-bw)) '(object))
	      ((eq domain 'bad-bw) '(object is-arm))
	      ((eq domain 'strips) nil)
;;removed: composite-object.	      
;;; generators: is-time is-width.
;;; has-clamp is always true.
;;; regular-shape (see top)
;;;  As a result 
;;; later, last-scheduled, and scheduled are removed.
;;; if last-scheduled is added then interactions are developed
;;; because the timing stuff deletes (last-schedule).  I might be able
;;; to get around this with simple bindings analysis, but for now I
;;; don't need the headache.
	      ((eq domain 'sched)	
	       '(is-time regular-shape is-width is-bolt 
		       ))
	      ((member domain '(hanoi hanoi-ebl))
	       '(is-disk))
	      ((eq domain 'iso) '(paint-machine polish-machine
						cut-machine leg-machine
						glue-machine lathe-machine
						saw-machine object))
	      ((eq domain 'test) 'nil)

	      ((eq domain 'kites) '(is-stick is-kite is-frame
				    is-bulk is-stick-shape is-stick-type 
				    is-fabric-type is-fabric-surface
				    is-fabric-shape is-frame-shape
				    is-frame-addition
				    func-subtract func-multiple
				    func-get-new-stick-name-or-return-true
				    func-get-new-frame-name
				    func-get-new-fabric-name
				    ))

	      ;;other domains
	      (t nil))))

; List of literals that are never set as top level goals.
; Statics are not included.
; etzi: I removed joined from the list and this seems to be ok.
; Comment remains in case the problem resurfaces.
;  joined is in to supress gi rules.  The problem is that
; otherwise the following rule is formed:
;IF top-level-goal=(JOINED B7 B8 B9) AND
;top-level-goal=(SURFACE-CONDITION B7 POLISHED)
;   NOT (CAN-BE-WELDED B7 B8 B9)
;   NOT ( OR (HAVE-BIT B5) (IS-DRILLABLE B4 B6))
;THEN (prefer goal (JOINED B7 B8 B9) to goal
;                  (SURFACE-CONDITION B7 POLISHED))
; Becuase in this case you have to PUNCH a hole which deletes the
; surface condition.  Unfortunately, joining deletes a prereq for sc
; polished (which I can't detect) so what should really be formed is a
; node rejection rule.  But I can only do that once I detect pre-req
; violations.


;;The format for the non-goals list seems to be the first predicate
;; of the goal followed by a list of unacceptable following predicates
;; which must be present to make something a non-goal.
;;
;; For example, (surface-condition rough) means the for the goal
;;   of (surface-condition ??? ??? ??? ???) if it contains the 
;;   predicate rough as any of the ??? it is a non-goal.
;;
;; Likewise, (is-object nil) means that for the goal of 
;;;  (is-object ?? ?? ?? ??) the goal is a non goal.
;;
;; If the non-goals list contained something like 
;;   (inroom robot1 robot2) then what would be non-goal as a 
;;   result of this would be anything like 
;;   (inroom ??? robot1 ??? robot2 ???)

(defun make-non-goals (domain)
  (setq *non-goals*
    (cond ((eq domain 'processp) nil)
	  ((eq domain 'mbw) nil)

	  ;;original domains
	  ((eq domain 'bw) nil)
	  ((eq domain 'iso) nil)
	  ((eq domain 'strips) '((holding nil))) ;hack
	  ((eq domain 'sched)
	   '((temperature hot) (is-object nil) (clampable nil) (scheduled nil)
	     (surface-condition rough) (last-scheduled nil))) 
	  ((eq domain 'test) nil)

	  ((eq domain 'kites)
	   '((IS-STICK nil)))

;; '((IS-FABRIC <X>) (STICK-PROPERTIES <X> TYPE <Y>)
;; (STICK-PROPERTIES <X> SIZE <Y>))    ;;these are not acceptable
;; formats for non-goals!!!!
	  
	  ;;other domains
	  (t nil))))


;Hack:
; Although there's always some clear block.  The following constraint
; misleads STATIC: (((exists <x>)) ((clear <x>))).  Why? Well, it
; turns out that in some cases when this constraint would seem to
; apply, the value for <x> is actually instantiated and so the
; constraint should NOT apply.  For example,
;NAME:  unstack 
;goal:  (clear b4) 
;precs:  ((on <v30> b4) (clear <v30>) (arm-empty)) 
; Here,  <v30> is actually bound because ON is a unique-attribute and
; b4 is bound.  However, STATIC doesn't know that.  In fact, even if
; ON was not a unique-attribute, its veracity could limit the scope of
; <v30> and hence the constraint shouldn't apply.  To avoid
; complicating the theorm-proving here, I replaced this constraint with
; the more specialized: 	 (((exists <x>)) ((arm-empty) (clear <x>))).

; Variables are implicitly universally quantified!

(defun make-invariant-list (domain)
  (setq *invariant-list*
    (cond ((eq domain 'processp)
	   '(
	     ))
	  ((eq domain 'mbw)
	   '(
	     
	     ;; this doesn't work, since <y> may be constrained to some type
	     ;; (((exists <y>))
	     ;;  ((on <x> <y>) (holding <x>)))
	     (((exists <y>))
	      ((surface <x>) (holding <x>) (clear <x>) (on <y> <x>)))
	     ))

	  ;;original domains
	  ((member domain '(bw mbb min-bad-bw))
	   '(
	     ;; Block x is either:
	     (((exists <y>))
	      ((on-table <x>) (on <x> <y>) (holding <x>)))
	     (((exists <x>))
	      ((arm-empty) (holding <x>)))
	     (((exists <y>))
	      ((holding <x>) (clear <x>) (on <y> <x>)))
	     (((exists <x>)) ((arm-empty) (clear <x>)))
	     ))
	  ((eq domain 'strips)
	   '(
	     ;;predicate negation.
	     (nil ((locked <d>) (unlocked <d>)))
	     (((exists <x>)) ((arm-empty) (holding <x>)))
	     (nil ((dr-closed <d>) (dr-open <d>)))
	     
	     ;;door typing inferences:
	     (nil ((unlocked <d>) (is-door <d>)))
	     (nil ((locked <d>) (is-door <d>)))
	     (nil ((dr-closed <d>) (is-door <d>)))
	     (nil ((dr-open <d>) (is-door <d>)))

	     ;; <d> constrained by is-door, and <r> is unbound
	     ;; or <d> is unbound and <r> is constrained,
	     ;; meaning every room has a door.
	     ;; messes up bindings selection!
	     ;;		 (nil ((dr-to-rm <d> <r>)))
		 
	     ;; every room is connected.
	     (((exists <r>)) ((connects <d> <r1> <r>)))

	     ;;addnl typing inferences:
	     (nil ((inroom <x> <r>) (is-object <x>)))
	     (nil ((inroom <x> <r>) (is-room <r>)))

	     ;;every object is in some room.
	     (((exists <r>)) ((next-to <y> <x>) (inroom <x> <r>)))
	     (((exists <r>)) ((next-to <y> <x>) (inroom <y> <r>)))

	     ))
	  ((eq domain 'sched)
	   '(
	     (((exists <y>))
	      ((shape <x> <y>)))	;everything has a shape.
           
	     (nil			; no exi vars.
	      ((temperature <x> cold) (temperature <x> hot)))

	     (((exists <y>))		; there is a last-scheduled time.
	      ((last-scheduled <x> <y>)))
	     (((exists <y>))
	      ((later <x> <y>)))	;there is always a later time.
	     ))
	  ((eq domain 'paper-bw)
	   '(
	     (((exists <y>))
	      ((on-table <x>) (on <x> <y>) (holding <x>)))
	     (nil			; mod from real bw.
	      ((arm-empty) (holding <x>)))
	     (((exists <y>))
	      ((holding <x>) (clear <x>) (on <y> <x>)))
	     (((exists <x>)) ((arm-empty) (clear <x>)))
	     ))
	  ((eq domain 'iso)
	   '(
	     ((small x) (medium x) (large x))
	     ((cylindrical x) (rectangular x) (shape-less x))
	     ((painted x) (unpainted x))
	     ((leg-less x) (has-legs x))
	     ((clamped x m) (on-table x) (holding x))
	     ((free m) (clamped x m))
	     ((arm-empty) (holding x))
	     ))
	  ((eq domain 'bad-bw)
	   '(
	     ;; Block x is either:
	     (((exists <x>) (exists <a>))
	      ((on-table <x>) (on <x> <y>) (holding <x> <a>)))
	     (((exists <x>))
	      ((arm-empty <a>) (holding <x> <a>)))
	     (((exists <a>) (exists <y>))
	      ((holding <x> <a>) (clear <x>) (on <y> <x>))
	      )))


	  ;;other domains
	  (t nil))))

; can't say (next-to <x> <x>) because (next-to robot <v1>) will match
; that.   This is here to rule out crazy operations.  This stuff is
; actually quite useful in strips.
(defun make-false-list (domain)
  (setq *false-list*
	(cond ((eq domain 'processp) nil)
	      ((eq domain 'mbw) nil)

	      ;;original domains
	      ((eq domain 'strips)
	       '((holding robot)
		 (pushable robot)
		 (is-object robot)
         (is-door robot)
		 (carriable robot)
		 (next-to robot robot)
		 ))

	      ;;other domains
	      (t nil))))
	 

(defun make-rarely-unique-list (domain)
 (setq *rarely-unique-list*
  (cons 'not-equal
  (cond ((eq domain 'mbw)    '(object))
        ((eq domain 'strips) '(is-object 
                               carriable
                               is-door
                               dr-to-rm
                               is-room))
        ((eq domain 'bw)     '(object))
        (t nil))))) ;;other domains

;;template for typing information is a list of:
;;
;; 1.  variable typing information consisting of a list of:
;;
;;      A.  each static predicate in the domain with variables in it.
;;      B.  each other static predicate those variables may occur in.
;;
;; 2.  constant typing information
;;
;;      A.  a constant
;;      B.  a list of the allowed static predicates for the constant.
;;
;; If no typing information is present then variable typing information
;;  will not be used.  (no typing information is considered to be present
;;  when the *typing-info-list* is set to '(nil nil))  If any typing information
;;  is present in *typing-info-list* then STATIC assumes the list of typing
;;  information is complete.  (In other words, don't enter it partially.)
;;
;;  For more information about the typing-info-list see rob-simplfy/var-typing.lisp.


(defun make-typing-info-list (domain)
 (setq *typing-info-list*
 (cond ((eq domain 'mbw)    '((((object <x>) ((object <x>)))) 
                              nil))
       ((eq domain 'sched)
         '((((is-time <time-l>) ((is-time <time>)
                                 (later <other-time> <time>)
                                 (later <time> <other-time>)))
            ((later <time> <other-time>)
                                ((is-time <time>)
                                 (is-time <other-time>)
                                 (later <time> <time2>)
                                 (later <time2> <time>)
                                 (later <other-time> <time2>)
                                 (later <time2> <other-time>)))
            ((is-bolt <bolt>)   ((is-bolt <bolt>)
                                 (is-width <width> <bolt>)))
            ((is-width <width> <bolt>)
                                ((is-width <width> <other-bolt>)
                                 (is-width <other-width> <bolt>)
                                 (have-bit <width>)
                                 (is-punchable <obj-u> <width> <orientation-u>)
                                 (is-bolt <bolt>)))
            ((have-bit <width>) ((have-bit <width>)
                                 (is-width <width> <bolt>)
                                 (is-punchable <obj-u> <width> <orientation-u>)))
            ((is-punchable <obj> <width> <orientation>)
                                ((is-width <width> <bolt>)                ;;width var
                                 (is-punchable <obj-x> <width> <orientation-x>)
                                 (have-bit <width>)
                                 (is-drillable <obj> <orientation-d>)          ;; obj var
                                 (is-punchable <obj> <hole-width-x> <orientation-x>)
                                 (can-be-bolted <obj> <obj2-b> <orientation-b>)
                                 (can-be-bolted <obj1-b> <obj> <orientation-b>)
                                 (can-be-welded <obj> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj> <orientation-w>)
                                 (composite-object <obj> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj>)
                                 (is-drillable <obj-d> <orientation>)   ;;orientation var
                                 (is-punchable <obj-x> <hole-width-x> <orientation>)
                                 (can-be-bolted <obj1-b> <obj2-b> <orientation>)
                                 (can-be-welded <obj1-w> <obj2-w> <orientation>)
                                 (composite-object <new-obj-w> <orientation> <obj1-w> <obj2-w>)))
            ((is-drillable <obj> <orientation>)
                                ((is-drillable <obj-x> <orientation>)          ;;orientation var
                                 (is-punchable <obj-x> <hole-width-u> <orientation>)
                                 (can-be-bolted <obj1-b> <obj2-b> <orientation>)
                                 (can-be-welded <obj1-w> <obj2-w> <orientation>)
                                 (composite-object <new-obj-w> <orientation> <obj1-w> <obj2-w>)
                                 (is-drillable <obj> <orientation-x>)          ;;obj var
                                 (is-punchable <obj> <hole-width-u> <orientation-u>)
                                 (can-be-bolted <obj> <obj2-b> <orientation-b>)
                                 (can-be-bolted <obj1-b> <obj> <orientation-b>)
                                 (can-be-welded <obj> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj> <orientation-w>)
                                 (composite-object <obj> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj>)))
            ((can-be-bolted <obj1> <obj2> <orientation>)
                                ((can-be-bolted <obj1> <obj2x> <orientationx>)
                                 (can-be-bolted <obj2> <obj2x> <orientationx>)
                                 (can-be-bolted <obj1x> <obj1> <orientationx>)
                                 (can-be-bolted <obj1x> <obj2> <orientationx>)
                                 (is-punchable <obj1> <hole-width-u> <orientation-u>)
                                 (is-punchable <obj2> <hole-width-u> <orientation-u>)
                                 (is-drillable <obj1> <orientation-d>)
                                 (is-drillable <obj2> <orientation-d>)
                                 (can-be-welded <obj1> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj2> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj1> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj2> <orientation-w>)
                                 (composite-object <obj1> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <obj2> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj2> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj1>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj2>)
                                 (can-be-bolted <obj1x> <obj2x> <orientation>)
                                 (is-punchable <obj-u> <hole-width-u> <orientation>)
                                 (is-drillable <obj> <orientation>)
                                 (can-be-welded <obj1-w> <obj2-w> <orientation>)
                                 (composite-object <new-obj-w> <orientation> <obj1-w> <obj2-w>)))
            ((can-be-welded <obj1> <obj2> <orientation>)
                                ((can-be-bolted <obj1> <obj2x> <orientationx>)
                                 (can-be-bolted <obj2> <obj2x> <orientationx>)
                                 (can-be-bolted <obj1x> <obj1> <orientationx>)
                                 (can-be-bolted <obj1x> <obj2> <orientationx>)
                                 (is-punchable <obj1> <hole-width-u> <orientation-u>)
                                 (is-punchable <obj2> <hole-width-u> <orientation-u>)
                                 (is-drillable <obj1> <orientation-d>)
                                 (is-drillable <obj2> <orientation-d>)
                                 (can-be-welded <obj1> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj2> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj1> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj2> <orientation-w>)
                                 (composite-object <obj1> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <obj2> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj2> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj1>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj2>)
                                 (can-be-bolted <obj1x> <obj2x> <orientation>)
                                 (is-punchable <obj-u> <hole-width-u> <orientation>)
                                 (is-drillable <obj> <orientation>)
                                 (can-be-welded <obj1-w> <obj2-w> <orientation>)
                                 (composite-object <new-obj-w> <orientation> <obj1-w> <obj2-w>)))
            ((composite-object <obj1> <orientation> <obj2> <obj3>)
                                ((can-be-bolted <obj1> <obj2x> <orientationx>)
                                 (can-be-bolted <obj2> <obj2x> <orientationx>)
                                 (can-be-bolted <obj3> <obj2x> <orientationx>)
                                 (can-be-bolted <obj1x> <obj1> <orientationx>)
                                 (can-be-bolted <obj1x> <obj2> <orientationx>)
                                 (can-be-bolted <obj1x> <obj3> <orientationx>)
                                 (is-punchable <obj1> <hole-width-u> <orientation-u>)
                                 (is-punchable <obj2> <hole-width-u> <orientation-u>)
                                 (is-punchable <obj3> <hole-width-u> <orientation-u>)
                                 (is-drillable <obj1> <orientation-d>)
                                 (is-drillable <obj2> <orientation-d>)
                                 (is-drillable <obj3> <orientation-d>)
                                 (can-be-welded <obj1> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj2> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj3> <obj2-w> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj1> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj2> <orientation-w>)
                                 (can-be-welded <obj1-w> <obj3> <orientation-w>)
                                 (composite-object <obj1> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <obj2> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <obj3> <orientation-w> <obj1-w> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj2> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj3> <obj2-w>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj1>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj2>)
                                 (composite-object <new-obj-w> <orientation-w> <obj1-w> <obj3>)
                                 (can-be-bolted <obj1x> <obj2x> <orientation>)
                                 (is-punchable <obj-u> <hole-width-u> <orientation>)
                                 (is-drillable <obj> <orientation>)
                                 (can-be-welded <obj1-w> <obj2-w> <orientation>)
                                 (composite-object <new-obj-w> <orientation> <obj1-w> <obj2-w>)))
            ((regular-shape <s-s>)
                                ((regular-shape <s-s>)))
            ((sprayable <paint-s>)
                                ((sprayable <paint-s>)
                                 (have-paint-for-immersion <paint-i>)))
            ((have-paint-for-immersion <paint-i>)
                                ((have-paint-for-immersion <paint-i>)
                                 (sprayable <paint-s>)))
            ((has-clamp <machine>)
                                ((has-clamp <machine>))))
	   ((SPRAY-PAINTER      ((has-clamp SPRAY-PAINTER)))
	    (POLISHER           ((has-clamp POLISHER)))
	    (PUNCH              ((has-clamp PUNCH))))))

                         
       ((eq domain 'strips) '((((is-room <ROOM>) ((is-room <ROOM>)
                                                  (connects <DOOR> <ROOM> <ROOM2>) 
                                                  (connects <DOOR> <ROOM2> <ROOM>) 
                                                  (dr-to-rm <DOOR> <ROOM>)))
                               ((dr-to-rm <DOOR> <ROOM>)
                                                 ((dr-to-rm <DOOR> <ROOM>)
	                                          (is-door <DOOR>)
                                                  (connects <DOOR> <ROOMX> <ROOMY>)
                                                  (connects <DOORX> <ROOM> <ROOMY>)
                                                  (connects <DOORX> <ROOMX> <ROOM>)
                                                  (is-key <DOOR> <KEY>)
                                                  (dr-to-rm <DOOR> <ROOMX>)
                                                  (dr-to-rm <DOORX> <ROOM>)
                                                  (is-room <ROOM>)))
                               ((connects <DOOR> <ROOM1> <ROOM2>)
                                                 ((connects <DOOR> <ROOM1> <ROOM2>)
                                                  (dr-to-rm <DOOR> <ROOM>)
                                                  (dr-to-rm <DOORX> <ROOM1>)
                                                  (dr-to-rm <DOORX> <ROOM2>)
                                                  (is-room <ROOM1>)
                                                  (is-room <ROOM2>)
                                                  (is-door <DOOR>)
                                                  (is-key  <DOOR> <KEY>)
                                                  (connects <DOORX> <ROOM1> <ROOMX>)
                                                  (connects <DOORX> <ROOMX> <ROOM2>)))
                               ((pushable <BOX>) 
                                                 ((pushable <BOX>)
                                                  (carriable <BOX>)
                                                  (is-object <BOX>)))
                               ((is-object <OBJ>) 
                                                 ((is-object <OBJ>)
                                                  (carriable <OBJ>)
                                                  (pushable <OBJ>)
                                                  (is-key <DOOR> <OBJ>)))
                               ((is-door <DOOR>)
                                                 ((is-door <DOOR>)
                                                  (connects <DOOR> <ROOM1> <ROOM2>)
                                                  (is-key <DOOR> <KEY>)
                                                  (dr-to-rm <DOOR> <ROOM>)))
                               ((carriable <OBJ>) 
                                                 ((carriable <OBJ>)
                                                  (is-object <OBJ>)
                                                  (is-key <DOOR> <OBJ>)
                                                  (pushable <OBJ>)))
                               ((is-key <DOOR> <KEY>)
                                                 ((is-key <DOOR> <KEY>)
                                                  (is-key <DOOR> <KEY2>) 
                                                  (is-key <DOOR2> <KEY>) 
                                                  (carriable <KEY>)
                                                  (is-object <KEY>)
                                                  (connects <DOOR> <ROOM1> <ROOM2>)
                                                  (is-door <DOOR>)
                                                  (dr-to-rm <DOOR> <ROOM>))))
                              ((ROBOT nil))))
        ((eq domain 'bw)     '((((object <x>) ((object <x>)))) 
                              nil))
        (t                   '(nil nil))))) ;;other domains


;;stores the typing information of constants in the *typing-info-list*

(defun initialize-typing-info-constants ()
 (unless (OR (not *use-typing-info-list*)
             (equal '(nil nil) *typing-info-list*))
  (iter:iterate
  (iter:for constant iter:in (second *typing-info-list*))
   (putprop (first constant) (second constant) 'type))))

; on,on neq is necessary to rule out an incorrect on < on rule.
; semantics of: ((p x) (q x y))
; forall x,y st not-equal(x,y) p(x) --> not(q(x,y)), and by
; contraposition: q(x,y) --> not(p(x)).
(defun make-negation-list (&optional (domain 'bw))
  (setq *negation-list*
	(cond ((eq domain 'mbw)
	       '(
		 ((on <x> <y>) (on <y> <x>))
		 ((on <x> <y>) (on <x> <z>)) ;(neq <y> <z>)
		 ((on <z> <y>) (on <x> <y>))

         ((on <x> <y>) (holding <x>))
         ((on <x> <y>) (holding <y>))
         ((on <x> <y>) (clear <y>))
         ((on <x> <y>) (on-table <x>))
         ((on-table <x>) (holding <x>))
         ((clear <x>) (holding <x>))
		 ((holding <x>) (holding <y>));;  this was commented out in the original ini.lisp
         ((arm-empty) (holding <x>))


		 ))
	      ((eq domain 'processp) nil)

	      ;;original domains
	      ((member domain '(bw mbb min-bad-bw))
	       '(
		 ((on <x> <y>) (on <y> <x>))
		 ((on <x> <y>) (on <x> <z>)) ;(neq <y> <z>)
		 ((on <z> <y>) (on <x> <y>))

		 ((holding <x>) (arm-empty))
		 ((holding <x>) (holding <y>))
		 ((holding <x>) (on-table <x>))
		 ((holding <x>) (on <x> <y>))
		 ((holding <x>) (on <y> <x>))
		 ((holding <x>) (clear <x>))
		 
		 ((on-table <x>) (on <x> <y>))
		 
		 ((clear <x>) (on <y> <x>))
		 ))
	      ((eq domain 'strips)
	       '(
		 ((holding <x>) (arm-empty))
 		 ((inroom <x> <r1>) (inroom <x> <r2>)) ;want r1<>r2!
		 ((locked <x>) (unlocked <x>))
		 ((locked <x>) (dr-open <x>))
		 ((dr-open <x>) (dr-closed <x>))
         ((~ (dr-open <x>)) (~ (dr-closed <x>))) ;due to binary
												 ;relationship between
												 ;dr-open and
												 ;dr-closed
         ((~ (unlocked <x>)) (~ (locked <x>)))   ;due to binary
												 ;relationship between
                                                 ;locked and unlocked
		 ((holding <x>) (next-to <x> <y>))
		 ((holding <x>) (next-to <y> <x>))
         ((is-door <X>) (is-object <X>))
         ((is-door <X>) (carriable <X>))
         ((is-door <X>) (holding   <X>))
         ((is-door <X>) (pushable  <X>))

		 ))
	      ((eq domain 'sched)
	       '(
		 ((painted <x> <p1>) (painted <x> <p2>))
		 ((shape <x> <s1>) (shape <x> <s2>)) ;not nec?
		 ((surface-condition <x> <s1>)
		  (surface-condition <x> <s2>))
		 ((temperature <x> hot) (temperature <x> cold))
		 ))
	      ((eq domain 'hanoi)
	       '(((on <d> <p>) (noton <d> <p>)))
	       )
	      ((eq domain 'iso)
	       '((holding ((holding 5)
			   (on-table 1)
			   (arm-empty 0)
			   (clamped 1)))
		 (arm-empty ((holding 0)))
		 (clamped ((holding 1)
			   (on-table 1)
			   (free 4)))
		 (on-table ((holding 1)
			    (clamped 1)))
		 (has-legs ((leg-less 1)))
		 (leg-less ((has-legs 1)))
		 (painted ((unpainted 1)))
		 (unpainted ((painted 1)))
		 (small ((large 1) (medium 1)))
 		 (medium ((large 1) (small 1)))
		 (large ((small 1) (medium 1)))
		 (shape-less ((rectangular 1) (cylindrical 1)))
		 (cylindrical ((rectangular 1) (shape-less 1)))
		 (rectangular ((shape-less 1) (cylindrical 1)))
	       ))
	      (t nil))))


; Generates a list that contains all the wild-card variables for
; operators in this domain.  A wild-card variable is a variable
; that appears on the rhs of an operator but not on its lhs.
; Consequently, the variable is unconstrained.  This isn't entirely
; right, because the variable can be bound when the operator
; unifies with the goal, but I assume this doesn't happen.
; NOTES: CURRENTLY NOT CALLED SINCE ASSUMES NO KEYWORD COMMANDS
; NEEDS TO BE IMPLEMENTED.
; relies on the fact that the ops have unique vars.
(defun make-wild-cards ()
  (setq *wild-cards*
  (remove-duplicates
  (iter:iterate
   (iter:for op iter:in (mapcar #'car *achievers*))
   (appending
    (set-difference (rhs-vars op)
		   (lhs-vars op)))))))



; Used for predicates which have very large subtrees in static, but which
; when problem solving the predicates do not generate large subtrees.
(defun make-dont-expand-list (&optional (domain 'bw))
  (setq *dont-expand-list* 
	(cond ((eq domain 'kites)
	       '(
                 (inter-stick-properties <prop> <stick1> at <point1> <stick2> at <point2>)
                ))
;; it would be nice not to subgoal on (frame-properties <frame> shape <shape>)
;; but its not too expensive. 

              ((eq domain 'strips)
               '(
                 (inroom robot <r>)      ;;the robot can reach any room.
                 (next-to robot <x>)     ;;the robot can reach any object
                ))
	     

	      (T nil))))



(defun rhs-vars (op)
  (extract-vars (find-op-effects op)))

(defun lhs-vars (op)
  (extract-vars (get op 'preconds)))

(defun extract-vars (l)
  (remove-duplicates
  (iter:iterate
   (iter:for c iter:in l)
   (if (eq (car c) '~)
       (appending (vars-only (cdr (cadr c))))
     (appending (vars-only (cdr c)))))))


(defun vars-only (l)
  (iter:iterate
   (iter:for x iter:in l)
   (when (rob-is-var? x) (iter:collect x))))


; Stack is reversed to get the variable constraints right!

;; changed to include negation invariant




(defun holds? (p stack)
  (setq stack (reverse stack))
  (let ((res (or
	      (holds-with-goal-stack? p stack)
	      (iter:iterate
	       (iter:for inv iter:in *invariant-list*)
	       (iter:thereis
		(invariant-applies inv stack p))))))   
    res))

(defun holds-with-goal-stack? (p	;literal
			       stack)	;goal stack

  (let* ((negated-literal? (eq '~ (first p)))
	 (literal          (if negated-literal?
			       (second p)
			     p))
	 (predicate        (first literal))
	 (params           (rest literal)))
    
    (cond
     ;; CASE 1: NEGATED P
     (negated-literal?
      (find-if #'(lambda (goal)
		   (and (eq predicate (first goal))
			(iter:iterate
			 (iter:for param iter:in params)
			 (iter:for goal-param iter:in (rest goal))
			 (iter:always (holds-with-params? param goal-param))))
		   )
	       stack))
     
     ;; CASE 2: NON-NEGATED P
     ;; bound parameters match bound and variable parameters
     ;; variable parameters do not match
     (T
      (find-if #'(lambda (goal)
		   (and (eq (first goal) '~)
			(eq (first (second goal)) predicate)
			(iter:iterate
			 (iter:for param iter:in params)
			 (iter:for goal-param iter:in (rest (second goal)))
			 (iter:always (holds-with-params? param goal-param))))
		   )
	       stack))
     )))

(defun holds-with-params? (param goal-param)
  (cond ((random-const? goal-param)
	 (eq goal-param param)
	 )
	((my-var? goal-param)
	 T
	 )
	((wild? goal-param)
	 T)
))

;;Negates the predicates on the stack to find a list of things with are known to be true.
;;Finds all the variables in the invariant axiom
;;Finds those variables which should not be bound to a constant in what p binds to.
;;Finds the a list of the predicates in the invariant-axiom.
;;Tries binding p to each one of the predicates in the invariant-axioms.
;;Puts the result of each of those attempted bindings into blists.
;;Calls sub-invariant-applies to recursively analyze each blist in blists individually.

(defun invariant-applies (inv stack p)
 (let* (
   (stack       (mapcar #'negate-exp stack))
   (inv-vars    (find-variables inv))
   (exists-vars (find-variables (first inv)))
   (inv-preds   (second inv))
   (blists      (mapcar #'(lambda (pred) (rob-match pred p)) inv-preds)))
  (sub-invariant-applies inv-preds blists p stack exists-vars inv-vars)))

;;Look at the first blists.
;;If there was no possible binding or some of the exists variables in the axiom were bound 
;;     to constants then try the next blist by calling itself recursively without the first blist.
;;

(defun sub-invariant-applies (inv-preds blists p stack exists-vars inv-vars)
 (unless (null blists)     ;;termination condition
  (let* (
    (blist (first blists)))
   (if (or (eq blist 'no-match) 
           (find-if #'(lambda (bpair)
                       (and (is-constant (second bpair))
                            (member (first bpair) exists-vars)))
            blist))  ;;does exist-var bind to a constant in conclusion of result?
                     ;;in other words, does a constant in p bind with an exists-var?
                     ;;if so, consider it the same as a no-match.
  
    (sub-invariant-applies inv-preds (rest blists) p stack exists-vars inv-vars)
    (let* (
      (new-inv-preds (instantiate inv-preds blist))
      (new-inv-preds (remove-if #'(lambda (x) (equal x p)) new-inv-preds))
      (new-exists-vars (intersection exists-vars (find-variables blist)))
      (new-exists-vars (instantiate new-exists-vars blist)) 
      (result-of-this-blist (sub-sub-invariant-applies new-inv-preds stack new-exists-vars inv-vars)))
     (if result-of-this-blist
      t
      (sub-invariant-applies inv-preds (rest blists) p stack exists-vars inv-vars)))))))



;;returns true if things in stack negate everything in inv-preds
;; without binding exists-vars which were present in the predicate p matched with
(defun sub-sub-invariant-applies (inv-preds stack exists-vars inv-vars)
 (if (null inv-preds)
  t
  (unless (null stack)
   (let* (
     (pstack (first stack))
     (preds-blists (mapcan #'(lambda (inv-pred)
                       (mapcar #'(lambda (x) (list inv-pred x))
                       (rob-negate-return-blists-w-axioms
                        inv-pred pstack))) inv-preds)))
    (if (sub-sub-sub-invariant-applies inv-preds (rest stack) 
           exists-vars preds-blists pstack inv-vars)
     t
     (sub-sub-invariant-applies inv-preds (rest stack) exists-vars inv-vars))))))

(defun sub-sub-sub-invariant-applies (inv-preds stack exists-vars 
                            preds-blists pstack inv-vars)
 (unless (null preds-blists)
  (let* (
    (blist (second (first preds-blists)))
    (pred  (first  (first preds-blists))))
   (or 
    (unless (find-if #'(lambda (bpair)
                        (or
                         (not (member (first bpair) inv-vars))
                         (and (is-constant (second bpair))
                              (member (first bpair) exists-vars))))
              blist)
      (let* (
        (inv-preds (remove-if #'(lambda (x) (equal x pred)) inv-preds))
        (inv-preds (instantiate inv-preds blist))
        (inv-preds (remove-if #'(lambda (x) (rob-negate-exactly-w-axioms
                                             x pstack)) inv-preds)))
       (sub-sub-invariant-applies inv-preds stack exists-vars inv-vars)))
    (sub-sub-sub-invariant-applies inv-preds stack exists-vars (rest preds-blists) 
                                   pstack inv-vars)))))


                                      

 
     


    
    






;;; invariant=(((exists <x>)..) (list of literals)).
;;; p=(clear <x>)
;;; lbl=list of bindings lists.
;;; ALGORITHM:
;;; 1. check if p matches the inv.
;;; 2. If so, try to match the rest of the inv against the stack,
;;; checking that the exi are not bound in the match.
;;
;;
;;; inv is first, because the constraint has to match against the stack
;;; not the other way around.
;;
;;(defun invariant-applies (inv stack p)
;;  (let* ((exi-vars (mapcar #'cadr (car inv)))
;;	 (rule (cadr inv))
;;	 (lbl (match p rule)))
;;    (iter:iterate
;;     (iter:for bl iter:in lbl)
;;     (iter:thereis (unless (exi-bound? bl exi-vars) 
;;		      (match-body (find-body rule bl p)
;;				  stack exi-vars))))))
;;
;;
;;; subs bl into p and removes it from rule.
;;; caar to strips bindings and list from p.
;;; car to strip bindings.
;;; Could this be optimized by doing the match directly (with an lbl
;;; argument) with the consing?
;;(defun find-body (rule bl p)
;;  (setq p (caar (bind-clauses (list p) bl nil)))
;;  (setq rule (car (bind-clauses rule bl nil)))
;;  (rem-equal p rule))
;;
;;  
;;(defun match-body (body stack exi-vars)
;;  (let ((lbl (match body stack)))
;;    (iter:iterate
;;     (iter:for bl iter:in lbl)
;;     (iter:thereis (not (exi-bound? bl exi-vars))))))
;;    
;;
;;; Is an existential variable bound to a constant in one of the
;;; bindings in this bindings list?
;;; hack? Key bias here: this assumes that variables in the goal are
;;; (potentially) bound, whereas randomly generated vars are not.
;;; Consequently, the check below for goal-arg?
;;(defun exi-bound? (bl exi-vars)
;;  (iter:iterate
;;   (iter:for b iter:in (butlast bl))		;last(bl)= (nil nil).
;;   (iter:thereis
;;    (iter:iterate
;;     (iter:for v iter:in exi-vars)
;;     (when (and (equal v (car b)) (goal-arg? (cadr b)))
;;	   (return t))))))
;;
;;
;;
;;; arguments in goal are consts!
(defun goal-arg? (x)
  (and x
       (not (eq (schar (symbol-name x) 0) #\<))))
