;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Common Lisp NONLIN - University of Maryland at College Park 
;;;;
;;;; Version 1.2, 11/91
;;;; (email nonlin-users-request@cs.umd.edu for more info)
;;;; 
;;;; LINK.LISP:  Routines to handle interactions 
;;;;
;;;; History:


;;; added DEVISOR mods - 4/91 bpk

;;; The two routines below are called by add-effects in expand.lisp and
;;; try-to-establish in establish.lisp to remove interactions. They 
;;; construct a set of interacting ranges and call nonlin-link 
;;; to solve the interactions between the ranges

(defun resolve-interaction (&key conflict-nodes offender offender-effect 
				 (extra-links '(nil)))
   (let*  ((conflict-ranges (for (inode :in conflict-nodes)
				 :save (list inode (purpose-nodes inode 
				 (negate-pat offender-effect)))))
	   (node offender)
	   (node-purposes (purpose-nodes offender offender-effect))
	   (linearizations (nonlin-link  (list (append (list (list node node-purposes)) conflict-ranges)) offender-effect extra-links)))
	  (return-from resolve-interaction linearizations)
;;;semantics: NODE IS GIVING THE OFFENDER-EFFECT. INODE IS GIVING (NOT(OFFENDER-EFFECT))
 ))


(defun resolve-establish-conflict (&key offender-range conflict-nodes offender-cond (extra-links '(nil)))
    ;;offender-range will have (chpvnode--atnode) and conflict nodes will be
    ;;all the pvbar nodes. chpvnode is the node which establishes the condition
    ;;at atnode
   (let*  ((conflict-ranges (for (inode :in conflict-nodes)
				 :save (list inode (purpose-nodes inode 
				 (negate-pat offender-cond)))))
	   (node (first offender-range))
	   (node-purposes (list (second offender-range)))
	   (linearizations (nonlin-link  (list (append (list (list node node-purposes)) conflict-ranges)) offender-cond extra-links)))
	  (return-from resolve-establish-conflict linearizations)
;;;semantics: NODE IS GIVING THE OFFENDER-EFFECT. INODE IS GIVING (NOT(OFFENDER-EFFECT))
 ))
    
; suggest contains the suggested extra-linkss
; (i.e., suggested links) to remove the interactions
; interact-list is a list of interact records of the form
; ((node (purposes of node))(inode1 (purposes of inode1))
;		    	    (inode2 (purposes of inode2)))
; where node's effect is interacting with inode1, inode2 etc
; each of which have the purpose nodes given in the next list
   
; this procedure is supposed to loop through the interact-list,
; removing interactions one by one by suggesting new extra-linkss 
; (ie) new links.  After all the interactions are removed, it will
; send the suggested extra-links back 
    
(defun NONLIN-LINK (interact-list offender-effect extra-links)
  (let ((newsuggest extra-links)
	suggest)
    (for (interact-record :in interact-list)
	 :do
	 (let* ((node-particulars (pop interact-record))
		(node (car node-particulars))
		(node-purposes (cadr node-particulars)))
	       ; interact-record now contains the particulars
	       ; for the interacting nodes
	       (for (interaction :in interact-record)
		    :do
		   (let* ((inode (car interaction))
			  (inode-purposes (cadr interaction))
			  interaction-ranges)
			 (setf interaction-ranges
			       (construct-interaction-ranges 
				   node node-purposes inode inode-purposes))
			 ; this will return a list of interacting
			 ; ranges 
			 
			 (for (ranges :in interaction-ranges)
			      :do
			      (let ((vrange (car ranges))
				    (vbarange (cadr ranges)))
		                   (setf suggest newsuggest)
			           (setf newsuggest nil)
				   (for (net :in suggest)
					:do
					(let ((linearizations (RESOLVE-LINK 
				 	   vrange vbarange offender-effect net)))
					     (if (null linearizations)
					      (then
						;there was no interaction
						(setf newsuggest (append1 newsuggest net)))
					      (else
						(if (and (no-valid-linearizations linearizations) (cadr vbarange)
							 (not (eql (car vbarange) (cadr vbarange))))
						  ;this is where we call try-to-modify-binding
						  ;to check if the condition type is use-only-for-query
						  ;and if yes then we try to get a new binding.  
						  (then
					             (if (try-to-modify-binding  (cadr vbarange) (car vbarange) (negate-pat offender-effect))
							(setf newsuggest (append1 newsuggest net))))
						;else   
					        (for (linearization :in linearizations)
						     :do
						     (if (not (null linearization))
                                                       (setf newsuggest (append1 newsuggest  (cons linearization net ))))
					        ))))
					)) ;end of for-net
			      )) 	; ranges
			 (if (null newsuggest)
			    (return-from nonlin-link (fail)))
			;interaction can not be removed
		    ))	; interaction
	 ))	; interaction record
    ; at this point we have removed all
    ; interactions in newsuggest. Any of the extra-links
    ;  in newsuggest would be a good selection.
    (return-from nonlin-link newsuggest)
))
			 

;RESOLVE-LINK takes one interaction range and returns all possible ways to remove
;that interaction. vrange is of the type (node node-purpose) and vbarange is of the
;type (inode inode-purpose). One establishes offebder-effect and other establishes
;not(offender-effect)
 
(defun RESOLVE-LINK (vrange vbarange offender-effect extra-links)
    ;extra-links is basically an incremental change in the network.
    ;current network+extra-links is the actual modified network. extra-links is 
    ;a list of suggest-entrys
    (let ((node (car vrange))(node-purpose (cadr vrange))
	  (inode (car vbarange)) (inode-purpose (cadr vbarange)))
	 ; HOW ABOUT MAKING SURE THAT THE RANGES WILL HAVE BEGINNING AND
	 ; END EVEN IF THEY ARE SINGLE NODES OR THE RANGE IS DELETED IN THE
         ; SUGGESTED NETWORK (BY REMOVE CONTRIBUTOR SUGGESTION)?
	 ; (IF THEY ARE SINGLE NODES, THEN, BOTH BEGINNING AND END WILL
	 ; BE MADE SAME NODES)
	 ; now a small trick
	 (if (or (null node-purpose) (range-deleted-in-suggested-net vrange offender-effect extra-links))
	     (setf node-purpose node))
	 (if (or (null inode-purpose) (range-deleted-in-suggested-net vbarange 
				       (negate-pat offender-effect) extra-links))
	     (setf inode-purpose inode))
        ;if the interaction has already  been removed in extar-links then 
        ;just return
	(if (not (vrange-intersects-vbarange-p vrange vbarange extra-links))
	   (return-from resolve-link nil))
	; if the ranges do not intersect then no interaction exists.
        (if (and (eql inode inode-purpose)
		 (not (eql node node-purpose)))
		 ;;for cases like (7 7) :offender-range (9 6)
		 (psetq node inode node-purpose inode-purpose
		       inode node inode-purpose node-purpose)
	)
	    
	 ;send all possible alternative linearizations in a list of suggest-
	 ;entrys
	 (list
	   ; 1.a put vbarange before vrange
	   (if (put-range1-before-range2  `(,inode ,inode-purpose)
		   `(,node ,node-purpose) extra-links)
		(make-suggest-entry :link (list inode-purpose node)))
	   ;1.b put vrange before vbarange
	   (if (put-range1-before-range2  `(,node ,node-purpose)
		   `(,inode ,inode-purpose) extra-links)
		(make-suggest-entry :link (list node-purpose inode)))
	   ;3.a remove node as a contributor to node-purpose and put
	       ; vrange before vbarange
	   (if (and
		     (check-contributor-removal node-purpose node  
						offender-effect	)
		     (put-range1-before-range2 `(,inode ,inode-purpose)
			     `(,node ,node) extra-links)
		)
	     (if (eql inode inode-purpose)
	       (then
		 (make-suggest-entry :cond offender-effect
	                             :contributor (list (list node node-purpose))))
               (else
	         (make-suggest-entry :link (list inode-purpose node)
				     :cond offender-effect
	                             :contributor (list (list node node-purpose)))))
	   )
	   ;3.b remove node as a contributor to node-purpose and put
	       ; vbarange before vrange
	   (if (and
		     (check-contributor-removal node-purpose node  
						offender-effect	)
		     (put-range1-before-range2 `(,node ,node)
			     `(,inode ,inode-purpose) extra-links)
		)
	     (if (eql inode inode-purpose)
	       (then
		 (make-suggest-entry :cond offender-effect
	                             :contributor (list (list node node-purpose)))) 
	       (else
	         (make-suggest-entry :link (list node inode)
				     :cond offender-effect
	                             :contributor (list (list node node-purpose)))))
	   )

	   ;2.a remove "inode" from the contributor list of inode-purpose
	       ;and put vrange before vbarange
	       
	   (if (and
		     (check-contributor-removal inode-purpose inode 
			 (negate-pat offender-effect))
		     (put-range1-before-range2 `(,inode ,inode)
			     `(,node ,node-purpose) extra-links)
		)
	     (if (eql node node-purpose)
	       (then
		 (make-suggest-entry :cond (negate-pat offender-effect)  
				     :contributor (list (list inode inode-purpose))))
	       (else
	         (make-suggest-entry :link (list inode node)
				     :cond (negate-pat offender-effect)
	                             :contributor (list (list inode inode-purpose)))))
	   )
	   ;2.a remove "inode" from the contributor list of inode-purpose
	       ;and put vrange after vbarange
	   (if (and
		     (check-contributor-removal inode-purpose inode 
			 (negate-pat offender-effect))
		     (put-range1-before-range2 `(,node ,node-purpose)
			     `(,inode ,inode) extra-links)
	       )
	     (if (eql node node-purpose)
	       (then
		 (make-suggest-entry :cond (negate-pat offender-effect)
	                             :contributor (list (list inode inode-purpose))))
	       (else
	         (make-suggest-entry :link (list node-purpose inode)
	                             :contributor (list (list inode inode-purpose))
				     :cond (negate-pat offender-effect))))
	   )
	       
	   ;4. remove both inode and node as contributors to inode-purpose
	       ; and node-purpose -then no  extra-links needed
	   (if (and
		     (check-contributor-removal inode-purpose inode
			 (negate-pat offender-effect))
		     (check-contributor-removal node-purpose node
			 offender-effect)
	       ) 
	     (make-suggest-entry :cond offender-effect
                                 :contributor (list (list node node-purpose)
						    (list inode inode-purpose)))
	   )
	 )))
    
                                      				           
;; This function tries to rebind a use-only-for-query condition when the
;; interaction can not be removed in any other way (i.e. RESOLVE-LINK 
;; returns fail to NONLIN-LINK)

(defun try-to-modify-binding (conflict-atnode conflict-node conflict-cond)
    ;retrieve current condition and binding into old-conflict-cond
    ;and old-binding
    (let* ((old-conflict-cond (find conflict-cond
				    (node-expanconds conflict-atnode)
				    :key #'scondition-pattern
				    :test #'equal))
	   (old-binding (get-scondition-binding old-conflict-cond))
	  )
	 (if (eql (scondition-type old-conflict-cond) :use-only-for-query)
	   (then
	     (mark conflict-atnode)
	     (let ((new-conflict-cond (my-copy-scondition old-conflict-cond)))
	     (setf (scondition-pattern new-conflict-cond)
		   (remove-binding (scondition-pattern new-conflict-cond) 
		   (mapcar #'reverse old-binding)))
	     ;query the current network to get a new binding
	     (multiple-value-bind (already-true result new-binding)
		(q&a (scondition-pattern new-conflict-cond) conflict-atnode)
	      (if already-true
	        ;replace the old condition and binding by new condition and
		;binding
		(then
		  (destructive-replace-variables 
		    (scondition-pattern new-conflict-cond) new-binding)

		  (setf (scondition-contributors new-conflict-cond)
			result)
                  (push (add-context new-binding)
			(scondition-binding new-conflict-cond))
		  (dremove old-conflict-cond
	    		   (node-expanconds conflict-atnode))
		  (pushnew new-conflict-cond
		   	   (node-expanconds conflict-atnode))
		  (remove-use-only-for-query-cond conflict-atnode
		    conflict-node (scondition-pattern old-conflict-cond))
		  ;modify gost
		  (enter-gost (scondition-pattern new-conflict-cond)
		      	      :use-only-for-query
			      conflict-atnode result)
		  ;modify effects of the node  the new binding
		  (for (seffect :in (node-ctxt conflict-atnode))
		       :do
		       (let((seffect-old-pat (seffect-pattern seffect))
			    (seffect-new-pat))
			  (setf seffect-new-pat
				(remove-binding seffect-old-pat
				     (mapcar #'reverse old-binding)))
			  (destructive-replace-variables
			     seffect-new-pat new-binding)
			  (delete-tome seffect-old-pat (seffect-type seffect)
							conflict-atnode)
		          (enter-tome seffect-new-pat (seffect-type seffect)
						       conflict-atnode)
			  (setf (seffect-pattern seffect) seffect-new-pat)))
		  (return-from try-to-modify-binding t)
 	    )))))))  
)

;; checks if range1 can be put before range2. Succeeds only if no link exists
;; in the reverse direction
(defun put-range1-before-range2 (range1 range2 extra-links)
    (let (;(range1-beg (car range1))
	  ( range1-end (cadr range1))
	  (range2-beg (car range2))
	  ;( range2-end (cadr range2))
	  )
	  (not (predecessor-p range2-beg range1-end extra-links))
    ))
    
     
;; tests whether "isprenode" is a predecessor of node "node"

(defun predecessor-p (isprenode node extra-links)
    ; checks if isprenode is a predecessor of node
    ;in the suggested (network+extra-links) network
    ; for now, we can try brute-force scan 
    ; back-wards
    (if (eql isprenode node) (return-from predecessor-p t))
    (let ((prenodes (append (get-prenodes node)
                    (get-prenodes-from-extra-links node extra-links))))
	 (if (member isprenode prenodes)
	     
	     ; if isprenode is a member of prenodes of
	     ; node 
	     t
	     ;else	
	     (if (there-exists (newnode :in prenodes)
			 (predecessor-p isprenode newnode extra-links))
                   t
		   )))
)

;; retrieves the predecessors of "node" in extra-links

(defun get-prenodes-from-extra-links (node extra-links)
  (for (suggest-entry :in extra-links)
       :when (eql (cadr (suggest-entry-link suggest-entry)) node)
       :save
        (car (suggest-entry-link suggest-entry))
  )
)

	   
;; Given interact record of the type ((node purpse-nodes)(inode purpose-inodes))
;; this contructs ranges of the type (((node purpose1)(inode inode-purpose1))
;; ((node purpose1)(inode inode-purpose2))...((node purosei)(inode inode-purposej)))

(defun construct-interaction-ranges (node node-purposes inode inode-purposes)
    (let (interaction-ranges vranges vbaranges)
	 (setf vranges (if node-purposes
			   ; if there are some purpose nodes,
			   ; then make ranges of form (node purpose1)
			   (for (purpose :in node-purposes)
				:save
				(list node purpose))
			   ; if there are no ranges, make an artificial
			   ; range of form (node node)
			   (list (list node node))))
	 (setf vbaranges (if inode-purposes
			     (for (ipurpose :in inode-purposes)
				  :save
				  (list inode ipurpose))
			     (list (list inode inode))))
	 (for (vrange :in vranges)
	      :do
	      (for (vbarange :in vbaranges)
		   :do
		   (push (list vrange vbarange)
			 interaction-ranges)))
  (return-from construct-interaction-ranges interaction-ranges)
  )
)


;; checks if an contributor can be removed. A contributor can be removed if
;; either there are more than one contributor or the node is of type :phantom

(defun check-contributor-removal (node contributor cond)
  (if (eql node contributor) (return-from check-contributor-removal nil))
  ;;;this is so that single node ranges (like (6 6)) will be taken care of
  ;;;efficiently
  (let* ((gost-entry (lookup-gost (convert-pat cond)))
	 ;;needed so that gentry can be removed if required
	 gentry)
    (if (null gost-entry) (return-from check-contributor-removal nil))
    (setf gentry (find node (gost-entrys cond) :key #'get-gentry-node))
    (if (null gentry) (return-from check-contributor-removal nil))
    (or (> (length (get-gentry-contributors gentry)) 1)
	(eq (node-type node) :phantom)
    )
  )
)

;; Given a list of suggest-entries, this function adds the links and removes
;; the contributors in the suggest-entries to linearize

(defun linearize (linearization)
  (for (suggest-entry :in linearization)
       :do
       (let ((link (suggest-entry-link suggest-entry))
	     (cond (suggest-entry-cond suggest-entry))
	     (contributor1 (car (suggest-entry-contributor suggest-entry)))
	     (contributor2 (cadr (suggest-entry-contributor suggest-entry))))
           (make-prenode (car link) (cadr link))
           (make-succnode (cadr link) (car link))
           (if contributor1 (try-to-remove-contributor (cadr contributor1) (car contributor1) cond))
           (if contributor2 (try-to-remove-contributor (cadr contributor2) (car contributor2) (negate-pat cond)))

           ;; DEVISOR mods:
           ;; when ordering 2 nodes, call modify-start-times to adjust
           ;;   the nodes' windows if necessary
           (when *devisor-mods* 
              (if (not
                 (modify-start-times (allnodes (car link)) 
                    (allnodes (cadr link))))
                 (backtrack)) ; backtrack if unsuccessful
	   )

       )
  )
)

;;; checks if vrange interacts with vbarange. Two ranges interact if and only if
;;; they have some part in parallel

(defun vrange-intersects-vbarange-p (vrange vbarange extra-links)
  (let((node (car vrange)) (node-purpose (cadr vrange))
       (inode (car vbarange))
       (inode-purpose (cadr vbarange)))
      (not (or (and (eql node node-purpose)
	            (eql inode inode-purpose))
	       (predecessor-p node-purpose inode extra-links)
	       (predecessor-p inode-purpose node extra-links)))
  )
)

;this functions checks if the range "range" has been removed in the suggested
;network "current-net+extra-links"
(defun range-deleted-in-suggested-net (range cond extra-links)
  (if (eql (car range) (cadr range)) (return-from range-deleted-in-suggested-net nil))
  ;if the range is singleton then it can not be deleted
  (do  ((suggest-entry (car extra-links) (car extra-links)))
       ((null suggest-entry) nil)
       (let ((contributor-cond (suggest-entry-cond suggest-entry))
             (contributor1 (car (suggest-entry-contributor suggest-entry)))
             (contributor2 (cadr (suggest-entry-contributor suggest-entry))))
	    (if (or (and (eql range contributor1)
			(eql cond contributor-cond))
	            (and (eql range contributor2)
			(eql cond (negate-pat contributor-cond))))
	       (return t))
            (setq extra-links (cdr extra-links))))
)
(defun no-valid-linearizations (linearizations)
   (do ((linearization (car linearizations) (car linearizations)))
       ((null linearizations) t)
       (if linearization
	   (return nil)
	   ;else
	   (setq linearizations (cdr linearizations)))
   ) 
)
(defun get-scondition-binding (cond)
 (cdar (scondition-binding cond)))
