;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Common Lisp NONLIN - University of Maryland at College Park 
;;;;
;;;; Version 1.2, 11/91
;;;; (email nonlin-users-request@cs.umd.edu for more info)
;;;; 
;;;; GOST.LISP: Routines for manipulating GOST (Goal Structure Table)  
;;;;
;;;; History:

(defstruct (gost-entry (:print-function print-gost-entry))
    condition  ;; a ground pattern
    pluses     ;; pluses contain plus polarity gentrys
    minuses    ;; minuses contain negated gentrys
)
				
(defstruct (gentry  (:print-function print-gentry))
    node ;;the node at which condition holds
    type ;;the type of entry (one of :phantom, :use-when, :precond)
    contributors ;;a list of (context nodes) lists such that nodes are the 
                 ;; contributors to that condition at node in contex "context"
    )

(defun print-gost-entry (gost-entry stream depth)
  (declare (ignore depth))
  (format stream "Cond:~s  +[~s ]  -[~s ]~%" (gost-entry-condition gost-entry)
	  (gost-entry-pluses gost-entry) (gost-entry-minuses gost-entry))
  )

(defun print-gentry (gentry stream depth)
  (declare (ignore depth))
  (format stream "<< ~s :at ~s :from ~s>>" (gentry-type gentry) (get-gentry-node gentry)
	  (get-gentry-contributors gentry)))


(defvar *gost* (make-hash-table)
 "the global table of gost-entrys hashed under predicates")

(defun print-gost (&optional(stream t) (depth t))
  (declare (ignore depth))
  (format stream "~%Gost Table Entrees:~%")
	       (for (gost-entry :in (all-gost-entrys))
                 :do
	       (format stream "~s" gost-entry))
nil
    )

    
(defun all-gost-entrys ()
    (let (glist)
	 (maphash #'(lambda (key gost-entry-list)
			    (declare (ignore key))
			    (setf glist (append glist gost-entry-list))
		    ) *gost*)
	 glist))
    
(defun reset-gost ()
    (setf *gost* (make-hash-table)))
    
;; gost-entrys are hashed on their pattern.

(defmacro relevant-gost-entrys (pattern)
    `(gethash (predicate ,pattern) *gost*))

;; adds entry to *gost*

(defun  enter-gost-table (entry)
    (push entry (gethash
		    (predicate (gost-entry-condition entry)) 
		      *gost*)

     ))

;;; Note :GOST never contains any partially instantiated patterns
;;; returns a gost entry for the condition pattern    

(defun lookup-gost (condition)
    (let (result (datalist (relevant-gost-entrys condition)))    
	 (setf result (find condition datalist
       		            :key 'gost-entry-condition :test #'equal))
    )
)

;;; if there is no entry for the  condition,
;;; it makes one and indexes in the gost

(defun index-gost (condition)
    (let ((entry (lookup-gost condition)))
      (if (null entry)
	  (then
	    (setf entry (make-gost-entry :condition condition))
	    (enter-gost-table entry)))
     entry
))

;;; We will have the condition sent as (pat) or (not (pat))
;;; the polarity is found inside the function        
;;; called by: expand-goal-node, expand-phantom-node, 
;;; expand-node-using-schema-instance

(defun ENTER-GOST (cond   entry-type  node 
		   contributors)
  (multiple-value-bind (condition sign) 
      (convert-pat cond)
    (let* ((entry (index-gost condition)) 
	   (gentrys (case sign
		      (:plus
		       (gost-entry-pluses entry))
		      (:minus
		       (gost-entry-minuses entry))))
	   (gentry (find node gentrys :key #'get-gentry-node))
	   ;;gentry will contain the gost entry (if any) that is already 
	   ;;present for the given condition at the given node
	   )
      (if  (null gentry)
	   ;;if such a gentry does not exist
	   (then 
	     (setf gentry (make-gentry :node (list (add-context node))))
	     (case sign
	       (:plus
		(pushnew gentry  (gost-entry-pluses entry)))
	       (:minus
		(pushnew gentry  (gost-entry-minuses entry)))
	       )
	; if this does not change the gentrys field 
        ; of entry, then use explicit methods
	     ))

      (setf (gentry-type gentry) entry-type)

      (add-contributors gentry contributors)
             
      (return-from enter-gost entry)
      ))
  )

; returns all the entrys listed under a
; given effect  of specified polarity
; search of GOST

(defun gost-entrys ( cond)
  (multiple-value-bind (condition sign)
      (convert-pat cond)
    (let ((entry (lookup-gost condition)))
      (if (null entry) ;no such entry exists
           (return-from gost-entrys nil))
      (case sign
	(:plus
	  (gost-entry-pluses entry))
	(:minus
	 (gost-entry-minuses entry))
	))

	; returns a list of gentry-s with the fields
	; node type contributors
    
    ))

    
;;gives the list of nodes which have node as a contributor for 
;;the given condition

(defun purpose-nodes (node cond)
    (let ((gentrys (gost-entrys cond)))
	 
	(for (gentry :in gentrys)
	     :when (member node (get-gentry-contributors gentry))
	     ;;when node is a member of the contributors for the condition
	     :save 
	     (get-gentry-node gentry)
	     ;;then the node of that gentry will be a purpose node
	))
)

;; this tries to remove the  contributor 'contributor' to the node 
;; 'node' for the condition 'cond'
;; this is called by function linearize.

(defun try-to-remove-contributor (node contributor cond)
  (let ((gost-entry (lookup-gost (convert-pat cond)))
	 ;;needed so that gentry can be removed if required
	 gentry)
    (setf gentry (find node (gost-entrys cond) :key #'get-gentry-node))
    (remove-contributor gentry contributor)
    (if (< (length (get-gentry-contributors gentry)) 1)
      ; this was the only contributor
      (then
	; make the phantom node a goal node and enter into the taskqueue
        (let ((newnode (my-copy-node (allnodes (node-nodenum node)))))
             (setf (node-type newnode) :goal)
             (setf (node-expanded newnode) nil)
	     (set-allnodes newnode (node-nodenum node))
             (enter-taskqueue newnode)
     )))
 ))

(defun remove-use-only-for-query-cond (node contributor cond)
  (let ((gost-entry (lookup-gost (convert-pat cond)))
        gentry)
    (setf gentry (find node (gost-entrys cond) :key #'get-gentry-node))
    (remove-contributor gentry contributor)
))

;; adds a list of contributors to an already existing gentry. It does
;; that by appending the new contributors with the old contributors
;; and adding the context to the resulting list

(defun add-contributors (gentry contributors)
  (let* ((contris (get-gentry-contributors gentry))
	 (contrib (cond ((listp contributors) contributors)
		        (t (list contributors)))))
  (setf contrib (for (contributor :in contrib)
		     :when (not (member contributor contris))
		     :save contributor))
  (setf (gentry-contributors gentry) (push (add-context (append contris contrib))
                                           (gentry-contributors gentry)))))

;; deletes a contributor from a list of contributors in gentry

(defun remove-contributor (gentry contributor)
  (let ((contris (get-gentry-contributors gentry)))
  (setf (gentry-contributors gentry) 
        (push (add-context (remove-context (remove-list contributor contris)))  
              (gentry-contributors gentry)))))

(defun get-gentry-contributors (gentry)
  (cdar (gentry-contributors gentry)))

(defun get-gentry-node (gentry)
  (cdar (gentry-node gentry)))

