;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Common Lisp NONLIN - University of Maryland at College Park 
;;;;
;;;; Version 1.2, 11/91
;;;; (email nonlin-users-request@cs.umd.edu for more info)
;;;; 
;;;; ESTABLISH.LISP:  Functions for querying/establishing conditions
;;;;
;;;; History:


;; This routine tries to establish the condition 'condition' at the node 
;; 'atnode'. This is called while expanding a Goal node. It first checks if
;; the condition is already true at atnode. If it is found to be not true
;; then it computes the four critical lists VL, PARVL, VNOTL and PARVNOTL.
;; if it succeeds it returns t and a list of contributors if not it returns 
;; nil. It calls q&a procedure for computing the above mentioned four lists
;; and linking procedure to link out all PNOTV-nodes.

(defun try-to-establish (condition atnode)
    (multiple-value-bind
	( already-true result binding ) (q&a  condition  atnode)
	
	(if already-true
	    ;; q&a has contributor nodes--
	    ;; result contains them
	    (then 
		  ;;this procedure will not enter anything in gost
		  ;; any procedure that calls this should worry about 
		  ;; entering stuff into gost
		  
		  (return-from try-to-establish (values t result binding)))
	    
	    (else
	      (if (null result)
                (then
       		 ;;there is no contributor and therefore the condition can  
	         ;;not be established without further reduction.		
	         (return-from try-to-establish (values nil nil)))
		 ;;else  result  contains criticial pv nodes and
		 ;; critical pvbar nodes
		(else
		  (let*((vl-nodes (car result))
		        (parvl-nodes (cadr result))
                        (vnotl-nodes (append (caddr result) (cadddr result)))
			(alternative-nets nil))
		  ;for each possible contributor node in vl and parvl find
		  ;out corresponding network (interaction free)
		  (for (contributor-node :in (append vl-nodes parvl-nodes))
		       :do
		       (let* ((suggested-net (if (member contributor-node parvl-nodes)
		                                (list (list (make-suggest-entry 
                                                 :link (list contributor-node atnode))))
					        '(nil)))
                       ;if we choose a parallel contributor then the link 
                       ;contributor-node->atnode must be included in the suggested network
		      ;;link_out all critical PNOTV nodes from the range
		      ;; (contributor-node atnode)
		             (result (resolve-establish-conflict
			           :offender-range (list contributor-node atnode)
				   ;;we consider that range chpv-node atnode
				   ;;is actually violating the protection
				   ;;interval of pvbar-node and its purpose
				   :conflict-nodes vnotl-nodes
				   :offender-cond condition
				   :extra-links suggested-net)))
			     (if (not (fail-p result))
 				(for (linearization :in result)
				     :do
				     (setf alternative-nets (append1 alternative-nets 
							  (list atnode contributor-node linearization condition)))))
   			))
		    (if (null alternative-nets)
		      (then
			   (return-from try-to-establish (values nil nil nil)))
		      (else
			(let* ((first-net (car alternative-nets))
			       (contributor (cadr first-net))
			       (linearization (caddr first-net))
			       ;;save alternative choices
			       (save-entry (make-backtrack-entry :type :establish :alternatives alternative-nets)))
			      (push (add-context save-entry) *context-list*)
			      (create-new-context)
			      (if (not-empty linearization)
				 (linearize linearization))
			      (return-from try-to-establish
				 (values t (list contributor) binding))
			))

		)    )
	  )))))
)
	
;;; q&a acts as a front end to tome... so, it will be here
;;; :all-bindings will make sure that all possible schemas will be returned

(defun q&a ( cond atnode &key (all-bindings nil) &aux nd-bd-pairs)
    ;; treat condition use-when (not (equal ?x ?y)) and (equal ?x ?y)
    ;; separately.
    (if (equal (car cond) 'equal)
      (then
	(let ((var1 (cadr cond))
	      (var2 (caddr cond)))
	  (if (equal var1 var2)
	     (return-from q&a (values t nil))
	     ;else
	     (return-from q&a (values nil nil)))
    )))
    (if (equal (car cond) 'not)
      (then
	(let ((pred (caadr cond))
	      (var1 (cadr (cadr cond)))
	      (var2 (cadr (cdadr cond))))
             (case pred
		(equal
		  (if (equal var1 var2)
		     (return-from q&a (values nil nil))
		     ;else
		     (return-from q&a (values t nil))))
	     )
    )))
    
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    ;;; check if cond holds in *always-ctxt*, 
    ;;;if yes, return it with *always-ctxt*
    ;;;as contributor (send the bind (if any))
    ;;; check if cond holds in *init-ctxt*
    ;;;if yes,  give *init-ctxt* as the contributor--send the binding (if any)
    ;;;until here, ground and non-ground cond are treated equally
    ;;; IF NOT (serious partial context searching using tome)
    ;;;if cond is ground, process it using q&a-process-tome-entry
    ;;;this is the only case when multiple contributors can result..
    ;;;if the condition is not ground, get all the matching tome-entrys
    ;;;and process each of them with q&a-process-tome-entry.  if any
    ;;;of them return a successful result, send that as the contributor
    ;;;else, choose one of the results for linking, and send it along.
    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
    
    (let (init-result always-result)
	 ;;;if a pattern is true in init-ctxt or always-ctxt, we don't
	 ;;;look for other contributors
	 (setf always-result  (q&a-always-ctxt cond))
	 (if always-result 
	     (if (not all-bindings)
	     ;;if the condition hosts in always ctxt
	     (return-from q&a (values t (list (node-nodenum *always-ctxt*)) 
		     (retrieval-result-binding (car always-result))))
	     ;;car chooses one binding of the possibly many
	     ;;else
	     (for (rbd :in always-result)
		  :do (push (list (LIST (node-nodenum *always-ctxt*))
				  (retrieval-result-binding rbd))
			    ND-BD-PAIRS))
	     ;;if all-bindings is true, we just push all possible 
	     ;;bindings into the nd-bd-pairs
	     ;;if init-ctxt is not true and always ctxt is,
	     ;;then we just sent the nd-bd-pairs that has been
	     ;;already set.
	     )
	     
	   )
	 (setf init-result (q&a-init-ctxt cond))
	 (if init-result 
	     (if (not all-bindings)
		 (return-from q&a (values t (list (node-nodenum *init-ctxt*))
		     (retrieval-result-binding (car init-result))))
		 ;;only if all-bindings is not set will we return one binding
		 ;;else we will make many bindings corresponding to this
		 ;;node
		 ;;else, we don't really have to check always result
		 ;;we just push our results 
		 ;;if there was a resutl from always, and we also have
		 ;;bindings for it...
		 (else
		      (for (rbd :in init-result)
			   :do (push (list (LIST (node-nodenum *init-ctxt*))
					   (retrieval-result-binding rbd))
				     ND-BD-PAIRS)
		      )
		      ;;don't return yet... we will also see
		      ;;for ground pat in tome
		      ))
	     )
	     (if ND-BD-PAIRS
		 ;;ONLY WHEN WE DON'T GET ANY THING
		 ;;FROM INPUT STATE WILL WE GO TO TOME--
		 ;;BASICALLY A WAY OF AVOIDING WRONG "AND"
		 ;;BINDINGS WHERE IN ONE COND IS CONTRIBUTED
		 ;;BY ONE NODE ANOTHER BY ANOTHER NODE
		 ;;AND THE PREVIOUS COND CAN'T BE CONTRIBUTED BY
		 ;;THE SECOND NODE
		  (return-from q&a
		      (values t nd-bd-pairs)))
    )
    ;;;;no more easy stuff--now for searching networks
    ;;;MAY HAVE TO RETURN BINDING ALSO SO THAT SELECT SCHEMA CAN USE IT
    (multiple-value-bind (condition sign)
	(convert-pat cond)
	(if (ground-pat-p condition)
	    (then (let ((entry (lookup-tome condition)))
		    (if entry
			;;if an entry exists
			(MULTIPLE-VALUE-BIND (result contributors)
			    (q&a-process-tome-entry entry 
						    condition sign atnode)
			      (IF all-bindings
				  (then
				    (if result
					(then
					  (PUSH (LIST contributors nil) nd-bd-pairs)
					  (RETURN-FROM q&a (VALUES result nd-bd-pairs)))
					;;(RETURN-FROM q&a (VALUES result (LIST (LIST contributors (LIST nil)))))
				 	;;if not all-bindings
				   ))	
				  (else
				    (RETURN-FROM q&a (VALUES result contributors))
				    ))))
		    ;;if not result
		    (IF (AND all-bindings nd-bd-pairs)
			(RETURN-FROM q&a (VALUES t nd-bd-pairs))
			
			(return-from q&a	;else
			  (values nil nil)))))
			    ;;since we don't do step addition, this condition
			    ;;should never arise.. except in goal expansion??
	    (else ; the condition is partially instantiated
		  (let ((matching-rentrys 
			    (lookup-tome-partial-pat condition))
			tome-query-results
		       )
		       (if (null matching-rentrys)
			   ;;no entrys match
		        (IF (AND all-bindings nd-bd-pairs)
			   (RETURN-FROM q&a (VALUES t nd-bd-pairs))
			
			   (return-from q&a	;else
			         (values nil nil nil)))
		       )
		       (for (rentry :in matching-rentrys)
			    :when (tome-entry-p (retrieval-result-data rentry))
			    ;;to avoid the problem of cleartop ?table coming in
			    ;;hack...
			    :do (let ((entry (retrieval-result-data rentry))
				      (binding (retrieval-result-binding 
						   rentry))
				      tome-query-result
				     )
				     (setf tome-query-result
					   (append1 (multiple-value-list
					       (q&a-process-tome-entry
						   entry condition sign atnode))
						    ;;q&a-process-tome-entry
						    ;;only deals with ground pat
						    ;;and as such returns no binding

						    ;;the binding comes from lookup-tome-partial-pat
						    binding))
				     (if (first tome-query-result)
					 ;;if the tome has direct contributor
					 (IF all-bindings
					     ;;we want all the contributors
					     (PUSH (LIST (second tome-query-result) binding) nd-bd-pairs)
					     ;;else
					     (return-from q&a 
					       (values (first tome-query-result)
						       (second tome-query-result)
						       (third tome-query-result)))
					     ;;;if we have any entry that matches,
					     ;;and it is not allbindings, thenwe will take it, throwing th
					     ;;;rest to wind    
					     )
					 ;else
					 (push tome-query-result
					       tome-query-results))
				))

		       (IF nd-bd-pairs
			   (RETURN-FROM q&a (VALUES t nd-bd-pairs))
			   )
		       ;;if we reach here,, none of the rentrys establish
		       ;;the condition.. a critical node analysis has to
		       ;;be done--FOR THAT WE CHOOSE ONE OF THE 
		       ;;TOME-QUERY-RESULTS AND SEND IT ALONG
		       (let ((tome-query-result 
				 (choose tome-query-results)))
			    (return-from q&a 
				(values (first tome-query-result)
					(second tome-query-result)
					(third tome-query-result)
					     )))
		  ))
	)
    ))
				     


;;; Given a ground condition  and a node this function computes the four
;;; critical nodes  and returns them to the caller.
;;; name correspondences pvs(VL), cpvs(PARVL), pvbars(VNOTL) and cpvbars(PARVNOTL).
(defun q&a-process-tome-entry (entry condition sign atnode &aux ppvs ppvbars
				     pvs cpvs pvbars cpvbars)
    ;;; 1. find p-nodes from tome
    ;;; 2. throw out the nodes that are "after" the atnode (from node-marks)
    ;;; 3. If there are no possible contributors then return nil.
    ;;; 4. if there are no cpv-bar nodes, return t and cpv-nodes as contributors
    ;;; 5.  else return four lists of critical nodes
    (case sign
	  (:plus
		(setf ppvs  (get-tome-entry-asserts entry))
		(setf ppvbars  (get-tome-entry-deletes entry)))
	  (:minus
		 (setf ppvbars  (get-tome-entry-asserts entry))
		 (setf ppvs  (get-tome-entry-deletes entry)))
	  ;; ppvs stands for potential pv node
    )
    
    (for (ppv :in ppvs)
	 :do
	 (case (node-mark ppv)
	       (:parallel
		   (push ppv cpvs))
	       ;; if the node is parallel, it is already
	       ;; critical
	       (:before
		       (push ppv pvs)))
	 ;; if the node is marked before, it is a pv
	 ;; node--it can possibly be cpv
    )
    (for (ppvbar :in ppvbars)
	 :do
	 (case (node-mark ppvbar)
	       (:parallel
		   (push ppvbar cpvbars))
	       ;; if the node is parallel, it is already
	       ;; critical
	       (:before
		       (push ppvbar pvbars))
	       ;; if the node is marked before, it is a pvbar
	       ;; node--it can possibly be cpvbar
	 )
	 
    )
    ;; at this point, we have four critical lists 
    ;; in pvs,cpvs,pvbars and cpvbars  
    
    (if (and (not (null pvs))
             (null pvbars)
             (null cpvbars))
       (then
	    (return-from q&a-process-tome-entry (values t pvs)))
	;; there are contributors in pvs (non null) and there are no deleters
	;; and therefore return true with contributors.
    )
    
    (if (and (null pvs)
             (null cpvs))
       (then 
	(return-from q&a-process-tome-entry (values nil nil)))
   
	;; there are no pv nodes and therefore the condition can not be 
	;; established.
	(else
	     ;; i.e. the condition has contributors
	     ;; but there are deleters also
	     (return-from q&a-process-tome-entry (values nil `(,pvs ,cpvs ,pvbars ,cpvbars))))
	     ;; send back both critical lists
    )
    
)
(defun choose (list)
  (car list))
				     
     
(defun not-empty (net)
  (not (equal net '(nil))))
		 

