;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Common Lisp NONLIN - University of Maryland at College Park 
;;;;
;;;; Version 1.2, 11/91
;;;; (email nonlin-users-request@cs.umd.edu for more info)
;;;; 
;;;; DEF.LISP:  Basic Structure Definitions/Operations
;;;;
;;;; History:
;;;; 10/04/91 bpk - added support for limit on planner cycles executed
;;;; 09/20/91 bpk - fixed bug in MAKE-CHILD
;;;; 09/13/91 bpk - added support for batch use mode

(defstruct (node (:conc-name snode-) (:print-function print-node) )
    				; this conc-name is used such that
				; we can write uniform node access functions
    (id (gensym "ND"))
    nodenum		        ; the number of the node in *allnodes*
    todo 			; the pattern with which the node is found
    parent 			; the parent of this node
    expansion 			; contains the relocated schema 
                                ; used in expannding the node
    expanconds  		; The list of conditions in the expansion
    children 			; list of children (:type node)
    prenodes 			; list of preceding nodes (node numbers)
    succnodes 			; list of following nodes (node numbers)
    ctxt  			; the effects of this node (that enter gost
								 ; and tome)
    type 			; the node type 
    				;one of{:dummy :planhead :goal :action :phantom}
    
    mark 			; the position of this node with respect
    				; to node in *netmark*
    				; one of {:before :after :parallel :node}
    expanded                    ;nil (not expanded) or the context in which it
				;was expanded
    nodevars
    ;; DEVISOR mods:
    (window (create-window nil)) ; time window
    (duration 0)                 ; time duration
    eventname                    ; event name (event-type nodes only)
    )

    ; effects will go into ctxt.  
    ; conditions can go in conds
    

(defun print-node (node stream depth)
  (let ((*print-gensym* nil)
	;;print the gensym without sharpsign
	(todo (snode-todo node))
	(parent (snode-parent node))
	(type (snode-type node))
	(children (snode-children node))
        (mark (snode-mark node))
        (duration (snode-duration node))  ; DEVISOR mods
        (window (snode-window node))
        (eventname (snode-eventname node))
	)
            
    (format stream "{")
    (format stream "<~s>" (snode-id node))
    (if (or type todo) (format stream "["))
    (if type (format stream "~s" type))
    (if todo (format stream "~s" todo))
    (if eventname (format stream "(~s)" eventname)) ; DEVISOR mod
;    (if mark (format stream "mk=~s" mark))
    (when *devisor-mods*
       (format stream " dur=~s wdw=~s " duration window))
    (if (or type todo) (format stream "]"))
    (when *devisor-mods* (format t "~%~35T"))  ; DEVISOR mods
    (if (not (> depth 0))
	(then
         (let ((prenodes (get-prenodes node))
               (succnodes (get-succnodes node)))
	  (if prenodes (format stream "{PRE:~s}" prenodes))
	  (if succnodes (format stream "{SUCC:~s}" succnodes))
	  (if parent (format stream " PAR:{<~s>} " (snode-id parent)))
;	  (if children
;	      (progn (format stream "CHL:(")
;		     (for (child :in children)
;			  :do (format stream "{<~s>}"  (snode-id child)))
;		     (format stream ")")))
	  ))
    (format stream "}"))
;   (format stream "CTXT: ~s " (snode-ctxt node))
))


;A LITTLE HACK TO GET UNIFORM ACCESS FUNCTIONS
; Uniform node-access functions 

; this macro is supposed to define node access functions such tha
; access function can handle both nodenumbers and nodes 
; SETTING FUNCTIONS STILL HAVE TO BE TAKEN CARE OF

(defmacro define-node-function (function-name sfunction-name)
      `(progn (defun ,function-name  (nodearg)
		   (typecase nodearg
		       (node (,sfunction-name  nodearg))
		       (integer (,sfunction-name (allnodes nodearg)))))
	    (defsetf ,function-name (nodearg) (new-value)
		 `(typecase  ,nodearg
		    (node (setf (,',sfunction-name ,nodearg) ,new-value))
		    (integer (setf (,',sfunction-name (allnodes ,nodearg))
				  ,new-value)))))
)

(eval-when (compile load eval)
  (let (( *correspondence-list*
	 '((node-nodenum snode-nodenum)( node-mark snode-mark)
	   (node-type snode-type)( node-todo snode-todo) 
	   (node-expansion snode-expansion) (node-parent snode-parent)
	   (node-expanconds snode-expanconds) ( node-children snode-children)
	   (node-prenodes snode-prenodes) (node-succnodes snode-succnodes)
	   (node-expanded snode-expanded)(node-ctxt snode-ctxt)
           (node-eventname snode-eventname) ; DEVISOR mods
           (node-window snode-window) (node-duration snode-duration))))
    
    (do ((x *correspondence-list* (cdr  x) ))
	((null x))
      (eval `(define-node-function ,(car (car x)) ,(cadr (car x))))
      )))

;;the following two routines take care of both nodes and nodenumbers
;;they first check if the current context is same as the latest context of
;;the successor nodes of 'node'. If yes then it adds 'snode' to the list
;;of successor nodes of 'node' otherwise a new list of successor/predecessor
;;nodes is created with the current context and pushed to the list 

(defun make-succnode (snode node)
  (if (member (node-nodenum snode) (get-succnodes node)) 
      (node-succnodes node)
  (else
  (if (equal *current-context* (get-context (car (node-succnodes node))))
     (setf (node-succnodes node)(push (append1 (car (node-succnodes node))
                         (node-nodenum snode)) (cdr (node-succnodes node))))
     ;else
     (setf (node-succnodes node)
        (push (add-context (append1 (get-succnodes node)
                              (node-nodenum snode))) (node-succnodes node)))
  ))
  )
)

(defun make-prenode (pnode node)        
  (if (member (node-nodenum pnode) (get-prenodes node)) 
     (node-prenodes node) 
  (else
  (if (equal *current-context* (get-context (car (node-prenodes node))))
     (setf (node-prenodes node)(push (append1 (car (node-prenodes node))
                         (node-nodenum pnode)) (cdr (node-prenodes node))))
     ;else
     (setf (node-prenodes node)
        (push (add-context (append1 (get-prenodes node)
                              (node-nodenum pnode))) (node-prenodes node)))
  ))
  )
)	

; this routine makes 'child' a child of 'node'.

;; FIXED 9/18/91
(defun make-child (child node)
  (when (not (node-p node)) (setf node (allnodes node)))
  (setf (node-children node) (pushnew child (node-children node)))
  (setf (node-parent child) node)
)
			         
; ************schemas and conditions***************************************
        
(defstruct  schema
   (id (gensym "SCH"))
    name 
    todo 	                ; this is the pattern on which fetch is done
    				; it should actually go into a ACHSCHEMA
				; index (or relevance list)

    strip 			; array of nodes
    size                        ; the size of the array
    conditions 		        ; these conditions refer to the node numbers
                                ; in the expansion
    effects			
                                ; effects also refer to the nodenumbers in
                                ; the expansion... Generally opschema's have
                                ; conditions and actschemas have effects
    (duration 0)                ; duration of action/event - DEVISOR mods
    (window (create-window nil))      ; window for action/event
     vars)

(defun print-schema (sch &optional (stream t) (depth 0))
  (let ((*print-array* t) (*print-gensym* nil)
	(conditions (schema-conditions sch))
	(effects (schema-effects sch))
	(strip (schema-strip sch))
	(size  (schema-size sch))
	(todo  (schema-todo sch))
	(variables (schema-vars sch))
        (duration (schema-duration sch)) ; DEVISOR mods
        (window (schema-window sch))
	)
    (format stream "~&{~s} ~%" (schema-id sch))
    (format stream "~2T ~s::~s~%" (schema-name sch) todo)
    (if strip
	(then
	  (format stream "~5T Expansion:~%")
	  (do ((index 0 (+ index 1)))
		  ((> index (- size 1)))
		(format stream "~10T ~s " index)
		(print-node (aref strip index) stream (1+ depth))
		(format stream "~%")
		)
	  ))
    (if conditions
	(then
	  (format stream "~5T Conditions:~%")
	  (for (scond :in conditions)
	      :do (format stream "~10T ~s" scond))
	  ))
    (if effects
	(then
	  (format stream "~& ~5T Effects:~%")
	  (for (seff :in effects)
	       :do (format stream "~10T ~s" seff))
	  ))

    (when *devisor-mods* ; DEVISOR mods
       (format stream "~& ~5T Time: dur=~s  wdw=~s ~%"
          duration window))

    (if variables
	(then (format stream "~5T Vars: ~s~%" variables))
	)
    ))
    

(defstruct (scondition (:print-function print-scondition))
    (id (gensym "SC"))
    type			; :use-when, :precond or :use-only-for-query
    pattern			; condition pattern (+ and -)??
    atnode			; should be true at the given node
    contributors		; is made true from the contributors
    binding			; is kept for use-only-for-query conditions
				; which may need rebinding at a later stage
)
(defun print-scondition (scond stream depth)
  (let ((*print-gensym* nil))
    (format stream "<<~s>>" (scondition-id scond))
    (if (< depth 1)
	(then-let ((type (scondition-type scond))
		   (pattern (scondition-pattern scond))
		   (atnode (scondition-atnode scond))
		   (contributors (scondition-contributors scond))
		   )
		  (if type (format stream " ~s " type))
		  (if pattern (format stream "~s" pattern))
		  (if atnode (format stream " :at ~s " atnode))
		  (if contributors (format stream " :from ~s " contributors))
		  (if (eql type :use-only-for-query) (format stream " :binding ~s " (scondition-binding scond)))
		  (format stream "~%")
		  ))))
    

(defstruct (seffect (:print-function print-seffect))
    (id (gensym "SE"))
    type    ;add/delete????
    pattern ;+ and -
    atnode  ;this is the effect of atnode
)    
        
(defun print-seffect (seff stream depth)
  (let ((*print-gensym* nil))
    (format stream "<<~s>>" (seffect-id seff))
    (if (< depth 1)
	(then-let ((type (seffect-type seff))
		   (pattern (seffect-pattern seff))
		   (atnode (seffect-atnode seff))
		   )
		  (if type (format stream " ~s " type))
		  (if pattern (format stream "~s" pattern))
		  (if atnode (format stream " :at ~s " atnode))
		  (format stream "~%")
		  ))))
;***********************Suggest-entrys(for linearization)*********************
(defstruct (suggest-entry (:print-function print-suggest-entry))
   link		  ; the link which must be removed. 
   cond		  ; the offender condition i.e. the interacting condition.
   contributor    ; list of atmost two (node node-purpose) pairs. If it
		  ; contains two pairs, then the first pair supplies the 'cond'
		  ; and the other pair supplies ~'cond'.
)

(defun print-suggest-entry (suggest-entry stream depth)
 (declare (ignore depth))
 (format stream "link:~s  Remove-contributor:~s ~%" (suggest-entry-link suggest-entry)
		(suggest-entry-contributor suggest-entry)))

(defvar *suggest* nil)

;*******************taskqueue*******************
(defvar *taskqueue* nil "the task queue for the planner")
    
;;enters a new task to the taskqueue. Current context is added to the task
;;for remembering when the task was created and is used for backtracking 
;;purposes.

(defun enter-taskqueue (node &optional first)
    (unless (node-p node)
	    (setf node (allnodes node)))
    (if (eq first :first)
	(then 
	      (push (add-context node) *taskqueue*))
	(else 
	      (setf *taskqueue* (append1 *taskqueue* (add-context node)))
	)))
    
;;pick-taskqueue returns the first task which has not yet been expanded.
;;expanded field of a node(task) keeps track of when a task was expanded.
;;a null value of for expanded indicates that the task has not been 
;;expanded.

(defun pick-taskqueue ()
    ; THIS FUNCTION SHOULD BE CHANGED TO IMPLEMENT DIFFERENT SEARCH
    ; STRATEGIES
  (let ((tasknode (car (for (node-context :in *taskqueue*)
		            :when (null (snode-expanded (cdr node-context)))
			    :save (cdr node-context)))))
       (if tasknode
	  (setf (snode-expanded tasknode) *current-context*))
       (return-from pick-taskqueue tasknode)
))

(defun print-taskqueue ()
  (for (node-context :in *taskqueue*)
       :do (let ((node (cdr node-context)))
	        (if (null (snode-expanded node))
                 (format t "~& ~s  ~s ~s" (node-nodenum node) (node-type node) (node-todo node))))))
    

(defun reset-taskqueue ()
    (setf *taskqueue* nil))        
    

(defvar *planhead*   nil "the beginning of the plan")
(defvar *plantail* nil "the end of the plan")        
    
;********allnodes*************************
    
(defvar *allnodes* nil "points to the array containing the flexible strip")
(defvar *striplen* 0 "the current length of the flexible strip allnodes")    
(defvar MAXLEN 100 "the maximum length of flexible strip")
(defconstant incr 25 "the increment for flexible strip")

; this  will serve to access the flexible strip
(defun allnodes (nodenum)
    (cond ((not (numberp nodenum))
	   (cerror "~s" "allnodes is getting a non-number argument"))
	  ((eql nodenum -1) *init-ctxt*)
	  ((eql nodenum -2) *always-ctxt*)	  
	  ;;We allow -1 and -2 as special nodenumbers
          ((> nodenum (1- *striplen*))
           (cerror "~s" "The nodenumber is more than striplength")
          )
          (t (cdar (aref *allnodes* nodenum)))))

(defun print-allnodes (&optional (stream t) (depth 0))
  (format stream "~&CONTENTS of ALLNODES:")
  (do ((index 0 (1+ index)))
      ((> index (1- *striplen*)))
      (if (not (null (aref *allnodes* index)))
        (then
          (format stream "~&~5T~s " index)
          (print-node (cdar (aref *allnodes* index)) stream depth)
        )))
) 

;; set-allnodes puts node at *allonodes*[nodenum]. This works in a fashion
;; similar to make-prenode/make-succnode i.e. checks the current context 
;; with the latest context and creates a new context if they differ. 
;; it also checks if the nodenum is greater than the maxlen.
;; if yes, increment the maxlen and 

(defun set-allnodes	(node &optional (nodenum *striplen*))
    ; adjust the size of the array
    (if (not (array-in-bounds-p *allnodes* nodenum))
	(progn
	      (setf maxlen (+ maxlen incr))
	      (adjust-array *allnodes* (list maxlen) :initial-element nil)))
    (setf (node-nodenum node) nodenum)
    ; this also sets the nodenum. Infact, this is the only routine
    ; that sets the nodenum
    (if (equal *current-context* (get-context 
       (if (atom (aref *allnodes* nodenum)) nil
           (car (aref *allnodes* nodenum)))))
       (setf (aref *allnodes* nodenum) (push (add-context node)
					  (cdr (aref *allnodes* nodenum))))
       ;else
       (setf (aref *allnodes* nodenum) (push (add-context node) (aref *allnodes* nodenum))))
    (if (> (+ nodenum 1) *striplen*)
	; if the node number is greater than the
	; current strip length
	; the plus 1 is to take care of 0 offset array
	(setq *striplen* (+ nodenum 1))
	; make *striplen*gth to be nodenum
	; this will ensure that the strip length is not changed when a
	; new node is placed in the middle.
    )
)

(defun reset-allnodes ()
    (setq *striplen* 0 *MAXLEN* 100)
    (setf *allnodes* (make-array (list MAXLEN) :adjustable t :initial-element nil))    
)    
        
;********reset all datastructures & History mechanism*****************
(defvar **HISTORY** nil "History persists between planning sessions!!")
(defvar *tasknet* nil "the tasknet")
(defvar *nonlin-use-mode* 'interactive "nonlin use mode: interactive or batch")
(defvar *debug* nil)

(defvar *cycle-count* 0 "the number of cycles executed by NONLIN")
(defvar *cycle-limit-p* nil "does a limit on the num. of cycles exist")
(defvar *cycle-limit* nil "the limit on the num. of cycles to execute")

(defvar *kids* nil "parent-child hierarchy")
(defvar *planner-in*  nil "planner input")
(defvar *planner-out* nil "planner output")
(defvar *goals* nil "goal specification for problem")

(defun reset-datastructures ()
    (if *tasknet*
	;if tasknet has been initialized already
	(then (push (list :tasknet *tasknet* 
			  :plan *allnodes* 
			  :size *striplen* 
			  :tome *TOME* 
			  :gost *GOST*
			  :init-ctxt *init-ctxt*
			  :always-ctxt *always-ctxt*
			  :planhead *planhead*)
		    **HISTORY**)))
    (reset-allnodes)
    (reset-tome)
    (reset-gost)
    (reset-taskqueue)
    (reset-terminate-flag)
    (reset-globals)
)

(defun reset-globals ()
   ;;; resets values of globals (invoked before each new problem)

   (setf *cycle-count* 0)

   (setf *goals* nil)
   (setf *kids* nil)
   (setf *planner-in* nil)
   (setf *planner-out* nil)
) 

(defvar *netmarked* nil   "the node with respect to which the net has been marked")   
    
(defmacro predicate (pattern)
    ;;used by hash tables for indexing    
    `(car ,pattern))    

(defvar *terminate* nil "flag for proper termination of the planner")
(defun reset-terminate-flag()
   (setf *terminate* nil)
)

(defun debug-p ()
   *debug*)
(defun debug (&optional (value t))
  (setq *debug* value))
