;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Common Lisp NONLIN - University of Maryland at College Park 
;;;;
;;;; Version 1.2, 11/91
;;;; (email nonlin-users-request@cs.umd.edu for more info)
;;;; 
;;;; BACKTRACK.LISP:  Backtracking Procedures
;;;;
;;;; History:
;;;; 10/04/91 bpk - added support for batch use mode
;;;; 09/22/91 bpk - Fixed bug in BACKTRACK-THRU-CONTEXT
       

(defvar *context-counter* 0 "generates monotonically increasing contexts")
(defvar *current-context* nil "defines the context for nodes and links")
(defvar *context-list* nil "Remembers the context history")
(defvar *init-ctxt-modification-list* nil "remembers when a pattern was moved 
					   from *init-ctxt* to *planhead*")

;; generates context variables

(defun make-context-symbol ()
   (make-symbol (format nil "C~D" (incf *context-counter*))))

(defstruct (backtrack-entry (:print-function  print-backtrack-entry))
   type ; one of :schema, :expand, :establish
   alternatives ; list of alternative choices
)
(defun print-backtrack-entry(backtrack-entry stream depth)
  (declare (ignore depth))
  (format stream "TYPE:~s ALTERNATIVES:~s~%" (backtrack-entry-type backtrack-entry)
	         (backtrack-entry-alternatives backtrack-entry))
)


(defun create-new-context ()
  (setf *current-context* (make-context-symbol))
)

(defun reset-context()
  (setf *context-list* nil)
  (create-new-context)
)
;; this function basically implements backtracking through contexts in
;; a depth first manner. When called, it backtracks to the point where
;; there is an alternative choice. Depending on the type of backtracking
;; point, it takes corresponding action

(defun backtrack-thru-context ()
  (prog()
   restart
   (if *context-list*
    ;there are backtracking points
    (then
      ;backup one level
      (backup-context *current-context*)
      (setf *current-context* (get-context (car *context-list*)))
      (let* ((type (backtrack-entry-type (cdar *context-list*)))
	     (alternative (pop (backtrack-entry-alternatives (cdar *context-list*)))))
       (if (null alternative)
	;if there is no alternative then backtrack further
	(then
	     (pop *context-list*)
	     (go restart))
        (else
	  ;we have found a backtracking point with an alternative
	  (case type
	    (:expand
		(create-new-context)
	        ;linearize with the new choice
		(linearize alternative)
	    )
	    (:establish
	        ;select the alternate contributor, linearize and update gost
		(let* ((atnode (car alternative))         ; atnode is a node number
		       (contributor (cadr alternative))
		       (linearization (caddr alternative))
		       (condition (cadddr alternative))
		       (nchild (my-copy-node (allnodes atnode))))
		    (create-new-context)
		    (if (not-empty linearization)
		       (linearize linearization))
		    (setf (node-type nchild) :phantom)
		    (make-child nchild atnode)
                    (set-allnodes nchild atnode) ; FIXED BUG HERE      		    
 
		    (enter-gost (node-todo nchild) :phantom
				(node-nodenum nchild) (list contributor))
	      ))
	      (:schema
		   (let ((atnode (car alternative))
			 (chosen-schema (cadr alternative)))
			(create-new-context)
			;make a expansion with the new schema
		        (expand-node-using-schema-instance chosen-schema atnode)
	      )))
            (return-from backtrack-thru-context t)
           ))))
    (else
      (format t "no more solutions")
      (setf *terminate* t)
      (return-from backtrack-thru-context nil))
   ))
)
;backtracking planner. Using "catch" and "throw" feature of commnon lisp
;this cleans up the pending calls while backtracking
 
(defun bplanner () 
   (do ((result nil)(finished nil))
       (finished :good-bye)
      (setq result (catch :backtrack (planner)))
      (if (and (equal *nonlin-use-mode* 'interactive) (or (eq result :failure)
              (y-or-n-p "Try again? (~D more alternatives)" (length *context-list*))))
          (setq finished (not (backtrack-thru-context)))
          ;else
          (setq finished t)))
)

(defun backtrack ()
  (throw :backtrack :failure))

;backs up to the context previous to "context"

(defun backup-context (context)
  (backup-allnodes context)
  (backup-gost context)
  (backup-tome context)
  (backup-taskqueue context)
  (restore-init-ctxt context)
)

;; this function backsup allnodes one level i.e. to the context previous
;; to "context". Basicallt it matchs the context "context" with the 
;; context of the nodes,succnodes,prenodes etc which changes with
;; context and throws it away if the two context match. It also backtracks
;; binding of the use-only-for-query conditions.

(defun backup-allnodes (context)
 (do ((index 0 (1+ index)))
     ((> index (1- *striplen*)))
     (let ((node-context-pair (car (aref *allnodes* index))))
      (if (eql context (get-context node-context-pair))
	 ;this node was created in the current context and therefore
	 ;delete it
         (setf (aref *allnodes* index) (cdr (aref *allnodes* index))))
     (let* ((node (allnodes index))
	    (expanconds (node-expanconds node))
	    (effects (node-ctxt node))
            (prenodes (car (node-prenodes node)))
            (succnodes (car (node-succnodes node))))
          (if (eql context (get-context prenodes))
            (setf (node-prenodes node) (cdr (node-prenodes node))))
          (if (eql context (get-context succnodes))
            (setf (node-succnodes node) (cdr (node-succnodes node))))
	  (for (expancon :in expanconds)
	       :do
	       (if (eql (scondition-type expancon) :use-only-for-query)
		  (if (eql context (caar (scondition-binding expancon)))
		     (then
		       ;reverse of rebinding 
		       (let ((new-expancon (my-copy-scondition expancon))
			     (current-binding (get-scondition-binding expancon)))
			  (setf (scondition-binding new-expancon) (cdr (scondition-binding new-expancon)))
			  (setf (scondition-pattern new-expancon)
				(remove-binding (scondition-pattern new-expancon)
						(mapcar #'reverse current-binding)))
			  (destructive-replace-variables (scondition-pattern new-expancon)
							 (get-scondition-binding new-expancon))
			  (dremove expancon (node-expanconds node))
			  (pushnew new-expancon (node-expanconds node))
			  (for (effect :in effects)
			       :do
			       (setf (seffect-pattern effect)
				     (remove-binding (seffect-pattern effect) (mapcar #'reverse current-binding)))
			       (destructive-replace-variables (seffect-pattern effect)
							      (get-scondition-binding new-expancon)))
			  )))))
        )))
)
    
;; backsup gost through the context "context". Checks the contexts of the gentrys
;; and the contributors and removes them if a match is found

(defun backup-gost (context)
 (for (gost-entry :in (all-gost-entrys))
      :do
      (let ((pluses-gentrys (gost-entry-pluses gost-entry))
           (minuses-gentrys (gost-entry-minuses gost-entry)))
       (for (pluses-gentry :in pluses-gentrys)
            :do
            (if (eql context (get-context (car (gentry-node pluses-gentry))))
             (setf (gentry-node pluses-gentry)(cdr (gentry-node pluses-gentry))))
            (if (eql context (get-context (car (gentry-contributors pluses-gentry))))
            (setf (gentry-contributors pluses-gentry)(cdr (gentry-contributors pluses-gentry))))
            (if (null (gentry-contributors pluses-gentry))
              (dremove pluses-gentry (gost-entry-pluses gost-entry))))
       (for (minuses-gentry :in minuses-gentrys)
          :do
            (if (eql context (get-context (car (gentry-node minuses-gentry))))
             (setf (gentry-node minuses-gentry)(cdr (gentry-node minuses-gentry))))
	  (if (eql context (get-context (car (gentry-contributors minuses-gentry))))
	    (setf (gentry-contributors minuses-gentry)(cdr (gentry-contributors minuses-gentry))))
            (if (null (gentry-contributors minuses-gentry))
              (dremove minuses-gentry (gost-entry-minuses gost-entry)))))
))

;; backsup tome one level
(defun backup-tome (context)
 (for (tome-entry :in (all-tome-entrys))
      :do
      (if (eql context (get-context (car (tome-entry-asserts tome-entry))))
	 (setf (tome-entry-asserts tome-entry) (cdr (tome-entry-asserts tome-entry))))
      (if (eql context (get-context (car (tome-entry-deletes tome-entry))))
	 (setf (tome-entry-deletes tome-entry) (cdr (tome-entry-deletes tome-entry))))
))

;restore *init-ctxt* if it was modified in the context "context"
(defun restore-init-ctxt (context)
 (do ((modify-context (get-context (car *init-ctxt-modification-list*)) 
 		      (get-context (car *init-ctxt-modification-list*))))
     ((not (eql modify-context context)))
     (let ((effect (cdr (car *init-ctxt-modification-list*))))
           (pop *init-ctxt-modification-list*)
	   (dremove effect (node-ctxt *planhead*) :test 'equal)
	   (store-pat effect :place (node-ctxt *init-ctxt*))
           (let ((gentrys (gost-entrys effect)))
		(for (gentry :in gentrys)
		     :do
		     (setf (gentry-contributors gentry)
			   (push (substitute (node-nodenum *init-ctxt*)
				   (node-nodenum *planhead*)
			    	   (pop (gentry-contributors gentry)))
				 (gentry-contributors gentry)))
		)
            )
       ))
)

;; backsup taskqueue one level. Deletes all tasks that were generated
;; in context "context". It also resets the expanded field of all tasks
;; to nil that were expanded after the context "context". The "later"
;; condition is easy to check because the context variables are made
;; with monotonicity of the context-counter

(defun backup-taskqueue (context)
  (setf *taskqueue* (for (node-context :in *taskqueue*)
                         :when (not (eql context (car node-context)))
                         :save node-context))
  (for (node-context :in *taskqueue*)
       :do
       (if (later (snode-expanded (cdr node-context)) context)
          (setf (snode-expanded (cdr node-context)) nil))
  )
)


(defun get-context (item)
  (car item))

(defun add-context (node-or-nodelist)
  (cons *current-context* node-or-nodelist))

(defun remove-context (node-or-nodelist)
  (cdr node-or-nodelist))

(defun deleters (entrys context)
  (for (entry :in entrys)
       :when (not (eql (get-context entry) context))
       :save
       entry
  )) 
	
(defun later (context1 context2)
 (let ((string1 (string context1))
       (string2 (string context2)))
      (and (>= (length string1) (length string2))
           (string>= string1 string2))
))
