;;;-----------------------------------------------------
;;;
;;;  disttaskalloc.lisp
;;;  Nick Short
;;;  6/92
;;;
;;;  This file contains the branch and bound method
;;;  for doing task allocation of m tasks to n processors
;;;  as a function of constraints.
;;;
;;;
;;;  Two functions are implemented:  Depth first and
;;;  best first.  The best-first algorithm is in this
;;;  file.
;;;
;;;-----------------------------------------------------

;;;----------------------------------------------------
;;;
;;;   BEST FIRST ALLOCATION OF TASKS
;;;
;;;   In this algorithm, only the frontier (i.e., tqueue for task queue)
;;;    of the search tree is stored, as backtracking involves expanding
;;;   upon a node in the frontier.  Each node in tqueue has
;;;   an assignment arrary property X along with the
;;;   cost of that assignment.  The cost is used
;;;   the sort the list tqueue.  Each node is expanded
;;;   in a breadth first manner, with constraints being
;;;   used to decide what nodes will be added to tqueue.  When
;;;   an assignment is completed, it is compared to the current
;;;   task assignment to see if a better solution was found.
;;;
;;;   Of particular note,  computation can be halted either by
;;;   limiting the number of nodes examined via icount, by 
;;;   establishing a deadline (in universal time), or by specifying
;;;   the amount of real seconds will be alloted for execution.
;;;   

(defvar *optimal-allocation* nil)

;;; returns multiple values where the first is the assignment matrix and
;;; the second is the cost

(defun allocate-tasks (C &key (verbose t) (icount nil) (deadline nil) 
			 (allocated-time nil) (search-strategy 'best-first))
  (declare (special verbose icount deadline search-strategy))
  (let ((task (make-symbol "N"))
	value-node)
    (setf *optimal-allocation* nil)
    (setf (get task 'id) 0)
    (setf (get task 'assignment)
	  (make-array (list 0 (length (cnode-processors C)))
		      :element-type 'bit
		      :initial-element 0
		      :adjustable t))
    (if allocated-time (setf deadline (+ (get-universal-time) allocated-time)))
    (setf value-node (allocation-search (list task) C))
    (values (get value-node 'assignment)
	    (get value-node 'cost))))

(defun allocation-search (queue C)
  (declare (special verbose icount deadline search-strategy)
	   (inline insert-task add-new-task-dimension 
		   print-assignment rule-1 rule-2 rule-3 rule-4))
  (do ((tqueue queue)
       (curr-count 0))
      ((or (null tqueue) 
	     (and icount 
		  (>= curr-count icount))
	     (and deadline
		  (> (get-universal-time) deadline)))
	 (if verbose
	     (format t "~%~A nodes examined out of ~A possible~%"
                    curr-count 
		    (/ (1- (expt (length (cnode-processors C))
				 (1+ (length (cnode-tasks C)))))
		       (1- (length (cnode-processors C))))))
	 *optimal-allocation*)
	(do* ((i 0 (1+ i))
		 (task (get-task-choice tqueue search-strategy))    
		 (task-array (get task 'assignment))
		 (new-task (make-symbol 
			    (format nil "~A~A" task i))
			   (make-symbol 
			    (format nil "~A~A" task i)))
		  X-temp 
		 (tqueue-new (remove task tqueue)) 
		 (nproc (length (cnode-processors C)))) 
	       ((>= i nproc) (setf tqueue tqueue-new)
			     (incf curr-count nproc))
	       
	       ;;; create candidate node 
	       (setf (get new-task 'id) (1+ (get task 'id)))
	       (setf X-temp
		     (add-new-task-dimension (get new-task 'id) i
					     task-array))
	       
	       ;;; diagnostic code
	       (if verbose
		   (progn
		     (format t "~%Examining Node ~A : cost ~A~%" task 
			     (get task 'cost))
		     (format t "Selection Queue: <~{~A, ~}>~%" tqueue)))

	       ;;; add this node to the frontier tqueue?

	       (if (and (rule-1 i X-temp C) 
			(rule-2 i X-temp C)
			(rule-3 i X-temp C) 
			(rule-4 i X-temp (get *optimal-allocation* 'cost) C))
		   (progn 
			  ;;; assign new task assignment array
			  (setf (get new-task 'assignment)
				X-temp)

			  ;;; calculate cost
			  (setf (get new-task 'cost)
				(total-cost 
				 (get new-task 'assignment) C))

			  ;;; add node to appropriate queue
			  (if (equal (length (cnode-tasks C))
				     (get new-task 'id))
			      (progn
				(if verbose 
				    (progn
				      (print-assignment X-temp)
				      (format t "Ass. Cost = ~A~%"
					      (get new-task 'cost))))
				(setf *optimal-allocation*  new-task))
			      (progn 
				(if verbose
				     (format t "Adding ~A: cost ~A~%"
					     new-task (get new-task 'cost)))
				     (setf tqueue-new
					   (insert-task 
					        new-task tqueue-new)))))))))

;;;------------------------------------------
;;; inserts task into a sorted  list based on an index of cost
;;; and the lexicographic ordering of node labels (done to bias
;;; the search from left to right when node costs are equal)

(defun insert-task (new-task tqueue)
  (cond ((null tqueue) (list new-task))
	((< (get new-task 'cost)
	    (get (car tqueue) 'cost))
	 (cons new-task tqueue))
	((= (get new-task 'cost)
	    (get (car tqueue) 'cost))
	 (if (<-task new-task (car tqueue))
	     (cons new-task tqueue)
	     (cons (car tqueue) (insert-task new-task (cdr tqueue)))))
	(t (cons (car tqueue) (insert-task new-task (cdr tqueue))))))

;;; lexicographic ordering test of the task labels

(defun <-task (ni nj)
  (let ((n1 (string-trim "N" (string ni)))
	(n2 (string-trim "N" (string nj))))
  (cond ((and (= (length n1) (length n2))
	      (< (read-from-string n1) (read-from-string n2))))
	((< (length n1) (length n2))))))

;;;------------------------------------------

(defun add-new-task-dimension (tid pid X)		    
 (let* ((ntasks (car (array-dimensions X)))
       (nproc (cadr (array-dimensions X)))
       (X-new  (make-array
	  (list (1+ ntasks)  nproc)
	  :element-type 'bit
	  :initial-element 0
	  :adjustable t)))

   
   ;;; modify array for creating the new array X-new
  (adjust-array X
		(list (1+ ntasks) nproc)
		 :element-type 'bit
		 :initial-element 0)
  (bit-ior X-new X t)               ;;; destructively modify X-new
  (setf (bit X-new (1- tid) pid) 1)
  ;;; return old array to original form
  (adjust-array X (list ntasks nproc))
  X-new))

;;;------------------------------------------
;;;
;;;  This allows the search strategy of the routine
;;;  allocation-search to have some variance in the
;;;  selection of the task to expand.  As of 6/92, only
;;;  two search strategies are used: best-first and 
;;;  random.

(defun get-task-choice (queue strategy)
  (case strategy
	(best-first (car queue))
	(random (nth (random (length queue)) queue))))


;;;------------------------------------------
;;; rule that checks if a processors allows a task to execute on its platform

(defun rule-1 (pid X C)
  (declare (special verbose))
  (let ((value 
	 (not (zerop 
	       (bit (cnode-P C) (1- (car (array-dimensions X))) pid)))))
    (if (and verbose (not value))
	(format t "Rule-1 failure~%"))
    value))

;;;------------------------------------------
;;; rule that checks if the assignment of a task to processor-i will not
;;; conflict with another task's assignment to the same processor

(defun rule-2 (pid X C)
  (declare (special verbose))
  (let ((value
  (do ((tid (1- (car (array-dimensions X))))
       (i 0 (1+ i))
       val)
       ((or (>= i tid) val) (not val))
	  (setf val (and (bit X i pid)
			 (not (zerop (bit (cnode-E C) 
					   i tid))))))))
    (if (and verbose (not value)) (format t "Rule-2 failure~%"))
    value))


;;;------------------------------------------
;;; rule for seeing if a task assignment to a processor will
;;; force the system to exceed its storage requirement
	 
(defun rule-3 (pid X C) 
  (declare (special verbose))
  (let ((value 
	 (do ((ntasks (car (array-dimensions X)))
	      (i 0 (1+ i))
	      (M 0))
	     ((>= i ntasks) M)
	     (if  (bit X i pid)
		 (incf M 
		       (tnode-Memory-requirement 
			(nth i (cnode-tasks C))))))))
    
    (cond ((> value (pnode-available-storage 
			      (nth pid (cnode-processors C))))
	   (if verbose
	       (format t "Rule-3 failure: M = ~A, S = ~A~%" value
		       (pnode-available-storage 
			(nth pid (cnode-processors C)))))
	   nil)
	  (t t))))


;;;------------------------------------------
;;; rule that sees if the cost of a task assignment exceeds the
;;; the cost of the best known assignment.

(defun rule-4 (pid X optimal-allocation C) 
  (declare (special verbose))
  (let* ((TC (total-cost X C))
	 (value  (or (and optimal-allocation
			  (< TC optimal-allocation))
		     (not  optimal-allocation))))
    (if (and verbose (not value))
	(format t "rule-4 failure: total cost: ~A, Optimal cost: ~A~%"
		TC optimal-allocation))
    value))

;;; w is used to scale processing cost and IPC to take into
;;; account any difference in measuring units.

(defun Total-Cost (X C &optional (w 1))
    (do ((m (car (array-dimensions X)))
	 (n (cadr (array-dimensions X)))
	 (i 0 (1+ i))
	 (result 0))
        ((>= i m) result)
	(dotimes (k n)
	 (incf result
	       (+ (* w (aref (cnode-Q C) i k)
		     (bit X i k))
		  (do ((l 0 (1+ l))
		       (result2 0))
		      ((>= l n) result2)
		      (dotimes (j m)
			 (incf result2
			       (* (aref (cnode-C C) i j)
				  (aref (cnode-D C) k l)
				  (bit X i k)
				  (bit X j l))))))))))

;;;----------------------------------------------
;;;
;;;  prints the assignment matrix of tasks
;;;  to processors.  The ith element corresponds
;;;  to the assignment of task-i to the displayed processor.
;;;

(defun print-assignment (X &optional (stream t))
  (format stream "Assignment <")
  (dotimes (i (car (array-dimensions X))
	      (format stream ">~%"))
	   (do ((j 0 (1+ j))
                (nproc (cadr (array-dimensions X))))
	       ((>= j nproc))
		(if (= (bit X i j) 1)
		    (format stream "P~A," j)))))

		      

