
; ------------------------- utilitee.s  ---------------------------------

;>>>
;>>> general demon utility functions
;>>>

;> expand-cd replaces every gap with its value
(define (expand-cd cd)
  (cond ((null? cd) nil)
	((or (string? cd) 
	     (eq? '? cd)) cd)
	((atom? cd) (expand-cd (eval cd)))
	(else (cons (kar cd) (expand-sf (kdr cd))))))

;> expand-sf expects a list of slot-fillers
(define (expand-sf sf)
  (letrec ((ans nil)
	   (loop 
	    (lambda ()
		    (cond ((null? sf) ans)
			  (else (set! ans (append1 ans (pop sf)))
				(set! ans (append1 ans (expand-cd (pop sf))))
				(loop))))))
	  (loop)))

;> traverses levels of indirection until the non-atom cd structure is found
(define (deatomize cd)
  (cond ((null? cd) nil)
	((atom? cd) (deatomize (eval cd)))
	(else cd)))

;> given cd-atm = (head slot fil slot fil ...), returns head
(define (con-head cd) 
  (let ((cd-struc (deatomize cd)))
       (if cd-struc 
	   (kar cd-struc)
	   nil)))

;> given cd-atm = (head slot fil slot fil ...), returns tail
(define (con-tail cd) 
  (let ((cd-struc (deatomize cd)))
       (if cd-struc 
	   (kdr cd-struc)
	   nil)))

(define (make-con head tail) (cons head tail))

;> given slot-name and cd, returns the top-level gap-name
;> of that slot-name (NOT the gap value)
(define (find-gap slot cd)
  (and (deatomize cd)
       (let ((rest (member-alikev slot (deatomize cd))))
	    (kar (kdr rest)))))

;> given a gap (NOT the gap value) and a cd,
;> returns the top-level slot-name that has
;> gap-name associated with it
(define (find-slot gap cd)
  (and (deatomize cd)
       (let ((rest (member-alikev gap (reverse (deatomize cd)))))
	    (kar (kdr rest)))))

;>>>
;>>> utility functions used by parsing demons
;>>>

;> Binds a GAP with a CON in Working Memory and
;> marks the CON as INSIDE the MYCON associated with this GAP
;> During tracing, prints: "GAP <-- CON " when the binding occurs.
;>
(define (link mycon mygap con-found)
  (and mygap
       (let ((myslot (find-slot mygap mycon)))
	    (and (atom? con-found)
		 (put con-found 'inside (list mycon myslot mygap)))
	    (set1 mygap con-found))))

;> Sets the CON to the appropriate meaning when
;> disambiguation has occurred.
(define (con-set mycon meaning)
  (set-top-level-value! mycon (split-config mycon meaning))
  (out ^p 2 "~a = ~a~%" mycon (eval mycon)))

;> Returns non-nil if atm is a member of a class (searches recursively
;> through atm's classes for this class)
(define (class? atm class)
  (cond
   ((pair? atm) (out #t 2 "Error: CLASS? expects atomic 1st arg"))
   ((and atm 
	 (cond ((pair? class) (member-eq atm class))
	       (else (eq? atm class)))))
   (else nil)))

;> path is used to selectively examine the contents of
;> conceptual structures in Working Memory.
;> It allows the user to ignore the fact that
;> all the bindings are done via atoms.
;> e.g. given X = (INGEST ACTOR (HUMAN NAME (JOHN)
;>				       GENDER (MALE) )
;>			  OBJECT (FOOD TYPE (LOBSTER)) )
;> Then (PATH '(ACTOR NAME) X) returns (JOHN)
;>  and (PATH '(OBJECT TYPE *) X) returns LOBSTER
;>
(define (path l cd)
  (cond ((null? cd) nil)
	((atom? cd) (path l (eval cd)))
	((null? l) cd)
	((atom? l) (out ^p 2 "PATH expects 1st arg to be a list"))
	((eq? (kar l) '*) (kar cd))
	(else (path-rest l (kdr cd)))))

(define (path-rest l rv-lis)
  (let ((val (kar (kdr (member-eq (kar l) rv-lis)))))
       (cond (val (path (kdr l) val))
	     (else nil))))

;>>> working memory search utilities
;> SEARCH takes 4 parameters:
;>   1. what to look for:		a fcn-name or lambda-exp of 1 arg
;>   2. where to start looking: 	a con atom (or variable *wm*)
;>   3. when to give up looking:	a fcn-name or lambda-exp of 1 arg
;>   4. what direction to look in:	BEFore or AFTer
;> Note: a. it tries 3 on a con atom first, then tries 1.
;> 	 b. it quits if it runs out of atoms to look at
;>
(define (search test-fcn start stop-fcn dir)
  (cond ((null? dir) (set! start *wm*) (set! dir 'bef)))
  (letrec 
   ((found nil)
    (loop 
     (lambda (ptr)
	     (if (or 
		  (or (null? ptr)
		      (and stop-fcn
			   (let ((st-val (stop-fcn ptr)))
				(cond ((eq? st-val t))
				      (st-val (set! found t) ;--> for result
					      (set! ptr st-val))
				      (else nil)))))
		  (set! found (test-fcn ptr)))
		 (and found ptr)
		 (loop (get ptr dir))))))
   (loop start)))

;> Look for class throughout wm
(define (find-wm class mycon)
  (search (lambda (con)
		  (and (not (eq? con mycon))
		       (class? (con-head con) class)))
	  *wm* nil 'bef))

;>>>
;>>> general purpose miscellaneous utilities
;>>>

;> (append1 '(a b) 'c) => (a b c)
(define (append1 lst atm)
  (append lst (list atm)))

;> concatenate the symbols in lst
(define (symbol-append . lst)
  (string->symbol
   (apply string-append
	  (map (lambda (x) (symbol->string x))
	       lst))))

;> remove the first element from the list
(define (remove1st elem l)
  (if (memq? elem l)
      (delq elem l)
      l))

;> returns "diff" between lst and sublst
(define (ldiff lst sublst)
  (cond ((alikev? lst sublst) nil)
	((null? lst) (error "Second argument not sublist of first argument"))
	(else (cons (kar lst) (ldiff (kdr lst) sublst)))))

;> returns remainder of lst with first element elem (using eq?)
(define (member-eq elem lst)
  (cond ((null? lst) nil)
	((eq? elem (kar lst)) lst)
	(else (member-eq elem (kdr lst)))))

;> returns remainder of lst with first element elem (using alikev?)
(define (member-alikev elem lst)
  (cond ((null? lst) nil)
	((alikev? elem (kar lst)) lst)
	(else (member-alikev elem (kdr lst)))))

