;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package: ADVISOR -*-
;;; -----------------------------------------------------------------------
;;; File:         circumstance.l
;;; Description:  Grammatical systems of the clause dealing with circumstants
;;; Author:       Michael Elhadad
;;; Created:      19 Dec 1991
;;; Modified:     07 Jan 1992: add a realization link when mapping is not
;;;                            direct constituent to constituent.  To be
;;;                            used by the relative system and add info
;;;                            about relative-marker and question-pronoun. 
;;;               07 Jul 1992: added temporal-background.
;;;               27 Jun 1993: allow temporal-background to take
;;;                            proc/process and conjunctions as fillers.
;;; Package:      ADVISOR
;;; -----------------------------------------------------------------------

;;; (in-package "ADVISOR")


;; ============================================================
;; Circumstantial
;; ============================================================

(def-conj circumstantial
  ;; START OF TREATMENT OF CIRCUMSTANCIAL PARTICIPANTS
  ;; OPTIONAL CASES:
  ;; These cases can occur with all process-types.
  ;; They handle "circumstances".
  ;; All have the same structure.
  ;; Order in pattern should be studied with care. I have now a
  ;; standard order. 
  ;; All roles are mapped to corresponding syntactic complements from
  ;; circumstances to adjuncts.
  ;; CAREFUL that most are inherent participants in the context of a
  ;; relational clause (eg, "The meeting is at 9" - "at 9" is not an
  ;; at-time circumstance, it is an attribute of the temporal
  ;; clause.)
  ;; ***** Should list criteria for each role and work on a more
  ;; exhaustive list of roles.  Relate this list to relational
  ;; processes. 
  
  ;; Some synonyms first to make life easier
  (circum {^ circumstances})
  ;; List here all known circumstances
  (circum ((fset (at-loc to-loc from-loc on-loc in-loc
			 instrument accompaniment manner
			 temporal-background
			 effect
			 purpose reason behalf time))))
  (:! at-loc)
  (:! to-loc)
  (:! from-loc)
  (:! in-loc)
  (:! on-loc)
  (:! instrument)
  (:! accompaniment)
  (:! manner)
  #+ignore(:! ao-adverbial)
  ;; THREE CAUSE COMPLEMENTS (as by Hallyday): reason, purpose, behalf
  (:! purpose)
  (:! reason)
  (:! behalf)
  (:! effect)
  (:! time)
  (:! temporal-background)
  (:! time-relater)
  (:! cond-relater)
  ;; END OF CIRCUMSTANTIAL ROLES TREATMENT
)


;; Most circumstantial roles have a grammar of the same shape.
;; This macro builds an alt with the appropriate variables for a given
;; role.  The alt at-loc indicates what the output is for a call at the end
;; of this file.


;; binder is the binder used when filler is a clause.

(import 'fug5::make-path)

(defmacro make-role-alt (role &key prep relative (relative-embedded 'yes)
			      question (question-embedded 'yes) binder
			      (clause-embedded 'yes))
  (let* ((alt-flag (intern (format nil "~s-PRESENT" role)))
	 (adjunct-path (make-path :l (list '^ 'adjuncts role)))
	 (circum-prep-path (make-path :l (list '^ '^ '^ 'circum role 'prep)))
	 (process-prep-name (intern (format nil "~s-PREP" role)))
	 (process-prep-path (make-path
			     :l (list '^ '^ '^ 'process process-prep-name)))
	 (circum-rel-path (make-path
			   :l (list '^ '^ '^ 'circum role 'relative-marker)))
	 (process-rel-name (intern (format nil "~s-RELATIVE" role)))
	 (process-rel-path (make-path
			    :l (list '^ '^ '^ 'circum role process-rel-name)))
	 (circum-q-path (make-path
			 :l (list '^ '^ '^ 'circum role 'question-pronoun)))
	 (process-q-name (intern (format nil "~s-QUESTION" role)))
	 (process-q-path (make-path
			  :l (list '^ '^ '^ 'circum role process-q-name)))
	 (adjunct3-path (make-path 
			 :l (list '^ '^ `^ 'adjuncts role)))
	 (circum3-path (make-path 
			:l (list '^ '^ `^ 'circum role)))
	 (circum2-path (make-path
			:l (list '^ '^ 'circum role))))

    `(def-alt ,role (:demo ,(format nil "Is there a ~s role?" role))
       (((circum ((,role none)))
	 (cset ((- ,adjunct-path))))
	((circum 
	  ((,role given)
	   (,role ((relative-embedded ((alt (given ,relative-embedded))))
		   (question-embedded ((alt (given ,question-embedded))))
		   (clause-embedded ((alt (given ,clause-embedded))))))))
	 (alt ,alt-flag (:index (circum ,role cat))
	   (:demo ,(format nil "What is the cat of the ~s role?" role))
	   (((circum ((,role ((cat np)))))
	     ;; Get the preposition, relative marker and question pronoun
	     ;; from either: the semantic role if given, else from the
	     ;; process if given, else default values.
	     (alt choose-prep
		 (((circum ((,role ((prep given)))))
		   (adjuncts ((,role ((prep ,circum-prep-path))))))
		  ((process ((,process-prep-name given)))
		   (adjuncts ((,role ((prep ,process-prep-path))))))
		  ((adjuncts ((,role ((prep ((lex ,prep))))))))))
	     (alt choose-rel
		 (((circum ((,role ((relative-marker given)))))
		   (adjuncts ((,role ((relative-marker ,circum-rel-path))))))
		  ((process ((,process-rel-name given)))
		   (adjuncts ((,role ((relative-marker ,process-rel-path))))))
		  ((adjuncts ((,role ((relative-marker ((lex ,relative))))))))))
	     (alt choose-question
		 (((circum ((,role ((question-pronoun given)))))
		   (adjuncts ((,role ((question-pronoun ,circum-q-path))))))
		  ((process ((,process-q-name given)))
		   (adjuncts ((,role ((question-pronoun ,process-q-path))))))
		  ((adjuncts ((,role ((question-pronoun ((lex ,question))))))))))
	     (adjuncts ((,role ((cat pp)
				(synt-funct ,role)
				(np ,circum3-path)))))
	     ;; Keep a realization link.
	     (circum ((,role ((cset ((- realization)))
			      (realization ,adjunct3-path))))))
	    ((circum 
	      ((,role ((cat pp)
		       (synt-funct ,role)
		       (relative-embedded ((alt (given ,relative-embedded))))
		       (question-embedded ((alt (given ,question-embedded))))
		       (relative-marker ((lex ((alt (given ,relative))))))
		       (question-pronoun ((lex ((alt (given ,question))))))
		       (prep ((lex ((alt (given ,prep))))))))))
	     (adjuncts ((,role ,circum2-path))))
	    ((circum ((,role ((cat clause)))))
	     (adjuncts 
	      ((,role ,circum2-path)
	       (,role ((mood bound)
		       (synt-funct ,role)
		       (binder ((lex ((alt (given ,binder))))))
		       (relative-embedded ((alt (given ,relative-embedded))))
		       (question-embedded ((alt (given ,question-embedded))))
		       (question-prep ((alt (given ,prep))))
		       (relative-marker ((lex ((alt (given ,relative))))))
		       (question-pronoun 
			((lex ((alt (given ,question))))))))))))))))))
				



(make-role-alt at-loc 
	       :prep "at" 
	       :relative "where" 
	       :relative-embedded no
	       :question "where" 
	       :question-embedded no
	       :binder "where"
	       :clause-embedded no)
 
(make-role-alt to-loc 
	       :prep "to" 
	       :relative "where"
	       :relative-embedded no
	       :question "where"
	       :question-embedded no
	       :binder   "where")

(make-role-alt from-loc 
	       :prep "from" 
	       :question "where"
	       :binder   "where")

(make-role-alt in-loc 
	       :prep "in" 
	       :question "where"
	       :question-embedded no
	       :binder   "where")

(make-role-alt on-loc 
	       :prep "on" 
	       :question "where"
	       :question-embedded no
	       :binder   "where")

(make-role-alt instrument
	       :prep "with" )

(make-role-alt accompaniment
	       :prep "with" )

(make-role-alt reason
	       :prep "because of"
	       :binder "because"
	       :question "why"
	       :question-embedded no
	       :relative "why"
	       :relative-embedded no)


;; Circumstantials of a different forms:

(def-alt manner (:demo "Is there a manner role?")
  (((circum ((manner none)))
    (cset ((- {^ adjuncts manner}))))
   ;; manner as an adverb
   ((circum ((manner given)
	     (manner ((realization {^3 adjuncts manner})
		      (cset ((- realization)))
		      (manner-conveyed adverb)
		      (question-embedded no)))))
    (adjuncts ((manner ((cat adv)
			(synt-funct manner-adverbial)
			(semr {^3 circum manner semr})
			(question-pronoun ((lex "how")))
			(lex {^3 circum manner lex})))))
    (pattern (dots {^ adjuncts manner} process dots)))
   ;; or to a pp 
   ((circum ((manner given)
	     (manner ((realization {^3 adjuncts manner})
		      (manner-conveyed pp)
		      (cset ((- realization)))
		      (relative-embedded yes)
		      (question-embedded no)))))
    (adjuncts ((manner ((cat pp)
			(synt-funct manner-pp)
			(question-pronoun ((lex "how")))
			(np ((semr {^4 circum manner semr})))
			(prep {^3 circum manner prep})
			(prep ((lex ((alt (given "with"))))))))))
    (pattern (dots process dots {^ adjuncts manner} dots)))))


#+ignore(def-alt ao-adverbial
  ;; Clause-level AO realized by an adverb
  (((ao none))
   ((ao given)
    (adjuncts ((ao ((cat adv)
		    (semr {^3 ao semr})
		    (synt-funct ao-adverbial)
		    (question-pronoun ((lex "how")))
		    (lex {^3 ao lex})))))
    (ao ((ao-conveyed adverb)
	 (question-embedded no)
	 (cset ((- realization)))
	 (realization {^2 adjuncts ao})))
    (pattern (dots {^ adjuncts ao} process dots)))))


(def-alt purpose (:demo "Is there a purpose role?")
  ;; purpose: answer to "what for?"
  (((circum ((purpose none)))
    (cset ((- {^ adjuncts purpose} {^ adjuncts-purpose-pp} {^ adjuncts-purpose-cl}))))
   ((circum ((purpose given)))
    (alt purpose-present (:index (circum purpose cat))
      (:wait (({^ circum purpose cat} #(< lexical-cat))))
      (((circum ((purpose ((cat clause)
			   (mood infinitive)
			   (syntax ((case purposive)))
			   (question-embedded no)
			   (question-pronoun ((lex "why")))
			   (alt (((keep-in-order yes)
				  (in-order ((lex "in order") 
					     (cat conj))))
				 ((keep-in-order no)
				  (in-order none))))))))
	(adjuncts ((purpose {^2 circum purpose})
		   (purpose ((synt-funct purpose)))
		   ;; Give it a different name for the pattern unifier.
		   (purpose-cl {^ purpose})))
	(alt (((pattern (dots {^ adjuncts purpose-cl} start dots))
	       (circum ((purpose ((position front)))))
	       (adjuncts ((purpose ((punctuation ((after ","))))))))
	      ((pattern (dots {^ adjuncts purpose-cl}))
	       (circum ((purpose ((position end)))))))))
       ((circum ((purpose ((cat pp)
			   (synt-funct purpose)
			   (question-embedded no)
			   (question-pronoun ((lex "why")))
			   (prep ((lex ((alt (given "for"))))))))))
	(adjuncts ((purpose-pp {^2 circum purpose}))))
       ((circum ((purpose ((cat np)
			   (realization {^3 adjuncts purpose})
			   (cset ((- realization)))
			   (question-embedded ((alt (given no))))
			   (relative-embedded ((alt (given yes))))))))
	(alt choose-prep (:demo "Choose a preposition for a PP purpose")
	  (((circum ((purpose ((prep given)))))
	    (adjuncts ((purpose ((prep {^3 circum purpose prep}))))))
	   ((process ((purpose-prep given)))
	    (adjuncts ((purpose ((prep {^3 process purpose-prep}))))))
	   ((adjuncts ((purpose ((prep ((lex "for"))))))))))
	(alt choose-rel (:demo "Choose a relative pronoun for a PP purpose")
	  (((circum ((purpose ((relative-marker given)))))
	    (adjuncts ((purpose ((relative-marker {^3 circum purpose prep}))))))
	   ((process ((purpose-relative given)))
	    (adjuncts ((purpose ((relative-marker {^3 process purpose-relative}))))))
	   ((adjuncts ((purpose ((prep ((lex "why")))))))
	    (circum ((purpose ((relative-embedded no))))))
	   ((circum ((purpose ((relative-embedded yes))))))))
	(alt choose-quest (:demo "Choose a question pronoun for a PP purpose")
	  (((circum ((purpose ((question-pronoun given)))))
	    (adjuncts ((purpose ((question-pronoun 
				  {^3 circum purpose question-pronoun}))))))
	   ((process ((purpose-question given)))
	    (adjuncts ((purpose ((question-pronoun 
				  {^3 process purpose-question}))))))
	   ((adjuncts ((purpose ((question-pronoun ((lex "why"))))))))))
	(adjuncts ((purpose-pp {^ purpose})         ;; for the patterns
		   (purpose ((synt-funct purpose)
			     (cat pp)
			     (np {^3 circum purpose})))))))))))


(def-alt behalf (:demo "Is there a behalf role?")
  ;; behalf: answer to "who for?"
  (((circum ((behalf none)))
    (cset ((- {^ adjuncts behalf}))))
   ;; behalf as a for-to infinitive clause
   ;; Note: subject must be given and is actually the behalf
   ;; "You have to do it for John to read" (Winograd p.472)
   ((circum ((behalf given)))
    (alt behalf-present (:index (circum behalf cat))
      (:wait (({^ circum behalf cat} #(< lexical-cat))))
      (((circum ((behalf ((cat clause)))))
	(adjuncts ((behalf {^ ^ circum behalf})
		   (behalf ((synt-func behalf)
			    (question-pronoun ((lex "what")))
			    (question-embedded yes)
			    (question-prep ((lex "for")))
			    (cat clause)
			    (mood infinitive))))))
       ((circum ((behalf ((cat pp)
			  (synt-funct behalf)
			  (prep ((lex ((alt (given "for"))))))
			  (question-embedded yes)
			  (relative-embedded yes)))))
	(adjuncts ((behalf {^2 circum behalf}))))
       ((circum ((behalf ((cat np)
			  (relative-embedded ((alt (given yes))))
			  (question-embedded ((alt (given yes))))
			  (cset ((- realization)))
			  (realization {^3 adjuncts behalf})))))
	(adjuncts ((behalf ((synt-func behalf)))))
	(alt choose-prep
	    (((circum ((behalf ((prep ((lex given)))))))
	      (adjuncts ((behalf ((prep {^3 circum behalf prep}))))))
	     ((process ((behalf-prep given)))
	      (adjuncts ((behalf ((prep {^3 process behalf-prep}))))))
	     ((adjuncts ((behalf ((prep ((lex "for"))))))))))
	(alt choose-rel
	    (((circum ((behalf ((relative-marker given)))))
	      (adjuncts ((behalf ((relative-marker 
				   {^3 circum behalf relative-marker}))))))
	     ((process ((behalf-relative given)))
	      (adjuncts ((behalf ((relative-marker {^3 process behalf-relative}))))))
	     ((adjuncts ((behalf ((relative-marker nil))))))))
	(alt choose-question
	    (((circum ((behalf ((question-pronoun given)))))
	      (adjuncts ((behalf ((question-pronoun 
				   {^3 circum behalf question-pronoun}))))))
	     ((process ((behalf-question given)))
	      (adjuncts ((behalf ((question-pronoun {^3 process behalf-question}))))))
	     ((adjuncts ((behalf ((question-pronoun nil))))))))
	(adjuncts ((behalf ((cat pp)
			    (np {^3 circum behalf})))))))))))



;; Hack to add an effect circum expecting all syntactic specification in
;; the input.
(def-alt effect (:demo "Is there an effect role?")
  (((circum ((effect none)))
    (cset ((- {^ adjuncts effect}))))
   ((circum ((effect given)))
    (adjuncts ((effect ((synt-funct effect)
			(punctuation ((before ",")))))
	       (effect {^2 circum effect}))))))


(def-alt time (:demo "Is there a time role?")
  ;; All the possible time roles under one time plus a time-type
  ;; feature specializing it.  
  ;; Can be a (cat list) for agglutination of time complements but then
  ;; the list should contain complete syntactic fds.
  ;; The list of time-type is given in Quirk 11.27
  ;; Ex: in the afternoon, later, when I have time, last Thursday
  ;; ***** Should implement the semantics of time-type with tpattern.
  (((circum ((time none)))
    (cset ((- {^ adjuncts time}))))
   ((circum ((time given)))
    (adjuncts ((time ((synt-funct time)))))
    (alt time-adjunct (:index (circum time cat))
      (((circum ((time ((cat adv)
			(relative-embedded no)
			(question-embedded no)
			(question-pronoun ((lex "when")))
			(synt-funct time)))))
	(adjuncts ((time {^ ^ circum time}))))
       ((circum ((time ((cat clause)
			(synt-funct time)
			(relative-embedded no)
			(question-embedded no)
			(question-pronoun ((lex "when")))
			(mood ((alt (bound 
				     #(under infinitive) 
				     #(under present-participle)))))
			(binder ((lex {^ ^ time-type})))
			(time-type 
			 ((alt ("after" "as" "before" "once" "since"
				"until" "when" "while" "now that"))))))))
	(adjuncts ((time {^ ^ circum time}))))
       ((circum ((time ((cat list)))))
	(adjuncts ((time {^ ^ circum time}))))
       ((circum ((time ((cat pp)
			(synt-funct time)
			(relative-embedded yes)
			(question-embedded no)
			(prep ((opt ((lex "at")))))))))
	(adjuncts ((time {^2 circum time}))))
       ((circum ((time ((cat np)
			(cset ((- realization)))
			(realization {^3 adjuncts time})
			(relative-embedded yes)
			(question-embedded no)
			(time-type 
			 ((alt ("at" "on" "in" "for" "before" "after"
				"since" "until"))))))))
	(adjuncts ((time ((cat pp)
			  (synt-funct time)
			  (question-pronoun ((lex "when")))
			  (prep ((lex {^4 circum time time-type})))
			  (np {^3 circum  time}))))))))
    (position-time {^ circum time position})
    (alt position-time (:index position-time)
      (((position-time end)
	(pattern (dots end-partic dots {^ adjuncts purpose-pp} 
		       {^ adjuncts time} dots))) 
       ((position-time front)
	(adjuncts ((time ((punctuation ((after ",")))))))
	(pattern (dots {^ adjuncts time} dots start dots))))))))


(def-alt temporal-background (:demo "Is there a temporal background?")
  ;; A when subordinate clause 
  ;; particular circumstance expressing the conditions under which the
  ;; action realized by the main clause can occur.
  (((circum ((temporal-background none))))
   ((circum ((temporal-background given)))
    (adjuncts 
     ((when ((cat phrase)
	     (pattern (connective clause))
	     (connective ((cat conj) (lex "when")))
	     (clause
	      ((cat clause)
	       (mood {^4 circum temporal-background mood})
	       (complex {^4 circum temporal-background complex})
	       (common {^4 circum temporal-background common})
	       (distinct {^4 circum temporal-background distinct})
	       (alt
		(((mood declarative))
		 ((mood present-participle)
		  (synt-roles
		   ((subject ((gap yes)))
		    (subject
		     ((semantics {^6 synt-roles subject semantics}))))))
		 ((mood past-participle)
		  (subject 
		   ((semantics {^6 synt-roles subject semantics}))))))
	       (process {^4 circum temporal-background process})
	       (process {^4 circum temporal-background proc})
	       (partic {^4 circum temporal-background partic})
	       (circum {^4 circum temporal-background circum})))))))
    (position-when {^ circum temporal-background position})
    (alt position-when (:index position-when)
      (((position-when end)
	(pattern (dots end-circum dots {^ adjuncts when} dots)))
       ((position-when front)
	(adjuncts ((when ((punctuation ((after ",")))))))
	(pattern (dots {^ adjuncts when} dots start dots))))))))


  

(def-alt time-relater (:demo "Is there a time-relater?")
  ;; time-relater are "first", "next", "then" occuring first in the
  ;; clause.  Note that they are not movable when used in this relater
  ;; sense.  So they are not just simply adverbials.
  ;; ***** To re-do when complex clauses are implemented.
  (((time-relater none))
   ((time-relater given)
    (time-relater ((cat adv)
		   (punctuation ((after ",")))))
    (pattern (time-relater dots)))))


(def-alt cond-relater (:demo "Is there a cond-relater?")
  ;; cond-relater is "if", "then" or "else" - this is used until we
  ;; come up with a better treatment of complex clauses. *****
  (((cond-relater none))
   ((cond-relater given)
    (cond-relater ((cat adv)))
    (pattern (time-relater cond-relater dots)))))


#|
;; ------------------------------------------------------------
;; Example of expansion of macro make-role-alt:
;; ------------------------------------------------------------

(def-alt at-loc (:demo "Is there an at-loc role?")
  (((circum ((at-loc none)))
    (cset ((- {^ adjuncts at-loc}))))
   ((circum ((at-loc given)))
    (alt at-loc-present (:index (circum at-loc cat))
      (:demo "What cat is the at-loc role?")
      ;; Map np to a pp and add appropriate prep
      ;; All others have a direct mapping.
      (((circum ((at-loc ((cat np)))))
	;; get prep from role if given, otw from verb, otw default.
	(alt choose-prep
	    (((circum ((at-loc ((prep given)))))
	      (adjuncts 
	       ((at-loc ((prep ((lex {^4 circum at-loc prep}))))))))
	     ((process ((at-loc-prep given)))
	      (adjunts 
	       ((at-loc ((prep ((lex {^4 process at-loc-prep}))))))))
	     ((adjuncts ((at-loc ((prep ((lex "at"))))))))))
	(alt choose-rel
	    (((circum ((at-loc ((relative-marker given)))))
	      (adjuncts 
	       ((at-loc ((relative-marker 
			  ((lex {^4 circum at-loc relative-marker}))))))))
	     ((process ((at-loc-relative given)))
	      (adjunts 
	       ((at-loc ((relative-marker
			  ((lex {^4 process at-loc-relative}))))))))
	     ((adjuncts ((at-loc ((relative-marker ((lex "where"))))))))))
	(alt choose-quest
	    (((circum ((at-loc ((question-pronoun given)))))
	      (adjuncts 
		((at-loc ((question-pronoun 
			   ((lex {^4 circum at-loc question-pronoun}))))))))
	     ((process ((at-loc-question given)))
	      (adjunts 
	       ((at-loc ((question-pronoun ((lex {^4 process at-loc-question}))))))))
	      ((adjuncts ((at-loc ((question-pronoun ((lex "where"))))))))))
	(adjuncts ((at-loc ((cat pp)
			    (np {^3 circum at-loc})))))
	(circum ((at-loc ((realization {^3 adjuncts at-loc}))))))
       ((circum ((at-loc ((cat pp)))))
	(adjuncts ((at-loc {^2 circum at-loc}))))
       ((circum ((at-loc ((cat clause)))))
	(adjuncts ((at-loc {^2 circum at-loc})
		   (at-loc ((mood bound)))
		   (alt (((binder ((lex given))))
			 ((binder ((lex "where"))))))))))))))

|#

;; ============================================================
(provide "circumstance")
;; ============================================================
