# include <stdio.h>			/* tprep.c (rev3.7) */
# include <ctype.h>
# include <signal.h>

usage()		/* print usage and synopsis of options */
{
	puts("Usage: tprep [-y -tpu] filename(s)\t\t(rev3.7)");
	puts("-y: say yes and suppress interactive prompting");
	puts("-t: trim lines, removing trailing blanks and tabs");
	puts("-p: pad, inserting blank at beginning of each line");
	puts("-u: unpad, deleting blank at beginning of each line");
	puts("Tprep rewrites files back on top of themselves.");
	exit(1);
}

char *tempfile;		/* storage file for modified text */
int askflag = 1;	/* ask user's permission (default on) */
int tflag = 0;		/* toggle option for trimming */
int pflag = 0;		/* toggle option for padding */
int uflag = 0;		/* toggle option for unpadding */

main(argc, argv)	/* insert or delete leading blanks */
int argc;
char *argv[];
{
	FILE *fp, *fopen();
	char *s, *mktemp();
	int i = 1;

	if (argc == 1)
		usage();

	for (i = 1; i < argc && *argv[i] == '-'; i++)
		for (s = argv[i]+1; *s != NULL; s++)
			if (*s == 'y')
				askflag = 0;
			else if (*s == 't')
				tflag = 1;
			else if (*s == 'p')
				pflag = 1;
			else if (*s == 'u')
				uflag = 1;
			else
			{
				fprintf(stderr,
				"Illegal tprep option: -%c\n", *s);
				exit(1);
			}
	if (tflag + pflag + uflag > 1)
	{
		fprintf(stderr, "Using incompatible tprep options\n");
		exit(1);
	}
	if (askflag)
		permission();
	if (!tflag && !pflag && !uflag)
		options();

	signal(SIGINT, SIG_IGN);
	tempfile = "/tmp/PrepXXXXX";
	mktemp(tempfile);

	for (; i < argc; i++)
	{
		if ((fp = fopen(argv[i], "r")) != NULL)
		{
			fprintf(stderr,
			"Rewriting the file: %s\n", argv[i]);
			if (tflag)
				trim(fp);
			if (pflag)
				pad(fp);
			if (uflag)
				unpad(fp);
			if (freopen(argv[i], "w", fp) == NULL)
			{
				fprintf(stderr,
				"%s: Write permission denied\n", argv[i]);
				unlink(tempfile);
				exit(1);
			}
			overwrite(fp);
			fclose(fp);
		}
		else  /* attempt to open file failed */
		{
			fprintf(stderr,
			"Tprep cannot access the file: %s\n", argv[i]);
			unlink(tempfile);
			exit(1);
		}
	}
	unlink(tempfile);
	exit(0);
}

permission()		/* make sure user agrees to rewrite files */
{
	FILE *term, *fopen();
	char ans[512];

	term = fopen("/dev/tty", "r");
	fprintf(stderr, "Do you know tprep rewrites files in command line?\n");
	fprintf(stderr, "Type \"y\" if this is OK with you: ");
	fgets(ans, 512, term);
	if (*ans == 'y' || *ans == 'Y')
		putc('\n', stderr);
	else
	{
		fprintf(stderr, "If it makes you feel safer, ");
		fprintf(stderr, "you could copy files to /usr/tmp.\n");
		exit(1);
	}
	return;
}

options()		/* ask user whether to trim, pad or unpad */
{
	FILE *term, *fopen();
	char ans[512];
	int done = 0;

	term = fopen("/dev/tty", "r");
	while (!done)
	{
		fprintf(stderr, "Do you want to trim, pad, or unpad? ");
		fgets(ans, 512, term);
		if (*ans == 't' || *ans == 'T')
		{
			tflag = 1;
			done = 1;
		}
		else if (*ans == 'p' || *ans == 'P')
		{
			pflag = 1;
			done = 1;
		}
		else if (*ans == 'u' || *ans == 'U')
		{
			uflag = 1;
			done = 1;
		}
		else if (*ans == 'q' || *ans == 'Q')
			exit(1);
		else
			fprintf(stderr,
			"\tJust answer \"t\", \"p\" or \"u\" (\"q\" exits).\n");
	}
	putc('\n', stderr);
	return;
}

trim(fp)		/* trim off blanks and send to tempfile */
FILE *fp;
{
	FILE *tfp, *fopen();
	char line[512];
	long count = 0;
	int n;

	tfp = fopen(tempfile, "w");
	while (fgets(line, 512, fp))
	{
		n = strlen(line);
		while (--n >= 0)
		{
			if (!isspace(line[n]))
				break;
			count++;
		}
		line[++n] = '\n';
		line[++n] = NULL;
		fputs(line, tfp);
		count--;		/* uncount newline */
	}
	fclose(tfp);
	fprintf(stderr, "\t%ld blanks and tabs removed\n", count);
}

pad(fp)			/* pad line with blank and send to tempfile */
FILE *fp;
{
	FILE *tfp, *fopen();
	char line[512];
	int count = 0;

	tfp = fopen(tempfile, "w");
	while (fgets(line, 512, fp))
	{
		fprintf(tfp, " %s", line);
		count++;
	}
	fclose(tfp);
	fprintf(stderr, "\t%d lines padded\n", count);
}

unpad(fp)		/* strip leading blank and send to tempfile */
FILE *fp;
{
	FILE *tfp, *fopen();
	char line[512], copy[512];
	int i, j, count = 0;

	tfp = fopen(tempfile, "w");
	while (fgets(line, 512, fp))
	{
		i = 0;
		if (line[i] == ' ')
		{
			i++;
			count++;
		}
		for (j = 0; line[i] != NULL; i++, j++)
			copy[j] = line[i];
		copy[j] = NULL;
		fputs(copy, tfp);
	}
	fclose(tfp);
	fprintf(stderr, "\t%d lines unpadded\n", count);
}

overwrite(fp)		/* write tempfile back over original file */
FILE *fp;
{
	FILE *tfp, *fopen();
	char line[512];

	tfp = fopen(tempfile, "r");
	while (fgets(line, 512, tfp))
		fputs(line, fp);
	fclose(tfp);
}
