/*  $Id: rnscalc.c,v 1.18 1991/10/10 16:44:20 richard Exp richard $  */

/*  Part of RNS -- Recurrent Network Simulator
 *
 *     by R. Kooijman
 *        T.U. Delft
 *        Faculteit Elektrotechniek
 *        Vakgroep Computerarchitectuur
 *        Sectie Neurale Netwerken
 */


/*  $Log: rnscalc.c,v $
 * Revision 1.18  1991/10/10  16:44:20  richard
 * Switched to UNIX RCS
 *
 * Revision 1.17  91/05/09  14:38:13  ROOT_DOS
 * Added USERERR learn property
 * Changed accuracy of neuron output values, EPS
 * Changed don't care and undefined values
 *
 * Revision 1.16  91/03/24  15:58:06  ROOT_DOS
 * Added RANDOM learning property
 * Fixed Differential Hebb learning algorithm
 *
 * Revision 1.15  91/02/26  21:48:49  ROOT_DOS
 * Added comments
 * Added support for SEED command
 *
 * Revision 1.14  91/02/17  14:27:03  ROOT_DOS
 * Some speed-up changes
 *
 * Revision 1.13  91/02/05  21:51:08  ROOT_DOS
 * Added lim_step constraint by adding STEP command
 *
 * Revision 1.12  91/02/05  01:27:16  ROOT_DOS
 * Made version that can be compiled to single or double arithmetic simulator
 *
 * Revision 1.11  91/02/01  23:27:18  ROOT_DOS
 * Added ->constraint==NULL checks
 *
 * Revision 1.10  91/01/30  21:43:39  ROOT_DOS
 * Forget ANSI C and make traditional C / Turbo C dual mode
 *
 * Revision 1.9  91/01/28  22:19:51  ROOT_DOS
 * Changed handling of bias of neurons with 0/1 (!=INPUT) inputs
 *
 * Revision 1.8  91/01/27  21:01:06  ROOT_DOS
 * Changes to make code strict ANSI C compatible
 *
 * Revision 1.7  91/01/22  16:55:05  ROOT_DOS
 * Made ITER command conditional compilable
 *
 * Revision 1.6  91/01/19  22:23:35  ROOT_DOS
 * Added support for real-valued patterns, relative learning rates
 *
 * Revision 1.4  91/01/03  14:43:58  ROOT_DOS
 * Added simulated annealing learning algorithme
 * Moved set_pattern_result() to rnstests.c
 *
 * Revision 1.3  90/12/19  21:27:08  ROOT_DOS
 * Changed max_step initial value
 *
 * Revision 1.2  90/12/08  03:50:25  ROOT_DOS
 * Some minor cosmetic changes
 *
 * Revision 1.1  90/12/08  03:12:47  ROOT_DOS
 * Initial revision
 *   */



/*  functies voor het doen van berekeningen op het neurale netwerk  */


#define EXTERN
#include "rnsdefs.h"
#include "rnsfuncs.h"


#ifdef TURBOC
#include <conio.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#else
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <values.h>
#endif



#ifdef USEPROTOS
int process_net(void)
#else
int process_net()
#endif
{
int rc = OK;
int pattern, pair;
boolean more;

   if ((rc=init_screen())==OK)
   {
      init_time();                     /*  initialiseer tijdmetingen  */

      init_net();

      init_weights();

      do
      {
         reset_dweights();

         if (learn & RANDOM)
         {
            pattern=Random(nrpatterns);

            if (!DESELECTED(pattern))
            {
               reset_pattern_error(pattern);

               for (pair=0; pair<PAIRS(pattern); pair++)
               {
#ifdef DEBUG
                  printf("pattern: %d   pair: %d\n", pattern, pair);
#endif

                  reset_activations(pattern, pair, OUT);
                  while (calc_output(pattern, pair)==FALSE)
                     ;

                  calc_deriv(pattern, pair);

                  calc_pattern_error(pattern, pair);

                  refresh_display(pattern, pair, FALSE);

                  reset_deltasums(pattern, pair, DELTA);
                  while (calc_delta()==FALSE)
                     ;

                  adjust_lrates(pattern);

                  calc_dweights(pattern);

                  calc_delay();
               }
            }
            else
               verbose_display(pattern, 0, FALSE);
         }
         else
            for (pattern=0; pattern<nrpatterns; pattern++)
            {
               if (!DESELECTED(pattern))
               {
                  reset_pattern_error(pattern);

                  for (pair=0; pair<PAIRS(pattern); pair++)
                  {
#ifdef DEBUG
                     printf("pattern: %d   pair: %d\n", pattern, pair);
#endif

                     reset_activations(pattern, pair, OUT);
                     while (calc_output(pattern, pair)==FALSE)
                        ;

                     calc_deriv(pattern, pair);

                     calc_pattern_error(pattern, pair);

                     refresh_display(pattern, pair, FALSE);

                     reset_deltasums(pattern, pair, DELTA);
                     while (calc_delta()==FALSE)
                        ;

                     adjust_lrates(pattern);

                     calc_dweights(pattern);

                     calc_delay();
                  }
               }
               else
                  verbose_display(pattern, 0, FALSE);
            }

         rc=log_info(FALSE);

         calc_error();

         if ((more=step_check()))      /*  pas de gewichten aan als  */
            update_weights();          /*  we nog verder moeten      */

      } while (rc==OK && more);
   }

   refresh_display(0, 0, TRUE);

   rc=log_info(TRUE);

   if (rc==OK)
      rc=close_screen();               /*  gebruik rc  */
   else
      close_screen();                  /*  bewaar fout in rc  */

   update_time();                      /*  doe nieuwe tijdmetingen  */

   return(rc);
}


#ifdef USEPROTOS
void init_net(void)
#else
void init_net()
#endif
{
   if (seed==0)
      Srand(gseed=(unsigned) time(NULL));
   else                                /*  geef random-generator  */
      Srand(seed);                     /*  een startwaarde        */

   complexity=nrneurons*nrpatterns;

   iter=0;

   if (max_iter==0)
      max_iter=nrneurons*2;

   step=0;
   con_step=0;

   if (learn & ANNEAL)
      max_step=500*complexity;         /*  meer stappen toegestaan    */
   else                                /*  bij 'simulated annealing'  */
      max_step=100*complexity;

   max_error=0.5*1.2*nrpatterns*nroutputs/sqr(MAXMINOUT);
   min_error=max_error;

   total_error=error;                  /*  zorg voor !(total_error<error)  */
}


#ifdef USEPROTOS
void init_weights(void)
#else
void init_weights()
#endif
{
int weight;
Neuronptr neuronptr;
Weightptr weightptr;
FLOAT value;

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      for (weight=0; weight<neuronptr->nrweights; weight++)
      {
         value=(FLOAT)(Random(10000)-5000)/10000;

         weightptr=neuronptr->weights+weight;

         weightptr->epsilon=epsilon;
                         /*  zet lokale epsilon op globale waarde  */
         weightptr->addrate=addrate;
                         /*  zet lokale addrate op globale waarde  */
         weightptr->mulrate=mulrate;
                         /*  zet lokale mulrate op globale waarde  */

         if (weightptr->constraint==NULL)
         {
            if (weightptr->value==0)
               weightptr->value=value;
         }
         else
         {
            if (strchr(weightptr->constraint, '<')!=NULL)
               value=weightptr->orgvalue+value-0.5;
            else
            if (strchr(weightptr->constraint, '>')!=NULL)
               value=weightptr->orgvalue+value+0.5;
            else
            if (strchr(weightptr->constraint, '=')!=NULL ||
                strchr(weightptr->constraint, '!')!=NULL)
               value=0;
            else
            if (weightptr->value==0)
               value=weightptr->orgvalue+value;
            else
               value=0;

            if (value!=0)
               weightptr->value=value;
         }
      }

      neuronptr->epsilon=epsilon;
                                /*  zet lokale epsilon op globale waarde  */
      neuronptr->addrate=addrate;
                                /*  zet lokale addrate op globale waarde  */
      neuronptr->mulrate=mulrate;
                                /*  zet lokale mulrate op globale waarde  */
   }
}


#ifdef USEPROTOS
void reset_activations(int pattern, int pair, char mode)
#else
void reset_activations(pattern, pair, mode)
        int  pattern;
        int  pair;
        char mode;
#endif
{
int number;
Neuronptr neuronptr;
Weightptr weightptr;
int weight;
FLOAT dinput, doutput;

   for (number=0; number<nrinputs; number++)
   {
      neuronptr=INPUTNEURON(number);
      if ((dinput=pattern_val(pattern, pair, number, INPUT))==DONTCARE)
         dinput=(mode & RESULT) ? UNDEFINED
                   : dontcare*((FLOAT)(Random(20000)-10000)/20000*MAXMINOUT)+UNDEFINED;
                                /*  dinput = dontcare*(-1 <--> 1)*MAXMIN/2+UNDEFINED  */

      neuronptr->activation=dinput;
      for (weight=0; weight<neuronptr->nrweights; weight++)
      {
         weightptr=neuronptr->weights+weight;
         weightptr->partact=0;
      }

      neuronptr->output=neuron_output(neuronptr);
      neuronptr->changed.output=TRUE;
   }

   if ((learn & FORCED) && !(mode & RESULT))
      for (number=0; number<nroutputs; number++)
      {
         neuronptr=OUTPUTNEURON(number);
         if ((doutput=pattern_val(pattern, pair, number, OUTPUT))==DONTCARE)
            doutput=(mode & RESULT) ? UNDEFINED
                       : dontcare*((FLOAT)(Random(20000)-10000)/20000*MAXMINOUT)+UNDEFINED;
                                /*  doutput = dontcare*((-1 <--> 1)*MAXMIN/2+UNDEFINED)  */

         neuronptr->output=doutput;
         neuronptr->changed.output=TRUE;

         neuronptr->activation=neuron_inverse(neuronptr);
         for (weight=0; weight<neuronptr->nrweights; weight++)
         {
            weightptr=neuronptr->weights+weight;
            weightptr->partact=0;
         }
      }


   /*  reset alle neuronen waarbij dit nog niet is gedaan: activaties op 0         */
   /*  om rekenfouten te resetten en geef aan dat outputs veranderd zijn om        */
   /*  partiele activaties geforceerd te updaten na mogelijke gewichtswijzigingen  */
   if (pattern==0 && pair==0)
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      {
         if (neuronptr->type==HIDDEN ||
             (neuronptr->type==OUTPUT && (!(learn & FORCED) || (mode & RESULT))))
         {
            /*  reset (partiele) activaties  */
            neuronptr->activation=0;
            for (weight=0; weight<neuronptr->nrweights; weight++)
            {
               weightptr=neuronptr->weights+weight;
               weightptr->partact=0;
            }

            /*  reset output en forceer update  */
            neuronptr->output=0;
            neuronptr->changed.output=TRUE;
         }
      }

   if (pair==0)
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      {
         neuronptr->prevoutput=0;
         neuronptr->changed.prev=TRUE;
      }

   iter=0;
}


#ifdef USEPROTOS
boolean calc_output(int pattern, int pair)
#else
boolean calc_output(pattern, pair)
           int pattern;
           int pair;
#endif
{
boolean convergence = TRUE;
Neuronptr neuronptr;
Weightptr weightptr;
int weight;
FLOAT value, activation;

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      for (weight=0; weight<neuronptr->nrweights; weight++)
      {
         weightptr=neuronptr->weights+weight;

         if (weightptr->delay==TRUE)
         {
            if (weightptr->from->changed.prev ||
                (weightptr->from2!=NULL && weightptr->from2->changed.prev))
            {
               neuronptr->activation-=weightptr->partact;

               value=weightptr->from->prevoutput;
               if (weightptr->from2==NULL)
                  activation=weightptr->value*value;
               else
                  activation=weightptr->value*value*weightptr->from2->prevoutput;

               weightptr->partact=activation;

               neuronptr->activation+=activation;
               neuronptr->changed.extra=TRUE;
                                /*  geef activation verandering aan  */
            }
         }
         else
         {
            if (weightptr->from->changed.output ||
                (weightptr->from2!=NULL && weightptr->from2->changed.output))
            {
               neuronptr->activation-=weightptr->partact;

               value=weightptr->from->output;
               if (weightptr->from2==NULL)
                  activation=weightptr->value*value;
               else
                  activation=weightptr->value*value*weightptr->from2->output;

               weightptr->partact=activation;

               neuronptr->activation+=activation;
               neuronptr->changed.extra=TRUE;
                                /*  geef activation verandering aan  */
            }
         }
      }

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      if (neuronptr->changed.extra)
      {
         value=neuron_output(neuronptr);

         if (fabs((neuronptr->output-value)/
                  (neuronptr->output!=0 ? neuronptr->output : 1))>EPS)
         {
            neuronptr->output=value;
            neuronptr->changed.output=TRUE;
            convergence=FALSE;
         }
         else
            neuronptr->changed.output=FALSE;

         neuronptr->changed.extra=FALSE;
      }
      else
         neuronptr->changed.output=FALSE;
   }

   if (++iter>=max_iter)
      convergence=TRUE;

   return(convergence);
}


#ifdef USEPROTOS
void calc_deriv(int pattern, int pair)
#else
void calc_deriv(pattern, pair)
        int pattern;
        int pair;
#endif
{
Neuronptr neuronptr;
FLOAT doutput;

   if (learn & DHEBB)
   {
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      {
         if (neuronptr->type==OUTPUT && !(learn & SELF) &&
             (doutput=pattern_val(pattern, pair, neuronptr->ionum, OUTPUT))!=DONTCARE)
            neuronptr->derivation=doutput-neuronptr->output;
         else
            neuronptr->derivation=neuronptr->output-neuronptr->prevoutput;
      }
   }
}


#ifdef USEPROTOS
void calc_error(void)
#else
void calc_error()
#endif
{
int pattern;
FLOAT tmp_error, max_pat_error;

   max_pat_error=0;

   for (tmp_error=0, pattern=0; pattern<nrpatterns; pattern++)
   {
      if (GETPATTERN(pattern)->count>0)
         GETPATTERN(pattern)->count--;
      else
         if (pselect_error>0 && GETPATTERN(pattern)->error<pselect_error)
            GETPATTERN(pattern)->count=pselect_steps;

#ifdef DEBUG
      printf("pattern: %d   error: %7.4f   select_error: %7.4f   count: %d\n",
             pattern, GETPATTERN(pattern)->error, pselect_error, GETPATTERN(pattern)->count);
#endif

      tmp_error+=GETPATTERN(pattern)->error;

      if (pselect_adapt==TRUE)
         max_pat_error=max(max_pat_error, GETPATTERN(pattern)->error);
   }

   /*  Geef aan of oplossing direct geaccepteerd moet worden of niet  */
   /*  Heeft alleen effect bij 'simulated annealing' leeralgoritme    */
   if (tmp_error<=total_error)
      accept=TRUE;
   else
      accept=FALSE;

   total_error=tmp_error;

   if (pselect_adapt==TRUE)
      pselect_error=max_pat_error/4;
}


#ifdef USEPROTOS
void reset_pattern_error(int pattern)
#else
void reset_pattern_error(pattern)
        int pattern;
#endif
{
   GETPATTERN(pattern)->error=0;
}


#ifdef USEPROTOS
void calc_pattern_error(int pattern, int pair)
#else
void calc_pattern_error(pattern, pair)
        int pattern;
        int pair;
#endif
{
int output;
FLOAT error, doutput, value;

   for (error=0, output=0; output<nroutputs; output++)
   {
      if ((doutput=pattern_val(pattern, pair, output, OUTPUT))!=DONTCARE)
      {
         value=OUTPUTNEURON(output)->output;

         if (learn & CLIP424)
            value=value>0.2*MAXOUTPUT ? MAXOUTPUT
                     : (value<0.2*MINOUTPUT ? MINOUTPUT : UNDEFINED);
         else
            if (learn & CLIP262)
               value=value>0.6*MAXOUTPUT ? MAXOUTPUT
                        : (value<0.6*MINOUTPUT ? MINOUTPUT : UNDEFINED);

         if (learn & USERERR)
            error+=(*errorfp)(doutput, value, erroropt);
         else
          if (learn & SELF)
             error+=0.5*sqr(1-(value-MINOUTPUT)/MAXMINOUT);
          else
             error+=0.5*sqr((doutput-value)/MAXMINOUT);
      }
   }

   GETPATTERN(pattern)->error+=GETPATTERN(pattern)->relative*error;
}


#ifdef USEPROTOS
void reset_deltasums(int pattern, int pair, char mode)
#else
void reset_deltasums(pattern, pair, mode)
        int pattern;
        int pair;
        char mode;
#endif
{
int number;
Neuronptr neuronptr;
Weightptr weightptr;
int weight;
FLOAT doutput;

   if (learn & FBPR)
   {
      for (number=0; number<nroutputs; number++)
      {
         neuronptr=OUTPUTNEURON(number);

         if (learn & USERERR)
         {
            doutput=pattern_val(pattern, pair, number, OUTPUT);
            neuronptr->deltasum=(*derrorfp)(doutput, neuronptr->output, derroropt);
         }
         else
          if (learn & SELF)
          {
             if (neuronptr->output<UNDEFINED)
                neuronptr->deltasum=MINOUTPUT-neuronptr->output;
             else
                neuronptr->deltasum=MAXOUTPUT-neuronptr->output;
          }
          else
          {
             if ((doutput=pattern_val(pattern, pair, number, OUTPUT))!=DONTCARE)
                neuronptr->deltasum=doutput-neuronptr->output;
             else
                neuronptr->deltasum=0;
          }

         neuronptr->changed.extra=TRUE;       /*  geef deltasum reset aan  */

         neuronptr->delta=
            neuronptr->deltasum*neuron_derivative(neuronptr);
         neuronptr->changed.delta=TRUE;
      }


      /*  reset alle neuronen waarbij dit nog niet is gedaan: deltasums op 0 om  */
      /*  rekenfouten te resetten en geef aan dat deltas veranderd zijn om       */
      /*  deltasums geforceerd te updaten na mogelijke gewichtswijzigingen       */
      if (pattern==0 && pair==0)
         for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
         {
            if (neuronptr->type!=OUTPUT)
            {
               /*  reset deltasum  */
               neuronptr->deltasum=0;
               neuronptr->changed.extra=TRUE; /*  geef deltasum reset aan  */

               /*  reset delta en forceer update  */
               neuronptr->delta=0;
               neuronptr->changed.delta=TRUE;
            }
         }

      /*  zoek deltasum resets door te testen op changed.extra  */
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
         for (weight=0; weight<neuronptr->nrweights; weight++)
         {
            weightptr=neuronptr->weights+weight;
            if (weightptr->from->changed.extra)
               weightptr->partdel=0;
            if (weightptr->from2!=NULL && weightptr->from2->changed.extra)
               weightptr->partdel2=0;
         }

      /*  reset alle changed.extra variabelen  */
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
         neuronptr->changed.extra=FALSE;
   }
   else
    if (learn & HEBB)
    {
       for (number=0; number<nroutputs; number++)
       {
          neuronptr=OUTPUTNEURON(number);
          if (!(learn & SELF) &&
              (doutput=pattern_val(pattern, pair, number, OUTPUT))!=DONTCARE)
             neuronptr->delta=doutput;
          else
             neuronptr->delta=neuronptr->output;
       }

       for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
          if (neuronptr->type!=OUTPUT)
             neuronptr->delta=neuronptr->output;
    }

   if (adjust_net!=ADJUST_ALL && pair==0)
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
         neuronptr->prevdelta=0;

   iter=0;
}


#ifdef USEPROTOS
boolean calc_delta(void)
#else
boolean calc_delta()
#endif
{
boolean convergence = TRUE;
int weight;
Neuronptr neuronptr;
Weightptr weightptr;
FLOAT value;

   if (learn & FBPR)
   {
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
         if (neuronptr->changed.delta)
         {
            for (weight=0; weight<neuronptr->nrweights; weight++)
            {
               weightptr=neuronptr->weights+weight;

               if (weightptr->delay==FALSE &&
                   (weightptr->constraint==NULL ||
                    (strchr(weightptr->constraint, '!')==NULL &&
                     strchr(weightptr->constraint, '|')==NULL)))
               {
                  if (weightptr->from2==NULL)
                  {
                     weightptr->from->deltasum-=weightptr->partdel;
                     weightptr->partdel=weightptr->value*neuronptr->delta;
                     weightptr->from->deltasum+=weightptr->partdel;
                     weightptr->from->changed.extra=TRUE;
                  }
                  else
                  {
                     weightptr->from->deltasum-=weightptr->partdel;
                     weightptr->partdel=weightptr->value*
                                        neuronptr->delta*
                                        weightptr->from2->output;
                     weightptr->from->deltasum+=weightptr->partdel;
                     weightptr->from->changed.extra=TRUE;

                     weightptr->from2->deltasum-=weightptr->partdel2;
                     weightptr->partdel2=weightptr->value*
                                         neuronptr->delta*
                                         weightptr->from->output;
                     weightptr->from2->deltasum+=weightptr->partdel2;
                     weightptr->from2->changed.extra=TRUE;
                  }
               }
            }
         }

      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      {
         if (neuronptr->changed.extra)
         {
            value=neuronptr->deltasum*neuron_derivative(neuronptr);

            if (fabs((neuronptr->delta-value)/
                     (neuronptr->delta!=0 ? neuronptr->delta : 1))>EPS)
            {
               neuronptr->delta=value;
               neuronptr->changed.delta=TRUE;
               convergence=FALSE;
            }
            else
               neuronptr->changed.delta=FALSE;

            neuronptr->changed.extra=FALSE;
         }
         else
            neuronptr->changed.delta=FALSE;
      }

      if (++iter>=max_iter)
         convergence=TRUE;
   }

   return(convergence);
}


#ifdef USEPROTOS
void adjust_lrates(int pattern)
#else
void adjust_lrates(pattern)
        int pattern;
#endif
{
int weight;
FLOAT muldelta_sign;
Neuronptr neuronptr;
Weightptr weightptr;

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      muldelta_sign=neuronptr->delta*neuronptr->prevdelta;

#ifdef DEBUG
      printf("sign of delta*prevdelta = %c\n", muldelta_sign<0 ? '-' : '+');
#endif

      if (muldelta_sign!=0)
      {
         for (weight=0; weight<neuronptr->nrweights; weight++)
         {
            weightptr=neuronptr->weights+weight;

            if (muldelta_sign>0)
            {
#ifdef DEBUG
               printf("positive rate adjustment\n");
#endif

               switch (adjust_mode)
               {
               case ADJUST_RELATIVE:
                  weightptr->epsilon+=weightptr->addrate*weightptr->epsilon;
                  break;
               case ADJUST_LOCAL:
                  weightptr->epsilon+=weightptr->addrate*GETPATTERN(pattern)->error;
                  break;
               case ADJUST_GLOBAL:
                  weightptr->epsilon+=weightptr->addrate;
                  break;
               }
            }
            else
            {
#ifdef DEBUG
               printf("negative rate adjustment\n");
#endif

               weightptr->epsilon*=1-weightptr->mulrate;
            }
         }

         if (muldelta_sign>0)
         {
#ifdef DEBUG
            printf("positive rate adjustment\n");
#endif

            switch (adjust_mode)
            {
            case ADJUST_RELATIVE:
               neuronptr->epsilon+=neuronptr->addrate*neuronptr->epsilon;
               break;
            case ADJUST_LOCAL:
               neuronptr->epsilon+=neuronptr->addrate*GETPATTERN(pattern)->error;
               break;
            case ADJUST_GLOBAL:
               neuronptr->epsilon+=neuronptr->addrate;
               break;
            }
         }
         else
         {
#ifdef DEBUG
            printf("negative rate adjustment\n");
#endif

            neuronptr->epsilon*=1-neuronptr->mulrate;
         }
      }
   }
}


#ifdef USEPROTOS
void calc_delay(void)
#else
void calc_delay()
#endif
{
Neuronptr neuronptr;

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      if (fabs((neuronptr->output-neuronptr->prevoutput)/
               (neuronptr->output!=0 ? neuronptr->output : 1))>EPS)
      {
         neuronptr->prevoutput=neuronptr->output;
         neuronptr->changed.prev=TRUE;
      }
      else
         neuronptr->changed.prev=FALSE;

      neuronptr->prevdelta=(1-dmomentum)*neuronptr->delta+
                            dmomentum*neuronptr->prevdelta;
   }
}


#ifdef USEPROTOS
void reset_dweights(void)
#else
void reset_dweights()
#endif
{
int weight;
Neuronptr neuronptr;
Weightptr weightptr;

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      for (weight=0; weight<neuronptr->nrweights; weight++)
      {
         weightptr=neuronptr->weights+weight;
         weightptr->delta=0;
      }

      neuronptr->dbias=0;
   }
}


#ifdef USEPROTOS
void calc_dweights(int pattern)
#else
void calc_dweights(pattern)
        int pattern;
#endif
{
int weight;
Neuronptr neuronptr;
Weightptr weightptr;
FLOAT value, value2;

   if (!(learn & ANNEAL))
      for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
      {
         for (weight=0; weight<neuronptr->nrweights; weight++)
         {
            weightptr=neuronptr->weights+weight;

            if (weightptr->constraint==NULL ||
                (strchr(weightptr->constraint, '=')==NULL &&
                 strchr(weightptr->constraint, '!')==NULL))
            {
               if (weightptr->delay==TRUE)
               {
                  value=weightptr->from->prevoutput;
                  if (weightptr->from2==NULL)
                     value2=1;
                  else
                     value2=weightptr->from2->prevoutput;
               }
               else
               {
                  value=weightptr->from->output;
                  if (weightptr->from2==NULL)
                     value2=1;
                  else
                     value2=weightptr->from2->output;
               }

               if (learn & DHEBB)
                  weightptr->delta+=GETPATTERN(pattern)->relative*
                     neuronptr->derivation*weightptr->from->derivation;
               else
                  if (learn & GROSSBERG)
                     weightptr->delta+=
                        GETPATTERN(pattern)->relative*(value*value2-weightptr->value);
                  else
                     weightptr->delta+=
                        GETPATTERN(pattern)->relative*(neuronptr->delta*value*value2);
            }
         }


         if ((neuronptr->constraint==NULL ||
              strchr(neuronptr->constraint, '=')==NULL) &&
             !(neuronptr->nrweights==0 || (neuronptr->type!=INPUT &&
                                           neuronptr->nrweights==1)))
            if (learn & (GROSSBERG | DHEBB))
               neuronptr->dbias=0;
            else
               neuronptr->dbias+=GETPATTERN(pattern)->relative*neuronptr->delta;
      }
}


#ifdef USEPROTOS
void update_weights(void)
#else
void update_weights()
#endif
{
int weight;
FLOAT delta_weight, delta_bias;
Neuronptr neuronptr;
Weightptr weightptr;

   if ((learn & ANNEAL) && accept==FALSE &&
       (FLOAT)(Random(10000)/10000) < chance)
      accept=TRUE;

   for (neuronptr=neuronlist; neuronptr!=NULL; neuronptr=neuronptr->next)
   {
      for (weight=0; weight<neuronptr->nrweights; weight++)
      {
         weightptr=neuronptr->weights+weight;

         if (weightptr->constraint==NULL ||
             (strchr(weightptr->constraint, '=')==NULL &&
              strchr(weightptr->constraint, '!')==NULL))
         {
            if (learn & ANNEAL)
            {
               if (accept==FALSE)
                  weightptr->value-=weightptr->prevdelta;

               /*  geef 'dw' een random waarde tussen -1 en 1  */
               weightptr->delta=(FLOAT)(Random(20000)-10000)/10000;
            }

            delta_weight=(1-momentum)*weightptr->epsilon*weightptr->delta+
                         momentum*weightptr->prevdelta;

            weightptr->value+=delta_weight;

            weightptr->value*=1-forget;       /*  vermenigvuldig met 'vergeet'-factor  */

            weightptr->value=max(weightptr->value, -maxweight);
            weightptr->value=min(weightptr->value, maxweight);

            if (weightptr->constraint!=NULL &&
                ((strchr(weightptr->constraint, '>')!=NULL &&
                  weightptr->value<weightptr->orgvalue) ||
                 (strchr(weightptr->constraint, '<')!=NULL &&
                  weightptr->value>weightptr->orgvalue)))
               weightptr->value=weightptr->orgvalue;

            weightptr->prevdelta=delta_weight;
         }
      }


      if ((neuronptr->constraint==NULL ||
           strchr(neuronptr->constraint, '=')==NULL) &&
          !(neuronptr->nrweights==0 || (neuronptr->type!=INPUT &&
                                        neuronptr->nrweights==1)))
      {
         if (learn & ANNEAL)
         {
            if (accept==FALSE)
               neuronptr->bias-=neuronptr->prevdbias;

            /*  geef 'dbias' een random waarde tussen -1 en 1  */
            neuronptr->dbias=(FLOAT)(Random(20000)-10000)/10000;
         }

         delta_bias=(1-momentum)*neuronptr->epsilon*neuronptr->dbias+
                    momentum*neuronptr->prevdbias;

         neuronptr->bias+=delta_bias;

         neuronptr->bias*=1-forget;           /*  vermenigvuldig met 'vergeet'-factor  */

         neuronptr->bias=max(neuronptr->bias, -maxweight);
         neuronptr->bias=min(neuronptr->bias, maxweight);

         if (neuronptr->constraint!=NULL &&
             ((strchr(neuronptr->constraint, '>')!=NULL &&
                neuronptr->bias<neuronptr->orgbias) ||
              (strchr(neuronptr->constraint, '<')!=NULL &&
                neuronptr->bias>neuronptr->orgbias)))
            neuronptr->bias=neuronptr->orgbias;

         neuronptr->prevdbias=delta_bias;
      }
   }
}


#ifdef USEPROTOS
boolean step_check(void)
#else
boolean step_check()
#endif
{
char key;

   step++;
   con_step++;

   if (decay!=0 && (step % labs(decay))==0)
      if (decay<0)
      {
         epsilon*=(FLOAT)decay_val/(decay_val+1);
         decay_val++;
      }
      else
         if (decay_val>1)
         {
            epsilon*=(FLOAT)decay_val/(decay_val-1);
            decay_val--;
         }

   if (step==REFRESH-1)                /*  verander vlak voor eerste   */
   {                                   /*  weergave van foutgrafiek    */
      max_error=total_error*1.2;       /*  1.2 is een correctiefactor  */
      min_error=total_error;
   }

   if (((min_error-total_error)/(min_error!=0 ? min_error : 1))>EPSERROR)
   {
      min_error=total_error;
      con_step=0;
   }

   if (kbhit())
   {
      key=getch();

      if (key=='S' || key=='s')
         con_step=max_step;

      if (key=='T' || key=='t')
         trace=!trace;
   }

   chance=0.5*exp(-2.0*step/max_step);

   update_time();                      /*  bepaal meest recente tijden  */

   return((lim_step>0 ? step<lim_step : TRUE) &&
          error<total_error && con_step<max_step);
}


#ifdef USEPROTOS
FLOAT neuron_output(Neuronptr neuronptr)
#else
FLOAT neuron_output(neuronptr)
         Neuronptr neuronptr;
#endif
{
FLOAT activation = neuronptr->activation+neuronptr->bias;

   switch (neuronptr->function)
   {
      case STEP:
         return(activation==UNDEFINED ? UNDEFINED
                               : (activation>0.0 ? MAXOUTPUT : MINOUTPUT));
      case LOGISTIC:
         return(MAXMINOUT/(1.0+exp(-activation))+MINOUTPUT);
      case LOGC4:
         return(MAXMINOUT/(1.0+exp(-4.0*activation))+MINOUTPUT);
      case LOGC8:
         return(MAXMINOUT/(1.0+exp(-8.0*activation))+MINOUTPUT);
      case LINEAR:
         return(activation);
   }

   return(0.0);                 /*  dummy, vermijdt 'warning'  */
}


#ifdef USEPROTOS
FLOAT neuron_derivative(Neuronptr neuronptr)
#else
FLOAT neuron_derivative(neuronptr)
         Neuronptr neuronptr;
#endif
{
FLOAT output = neuronptr->output;

   switch (neuronptr->function)
   {
      case STEP:
         return(0.0);
      case LOGISTIC:
         return(0.25*MAXMINOUT*(1.0+output)*(1.0-output));
      case LOGC4:
         return(1.00*MAXMINOUT*(1.0+output)*(1.0-output));
      case LOGC8:
         return(2.00*MAXMINOUT*(1.0+output)*(1.0-output));
      case LINEAR:
         return(1.0);
   }

   return(0.0);                 /*  dummy, vermijdt 'warning'  */
}


#ifdef USEPROTOS
FLOAT neuron_inverse(Neuronptr neuronptr)
#else
FLOAT neuron_inverse(neuronptr)
         Neuronptr neuronptr;
#endif
{
FLOAT output = neuronptr->output;

   switch (neuronptr->function)
   {
      case STEP:
         return((output==UNDEFINED ? UNDEFINED
                            : (output>0.0 ? MAXOUTPUT : MINOUTPUT))-neuronptr->activation);
      case LOGISTIC:
         return((output>=MAXOUTPUT-1E-10 ? 25.0
                   : (output<=MINOUTPUT+1E-10 ? -25.0
                         : -log((MAXMINOUT+MINOUTPUT-output)/(output-MINOUTPUT))))-neuronptr->activation);
      case LOGC4:
         return((output>=MAXOUTPUT-1E-10 ?  6.0
                   : (output<=MINOUTPUT+1E-10 ?  -6.0
                         : -log((MAXMINOUT+MINOUTPUT-output)/(output-MINOUTPUT))/4.0))-neuronptr->activation);
      case LOGC8:
         return((output>=MAXOUTPUT-1E-10 ?  3.0
                   : (output<=MINOUTPUT+1E-10 ?  -3.0
                         : -log((MAXMINOUT+MINOUTPUT-output)/(output-MINOUTPUT))/8.0))-neuronptr->activation);
      case LINEAR:
         return(output-neuronptr->activation);
   }

   return(0.0);                 /*  dummy, vermijdt 'warning'  */
}


#ifdef USEPROTOS
FLOAT pattern_val(int pattern, int pair, int number, char type)
#else
FLOAT pattern_val(pattern, pair, number, type)
         int pattern;
         int pair;
         int number;
         char type;
#endif
{
   switch (type)
   {
      case INPUT:
         return(*(GETPATTERN(pattern)->in+pair*nrinputs+number));
      case OUTPUT:
         return(*(GETPATTERN(pattern)->out+pair*nroutputs+number));
      case RESULT:
         return(*(GETPATTERN(pattern)->result+pair*nroutputs+number));
   }

   return(0.0);                 /*  dummy, vermijdt 'warning'  */
}


#ifdef USEPROTOS
FLOAT clip_value(FLOAT value)
#else
FLOAT clip_value(value)
         FLOAT value;
#endif
{
   if (value > MAXOUTPUT-tolerance*MAXMINOUT)
      return(MAXOUTPUT);
   else
    if (value < MINOUTPUT+tolerance*MAXMINOUT)
       return(MINOUTPUT);

   return(UNDEFINED);
}


#ifdef USEPROTOS
FLOAT sqr(FLOAT x)
#else
FLOAT sqr(x)
         FLOAT x;
#endif
{
   return(x*x);
}


#if !defined(LINUX) && !defined(SOLARIS)
#ifdef USEPROTOS
int matherr(struct exception *e)
#else
int matherr(e)
       struct exception *e;
#endif
{     
   switch (e->type)
   {
      case DOMAIN:
      case SING:
      case TLOSS:
#ifndef TURBOC      
      case PLOSS:
#endif
         return(0);
      case OVERFLOW:
         if (strcmp(e->name, "exp")==0)
         {
            e->retval=MAXFLOAT;
            return(1);
         }
         return(0);
      case UNDERFLOW:
         if (strcmp(e->name, "exp")==0)
         {
            e->retval=0;
            return(1);
         }
         return(0);
      default:
         return(0);
   }         
}
#endif
