/*
 * xcore.c
 *    Xlib initialization. - entirely based on John A. Artoux's functions.
 *    some X specific routines
 * the path for the include files should be changed to appropriate one
 * for each machine.
 */

#include <stdio.h>
#include <pwd.h>
#include <X11/Xlib.h>  /* get all the X structures */
#include <X11/X.h>     /* types of events we need */
#include <X11/Xutil.h> /*get structures for attributes*/
#include <X11/Xresource.h>
#include <X11/cursorfont.h> /* cursor constants */
#include "xgraph.h"       /* for all the X headers and global definitions */
#include "graph.h"
#include "i.xbm"         /* the icon (presently a synapse by J. Artoux) */
#include "error.h"
#include "userdefs.h"
#include "msg.h"

#define CWAttbs (CWBackPixel|CWBorderPixel|CWBitGravity|CWWinGravity \
     |CWBackingStore|CWBackingPlanes|CWOverrideRedirect|CWEventMask   \
     |CWDontPropagate) /* |CWColormap) */
/* this mask lets the window manager know what attributes I'm interested in
 * setting */

#define GoodEvents (ButtonMotionMask | ButtonPressMask | ButtonReleaseMask)
/* The events I'm interested in */

char * font = GraphicsFont;  /* the default font */

#define min(a, b) ((a)<(b)?(a):(b))

XSetWindowAttributes WAttbs = {  /* the default attributes */
   None,       /* no background Pixmap */
   0,	       /* background pixel will be set to white at run time */
   None,       /* no border Pixmap */
   0,	       /* border color to be set to black at run time */
   NorthWestGravity, /* window remains relative to upper left corner */
   NorthWestGravity, /* when parent moves, this window stays */
   Always,     /* xnet does not do exosure event handling */
   ~0,         /* save all planes (~0 = all bits set to one) */
   0,          /* no backing pixel */
   False,      /* don't save bits during pop-ups */
   GoodEvents, /* XEvents we do care about */
   NoEventMask,/* XEvents we don't care about */
   False,      /* let the window manager intercept commands to this window */
   CopyFromParent, /* copy colormap from parent */
   None        /* no cursor Pixmap (yet) */
};

#define do_or_die( CALL, MSG, ERRVAL )	\
	if((CALL) == NULL && ERRVAL != 0 )	\
	    {	\
	    fprintf( stderr, "Fatal error from `%s' (#%d)\n", MSG, ERRVAL ); \
	    perror("exiting");\
	    exit( ERRVAL ); \
	    } else {}

char xMarker;
float xX, xY,newX, newY, xCharSize;
float scaleX,scaleY;
short fCenterX, fCenterY;
int XLineWidth=1, XLineStyle=LineSolid;

int bd_width = 3;
char *bg_color, *bd_color;
XSizeHints sz = { PPosition, 0, 0, 600, 450 };
XGCValues xorgcv = { GXxor, ~0, 1L, 0L}; /* gc for the rubber band box */
Display *netD;
Window netW,intrW;
GC netGC, xorgc;
Colormap netC;
Visual *netV;
XFontStruct *netF;
unsigned long intr_pixel; /* pixel for interrupt window intrW */
#define intrSize 10
int scrNum; /* the screen number */
char DisplayName[256];

static int opTableEntries = 18;
static XrmOptionDescRec opTable[] = {
{"-background",   "*background",	      XrmoptionSepArg, (caddr_t) NULL},
{"-bd",   	  "*borderColor",	      XrmoptionSepArg, (caddr_t) NULL},
{"-bg",  	  "*background",	      XrmoptionSepArg, (caddr_t) NULL},
{"-borderwidth",  "*ToplevelShell.borderWidth",XrmoptionSepArg,(caddr_t) NULL},
{"-bordercolor",  "*borderColor",	      XrmoptionSepArg, (caddr_t) NULL},
{"-bw",  	  "*ToplevelShell.borderWidth",XrmoptionSepArg,(caddr_t) NULL},
{"-display",   	  ".display",		      XrmoptionSepArg, (caddr_t) NULL},
{"-fg",   	  "*foreground",	      XrmoptionSepArg, (caddr_t) NULL},
{"-font",	  "*font",		      XrmoptionSepArg, (caddr_t) NULL},
{"-foreground",   "*foreground",	      XrmoptionSepArg, (caddr_t) NULL},
{"-fn",	  	  "*font",		      XrmoptionSepArg, (caddr_t) NULL},
{"-geometry",     ".geometry",  	      XrmoptionSepArg, (caddr_t) NULL},
{"-iconic",	  ".TopLevelShell.iconic",    XrmoptionNoArg,  (caddr_t) "on"},
{"-name",   	  ".name",		      XrmoptionSepArg, (caddr_t) NULL},
{"-reverse",   	  "*reverseVideo",	      XrmoptionNoArg,  (caddr_t) NULL},
{"-rv",   	  "*reverseVideo",	      XrmoptionNoArg,  (caddr_t) NULL},
{"-title",   	  ".TopLevelShell.title",     XrmoptionSepArg, (caddr_t) NULL},
{"-xrm",   	  NULL,			      XrmoptionResArg, (caddr_t) NULL},
};

static XrmDatabase resourceDB;	/* resource data base */
static XrmDatabase commandlineDB;   /* resource database obtained from command line*/
static int fgColor, bgColor, borderColor;
static char Geostr[20];
static char *GetHomeDir();
char *getenv();

initialize_display( xmax, ymax, argc, argv )
float *xmax, *ymax;
int argc; char *argv[];
{
  int depth;  /* the screen depth */
  char *usage = "usage: %s [options ...]\nwhere options are:\n\t-bg background color\n\t-bd border color\n\t-bw border width\n\t-geom geometry\n\t-fn fontname\n";
  char *str_type[20];
  char buffer[20];
  XrmValue value;
  int x,y,width,height;
  long flags;

  XrmInitialize();

  /* parse command line arguments and get resource database */

  XrmParseCommand(&commandlineDB, opTable, opTableEntries, argv[0], 
		  &argc, argv );
  if( argc != 1 ) printf(usage);

  /*get display now, because we need it to get otehr databases */
  if ( XrmGetResource(commandlineDB, "xnet.display","XNet.Display",
		      str_type, &value )== True) {
    (void) strncpy(DisplayName, value.addr, (int) value.size);
  }

  /* Open the connection to the server. A null string tells XOpenDisplay to
   * use the UNIX DISPLAY environment variable. Also get the screen number */

  do_or_die( netD = XOpenDisplay(DisplayName), 
	     "OpenDisplay: Set DISPLAY environment or run 'xhost' on the server machine.", 1 ) ;
  scrNum = DefaultScreen(netD);

  /* get all the resource databases - app-defaults & .Xdefaults */
  getUsersDatabase(argc,argv,&resourceDB);

  /* finally, merge command line database into existing database */
  XrmMergeDatabases(commandlineDB, &resourceDB);

  /* get the default colormap and visual for use in color allocation */
  do_or_die( netC = DefaultColormap(netD, scrNum), "DefaultColorMap", 1);
  
  /* look at what's specified in the database */
  /* get geometry */
  if( XrmGetResource(resourceDB, "xnet.geometry", "XNet.Geometry",
		     str_type, &value) == True) {
    (void) strncpy(Geostr, value.addr, (int) value.size);
  } else Geostr[0] = NULL;

  /* get background color */
  if( XrmGetResource(resourceDB, "xnet.background", "XNet.Background",
		     str_type, &value) == True) {
    (void) strncpy(buffer, value.addr, (int) value.size);
    if( allocColor(netD, netC, buffer, &bgColor ));
    else bgColor = BlackPixel(netD,scrNum);
  } else bgColor = BlackPixel(netD,scrNum);

  /* get foreground color */
  if( XrmGetResource(resourceDB, "xnet.foreground", "XNet.Foreground",
		     str_type, &value) == True) {
    (void) strncpy(buffer, value.addr, (int) value.size);
    if( allocColor(netD, netC, buffer, &fgColor ));
    else fgColor = WhitePixel(netD,scrNum);
  } else fgColor = WhitePixel(netD,scrNum);

  /* get border color */
  if( XrmGetResource(resourceDB, "xnet.borderColor", "XNet.BorderColor",
		     str_type, &value) == True) {
    (void) strncpy(buffer, value.addr, (int) value.size);
    if( allocColor(netD, netC, buffer, &borderColor ));
    else borderColor = WhitePixel(netD,scrNum);
  } else borderColor = WhitePixel(netD,scrNum);
  
  /* get font */
  if( XrmGetResource(resourceDB, "xnet.font", "XNet.Font",
		     str_type, &value) == True) {
    strncpy(font, value.addr, (int) value.size );
  }

  /* set the background/border colors */
  WAttbs.background_pixel = bgColor;
  WAttbs.border_pixel = borderColor;

  /* set geometry */
  if( Geostr[0] != NULL ) {
    flags = XParseGeometry(Geostr, &x, &y, &width, &height);
    if(WidthValue & flags ) sz.width = width;
    if(HeightValue & flags ) sz.height = height;
 
    if( XValue & flags ) {
      if( XNegative & flags ) 
	x = DisplayWidth(netD, scrNum) + x - sz.width;
      sz.flags |= USPosition;
      sz.x = x;
    }
    if( YValue & flags ) {
      if ( YNegative & flags ) 
	y = DisplayHeight(netD, scrNum) + x - sz.width;
      sz.flags |= USPosition;
      sz.x = y;
    }
  }

  /* graphics routines use these values */
  *xmax = (float) sz.width, *ymax = (float) sz.height;

  do_or_die( netV = DefaultVisual(netD, scrNum), "DefaultVisual", 1);
  do_or_die( depth = DefaultDepth(netD, scrNum), "DefaultDepth", 1);

   /* create a window with all the defaults that should be set by now.
    * set a few goodies like the icon and the name of the icon and title bar.
    * get a graphics context to be used later in drawing. Select the event
    * inputs were interested in. Load the font */
  
  do_or_die( netW = XCreateWindow(netD, 		/* display */
				  DefaultRootWindow(netD), /* parent */
				  sz.x, sz.y, sz.width, sz.height, /* size */
				  bd_width,		/* border width */
				  depth,	 	/* depth */
				  InputOutput, 		/* class */
				  netV,			/* visual */
				  CWAttbs, 		/* attribute mask */
				  &WAttbs),		/* attributes */
	     "CreateWindow", 1);

  XSetStandardProperties(netD, netW, argv[0], argv[0],
		 XCreateBitmapFromData(netD, netW, i_bits, i_width, i_height),
	         argv, argc, &sz);

  XDefineCursor(netD, netW, XCreateFontCursor(netD, XC_hand1));

  /* just take the default GC */
  do_or_die( netGC = DefaultGC(netD, scrNum),"DefaultGC",1); 

  /* set foreground color */
  XSetForeground(netD, netGC, fgColor);

  if ((netF = XLoadQueryFont(netD, font)) != NULL)
    XSetFont(netD, netGC, netF -> fid);	/* use this font ID */
  else {
    (void) fprintf(stderr,"%s: bad font name.\nusing default font.\n", font);
    netF = XQueryFont(netD, netGC -> gid);
  }

  fCenterX = netF->max_bounds.rbearing;	/* center of char font */
  fCenterY = netF->max_bounds.ascent/2;	/* used for adjustment */
  xorgc = XCreateGC(netD, netW, GCFunction | GCBackground | GCForeground,
		    &xorgcv);   /* initialize the xorgc */

  /* create interrupt window */
  
  do_or_die( intrW = XCreateSimpleWindow(netD, netW,0,0,intrSize,intrSize,0,
					 CopyFromParent, fgColor),
	     "CreateSimpleWindow", 1);

  /* last, but not least, map the window */
  XMapWindow(netD, netW);
  XMapWindow(netD, intrW);

  /* print information about the visual */
  printf("default depth=%d\n", depth);

  if( netV->class == PseudoColor ) printf("multi-plane color\n");
  else if( netV->class == GrayScale ) printf("multi-plane monochrome\n");
  else if( netV->class == DirectColor ) printf("direct color\n");
  else if( netV->class == TrueColor ) printf("direct color - unchangeable\n");
  else if( netV->class == StaticColor ) printf("static color\n");
  else if( netV->class == StaticGray ) printf("static gray\n");

  XFlush(netD); /* flush all pending graphics requests to expose window */
}

getUsersDatabase(rDB)
XrmDatabase *rDB;
{
  XrmDatabase homeDB, serverDB, applicationDB;

  char filenamebuf[1024];
  char *filename = &filenamebuf[0];
  char *environment;
  char *classname = "XNet";
  char name[255];

  (void) sprintf(name, "/usr/lib/X11/app-defaults/%s", classname);
  /* get application defaults file, if any */
  applicationDB = XrmGetFileDatabase(name);
  (void) XrmMergeDatabases(applicationDB, &resourceDB);
  
  /* Merge server defaults, created by xrdb, loaded as a property of the root
   * window when the server initializes, and loaded into the display structure
   * on XOpenDisplay.  If not defined, use .Xdefaults */
  if (netD->xdefaults != NULL ) {
	/* NOTE: we are not supposed to access components of netD 
	 * change this to a macro as soon as it is available.  */
    serverDB = XrmGetStringDatabase(netD->xdefaults);
  }  else {
    /* Open .Xdefaults file and merge into existing database */
    (void) GetHomeDir(filename);
    (void) strcat(filename, "/.Xdefaults");
    serverDB = XrmGetFileDatabase(filename);
  }
  XrmMergeDatabases(serverDB, &resourceDB);

  /* Open XENVIRONMENT file, or if not defined, the .Xdefaults, and merge
   * into existing database */
  if ((environment = getenv("XENVIRONMENT")) == NULL ) {
    int len;
    environment = GetHomeDir(filename);
    (void) strcat(environment, "/.Xdefaults-");
    len = strlen(environment);
    (void) gethostname(environment+len,1024-len);
  }
  homeDB = XrmGetFileDatabase(environment);
  XrmMergeDatabases(homeDB, &resourceDB);
}

allocColor(dpy, colormap, color_name, pixel_return)
/* allocates the RGB values in colormap for the color named by color_name.
 * if the named color exists in the data base, the new pixel value is retrurned
 * through pixel_return. pixel_return is unchanged if the color cannot be
 * allocated and an error message is printed. Color names can be a `#' followed
 * by hex digits specifying red, green and blue pixel values or a number of
 * common English color names such as red, green, blue, violet and yellow.
 * See the manual on color for names that are in the data base */
     Display * dpy;
     Colormap colormap;
     char * color_name;
     unsigned long * pixel_return;
{
  XColor color;  /* will hold the values of pixel and r, g and b intensities */

  if (XParseColor(dpy, colormap, color_name, &color)) { /* if valid name */
    (void) XAllocColor(dpy, colormap, &color);
    *pixel_return = color.pixel;  /* set value for return */
    return(OK);
  } else {
    (void) fprintf(stderr, "bad color: %s\n", color_name);
    return(ERR);
  }
}

set_graphics_font( name )
char *name;
{
  IfErr((netF = XLoadQueryFont(netD, name )) != NULL) 
    Erreturn1("%s: font not found", name );
  XSetFont(netD, netGC, netF -> fid);	/* use this font ID */
  return( OK );
}

clickInterrupt()
{
  XEvent e; void clearButton();
  if( True==XCheckMaskEvent( netD, ButtonReleaseMask, &e ) &&
      e.xbutton.subwindow == intrW ) { 
    clearButton(); return(OK);
  }
  return(ERR);
}

Bool isButton( display, event, arg )
     Display *display;
     XEvent *event;
     char *arg;
{
  if( event->type==ButtonPress || event->type==ButtonRelease ) return(True);
  else return( False );
}

void clearButton()	/* clears button click events in the queue */
{
  XEvent e;
  while( True==XCheckMaskEvent( netD, ButtonPressMask|ButtonReleaseMask, &e ) );
}

locate_box( box )
BOX *box;
{
  XRectangle rect;
  void clearButton();
  clearButton();
  IfErr( rubberBand( netD, netW, &rect, Button3 ) ) return(ERR);
  box->x = rect.x, box->y = YMAX - (rect.y + rect.height);
  box->width = rect.width, box->height = rect.height;
  return( OK );
}

void waitFor(dpy, type, event)
     Display * dpy;
     int type;
     XEvent *event;   /* wait for the event specified to happen */
{
   do {
      XNextEvent(dpy, event);
   } while (event->type != type);
}

wait_any_button()
{
  XEvent e;
  waitFor( netD, ButtonRelease, &e );
}

getMouseClick(button)	/* returns mouse location and the button # clicked */
float button[];
{
  XEvent e;
  waitFor( netD, ButtonRelease, &e );
  button[0] = (float) e.xbutton.x;
  button[1] = (float) e.xbutton.y;
  button[2] = (float) e.xbutton.button;
  return( e.xbutton.button );
}

locate_mouse( x, y )
float *x, *y;
{
  XEvent e;
  clearButton();
  waitFor( netD, ButtonRelease, &e );
  *x = e.xbutton.x, *y = YMAX - e.xbutton.y;
  return(0);
}


int rubberBand(dpy, w, rt, exit_button)
 /* interactively get a rectangle. Uses dpy and w for drawing functions.
  * The rectangle is put into rt. a true value is returned if a vallid 
  * rectangle was specified, 0 if not */
   XRectangle *rt;
   Display *dpy;
   Window w;
   int exit_button;
{
   int sx, sy; /* the saved x and y positions */
   XEvent e;  /* the event we are working with */
   int pressed=0;

   rt->width = rt->height = 0; /* the initial rectangle is size zero */

   sendMsg1("Click button %d to cancel.\n", exit_button);
   for (;;) { /* must return from inside */
      XNextEvent(dpy, &e); /* get the next event for this connection */
      switch (e.type) {
         case ButtonPress: /* designate location as origin of rectangle */
            if (e.xbutton.button == exit_button) { /* cancel button */
               waitFor(dpy, ButtonRelease, &e);
               return(0);
            }
            rt -> x = sx = e.xbutton.x; /* save pointer location */
            rt -> y = sy = e.xbutton.y;
	    pressed = 1;
            break;
         case MotionNotify:
	    if( !pressed ) break;
            /* erase old rectangle */
            XDrawRectangle(dpy, w, xorgc, rt->x, rt->y, rt->width, rt->height);
            rt -> x = min(sx, e.xbutton.x); /* figure out the dimensions */
            rt -> y = min(sy, e.xbutton.y); /* of the new rectangle */
            rt -> width = abs(sx - e.xbutton.x);
            rt -> height = abs(sy - e.xbutton.y);
            /* draw the new rectangle */
            XDrawRectangle(dpy, w, xorgc, rt->x, rt->y, rt->width, rt->height);
            break;
         case ButtonRelease:
            if( !pressed ) break;
            /* erase the rectangle */
            XDrawRectangle(dpy, w, xorgc, rt->x, rt->y, rt->width, rt->height);
            XFlush(dpy);
            return(1); /* executed perfectly */
         default:
            break; /* do nothing */
      }
   }
}

ClipBox( x, y, w, h )
float x,y,w,h;
{
  XRectangle rt;
  rt.x = x, rt.y = YMAX-(y+h), rt.width = w+1, rt.height = h+1;
  XSetClipRectangles( netD, netGC, 0, 0, &rt, 1, Unsorted );
}

  /**** rewrite movie routines in graph.c for XNet.
   **** 
   ****/

Pixmap
take_frame( box )
BOX *box;
{
  Pixmap pixmap;
  pixmap = XCreatePixmap( netD, netW, (int)box->width, (int)box->height, 1 );
  XCopyArea( netD, netW, pixmap, netGC,
	    (int)box->x, (int) (YMAX-(box->y+box->height)),
	    (unsigned int)box->width, (unsigned int)box->height, 0, 0 );
  printf("x=%d y=%d w=%d h=%d\n",
	    (int)box->x, (int) (YMAX-(box->y+box->height)),
	    (int)box->width, (int)box->height );
  XFlush(netD);
  return( pixmap );
}

copy_frame( box1, box2 )
BOX *box1, *box2;
{
  XCopyArea( netD, netW, netW, netGC,
	    (int)box1->x, (int) (YMAX-(box1->y+box1->height)),
	    (unsigned int)box1->width, (unsigned int)box1->height, 
	    (int)box2->x, (int) (YMAX-(box2->y+box2->height)));
  printf("x=%d y=%d w=%d h=%d x2=%d y2=%d\n",
	    (int)box1->x, (int) (YMAX-(box1->y+box1->height)),
	    (unsigned int)box1->width, (unsigned int)box1->height, 
	    (int)box2->x, (int) (YMAX-(box2->y+box2->height)));
  XFlush(netD);
}

erase_frame( frame )
Pixmap frame;
{
  XFreePixmap( netD, frame );
}

show_frame( frame, x1, y1, t_frame )
Pixmap frame;
float x1,y1;		/* lower left corner of frame */
int t_frame;		/* time in msec to show frame */
{
  Window *root;
  int x,y; unsigned int w, h, border_w, depth;
  XGetGeometry( netD, frame, &root, &x, &y, &w, &h, &border_w, &depth );
  XCopyArea( netD, frame, netW, netGC, 0, 0, w, h, 
	   (int) x1, (int) (YMAX-(y1+h)));
  printf("x=%d y=%d w=%d h=%d x2=%d y2=%d\n",0,0,w,h,
	   (int) x1, (int) (YMAX-(y1+h)));
  XFlush(netD);
  return( OK );
}

save_frame( fname, frame )
char *fname;
Pixmap frame;
{
  Window *root;
  int x,y; unsigned int w, h, border_w, depth, status;
  XGetGeometry( netD, frame, &root, &x, &y, &w, &h, &border_w, &depth );
  status=XWriteBitmapFile( netD, fname, frame, w, h, -1, -1 );
  if(status == BitmapSuccess) return( OK );
  if(status == BitmapOpenFailed) Erreturn("cannot open file");
  if(status == BitmapNoMemory) Erreturn("not enough memory");
  return( ERR );
}

Pixmap
read_frame( fname )
char *fname;
{
  unsigned int w, h; int x_hot, y_hot;
  Pixmap frame;
  int status;
  status = XReadBitmapFile( netD, netW, fname, &w, &h, &frame, &x_hot, &y_hot);
  if(status == BitmapSuccess) return( frame );
  if(status == BitmapOpenFailed) Erreturn("cannot open file");
  if(status == BitmapFileInvalid) Erreturn("file invalid");
  if(status == BitmapNoMemory) Erreturn("not enough memory");
  return( ERR );
}

static char *GetHomeDir(dest)
char *dest;
{
  int uid;
  extern char *getenv();
  extern int getuid();
  extern struct passwd *getpwuid();
  struct passwd *pw;
  register char *ptr;
  
  if (( ptr = getenv("HOME")) != NULL ) {
    (void) strcpy(dest, ptr);
  } else {
    if((ptr = getenv("USER")) != NULL ) {
      pw = getpwnam(ptr);
    } else {
      uid = getuid();
      pw = getpwuid(uid);
    }
    if(pw) {
      (void) strcpy(dest, pw->pw_dir);
    } else {
      *dest = ' ';
    }
  }
  return dest;
}
