/* $Header: /soma/users/miyata/planet/src/RCS/vector.c,v 5.6.0.4 91/02/13 15:41:46 miyata Exp $ */
static char rcsid[] = "$Header: /soma/users/miyata/planet/src/RCS/vector.c,v 5.6.0.4 91/02/13 15:41:46 miyata Exp $";
#include <math.h>
#include "error.h"
#include "net.h"
#include "alloc.h"
#include "vector.h"

/*****
  It is very easy to extend SunNet by adding your own vector functions.
  Vector functions are those functions you use in arithmetic expressions
  in procedures, such as sin(), exp(), log(), etc.  This file contains
  the definitions of all those functions.  It should be straightforward
  by looking at the following, how you can add new functions.

  Let's look at how to use a unary arithmetic function in the standard math 
  library, say the function sin().  You only need to do two things.  First 
  put a line:
****/

UnaryVectorFunction( sin_vector, sin )

/****
  This makes "sin_vector" the name of the function.  Pick any name.
  Next thing is to add a line in the function Install() defined below.
  (See explanation there.)
  Here are some more familiar functions.  See /usr/include/math.h to 
  find out what other functions are available in the standard library.
****/

UnaryVectorFunction( cos_vector, cos )
UnaryVectorFunction( tan_vector, tan )
UnaryVectorFunction( asin_vector, asin )
UnaryVectorFunction( acos_vector, acos )
UnaryVectorFunction( atan_vector, atan )

UnaryFunctionTable( exp_table, exp, 512, -5.0, 5.0 )

/**** you can add more functions here ****/

/****
  To implement your own function, you need to define the function so that
  it takes one argument and returns a value (float or double).  It can be
  either a macro as in the following example:
****/

#define BOUND01( x ) x>1.0? 1.0 : x<0.0? 0.0 : x

/****
 or a real function.
****/

NUMBER _logistic();	/* this is defined in net.c */

/***
  Then the vector functions can be defined in exactly the same way.
***/

UnaryVectorFunction( bound01_vector, BOUND01 )
UnaryVectorFunction( logistic_vector, _logistic )

/*****
  Here are some more functions.
*****/

#define BOUND( x ) (x)>MAXBOUND? MAXBOUND : (x)<-MAXBOUND? -MAXBOUND : (x)
#define BOUND0( x ) (x)>MAXBOUND0? MAXBOUND0 : (x)<0.0 ? 0.0 : (x)
#define EXP( x ) ((x)<80.0)? exp((x)) : exp(80.0)
#define SQRT( x ) sqrt( (x)>0.0? (x) : -(x) )
#define LOG( x ) log( (x)>1e-8? (x) : 1e-8 )
#define ABS( x ) (x)>0.0? (x) : -(x)
double   nrandom();int urandom();
	/* random number with gausian distribution with given variance */
#define RANDOM( x ) ((float) nrandom())*(x)
	/* random number with uniform distribution in range [0,x] */
#define URANDOM( x ) ((float) urandom( 0, 4096)) * ( (x)/4096 )
	/* on with probability x, off with probability 1-x */
#define RANDSTATE(x) URANDOM(1.0)<(x)? MAXACTV:MINACTV

#define INTEGER(x) (float) ((int) (x))

UnaryVectorFunction( sqrt_vector, SQRT )
UnaryVectorFunction( exp_vector, EXP )
UnaryVectorFunction( log_vector, LOG )
UnaryVectorFunction( abs_vector, ABS )
UnaryVectorFunction( bound_vector, BOUND )
UnaryVectorFunction( bound0_vector, BOUND0 )
UnaryVectorFunction( random_vector, RANDOM )
UnaryVectorFunction( urandom_vector, URANDOM ) 
UnaryVectorFunction( randomstate_vector, RANDSTATE )
UnaryVectorFunction( integer_vector, INTEGER )

/****  add your own functions here.  ***/

/***
  If you want a function that does not work in element-by-element fashion,
  you need to define it yourself as in the following example which computes
  the sum of the values in a vector.
***/

sum_vector( vector, result )
VECTOR *vector, *result; 
{
  register REAL *res = result->value, *val = vector->value;
  register int i;
  *res = 0;
  for( i= vector->nvalue; i; i--, val++ ) *res += *val;
  result->nvalue = 1;		/* length of result is 1 */
  return(OK);
}

/** the following functions are defined later in this file **/

int average_vector(), transpose_matrix(), sum_matrix_row(), sum_matrix_col();
int permute_vector(), index_max_vector(), index_min_vector(), smooth_matrix();
int min_vector(), max_vector(), sizeof_vector(), input_vector(),target_vector();
int reverse_vector(), norm_vector();

/*****
  Finally, you should put for each function defined above a line of the form:

  InstallVectorFunction( string-name, function-name );

  in the function Install() below,
  where function-name is the name used above, and string-name is the name
  you wish to use to refer to this function in your procedure.
  Don't forget to describe what your new functions do in the help section below.
****/

void Install()
{
  InstallVectorFunction( "sin", 	sin_vector );
  InstallVectorFunction( "cos", 	cos_vector );
  InstallVectorFunction( "tan", 	tan_vector );
  InstallVectorFunction( "asin", 	asin_vector );
  InstallVectorFunction( "acos", 	acos_vector );
  InstallVectorFunction( "atan", 	atan_vector );
  InstallVectorFunction( "logistic", 	logistic_vector );
  InstallVectorFunction( "sqrt", 	sqrt_vector );
  InstallVectorFunction( "exp", 	exp_vector );
  InstallVectorFunction( "exptbl",	exp_table );
  InstallVectorFunction( "log", 	log_vector );
  InstallVectorFunction( "abs", 	abs_vector );
  InstallVectorFunction( "bound", 	bound_vector );
  InstallVectorFunction( "bound01", 	bound01_vector );
  InstallVectorFunction( "bound0", 	bound0_vector );
  InstallVectorFunction( "integer", 	integer_vector );
  InstallVectorFunction( "sum", 	sum_vector );
  InstallVectorFunction( "average", 	average_vector );
  InstallVectorFunction( "transpose", 	transpose_matrix );
  InstallVectorFunction( "mean", 	average_vector );
  InstallVectorFunction( "min", 	min_vector );
  InstallVectorFunction( "max", 	max_vector );
  InstallVectorFunction( "T",	 	transpose_matrix );
  InstallVectorFunction( "smooth",	smooth_matrix );
  InstallVectorFunction( "sumrow", 	sum_matrix_row );
  InstallVectorFunction( "sumcol", 	sum_matrix_col );
  InstallVectorFunction( "permute", 	permute_vector );
  InstallVectorFunction( "reverse", 	reverse_vector );
  InstallVectorFunction( "norm", 	norm_vector );
  InstallVectorFunction( "indexmax", 	index_max_vector );
  InstallVectorFunction( "indexmin", 	index_min_vector );
  InstallVectorFunction( "sizeof", 	sizeof_vector );
  InstallVectorFunction( "input",	input_vector );
  InstallVectorFunction( "target",	target_vector );

  /* If a function is not deterministic, put a third argument   *\
  \* RandomFunc as in the following functions.	 		*/

  InstallVectorFunction( "random", 	random_vector, RandomFunc );
  InstallVectorFunction( "urandom", 	urandom_vector, RandomFunc );
  InstallVectorFunction( "randomstate",	randomstate_vector, RandomFunc );

/***** 
  put InstallVectorFunction( string-name, function-name ); for each of 
  your functions here.
*****/

}

/**
.HE "function" "description of vector functions" VECTOR-HELP
.BB
The following functions are used in the form \fB<function>(<vector>)\fP
\fB<function>(<matrix>)\fP or \fB<function>(<expression>)\fP where <expression> 
evaluates to a vector or a matrix, and apply <function> to every element in 
the vector or the matrix, generating a vector or a matrix of the same dimension.
.Bi
\fBsin(), cos(), tan(), asin(), acos(), atan(), sqrt(), exp(), log(), abs(),
power(),logistic(), bound(), bound0(), bound01(), random(), urandom(),
randomstate()\fP.
.Ni
\fBbound()\fP limits values in a vector between $bound and \(mi$bound.
.Ni
\fBbound0()\fP limits values in a vector between $bound0 and 0.
.Ni
\fBbound01()\fP limits values between 0.0 and 1.0.
.Ni
The parameters $bound and $bound0 can be changed using the command
\fBset\fP or in procedures ( $bound = .. ).
.Ni
\fBrandom(x)\fP generates random numbers with gausian distribution with variance x.
.Ni
\fBurandom(x)\fP generates random numbers with uniform distribution in range [0,x].
.Ni
\fBrandomstate(x)\fP generates a vector whose elements are turned on or off 
according to the probabilities given in its argument.

The following functions have effects that are not element-by-element.  X is
a vector or a matrix.
.Bi
\fBtranspose(X)\fP or \fBT(X)\fP transposes a matrix, including a row vector 
or a column vector.  Transposing a matrix has some overhead for copying it.
Transposing a vector is fast.
.Ni
\fBpermute(X)\fP randomly permutes the elements in X.
.Ni
\fBsumcol(<matrix>)\fP generates a vector of which each element is the sum of 
the elements in each column of <matrix>.
.Ni
\fBsumrow(<matrix>)\fP generates a vector of which each element is the sum of 
the elements in each row of <matrix>.
.Ni
\fBaverage(X)\fP generates a single value which is the average of the elements 
of X.
.Ni
\fBsum(X)\fP generates a single value which is the sum of the elements of X.
.Ni
\fBmin(X)\fP generates a single value which is the minimum value in X.
.Ni
\fBmax(X)\fP generates a single value which is the maximum value in X.
.Ni
\fBnorm(X)\fP generates a single value which is the norm of X.  This is 
equivalent to sqrt(sum(X^2)).
.Ni
\fBindexmax(X)\fP generates a single value which is the index 
for the element with the largest value in X.  
The index for a matrix element is (row #)*(no. of columns)+(column #).
.Ni
\fBindexmin(X)\fP generates a single value which is the index
for the element with the smallest value in X.  
The index for a matrix element is (row #)*(no. of columns)+(column #).
.EE
**/

/** Some additional functions are defined below */

average_vector( vector, result  )
VECTOR *vector, *result; 
{
  register REAL *res = result->value, *val = vector->value;
  register int i;
  *res = 0;
  for( i=vector->nvalue; i; i--, val++ ) *res += *val;
  if( vector->nvalue ) *res /= vector->nvalue;
  result->nvalue = 1;		/* length of result is 1 */
  return(OK);
}

min_vector( vector, result )
VECTOR *vector, *result; 
{
  register REAL *res = result->value, *val = vector->value;
  register int i;
  *res = *val;
  for( i= vector->nvalue; i; i--, val++ ) 
    if( *res > *val ) *res = *val;
  result->nvalue = 1;		/* length of result is 1 */
  return(OK);
}

max_vector( vector, result )
VECTOR *vector, *result; 
{
  register REAL *res = result->value, *val = vector->value;
  register int i;
  *res = *val;
  for( i= vector->nvalue; i; i--, val++ ) 
    if( *res < *val ) *res = *val;
  result->nvalue = 1;		/* length of result is 1 */
  return(OK);
}

transpose_matrix( vector, result  )
VECTOR *vector, *result; 
{
  register REAL *res, *val = vector->value;
  register int i,j;
  int nval1 = vector->nvalue1, nval2 = vector->nvalue2;
  if( !nval1 ) { /* its a vector -> change to a column vector (1 col. matrix)*/
    copy_vector_( val, result->value, &vector->nvalue );
    result->nvalue1 = vector->nvalue;
    result->nvalue2 = 1;
    return(OK);
  }
  for( i=0; i< nval1; i++ )	/* its a matrix ->transpose */
    for( j=nval2, res=result->value+i; j; j--, val++, res += nval1 ) 
      *res = *val;
  result->nvalue1 = vector->nvalue2;
  result->nvalue2 = vector->nvalue1;
  return(OK);
}

sum_matrix_row( vector, result )
VECTOR *vector, *result;
{
  register int i,j;
  register REAL *res = result->value, *val = vector->value;
  int nval1 = vector->nvalue1, nval2 = vector->nvalue2;
  if( !nval1 ) nval1 = 1, nval2 = vector->nvalue;
  for( i=nval1; i; i--, res++ ) {
    *res = 0;
    for( j=nval2 ; j; j--, val++ ) *res += *val;
  }
  result->nvalue1 = nval1;
  result->nvalue2 = 1;
  result->nvalue = nval1;	/* number of rows */
  return(OK);
}

sum_matrix_col( vector, result )
VECTOR *vector, *result;
{
  register int i,j;
  register REAL *res = result->value, *val;
  int nval1 = vector->nvalue1, nval2 = vector->nvalue2;
  if( !nval1 ) nval1 = 1, nval2 = vector->nvalue;
  for( i=0; i < nval2; i++, res++ ) {
    *res = 0;
    for( j=nval1, val = vector->value+i ; j ; j--, val += nval2 )
      *res += *val;
  }
  result->nvalue = nval2;	/* number of columns */
  return(OK);
}

sizeof_vector( vector, result )
VECTOR *vector, *result;
{
  result->value[0] = vector->nvalue;
  result->nvalue = 1;
  return( OK );
}

smooth_matrix( vector, result )
VECTOR *vector, *result;
{
  register int row,col;
  register REAL *res = result->value, *val;
  int nval1 = vector->nvalue1, nval2 = vector->nvalue2, index;
  if( vector->nvalue == 1 ) return( OK ); /* onlye 1 value -nothing to do */
  if( !nval1 ) {
    nval1 = vector->nvalue;
    val = vector->value;
    res[0] = (val[0]+val[1])*0.5;
    for( col=1; col<nval1-1; col++ ) 
      res[col] = (val[col-1]+val[col]+val[col+1])/3;
    res[col] = (val[col-1]+val[col])*0.5;
  }
  else {
	/* the four corners */
    /* row = 0, col = 0, index = 0;*/
    val = vector->value;
    res[0] = (*val+ *(val+1)+ *(val+nval2)+ *(val+nval2+1))/4;

    row = 0, col = nval2-1, index = row*nval2+col;
    val = &(vector->value[index]);
    res[index] = (*val+ *(val-1)+ *(val+nval2)+ *(val+nval2-1))/4;

    row = nval1-1, col = 0, index = row*nval2+col;
    val = &(vector->value[index]);
    res[index] = (*val+ *(val+1)+ *(val-nval2)+ *(val-nval2+1))/4;

    row = nval1-1, col = nval2-1, index = row*nval2+col;
    val = &(vector->value[index]);
    res[index] = (*val+ *(val-1)+ *(val-nval2)+ *(val-nval2-1))/4;

    /* for row = 0 */
    for( row = 0, col=1; col<nval2-1; col++ ) {
      index = row*nval2+col;
      val = &(vector->value[index]);
      res[index] = (*(val-1)+ *val + *(val+1) +
		    *(val+nval2-1) + *(val+nval2) + *(val+nval2+1))/6;
    }
    /* for row = nval1-1 */
    for( row = nval1-1, col=1; col<nval2-1; col++ ) {
      index = row*nval2+col;
      val = &(vector->value[index]);
      res[index] = (*(val-1)+ *val + *(val+1) +
		    *(val-nval2-1) + *(val-nval2) + *(val-nval2+1))/6;
    }
    /* for col = 0 */
    for( row = 1, col=0; row<nval1-1; row++ ) {
      index = row*nval2+col;
      val = &(vector->value[index]);
      res[index] = (*(val-nval2)+ *(val) + *(val+nval2) +
		    *(val-nval2+1)+ *(val+1) + *(val+nval2+1))/6;
    }
    /* for col = nval2-1 */
    for( row = 1, col=nval2-1; row<nval1-1; row++ ) {
      index = row*nval2+col;
      val = &(vector->value[index]);
      res[index] = (*(val-nval2)+ *(val) + *(val+nval2) +
		    *(val-nval2-1)+ *(val-1) + *(val+nval2-1))/6;
    }
	/* other values */
    for( row=1; row<nval1-1; row++ ) { /* i = row */
      for( col=1; col<nval2-1; col++ ) {		/* j = column */
	index = row*nval2+col;
	val = &(vector->value[index-nval2]);
	res[index] = *(val-1)+ *(val)+ *(val+1);
	val += nval2;
	res[index] += *(val-1)+ *(val)+ *(val+1);
	val += nval2;
	res[index] += *(val-1)+ *(val)+ *(val+1);
	res[index] /= 9;
      }
    }
  }
  result->nvalue = vector->nvalue,
  result->nvalue1 = vector->nvalue1,
  result->nvalue2 = vector->nvalue1;
  return(OK);
}

permute_vector( vector, result )
VECTOR *vector, *result;
{
  copy_vector_( vector->value, result->value, &vector->nvalue );
  permut( result->value, vector->nvalue, sizeof( *result->value) );
  result->nvalue = vector->nvalue;
  return(OK);
}

norm_vector( vector, result )
VECTOR *vector, *result;
{
  register int i; REAL *val=vector->value, res;
  for( i= vector->nvalue, res = 0.0 ; i; i--, val++ )
    res += *val * *val;
  *result->value = sqrt( res );
  result->nvalue = 1;
  return( OK );
}

reverse_vector( vector, result )
VECTOR *vector, *result;
{
  register int i; 
  REAL *val = vector->value, *res = result->value + vector->nvalue - 1;
  for( i=vector->nvalue; i; i--, val++, res-- )
    *res = *val;
  return( OK );
}

index_max_vector ( vec, result )
     VECTOR *vec, *result;
{
  register int i;
  register REAL *val = vec->value;
  REAL max;
  int    max_i = 0;

  for ( max= *(val++), i=vec->nvalue-1; i ; i--, val++ ) 
    if ( max < *val ) {
      max = *val ;
      max_i = vec->nvalue-i;
    }
  result->value[0] = (float) max_i ;
  result->nvalue = 1;
  result->nvalue1 = result->nvalue2 = 0;
  return ( OK );
}

index_min_vector ( vec, result )
     VECTOR *vec, *result;
{
  register int i;
  register REAL *val=vec->value;
  REAL min;
  int    min_i = 0;

  for ( min = *(val++), i=vec->nvalue-1; i ; i--, val++ ) 
    if ( min > *val ) {
      min = *val ;
      min_i = vec->nvalue-i;
    }
  result->value[0] = (float) min_i ;
  result->nvalue = 1;
  result->nvalue1 = result->nvalue2 = 0;
  return ( OK );
}
/* input_vector and target_vector functions get input or target pattern *
 * # N, N being specified by the argument */
extern REAL **Input, **Target;
extern int N_input,N_target,Npattern;

input_vector( vector, result )
VECTOR *vector, *result;
{
  register int i;
  register REAL *input, *val;
  int nIn = (int) vector->value[0];
  if( nIn >= Npattern ) Erreturn1("only %d patterns", Npattern);
  input = Input[nIn];
  if( result->nalloc < N_input ) {
    if( result->value ) free( result->value );
    IfErr( result->value = new_array_of( N_input, REAL ) ) 
      Erreturn("not enough memory");
    result->nalloc = N_input;
  }
  for( val = result->value, i=N_input; i; i--, input++, val++ ) *val = *input;
  result->nvalue = N_input;
  return( OK );
}

target_vector( vector, result )
VECTOR *vector, *result;
{
  register int i;
  register REAL *target, *val;
  int nTarg = (int) vector->value[0];
  if( nTarg >= Npattern ) Erreturn1("only %d patterns", Npattern);
  target = Target[nTarg];
  if( result->nalloc < N_target ) {
    if( result->value ) free( result->value );
    IfErr( result->value = new_array_of( N_target, REAL ) ) 
      Erreturn("not enough memory");
    result->nalloc = N_target;
  }
  for( val = result->value, i=N_target; i; i--, target++, val++ ) *val = *target;
  result->nvalue = N_target;
  return( OK );
}
  
/* matrix-matrix multiplication: includes, as special cases, 
 * matrix**transpose(vec): vector-matrix multiplication ; 
 * transpose(vec)**vec : vector-vector inner product ;
 * vec**transpose(vec) : vector-vector outer product ;
 */

matrix_multi( vec1, vec2, result, type )
VECTOR *vec1, *vec2, *result; int type;
{
  register REAL *val1, *val2, *res;
  register int i,j,k;
  int row1,col1,row2,col2, nval;
  REAL *val1_start, *val2_start;

  if( vec1->nvalue1 ) col1 = vec1->nvalue2, row1 = vec1->nvalue1;
  else 		      col1 = vec1->nvalue, row1 = 1;
  if( vec2->nvalue1 ) col2 = vec2->nvalue2, row2 = vec2->nvalue1;
  else 		      col2 = vec2->nvalue, row2 = 1;
  if( col1 != row2 ) Erreturn("incompatible matrices for multiplication");
			/* allocate memory for result if not enough */
  if( (nval = col2*row1) <= result->nalloc ) result->nvalue = nval;
  else IfErr(alloc_vector( result, nval )) return(ERR);

  for(i=row1, res=result->value, val1_start=vec1->value;
	i; i--, val1_start += col1 )
    for( j=col2, val2_start=vec2->value; j; j--, res++, val2_start++ )
      for( k=col1, val1=val1_start, val2=val2_start, *res=0.0; 
	   k; k--, val1++ , val2 += col2)
	*res += *val1 * *val2;
   result->nvalue1 = row1;
   result->nvalue2 = col2;
   return(OK); 
}

column_matrix( vec, ncol, result )
VECTOR *vec, *result; int ncol;
{
  register int i; REAL *val, *res;
  if( ! vec->nvalue1 ) Erreturn("not a matrix");
  if( ncol >= vec->nvalue2 ) Erreturn1("subscript %d out of range",ncol);
  for( val=vec->value+ncol, res=result->value, i=vec->nvalue1;
      i; i--, val += vec->nvalue2, res++ )
    *res = *val;
  result->nvalue = result->nvalue1 = vec->nvalue1;
  result->nvalue2 = 1;
  return(OK);
}

/* The following functions implement actions such as "push", "pop", "shift",
 * "threshold", etc.  These could be converted to binary functions like
 * shift(vector,n) to access them in expressions.  */

#define evaluate_expression(ex) (*ex->evalfunc)(ex)
#define min(x,y) (x<y? (x):(y))

threshold( ex1, ex2, exThresh )
EXPRESS *ex1, *ex2, *exThresh;
{
  int nval; 
  register int i;
  register REAL *val1, *val2, *threshold;
  if(Err( evaluate_expression(ex1) ) || Err( evaluate_expression(ex2)) ||
     Err( evaluate_expression(exThresh))) return(ERR);
  nval = min( ex1->vector->nvalue, ex2->vector->nvalue );
  val1 = ex1->vector->value, val2 = ex2->vector->value;

  if( exThresh->vector->nvalue == 1 ) { /* one threshold value */
    for( i=nval; i ; i--, val1++, val2++ )
      *val2 = (*val1 > *threshold)? MAXACTV : MINACTV ;
    return(OK);
  }
  			/* otherwise threshold is a vector */
  nval = min( nval, exThresh->vector->nvalue );
  for( i=nval; i ; i--, val1++, val2++, threshold++ )
    *val2 = (*val1 > *threshold)? MAXACTV : MINACTV ;
  return( OK );
}

push_vector( ex1, ex2 )
EXPRESS *ex1, *ex2;
{
  int nval;
  VECTOR *vec1, *vec2;
  if(Err( vec1=evaluate_expression(ex1) ) || 
     Err( vec2=evaluate_expression(ex2)) ) return(ERR);
  nval = min( vec1->nvalue, vec2->nvalue );
	/* shift vec2->value to the right by nval values */
  IfErr( shift_vector_(&nval, vec2->value, &vec2->nvalue) ) return( ERR );
  copy_vector_( vec1->value, vec2->value, &nval );
  return( OK );
}

push_from_end( ex1, ex2 )
EXPRESS *ex1, *ex2;
{
  int	nshift;
  VECTOR *vec1,*vec2;
  if( Err( vec1=evaluate_expression(ex1) ) || 
      Err( vec2=evaluate_expression(ex2)) ) return(ERR);
  if( vec1->nvalue >= vec2->nvalue ) { /* vec1 larger. copy all */
    copy_vector_( vec1->value, vec2->value, &vec2->nvalue );
    return(OK);
  }
  nshift = - vec1->nvalue;	/* shift vector2 to the left */
  IfErr( shift_vector_( &nshift, vec2->value, &vec2->nvalue ) ) return( ERR );
  copy_vector_( vec1->value, vec2->value + (vec2->nvalue-vec1->nvalue), 
		&vec1->nvalue );
  
  return( OK );
}

pop_vector( ex1, ex2 )
EXPRESS *ex1,*ex2;
{
  int shift, nval;
  VECTOR *vec1,*vec2;
  if( Err( vec1=evaluate_expression(ex1) ) || 
      Err( vec2=evaluate_expression(ex2)) ) return(ERR);
  nval = min( vec2->nvalue, vec1->nvalue );
	/* copy first nval values of vec1->value to vec2->value */
  copy_vector_( vec1->value, vec2->value, &nval );
	/* shift vec1->value to the left by nval values */
  shift = -nval;
  return( shift_vector_( &shift, vec1->value, &vec1->nvalue ) );
}

pop_from_end( ex1, ex2 )
EXPRESS *ex1, *ex2;
{
  VECTOR *vec1,*vec2;int nval;
  if( Err( vec1=evaluate_expression(ex1) ) || 
      Err( vec2=evaluate_expression(ex2)) ) return(ERR);
  nval = min( vec1->nvalue, vec2->nvalue );
	/* copy last nval values of vec1->value to vec2->value */
  copy_vector_( vec1->value + vec1->nvalue - nval, vec2->value, &nval );
  	/* shift vec1->value by nval values to the right */
  return( shift_vector_( &nval, vec1->value, &vec1->nvalue ) );
}

shift_vector( exp, shift )
EXPRESS *exp, *shift;
{
  VECTOR *vec1,*vec2; int nshift;
  if( Err( vec1=evaluate_expression(exp) ) || 
      Err( vec2=evaluate_expression(shift)) ) return(ERR);
  nshift = (int) vec2->value[0];
  return( shift_vector_( &nshift, vec1->value, &vec1->nvalue ));
}

#ifndef allie
shift_vector_( nshift, vector, nvector ) /* shift content of vector */
int	*nshift;		/* positive/negative -> shift right/left */
REAL	*vector;
int     *nvector;		/* # of values in the vector */
{
  register int i,n;
  if( *nshift > 0 ) {		/* shift to right */
    i = *nvector-1;		/* end of vector */
    n = i - *nshift;		/* no. of shift from end of vector */
    for( ; n >= 0 ; i--, n--) vector[i] = vector[n];
    bzero( vector, sizeof(REAL)*(i+1) );
  }
  
  else if( *nshift < 0 ) {
    i = 0;			/* beginning of vector */
    n = -*nshift;		/* no. of shift from beginning of vector*/
    for( ; n < *nvector ; i++, n++ ) vector[i] = vector[n];
    for( ; i < *nvector ; i++ ) vector[i] = 0.0;
  }
  return( OK );
}
#endif

clamp_vector( ex1, ex2 )
EXPRESS *ex1, *ex2;
{
  register REAL	*vector1, *vector2; 
  int nval;
  register int i;
  if( Err( evaluate_expression(ex1)) || Err( evaluate_expression(ex2)) )
    return(ERR);
  vector1= ex1->vector->value, vector2= ex2->vector->value;
  nval = min( ex1->vector->nvalue, ex2->vector->nvalue);
  for( i=nval; i ; i--, vector1++, vector2++ )
    if( *vector2 != DONT_CARE ) *vector1 = *vector2;
  return(OK);
}

add_noise_to_vector( exp, noiseExp )
EXPRESS *exp, *noiseExp;
{
  REAL	*vector;
  double nrandom();
  REAL   noise;
  register int i;

  if( Err( evaluate_expression(exp)) || Err( evaluate_expression(noiseExp)) )
    return(ERR);
  vector= exp->vector->value, noise = noiseExp->vector->value[0];

  for ( i= exp->vector->nvalue; i ; i--, vector++ ) {
    *vector += noise * ((float) nrandom()) ;
  }
  return ( OK );
}

ivector_times_float( ivec, val, result, nval )
IVECTOR *ivec; REAL *val, *result; int *nval;
{
  register int i, index;
/*int nindex = ivec->nindex;*/
  int nindex = ivec->nvalue;	/* no. of values in index */
  REAL *sval=ivec->value, *res = result;
  REAL *ind=ivec->index;
  for( i= *nval; i; i--, res++) *res = 0.0;	/* initialize to zeros */
  for( i=nindex; i; i--, sval++, ind++ )
    if( (index= *ind)>=0 && index < *nval )
      result[index] = *sval * *val;
  return( OK );
}
  
ivector_dot_vector( ivec, vec, result, nval )
IVECTOR *ivec; REAL *vec, *result; int *nval;
{
  register int i, index;
  int nindex = ivec->nvalue;
  REAL *sval=ivec->value;
  REAL *ind=ivec->index;
  *result = 0.0;
  for( i=nindex; i; i--, sval++, ind++ )
    if( (index= *ind)>=0 && index < *nval )
      *result += *sval * vec[index];
  return( OK );
}

ivector_times_float_inc( ivec, val, result, nval )
IVECTOR *ivec; REAL *val, *result; int *nval;
{
  register int i, index;
/*int nindex = ivec->nindex; */
  int nindex = ivec->nvalue;	/* no. of values in index */
  REAL *sval=ivec->value;
  REAL *ind=ivec->index;

  for( i=nindex; i; i--, sval++, index++, ind++ )
    if( (index= *ind)>=0 && index < *nval )
      result[index] += *sval * *val;
  return( OK );
}
  
ivector_dot_vector_inc( ivec, vec, result, nval )
IVECTOR *ivec; REAL *vec, *result; int *nval;
{
  register int i, index;
  int nindex = ivec->nvalue;
  REAL *sval=ivec->value;
  REAL *ind=ivec->index;
  for( i=nindex; i; i--, sval++, ind++ )
    if( (index= *ind) >=0 && index < *nval )
      *result += *sval * vec[index];
  return( OK );
}

make_index( ivec, vec, nval )
IVECTOR *ivec; REAL *vec; int *nval;
{
  int index=0,i; REAL *ival=ivec->value, *iind=ivec->index; 
	/* initialize indeces and values to zeros. */
  for( i=ivec->nindex; i; i--, ival++, iind++ ) *ival = *iind = 0.0;

  for( i=0, index=0; i< *nval; i++, vec++ )
    if( *vec != 0.0 ) {
      if( index >= ivec->nindex )
	Erreturn("more non-zero values than index space");
      ivec->value[index] = *vec;
      ivec->index[index++] = i;
    }
  ivec->nvalue = index;
  return( OK );
}

sort_index( ivec )	/* sort index by its values */
IVECTOR *ivec;
{
  register int i; register REAL *val, *ind;
  REAL tmp;
  int change; 
  do {
    change=0;
    for(i=ivec->nvalue-1, ind=ivec->index, val=ivec->value;i;i--,val++,ind++)
      if( *val > *(val+1) ) {
	tmp = *val; *val = *(val+1); *(val+1) = tmp;
	tmp = *ind; *ind = *(ind+1); *(ind+1) = tmp;
	change=1;
      }
  } while( change );
  return( OK );
}

sort_vector( vector, nval )	/* sort vector by its values */
REAL *vector;int *nval;
{
  register int i;
  int change; 
  register REAL *val; REAL tmp;
  for(change=1; change; change=0 ) {
    for(i= *nval-1, val=vector; i; i--, val++ )
      if( *val > *(val+1) ) {
	tmp = *val; *val = *(val+1); *(val+1) = tmp;
	change=1;
      }
  }
  return( OK );
}

/**** Definitions of basic vector operators ****/

/*** Arithmetic Operators ***/

#ifndef allie	/* these are written in fortran for AllieNet */
#define ADD( r, x, y )  r = (x) + (y)
#define MINUS( r, x, y ) r = (x) - (y)
#define TIMES( r, x, y ) r = (x) * (y)
#define DIVIDE( r, x, y )  { \
			     Check(y==0.0, "division by zero");\
			     r = (x) / (y);\
			   }
#define POWER(r,x,y) r = pow((x),(y))

VectorOperation( add_vector, ADD,  )
VectorOperation( minus_vector, MINUS,  )
VectorOperation( times_vector, TIMES,  )
VectorOperation( divide_vector, DIVIDE,  )
VectorOperation( power_vector, POWER, )

#endif allie

/*** Comparison Operators ***/

#define EQUAL(r,x,y)  r = (x)==(y)? TrueVal : FalseVal
#define NEQUAL(r,x,y) r = (x)!=(y)? TrueVal : FalseVal
#define GT(r,x,y)     r = (x)> (y)? TrueVal : FalseVal
#define GEQ(r,x,y)    r = (x)>=(y)? TrueVal : FalseVal
#define LT(r,x,y)     r = (x)< (y)? TrueVal : FalseVal
#define LEQ(r,x,y)    r = (x)<=(y)? TrueVal : FalseVal

/*** Logical Operators ***/

/* and (&) returns larger of two operands when both are greater than MINACTV.*
 * otherwise it returns MINACTV.*/

#define AND(r,x,y) r = ((x)>MINACTV&&(y)>MINACTV)? \
			((x)>(y)? (x):(y)) : MINACTV

/* or (|) returns larger of two operands when at least one of them is greater*
 * than MINACTV.  otherwise it returns MINACTV.*/

#define OR(r,x,y) r = ((x)>MINACTV||(y)>MINACTV)? \
		       (((x)>(y))? (x):(y)) : MINACTV

/* logical and (&&) returns TRUE if both operands are NON-FALSE */

#define LOGIC_AND(r,x,y) r = ((x)!=FalseVal&&(y)!=FalseVal)? TrueVal:FalseVal
/* logical or (||) returns TRUE if at least one operand is NON-FALSE */

#define LOGIC_OR(r,x,y) r = ((x)!=FalseVal||(y)!=FalseVal)? TrueVal:FalseVal

VectorOperation( equal_vector, EQUAL, )
VectorOperation( nequal_vector, NEQUAL, )
VectorOperation( gt_vector, GT, )
VectorOperation( geq_vector, GEQ, )
VectorOperation( lt_vector, LT, )
VectorOperation( leq_vector, LEQ, )
VectorOperation( and_vector, AND, )
VectorOperation( or_vector, OR, )
VectorOperation( logic_and_vector, LOGIC_AND, )
VectorOperation( logic_or_vector, LOGIC_OR, )

#ifdef allie	/* operations passed to fortran routines in AllieNet */
    
divide_vector( vec1, vec2, result, type )
VECTOR *vec1, *vec2, *result; int *type;
{
  register int i;
  		/* check floating exception first and pass to fortran routine*/
  REAL *v2;
  switch( *type ) {
  case VopS:
    if( *(vec2->value) == 0.0 ) Erreturn("division by zero");
    break;
  case SopV:   case VopV:
    v2 = vec2->value+result->nvalue-1 ;
    for( i=result->nvalue; i> 0; i--, v2-- ) {
      if( *v2 == 0.0 ) Erreturn("division by zero");
    }
    break;
  }
  return( divide_vector_( vec1->value, vec2->value, 
			  &result->nvalue, result->value, type ) );
}

add_vector( vec1, vec2, result, type )
VECTOR *vec1, *vec2, *result; int *type;
{
  return( add_vector_( vec1->value, vec2->value, 
		      &result->nvalue, result->value, type ) );
}

minus_vector( vec1, vec2, result, type )
VECTOR *vec1, *vec2, *result; int *type;
{
  return( minus_vector_( vec1->value, vec2->value, 
		      &result->nvalue, result->value, type ) );
}

times_vector( vec1, vec2, result, type )
VECTOR *vec1, *vec2, *result; int *type;
{
  return( times_vector_( vec1->value, vec2->value, 
		      &result->nvalue, result->value, type ) );
}

power_vector( vec1, vec2, result, type )
VECTOR *vec1, *vec2, *result; int *type;
{
  return( add_vector_( vec1->value, vec2->value, 
		      &result->nvalue, result->value, type ) );
}
#endif allie

#if 0	/* the following vector functions for actions have been scrapped */
	/* replaced by equivalent expressions.  */

average_vector_( vector, nvector, value  )
REAL	*vector,*value;
int	*nvector;
{
  register int i;
  *value = 0.0 ;
  for ( i= *nvector ; i ; i--, vector++ ) *value += *vector ;
  if ( *nvector )  *value /= *nvector ;
  return ( OK );
}

sum_vector_( vector, nvector, value  )
REAL	*vector,*value;
int	*nvector;
{
  register int i;
  *value = 0.0 ;
  for ( i= *nvector ; i ; i--, vector++ ) *value += *vector ;
  return ( OK );
}

min_vector_( vector, nvector, minval  )
REAL	*vector,*minval;
int	*nvector;
{
  register int i;
  *minval = *vector ;
  for ( i= *nvector ; i ; i--, vector++ ) 
      if( *minval > *vector ) *minval = *vector ;
  return ( OK );
}

max_vector_( vector, nvector, max  )
REAL	*vector,*max;
int	*nvector;
{
  register int i;
  *max = *vector ;
  for ( i= *nvector ; i ; i--, vector++ ) 
      if( *max < *vector ) *max = *vector ;
  return ( OK );
}

#if	mac|hp		/* Mac II or Bobcat */
#define	random		rand
#define	srandom		srand
#endif mac|hp

long random ();

extern double	divisor;
	/* the divisor is used to map integers returned by random()
	 * to between zero and one.  It is one larger than the
	 * largest integer on a machine - from Gary Perlman's rand.c
	#define DIVISOR (sizeof(int)==4?2147483648.0:32768.0)
	 */

random_state( vector, prob, nvector )
     REAL *vector, *prob;
     int    *nvector;
{
  register int i;
  for( i=0; i< *nvector; i++ ) 
    *(vector++) = ( *(prob++) > (random() / divisor) )? MAXACTV : MINACTV ;
  return( OK );
}

outer_product( vector1, n1, vector2, n2, matrix )
     REAL *vector1, *vector2, *matrix;
     int *n1, *n2;
{
  register int i,j;
  REAL *v1;
  for( i=0; i< *n2; i++, vector2++ ) {
    v1 = vector1;
    for( j=0; j< *n1; j++ ) *(matrix++) = *(v1++) * *vector2;
  }
  return( OK );
}
#endif 0
