/* $Header: /soma/users/miyata/planet/src/RCS/arith.c,v 5.6.0.5 91/02/13 15:41:07 miyata Exp $ */
static char rcsid[] = "$Header: /soma/users/miyata/planet/src/RCS/arith.c,v 5.6.0.5 91/02/13 15:41:07 miyata Exp $";
/********** UPDATES *********************************************************
arith.c:
9/27/90  implemented in-expression assignment eval_assign_expression()
9/10/90  fixed bug in optimize_expression() evaluating procedure argument.
4/20/90  fixed bug in optimize_expression() evaluating matrix[n][i->j] even
	 when n is a variable.
11/21/89 fixed bug in eval_range_expression() making some matrices vectors.
	- use eval_range2_expression() to evaluate double range. 
****************************************************************************/

#include <stdio.h>
#include <math.h>
#include "error.h"
#include "alloc.h"
#include "stack.h"
#include "net.h"
#include "arith.h"
#include "vector.h"

/* Find_variable_expression() returns an expression that is a variable,
   ie. that can be assigned values, as in "variable = expression".  This 
   includes vector/matrix, & vector/matrix subscripted by expressions */

EXPRESS *
find_variable_expression( net, name )
NETWORK *net; char *name;
{
  EXPRESS *ex;
  IfErr(ex = find_optimize_expression( net, name ) ) return( ERR );
  if( ex->type == Terminal || ex->type == RangeOp ) return(ex);
  Erreturn1("%s: not a variable", name);
}

/* assign_expression() implements the action "variable = expression" */

assign_expression( exLhs, exRhs )
EXPRESS *exLhs, *exRhs;
{
  VECTOR *vecLhs, *vecRhs; int nval;
  if( Err( exLhs ) || Err( exRhs ) ) return(ERR);
  if(Err( vecLhs = (*exLhs->evalfunc)( exLhs ) ) ||
     Err( vecRhs = (*exRhs->evalfunc)( exRhs ) )) return( ERR );
  if( vecRhs->nvalue == 1 ) 
    set_vector_to_float( vecRhs->value, vecLhs->value, &vecLhs->nvalue );
  else {
    nval = min( vecLhs->nvalue, vecRhs->nvalue );
    copy_vector_( vecRhs->value, vecLhs->value, &nval );
  }
  return( OK );
}

evalExpression( ex )		/* used in setup.c */
EXPRESS	*ex;
{
  IfErr( ex ) return(ERR);
  IfErr( ex->vector = (*ex->evalfunc)( ex ) ) return(ERR);
  return( OK );
}

/* eval_xxx_expression() recursively traverses and evaluates an expression tree */

VECTOR *
eval_terminal_expression( ex )
EXPRESS	*ex;
{
  VECTOR *vec;
  IfErr( ex ) return( ERR );
  vec = ex->vector;
  if( vec->accessfunc ) 
    vec->value = (*vec->accessfunc)(vec->object,&vec->nvalue,&vec->nvalue1,&vec->nvalue2);
  IfErr( vec->value ) return(ERR); /* argument ptr may not be there yet */
  return( vec );
}

VECTOR *
eval_assign_expression( ex )		/* expression = expression */
EXPRESS *ex;
{
  IfErr( ex ) return( ERR );
  IfErr( assign_expression( ex->arg1, ex->arg2 ) ) return( ERR );
  return( ex->arg1->vector );
}

VECTOR *
eval_increment_expression( ex )		/* expression += expression */
EXPRESS *ex;
{
  VECTOR *vecLhs, *vecRhs; int nval;
  if(Err( vecLhs = (*ex->arg1->evalfunc)( ex->arg1 ) ) ||
     Err( vecRhs = (*ex->arg2->evalfunc)( ex->arg2 ) )) return( ERR );
  if( vecRhs->nvalue == 1 ) 
    increment_vector_with_float(vecRhs->value, vecLhs->value, &vecLhs->nvalue );
  else {
    nval = min( vecLhs->nvalue, vecRhs->nvalue );
    increment_vector_( vecRhs->value, vecLhs->value, &nval );
  }
  return( vecLhs );
}

VECTOR *
eval_decrement_expression( ex )		/* expression -= expression */
EXPRESS *ex;
{
  VECTOR *vecLhs, *vecRhs; int nval;
  if(Err( vecLhs = (*ex->arg1->evalfunc)( ex->arg1 ) ) ||
     Err( vecRhs = (*ex->arg2->evalfunc)( ex->arg2 ) )) return( ERR );
  if( vecRhs->nvalue == 1 ) 
    decrement_vector_with_float(vecRhs->value, vecLhs->value, &vecLhs->nvalue );
  else {
    nval = min( vecLhs->nvalue, vecRhs->nvalue );
    decrement_vector_( vecRhs->value, vecLhs->value, &nval );
  }
  return( vecLhs );
}

VECTOR *
eval_multiply_expression( ex )		/* expression *= expression */
EXPRESS *ex;
{
  VECTOR *vecLhs, *vecRhs; int nval;
  if(Err( vecLhs = (*ex->arg1->evalfunc)( ex->arg1 ) ) ||
     Err( vecRhs = (*ex->arg2->evalfunc)( ex->arg2 ) )) return( ERR );
  if( vecRhs->nvalue == 1 ) 
    multiply_vector_with_float(vecRhs->value, vecLhs->value, &vecLhs->nvalue );
  else {
    nval = min( vecLhs->nvalue, vecRhs->nvalue );
    multiply_vector_( vecRhs->value, vecLhs->value, &nval );
  }
  return( vecLhs );
}

VECTOR *
eval_divide_expression( ex )	/* expression /= expression */
EXPRESS *ex;
{
  VECTOR *vecLhs, *vecRhs; int nval;
  if(Err( vecLhs = (*ex->arg1->evalfunc)( ex->arg1 ) ) ||
     Err( vecRhs = (*ex->arg2->evalfunc)( ex->arg2 ) )) return( ERR );
  if( vecRhs->nvalue == 1 ) 
    divide_vector_with_float(vecRhs->value, vecLhs->value, &vecLhs->nvalue );
  else {
    nval = min( vecLhs->nvalue, vecRhs->nvalue );
    divide_vector_( vecRhs->value, vecLhs->value, &nval );
  }
  return( ex->arg1->vector );
}

VECTOR *
eval_range_expression( ex )/* vector[ expression -> expression ] */
EXPRESS	*ex;
{
  register VECTOR *vector;	/* results of eval(ex->arg1)*/
  register VECTOR *vstart, *vend;	/* results of eval(start), eval(end) */
  register int n1, n2;
  VECTOR *result;

  IfErr( vector = (*ex->arg1->evalfunc)( ex->arg1 )) return(ERR);

  result = ex->vector;
  IfErr(vstart = (*ex->start->evalfunc)( ex->start )) return( ERR );
  if( ex->end == NULL ) vend = NULL;
  else IfErr(vend = (*ex->end->evalfunc)( ex->end )) return( ERR );

  n1 = (int) vstart->value[0];
  n2 = vend? (int) vend->value[0] : n1;

  if(n1<0 || n2<0 || n2 < n1 ) Erreturn2("invalid range %d -> %d", n1, n2);

  if( vector->nvalue1 ) { 	/* its a matrix - extract rows */
    if( n2 >= vector->nvalue1 ) Erreturn1("subscript %d out of range", n2);
    result->value = vector->value + n1 * vector->nvalue2;
    result->nvalue1 = n2-n1+1; result->nvalue2 = vector->nvalue2;
    result->nvalue = result->nvalue1 * result->nvalue2;
  }
  else {		/* its a vector - extract columns */
    if( n2 >= vector->nvalue ) Erreturn1("subscript %d out of range", n2);
    result->value = vector->value+n1;
    result->nvalue = n2-n1+1;
  }
  result->name = vector->name;	/* should name carry range too? */
  return( result );
}

VECTOR *
eval_range2_expression( ex )/* vector[ expr ][ expr -> expr ] */
EXPRESS	*ex;
{
  register VECTOR *vector;	/* results of eval(ex->arg1)*/
  register VECTOR *vstart, *vend;	/* results of eval(start), eval(end) */
  register VECTOR *vrow ;	/* result of eval(row) */
  register int n1, n2, row;
  VECTOR *result;
	/* evaluate the vector to be indexed */
  IfErr( vector = (*ex->arg1->evalfunc)( ex->arg1 )) return(ERR);
  if( ! vector->nvalue1 ) Erreturn("double range for a vector");
	/* evaluate the first index -> row */
  IfErr(vrow = (*ex->row->evalfunc)( ex->row )) return( ERR );
	/* evaluate the second index -> column */
  IfErr(vstart = (*ex->start->evalfunc)( ex->start )) return( ERR );
			/* single column */
  if( ex->end == NULL ) vend = NULL;
			/* column range */
  else IfErr(vend = (*ex->end->evalfunc)( ex->end )) return( ERR );
			/* put result in ex->vector */
  result = ex->vector;
			/* row index */
  row = (int) vrow->value[0];
			/* column n1 through n2 */
  n1 = (int) vstart->value[0];
  n2 = vend? (int) vend->value[0] : n1;
			/* check validity */
  if(row<0 ) Erreturn1("invalid row # %d", row );
  if(row >= vector->nvalue1 ) Erreturn1("row subscript %d out of range", row);
  if(n1<0 || n2<0 || n2 < n1 ) Erreturn2("invalid range %d -> %d", n1, n2);
  if( n2 >= vector->nvalue2 ) Erreturn1("column subscript %d out of range",n2);

  result->value = vector->value + row * vector->nvalue2 + n1 ;
  result->nvalue1 = 1 ; result->nvalue2 = n2-n1+1;
  result->nvalue = n2-n1+1;

  result->name = vector->name;	/* should name carry range too? */
  return( result );
}

VECTOR *
eval_func_expression( ex )		/* function( expression ) */
EXPRESS	*ex;
{
  register VECTOR *vec, *result = ex->vector;
  register EXPRESS *arg1 = ex->arg1;
  IfErr( vec = (*arg1->evalfunc)( arg1 ) ) return( ERR );
  if( vec->nvalue <= result->nalloc ) result->nvalue=vec->nvalue;
  else IfErr( alloc_vector( result, vec->nvalue )) return(ERR);
  IfErr( (*ex->func)( vec, result )) return(ERR);
  return( result );		/* ptr to value/vector */
}

VECTOR *
eval_multi_expression( ex )	 	/* expression ** expression */
EXPRESS	*ex;
{
  register VECTOR *vec1, *vec2;
  register EXPRESS *arg1 = ex->arg1, *arg2 = ex->arg2;
  if(Err( vec1 = (*arg1->evalfunc)( arg1 )) ||
     Err( vec2 = (*arg2->evalfunc)( arg2 )) ||
     Err( matrix_multi(vec1, vec2, ex->vector ))) 
    return(ERR);
  return( ex->vector );		/* ptr to value/vector */
}

VECTOR *
eval_op_expression( ex )		/* expression OP expression */
EXPRESS	*ex;
{
  register VECTOR *vec1, *vec2; 
  register EXPRESS *arg1 = ex->arg1, *arg2 = ex->arg2;
  if(Err(vec1 = (*arg1->evalfunc)( arg1 )) ||
     Err(vec2 = (*arg2->evalfunc)( arg2 )) ||
     Err((*ex->func)(vec1, vec2, ex->vector ))) 
    return(ERR);
  return( ex->vector );		/* ptr to value/vector */
}

VECTOR *
eval_column_expression( ex ) /* matrix[][expr] */
EXPRESS	*ex;
{
  register VECTOR *vec1,*vec2;	/* results of eval(ex->arg1) & eval(ex->arg2)*/
  register VECTOR *result = ex->vector;
  if(Err(vec1 = (*ex->arg1->evalfunc)( ex->arg1 )) ||
     Err(vec2 = (*ex->arg2->evalfunc)( ex->arg2 )) ) return( ERR );
  if( vec1->nvalue1 <= result->nalloc ) result->nvalue=vec1->nvalue1;
  else IfErr( alloc_vector( result, vec1->nvalue1 )) return(ERR);
  IfErr( column_matrix(vec1, (int) vec2->value[0], result )) return(ERR);
  return( result );		/* ptr to value/vector */
}

VECTOR *
eval_expression( ex )		/* used in sunnet.c, vector.c etc. */
EXPRESS	*ex;
{
  IfErr( ex ) return(ERR);
  return( ex->vector = (*ex->evalfunc)( ex ) );
}

EXPRESS *
find_optimize_expression( net, name )
NETWORK *net; char *name;
{
  EXPRESS *ex;
  IfErr(ex = find_expression( net, name ) ) return( ERR );
  IfErr( optimize_expression( &ex ) ) return( ERR );
  return( ex );
}

/* optimize_expression() tries to minimize run-time evaluation steps *
 * by replacing constant expression (const OP const, FUNC(const)) *
 * and constant range (vector[ const -> const]) with evaluated result. */

optimize_expression( exP )
EXPRESS	**exP;
{
  EXPRESS *ex, *result = NULL;
  REAL constValue;

  if( Err( exP ) || Err( ex = *exP ) ) return( ERR );
  switch( ex->type ) {
  case Terminal: case Constant:		/* vector or constant */
    return( OK );			/* cannot optimize any more */
  case RangeOp:			
    if(Err( optimize_expression( &ex->arg1 )) ||
       Err( optimize_expression( &ex->start )) ) return( ERR );
			/* terminal[start] or terminal[ start -> end ] */
    if( ex->end && Err( optimize_expression( &ex->end ))) return( ERR );
			/* or terminal[ row ][ start -> end ] */
    if( ex->row && Err( optimize_expression( &ex->row ))) return( ERR );

#if 0	/* terminal[ constant range ] cannot be optimized because *
	 * terminal ptr may change & accessfunc may need be exec'ed *
	 * at run-time for some objects (like $mouseClick).  */

    if( ex->arg1->type == Terminal && ex->start->type == Constant &&
       (ex->arg1->vector->value) && /* null if its a procedure argument */
       (ex->end==NULL || ex->end->type == Constant ) &&
       (ex->row==NULL || ex->row->type == Constant )) { 
		/* terminal[ constant range ] -> terminal */
      if( Err( (*ex->evalfunc)( ex ) )  ||
	 Err( result = terminal_expression( ex->vector ))) return(ERR);
    }
#endif 0
    break;
  case UnaryFunc:		/* function( expression ) */
    IfErr( optimize_expression( &ex->arg1 ) ) return( ERR );
    if( ex->arg1->type == Constant ) { /* FUNC( const ) -> const */
      IfErr( (*ex->evalfunc)( ex ) ) {
	Erreturn1("%s: cannot evaluate constant expression", ERR_MSG);
      }
      constValue = ex->vector->value[0];
      IfErr( result = constant_expression( constValue )) return(ERR);
    }
    break;
  case UnaryRandomFunc:		/* random-function( expression ) */
    IfErr( optimize_expression( &ex->arg1 ) ) return( ERR );
    break;
  case BinaryOp:		/* expression OP expression */
  case MultiOp:		/* expression ** expression */
    if(Err( optimize_expression( &ex->arg1 )) ||
       Err( optimize_expression( &ex->arg2 )) ) return( ERR );
    if( ex->arg1->type == Constant && ex->arg2->type == Constant ) {
      				/* const OP const -> const */
      IfErr( (*ex->evalfunc)( ex ) ) {
	Erreturn1("%s: cannot evaluate constant expression", ERR_MSG);
      }
      constValue = ex->vector->value[0]; 
      IfErr( result = constant_expression( constValue )) return(ERR);
    }
    break;
  case Assign:		/* expression = expression */
  case ColumnOp:	/* expression[][expression] */
    if(Err( optimize_expression( &ex->arg1 )) ||
       Err( optimize_expression( &ex->arg2 )) ) return( ERR );
    break;
  default:
    Erreturn("unknown expression type");
  }
  if( result ) {      /* optimized => replace with new expression */
    if( ex->name ) result->name = new_string( ex->name, result->name );
    *exP = result;
    delete_expression( ex );
  }
  return( OK );
}

alloc_vector( vec, nval )
VECTOR *vec; int nval;
{
  if(vec->value) free( vec->value );
  IfErr((vec->value = new_array_of(nval, REAL))) Erreturn("not enough memory");
  vec->nvalue = vec->nalloc = nval;
  return(OK);
}

EXPRESS *
op_expression( op, ex1, ex2 )
char op; EXPRESS *ex1, *ex2;
{
  EXPRESS *ex;
  if(Err(ex1) || Err(ex2)) return(ERR);
  if(Err(ex=new(EXPRESS)) || Err(ex->vector = new(VECTOR))) 
    Erreturn("not enough memory");
  find_operation( ex->op = op, &ex->func );
  ex->arg1 = ex1; ex->arg2 = ex2;
  ex->type = BinaryOp;
  ex->evalfunc = eval_op_expression ;
  return(ex);
}

EXPRESS *
assignment_expression( type, ex1, ex2 )
char type;
EXPRESS *ex1, *ex2;
{
  EXPRESS *ex;
  if(Err(ex1) || Err(ex2)) return(ERR);
  IfErr( is_lvalue( ex1 ) ) Erreturn("assignment to non-lvalue expression");
  if(Err(ex=new(EXPRESS)) || Err(ex->vector = new(VECTOR))) 
    Erreturn("not enough memory");
  ex->arg1 = ex1; ex->arg2 = ex2;
  ex->type = Assign;
  switch( type ) {
  case '=': ex->evalfunc = eval_assign_expression ; break;
  case '+': ex->evalfunc = eval_increment_expression ; break;
  case '-': ex->evalfunc = eval_decrement_expression ; break;
  case '*': ex->evalfunc = eval_multiply_expression ; break;
  case '/': ex->evalfunc = eval_divide_expression ; break;
  default: Erreturn1("%c=: unknown assignment operation", type );
  }
  return(ex);
}

is_lvalue( ex ) /* OK if ex is an l-value, ERR otherwise */
EXPRESS *ex;
{
  IfErr(ex) return(ERR);
  if( ex->type == Terminal || ex->type == Assign ) return(OK);
  if( ex->type == RangeOp ) return( is_lvalue( ex->arg1 ) );
  return(ERR);
}

EXPRESS *
multi_expression( ex1, ex2 )
EXPRESS *ex1, *ex2;
{
  EXPRESS *ex;
  if(Err(ex1) || Err(ex2)) return(ERR);
  if(Err(ex=new(EXPRESS)) || Err(ex->vector = new(VECTOR))) 
    Erreturn("not enough memory");
  ex->arg1 = ex1; ex->arg2 = ex2;
  ex->type = MultiOp;
  ex->evalfunc = eval_multi_expression ;
  return(ex);
}

EXPRESS *
range_expression( arg, range )
EXPRESS *arg; RANGE *range;
{
  EXPRESS *ex;
  if( Err(arg) || Err(range) || Err(range->start)) return(ERR);
  if( Err(ex = new(EXPRESS)) || Err(ex->vector = new(VECTOR)))
    Erreturn("not enough memory");
  ex->type = RangeOp;
  ex->row = NULL;
  ex->start = range->start, ex->end = range->end;
  ex->arg1 = arg;
  ex->evalfunc = eval_range_expression ;
  return( ex );
}

EXPRESS *
range2_expression( arg, range1, range2 )
EXPRESS *arg; RANGE *range1, *range2;
{
  EXPRESS *ex;
  if( Err(arg) || Err(range1) || Err(range2) ) return(ERR);
  if( Err(ex = new(EXPRESS)) || Err(ex->vector = new(VECTOR)))
    Erreturn("not enough memory");
  ex->type = RangeOp;
  ex->row = range1->start;
  ex->start = range2->start, ex->end = range2->end;
  ex->arg1 = arg;
  ex->evalfunc = eval_range2_expression ;
  return( ex );
}

EXPRESS *
column_expression( ex1, ex2 )
EXPRESS *ex1, *ex2;
{
  EXPRESS *ex;
  if( Err(ex1) || Err(ex2) ) return(ERR);
  if( Err(ex = new(EXPRESS)) || Err(ex->vector = new(VECTOR)))
    Erreturn("not enough memory");
  ex->type = ColumnOp;
  ex->arg1 = ex1;
  ex->arg2 = ex2;
  ex->evalfunc = eval_column_expression ;
  return( ex );
}

EXPRESS *
func_expression( func, arg )
int (*func)(); EXPRESS *arg;
{
  EXPRESS *ex;
  IfErr(arg) return(ERR);
  if(Err(ex=new(EXPRESS)) || Err(ex->vector = new(VECTOR))) 
    Erreturn("not enough memory");
  ex->func = func;
  ex->arg1 = arg;
  ex->type = UnaryFunc;
  ex->evalfunc = eval_func_expression ;
  return(ex);
}

EXPRESS *
randomfunc_expression( func, arg )
int (*func)(); EXPRESS *arg;
{
  EXPRESS *ex;
  IfErr(arg) return(ERR);
  if(Err(ex=new(EXPRESS)) || Err(ex->vector = new(VECTOR))) 
    Erreturn("not enough memory");
  ex->func = func;
  ex->arg1 = arg;
  ex->type = UnaryRandomFunc;
  ex->evalfunc = eval_func_expression ;
  return(ex);
}

EXPRESS *
terminal_expression( vector )
VECTOR *vector;
{
  EXPRESS *ex = new(EXPRESS); 
  if(Err(ex) || Err(ex->vector = new(VECTOR))) Erreturn("not enough memory");
  ex->type = Terminal;
  ex->vector->value = vector->value;
  ex->vector->object = vector->object;
  ex->vector->accessfunc = vector->accessfunc;
  ex->vector->nvalue = vector->nvalue;
  ex->vector->nvalue1 = vector->nvalue1;
  ex->vector->nvalue2 = vector->nvalue2;

  IfErr( ex->vector->name = new_string(vector->name, NULL)) 
    Erreturn("not enough memory");
  ex->evalfunc = eval_terminal_expression ;
  return( ex );
}

EXPRESS *
constant_expression( value )
REAL value;
{
  char name[16];
  EXPRESS *ex = new(EXPRESS); 
  if(Err(ex) || Err(ex->vector = new(VECTOR))) Erreturn("not enough memory");
  ex->type = Constant;
  ex->vector->value = new_array_of( 1, float);
  ex->vector->value[0] = value;
  ex->vector->nvalue = 1;
  ex->evalfunc = eval_terminal_expression ;
  sprintf(name, "%g", value);
  IfErr(ex->name = new_string(name, NULL )) Erreturn("not enough memory");
  return( ex );
}

char *
pop( st ) STACK *st; {
  if( st->pointer > 0 )
    return( st->array[--st->pointer] );
  else
    return( ERR );
}

push( c, st )
char	*c;	STACK *st;
{
  if( st->pointer >= STACKSIZE ) return( ERR );
  st->array[st->pointer++] = c;
  return( OK );
}

empty( st ) STACK *st; {
  if( st->pointer <= 0 ) 
    return( 1 );
  else
    return( 0 );
}

char *
top( st ) STACK *st; {
  if( st->pointer > 0 && st->pointer <= STACKSIZE )
    return( st->array[ st->pointer-1 ] );
  else
    return( NULL );
}

VECTORFUNC VectorFunction[MaxVectorFunction];
int NVectorFunction = 0;

VECTORFUNC RandomFunction[MaxRandomFunction];
int NRandomFunction = 0;

void InstallRandomFunction();

void
InstallVectorFunction( name, func, type )
char *name; int (*func)(); int type;
{
  if( type == RandomFunc ) {
    InstallRandomFunction( name, func );
    return;
  }
  if( NVectorFunction >= MaxVectorFunction ) {
    fprintf( stderr, "Sorry.  Cannot install more than %d functions.\n", 
	     MaxVectorFunction );
    finish( 1 );
  }
  VectorFunction[ NVectorFunction ].name = new_string( name, NULL );
  VectorFunction[ NVectorFunction ].func = func ;
  NVectorFunction ++ ;
}

void
InstallRandomFunction( name, func )
char *name; int (*func)();
{
  if( NRandomFunction >= MaxRandomFunction ) {
    fprintf( stderr, "Sorry.  Cannot install more than %d random functions.\n", 
	     MaxRandomFunction );
    finish( 1 );
  }
  RandomFunction[ NRandomFunction ].name = new_string( name, NULL );
  RandomFunction[ NRandomFunction ].func = func ;
  NRandomFunction ++ ;
}

find_function( name, funcP )
char *name; int (**funcP)();
{
  register int i;
  IfErr( name ) return( ERR );
  for(i=0; i < NVectorFunction; i++ ) 
    IF( name, VectorFunction[i].name ) { 
      *funcP = VectorFunction[i].func;
      return( OK );
    }
  Erreturn1( "cannot find function %s", name );
}

find_random_function( name, funcP )
char *name; int (**funcP)();
{
  register int i;
  IfErr( name ) return( ERR );
  for(i=0; i < NRandomFunction; i++ ) 
    IF( name, RandomFunction[i].name ) { 
      *funcP = RandomFunction[i].func;
      return( OK );
    }
  Erreturn1( "cannot find function %s", name );
}

find_operation( op, funcP )
int op; int (**funcP)();
{
  switch( op ) {
    case '+': *funcP = add_vector ; return( OK );
    case '-': *funcP = minus_vector ; return( OK );
    case '*': *funcP = times_vector ; return( OK );
    case '/': *funcP = divide_vector ; return( OK );
    case '^': *funcP = power_vector ; return( OK );
    case '&': *funcP = and_vector ; return( OK );
    case '|': *funcP = or_vector ; return( OK );
    case LAND: *funcP = logic_and_vector ; return( OK );
    case LOR: *funcP = logic_or_vector ; return( OK );
    case EQ : *funcP = equal_vector ; return( OK );
    case NEQ: *funcP = nequal_vector ; return( OK );
    case GT : *funcP = gt_vector ; return( OK );
    case GEQ: *funcP = geq_vector ; return( OK );
    case LT : *funcP = lt_vector ; return( OK );
    case LEQ: *funcP = leq_vector ; return( OK );
    default: return( ERR );
  }
}

#if 0		/* attempt to implement run-time installment of operations
		 * also need to modify infix->postfix function ;
		 * an operator should have a priority! */
typedef struct {
  char name;
  int  (**func);
} VECTOROP;
#define MaxVectorOperation 128
VECTOROP VectorOperation[MaxVectorOperation];

void
InstallVectorOperation( name, func )
char name; int (*func)();
{
  if( NVectorOperation >= MaxVectorOperation ) {
    fprintf( stderr, "Sorry.  Cannot install more than %d operations.\n", 
	     MaxVectorOperation );
    finish( 1 );
  }
  VectorOperation[ NVectorOperation ].name =  name;
  VectorOperation[ NVectorOperation ].func = func ;
}

find_operation( name, funcP )
char name; int (**funcP)();
{
  register int i;
  IfErr( name ) return( ERR );
  for(i=0; i < NVectorOperation; i++ ) {
    if( name == VectorOperation[i].name ) *funcP = VectorOperation[i].func;
    return( OK );
  }
  return( ERR );
}
#endif 0

extern BINARY DeBug, yyDebug;
set_yydebug() { yyDebug = ( DeBug == ON )? 1 : 0; }
