;;; PlaNet mode for GNU Emacs
;;; Written by Michael S. Littman (mlittman@Bellcore.COM).
;;; Simple-minded attempt to a major mode for editting PlaNet files.
;;;    So far the indentation rule is that things between "procedure"
;;;    and "end" are indented and things between "if" and "endif" are
;;;    indented.  These commands must occur at the beginning of the
;;;    line. 

; More advanced: Knows about if-do, if-end, if-stop, if-x, if-x;,
; if-endwhile, if-endrepeat.  Doesn't deal with semis in general yet.
; Attempt to deal with semis.
; Attempt to deal with if-else (Y Miyata).

(defvar planet-block-indent 4
  "*Extra indentation applied to blocks.")

(defvar planet-block-commands
  '(("if-then" "endif") ("if-then" "else")
    ("if-do" "endif") ("if-do" "else") ("else" "endif")
    ("procedure" "end") ("while" "endwhile")
    ("repeat" "endrepeat") ("if" 1) ("if-end") ("if-stop") ("if-continue")
    ("if-endwhile") ("if-endrepeat"))
  "*Recognized keywords.  Pairs delimit an indented block.  If the
second element in the pair is a 1, only one line is indented
afterwards.  If the second element is nil, nothing is indented.")

(defvar planet-end-command-list
  '(("then") ("do") ("end") ("stop") ("continue") ("endwhile") ("endrepeat"))
  "*Recognized keywords at the end of an 'if' statement.")

(defvar planet-mode-map () 
  "Keymap used in planet mode.")

(if planet-mode-map
    ()
  (setq planet-mode-map (make-sparse-keymap))
  (define-key planet-mode-map "\r" 'planet-newline)
  (define-key planet-mode-map "\t" 'planet-indent-current-line)
  (define-key planet-mode-map "#" 'planet-electric-comment))

; This routine adapted from fortran.el
(defun planet-mode ()
  "Major mode for editing planet code.
Tab indents the current planet line correctly. 

Variables controlling indentation style and extra features:

 planet-block-indent
    Extra indentation within blocks.  (default 4)

Turning on PlaNet mode calls the value of the variable planet-mode-hook 
with no args, if that value is non-nil.
\\{planet-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (make-local-variable 'require-final-newline)
  (setq require-final-newline t)
  (use-local-map planet-mode-map)
  (setq mode-name "PlaNet")
  (setq major-mode 'planet-mode)
  (run-hooks 'planet-mode-hook))

(defun planet-indent-current-line ()
  "Indent current line according to previous and current lines."
  (interactive)
  (let ((old-indent 0) (old-command-string nil) new-indent new-command)
    (setq new-command (save-excursion
			(beginning-of-line) (planet-command-on-line t)))
    (save-excursion
      (beginning-of-line)
      (if (not (planet-enclosing-statement)) ()	
	(setq old-indent (planet-indentation-of-command))
	(setq old-command-string (planet-command-string))))
    (setq new-indent
	  (+ old-indent (planet-compute-indentation-difference
			 old-command-string new-command)))
    (setq new-indent (max new-indent 0))
    (planet-do-indentation new-indent)))

(defun planet-start-command-p (cmd)
  "Checks if CMD is a car in the block list.  Returns matching end."
  (let ((a (assoc cmd planet-block-commands)))
    (if a (car (cdr a)))))

(defun planet-end-command-p (cmd)
  "Checks if CMD is a cadr in the block list.  Returns matching start."
  (if (or (equal cmd 1) (equal cmd nil)) nil
    (let ((end-p nil)
	  (block-commands planet-block-commands))
      (while block-commands
	(if (equal cmd (car (cdr (car block-commands))))
	    (progn
	      (setq end-p (car (car block-commands)))
	      (setq block-commands nil))
	  (setq block-commands (cdr block-commands))))
      end-p)))

(defun planet-command-on-line (&optional quick-p)
  "Return a string of the first word on the current line.  Without
optional QUICK-P, carefully exams if statements for syntactic clues."
  (save-excursion
    (let ((command
	   (progn
	     (planet-beginning-of-statement)
	     (if (re-search-forward 
		  "[ \t]*\\([a-zA-Z0-9_]*\\)" (point-max) 0)
		 (buffer-substring (match-beginning 1) (match-end 1))
	       ""))))
      (if (or quick-p (not (string-equal command "if")))
	  command
	; Ifs... -do, -end, -stop, -endwhile, -endrepeat
	(let ((end-command
	       (if (re-search-forward
		    "\\([a-zA-Z0-9_]*\\)[ \t]*[;\n#]" (point-max) 0)
		   (buffer-substring (match-beginning 1) (match-end 1))
		 "")))
	  (if (assoc
	       end-command
	       planet-end-command-list)
	      (setq command (concat command "-" end-command)))
	  command)))))

(defun planet-indentation-of-command ()
  "Returns the column number of the command."
  (save-excursion
    (planet-beginning-of-statement)
    (skip-chars-forward " \t")
    (current-column)))
    
(defun planet-do-indentation (COL)
  "Deletes initial whitespace and does an `indent-to.'  Problem is,
where should the cursor end up?"
  (let (start)
    (save-excursion
      (beginning-of-line)
      (setq start (point))
      (skip-chars-forward " \t")
      (delete-region start (point))
      (indent-to COL)))
  (if (bolp) (skip-chars-forward " \t")))

(defun planet-newline (ARG)
  "Adjust indentation before newline."
  (interactive "P")
  (planet-indent-current-line)
  (newline ARG)
  (planet-indent-current-line))

(defun planet-compute-indentation-difference (old-com-str new-com)
  "Returns the relative indentation of the NEW-COM.  OLD-COM-STR is
used for displaying."
  (let ((end-com (planet-end-command-p new-com)))
    (if (not end-com)
	(if old-com-str planet-block-indent 0)
      (if old-com-str (message "Matches %s" old-com-str))
      0)))

(defun planet-enclosing-statement ()
  "Puts cursor on line which encloses original line."
  (let ((mismatch nil)
	(command-stack 0)
	(lines-processed 0)
	start end cmd)
    (while (and command-stack (not mismatch))
      (setq cmd (planet-command-on-line))
      (setq start (planet-start-command-p cmd))
      (setq end (planet-end-command-p cmd))
      (cond
       (start ; command-start, try to pop.
	(cond
	 ((and (equal 0 command-stack)  ; Procedures always COL 0
	       (string-equal "procedure" cmd))
	  (setq command-stack (list nil)) ; (hack)
	  (setq mismatch t))
	 ((equal 0 command-stack) ; First command
	  (setq command-stack (list nil)))
	 ((and (equal start 1) (= 1 lines-processed)) ; if-x
	  (setq command-stack nil)) ; assume match to close.
	 ((equal start 1) ; close if-x
	  )
	 ((equal start (car command-stack)) ; Pop!
	  (setq command-stack (cdr command-stack)))
	 ((equal nil (car command-stack)) ; Done!
	  (setq command-stack nil))
	 (t (setq mismatch t)))) ; Bad pop
       (end ; command-end, push
	(if (equal command-stack 0) (setq command-stack nil))
	(setq command-stack (cons cmd command-stack)))
       (t ; other command
	(if (equal command-stack 0)
	    (setq command-stack (list nil)))))
      (if (and (string-equal "procedure" cmd) ; Procedure ends the search
	       (equal nil (car command-stack)))
	  (setq mismatch (not (null command-stack))))
      (if (and command-stack (not (planet-previous-statement)))
	  (setq mismatch t))
      (setq lines-processed (+ lines-processed 1)))
    (if (and (= (length command-stack) 1)
	     (equal (car command-stack) nil))
	nil
      (if mismatch (progn (beep) (message "Mismatch!") nil) t))))

(defun planet-command-string ()
  "Returns a string of the line at point."
  (save-excursion
    (planet-beginning-of-statement)
    (if (re-search-forward
	 "[ \t]*\\([^;\n]*\\)" (point-max) 0)
	(buffer-substring (match-beginning 1) (match-end 1))
      "(match error)")))

(defun planet-beginning-of-statement ()
  "Move cursor to beginning of statement (before whitespace).  Skips
comments."
  (re-search-backward ; Skip over comment.
   "#" (save-excursion (beginning-of-line) (point)) t)
  (if (or (= (point) (point-min))
	  (save-excursion (forward-char -1)
			  (looking-at "[;\n]")))
      ()
    (re-search-backward "[;\n]" (point-min) 1)
    (forward-char 1)))

(defun planet-previous-statement ()
  "Move cursor to beginning of previous statement.  Return nil if at
first command in buffer."
  (planet-beginning-of-statement)
  (if (= (point) (point-min))
      nil
    (re-search-backward "[^ \t\n]" (point-min) 1)
    (planet-beginning-of-statement)
    t))

; Basically works.  Slow.

(defun planet-electric-comment (ARG)
  "Looks for comment on previous line and attempts to indent to match it."
  (interactive "P")
  (let* ((cur-col (current-column))
	 (cur-point (point))
	 (prev-col
	  (save-excursion
	   (beginning-of-line)
	   (forward-line -1)
	   (re-search-forward "#" cur-point t)
	   (current-column))))
    (indent-to (- prev-col 1))
    (insert-char ?# (or ARG 1))))
