#ifndef _DO_H
#define _DO_H

/* Copyright (C) 1991, 1992, 1993 by David B. Rosen.  All rights reserved. */

/*

do.h (See change log below for version number.)

  nrepeat(n, body)
   is a statement that executes body n times serially. Note that n
   is simply an expression that tells nrepeat how many times to 
   execute the body.  There is no loop variable available within the 
   loop. If you need to use a loop variable, use one of the macros 
   below instead. 

  doup(ivar,n1,n2,body)
   is a statement that executes body serially, with ivar
   successively taking on the ascending values n1, n1+1, ..., n2.

  dodown(ivar,n1,n2,body)
   is a statement that executes body serially, with ivar
   successively taking on the descending values n1, n1-1, ..., n2.

  do0(ivar,n,body)
   is a statement that executes body n times serially, with ivar
   successively taking on the descending values n-1, n-2, ... 0.   If
   you want the values of ivar to ascend, use doup instead.

  do1(ivar,n,body)
   is a statement that executes body n times serially, with ivar
   successively taking on the descending values n, n-1, n-2, ... 1.  If
   you want the values of ivar to ascend, use doup instead.

For all of the loop macros described above:
   o  The loop variable you name (ivar above) is defined by the macro as a
       register int local to the loop.  Do not declare it yourself. Do not
       modify its value, as the result of doing so should be considered
       unpredictable.
   o  The macro itself is syntactically a single statement. This means you
       should normally follow it by a semicolon, as you would any statement
       that does not end with ``}''.
   o  LIMITATION:  The body may not contain any commas except enclosed within
       parentheses () or within a string literal, otherwise a(n ANSI, but 
       perhaps not a traditional,) C preprocessor will think these delimit
       additional (spurious) arguments for the loop macro!
   o  The count (n) or range of values (n1 and n2) are expressions (rvalues)
       that are each evaluated only once,  before entering the loop the
       first time.  They can be of any type that can be implicitly converted
       to int, but do not use an expression that may be out of range of a
       (signed) int.
   o  break and continue may be used in the normal way within these loops.
   o  A final semicolon is automatically appended to the ``body'', so you need
       not supply one yourself, though you may if you wish.
   o  The body is enclosed directly in {} by the macro, so the body may, e.g.,
       begin with local declarations and definitions. Because you can't use
       commas (see above), use e.g. ``int i; int j;'' instead of ``int i, j;''.
   o  Problem:  The source line number reported in case of error
       anywhere within the body may reflect the first line of the macro
       rather than the line where the error occurred, depending on your
       preprocessor and compiler.
   o  Problem: C-specific editor modes and pretty-print software will 
       probably indent the loop as a function with arguments rather than 
       as a control structure with a body.

Note: do0 and do1 (and nrepeat) are likely to run slightly faster
than doup or dodown.  This is because doup and dodown have to
compare ivar to n2 every time through the loop, while do0 and do1 need
only test the sign of ivar.

For each of the macro names above, appending the letter `L' (do0L,
nrepeatL, etc.) gives a version of the macro that uses a long loop
variable instead of int.

*/

#if 0 /* example, so it can contain comments */

Example:
  #define N 100
  int a[N];
  ...
  ...
  nrepeat(5,
    /* body to be repeated 5 times */
    do0(i,N,    /* note that we needn't declare i */
      /* body to be repeated with i = N-1, N-2, ..., 2, 1, 0 */
      if (a[i] > 0 )
        func1(a[i]);
      else 
        func2(a[i]);
      doup(j, i, N-1, func3(a[i], a[j]) ); /* we chose to omit `;' in body */
    ); /* end of do0 i */
  ); /* end of nrepeat */

#endif /* example */

/*
To do:

  * implement do0L, nrepeatL, etc. using long loop variable.
  (currently only implemented on machines that happen to have int==long.)

*/

/* End of external specification and user documentation. 
   Implementation follows. */

/* 

Note: each macro is enclosed in a do {...} while(0) to make it syntactically
   a single statement, so as not to screw up if statements, etc. 

 Change log:

   Version 1.02:

   16 Mar 1993: Implemented L-suffix (long) portably.

   Version 1.01:

   02 Feb 1993: Added long (L-suffix) versions to documentation;
   implemented for int==long machines only.

   Version 1.0:

   18 Nov 1992 DBR - Improved documentation above.


   27 Oct 1992 DBR - Isolated and documented comma problem.  
                     Fixed buggy comma within doup and dodown macro defs.
   Version 0.8:

   1991
*/

#define do1(ivar,n,body) _do1Type(int,ivar,n,body)
#define do0(ivar,n,body) _do0Type(int,ivar,n,body)
#define doup(ivar,n1,n2,body) _doupType(int,ivar,n1,n2,body)
#define dodown(ivar,n1,n2,body) _dodownType(int,ivar,n1,n2,body)
#define nrepeat(n,body) _nrepeatType(int,n,body)

#define do1L(ivar,n,body) _do1Type(long,ivar,n,body)
#define do0L(ivar,n,body) _do0Type(long,ivar,n,body)
#define doupL(ivar,n1,n2,body) _doupType(long,ivar,n1,n2,body)
#define dodownL(ivar,n1,n2,body) _dodownType(long,ivar,n1,n2,body)
#define nrepeatL(n,body) _nrepeatType(long,n,body)

#define _do1Type(type,ivar,n,body) \
do { \
    register type (ivar)=((n)+1); \
    while ( --(ivar) > 0 ) \
      { \
	  body; /* include final ';' so user needn't for single-stmt body */ \
      } \
    } while(0)

#define _do0Type(type,ivar,n,body) \
do { \
    register type (ivar)=(n); \
    while ( --(ivar) >= 0 ) \
      { \
	  body; /* include final ';' so user needn't for single-stmt body */ \
      } \
    } while(0)

#define _doupType(type,ivar,n1,n2,body) \
do { \
    register type (ivar)=(n1)-1; register type _doup_final_=(n2); \
    while( (ivar)++ < _doup_final_ ) \
      { \
	  body; /* include final ';' so user needn't for single-stmt body */ \
      } \
    } while(0)

#define _dodownType(type,ivar,n1,n2,body) \
do { \
    register type (ivar)=(n1)+1; register type _doup_final_=(n2); \
    while( --(ivar) >= _doup_final_ ) \
      { \
	  body; /* include final ';' so user needn't for single-stmt body */ \
      } \
    } while(0)

#define _nrepeatType(type,n,body) _do0Type(type, _i_nrepeat_, (n), body)

#endif /* _DO_H */
