/* wave.cc: a program to produce relative spectral sensitivity
   arrays for the three cone pigment types.
*/

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <math.h>
void exit (int n);

#ifdef __cplusplus
}
#endif

#include "ncomp.h"
#include "ncsub.h"

/* double log(double), exp(double); */
double lamm;

#define LN10	2.30258509299404568402
#define LAMM 561.0

void calcwav(int pigm);
void wavout(double val);
void arrfill (double val, int pigm);

static int p=0;

double pigmarr[NUMREC][PIGMSIZ] = {0};

double lumlight [LUMINS][LIGHTS] = {0};

double pigmavg[NUMREC][LIGHTS] = {0};

				/* values taken from Bowmaker et al 1980 */
				/* log10 of abs sensitivity at lmax: */
double pigmconst[NUMREC] = {	
				-0.19031465,	/* rod log10(1-exp10(-.450)) */
				-0.153996,	/* red                .525   */
				-0.153996,	/* green	      .525   */
				-0.23784418	/* blue		      .375   */
			    } ;	
double pigmlen[NUMREC] = {	
				25.0,		/* rod path length */
				35.0,		/* red cone path length */
				35.0,		/* green cone path length */
				25.0,		/* blue cone path length */
			    } ;	
double lights[LIGHTS][PIGMSIZ] = {		/* standard light inten */
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0,
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  	/* sun */
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0,
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  	/* xenon */
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0,
	};

double luminos[LUMINS][PIGMSIZ] = {		/* standard luminosities */

	/* scotopic lumonosity function (Wyszecki and Stiles, 1982) */

  5.89e-4, 1.108e-3, 2.209e-3, 4.53e-3, 9.29e-3, 1.852e-2, 3.484e-2, 6.04e-2,
  9.66e-2, 0.1436, 0.1998, 0.2625, 0.3281, 0.3931, 0.4550, 0.5130,
  0.5670, 0.6200, 0.6760, 0.7340, 0.7930, 0.8510, 0.9040, 0.9490,
  0.9820, 0.9980, 0.9970, 0.9750, 0.9350, 0.8800, 0.8110, 0.7330,
  0.6500, 0.5640, 0.4810, 0.4020, 0.3288, 0.2639, 0.2076, 0.1602,
  0.1212, 8.99e-2, 6.55e-2, 4.69e-2, 3.315e-2, 2.312e-2, 1.593e-2, 1.088e-2,
  7.37e-3, 4.97e-3, 3.335e-3, 2.235e-3, 1.497e-3, 1.005e-3, 6.77e-4, 4.59e-4,
  3.129e-4, 2.146e-4, 1.480e-4, 1.026e-4, 7.15e-5, 5.01e-5, 3.533e-5, 2.501e-5,
  1.780e-5, 1.273e-5, 9.14e-6, 6.60e-6, 4.78e-6, 3.482e-6, 2.546e-6, 1.870e-6,
  1.379e-6, 1.022e-6, 7.60e-7, 5.67e-7, 4.25e-7, 3.196e-7, 2.413e-7, 1.829e-7,
  1.390e-7, 1.01e-7, 7.7e-8, 5.7e-8, 4.3e-8,
	
	/* photopic lumonosity function (Wyszecki and Stiles, 1982) */

	0.0000, 0.0000, 0.0001, 0.0003, 0.0004, 0.0009, 0.0012, 0.0028,
	0.0040, 0.0080, 0.0116, 0.0174, 0.0230, 0.0305, 0.0380, 0.0490,
	0.0600, 0.0755, 0.0910, 0.1150, 0.1390, 0.1735, 0.2080, 0.2655,
	0.3230, 0.4130, 0.5030, 0.6065, 0.7100, 0.7860, 0.8620, 0.9080,
	0.9540, 0.9745, 0.9950, 0.9950, 0.9950, 0.9735, 0.9520, 0.9110,
	0.8700, 0.8135, 0.7570, 0.6940, 0.6310, 0.5670, 0.5030, 0.4420,
	0.3810, 0.3230, 0.2650, 0.2200, 0.1750, 0.1410, 0.1070, 0.0840,
	0.0610, 0.0465, 0.0320, 0.0250, 0.0170, 0.0130, 0.0082, 0.0062,
	0.0041, 0.0031, 0.0021, 0.0016, 0.0010, 0.0008, 0.0005, 0.0004,
	0.0003, 0.0002, 0.0001, 0.0001, 0.0001, 0.0000, 0.0000, 0.0000,
	0.0000, 0.0000, 0.0000, 0.0000, 0.0000,

	};

double filts[FILTS][PIGMSIZ] = {
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0,
					/* macular pigment */
	0.8222, 0.8222, 0.8222, 0.8222, 0.8222, 0.7586, 0.6918, 0.5957,  
	0.5012, 0.4519, 0.4315, 0.4169, 0.3981, 0.3758, 0.3467, 0.3236,  
	0.3199, 0.3388, 0.3589, 0.3890, 0.3846, 0.3802, 0.3890, 0.4356,  
	0.5309, 0.6383, 0.7413, 0.8222, 0.8913, 0.9441, 0.9772, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,  
	1.0, 1.0, 1.0, 1.0, 1.0
	};

/*------------------------------------*/

main(void)

/* This program prints 6 tables:

   1) the log of specific optical density for rods and the 3 cone types.
   2) the spectral amplitude of 4 light sources.
   3) the spectral transmittance of 2 filters (100% and macular pigment).
   4) the calibration sensitivity of 2 standard luminosity filters (scot, phot)
	with the 4 light sources.
   5) the average optical sensitivity of rods and cones with 4 light sources.
   6) the standard path lengths for rod and cones.

   All spectral sensitivities are calculated from 380 to 800 nm wavelength.

   The log10 of optical density is produced for output because
   it gives a better interpolation than the optical density.

   The original pigment sensitivity data is in the form of
   log10 sensitivity vs. wavelength.
   To get from sensitivity to optical density, use the following
   derivation:

   light absorbed = absolute sensitivity;
   transmittance = 1 - light absorbed
   optical density = - log10 ( transmittance )
   specific density * path length = optical density
   absolute sensitivity = rel sens * (1 - exp10 ( - recep axial density ))
   log10 (abs sens) = log10 (rel sens) + log10 (1-exp10(-recep axial dens))
 
   optical density = - log10 ( 1 - sensitivity )

   optical density = - log10 ( 1 - exp10 ( log10 ( sensitivity ) ) )

   specific density = -log10 ( 1 - exp10 ( log10(sensitivity) ) ) / path length

*/
   
{
   int i,j,k,w;
   double tot,val,logod,specod,lumtot,lighttot;

				/* calculate pigment sens curves */
  calcwav(0);
  calcwav(1);
  calcwav(2);
  calcwav(3);

				/* calculate the calibrations */
				/*  for phot and scot luminosities */
				/*  with different light sources */
  for (i=0; i<LUMINS; i++) {	
    for (j=0; j<LIGHTS; j++) {	
      tot=0.0;
      for (w=0; w<PIGMSIZ; w++) {
        tot += luminos[i][w] * lights[j][w];	/* sum calib sensitivities */
      }
      lumlight[i][j] = tot/PIGMSIZ;		/* normalize sum */
    }
  }

				/* calculate the average sensitivity */
				/*  for standard light source */
  for (i=0; i<NUMREC; i++) {	
    for (j=0; j<LIGHTS; j++) {	
      tot = 0.0;
      for (w=0; w<PIGMSIZ; w++) {
        tot += exp(LN10*pigmarr[i][w]) * lights[j][w];  /* sum sensitivities */
      }
      pigmavg[i][j] = tot/PIGMSIZ * exp(pigmconst[i]*LN10); /* normalize sum */
    }
  }

				/* print the pigment table */

  printf ("float specdens[NUMREC][PIGMSIZ] = {\n    ");
  p=0;
  for (i=0; i<NUMREC; i++) {	
    p=0;
    for (w=0; w<PIGMSIZ; w++) {
	val = pigmarr[i][w] + pigmconst[i];		/* abs sensitivity */
        specod = (-log(1.0-exp((val)*LN10))/LN10) / pigmlen[i];
	wavout(log(specod)/LN10);
    }
    printf ("\n    ");
  }
  printf ("};\n");
						/* print the light table */
  printf ("float lights[LIGHTS][PIGMSIZ] = {\n    ");
  p=0;
  for (i=0; i<LIGHTS; i++) {	
    p=0;
    for (w=0; w<PIGMSIZ; w++) {
	wavout(lights[i][w]);
    }
    printf ("\n    ");
  }
  printf ("};\n");

						/* print the filter table */
  printf ("float filts[FILTS][PIGMSIZ] = {\n    ");
  p=0;
  for (i=0; i<FILTS; i++) {	
    p=0;
    for (w=0; w<PIGMSIZ; w++) {
	wavout(filts[i][w]);
    }
    printf ("\n    ");
  }
  printf ("};\n");

				/* print the luminosity calibration table */
  printf ("\n\n");

  printf ("float lumlight[LUMINS][LIGHTS] = {\n    ");
  for (i=0; i<LUMINS; i++) {	
    for (j=0; j<LIGHTS; j++) {
	val = lumlight[i][j];
        printf ("%8.5g,", val);
    }
    printf ("\n    ");
  }
  printf ("};\n");

				/* print the light source table */
  printf ("\n\n");

  printf ("float lightsens[NUMREC][LIGHTS] = {\n    ");
  for (i=0; i<NUMREC; i++) {	
    for (j=0; j<LIGHTS; j++) {
	val = pigmavg[i][j];
        printf ("%8.5g,", val);
    }
    printf ("\n    ");
  }
  printf ("};\n");


  printf ("float pigmlen[NUMREC] = {\n    ");
  for (i=0; i<NUMREC; i++) {	
	val = pigmlen[i];
        printf ("%8.5g,", val);
  }
  printf ("};\n");
  exit(0);
}

/*------------------------------------*/

double logwav(double w)
{
  return log(1/(w*.001 * LAMM/lamm)) / LN10;
}

/*------------------------------------*/

static double an[7] = {-5.2734, -87.403, 1228.4,
			  -3346.3, -5070.3, 30881.0, -31607.0};
void calcwav(int pigm)
           

/* Procedure to calculate spectral sensitivity
   curves for rod and three cone types.
   Spectral response curve taken from Baylor et al. 1987.
   Beyond 800 nm, the curve is linear with wavenumber,
   (1/w) (see Lewis, 1955 ), and the curve is
   extended in this region using linear extrapolation
   as a funtion of 1/w, using the data for S= -3 to -5,
   using Baylor's slope measurements.  

   The result of this routine is the log10 of sensitivity.
*/

{
    int i,n;
    double m, b, dy, dw, wold;
    double logw, w, sum, x, xx, od, y, y0;
    double logod;

  switch (pigm) {

   case 1: 
         lamm = 561.0;			/* red */
          b = 17200.0;
          m = 0;
          break;
		
   case 2: 
          lamm = 531.0;			/* green */
          b = 15900.0;
          m = 0;
          break;
		
   case 3: 
          lamm = 430.0;			/* blue */
          b = 12700.0;
          m = 0;
          break;
		
   case 0: 
          lamm = 500.0;			/* rod */
          b = 15000.0;
          m = 0;
          break;
	
  }	

  sum = 0;
  for (w=MINWAV; (sum > -5.5) && (w<=MAXWAV); w+= 5.0) { 
    sum = 0;
    x = logwav(w); 
    for (n=0; n<=6; n++) {
       xx = 1.0;
       for (i=0; i<n; i++) {
	 xx *= x;	
       }
       sum += xx * an[n];
    } 
/*    printf ("%5g  %8.3g\n", -1/w, sum);   	   /* log sensitivity */
/*    printf ("%5g  %8.3g\n", w, exp(LN10*sum)*.665);  /* sensitivity */
/*    printf ("%5g  %8.3g\n", w, -log(1.0-exp((sum-0.177)*LN10))/LN10 );  */
/*printf("%5g  %8.3g\n",-1/w,log(-log(1.0-exp((sum-0.177)*LN10))/LN10)/LN10); */
/*    wavout(sum); */
      arrfill(sum,pigm);
  }
  y0 = -27.75;
  y = sum;
  for (wold=w-5; w<=MAXWAV; wold=w, w+= 5.0) {
    dw = 1/w - 1/wold;
    dy = m * 1/(1/lamm - 1/w) + b;
    y += dy * dw; 
/*    y = y0 + -(1/(w*.001 * LAMM/lamm)) * -17.82946;   */
/*    y = y0 + -(1/w) * -17829.46; */
/*    printf ("%5g  %8.3g 1\n", -1/w, y);  */
/*    printf ("%5g  %8.3g\n", w, exp(LN10*y)*.665);  */
/*    printf ("%5g  %8.3g\n", w, -log(1.0-exp((y-0.177)*LN10))/LN10 );  */
/*  printf("%5g  %8.3g\n",-1/w,log(-log(1.0-exp((y-0.177)*LN10))/LN10)/LN10);*/
/*    wavout(y); */
    arrfill(y,pigm);
  } 
}

/*------------------------------------*/

void wavout (double val)
{
  printf ("%8.5g,", val);
  if (++p >= 8) {
     p = 0;
     printf ("\n    ", val);
  }    
}

/*------------------------------------*/

void arrfill (double val, int pigm)
             
           

/* fill pigment array */

{
  static int i,pg= -1;

  if (pg==pigm) {
    pigmarr[pigm][i++] = val;
  }
  else {
    pg = pigm;
    i = 0;
    pigmarr[pigm][i++] = val;
  } 
}
