/* Module ncsub in Program nc */

/* Models neuronal circuits */

/* Simulates a neuron with a compartmental model.
It assumes a dendrite is made up of small isopotential
compartments which are modeled as electrical nodes.
Each node is connected to its neighbors through axial 
resistances, and has conduction pathways to ground
through membrane resistance and capacitance.
Additional conduction pathways are allowed for any node. 

Unless otherwise noted, all length values are calibrated
in meters, all voltages are in volts, all currents are
in amperes, and capacitance is in farads.

        Oct 87                  R.G. Smith

*/

#include <stdio.h>
#include <math.h>
#include "adef.h"
#include "nc.h"
#include "y.tab.h"
#include "control.h"
#include "ncsub.h"
#include "ncelem.h"
#include "ncomp.h"

#ifdef __cplusplus
extern "C" {
#endif

#include "gr.h"

#ifdef __cplusplus
}
#endif

#define HMICSEC 0.0001

#define DEBUG

extern int cumcomp;
extern int cumconn;
extern int cumnode;
extern int cumsynap;
extern int cumrec;
extern int setdisp;
extern int setprmap;
extern int disp_ray;
extern double setlamcrit;
extern int pinitfl;			/* =1 -> plotinit() run */
extern synap *synpnt;
extern comp *compnt;
extern recep *recpnt;
extern recstim *recspnt;
extern recstim *recsend;
extern node *nodepnt;
extern elem *elempnt;
extern elem *elemend;
static elem *oelemend = 0;

extern Symbol *timeptr;
extern Symbol *ncompptr;

int iflag;                              /* interrupt flag set by onintr()  */

plotfr plotnod[PLOTNODSIZ] = {0};       /* node numbers, vmin, vmax voltages */

int numplots = 0;                       /* number of plots displayed */
int runyet=0;
int nocond = 0;				/* no condense */

double record(int cnod1, int cnod2, int cnod3, int pmod);
double getcurv(synap *spnt, double val);
double rsens(recep *rpnt, double wavel, int filt);
double ncabs(double x);
double lkexp(double val);
void initcomp();
void condense();
void prcomp();
void chksmall();
void compcon();
void maktables(double timinc);
void runmod(double timestep);
void plotinit(int numplots);
void runstim(double xtime, int nsteps);
void runsyn(int nsteps);
void runrec(int t); 
double getrecval(int i);
void mplot(double val, double time, int numplots, int i);
void runcomp();
node *findnode(short int num1, short int num2, short int num3, char *str);
int findrecep (int cnod1, int cnod2, int cnod3, recep **rpnt, char *s);
void execerror (char *s, char *t);
void getarg(int narg, double **val, short **vtyp);   
void rchan(synap *sdyad, int stime, double chanligand);
int pois(double m);
int readstim(double rtime);
void delrstim(recstim *rspnt);
conn *printconns(comp *pnt, int nprint);
comp *othercomp (comp *pnt,conn *cpnt);
char *prnode (int n1, int n2, int n3); 

/*------------------------------------*/

void actrun (double timestep)
{
  static int ovidmode=0;

  if (setdisp>=0) disp = setdisp;	/* "-d n" overrides "disp=n" */
  if (setprmap>=0) prmap = setprmap;	/* "-p n" overrides "prmap=n" */
  if (setlamcrit>=0) lamcrit=setlamcrit; /* "-l n" overrides "lamcrit=n" */

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"actrun start %g\n",timestep);
  if (debug & 256) nocond = 1;
#endif

    if (!runyet) {
      initcomp();         /* make compartments */
      if (!nocond) condense();
      ncompptr->val = cumcomp;	/* set number of comps for user */
      if (!vidmode && !disp_ray) {
          if (ovidmode) ctext();
	  if      (cumcomp> 1) printf ("#c %d comps",cumcomp);
          else if (cumcomp==1) printf ("#c %d comp",cumcomp);
	  if      (cumnode> 1) printf (", %d nodes",cumnode);
          else if (cumnode==1) printf (", %d node",cumnode);
	  if      (cumconn> 1) printf (", %d connections",cumconn);
          else if (cumconn==1) printf (", %d connection",cumconn);
	  if      (cumsynap> 1) printf (", %d synapses",cumsynap);
          else if (cumsynap==1) printf (", %d synapse",cumsynap);
	  if      (cumrec> 1) printf (", %d photoreceptors",cumrec);
          else if (cumrec==1) printf (", %d photoreceptor",cumrec);
          printf (".\n");
	  fflush (stdout);
      }
      else if (!ovidmode && !disp_ray) cgraphics(); 
      if (prmap & (PCOND|PCOMP)) prcomp();
/*    if (disp & (DCOMP|DCONN))  dcomp(); */
      chksmall();			/* check for small compartments */
      compcon();                  /* set up time interval in compartment */
  }
  else {         /* runyet == 1 */
      if (elemend != oelemend) {  /* if any new neural elements */
        initcomp();               /* make new compartments */
        compcon();                /* set up time interval in compartment */
        if (!nocond) condense();
        if (prmap & (PCOND|PCOMP)) prcomp();
/*      if (disp & (DCOMP|DCONN))  dcomp(); */
      }
      if (vidmode) {
        if (!ovidmode) cgraphics(); 
      }
      else 
        if (ovidmode)  ctext();
    }
  maktables(timinc);		/* make new Na, K rate tables */
  runmod(timestep);           /* run compartments, synapses and receptors */
  ovidmode = vidmode;
  oelemend = elemend;

}
 
/*------------------------------------*/

void chksmall(void)

/* Check for possible errors in constructing compartments. */

{
   comp *pnt,*pnt2;
   node *npnt,*onpnt;
   int found;
   conlst *lpnt;
   conn *cpnt;

  for (pnt=compnt; pnt; pnt=pnt->next) {
    if (pnt->rm > (LARGERES * .9) || pnt->cap < SMALLCAP * 1.1) {
       printf("# Warning: possible error:\n");
       printf("# Compartment # %u is missing: ",pnt->num);
       printf(" gm %6.3g, cap %6.3g",pnt->rm,pnt->cap);
       for (found=0,lpnt=pnt->nodlst; lpnt; lpnt=lpnt->next) {
            if (npnt=(node*)lpnt->conpnt) {
                printf (" at node %s.\n",
			prnode(npnt->nodenm1,npnt->nodenm2,npnt->nodenm3));
                 found = 1;
            }
       }
       if (!found) 
         for (found=0,npnt=nodepnt; npnt; npnt=npnt->next) {    /* find node */
           if (npnt->comptr == pnt) { 
                printf (" at node %s.\n",
			prnode(npnt->nodenm1,npnt->nodenm2,npnt->nodenm3));
	       found = 1;
               break; 
           }
       }
      if (found) 
         onpnt = npnt;
      else {
         printf ("# Small comp without node. ");
         onpnt = NULL;
      }
      printf ("# Check: \"");
      if (cpnt=printconns(pnt,1)) {
        pnt2 = othercomp (pnt,cpnt);		/* get other compartment */
        if (pnt2) {				/*  find its nodes */
          printf ("\" which connects");
           for (found=0,lpnt=pnt2->nodlst; lpnt; lpnt=lpnt->next) {
            if (npnt=(node*)lpnt->conpnt) {
                printf (" to node %s.\n",
			prnode(npnt->nodenm1,npnt->nodenm2,npnt->nodenm3));
                 found = 1;
            }
         }
        }  /* if pnt2 */
       }
       else printf ("\"\n");
       if (onpnt) printf ("# Node %s",
			prnode(onpnt->nodenm1,onpnt->nodenm2,onpnt->nodenm3));
       else printf ("# It"); 
       printf (" should also connect to a cable, sphere, or load.\n");
       fflush (stdout);

    }  /*  if (pnt-rm...) */
  }  /* for (pnt=compnt;; ) */
}

/*------------------------------------*/

void runmod(double timestep)
                    

/* Run the simulation; 
   schedule the receptors every msec,
   schedule the synapses to run every 100 microsec,
   and run the compartments every "timestep".
   Plot the nodes specified using "record".
*/

{
  static double ptime, oldtime=-1e6;
  static double stoptime;
  static double plotime;                /* time incr. for approximation */
  static double time5, otime5, time4, otime4;
  static int nsteps;
  extern double endexp;
  int i,newtim;

#ifdef DEBUG
 if (debug & 1) fprintf (stderr,"runmod step %g runyet %d\n",timestep,runyet);
#endif

#define ROUNDUP 0.0000001
#define STIME5 (-1000)

 newtim = (xtime!=oldtime);
 oldtime = xtime;		/* initial time setting from user */
 plotime = ploti;
 if (numplots && !pinitfl)  plotinit(numplots);
 if (timestep <= 0.0) {          /* run mode: run until endexp */
   if (newtim) {
        ptime = plotime;
        otime4 = otime5 = STIME5;
   }
   stoptime = endexp - 1e-10;
 }
 else {                          /* step mode */
   if (newtim) {
        ptime = plotime;
        otime4 = otime5 = STIME5;
   }
   stoptime = oldtime + timestep - 1e-10;
   if (stoptime > endexp) stoptime = endexp;
 }
 

 if ((timinc / HMICSEC) > MAXLONG) {
	timinc = (MAXLONG + STIME5) * HMICSEC;
	printf ("# runmod: 'timinc' is too large, limited to %9.5g\n",timinc);
 }
 
 for (xtime=oldtime; xtime<stoptime; xtime+=timinc, ptime+=timinc)
  {
    if (debug & 1) fprintf (stderr,"runmod: time %g\n",xtime);

    time5 = floor (xtime / HMICSEC + ROUNDUP);
/* fprintf (stderr,"time5 %g otime5 %g time %30.30g\n",time5,otime5,xtime); */
    if (time5 > otime5) {
        nsteps = (int)(time5 - otime5);	/* calculate steps actually taken */
        otime5 = time5;
	nsteps = 1;
        runstim(xtime,nsteps);      	/* run stimuli */
        runsyn(nsteps);			/* run synapses  */
        time4 = floor (time5 / 10);
/* fprintf (stderr,"time4 %g otime4 %g\n",time4,otime4); /* */
        if (time4 > otime4) {
                otime4 = time4;
                runrec(1);              /* run receptors */
        }
    }
    if (ptime >= plotime)
     {
      ptime = 1e-8; 
      for (i=0; i<numplots; i++)                /* display plots */
           if (plotnod[i].pmod != GREC)
		mplot (getrecval(i),xtime,numplots,i);
    }
    runcomp();                                  /* run compartments */

    if (iflag)
     {
       iflag = F;
       break;                           /* stop on interrupt */
     }
  }             /* for (xtime;;) */

  if (stoptime + 2e-10 >= endexp) {
   ptime = 1e-8;
   for (i=0; i<numplots; i++)           /* display plots one last time */
      if (plotnod[i].pmod != GREC)
          mplot (getrecval(i),xtime,numplots,i);
 }
 oldtime = xtime;                       /* save time for next runmod() */
 if (xtime >= endexp) runyet=0;         /* reset run flag for init */
 else                runyet=1;
 timeptr->val = xtime;                  /* save correct time for user */
 timeptr->vtype = NUMBER; 

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"runmod end\n");
#endif
}


/*------------------------------------*/

double record (int cnod1, int cnod2, int cnod3, int pmod)
{
  node *npnt;
  recep *rpnt;
  double rval, recsynap(int snum, int nf), recchan(int cnum, int nf);
  comp *cpnt;

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"record node %d %d %d mode %d",
					cnod1,cnod2,cnod3,pmod);
#endif

   rval = 0.0;
   switch (pmod) {

   case VREC: if (!(npnt=findnode(cnod1,cnod2,cnod3,"record"))) return(0.0);
	      if (npnt->comptr) rval = npnt->comptr->v;
              break;

   case IREC: if (!(npnt=findnode(cnod1,cnod2,cnod3,"record"))) return(0.0);
              if ((cpnt=npnt->comptr)) {
     	         if (cpnt->miscfl & VEXT)             /* comp is v-clamped */
                     rval = -cpnt->extvi;
	      else if (cpnt->miscfl & IEXT)           /* comp is i-clamped */
                     rval = cpnt->exti;
	      }
              break;

   case LREC: if (findrecep (cnod1, cnod2, cnod3, &rpnt, "record"))
                    rval = rpnt->flux;
              break;

   case NRECA0: case NRECA1: case NRECA2: case NRECA3: case NRECA4: 
   case NRECA9:
   case NRECB0: case NRECB1: case NRECB2: case NRECB3: case NRECB4:
   case NRECC0: case NRECC1: case NRECC2: case NRECC3: case NRECC4:
	  rval = recsynap(cnod1,pmod);
          break;

   case NRECG0: case NRECG1: case NRECG2: case NRECG3: case NRECG4:
   case NRECG5: case NRECG6: case NRECG7:
   case NRECG8:
	 rval = recchan(cnod1,pmod);
         break;

   }    /* switch */

#ifdef DEBUG
  if (debug & 1) fprintf (stderr," rval %g ...end\n",rval);
#endif
 
 return (rval);
}

/*------------------------------------*/

double recsynap(int snum, int nf)
           
/* Record from a synapse.
    Partly copied from modsynap() in ncmak.c.
*/

{
    elem *elpnt;
    synap *spnt;

    for (elpnt=elempnt; elpnt; elpnt=elpnt->next) {
      if (elpnt->elnum == snum) break;
    }
    if (elpnt==NULL) {
       fprintf (stderr,"recsyn: can't find element %d\n",snum);
       return (0.0);
    }
    spnt = (synap *)elpnt->lptr;	/* get pointer to synapse from elem */
    if (spnt==NULL) {
       fprintf (stderr,"recsyn: can't find synapse for element %d\n",snum);
       return 0.0;
    }
    if (spnt->ctype != SYNAPSE) {
       fprintf (stderr,"recsyn: element %d is not a synapse\n",snum);
       return (0.0);  
    }
    if (nf < NRECA9) {			/* look at filters before nt release */
      nf = nf - NRECA0;			/* look at any of the NUMFILT filters */
      if      (nf<0) nf = 0;
      else if (nf>=NUMFILT) nf = NUMFILT-1;
      if (spnt->filt1) {
/* fprintf (stderr,"f0 %d snum %d nf %d\n",spnt->filt1->lfilt[nf],snum,nf); */

        return (double)spnt->filt1->lfilt[nf];	/* voltage before filter */
      }
      else return 0.0;
   }
   else if (nf == NRECA9) {		/* look at vesicle rate */
	double trvolts,transrate;

      trvolts = (spnt->comp1->v - spnt->thresh); /* calib in volts */
      transrate = getcurv(spnt,trvolts);
      if (transrate < TRMIN) transrate = TRMIN;
      else if (transrate > TRMAX) transrate = TRMAX;
      return transrate/(spnt->vsize*HMICSEC);
   }
   else if (nf < NRECC0) {		/* look at filters after nt release */
      nf = nf - NRECB0;			/* look at any of the NUMFILT filters */
      if      (nf<0) nf = 0;
      else if (nf>=NUMFILT) nf = NUMFILT-1;
      if (spnt->filt2) {
/* fprintf (stderr,"f0 %d snum %d nf %d\n",spnt->filt2->lfilt[nf],snum,nf); */
       return (double)spnt->filt2->lfilt[nf];	/* neurotrans from filter */
      }
      else return 0.0;
   }
   else {	/* nf >= NRECC0 */	/* look at filters after nt sat. */
      nf = nf - NRECC0;			/* look at any of the NUMFILT filters */
      if      (nf<0) nf = 0;
      else if (nf>=NUMFILT) nf = NUMFILT-1;
      if (spnt->filt3) {
/* fprintf (stderr,"f0 %d snum %d nf %d\n",spnt->filt3->lfilt[nf],snum,nf); */
        return (double)spnt->filt3->lfilt[nf];	/* neurotrans from filter */
      }
      else {
         double c;

        if (c=spnt->maxcond) 
	     return (spnt->conduct/c);
	else return  spnt->conduct;
      }
   }
}

/*------------------------------------*/

double recchan(int cnum, int nf)
				    /* cnumm = element number of channel */

/* Record from a channel.
    Partly copied from recsynap() above.
*/

{
    elem *elpnt;
    sschan *chpnt;
    attrib *apnt;
    cacomp *capnt;
    int ncond;
    double cval;
    synap *spnt;

 for (elpnt=elempnt; elpnt; elpnt=elpnt->next) {
      if (elpnt->elnum == cnum) break;
 }
 if (elpnt==NULL) {
       fprintf (stderr,"recchan: can't find element %d\n",cnum);
       return (0.0);  
 }
 if (elpnt->ctype==SYNAPSE) {	/* record post-synaptic conductance */
    spnt = (synap *)elpnt->lptr;	/* get pointer to synapse from elem */
    if (spnt==NULL) {
       fprintf (stderr,"recsyn: can't find synapse for element %d\n",cnum);
       return 0.0;
    }
    if (spnt->ctype != SYNAPSE) {
       fprintf (stderr,"recsyn: element %d is not a synapse\n",cnum);
       return (0.0);  
    }
   if (nf==NRECG0) {		/* total channel conductance */
     cval = spnt->conduct;
   }
 }

 else {			/* record channel conductances */
  if ((apnt=elpnt->attpnt)==NULL) {
  fprintf (stderr,"recchan: can't find channel attrib for element %d\n",cnum);
       return 0.0;
  }

  if ((chpnt=(sschan *)apnt->lptr)==NULL) { /* get pnt to channel from elem */
       fprintf (stderr,"recchan: can't find channel for element %d\n",cnum);
       return 0.0;
  }

/*  fprintf(stderr,"chan type %d stype %d nf %d\n",
		chpnt->ctype,chpnt->stype,nf); /* */

  switch (chpnt->ctype) {	/* stop short on wrong element */
     case HH:
     case NA:
     case K:
     case CA: break;
     default:
       fprintf (stderr,"recchan: element %d is not a channel (%d)\n",
					cnum,chpnt->ctype);
       return (0.0);  
  }

  if (nf==NRECG0) {		/* total channel conductance */
     cval = chpnt->conduct;
  }
  else {    
      double v;

    ncond = nf - NRECG0;	/* conductance number */
    if (ncond > 9) ncond = 9;
    else if (ncond < 1) ncond = 1;

    switch (chpnt->ctype) {
     case HH:
     case NA:
	  switch (chpnt->stype) {
	    case 0:			/* m3h type Na channel */
	 	switch (ncond) {
		    case 1:	cval = ((hhchan *)chpnt)->m; break;
		    case 2:	cval = ((hhchan *)chpnt)->h; break;
		    case 3:	v = ((hhchan *)chpnt)->vrev -
				    ((hhchan *)chpnt)->comp1->v;
			 cval = -((hhchan *)chpnt)->conduct * v; break;
		    default:    cval = 0.0; break;
		}
		break;
	    case 1:
	    case 2:
	    default:
		if (--ncond >=chpnt->numstate) ncond = chpnt->numstate-1;
		cval = chpnt->conc[ncond].cest;
		break;
	  }
 	  break;
     case K:
	  switch (chpnt->stype) {
	    case 0:			/* n4 type K channel */
	    case 2:
	 	switch (ncond) {
		    case 1:	cval = ((hhchan *)chpnt)->m; break; /* n */
		    case 2:	v = ((hhchan *)chpnt)->vrev -	    /* Ik */
				    ((hhchan *)chpnt)->comp1->v;
			 cval = -((hhchan *)chpnt)->conduct * v; break;
		    default:    cval = 0.0; break;
		}
		break;
	    case 1:
	    default:
		if (--ncond >=chpnt->numstate) ncond = chpnt->numstate-1;
		cval = chpnt->conc[ncond].cest;
		break;
	    case 3:
	 	switch (ncond) {
		    case 1:	cval = ((hhchan *)chpnt)->m; break; /* n */
		    case 2:	cval = ((hhchan *)chpnt)->h; break; /* h */
		    case 3:	v = ((hhchan *)chpnt)->vrev -       /* Ik */
				    ((hhchan *)chpnt)->comp1->v;
			 cval = -((hhchan *)chpnt)->conduct * v; break;
		    default:    cval = 0.0; break;
		}
		break;
	  }
 	  break;
     case CA:
	capnt = (cacomp *)chpnt->comp2;
	switch (chpnt->stype) {
        case 0:			/* m3 type Ca channel */
 	switch (ncond) {
	    case 1:	cval = ((hhchan *)chpnt)->m; break;
	    case 2:	cval = capnt->cais[1]; break;
	    case 3:	cval = capnt->cais[2]; break;
	    case 4:	cval = capnt->cais[5]; break;
	    case 5:	cval = capnt->cais[NUMCASH-1]; break;
	    case 6:	cval = capnt->cao;   break;
	    case 7:	cval = capnt->vrev;  break;
	    case 8:	cval = -capnt->ica;   break;
	    case 9:	cval = -capnt->ipump; break;
	    default:    cval = 0.0; break;
		}
		break;
	 case 1:
	 case 2:
	 default:
	    if (--ncond >=chpnt->numstate) ncond = chpnt->numstate-1;
	    cval = chpnt->conc[ncond].cest;
	    break;
	}
 	  break;
    }  /* switch (chpnt->ctype) */
  } /* else nf > NRECG0 */
 } /* else channel conductance */

    return cval;
}

/*------------------------------------*/

double reccable(int elemnum, double fracdist)

/* record from a cable with element number "elemnum",
   at a fractional distance "fracdist" from the first
   end (the end that contacts node 1, the node given first).
*/

{
    int i,n,cnum;
    double r, v, vold;
    node *np1;
    elem *epnt;
    conn *cpnt,*cpnt1,*cpnt2,*tcpnt;
    comp *pnt,*pnt1,*pnt2;
    conlst *lpnt;


 for (epnt=elempnt; epnt; epnt=epnt->next) 	/* search all elems */
    if (epnt->elnum == elemnum) break;

if (epnt==NULL) {
         fprintf (stderr,"reccable: can't find element %d\n",elemnum);
	 execerror ("Missing element: "," stopping... ");
         return 0.0;  
 }
 np1 = epnt->nodp1;
 if (!np1) {
         fprintf (stderr,"reccable: can't find node for element %d\n",elemnum);
         return 0.0;  
 }
 cpnt = (conn *)epnt->lptr;
 if (!cpnt) {
         fprintf (stderr,"reccable: can't find low-lev conn for element %d\n",
							elemnum);
  	 return 0.0;
 } 
 		/* find first node's compartment */

 pnt = np1->comptr;
 if (!pnt) {
      fprintf (stderr,"reccable: can't find comp for node %d\n",np1->nodenm1);
      return 0.0;
 } 
 		/* Find connection from first comp to second comp. */ 
		/* Must be same as connection saved in cable. */

 cpnt1 = NULL;
 for (lpnt=pnt->clst; lpnt && (cpnt1 != cpnt); lpnt=lpnt->next) {
    if (!(cpnt1=lpnt->conpnt)) {
	fprintf (stderr,"reccable: missing conn pt\n");
	return 0.0;
    }
 }
 tcpnt = cpnt1;		/* remember the conn */

		/*  (Code below taken from dcomp() in "prcomp.c") */

		/* Make sure conn points back to orig comp. */

 if 	 ((pnt1=cpnt1->comp1)==pnt) pnt1 = cpnt1->comp2;
 else if ((pnt1=cpnt1->comp2)==pnt) pnt1 = cpnt1->comp1;
 else return 0.0;

 if (cpnt1->ctype!=AXIALRES) return 0.0;
 
 for (n=0; pnt1->nodlst==NULL; n++) {	       /* traverse cable, count comps */

    lpnt = pnt1->clst;
    if (!lpnt) break;
    if (!(cpnt2=lpnt->conpnt)) break;		 /* find conn */
    if (cpnt1==cpnt2) {
	lpnt  = lpnt->next;
	if (lpnt) cpnt2 = lpnt->conpnt; 	 /* wrong conn? */
    }
    if (!cpnt2) break;
    while (cpnt2->ctype!=AXIALRES) {
      lpnt  = lpnt->next;
      if (!lpnt) break;
      cpnt2 = lpnt->conpnt;
      if (!cpnt2) break;
    }

    if ((pnt2=cpnt2->comp1)==pnt1) pnt2 = cpnt2->comp2;
    pnt1=pnt2;
    cpnt1=cpnt2;

   }   /* for (n=0;;) */

   n++;					/* final comp at second node */
   cnum = (int)(fracdist * n);		/* integer number of requested comp */
  					/* end comps (0,n) are 1/2 size */
 
			/* Do it again; this time stop at comp "cnum" */
   cpnt1 = tcpnt;
   if 	   ((pnt1=cpnt1->comp1)==pnt) pnt1 = cpnt1->comp2;
   else if ((pnt1=cpnt1->comp2)==pnt) pnt1 = cpnt1->comp1;
   else return 0.0;

   if (cpnt1->ctype!=AXIALRES) return 0.0;

   vold = pnt->v;
   v = pnt1->v;
   for (i=1; i<=cnum; i++) {	       /* traverse cable, count comps */

      lpnt = pnt1->clst;
      if (!(cpnt2=lpnt->conpnt)) break;		 /* find conn */
      if (cpnt1==cpnt2) {
	  lpnt  = lpnt->next;
	  if (lpnt) cpnt2 = lpnt->conpnt; 	 /* wrong conn? */
      }
      if (!cpnt2) break;
      while (cpnt2->ctype!=AXIALRES) {
        lpnt  = lpnt->next;
        if (!lpnt) break;
        cpnt2 = lpnt->conpnt;
        if (!cpnt2) break;
      }

      if ((pnt2=cpnt2->comp1)==pnt1) pnt2 = cpnt2->comp2;
      vold = v;
      v = pnt2->v;
      pnt1=pnt2;
      cpnt1=cpnt2;

     }   /* for (i=0;;) */

   r = fracdist * n - cnum;		/* interpolate across axial resistors */
					/* between adjacent compartments */

/*  fprintf (stderr,"n %d fracdist %g cnum %d v %g vold %g r %g\n",
			n,fracdist,cnum,v,vold,r);	/* */
    
   return (vold * (1.0-r) + v * r);
}

/*------------------------------------*/

double getrecval (int plotnum)
{
  int cnod1,cnod2,cnod3,pmod;
  Symbol *spnt;
  int narg;
  short *vtyp;
  double *val;
  double rval, *garray(Symbol *s);
  double recsynap(int snum, int nf),reccable(int elemnum, double fracdist);

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"getrecval plotnum %d\n",plotnum);
#endif

   cnod1 = plotnod[plotnum].cnod1;
   cnod2 = plotnod[plotnum].cnod2;
   cnod3 = plotnod[plotnum].cnod3;
   pmod  = plotnod[plotnum].pmod;

   switch (pmod) {

   case VREC:
   case IREC:
   case LREC: 
             rval = record (cnod1,cnod2,cnod3,pmod);
             break;
   case GREC:
	     break;
   case SREC: 
        spnt = (Symbol *)plotnod[plotnum].spnt;  /* code is from getvar() */
        narg = (int)spnt;
        if (narg > 0 && narg < 20) {    /* if var is arg to a function */
                getarg(narg,&val,&vtyp);   
		rval = *val;
        }
        else {
          switch (spnt->type) {
          case UNDEF:
          case CONST:
          case VAR:    rval = spnt->val;
                      break;
          case ARRAY:  
                      rval = *garray(spnt); 
                      break;
          default: execerror("attempt to eval non-var", spnt->name);
                      break;
          } /* switch */
        } /* else */
        break;
			/* record from synapse */

case NRECA0: case NRECA1: case NRECA2: case NRECA3: case NRECA4:
case NRECA9:
case NRECB0: case NRECB1: case NRECB2: case NRECB3: case NRECB4:
case NRECC0: case NRECC1: case NRECC2: case NRECC3: case NRECC4:

			/* record conductance from channel */

case NRECG0: case NRECG1: case NRECG2: case NRECG3: case NRECG4:
case NRECG5: case NRECG6: case NRECG7:
case NRECG8:
			/*  cnod1 = element number, not node number */ 
		rval = record(cnod1,cnod2,cnod3,pmod);
        	break;
case CABLE:
     		rval = reccable(cnod1,((double) cnod2)/CABLFRAC);
		break;

   }    /* switch */
 
#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"getrecvalend\n");
#endif

 return (rval);
}

/*------------------------------------*/

double getcurv(synap *spnt, double val)
               
              

/* return instantaneous value for normalized conductance at synapse,
   given a voltage (calib in volts) and a pointer to the synapse. 
*/

{
   double rval;

  switch (spnt->curve) {

   case LINEAR:	if (val < 0) val = 0;		/* linear, sharp shoulder */
		rval = val * 10.0;		/* val=0.1V -> rval = 1 */
		break;

   case EXPON:	rval = .025 * lkexp(val*VTOMV/spnt->expon); /* exponential */
		if (rval < .001) rval = 0.0;	/* val is in volts */ 
		break;
  }
/* fprintf (stderr,"curv %d, expon %g val %g, rval %g igain %g\n",
			spnt->curve,spnt->expon,val,rval,spnt->igain); /* */
 return (rval * spnt->igain);
}

/*------------------------------------*/

void runsyn(int stime)
                        /* number of 100 microsec steps */

/* Run all the synapses, depending on their
   type and history.  Normally, call once every 100 microsec.
   For static runs, call with stime=0 and only
   the amplitude transfer function is produced, not the
   time functions.
*/

{
   int i,nsites;
   double trvolts,fltvolts,vesrel,transrate,p,vsize;
   double chanligand,bnldev(double pp, int n),synfilt(lpfilt *lpnt, double transrate);
   synap *spnt;

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"runsynap");
#endif

#define TR 3.0

 for (spnt=synpnt; spnt; spnt= (synap *)spnt->next)
   {
     switch (spnt->ctype)
      {
       case SYNAPSE:
	 if (spnt->curve==DYAD) break;		/* stop if 2nd in dyad */
	 if (stime) {
	    trvolts = (spnt->comp1->v - spnt->thresh);  /* calib in volts */
            for (i=0; i<stime; i++) {                   /* one or more steps */
	       if (spnt->filt1)
	         fltvolts = synfilt(spnt->filt1,trvolts); /* first lowp filt */
	       else fltvolts = trvolts;
	    }
	 }
	 else {
            trvolts = (spnt->comp1->vest - spnt->thresh); /* calib in volts */
	    fltvolts = trvolts;				/* if static */
	 }
	 transrate = getcurv(spnt,fltvolts);         /* static tranfer func */
         if (transrate < TRMIN) transrate = TRMIN;
         else if (transrate > TRMAX) transrate = TRMAX;
	 if (stime) {
           for (i=0; i<stime; i++) {                   /* one or more steps */
	 	if (nsites=spnt->vsites) {
#define TMAXP 10.0	     			     /* max prob at 10.0 */ 
		   vsize=spnt->vsize;    	     /* ves rate*timeinc */
		   transrate /= TMAXP;
		   if ((p=transrate/(vsize*nsites))<=1) { /* fast rate */
		     if (vsize<1) {		    /* if more than 1/tstep */
		       p = transrate/nsites;	    /*  reduce p */
		       nsites = (int)(nsites / vsize);  /* and increase nsites*/
		     }
		     nsites = (int)(nsites * TMAXP);
	             vesrel = bnldev(p,nsites);    /* vesicles released  */
		     transrate = vesrel * vsize; 
		   }
		   else {
		     transrate *= TMAXP;	   /* high prob, no noise */
		   }
/* fprintf(stderr,"vesrel %g transrate %g p %g size %g nsites %d\n",
			vesrel,transrate,p,vsize,nsites); 	/* */
	 	}
	       if (spnt->filt2)
		 chanligand = synfilt(spnt->filt2,transrate); /* 2nd lowp filt*/
	       else chanligand = transrate;
               if (chanligand < TRMIN) chanligand = TRMIN;
               else if (chanligand > TRMAX) chanligand = TRMAX;
            }
	 }  /* if (stime) */
	 else chanligand = transrate;

         rchan(spnt,stime,chanligand);		/* run the channels */
         if (spnt->sdyad) {
            rchan(spnt->sdyad,stime,chanligand); /* run the dyad's channels */
 	 }
         break;   /* case SYNAP */

       case GJ:                                 /* add for modulated gj's */
         break;

       case BUF:                                /* delay buffer */
        for (i=0; i<stime; i++) {                   /* one or more steps */
	   if (((dbuf *)spnt)->delay > 0) {	/* if delay */
		int v;
		double vf;
   	        dbuf *dlpnt;

	      dlpnt = (dbuf *)spnt;
	      vf = dlpnt->comp1->v / DELCONV; /* input voltage */
	      if (vf > 0) 
	         v = (int)(vf+0.5); 		/* input voltage */
	      else
	         v = (int)(vf-0.5); 		/* input voltage */
	      if (v>MAXSHORT) v = MAXSHORT;	/* no overflow of short int */
	      else if (v<MINSHORT) v = MINSHORT;
	      if (dlpnt->delbuf) {
	        *dlpnt->delpnt = v;
	        if (++dlpnt->delpnt >= dlpnt->delbuf+ (int)(dlpnt->delay))
		   dlpnt->delpnt = dlpnt->delbuf;  /* incr. pntr, make circ. */
					/* leave pointing to next output val */
	      }  /* if (dlpnt->delbuf) */
	    }  /* if (dlpnt->delay>0) */
	    else {
		((dbuf *)spnt)->comp2->v = ((dbuf *)spnt)->comp1->v;
	    }
	 }  /* if (stime;;) */
         break;  /* case BUF */

      } /* switch */
   }        /* for */
#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"...end\n");
#endif
}

/*------------------------------------*/

void rchan(synap *spnt, int stime, double chligand)

/* Run the channel conductance part of a synapse.  */
 
{
   int nsites;
   double bound,conduct,p,cdur,chanev,rpar;
   double bnldev(double pp, int n),synfilt(lpfilt *lpnt, double transrate);
   comp *cpnt;

 bound = chligand / (spnt->kd + chligand);  /* fraction bound */
 switch (spnt->ntact) {
  case OPEN:
   conduct =  bound;
    break;
  case CLOSE:
     conduct = (1 - bound); 
     break;
 }	
 if (stime) {
   if (spnt->filt3)
	      conduct = synfilt(spnt->filt3,conduct);   /* 3rd lowp filt*/
    if (nsites=spnt->csites) {
      cdur = spnt->cdur;	    /* ch event duration / timeinc */
      if (cdur>=1.0) {
        p = conduct / cdur; 	    /* binding probab */
        chanev = bnldev(p,nsites); /* bound  w/noise */
        conduct = chanev * cdur / nsites;
      }
      else {			 /* dur is less than timeinc */
	p = conduct;		 /*  but should be integer fraction */
	nsites = (int)(nsites/cdur); /* integer increase */		
        chanev = bnldev(p,nsites); /* bound  w/noise */
        conduct = chanev / nsites;
      }
   }
 }	/* if (stime) */
 else { 	/* stime==0,  static syn, update compartment implf */
   cpnt = spnt->comp2;
   if (cpnt) {
     if (cpnt->implf) {	/* if implf has been calculated yet... */
       rpar = ((1. / cpnt->implf) - 1.) / cpnt->k - spnt->conduct;
       rpar += conduct * spnt->maxcond; 
          cpnt->implf = 1. / (1. + rpar * cpnt->k);  /* implicit factor*/
     }
   }
   else fprintf (stderr,"runsyn: panic, no second compartment\n");
 }
 spnt->conduct = conduct * spnt->maxcond;
/* fprintf (stderr,"chlig %g bound %g kd %g ntact %d\n",
			chligand,bound,spnt->kd,spnt->ntact);  /* */
}

/*------------------------------------*/

double synfilt(lpfilt *lpnt, double transrate)

/* Make spnt->nfilt first-order low pass filters.
   Each filter has a time constant of spnt->ftau/10 * 0.1 msec */
{
   int i,nfilt;
   double ctr,tr, k;

  if (!lpnt) return transrate;
  tr=transrate;
  if (!(nfilt=lpnt->nfilt))		/* number of synaptic low pass filts */	
		return tr;
  k=lpnt->ftau;
  if (synaptau > 0.0) k /= synaptau;	/* possibly modify time constant */
  if (k >= 1.0) k = .9999;		/* limit tau to reasonable value */
  lpnt->lfilt[0] = tr;			/* zero filter holds input to filter */
  for (i=1; i<nfilt; i++) {		/* do one less filter than actual num */
    ctr = lpnt->lfilt[i];
    ctr += (tr - ctr) * k; 
    lpnt->lfilt[i] = tr = ctr; 
  }

		/* do the last one differently: separate falling phase */
  ctr = lpnt->lfilt[i];
  if (lpnt->tfall>0.0) {		/* separate falling phase */
	 double trel;

     trel = (tr - ctr) * k; 	 	/* transmitter released */
     if (trel > 0) ctr += trel;	 	/* rising phase, trel is non-neg */
     ctr *= lpnt->tfall;		/* falling phase */
  }
  else ctr += (tr - ctr) * k; 		/* normal falling phase */ 

  lpnt->lfilt[i] = ctr; 
 return (ctr);
}

/*------------------------------------*/

double bnldev(double pp, int n)
              
/* Taken from Press et al. 1988, "Numerical Recipes in C", p 223.
   Returns as a floating point number an integer value 
   that is a random deviate drawn from a binomial distribution
   of n trials each of probability pp, using drand() as a
   source of uniform random deviates.
*/

{
   int j;
   static int nold = -1;  			     /* must be static */
   static double am,em,g,angle,p,bnl,sq,t,y;
   static double pold = -1, pc, plog,pclog,en,oldg;  /* must be static */
   double drand(), gammln(double xx), floor(double);

   if (pp > 1.0) pp = 1.0;
   if (pp < 0.0) pp = 0.0;
   p = (pp <= 0.5? pp : 1-pp);
   am = n * p;				/* am = mean of deviate */
   if (n < 25) {			/* use direct method if n is small */
      bnl = 0.0;
      for (j=1; j<=n; j++) 
	 if (drand() < p) bnl += 1.0;
   }
   else if (am < 1.0) {			/* if fewer than 1 event is expected */
      g = exp(-am);			/*  in 25 or more trials, then */
      t = 1.0;				/*  use Poisson method */
      for (j=0; j<=n; j++) {
         t *= drand();
	 if (t < g) break;
      }
      bnl = (j <= n ? j : n);
   }
   else {				/* use rejection method */
      if (n != nold) {
	 en = n;
	 oldg = gammln(en+1.0);
	 nold = n;
      }
      if (p != pold) {			/* if p has changed */
	pc = 1.0 - p;
	plog = log(p);
	pclog = log(pc);
	pold = p;
      }
      sq = sqrt(2.0*am*pc);		/* rejection method with Lorentzian */
      do {				/* comparison function */
	do {
	   angle = drand();
	   if (angle == 0.5) angle = .49999999999;	
	   y = tan(MPI*angle);
	   em = sq*y+am;
	} while (em < 0.0 || em >= (en+1.0));	/* reject */
	em = floor(em);
	t = 1.2 * sq * (1.0+y*y) * exp(oldg-gammln(em+1.0)
		- gammln(en-em+1) + em * plog+ (en-em) * pclog);

      } while (drand() > t);		/* reject. This happens about 1.5 */
      bnl = em;				/* times per deviate, on average. */
   }
   if (p != pp) bnl = n -bnl;
   return bnl;
}

/*------------------------------------*/

double gammln(double xx)
            

/* Taken from Press et al. 1988, "Numerical Recipes in C", p 167.
   Returns the value ln(gamma(xx)) where xx > 0.
*/

{
   double x,tmp,ser;
   static double cof[6]={76.18009173,-86.50532033,24.01409822,
			-1.231739516, 0.120858003e-2,-0.536382e-5};
   int j;

   x = xx - 1.0;
   tmp = x + 5.5;
   tmp -= (x+0.5) * log(tmp);
   ser = 1.0;
   for (j=0; j<=5; j++) {
     x += 1.0;
     ser += cof[j]/x;
   }
   return -tmp+log(2.50662827465*ser);
}


/*------------------------------------*/

void runrec(int dmsec)

/* Run all the receptors. 
 depending on their type and history.
 Call at multiples of milliseconds only. */

{
   int i,tspeed;
   recep *rpnt;
   recpar *cpt;
   static double iflux,ca2;

#ifdef DEBUG
  if (debug & 1)  fprintf (stderr,"runrec");
#endif

  for (rpnt = recpnt; rpnt; rpnt=(recep *)rpnt->next) {

 
     switch (rpnt->ctype) {

       case ROD:
       case CONE:
          if (!(cpt=rpnt->consts)) {		/* get pointer to const */
	     break;
	  }
	  else {
	  tspeed = (int)(rpnt->timec * dmsec);
          for (i=0; i<tspeed; i++) {             /* maybe do several steps */
	    iflux = rpnt->flux * 0.001 / tspeed;
            if (rpnt->pnois) iflux = pois(iflux);

            rpnt->rhod += iflux * cpt->rhodgain;                /* */
            rpnt->gpr1 += rpnt->rhod * cpt->ggain1;
            rpnt->gpr2 += rpnt->gpr1 * cpt->ggain2;
            rpnt->gpr3 += rpnt->gpr2 * cpt->ggain3;

            rpnt->pde  += rpnt->gpr3 * (1.0 - rpnt->pde) * cpt->ggain4;
            ca2 = rpnt->cax;
            ca2 = ca2 * ca2;
            rpnt->gcyc  = 1.0 - (ca2 / (ca2 + cpt->kdgcyc));

            rpnt->cycg -= rpnt->pde * rpnt->cycg * cpt->pdegain / 
                        (rpnt->cycg + 1.0);
            rpnt->cycg += rpnt->gcyc * cpt->gcygain;

            rpnt->cond += rpnt->cycg * rpnt->cycg *
                         (cpt->totcond - rpnt->cond) /
                         cpt->totcond * cpt->condgain;
            rpnt->ca   += rpnt->cond * cpt->gca;
            rpnt->cax  += rpnt->ca * cpt->cxgain;

            rpnt->rhod *= cpt->dec1;
            rpnt->gpr1 *= cpt->dec2;
            rpnt->gpr2 *= cpt->dec3;
            rpnt->gpr3 *= cpt->dec4;
            rpnt->pde  = cpt->pdedec * (rpnt->pde- cpt->pdebase) + cpt->pdebase;
            rpnt->ca   -= cpt->capump * rpnt->ca / (rpnt->ca + 1.0); 
            rpnt->cond *= cpt->dec5;
            rpnt->cax  *= cpt->dec6;
   
 	  }	/* for */
        }    /* else */

         rpnt->conduct = rpnt->maxcond * rpnt->cond;   /* chan current */

if (debug & 1 && debugz & 16)
 fprintf (stderr,
        "rec num %d timec %g iflux %g %g rh %g pd %g gc %g cg %g c %g cond %g\n",
                rpnt->recnm1, rpnt->timec, iflux, rpnt->flux,
                 rpnt->rhod, rpnt->pde, rpnt->gcyc,
                 rpnt->cycg, rpnt->cond, rpnt->conduct);                /* */
 

         break;
        }       /* switch */
    }     /* for (rpnt; ; )  */
#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"...end\n");
#endif
}

/*------------------------------------*/

void runstim(double rtime, int dmsec)

/* Advance all the stimuli to time "rtime",
 depending on their type and history.
 Also run all voltage and current clamps.
 Call at multiples of tenths of milliseconds only. */

{
#define RECTIM 1			/* how many 0.1 msec intervals */
#define READRECTIM 10			/* how often to read stimuli */
#define RTIMESTEP 0.0001		/* time step for reading stimuli */
#define RTIMERND   0.000001		/* down-round value */
#define QEFF 0.67			/* rhod isomerizations per photon */

   recstim *rspnt,*last,*next;
   recep *rpnt;
   node *npnt;
   static int readmsec=READRECTIM;
   static double start,stop,diff,sens,rtimestep;

#ifdef DEBUG
  if (debug & 1)  fprintf (stderr,"runstim\n");
#endif

 rtimestep = max(timinc,RTIMESTEP);

/* The input from the file is sorted by time before being read in. 
   This is done as a final step by "stim".
*/

 if(readmsec++ >= READRECTIM) {
   if (!readstim(rtime+readmsec*rtimestep)) {  /* read new receptor inputs */
         fprintf (stderr,"runstim: invalid stimulus at time %g\n",rtime);
	 execerror ("Missing stimulus: "," stopping... ");
   }   
   readmsec = 0;                
 }

 if(dmsec >= RECTIM) {        /* calculate receptors for 0.1 msec intervals: */

  start = rtime - (rtimestep + RTIMERND);
  stop  = rtime + (rtimestep - RTIMERND);
  for (last=rspnt=recspnt; rspnt; ) {

/*  fprintf (stderr,"stim num %d type '%c' time %g rtime %g start %g stop %g\n",
            rspnt->recnm1, rspnt->ctype,rspnt->time, rtime, start, stop); /* */
 
   if ((rspnt->time >= start) && (rspnt->time <stop)) {
     switch (rspnt->ctype) {

       case 'a':                /* absolute */
          if (findrecep (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
					&rpnt, "runstim")) {
	     sens = rsens(rpnt,rspnt->wavel,rpnt->filt);
             rpnt->flux = rpnt->area * rspnt->val * sens * rpnt->attf * QEFF;
	  }
          if (rpnt->flux > MAXPHOT) rpnt->flux = MAXPHOT;
          if (rpnt->flux < 1e-10) rpnt->flux = 0.0;
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"abs stim %d %g %g\n",
                                        rspnt->recnm1,rspnt->val,rspnt->wavel);
#endif
         break;

       case 'd':                /* delta */
          if (findrecep (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
					&rpnt, "runstim")) {
	     sens = rsens(rpnt,rspnt->wavel,rpnt->filt);
             rpnt->flux += rpnt->area * rspnt->val * sens * rpnt->attf * QEFF;
	  }
          if (rpnt->flux > MAXPHOT) rpnt->flux = MAXPHOT;
          if (rpnt->flux < 1e-10) rpnt->flux = 0.0;
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"delta stim %d %g\n",
                                        rspnt->recnm1,rspnt->val,rspnt->wavel);
#endif
         break;

       case 'v':                /* vclamp */
          if (npnt=findnode (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
					 "runstim vclamp")) {
	    if (!(npnt->comptr->miscfl & VBUF)) { /* not buf already */
              npnt->comptr->extv = rspnt->val;  /* set up voltage clamp */
              npnt->comptr->v = rspnt->val;     /* set comp voltage (for plot)*/
              npnt->comptr->miscfl |= VEXT;
	    }
          }
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"vclamp %d %g\n",
                                        rspnt->recnm1,rspnt->val);
#endif
         break;

       case 'c':                /* cclamp */
          if (npnt=findnode (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
				 	"runstim cclamp")) {
            npnt->comptr->exti += rspnt->val;    /* set up current clamp */
            npnt->comptr->miscfl |= IEXT;
          }
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"cclamp %d %g\n",
                                        rspnt->recnm1,rspnt->val);
#endif
         break;

       case 'e':                /* volt clamp off */
          if (npnt=findnode (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3, 
				"runstim voff")) {
            diff = npnt->comptr->extv - rspnt->val; 
	    if (!(npnt->comptr->miscfl & VBUF)) { /* not buf already */
              if (ncabs(diff) < 1e-6) {
                 npnt->comptr->extv = 0.0;
                 npnt->comptr->miscfl &= ~VEXT;    /* turn off voltage clamp */
            }
	   }
          }
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"vclamp %d off\n", rspnt->recnm1);
#endif
         break;

       case 'i':                /* current clamp off */
          if (npnt=findnode (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
					"runstim voff")) {
            npnt->comptr->exti -= rspnt->val;
            if (ncabs(npnt->comptr->exti) < 1e-20) {
               npnt->comptr->exti = 0.0;
               npnt->comptr->miscfl &= ~IEXT;    /* turn off current clamp */
            }
          }
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"cclamp %d off\n",rspnt->recnm1);
#endif
         break;

        default:
         break;

      }         /* switch */
  
        next = rspnt->next;            /* delete and patch pointers */
        if (rspnt == recspnt) recspnt = next;
        else last->next = next;
        if (recsend==rspnt) {
	  if (!recspnt) recsend = NULL;
	  else recsend = last;
	}
        delrstim(rspnt);                /* delete stimulus after use */
        rspnt = next;
     }        /* if */
    else {   /* no stimulus here */
      last = rspnt;
      rspnt = rspnt->next;
    }
   }       /* for */
 }      /* if */
#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"runstim end\n");
#endif
}

/*------------------------------------*/

int pois(double xm)
             

/*
 Return a poisson distribution which on average equals "xm".

 Taken from Press et al. 1988, "Numerical Recipes in C", p 222.
   Returns as a floating point number an integer value 
   that is a random deviate drawn from a Poisson distribution
   of mean xm as a source of uniform random deviates.
*/

{
  static double sq,alxm,g,oldm= -1.0;
  double em,t,y;
  double drand(),gammln(double xx);

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"pois\n");
#endif

  if (xm < 12.0) {			/* use direct method */
     if (xm != oldm) {
  	oldm = xm;
	g = exp(-xm);			/* if xm is new, compute exp */
     }
     em = -1;
     t = 1.0;
     do {
       em += 1.0;			/* instead of adding exp deviates */
       t *= drand();			/* we multiply uniform deviates */
     } while (t > g);
  }
  else {				/* use rejection method */
    if (xm != oldm) {
      oldm = xm;
      sq = sqrt(2.0*xm);
      alxm = log(xm);
      g = xm * alxm - gammln(xm+1.0);
    }
    do {
       do {
	  y = tan(MPI * drand());
	  em = sq*y+xm;
       } while (em < 0.0);		/* reject if zero probability */
       em = (int)(floor(em));
       t = 0.9 * (1.0+y*y) * exp(em*alxm-gammln(em+1.0)-g);
    } while (drand() > t);
  }
 return (int)(em);
}

/*------------------------------------*/

 
