/* segment ncstim in program nc */

/* Reads stimulus arrays from disk files.
*/


#include <stdio.h>
#include "adef.h"
#include "nc.h"
#include "y.tab.h"
#include "ncelem.h"
#include "ncomp.h"
#include "ncsub.h"
#include "control.h"

extern int cumrecst;
extern recep *recpnt;
extern recstim *recspnt;
extern int stimfflg;			/* =1 -> get stimulus from file */
extern FILE *stimin;			/* stimulus file */

#ifdef __cplusplus
extern "C" {
#endif

double log(double), exp(double), sqrt(double);

#ifdef __cplusplus
}
#endif

double flookup(int filt, double wavel);
void wsav (int pigm, double pathl, double wavel, int filt, double sens);

/*------------------------------------*/

BOOL readstim(double endtime)
                   

/* Read the current stimulus from a disk file, until
   the stimulus time is equal or greater than endtime.
   The stimulus is read once every 1 msec (simulated time).
   Erase stimuli earlier than startime.
*/

{
   int recnm1, recnm2, recnm3;
   char linebuf[80],action[10];
   recstim *rpnt,*makrstim(double start, int recnm1, int recnm2, int recnm3, double inten, double wavel, char *action);
   double stime,inten,wavel;
   int error;
				/* first, delete only old stimuli: */

  if (! stimfflg) return(1);
				/* then, check for new stimuli */
  error = 0;		
  stime = 0;
  do {				/* get new stimuli */
    if ((fgets (linebuf,80,stimin)) == NULL) { break; }	/* not an error */
    if (*linebuf == '#') continue;		/* ignore comment lines */
    if ((sscanf (linebuf,"%lf %d %d %d %lf %lf %3s\n",
		&stime,&recnm1,&recnm2,&recnm3,&inten,&wavel,action))< 7) {
			 error = 1; break; }
    if (recnm3<0) recnm3 = NULLNOD;
    if (recnm2<0) recnm2 = NULLNOD;
    if (recnm1<0) recnm1 = NULLNOD;
    if ((rpnt=makrstim(stime,recnm1,recnm2,recnm3,inten,wavel,action))==NULL) {
	fprintf (stderr,"readstim: can't make rec stim %d\n",cumrecst);
	error = 1;
	break;
    }
  } while (stime < endtime);

  if (error)	return (0);
  else 		return (1);
}

/*------------------------------------*/

int incirc(double x, double y, double circx, double circy, double rad)
                              

/* calculate whether point is inside a circle
   of given radius.
*/

{
   double tempx, tempy;
   int inside;

   tempx = circx - x;
   tempy = circy - y;
   inside = (sqrt(tempx*tempx + tempy*tempy) <= rad);
   return inside;
}

/*------------------------------------*/

int inrect(double x, double y, double maxx, double minx, double maxy, double miny)
                                     

/* calculate whether point is inside a rectange
   of given limits.
*/

{
   int inside;

   if (maxx < minx) return 0;
   if (maxy < miny) return 0;
   inside = ((minx <= x) && (x <= maxx) && (miny <= y) && (y <= maxy));
   return inside;
}

/*------------------------------------*/

/* pigment action spectra tables */

			/* wave.h contains pigment and light tables */
#include "wave.h"

double dlookup(int pigm, double wavel)

/* calculate the specific optical density
   for given pigment type and wavelength of
   incident light.
*/

{
   double dens,logdens;
   int w,windx;
   double r;

  windx = (int)wavel;
  if (windx < MINWAV) {
     fprintf (stderr,"dlookup: error in wavelength '%g'\n",windx);
     return (-100.0);
  }
  else if (windx > MAXWAV) {
     fprintf (stderr,"dlookup: error in wavelength '%g'\n",windx);
     return (-100.0);
  }
  else {				/* interpolate between 5nm entries */
    windx -= MINWAV;
    w = windx / WAVINC;
    r = windx / WAVINC - w; 		/* remainder normalized to 1.0 */
    if (w >= (PIGMSIZ-1)) w = PIGMSIZ-2;
    logdens = specdens[pigm][w];
    if (r > .001)
       logdens += (specdens[pigm][w+1] - logdens) * r; 	/* interpolate */ 
  }
  dens = exp(logdens*LN10);
  return (dens);
}

/*------------------------------------*/

double flookup(int filt, double wavel)

/* calculate the transmission senstivity
   for given filter and wavelength of
   incident light.
*/

{
   double sens;

  int w,windx;
  double r;

  windx = (int)wavel;
  if (windx < MINWAV) {
     fprintf (stderr,"flookup: error in wavelength '%g'\n",windx);
     return (-100.0);
  }
  else if (windx > MAXWAV) {
     fprintf (stderr,"flookup: error in wavelength '%g'\n",windx);
     return (-100.0);
  }
  else {				/* interpolate between 5nm entries */
    windx -= MINWAV;
    w = windx / WAVINC;
    r = ((double) windx) / WAVINC - w; 	/* remainder normalized to 1.0 */
    if (w >= (PIGMSIZ-1)) w = PIGMSIZ-2;
    sens = filts[filt][w];
    if (r > .001)
       sens += (filts[filt][w+1] - sens) * r;  	/* interpolate */ 
  }
  return (sens);
}

/*------------------------------------*/

struct wparm {
	int pigm;
	int filt;
	double pathl;
	double wavel;
	double sens;
	};

#define OLDPSIZ 64
struct wparm oldparm[OLDPSIZ] = {0};	/* array to hold all combin. of */
					/*  pigm, pathl, wavel, filt calcs. */

int wfind (int pigm, double pathl, double wavel, int filt, double *oldsens)
                  
/* Look for a pigment sensitivity previously calculated
   with a specific combination of pigment, path length,
   light, and filter parameters.
*/

{
  int i, found;
  static int first=1;

  if (first) {
      first = 0;
      for (i=0; i<OLDPSIZ; i++) {
    	oldparm[i].pathl = 0;		/* initialize all entries first time */
      }
  }
	 
  found = 0;
  for (i=0; !found && i<OLDPSIZ; i++) {	/* try to find same calc done before */

    if (oldparm[i].pathl == 0) break;	/* stop when list is done */

    if (oldparm[i].pigm  != pigm)  continue;
    if (oldparm[i].pathl != pathl) continue;
    if (oldparm[i].wavel != wavel) continue;
    if (oldparm[i].filt  != filt)  continue;

    found = 1;
    *oldsens = oldparm[i].sens; 	/* old calc found; use it */
  }
  return (found);
}

/*------------------------------------*/

void wsav (int pigm, double pathl, double wavel, int filt, double sens)

/* Save a pigment sensitivity done with a specific combination
    of pigment, path length, light, and filter parameters
    for future use.
*/

{
  int i, found;

  found = 0;
  for (i=0; !found && i<OLDPSIZ; i++) {	/* try to find same calc done before */

    if (oldparm[i].pathl == 0) break;	/* stop when list is done */

    if (oldparm[i].pigm  != pigm)  continue;
    if (oldparm[i].pathl != pathl) continue;
    if (oldparm[i].wavel != wavel) continue;
    if (oldparm[i].filt  != filt)  continue;

    found = 1;				/* should never get here! */
  }
  if (i==OLDPSIZ) {
     fprintf (stderr,"Wsav: not enough space to save new cone calculation.\n");
     fprintf (stderr,"      Continuing with calculations without save...\n");
     return;
  }
  if (!found && oldparm[i].pathl==0) {	/* use first empty spot in list */
    oldparm[i].pigm  = pigm;		/* save new parameters and sens */
    oldparm[i].pathl = pathl;
    oldparm[i].wavel = wavel;
    oldparm[i].filt  = filt;
    oldparm[i].sens  = sens;
  }
}

/*------------------------------------*/

double rsens(recep *rpnt, double wavel, int filt)
               
/* Calculate receptor sensitivity given pigment, pathlength,
   wavelength, and filter.
   First, look to see if calculation has been done before;
   if so, then use old calculation.
   Second, if light is monochromatic, do the calculation directly.
   Otherwise, light must be a standard light source;
   check to see if path length and filter are standard; if so, then
   use standard lookup table () to get sensitivity.
   Lastly, if non-standard path length or filter,
   then calculate sensitivity over full wavelength range, 
   and save the calculation for future use.
*/

{
    int j,pigm,wav,lum;
    double od, pathl, logdens, specd, sens, totsens;
    double oldsens;

pigm = rpnt->consts->pigm;
lum = (pigm? 1 : 0);			/* scotopic or photopic calibration */
lum = 0;				/* use scotopic calib for now */
pathl = rpnt->pathl;
if (wfind (pigm,pathl,wavel,filt,&oldsens))
     sens = oldsens;		/* check to see if this has been done before */
else {
  if (filt >= FILTS) filt = FILTS-1;
  wav = (int)wavel;
  if (wav < 0) wav = 0;
  if (wav >= LIGHTS) {			/* monochromatic light */
     specd = dlookup(pigm,wavel);
     od = specd * pathl;
     sens = 1.0 - exp(-od * LN10) * flookup(filt,wavel);
  }
  else {			/* (wav < LIGHTS): standard light source */
    if (pathl == rpnt->consts->pathl && filt == 0) {	/* if pre-computed */
       sens = lightsens[pigm][wav] / lumlight[lum][wav];
    }
    else {			/* re-calculate pigm sens to light func */
      totsens = 0.0;
      for (j=0; j<PIGMSIZ; j++) {
        logdens = specdens[pigm][j];	/* no interpolation */
        specd = exp(logdens*LN10);	/* density table is log(spec od) */
        od = specd * pathl;
        sens = 1.0 - exp(-od * LN10);
	totsens += sens * filts[filt][j] * lights[wav][j];
      } 
      sens = totsens/(PIGMSIZ*lumlight[lum][wav]); /* divide by calib sens */
    }
    wsav (pigm,pathl,wavel,filt,sens); /* save calculation */
   }
  }

/* fprintf (stderr,"sens %g pigm %d wavel %g filt %d path %g\n",
			sens,pigm,wavel,filt,pathl);	/* */
  return sens;
 } 
 
