/* ncray */

/* Rudimentary interface to generate symbolic output file for 
   "POV" ray-tracer.  Requires "nc.pov" include file to set up
   lights and default surface.  Camera location defines 
   the rotations from the "nc" file.
*/

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <math.h>

#ifdef __cplusplus
}
#endif

#include "ncsub.h"
#include "ncelem.h"
#include "colors.h"
#include "adef.h"
#include "nc.h"
#include "y.tab.h"

extern double ncmatc[4][4];

void rayconn (double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	int ctype, double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4]);
void transf (double x,double y,double z,
	      double *tx,double *ty,double *tz, double (*mat)[4]); 
void setray (double xrot, double yrot, double zrot, 
	double xcent, double ycent, double zcent,
	double xtran, double ytran, double ztran,
	double scale);

/*--------------------------------------*/

char *color_strings[]= {
        "Black","Blue","Green","Cyan","Red","Magenta","Brown","White",
        "Gray60","LightBlue","Aquamarine","CadetBlue","Firebrick",
        "VioletRed","Yellow", "VLightGrey"};

char *getcol (int color)

/* Return the char string color.  */

{
 return (color_strings[color]);
}

double xr,yr,zr;

/*--------------------------------------*/

void initray (double xrot, double yrot, double zrot, 
	double xcent, double ycent, double zcent, 
	double rxcent, double rycent, double rzcent, double scal)
{

  double xcam,ycam,zcam,aspect;
  double xcamr,ycamr,zcamr;
  double xzoom,yzoom,zzoom;
  static int runyet=0;

  if (!runyet) runyet = 1;
  else return; 
    
  aspect = 1.0;
  xcam = 0;			/* original camera position */
  ycam = 0;
  zcam = 10*scal;
  xzoom = 0; 			/* frame size */ 
  yzoom = 10;  
  zzoom = 0;  
  printf ("/* Scene code for POV ray-tracer */\n");
  printf ("/* generated by nc. */\n");
  printf ("\n");
  printf ("#include \"nc.pov\"\n\n");

  setray (xrot,yrot,zrot,xcent,ycent,zcent,rxcent,rycent,rzcent,1.0);

/*  transf (xcam,ycam,zcam, &xcamr,&ycamr,&zcamr,ncmatc); */

xr = xrot;
yr = yrot;
zr = zrot;

  printf ("  camera {\n");
  printf ("    location <%g,  %g  %g>	/* viewpoint location */\n",
			xcam,ycam,zcam);
  printf ("     up      <%g, %g, %g>	/* aspect ratio */\n",
			0.0,1.0,0.0);
  printf ("     right   <%g, %g, %g>	/* aspect ratio */\n",
			1.0*aspect,0.0,0.0);
  printf ("     direction <%g, %g, %g>	/* length of viewing lens */\n",
			xzoom,yzoom,zzoom);
  printf ("     look_at <%g, %g, %g>	/* viewpoint direction */\n",
			0.0,0.0,0.0);
  printf ("  }\n");
  
}

/*--------------------------------------*/

void raycomp(double x, double y, double z, double dia)

/* draw a compartment */

{
  int color;

  if (z==NULLNOD) z = 0.0;
  color = BLUE;
  printf ("sphere { <%g,%g,%g>, %g pigment {%s} }\n\n",
		x,y,z,dia/2.0,getcol(color));
}

/*--------------------------------------*/

void raynode(node *npnt, double dscale, int color, double (*mat)[4])

/* display node numbers in 3D location */

{
    double x,y,z,tx,ty,tz,xoffs,charsiz;
    static char numbuf[20];

/*  if (!color) color = WHITE; 
  transf (x,y,z,&tx,&ty,&tz,mat);
  if (npnt->nodenm3 != NULLNOD)
      sprintf (numbuf,"%d %d %d",npnt->nodenm1,npnt->nodenm2,npnt->nodenm3);
  else if (npnt->nodenm2 != NULLNOD)
      sprintf (numbuf,"%d %d",npnt->nodenm1,npnt->nodenm2);
  else
      sprintf (numbuf,"%d",npnt->nodenm1);
  charsiz = .015 * dscale;
  xoffs = strlen(numbuf) / 2.0;		/* find center of number */
/*  xoffs *= charsiz * .7;
  gmove (tx-xoffs, ty - charsiz*1.1);	/* label node numbers */
/*  gcwidth (charsiz);
  gtext (numbuf);
*/
}

/*--------------------------------------*/

void raycable (double x1, double y1, double z1, 
		double x2, double y2, double z2, 
		double dia, double dscale, double n1dia, double n2dia, 
		int color, int hide, double (*mat)[4])
                
/*
   Describe a cable element in 3D for ray-tracer.
   Include file sets up camera angle and default surfaces.
*/

{
  double tx1,ty1,tz1;
  double tx2,ty2,tz2;

  if (z1==NULLNOD) z1 = 0.0;
  if (z2==NULLNOD) z2 = 0.0;
  transf (x1,y1,z1,&tx1,&ty1,&tz1,mat);
  transf (x2,y2,z2,&tx2,&ty2,&tz2,mat);
  if (color==NULLNOD) color = GREEN;
  if (dscale<0) {
     dscale = -dscale;
     dia *= dscale;
     if (dia < 0.3) dia = 0.3;
  }
  else dia *= dscale;
  if (x1!=x2 || y1!=y2 || z1!=z2) {
    printf ("cylinder { <%g,%g,%g> <%g,%g,%g>,%g pigment {%s} }\n\n",
		tx1,ty1,tz1,tx2,ty2,tz2,dia/2.0,getcol(color));
  }
}
/*--------------------------------------*/

void raysphere (double x, double y, double z, double dia, double dscale, 
	int color, int hide, double (*mat)[4],int fill)
{
    double tx,ty,tz;

  if (z==NULLNOD) z = 0.0;
  if (color==NULLNOD) color = BLUE;
  if (dscale<0) {
     dscale = -dscale;
     dia *= dscale;
     if (dia < 0.5) dia = 0.5;
  }
  else {
     dia *= dscale;
  }
  transf (x,y,z,&tx,&ty,&tz,mat);
  printf ("sphere { <%g,%g,%g>, %g pigment {%s} }\n\n",
		tx,ty,tz,dia/2.0,getcol(color));
}

/*--------------------------------------*/

void raysynap (double x1, double y1, double z1, 
	double x2, double y2, double z2, synapse* epnt,
	double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4])
{
    double tx1,ty1,tz1, tx2,ty2,tz2;
    double dia;
    double cx,cy,cz;

  if (color==NULLNOD) color=RED;
  if (z1==NULLNOD) z1 = 0.0;
  if (z2==NULLNOD) z2 = 0.0;
  transf (x1,y1,z1,&tx1,&ty1,&tz1,mat);
  transf (x2,y2,z2,&tx2,&ty2,&tz2,mat);
  dia = 0.5;
  cx = (tx1+tx2)*0.5;
  cy = (ty1+ty2)*0.5;
  cz = (tz1+tz2)*0.5;
  if (x1!=x2 || y1!=y2 || z1!=z2) {
    printf ("cylinder { <%g,%g,%g> <%g,%g,%g>,%g pigment {%s} }\n\n",
		tx1,ty1,tz1,tx2,ty2,tz2,dia/2.0,getcol(color));
  }
  printf ("sphere { <%g,%g,%g>, %g pigment {%s} }\n\n",
		cx,cy,cz,dia/2.0,getcol(color));
}

/*--------------------------------------*/

void raycconn (double x1, double y1, double z1, double x2, double y2, double z2)
{
   rayconn (x1,y1,z1,x2,y2,z2, AXIALRES, 1.0, 0.0, 0.0, YELLOW, 0, 0);
}

/*--------------------------------------*/

void rayconn (double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	int ctype, double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4])
{
  double dia,cx,cy,cz;
  double tx1,ty1,tz1, tx2,ty2,tz2;

  if (z1==NULLNOD) z1 = 0.0;
  if (z2==NULLNOD) z2 = 0.0;
  if (color==NULLNOD) color = YELLOW;
  transf (x1,y1,z1,&tx1,&ty1,&tz1,mat);
  transf (x2,y2,z2,&tx2,&ty2,&tz2,mat);
  cx = (tx1+tx2)*0.5;
  cy = (ty1+ty2)*0.5;
  cz = (tz1+tz2)*0.5;

  switch (ctype){

  case GJ: 
    if (dscale>0) {
      dia = dscale;
      printf ("sphere { <%g,%g,%g>, %g pigment {%s} }\n\n",
		cx,cy,cz,dia/2.0,getcol(color));
      dia = 1.0;
      printf ("cylinder { <%g,%g,%g> <%g,%g,%g>,%g pigment {%s} }\n\n",
		tx1,ty1,tz1,tx2,ty2,tz2,dia/2.0,getcol(color));
    }  
    else {
      dia = 1.0 * -dscale;
      printf ("sphere { <%g,%g,%g>, %g pigment {%s} }\n\n",
		cx,cy,cz,dia/2.0,getcol(color));
    }
    break;

  case RESISTOR:
  case AXIALRES:
    dia = 0.5;
    printf ("cylinder { <%g,%g,%g> <%g,%g,%g>,%g pigment {%s} }\n\n",
		tx1,ty1,tz1,tx2,ty2,tz2,dia/2.0,getcol(color));
    break;
  } 
}

/*--------------------------------------*/

void rayphotrec (double x, double y, double z, 
		     photrec *epnt, double dscale,
		     double n1dia,double n2dia,
		     int color, int hide, double (*mat)[4])
{
    double tx1,ty1,tz1,tx2,ty2,tz2;
    double dia,len;
    int ctype,pigm;
    static int color_trans[NUMREC] = {MAGENTA,RED,GREEN,BLUE};

  ctype = epnt->ctype;
  dia = epnt->dia*dscale;
  if (color==NULLNOD) {
      if ((pigm=(int)(epnt->pigm)) == NULLNOD)
         pigm = (ctype==CONE ? 1 : 0);
      pigm = limit(pigm,0,NUMREC);
      color= color_trans[pigm];
  }
  len = 1.0;				/* length of photorec in um */
  if (dia<=0) dia = 2.0;
  if (z==NULLNOD) z = 0.0;
  transf (x,y,z,&tx1,&ty1,&tz1,mat);
  transf (x,y,z+len,&tx2,&ty2,&tz2,mat);
  switch (ctype) {

  case CONE:
  printf ("object { nc_cone scale <%g,%g,1> rotate <%g,%g,%g> translate <%g,%g,%g> pigment {%s} }\n\n",
	 dia,dia,xr,yr,zr,tx1,ty1,tz1,getcol(color));
  break;

  case ROD:
  printf ("object { nc_rod scale <%g,%g,1> rotate <%g,%g,%g> translate <%g,%g,%g> pigment {%s} }\n\n",
	 dia,dia,xr,yr,zr,tx1,ty1,tz1,getcol(color));
  break;

 }  /* switch */
}

/*--------------------------------------*/

void rayload (double x, double y, double z, int ctype, 
	double n1dia, double n2dia, int color, int hide, double (*mat)[4])
                              
/* see "drphotrec()" above for explanation of
   frame, origin() and rotation. */
 
{
    double tx1,ty1,tz1,tx2,ty2,tz2;
    double dia,len;

  len = 10; 
  transf (x,y,z,&tx1,&ty1,&tz1,mat);
  transf (x,y,z+len,&tx2,&ty2,&tz2,mat);
  if (color==NULLNOD) color = BLUE;
  dia = 0.5;
  printf ("sphere { <%g,%g,%g>, %g pigment {%s} }\n\n",
		tx1,ty1,tz1,dia/2.0,getcol(color));
}

/*--------------------------------------*/

void rprintf (char *fmt,double x, double y, double z, double size, int color)

/* print a row of chars into 3D ray-traced picture */

{
   int i,slen,ch;

 slen = strlen(fmt);
 for (i=0; i<slen; i++) {
   ch = toupper(*(fmt+i));
   if (ch== ' ') continue;
   printf ("object {char_%c scale %g translate <%g,%g,%g> pigment {%s} }\n\n",
      ch, size, x+size*5*i,y,z,getcol(color));
 }

}
/*--------------------------------------*/

void raycalib(double x, double y, double len, double size, int color)
{
    static char numbuf[20];
    double xb,yb,zb,z,width,ls;

  if (color==NULLNOD) color = CYAN;
  z = 1;
  x = (x-0.5) * size * .95;
  y = (y-0.5) * size * .95;
  xb = x - len;
  ls = 0.003 * size;			/* letter scale */
  width = len * 0.1;
  yb = y + width;
  zb = z + width;
  sprintf (numbuf,"%g um",len);  
  printf ("box { <%g,%g,%g> <%g,%g,%g> pigment {%s} }\n\n",
			x,y,z,xb,yb,zb,getcol(color));
  rprintf (numbuf, x-len*0.8, y-ls*8,z,ls,color);
}

/*--------------------------------------*/

