/* Segment ncomp in Program nc */

/* Simulates neuronal circuits */
/*  by numerically integrating difference equations */
/*  using iterative relaxation technique */
/*  to implement the "implicit" integration method */ 

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include "nc.h"
#include "y.tab.h"
#include "control.h"

#ifdef __cplusplus
}
#endif

#include "ncelem.h" 
#include "ncomp.h"
#include "ncsub.h"

#define DEBUG			/* */
#define NOSKIP 128

#define SQRT10 3.16227766
#define SSQRT10 1.77827942

extern comp *compnt,*compend;

#ifdef __cplusplus
extern "C" {
#endif
    double log(double);
    double exp(double);
    double sqrt(double);
    void exit(int err);
#ifdef __cplusplus
}
#endif

double ncabs(double x);

#define min(a,b) ((a)<(b)?(a):(b))

extern double alpham,betam;
extern double alphah,betah;
extern double alphan,betan;
extern double alphac,betac;
extern double m1,m2;
extern double h1,h2;
extern double n1,n2;
extern double d1,d2;
extern double c1,c2;
extern double kca1,kca2;

static double maxerr;
static int ncomp,tcomp;

double akcacalc(double v, double ca, double tau, double d1, double k1);
double bkcacalc(double v, double ca, double tau, double d2, double k2);

void docomp(comp *p);
void runsyn(int t);
void catab(double vthr,int stype, double taum);
void natab(double vthr,int stype,double taum, double tauh);	
void ktab (double vthr,int stype,double taum);	
void kcatab(double vthr, double cais, int stype, double taum,
		     double d1, double d2, double k1, double k2);

/*------------------------------------*/

double dochan1(sschan *chpnt)
              

/* First-order calculation of  channel states */
/*  Assumes "cval" contains current value of state concentration */
/*  Leaves new concentration values in "cest" */
/*  Returns conductance */

{
   int i,j,t,nstate,numtr;
   static chanstate *stpnt;
   static stconc *conc;
   static double (**f_rate)(double v, int typ), trrate, conduct;
   static float *trmul;
   static char *ratev;
   double trv;			/* voltage value for channel transition */
   char *trans;

  nstate = chpnt->numstate;
  t  = chpnt->stype;
  conduct = 0.0;
  conc = chpnt->conc;
  for (i=0; i<nstate; i++,conc++) { /* save old estimate */
    conc->cval = conc->cest;
  }
  conc = chpnt->conc;
  stpnt = chpnt->consts->state;
  for (i=0; i<nstate; i++,stpnt++,conc++) {  /* find change in conc */
       trans = stpnt->trans;	     /* get parameters */
       f_rate  = stpnt->trate;
       trmul = stpnt->ratemul;
       numtr = stpnt->numtrans;
       ratev = stpnt->ratev;
       for (j=0; j<numtr; j++,trans++,f_rate++,trmul++,ratev++) {
         trv = (*ratev ? chpnt->comp2->v : chpnt->comp1->v) - chpnt->vthr;
         trrate = conc->cval * (*f_rate)(trv,t) * *trmul;      /* per timinc */
/*	 if (trrate > conc->cval) trrate = conc->cval;
	 if (trrate + chpnt->conc[*trans].cval > 1.0)
		 trrate = 1.0 - chpnt->conc[*trans].cval;
*/
/* if (chpnt->ctype==NA)
fprintf
  (stderr,"i %d j %d t %d trrat %10.4g cval %10.4g f_rate %10.4g v %g mul %g\n",
	i,j,t,trrate,conc->cval,(*f_rate)(trv,t),trv,*trmul); /* */
         chpnt->conc[*trans].dcon += trrate;
         conc->dcon -= trrate;
       }
  }
	/* after calculating all deltas, find new conc estimates: */

  conc = chpnt->conc;
  stpnt = chpnt->consts->state;
  for (i=0; i<nstate; i++,stpnt++,conc++) {
/*       if      (conc->dcon< -conc->cval) conc->dcon = -conc->cval;
       else if (conc->dcon+conc->cval>1.0) conc->dcon = 1.0 - conc->cval; /* */
       conc->cest = conc->cval + conc->dcon * 1.5;
/*       if      (conc->cest<0.0) conc->cest = 0.0;
       else if (conc->cest>1.0) conc->cest = 1.0;   /* */
/* fprintf (stderr,"1 typ %d conc %10.4g dconc %10.4g cond %g  i %d\n",
		chpnt->ctype,conc->cest,conc->dcon,stpnt->cond,i); /* */
  }
  conduct = 0.0;
  conc = chpnt->conc;
  stpnt = chpnt->consts->state;
  for (i=0; i<nstate; i++,stpnt++,conc++) { /* integrate conductance */
    conduct += conc->cest * stpnt->cond;
  }

 /* fprintf (stderr,"typ %d %d c %10.4g cond %10.4g\n",
		chpnt->ctype,chpnt->stype,conduct,chpnt->conduct); /* */
 return conduct;
}
/*------------------------------------*/

double dochan2(sschan *chpnt)
              

/* Second order calculation of channel states */
/*  Assumes "cest" contains old state concentration */
/*  Leaves new estimate in "cest" */
/*  Returns conductance */

{
   int i,j,t,nstate,numtr,errcnt;
   static chanstate *stpnt;
   static stconc *conc;
   static double (**f_rate)(double v, int typ), trrate, conduct,critc,tconc;
   static float *trmul;
   static char *ratev;
   double trv;			/* voltage value for channel transition */
   char *trans;
   double cerr,cmaxerr,cest;
   static stconc oconc[NUMSTATE];
   static stconc oldconc[NUMSTATE];


#define MAXERRCNT 200
  conduct = dochan1(chpnt);	     /* do first order forward estimate */
  t  = chpnt->stype;
  nstate = chpnt->numstate;
  conc = chpnt->conc;
  for (i=0; i<nstate; i++,conc++) {  /* zero deltas */
    oldconc[i].dcon = conc->dcon;
    conc->dcon = 0;
    oconc[i].dcon = 0;
  }	
	/* calculate delta concentrations from rate functions: */

  critc = 1e-4;
  for (cmaxerr=1.0,errcnt=0; cmaxerr>critc && errcnt<MAXERRCNT; errcnt++) {
    cmaxerr = 0.0;
    tconc = 0.0;
    conc = chpnt->conc;
    stpnt = chpnt->consts->state;
    for (i=0; i<nstate; i++,stpnt++,conc++) {  /* find change in conc */
       trans = stpnt->trans;	     /* get parameters */
       f_rate  = stpnt->trate;
       trmul = stpnt->ratemul;
       numtr = stpnt->numtrans;
       ratev = stpnt->ratev;
       for (j=0; j<numtr; j++,trans++,f_rate++,trmul++,ratev++) {
         trv = (*ratev ? chpnt->comp2->v : chpnt->comp1->v) - chpnt->vthr;
         trrate = conc->cest * (*f_rate)(trv,t) * *trmul;  /* per timinc */
/*	 if (trrate > 1*conc->cest) trrate = 1*conc->cest;
	 if (trrate + chpnt->conc[*trans].cest > 1.0)
		 trrate = 1.0 - chpnt->conc[*trans].cest;
*/
/* if (chpnt->ctype==NA) fprintf
(stderr,"i %d j %d t %d trrat %10.4g cest %10.4g f_rate %10.4g v %g mul %g\n",
	i,j,t,trrate,conc->cest,(*f_rate)(trv,t),trv,*trmul); /* */

         chpnt->conc[*trans].dcon += trrate;
         conc->dcon -= trrate;
       }  /* for (j=0; j<numtr; ...) */
    }   /*  for (i=0; i<nstate; ...) */

	/* after calculating all deltas, find new conc estimates: */

    conc = chpnt->conc;
    stpnt = chpnt->consts->state;
    for (i=0; i<nstate; i++,stpnt++,conc++) {

/* Problem: we can't change deltas here because we'd slowly lose total conc */

/*       if      (conc->dcon< -conc->cest) conc->dcon = -conc->cest;
       else if (conc->dcon+conc->cval>1.0) conc->dcon = 1.0 - conc->cval; /* */
       cest = conc->cval + (conc->dcon+oconc[i].dcon)*.5; 
       if ((cest >= 1.0) || (cest<0))
         cest = conc->cval + conc->dcon;   /* if unstable use implicit */
       conc->cest = cest;
       tconc += cest;

/* also, we can't modify state conc for same reason */

/*       if      (conc->cest<0.0) conc->cest = 0.0;
       else if (conc->cest>1.0) conc->cest = 1.0;   /* */
       cerr = conc->dcon - oconc[i].dcon; 
#ifdef DEBUG
  if (debug & 16 && debugz && 1)
 fprintf(stderr,"2 typ %d conc %10.4g dconc %10.4g err %10.4g cond %g  i %d\n",
		chpnt->ctype,conc->cest,conc->dcon,cerr,stpnt->cond,i); /* */
#endif
       if (cerr < 0.0) cerr = -cerr; 	/* absolute value */
       if (cerr > cmaxerr) cmaxerr = cerr;
       oconc[i].dcon = conc->dcon;
       conc->dcon = 0;			/* erase delta for next iteration */
    }  /*  for (i=0; i<nstate; ...) */

#ifdef DEBUG
  if (debug & 16 && debugz && 1)
     fprintf (stderr,"tconc %g\n",tconc);   /* */
#endif
/* fprintf (stderr,"cmax %10.4g\n",cmaxerr);   /* */
  }    /* for (cmaxerr;;) */

  conduct = 0.0;
  conc = chpnt->conc;
  stpnt = chpnt->consts->state;
  for (i=0; i<nstate; i++,stpnt++,conc++) { /* integrate and total cond */
    conc->cest = conc->cval + (conc->dcon + oldconc[i].dcon) * .5; 
    conduct += conc->cest * stpnt->cond;
/*    if (chpnt->ctype==NA) fprintf (stderr,"%10.4g ",conc->cest); /* */
  }
/* if (chpnt->ctype==NA) 
	 fprintf (stderr,"\n"); /* */

  return (conduct);
}
/*------------------------------------*/

void dochani(sschan *chpnt, double critc)

/* implicit calculation of equilibrium state concentrations */

{
   int i,j,t,nstate,numtr,oldfl;
   static chanstate *stpnt;
   static stconc *conc;
   static double (**f_rate)(double v, int typ), trrate, conduct;
   static float *trmul;
   static char *ratev;
   double trv;			/* voltage value for channel transition */
   char *trans;
   double cerr,cmaxerr;
   static stconc oconc[NUMSTATE];
   static stconc oldna[NUMSTATE];
   static stconc oldk[NUMSTATE];
   static double oldv = 1e30;
   static double oconduct=1e30;
   static int ostypena= -100;
   static int ostypek = -100;

#ifdef DEBUG
  if (debug & 1 && debugz && 16)
	 fprintf (stderr,"dochani type %d %d\n",chpnt->ctype,chpnt->stype);
#endif

  t  = chpnt->stype;
  oldfl = 0;
  if (oldv==chpnt->comp1->v) {
    switch (chpnt->ctype) {
     case NA:
      if (ostypena==chpnt->stype){
        nstate = chpnt->numstate;
        conc = chpnt->conc;
        for (i=0; i<nstate; i++,conc++) {  /* copy the old equilibrium values */
          conc->cest = oldna[i].cest;
        }
        oldfl = 1;
      }
      break;
     case K:
      if (ostypek==chpnt->stype){
        nstate = chpnt->numstate;
        conc = chpnt->conc;
        for (i=0; i<nstate; i++,conc++) {  /* copy the old equilibrium values */
          conc->cest = oldk[i].cest;
        }
        oldfl = 1;
      }
      break;
    }
  }  /* if (oldv) */

  if (oldfl) {
/* fprintf (stderr,"using old values %d\n",chpnt->ctype); */
        return;
  }

  nstate = chpnt->numstate;
  conc = chpnt->conc;
  for (i=0; i<nstate; i++,conc++) {  /* zero deltas */
    conc->dcon = 0;
    oconc[i].dcon = 0;
  }	
	/* calculate delta concentrations from rate functions: */

  for (cmaxerr=1.0; cmaxerr>critc; ) {
    cmaxerr = 0.0;
    conc = chpnt->conc;
    stpnt = chpnt->consts->state;
    for (i=0; i<nstate; i++,stpnt++,conc++) {  /* find change in conc */
       trans = stpnt->trans;	     /* get parameters */
       f_rate  = stpnt->trate;
       trmul = stpnt->ratemul;
       numtr = stpnt->numtrans;
       ratev = stpnt->ratev;
       for (j=0; j<numtr; j++,trans++,f_rate++,trmul++,ratev++) {
         trv = (*ratev ? chpnt->comp2->v : chpnt->comp1->v) - chpnt->vthr;
         trrate = conc->cest * (*f_rate)(trv,t) * *trmul;  /* per timinc */
/*	 if (trrate > conc->cest) trrate = conc->cest;
	 if (trrate + chpnt->conc[*trans].cest > 1.0)
		 trrate = 1.0 - chpnt->conc[*trans].cest;
*/
/* if (chpnt->ctype==NA)
  fprintf (stderr,"i %d j %d trrate %10.4g cest %10.4g f_rate %10.4g mul %g\n",
	i,j,trrate,conc->cest,(*f_rate)(trv,t),*trmul); /* */
         chpnt->conc[*trans].dcon += trrate;
         conc->dcon -= trrate;
       }
    }
	/* after calculating all deltas, find new conc estimates: */

    conc = chpnt->conc;
    stpnt = chpnt->consts->state;
    for (i=0; i<nstate; i++,stpnt++,conc++) {
/*       if      (conc->dcon< -conc->cest) conc->dcon = -conc->cest;
       else if (conc->dcon+conc->cest>1.0) conc->dcon = 1.0 - conc->cest; /* */
       conc->cest += conc->dcon  * 1.0;
/*      if      (conc->cest<0.0) conc->cest = 0.0;
       else if (conc->cest>1.0) conc->cest = 1.0;   /* */
       cerr = conc->dcon; 
/* fprintf
    (stderr,"init typ %d conc %10.4g dconc %10.4g err %10.4g cond %g i %d\n",
		chpnt->ctype,conc->cest,conc->dcon,cerr,stpnt->cond,i); /* */
       if (cerr < 0.0) cerr = -cerr; 	/* absolute value */
       if (cerr > cmaxerr) cmaxerr = cerr;
       oconc[i].dcon = conc->dcon;
       conc->dcon = 0;			/* erase delta for next iteration */
    }
/* fprintf (stderr,"\n");   /* */
  }    /* for (cmaxerr;;) */

  conduct = 0.0;
  conc = chpnt->conc;
  stpnt = chpnt->consts->state;
  for (i=0; i<nstate; i++,stpnt++,conc++) { /* integrate and total cond */
    switch (chpnt->ctype) {
      case NA: oldna[i].cest = conc->cest;
    		ostypena = chpnt->stype; 
	        break;
      case K:  oldk[i].cest  = conc->cest;
    		ostypek = chpnt->stype; 
		 break;
    }
  }
  oldv = chpnt->comp1->v;

#ifdef DEBUG
  if (debug & 1 && debugz && 16) fprintf (stderr,"dochani end\n");
#endif

  return;
}

/*------------------------------------*/

void docacomp (cacomp *capnt)
                

/* Calculate calcium concentration for next time step. 
   Concentration is computed for thin shell at inner surface
   of membrane.  Calcium diffuses through ten shells and then
   into the inner volume of the cell.  Calcium flux is converted
   into an equivalent concentration at a shell outside the membrane.
   This allows the diffusion equation to be run identically for
   all the compartments (except the inner core).  

   Based on Hines, 1989, Int. J. Biomed. Comput. 24: 55-68.

   Calcium diffusion const = 6e-6 cm2/sec (Hodgkin and Keynes 1957) 

   Use equation		caflux = D * time * surf/dist * (C1 - C2).

*/

{
  int i;
  comp *pnt;
  static conlst *lpnt;
  static chan *chpnt;
  static hhchan *hhpnt;
  extern double vna;
  double c,v,vm,conduct,cratio,vfact,ica,eica,pica,ina;
  double caflux,eex,vmax,kex,cfactr,ca1,cai,tcai,*cais,camaxerr,err;
  double casf0,casfn,casfc,cabnd;
  static double caise[NUMCASH] = {0};	/* calcium internal shell estimates */
  static double caiso[NUMCASH] = {0};	/* calcium shell old estimates */
  static int exchr=3;			/* sodium-calcium exchange ratio */

#ifdef DEBUG
  if (debug & 1 && debugz && 16) fprintf (stderr,"doca\n");
#endif

  if (!(pnt=capnt->comp1)) {		/* voltage compartment */
     fprintf (stderr,"doca: can't find voltage compartment\n");
     return;
  } 
  vm = pnt->v;

		/* First calculate ca channel conductances. */
		/* Could be several ca channels in this ca compartment */

  ica = 0.0;
  for (lpnt=capnt->clst; lpnt; lpnt=lpnt->next) {
     chpnt = (chan *)lpnt->conpnt;
     if (! chpnt) continue;
/* fprintf (stderr,"doca: chan %d %d\n",chpnt->ctype,chpnt->stype); /* */
     switch (chpnt->stype) {
       case 0:
          hhpnt = (hhchan *)chpnt; 
          catab(vm-hhpnt->vthr,	/* set Na rate consts */ 
	      hhpnt->stype,hhpnt->taum);	
          c = hhpnt->m * c1 + c2;
          hhpnt->m = c;

  /*      if((cratio=capnt->cais[1]/capnt->cao) < 1.0) { 
            vfact = exp(vm*F2R/ktemp);
            cfactr = c*c*c * (1.0-cratio*vfact) / (1.0 - vfact);
          }
          else cfactr = c*c*c;
  */
          cfactr = c*c*c;
          break;
       case 1:
       case 2:
		/* add sequential-state ca channels here */
            cfactr = c*c*c;
       break;
     }

     conduct = cfactr * chpnt->maxcond;
     chpnt->conduct = conduct;
     v = hhpnt->vrev - vm;
     ica += conduct * v;		/* calcium current, used here only */
  }

  capnt->ica = ica;			/* current through ca-selective chan */
  ca1 = capnt->cais[1];
  cai = capnt->cai;			/* starting calcium conc. inside */

  if ((kex=capnt->kex) > 0) {		/* sodium-calcium exchanger */
     eex = (exchr*vna - 2.0*capnt->vrev) / (exchr-2);
     eica = -kex * (vm - eex) * ca1 / (ca1 + capnt->ekm); /* exch ica */
     ina = -exchr * eica / 2.0;		/* sodium current */
     ica += eica;			/* additional calcium current */
  }
  else eica = ina = 0.0;

  if ((vmax=capnt->vmax) > 0) { 	/* calcium pump */
     tcai = ca1-cai;
     pica = -vmax * tcai / (capnt->pkm + tcai);
     ica += pica;			/* pumped calcium */
  }
  else pica = 0.0;

		 /* Compute Ca concentration at inner mem surface. */

/* fprintf (stderr,"ica %g pica %g eica %g ina %g\n",ica,pica,eica,ina); /* */

  capnt->ipump = eica + pica + ina;	/* total pump current */

  caflux = ica * timinc * F2;   	/* flux in moles */

/* fprintf (stderr,"ica %g caflux %g\n",ica,caflux);   /* */

  cais  = capnt->cais;
  casf0 = capnt->casf0;			/* shell factor = dr/(D*t*4PI*r*r) */
  casfn = capnt->casfn;			/* shell factor = 1 / (dr*dr) */
  casfc = capnt->casfc;			/* shell factor for core */
  cabnd = capnt->cabnd;			/* 1 / (1+ratio of ca bound / free) */

  for (i=0; i<NUMCASH; i++) {		/* reset old estimates */
    caise[i] = cais[i];
    caiso[i] = 0.0;
  }

 cais[0] = cais[1] + caflux * casf0;	/*convert ca flux to equiv conc.*/
 caise[0] = cais[0];

 crit = 1e-10;
 for (camaxerr=1e8; camaxerr>crit; ) {	/* relaxation */

  camaxerr = 0.0;

 if (implicit) {
  for (i=1; i<(NUMCASH-1); i++) {	     /* then solve ca concentration */
   caise[i]= (cais[i]+(caise[i-1]+caise[i+1])*casfn*cabnd)/(1.0+2.0*casfn*cabnd);
  }
  caise[i]= (cais[i] + caise[i-1]*casfc * cabnd) /
				(1.0+casfc*cabnd);   /* core */
 }

 else {		/* CN */

  for (i=1; i<(NUMCASH-1); i++) {	     /* then solve ca concentration */
   caise[i]= (cais[i]+
    (caise[i-1]+caise[i+1]+cais[i-1]+cais[i+1]-2.0*cais[i]) *
	0.5*casfn*cabnd) /(1.0+2.0*casfn*0.5*cabnd);
  }
  caise[i]= (cais[i] + (caise[i-1]+cais[i-1]-cais[i])*casfc * 0.5 * cabnd) /
				(1.0+casfc*0.5*cabnd);   /* core */
  }
				/* find error */
  for (i=0; i<NUMCASH; i++) {
    if ((err=ncabs(caise[i]-caiso[i])) > camaxerr) camaxerr = err;
    caiso[i] = caise[i];		/* save old estimate */
  }

/*  fprintf (stderr,"doca: camaxerr %g\n",camaxerr);  /* */

 }	/* for (camaxerr;;) */


  for (i=0; i<NUMCASH; i++) {
       cais[i] = caise[i];			/* save estimate */
  }

		 /* Compute calcium reversal potential */
		 /* Use Nernst equation */

  capnt->vrev = R2F*ktemp*log(capnt->cao/capnt->cais[1]); 

		/* Copy calcium reversal potential into channels */

  for (lpnt=capnt->clst; lpnt; lpnt=lpnt->next) {
     chpnt = (chan *)lpnt->conpnt;
     if (! chpnt) continue;
     chpnt->vrev = capnt->vrev;
  }

#ifdef DEBUG
  if (debug & 1 && debugz && 16) fprintf (stderr,"doca end\n");
#endif

  return;
}

/*------------------------------------*/

void runcomp(void)

/* Advance the network of compartments one step in time,
using a second-order implicit numerical integration.
Iterate the approximation until the error is
reduced below criterion.
*/

{
  static int left,niter,titer,less,noskip;
  static int stype;
  static double nodcur;                    /* total node current */
  static double nodcr;                     /* total unvarying node curr */
  static comp *pnt,*ocpnt;
  static conn *conpnt;
  static double delcrit,tdelcrit;
  static double relmult,critx,critxx,critk;
  static double tcond;			   /* total conductance in comp */
  static double tcondn;			   /* tot cond not varying w/vest */
  static double conduct;		   /* conductance of channel */
  static double vest;		   	   /* conductance of channel */
  static conlst *lpnt;
  static recpar *cpt;                      /* receptor constants */
  static hhchan *hhpnt;
  static kcachan *kcapnt;
  static sschan *chpnt;
  static cacomp *capnt;
  static double a,m,n,h,dm,dn,dh;
  static double mmaxerr;		   /* max maxerr for level of critx */
  static double oldmaxerr;		   /* looking for maxerr increases */

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"runcomp\n");
#endif

 
 if (timinc > 1.0) runsyn(0);			/* if static, calc synapses */


 if (euler) implicit = 0;

 for (pnt=compnt; pnt; pnt=pnt->next)                   /* First estimate */
  {                                                     /* for all comps */
   nodcur = nodcr = 0.0;
   tcond = tcondn = 0.0;
   if (capnt=pnt->capnt) {		/* Compute calcium concentration. */
     docacomp(capnt);			/* Must do first because other chans */
   }					/*  may be dependent on ca. */

   for (lpnt=pnt->clst; lpnt; lpnt=lpnt->next) 
    {                                           /* check all connections */
     conpnt = lpnt->conpnt;
     if (! conpnt) break;
     if (conpnt->comp1 == pnt) ocpnt = conpnt->comp2; /* get other compartment*/
     else                      ocpnt = conpnt->comp1;
     switch (conpnt->ctype)
      {                         /* add connection conductances which change */

       case GJ:                                 /* may vary gj in future */
                nodcur += conpnt->conduct * ocpnt->v;   /* current thru g j */
                tcond += conpnt->conduct;              /* total cond */
               break;
       case SYNAPSE:					 /* cur thru syn chan */
		if (pnt==((synap *)conpnt)->comp2) {
                  nodcr += ((synap *)conpnt)->conduct *((synap *)conpnt)->vrev;
                  tcondn += ((synap *)conpnt)->conduct;   /* total cond */
  	/* fprintf (stderr,"vrev %10.3g conduct %10.3g\n",
		((synap *)conpnt)->vrev,((synap *)conpnt)->conduct); /* */
		}
               break;

      case ROD:
      case CONE:                                /* find receptor instantan */
                                                /* conductance */
          cpt = ((recep *)conpnt)->consts;
          nodcr += ((recep *)conpnt)->conduct * cpt->vrev;/* cur thru rec chan*/
          tcondn  += ((recep *)conpnt)->conduct;               /* total cond */
/*   fprintf (stderr,"vrev %10.3g conduct %10.3g\n",
		cpt->vrev,((recep *)conpnt)->conduct); /* */
         break;

      case LOAD:
                nodcr += ((load *)conpnt)->conduct * 
			  ((load *)conpnt)->vrev;     /* cur to "ground" */
                break;              /* conductance is collected in pnt->tcond */

      case AXIALRES:
      case RESISTOR:
                nodcur += conpnt->conduct * ocpnt->v;  /* cur to next comp */
                break;              /* conductance is collected in pnt->tcond */
      case BATT:
     		if (ocpnt == conpnt->comp1)
                  nodcur += (ocpnt->v - conpnt->conduct) * BATTCOND;
		else 
                  nodcur += (ocpnt->v + conpnt->conduct) * BATTCOND;
                break;		/* cur to next comp through batt */

      case BUF:	if (pnt == ((dbuf *)conpnt)->comp2) {
		   if (((dbuf *)conpnt)->delay) {
		     pnt->extv = *((dbuf *)conpnt)->delpnt * DELCONV;
		   }
		   else pnt->extv = ((dbuf *)conpnt)->comp1->v;
		  }
		break;

      case CAP: nodcr += (pnt->oldv - ocpnt->oldv) * conpnt->conduct/timinc;
                break;			/* -(v3 - v2) * c / t */

      case NA:  hhpnt = (hhchan *)conpnt;
		stype = hhpnt->stype;
		switch (stype) {

		case 0:				/* Hodgkin-Huxley */

      		natab(pnt->v-hhpnt->vthr,	/* set Na rate consts */ 
		      stype,hhpnt->taum,hhpnt->tauh);	
		m = hhpnt->m * m1 + m2;
		h = hhpnt->h * h1 + h2;

		hhpnt->m = m;
		hhpnt->h = h;
  		conduct = m*m*m*h * hhpnt->maxcond;
		hhpnt->conduct = conduct;
                nodcr += conduct * hhpnt->vrev;
                tcondn += conduct;    			/* total cond */
		break;

		case 1:				/* Sequential state */
		case 2:

	 	chpnt = (sschan *)conpnt;
  		conduct = dochan2(chpnt) * chpnt->maxcond;
/* 		if (nsites=chpnt->csites) {			  /* */
/*    		  p = conduct / (nsites * chpnt->cq);     /* find open prob */
/*    		  conduct = bnldev(p,nsites) * chpnt->cq; /* conduct w/noise */
/*  		} */
		chpnt->conduct = conduct;
                nodcr += conduct * chpnt->vrev;
                tcondn += conduct;    			/* total cond */
		break;

		}	/* switch (stype) */
                break;

      case K:   
	 	hhpnt = (hhchan *)conpnt;
		stype = hhpnt->stype;
		switch (stype) {

		case 0:				/* Hodgkin-Huxley */

     		ktab(pnt->v - hhpnt->vthr,	/* set K rate consts */ 
		     stype,hhpnt->taum);
		n = hhpnt->m * n1 + n2;
		hhpnt->m = n;
  		conduct = (n*n)*(n*n) * hhpnt->maxcond;
		hhpnt->conduct = conduct;
                nodcr += conduct * hhpnt->vrev;
                tcondn += conduct;    			/* total cond */
                break;

		case 1:				/* Sequential state */
	 	chpnt = (sschan *)conpnt;
  		conduct = dochan2(chpnt) * chpnt->maxcond;
/* 		if (nsites=chpnt->csites) {			  /* */
/*    		  p = conduct / (nsites * chpnt->cq);     /* find open prob */
/*    		  conduct = bnldev(p,nsites) * chpnt->cq; /* conduct w/noise */
/*  		} */
		chpnt->conduct = conduct;
                nodcr += conduct * chpnt->vrev;
                tcondn += conduct;    		/* total cond */
		break;

		case 2:
	 	kcapnt = (kcachan *)conpnt;
		if (pnt->capnt) {  /* calcium compartment exists */

		  if (!kcapnt->initfl){	/* Calcium not set up before K chan. */
			double ca;	/* Find equilibrium value for "n". */ 
			double alph,bet;
		    ca = pnt->capnt->cais[1];
		    alph = akcacalc((pnt->v-kcapnt->vthr),ca,
			kcapnt->taum,kcapnt->d1,kcapnt->k1);
		    bet = bkcacalc((pnt->v-kcapnt->vthr),ca,
			kcapnt->taum,kcapnt->d2,kcapnt->k2);
		    kcapnt->m = alph / (alph + bet);
		    kcapnt->initfl = 1;
		  }
     		  kcatab((pnt->v-kcapnt->vthr),    /* set K rate consts */ 
		     pnt->capnt->cais[1],stype,kcapnt->taum,
		     kcapnt->d1,kcapnt->d2,kcapnt->k1,kcapnt->k2);
		}
		else {		/* Can't find ca comp for kca. */
		   kca1 = 1;	/* Don't change "n" */
		   kca2 = 0;
		}
		n = kcapnt->m * kca1 + kca2;
		kcapnt->m = n;
  		conduct = n * kcapnt->maxcond;
		kcapnt->conduct = conduct;
                nodcr += conduct * kcapnt->vrev;
                tcondn += conduct;    			/* total cond */
		break;

		case 3:				/* Type A channel */
						/* fast inactivating */
     		ktab(pnt->v - hhpnt->vthr,	/* set K rate consts */ 
		     stype,hhpnt->taum);
		a = hhpnt->m * n1 + n2;
		h = hhpnt->h * d1 + d2;
		hhpnt->m = a;
		hhpnt->h = h;
  		conduct = a*a*a*h * hhpnt->maxcond;
		hhpnt->conduct = conduct;
                nodcr += conduct * hhpnt->vrev;
                tcondn += conduct;    			/* total cond */
                break;

		}	/* switch (stype) */
		break;

       case CA:   
	 	hhpnt = (hhchan *)conpnt;
		stype = hhpnt->stype;
		switch (stype) {

		case 0:				/* Hines Ca type */
		conduct = hhpnt->conduct; 
                nodcr += conduct * hhpnt->vrev;
   		if (capnt)
		  nodcr += capnt->ipump;	/* extra currents */
                tcondn += conduct;    		/* total cond */
/* fprintf (stderr,"conduct %g ipump %g\n",conduct,capnt->ipump); /* */
		break;

		case 1:				/* Sequential-state type */
		break;

		}     /* switch (stype) */
		break;

      default:  break;

      } /* switch */
    }      /* for (lpnt=; ;) */

   tcond += tcondn + pnt->tcond;                 /* total conductance */
   nodcur -= pnt->v * tcond;                     /* current due to comp volts */
   nodcr  += pnt->vrev * pnt->rm;                /* membrane conductance */

/*  fprintf (stderr,"pnt->vrev %10.3g\n",pnt->vrev); /* */
/*  fprintf (stderr,"nodcr %10.3g nodcur %10.3g pntcond %g tcond %g\n",
			nodcr,nodcur,pnt->tcond,tcond); /* */

   if (pnt->miscfl & IEXT)                    /* comp has a current src */
      nodcr += pnt->exti;                     /* external current */
   nodcur += nodcr;			      /* total comp current */

 if (euler)	{     			      /* forward-Euler */
     vest = pnt->v + nodcur * pnt->k;	      /* simple first-order est */
     if (pnt->miscfl & VEXT) {                /* comp is a battery */
       pnt->extvi = (vest - pnt->extv) / (pnt->k);
       vest = pnt->extv;		      /*  battery voltage  */
     }
     else {
       pnt->extvi = 0.0;
     }
     pnt->vest = vest;				/* just save the estimate */
 }						/* (not quite) end of Euler */
						/*   (see below) */

 else {						/* Implicit modes */

   pnt->tcondn = tcondn;			/* cond not varying w/vest */
   pnt->implf = 1. / (1. + tcond * pnt->k);   /* implicit factor   */

   if (implicit) {			/* Backward Euler, fully implicit */
     pnt->nodc = nodcr;				/* curr for estimate */
     nodcur *=  pnt->k;                      	/* first-ord only */
     pnt->vest = pnt->v + nodcur;             /* simple first-order est */
     if (pnt->miscfl & VEXT) {                   /* comp is a battery */
          pnt->extvi = (pnt->vest - pnt->extv) / (pnt->k);
          pnt->vest = pnt->extv;
       }
     else {
       pnt->extvi = 0.0;
     }
     pnt->verr = 1.;
   }						/* end of implicit */

   else	{     /* Crank-Nicolson */
     pnt->nodc = nodcur + nodcr;	      /* curr for estimate */
     nodcur *=  pnt->k * 2.;                  /* first-ord only */
     pnt->vest = pnt->v + nodcur;             /* simple first-order est */
     if (pnt->miscfl & VEXT) {                /* comp is a battery */
       pnt->extvi = (pnt->vest - pnt->extv) / (pnt->k*2.);
       pnt->vest = pnt->extv;		      /*  battery voltage  */
     }
     else {
       pnt->extvi = 0.0;
     }
     pnt->verr = 1.;
   }    /* crank-nicholson */
  }  /* CN or implicit */
 }   /* for (pnt=;;) */

 if (euler) {					/* after everything else, */
   for (pnt=compnt; pnt; pnt=pnt->next) {
	pnt->v = pnt->vest;			/* save as new voltage */
   }
   return;
 }

 relmult = relincr * timinc / 1e-4;
 mmaxerr=oldmaxerr=maxerr = 1.0;                 /* set maxerr so loop runs */ 
 ncomp = tcomp = 0; 
 delcrit =   SQRT10; /* */
 left = 1;
for (titer=0,critx=1.000001e-2;
		 mmaxerr >= crit;
		 critx = min(mmaxerr,critx),
		  critx /= delcrit, maxerr = 1.0, titer+=niter, left++) /* */
 {
 critxx = critx;
 oldmaxerr=1.0;
 mmaxerr = 0.0;

 for (niter=0; maxerr>critx; niter++)	/* Iterate until convergence */
  {
   if (timinc > 1.0) runsyn(0);		/* if static, calc synapses */

/* fprintf (stderr,"maxerr %g critx %g\n",maxerr,critx); /* */


   maxerr = 0.0;
   if (left&1)
     for (ncomp=0,pnt=compend; pnt; pnt=pnt->last)     /* left estimate */
      {
#ifdef DEBUG
        if (!(debug&NOSKIP))
#endif
        if (!noskip)
          if (pnt->verr<critxx) continue; /* */
        docomp(pnt);
      }
   else
     for (ncomp=0,pnt=compnt; pnt; pnt=pnt->next)      /* right estimate */
      {
#ifdef DEBUG
        if (!(debug&NOSKIP))
#endif
        if (!noskip)
          if (pnt->verr<critxx) continue; /* */
        docomp(pnt);

      noskip = 0;
      }           /* for (ncomp=0,pnt=compnt;) */


   /* Reduce the criterion if error increases, and also at least
       once every 10 iterations.  This causes compartments
       outside the region of iteration (i.e. otherwise skipped over)
       to be recomputed and reduces the total number of iterations
       necessary for complete convergence.
   */
       
  if (maxerr>oldmaxerr || ((niter>0) && (niter%10==0))) {
    tdelcrit = sqrt(delcrit);
    if (tdelcrit>1.05) {
      delcrit=tdelcrit;			/* */
#ifdef DEBUG
      if (debug & 4 && debugz & 1) {
	 if (maxerr>oldmaxerr)
	 fprintf (stderr,"Maxerr incr:   reducing delcrit to %8.4g\n",delcrit);
	 else
	 fprintf (stderr,"10 iterations: reducing delcrit to %8.4g\n",delcrit);
      }
#endif
    } 
      oldmaxerr = 1.0; 
      noskip = 1; 
    critxx /= delcrit;

#ifdef DEBUG
  if (debug & 4 && debugz & 1)
	 fprintf (stderr,"maxerr %8.3g, reducing critxx to %8.3g\n",
					maxerr,critxx);
#endif

   }  /* if (maxerr>oldmaxerr || ) */

   else {
      oldmaxerr = maxerr; 
   }

   if (critxx < critx*1e-2) noskip = 1; 	/* don't skip when unstable */

   if (mmaxerr<maxerr) mmaxerr = maxerr;   /* remember largest maxerr */

#ifdef DEBUG
  if (debug & 4 && debugz & 1)
   fprintf 
    (stderr,"ncomp %4d mmaxerr %8.3g maxerr %8.3g critxx %8.3g\n",
					ncomp,mmaxerr,maxerr,critxx);
#endif

   if (critxx < 1e-30) {
      fprintf (stderr,"Ncomp: panic, no convergence, continuing...");
      return;
   }
 
 }            /* for (niter=0; maxerr>critx; niter++)  */




/* Reset error after each level of critx */
/*  This has the effect of spreading the region of iteration,
    and causes compartments just outside to be recomputed.
*/


 critk = .9 * critx / delcrit;
 for (pnt=compnt; pnt; pnt=pnt->next)
  {
   pnt->verr += critk;
  }			/* */


#ifdef DEBUG
  if (debug & 2 && debugz & 1)
   fprintf 
 (stderr,"niter %2d   mmaxerr %8.3g maxerr %8.3g critx  %8.3g critk %7.3g\n",
				niter,mmaxerr,maxerr,critx,critk);
#endif

  }         /* for (critx ; ; ) */

#ifdef DEBUG
  if (debug & 2 && debugz & 1)
	fprintf (stderr,"*tcomp %4d titer %d\n",tcomp,titer);
#endif

/* Just save it and calculate relax for next time step. */

 for (pnt=compnt; pnt; pnt=pnt->next)
  {
   pnt->oldv = pnt->v = pnt->vest;		/* save it */
  }

/* This code runs only once per time step, */
/* so it adds negligible time to computataion */
/* if model is large and requires many iterations */
/* per time step. */

if (relincr && titer > 10)			/* if lots of iterations */
 for (pnt=compnt; pnt; pnt=pnt->next) {
   if (!(pnt->miscfl & VEXT)) {			/* if comp is not a battery */
      less = pnt->t - pnt->g;			/* then compute oscillations */
      m = ((pnt->g < less) ? less : pnt->g);	/* max of less, greater */
      if (m==pnt->t) pnt->relax += relmult;	/* adjust comp relax */
      else           pnt->relax -= (1.0 - ((double) m)/pnt->t) * 10.0 * relmult;
      if      (pnt->relax > 1.0) pnt->relax = 1.0;
      else if (pnt->relax < 0.0) pnt->relax = 0.0;
    }

#ifdef DEBUG
  if (debug & 8 && debugz & 1)
     fprintf (stderr,"c %-4d relax %-8.4g g %-4d t %-4d misc %d\n",
			   pnt->num,pnt->relax,pnt->g,pnt->t,pnt->miscfl); /* */
#endif
   pnt->g = pnt->t = 0;				/* reset greater and total */
  }

#ifdef DEBUG
  if (debug & 1) fprintf (stderr,"runcomp end.\n");
#endif
}

/*------------------------------------*/

void docomp(comp *pnt)
{
  static double nodcur;                         /* total comp current */
  static double tcondv;                         /* total comp conductance */
  static double conduct;                        /* channel conductance */
  static double implf;				/* new implicit factor */
  static comp *ocpnt;
  static conn *conpnt;
  static double err;
  static double icomp;
  static conlst *lpnt;
  static hhchan *hhpnt;
  static double m,n,h,dm,dn,dh;
  int i;

#ifdef DEBUG
     tcomp++;
     ncomp++;
#endif
     nodcur = pnt->nodc;                        /* start with old node cur */
     tcondv = 0.0;				/* non-varying conductance */
     for (lpnt=pnt->clst; lpnt; lpnt=lpnt->next) 
      {
       conpnt = lpnt->conpnt;                   /* check connection */
       if (! conpnt) break;
       if (conpnt->comp1 == pnt) ocpnt = conpnt->comp2; /* get other comp */
       else                      ocpnt = conpnt->comp1;
       switch (conpnt->ctype)
        {
         case AXIALRES:
         case RESISTOR:
         case GJ:
                nodcur += ocpnt->vest * conpnt->conduct;  /* curr thru resist */
                break;
         case BATT:
     		if (ocpnt == conpnt->comp1) 
                  nodcur += (ocpnt->vest - conpnt->conduct) * BATTCOND;
		else
                  nodcur += (ocpnt->vest + conpnt->conduct) * BATTCOND;
                break;		     /* curr thru batt small resistor */

         case BUF: if (pnt == ((dbuf *)conpnt)->comp2) {
		     if (((dbuf *)conpnt)->delay) {
		       pnt->extv = *((dbuf *)conpnt)->delpnt * DELCONV;
		     }
		     else pnt->extv = ((dbuf *)conpnt)->comp1->vest;
		   }
		break;
         case CAP:
		nodcur += ocpnt->vest * conpnt->conduct / timinc;	
                break; 
         case NA:
      	 case K:   
         case CA:
		break;

         case SYNAPSE:                            /* currents don't need est */
         case ROD:
         case CONE:
         case LOAD:
         default:
                break;
        }  /* switch */
       }  /* for (lpnt= ; ; ) */

     err = pnt->vest;
     if (tcondv) {				/* if any new conductances */
        implf = 1. / (1. + (tcondv+pnt->tcondn+pnt->tcond) * pnt->k);
        pnt->vest = (pnt->v + nodcur * pnt->k) * implf;
     }
     else  {
 	implf = pnt->implf;
        pnt->vest = (pnt->v + nodcur * pnt->k) * implf;
     }
     if (pnt->miscfl & VEXT) {			/* comp is a battery */
	icomp = (pnt->vest - pnt->extv) / (pnt->k * implf);
	if (implicit) {
           pnt->extvi = icomp;
        }
	else {    			/* crank-nicholson */
           pnt->extvi = icomp * 0.5;
	}   
       pnt->vest = pnt->extv;
       pnt->verr = 1.0;			/* set verr so vclamp always runs */
     }
     else {
        err -= pnt->vest;                         /* find convergence deriv */
        pnt->vest -= err * pnt->relax;
        pnt->t++;
        if (err>0.0) pnt->g++;
	else err = -err;
        pnt->verr = err;
        if (err > maxerr) maxerr = err;
     }
}

/*------------------------------------*/

