/* Module ncmak in program nc */

/* Makes cable branches, nodes, compartments,
   connections, and synapses. 
*/


#include <stdio.h>
#include <math.h>
#include "adef.h"
#include "nc.h"
#include "y.tab.h"
#include "ncelem.h"
#include "ncomp.h"
#include "ncsub.h"
#include "control.h"

#define DEBUG 		/* */

#ifndef XSTIM
#define XMOD                    /* define XMOD when not XSTIM */
#endif

#ifdef XSTIM
#define XSTIMM          	/* define XSTIMM when stim should ignore */
#endif                          /*    neural elements for speed */

/* #define MALLOPT		/* */

#ifdef MALLOPT
#include <malloc.h>
#endif

#define MPI 3.14159265358979323846264


int cumelem=0;				/* lengths of lists */
int cumnode=0;
int cumcomp=0;
int cumconn=0;
int cumchan=0;
int cumsynap=0;
int cumload=0;
int cumrec=0;
int cumrecst=0;
int cumcacomp=0;
int reccum = 0;

elem *elempnt=0;			/* pointers to lists */
elem *elemend=0;			
elem *oepnt=0;			
extern elem *elpnt;			
node *nodepnt=0;
node *nodend=0;
comp *compnt=0;
comp *compend=0;
conn *connpnt=0;
conn *connend=0;
synap *synpnt=0;
synap *synend=0;
load *loadpnt=0;
load *loadend=0;
recep *recpnt=0;
recep *recend=0;
recstim *recspnt=0;
recstim *recsend=0;

#ifdef XSTIM
#include "stim.h"
extern recnod *reclist;
extern recnod *reclend;
#endif

extern chanpar natypes[];
extern chanpar ktypes[];
extern chanpar catypes[];
extern float pigmlen[];		/* path length through o.s. (in ncstim) */


extern double alpham,betam;
extern double alphah,betah;
extern double alphan,betan;
extern double alphad,betad;
extern double alphac,betac;

#ifdef __cplusplus
extern "C" {
#endif

   double pow(double, double);
   double log(double);
   double sqrt(double);
   void free (...);
#ifdef __cplusplus
}
#endif

double ncabs(double x);
char *emalloc(unsigned int n);
double calctau(double tau);
double akcacalc(double v, double ca, double tau, double d1, double k1);
double bkcacalc(double v, double ca, double tau, double d2, double k2);
void execerror (char *s, char *t);
void freelst (conlst *clst);
void maklst(conlst **head, conn *cpnt);
void coneset(int n);
void saverec(recep *rpnt);
void restorec(recep *rpnt);
void maktables(double timinc);
void narate(double v,int type);
void krate(double v,int type);
void carate(double v,int type);
void dochani(sschan *chpnt, double crit);

#ifndef XSTIM
void ninithash(void);
#endif

/*------------------------------------------------*/

void initmk(void)

/* reset all lists */

{
#ifdef MALLOPT
  static int called=0;
if (!called) {
   called = 1;
   mallopt(M_MXFAST,sizeof(comp)+1);	/* allocate comps, nodes, etc. faster */
   mallopt(M_NLBLKS,100);		/* set size of holding blocks  */
   mallopt(M_GRAIN,8);			/* set grain size  */
}
#endif

cumelem=0;				/* cumulative lengths of lists */
cumnode=0;
cumcomp=0;
cumconn=0;
cumchan=0;
cumcacomp=0;
cumsynap=0;
cumload=0;
cumrec=0;
cumrecst=0;
reccum = 0;

elempnt=0;				/* pointers to lists */
elemend=0;
oepnt=0;		/* used by initcomp() in "ncinit.c" */
nodepnt=0;
nodend=0;
compnt=0;
compend=0;
connpnt=0;
connend=0;
synpnt=0;
synend=0;
loadpnt=0;
loadend=0;
recpnt=0;
recend=0;
recspnt=0;
recsend=0;

#ifdef XSTIM
 reccum = 0;                     /* reset number of receptors */
 reclist = 0;
 reclend = 0;
#endif
}

/*------------------------------------------------*/

void ncleanup(void)

/* erase all lists */

{
   attrib *apnt,*tapnt;
   elem *epnt,*tepnt;
   comp *cpnt,*tcpnt;
   conn *cnpnt,*tcnpnt;
   node *npnt, *tnpnt;
   synap *spnt,*tspnt;
   load  *lpnt,*tlpnt;
   recep *rpnt,*trpnt;
   recstim *rspnt, *trspnt;
#ifdef XSTIM
   recnod *npt, *tnpt;
#endif

for (epnt=elempnt; epnt; ) {		/* elements */
    for (apnt=epnt->attpnt; apnt; ) {	/* free element's attributes first */
        tapnt = apnt;
        apnt = apnt->attpnt;
	free (tapnt);
    }
    tepnt = epnt;
    epnt = epnt->next;
    free (tepnt); 			/* then free the element */
}

 for (npnt=nodepnt; npnt; ) {		/* nodes */
    freelst (npnt->elemlst);		/* free the node's element list */
    tnpnt = npnt;
    npnt = npnt->next;
    free (tnpnt);
}
#ifndef XSTIM
 ninithash();				/* erase the node hash table */
#endif

 for (cpnt=compnt; cpnt; ) {		/* compartments */
    freelst (cpnt->clst);		/* free the compartment's list */
    tcpnt = cpnt;
    cpnt = cpnt->next;
    free (tcpnt);
}

 for (cnpnt=connpnt; cnpnt; ) {		/* connections */
    tcnpnt = cnpnt;
    cnpnt = cnpnt->next;
    free (tcnpnt);
}

 for (lpnt=loadpnt; lpnt; ) {		/* loads */
    tlpnt = lpnt;
    lpnt = lpnt->next;
    free (tlpnt);
}

 for (spnt=synpnt; spnt; ) {		/* synapses */
    tspnt = spnt;
    if (spnt->filt1) free (spnt->filt1);
    if (spnt->filt2) free (spnt->filt2);
    if (spnt->filt3) free (spnt->filt3);
    spnt = (synap*)spnt->next;
    free (tspnt);
}

for (rpnt=recpnt; rpnt; ) {		/* photoreceptors */
    trpnt = rpnt;
    rpnt = (recep *)rpnt->next;
    free (trpnt);
}

for (rspnt=recspnt; rspnt; ) {		/* photoreceptor stimuli */
    trspnt = rspnt;
    rspnt = (recstim *)rspnt->next;
    free (trspnt);
}

#ifdef XSTIM
for (npt=reclist; npt; ) {
    tnpt = npt;
    npt = npt->next;
    free (tnpt);
}
#endif

initmk();				/* reset all lists and counts */
}

/*------------------------------------------------*/

void freelst (conlst *pnt)
                

/* free a conlst (list of connections), part of either
   a node, compartment, or cacomp, but not its connection (conpnt).
*/

{
   conlst *lpnt,*tlpnt;

   for (lpnt=pnt; lpnt; ) {
        tlpnt = lpnt;
        lpnt = lpnt->next;
	free (tlpnt);
    }
}

/*------------------------------------------------*/

void freendp (node *nodp, elem *epnt)
             
/* Reverse of "setnodlst()" in modcode.c.
   Subtracts a "conlst" from a node's element list,
    but maintains the list by patching pointers.
   Frees the list pointer from a node to an element.
   Used when the element is to be freed. 
*/

{
  conlst *lpnt, *olpnt, *tlpnt;
 
  if (!nodp) return;
  olpnt=NULL; 
  for (lpnt=nodp->elemlst; lpnt;)  {
     if (lpnt->conpnt== (conn *)epnt) {
        if (olpnt) olpnt->next = lpnt->next;
        else      nodp->elemlst = lpnt->next;     
        tlpnt = lpnt;
        lpnt = lpnt->next;
        free (tlpnt);
    /*  return;		/* should only be one pointer from node to elem */
     }
     else { 
        olpnt = lpnt;
        lpnt = lpnt->next;
     }
  }
}

/*------------------------------------------------*/

elem *tmpelem(void)

/* Set up an element to hold
   nodes and locations temporarily. */

{
  static elem *epnt, nullelem = {0};

  epnt = &nullelem;
  epnt->node1a = NULLNOD;	/* reset node number */ 
  epnt->node1b = NULLNOD;
  epnt->node1c = NULLNOD;
  epnt->node2a = NULLNOD;
  epnt->node2b = NULLNOD;
  epnt->node2c = NULLNOD;
  epnt->attpnt = NULL;
  epnt->nodp1 = NULL;
  epnt->nodp2 = NULL;
  epnt->next = NULL;
  epnt->attpnt= NULL;
  epnt->lptr  = NULL;
  epnt->saved = 0;
  epnt->modif = 0;

  epnt->ctype = ELEMENT;	/* element is default type */
  elpnt = NULL;			/* reset current element pointer */
  return (epnt);
}

/*------------------------------------------*/

void elcopy (elem *src, elem *dest)

/* Copy a neural element's node information. */

{
  if (!src || !dest) return;

  dest->ctype  = src->ctype;
  dest->node1a = src->node1a;
  dest->node1b = src->node1b;
  dest->node1c = src->node1c;
  dest->node2a = src->node2a;
  dest->node2b = src->node2b;
  dest->node2c = src->node2c;
  dest->attpnt = src->attpnt;
  dest->nodp1 = src->nodp1;
  dest->nodp2 = src->nodp2;
  dest->next = src->next;
  dest->lptr  = src->lptr;
  dest->saved = src->saved;
  dest->modif = src->modif;
  dest->elnum = src->elnum;
}

/*------------------------------------------------*/

elem *makelem(int etype, elem *oepnt)

/* make a new neural element, and link it
   to the elem list. Return a pointer to the 
   new element. */

{
    elem *epnt;
    int esize;

#ifdef DEBUG 
  if (debug & 1 && debugz & 16)  fprintf (stderr,"elem %d\n",cumelem);
#endif

  if (oepnt && oepnt->ctype!=ELEMENT) 	/* If old element exists, */
	return (oepnt);			/*  but is already set up, return it. */ 

  switch (etype) {

        default:
 	case ELEMENT: esize = sizeof(elem);	break;
 	case SPHERE:  esize = sizeof(sphere);	break;
 	case CABLE:   esize = sizeof(cable);	break;
 	case SYNAPSE: esize = sizeof(synapse);	break;
 	case ROD:     esize = sizeof(photrec);	break;
 	case CONE:    esize = sizeof(photrec);	break;
 	case GJ:      esize = sizeof(gapjunc);	break;
 	case RESISTOR:esize = sizeof(resistor);	break;
 	case LOAD:    esize = sizeof(loadelem);	break;
 	case CAP:     esize = sizeof(capac);	break;
 	case BATT:    esize = sizeof(batt);	break;
 	case BUF:     esize = sizeof(vbuf);	break;
 	case CHAN:    esize = sizeof(elem);	break;
  }

#ifndef XSTIMM
  if ((epnt=(elem *)emalloc(esize)) == NULL) {
     fprintf (stderr,"no space left for elem %d\n",cumelem);
     return (NULL);  
  }

#else		/* if we are ignoring neural elements in "stim" */

    {  static char tmpelement[sizeof(synapse)*2] = {0};

       epnt = (elem *)&tmpelement;
    }
#endif

  if (oepnt) {
    elcopy (oepnt, (elem *)epnt);	/* first, copy node info. */
  }

  epnt->next = NULL;			/* next, set pointers */
  if (!elempnt) elempnt = epnt;  	/* save head if first branch */
  if (elemend)
    elemend->next = epnt;
  elemend = epnt;

  epnt->ctype  = etype;
  epnt->lptr  = NULL;
  epnt->saved = 0;
  epnt->modif = 0;
  epnt->elnum = ++cumelem; 			/* increment total */

  switch (etype) {

        default:
 	case ELEMENT: break;

 	case SPHERE:	((sphere *)epnt)->dia    = NULLNOD;
			((sphere *)epnt)->Rm     = NULLNOD;
		 	((sphere *)epnt)->Cm     = NULLNOD;
		 	((sphere *)epnt)->vrev   = NULLNOD;
		 	((sphere *)epnt)->vrest  = NULLNOD;
		 	break;

 	case CABLE:	((cable *)epnt)->dia    = NULLNOD;
		 	((cable *)epnt)->Ri     = NULLNOD;
		 	((cable *)epnt)->Rm     = NULLNOD;
		 	((cable *)epnt)->Cm     = NULLNOD;
		 	((cable *)epnt)->vrev   = NULLNOD;
		 	((cable *)epnt)->vrest  = NULLNOD;
		 	((cable *)epnt)->length = NULLNOD;
		 	((cable *)epnt)->cplam  = NULLNOD;
		 	break;

 	case SYNAPSE:	((synapse *)epnt)->thresh = NULLNOD;
		 	((synapse *)epnt)->timec1 = NULLNOD;
		 	((synapse *)epnt)->nfilt1 = NULLNOD;
		 	((synapse *)epnt)->timec2 = NULLNOD;
		 	((synapse *)epnt)->nfilt2 = NULLNOD;
		 	((synapse *)epnt)->timec3 = NULLNOD;
		 	((synapse *)epnt)->nfilt3 = NULLNOD;
		 	((synapse *)epnt)->tfall3 = NULLNOD;
		 	((synapse *)epnt)->igain  = NULLNOD;
		 	((synapse *)epnt)->maxcond= NULLNOD;
		 	((synapse *)epnt)->kd     = NULLNOD;
		 	((synapse *)epnt)->ntact  = NULLNOD;
		 	((synapse *)epnt)->expon  = NULLNOD;
		 	((synapse *)epnt)->curve  = NULLNOD;
		 break;

	case CONE:
 	case ROD:	((photrec *)epnt)->xpos   = NULLNOD;
		 	((photrec *)epnt)->ypos   = NULLNOD;
		 	((photrec *)epnt)->zpos   = NULLNOD;
		 	((photrec *)epnt)->dia    = NULLNOD;
		 	((photrec *)epnt)->maxcond= NULLNOD;
		 	((photrec *)epnt)->pigm   = NULLNOD; 
		 	((photrec *)epnt)->pathl  = NULLNOD;
		 	((photrec *)epnt)->attf   = NULLNOD;
		 	((photrec *)epnt)->timec1 = NULLNOD;
		 	((photrec *)epnt)->filt   = 0;
		 	((photrec *)epnt)->save   = 0;
		 	((photrec *)epnt)->restore= 0;
		 	((photrec *)epnt)->photnoise=0;
		 	break;

 	case GJ:      	((gapjunc *)epnt)->area   = NULLNOD;
		 	((gapjunc *)epnt)->specres= NULLNOD;
		 	break;

 	case RESISTOR:	((resistor *)epnt)->z = NULLNOD;
		 	break;

 	case LOAD:	 ((loadelem *)epnt)->z = NULLNOD;
		 	((loadelem *)epnt)->vrev  = NULLNOD;
		 	((loadelem *)epnt)->vrest = NULLNOD;
		 	break;

 	case GNDCAP:
 	case CAP:	((capac *)epnt)->c  = NULLNOD;
			((capac *)epnt)->vrest = NULLNOD;
		 	break;

 	case BATT:	((batt *)epnt)->v  = NULLNOD;
		 	break;

 	case BUF:	((vbuf *)epnt)->delay = NULLNOD;
		 	break;

  }	/* switch (etype)  to initialize the elements */

  return (epnt); 
}

/*------------------------------------------------*/

cattrib *makcattr(void)

/* make a new neural element to contain calcium attributes.
   Return a pointer to the new element. */

{
    cattrib *apnt;

#ifdef DEBUG 
  if (debug & 1 && debugz & 16)  fprintf (stderr,"makcattrib %d\n",cumelem);
#endif

  if ((apnt=(cattrib *)emalloc(sizeof(cattrib))) == NULL) {
     fprintf (stderr,"no space left for cattrib %d\n",cumelem);
     return (NULL);  
  }
  apnt->stype   = NULLNOD;
  apnt->vrev    = NULLNOD;
  apnt->thresh  = NULLNOD;
  apnt->taum    = NULLNOD;
  apnt->tauh    = NULLNOD;
  apnt->maxcond = NULLNOD;
  apnt->density = NULLNOD;

  apnt->cao     = NULLNOD;
  apnt->cai     = NULLNOD;
  apnt->kex     = NULLNOD;
  apnt->vmax    = NULLNOD;
  apnt->pkm     = NULLNOD;
  apnt->ekm     = NULLNOD;
  apnt->cabnd   = NULLNOD;
  apnt->pump    = 0;
  apnt->exch    = 0;
  apnt->attpnt  = NULL;
  return (apnt); 
}
/*------------------------------------------------*/

attrib *makattr(void)

/* make a new neural element to contain attributes.
   Return a pointer to the new element. */

{
    attrib *apnt;

#ifdef DEBUG 
  if (debug & 1 && debugz & 16)  fprintf (stderr,"makattrib %d\n",cumelem);
#endif

  if ((apnt=(attrib *)emalloc(sizeof(attrib))) == NULL) {
     fprintf (stderr,"no space left for attrib %d\n",cumelem);
     return (NULL);  
  }
  apnt->ctype   = NULLNOD;
  apnt->stype   = NULLNOD;
  apnt->lptr    = NULL;
  apnt->vrev    = NULLNOD;
  apnt->thresh  = NULLNOD;
  apnt->maxcond = NULLNOD;
  apnt->density = NULLNOD;
  apnt->taum    = NULLNOD;
  apnt->tauh    = NULLNOD;
  apnt->d1 	= NULLNOD;
  apnt->d2 	= NULLNOD;
  apnt->k1 	= NULLNOD;
  apnt->k2 	= NULLNOD;
  apnt->attpnt  = NULL;
  return (apnt); 
}

/*------------------------------------*/

load *makload(loadelem *epnt, comp *comp1)
               
/* make a new resistive load and link it to the load list. */

{
    load *lpnt;
    double rp, r;
 
    if ((lpnt=(load *)emalloc(sizeof(load))) == NULL) {
      fprintf (stderr,"no space left for load %d\n",cumload+1);
      return (NULL);  
    }
    lpnt->next = NULL;
    if (!loadpnt) loadpnt = lpnt;  	/* save head if first load */
    if (loadend)
      loadend->next = lpnt;
    loadend = lpnt;

    loadend->comp1  = comp1; 		 /* compartment we're connected to */
    maklst (&comp1->clst,(conn*)loadend); /* link presynaptic cell conn */
    loadend->ctype  = epnt->ctype;	 /* set load type */
    if (loadend->ctype == ELEMENT)
		 loadend->ctype = LOAD;  /* default load type */
 					 /* use default values if necess */
   
    if ((r=epnt->z)==NULLNOD)   r = 1e6; /* resistance of load */
    if ((rp=epnt->vrev)==NULLNOD) rp = 0; /* reversal potential */
    if (r==0) r = 1.0;
    loadend->conduct = 1.0 / r;
    loadend->vrev = rp;                 /* reversal potential */
    cumload++; 				/* increment total */
    return (loadend); 
}

/*------------------------------------*/

void modload(loadelem *ldpnt)
               
/* modify an old load */

{
    elem *epnt;
    load *lpnt;
    double z;

    for (epnt=elempnt; epnt; epnt=epnt->next) {
      if (epnt->elnum == ldpnt->modif) break;
    }
    if (epnt==NULL) {
       fprintf (stderr,"modload: can't find element %d\n",ldpnt->modif);
       return;  
    }
    lpnt = (load *)epnt->lptr;	   /* get pointer to synapse from elem */
    if (lpnt==NULL) {
       fprintf (stderr,"modload: can't find load for elem %d\n",ldpnt->modif);
       return;  
    }
    if (lpnt->ctype != LOAD) {
       fprintf (stderr,"modload: element %d is not a load\n",ldpnt->modif);
       return;  
    }

    if (ldpnt->z != NULLNOD) {
       z = ldpnt->z;
       if (z==0.0) z = 1e8;
       lpnt->conduct = 1.0/z;  				/* conductance */
    }
    if (ldpnt->vrev!=NULLNOD)  lpnt->vrev = ldpnt->vrev; /* reversal pot */
}

/*------------------------------------*/

synap *maksynap(synapse *epnt, comp *comp1, comp *comp2)
               
/* make a new synapse and link it to the synapse list. */

{

#define HMICSEC .0001

    int sf1,sf2,sf3,nt,i;
    synap *spnt;
    lpfilt *fpnt;
    attrib *apnt;
    double st, se, si, kd, sc, rp, mx;
    double ft1, ft2, tf2, ft3, tf3;
 
    if ((spnt=(synap *)emalloc(sizeof(synap))) == NULL) {
      fprintf (stderr,"no space left for synapse %d\n",cumsynap+1);
      return (NULL);  
    }
    spnt->next = NULL;
    if (!synpnt) synpnt = spnt;  	/* save head if first synap */
    if (synend)
      synend->next = spnt;
    synend = spnt;

    synend->comp1  = comp1; 		/* presynaptic cell */
    synend->comp2  = comp2; 		/* postsynaptic cell */
    maklst (&comp1->clst,(conn*)synend);/* link presynaptic cell conn */
    maklst (&comp2->clst,(conn*)synend);/* link postsynaptic cell conn */
    synend->ctype  = epnt->ctype;	/* set synapse type */
    if (synend->ctype == ELEMENT)
		 synend->ctype = SYNAPSE;  /* default synapse type */
    					/* use default values if necess */
    if ((se=epnt->expon)==NULLNOD) se = dse; 	/* exponent gain */
    if ((sc=epnt->curve)==NULLNOD) sc = dsc; 	/* synap curve: lin or expon */
    if ((si=epnt->igain)==NULLNOD)   si = dsi; 	/* default input gain */
    if ((nt=epnt->ntact)==NULLNOD)   nt = OPEN; /* neurotransmitter action */
    if ((kd=epnt->kd)==NULLNOD)      kd = dskd; /* half-max saturation point */
    if ((mx=epnt->maxcond)==NULLNOD) mx = dmaxsyn; /* max conductance */
    if ((st=epnt->thresh)==NULLNOD)  st = dst;     /* synaptic threshold */
    if ((sf1=(int)(epnt->nfilt1))==NULLNOD)sf1=(int)(dsfa);/* number of filters */
    if ((ft1=epnt->timec1)==NULLNOD) ft1 = dfta; /* tau of filters (msec) */
    if ((sf2=(int)(epnt->nfilt2))==NULLNOD)sf2=(int)(dsfb);/* number of filters */
    if ((ft2=epnt->timec2)==NULLNOD) ft2 = dftb; /* tau of filters (msec) */
    if ((tf2=epnt->tfall2)==NULLNOD) tf2 = 0.0;	 /* falling tau of filt (msec)*/
    if ((sf3=(int)(epnt->nfilt3))==NULLNOD)sf3=0;/* number of filters */
    if ((ft3=epnt->timec3)==NULLNOD) ft3 = 0.0; /* tau of filters (msec) */
    if ((tf3=epnt->tfall3)==NULLNOD) tf3 = 0.0;	 /* falling tau of filt (msec)*/
    if ((rp=epnt->vrev)==NULLNOD) {	/* reversal potential */
       rp = .66 * vna + .33 * vk;
    }
				/* allocate filter space if filters specified */
    if (sf1>0) {
      if ((fpnt=(lpfilt *)emalloc(sizeof(lpfilt))) == NULL) {
        fprintf (stderr,"no space left for filter 1 %d\n",cumsynap+1);
        return (NULL);  
      }
    synend->filt1  = fpnt; 		/* filter 1 */
    }
    else synend->filt1 = NULL;

    if (sf2>0) {
      if ((fpnt=(lpfilt *)emalloc(sizeof(lpfilt))) == NULL) {
        fprintf (stderr,"no space left for filter 2 %d\n",cumsynap+1);
        return (NULL);  
      }
      synend->filt2  = fpnt; 		/* filter 2 */
    }
    else synend->filt2 = NULL; 

    if (sf3>0) {
      if ((fpnt=(lpfilt *)emalloc(sizeof(lpfilt))) == NULL) {
        fprintf (stderr,"no space left for filter 3 %d\n",cumsynap+1);
        return (NULL);  
      }
      synend->filt3  = fpnt; 		/* filter 3 */
    }
    else synend->filt3 = NULL; 

    synend->csites = 0;
    synend->vsites = 0;
    for (apnt=epnt->attpnt; apnt; apnt=apnt->attpnt) {/* set up noise params */
      switch (apnt->ctype) {

 	case CHNOISE:
	if (apnt->density==NULLNOD) 	/* use defaults if not defined */
	    synend->csites = (int)(dscn);	
	else
	    synend->csites = (int)(apnt->density);	

	if (apnt->thresh==NULLNOD)
	    synend->cdur = dscd;	
	else
	    synend->cdur = apnt->thresh;	
	synend->cdur /= HMICSEC;
	break;

 	case VESNOISE:
	if (apnt->density==NULLNOD)     /* use defaults if not defined */	
	    synend->vsites = (int)(dsvn);	
	else
	    synend->vsites = (int)(apnt->density);	

	if (apnt->thresh==NULLNOD)
	    synend->vsize = dvsz;	
	else
	    synend->vsize = apnt->thresh;	
	if (synend->vsize==0) synend->vsize = 1.0;
	break;

        }  /* switch */
      }  /* for (apnt;;) */

    synend->ntact = nt;
    synend->expon = se;
    synend->igain = si;
    synend->curve = (int)(sc);
    if (synend->curve==DYAD) {		/* if this synapse is 2nd in dyad */
      for (epnt=(synapse *)elempnt; epnt; epnt=(synapse *)epnt->next) {
        if (epnt->elnum == se) break;	/* element number of 1st in dyad */
      }
      if (epnt==NULL) {
        fprintf (stderr,"maksyn: can't find element %d\n",(int)se);
        return (NULL);  
      }
      else {
        if (epnt->lptr) { 			  /* if 1st synap exists */ 
          ((synap *)epnt->lptr)->sdyad = synend; /* save ptr to 2nd in 1st */
          synend->sdyad = (synap *)epnt->lptr;	  /* save ptr to 1st in 2nd */
        }
      }
    }
    else synend->sdyad = 0;

    synend->thresh = st;
    synend->kd    = kd;
    synend->maxcond = mx;
    synend->vrev = rp;			/* reversal potential */


    if (sf1) {
      if (sf1 > NUMFILT) sf1 = NUMFILT;
      synend->filt1->nfilt = sf1;		/* number of filters */
      synend->filt1->ftau = calctau(ft1);	/* low-pass filt time const */
      synend->filt1->tfall = 0.0;		/* low-pass fall time const */
      for (i=0; i<NUMFILT; i++)
          synend->filt1->lfilt[i] = 0;	/* init low-pass filt array */
    }

    if (sf2) {

      if (sf2>NUMFILT) sf2 = NUMFILT;
      synend->filt2->nfilt = sf2;		/* number of filters */
      synend->filt2->ftau = calctau(ft2);	/* low-pass filt time const */
      if (tf2) 
	synend->filt2->tfall = 1.0-calctau(tf2);/* low-pass fall time const */
      else
	synend->filt2->tfall = 0.0;
      for (i=0; i<NUMFILT; i++)
          synend->filt2->lfilt[i] = 0;	/* init low-pass filt array */
    }

    if (sf3) {

      if (sf3>NUMFILT) sf3 = NUMFILT;
      synend->filt3->nfilt = sf3;		/* number of filters */
      synend->filt3->ftau = calctau(ft3);	/* low-pass filt time const */
      if (tf3)
        synend->filt3->tfall = 1.0-calctau(tf3);/* low-pass fall time const */
      else
        synend->filt3->tfall = 0.0;
      for (i=0; i<NUMFILT; i++)
          synend->filt3->lfilt[i] = 0;	/* init low-pass filt array */
    }

/* fprintf (stderr,"sf1 %d ft1 %g sf2 %d ft2 %g sf3 %d ft3 %g\n",
				sf1,ft1,sf2,ft2,sf3,tf3); /* */

/* fprintf (stderr,"nt %d se %g si %g sc %g st %g kd %g mx %g rp %g\n",
				nt,se,si,sc,st,kd,mx,rp); /* */

    cumsynap++; 			/* increment total */
    return (synend); 
}

/*------------------------------------*/

void setsynv(synap *spnt, double v)
              
/* Set a synapse's internal activation from 
   its presynaptic voltage.
   If "synapse" is a voltage buffer, then
   set "destination" voltage equal to the
   "source" voltage.
 */

{
   int i;
   double bnd,cond,sa,st,tr;
   dbuf *dpnt;
#ifndef XSTIM
    double getcurv(synap *spnt, double val);
#endif

switch (spnt->ctype) {

case SYNAPSE:
 st = spnt->thresh;			/* synaptic threshold */
 sa = v - st;				/* synaptic activation in mv */
 if (spnt->filt1) 
    for (i=0; i<NUMFILT; i++)
      spnt->filt1->lfilt[i] = sa;	/* init low-pass filt array */

#ifdef XSTIM
    tr = 0;        			/* find resting transmitter rel */
    cond = 0;        			/* resting conductance */
#else
    tr = getcurv(spnt,sa);        	/* find resting transmitter rel */
    bnd = tr / (spnt->kd + tr);  	/* fraction bound */
    switch (spnt->ntact) {		/* find resting conductance */
      case OPEN:
           cond = bnd;
           break;
      case CLOSE:
           cond = (1 - bnd);
           break;
	 }	
#endif

 if (spnt->filt2) { 
    for (i=0; i<NUMFILT; i++)
      if (spnt->vsites) {		/* if ves noise to be used */
        spnt->filt2->lfilt[i] = 0;	/* zero low-pass filt array */
      }
      else {
        spnt->filt2->lfilt[i] = tr;	/* init low-pass filt array */
      }
 }
 if (spnt->filt3) {
    for (i=0; i<NUMFILT; i++)
      if (spnt->vsites) {
        spnt->filt3->lfilt[i] = 0;	/* zero low-pass filt array */
      }
      else {
        spnt->filt3->lfilt[i] = cond;	/* init low-pass filt array */
      }
 }
    break;

  case BUF:

    dpnt = (dbuf *)spnt;
    if (dpnt->delay) {
       short int val;
       double vf;

      vf = v / DELCONV; 
      if (vf > 0) 
         val = (int)(vf + 0.5); 		/* input voltage */
      else
         val = (int)(vf - 0.5); 		/* input voltage */
      if (dpnt->delbuf)
        for (i=0; i<dpnt->delay; i++) 
           dpnt->delbuf[i] = val;
      dpnt->delpnt = dpnt->delbuf;
    }
   else {
     dpnt->comp2->v = v; 
   }
   break;
  }  /* end switch */
}

/*------------------------------------*/

synap *modsynap(synapse *sepnt)
               
/* modify an old synapse */

{
    int i,sfi,sti,tfi;
    elem *epnt;
    synap *spnt;
    attrib *apnt;
    lpfilt *fpnt;
#ifndef XSTIM
    double getcurv(synap *spnt, double val);
#endif

    for (epnt=elempnt; epnt; epnt=epnt->next) {
      if (epnt->elnum == sepnt->modif) break;
    }
    if (epnt==NULL) {
       fprintf (stderr,"modsyn: can't find element %d\n",sepnt->modif);
       return (NULL);  
    }
    spnt = (synap *)epnt->lptr;	/* get pointer to synapse from elem */
    if (spnt==NULL) return NULL; 
    if (spnt->ctype != SYNAPSE) {
       fprintf (stderr,"modsyn: element %d is not a synapse\n",sepnt->modif);
       return (NULL);  
    }

#ifndef XSTIM
   if (sepnt->expon!=NULLNOD)spnt->expon= sepnt->expon; /* exponent gain */
   if (sepnt->curve!=NULLNOD) spnt->curve= sepnt->curve; /* synap curv:lin,exp*/
   if (sepnt->igain!=NULLNOD) spnt->igain = sepnt->igain;/* input gain */
   if (sepnt->ntact!=NULLNOD) spnt->ntact = sepnt->ntact;/* neurotrans action*/
   if (sepnt->kd!=NULLNOD)    spnt->kd = sepnt->kd;      /* half-max sat point*/
   if (sepnt->maxcond!=NULLNOD)spnt->maxcond = sepnt->maxcond; /* max conduct*/
   if (sepnt->thresh!=NULLNOD)spnt->thresh = sepnt->thresh; /* synaptic thresh*/
   if (sepnt->vrev!=NULLNOD)  spnt->vrev = sepnt->vrev;  /* reversal potential*/

    if (sti=(sepnt->timec1!=NULLNOD)) {		/* synaptic time const 1 */
        if (spnt->filt1)
    	  spnt->filt1->ftau = calctau(sti);	/* initialize tau 1 */
    }
    if (sti=(sepnt->timec2!=NULLNOD)) {		/* synaptic time const 2 */
        if (spnt->filt2)
    	  spnt->filt2->ftau = calctau(sti);	/* initialize tau 2 */
    }
    if (sti=(sepnt->timec3!=NULLNOD)) {		/* synaptic time const 3 */
        if (spnt->filt3)
    	  spnt->filt3->ftau = calctau(sti);	/* initialize tau 3 */
    }

    if (tfi=(sepnt->tfall2!=NULLNOD)) {		/* synaptic fall time */
        if (spnt->filt2)
    	  spnt->filt2->tfall = 1.0-calctau(tfi);/* initialize tfall */
    }
    if (tfi=(sepnt->tfall3!=NULLNOD)) {		/* synaptic fall time */
        if (spnt->filt3)
    	  spnt->filt3->tfall = 1.0-calctau(tfi);/* initialize tfall */
    }

	/* If we're changing number of filters,
        /*  we initialize them to the value of filter before them. */
        /* But if number of filters is 0, don't do anything. */

    if (sfi=(sepnt->timec1 !=NULLNOD)) {	/* num of synaptic filters */
	if (fpnt=spnt->filt1) {
          if (!fpnt->nfilt)
              fpnt->lfilt[0] = spnt->comp1->v - spnt->thresh;  
          else 
	    for (i=fpnt->nfilt; i<sfi; i++) 	/* init new filters */
              fpnt->lfilt[i] = fpnt->lfilt[fpnt->nfilt-1];
    	  fpnt->nfilt = sfi; 			/* initialize num of filters */
        }
    }

    if (sfi=(sepnt->nfilt2!=NULLNOD)) {		/* num of synaptic filters */
	if (fpnt=spnt->filt2) {
          if (!fpnt->nfilt)
              fpnt->lfilt[0] = getcurv (spnt,spnt->comp1->v - spnt->thresh);  
	  else    /* fpnt->nfilt > 0 */
	    for (i=fpnt->nfilt; i<sfi; i++) 	/* init new filters */
              fpnt->lfilt[i] = fpnt->lfilt[fpnt->nfilt-1];
    	  fpnt->nfilt = sfi; 			/* initialize num of filters */
        }
    }

    if (sfi=(sepnt->nfilt3!=NULLNOD)) {		/* num of synaptic filters */
	if (fpnt=spnt->filt3) {
	  if (!fpnt->nfilt) {
              float tr,bnd,cond;

              tr = getcurv (spnt,spnt->comp1->v - spnt->thresh);  
              bnd = tr / (spnt->kd + tr);  	/* fraction bound */
    	      switch (spnt->ntact) {		/* find resting conductance */
      		case OPEN:
 			cond = spnt->maxcond * bnd;
			break;
		case CLOSE:
			cond = spnt->maxcond * (1 - bnd);
			break;
	      }	
              fpnt->lfilt[0] = cond; 
	  }
	  else /* fpnt->nfilt>0 */
	    for (i=fpnt->nfilt; i<sfi; i++) 	/* init new filters */
              fpnt->lfilt[i] = fpnt->lfilt[fpnt->nfilt-1];
    	  fpnt->nfilt = sfi; 			/* initialize num of filters */
        }
    }

    for (apnt=sepnt->attpnt; apnt; apnt=apnt->attpnt) {/* set up noise params */
      switch (apnt->ctype) {

 	case CHNOISE:
	if (apnt->density!=NULLNOD)
	    spnt->csites = (int)(apnt->density);	
	if (apnt->thresh!=NULLNOD)
	    spnt->cdur = apnt->thresh / HMICSEC;	
	break;

 	case VESNOISE:
	if (apnt->density!=NULLNOD)
	    spnt->vsites = (int)(apnt->density);	
	if (apnt->thresh!=NULLNOD)
	    spnt->vsize = apnt->thresh;	
	break;

        }  /* switch */
      }  /* for (apnt;;) */

#endif
    return (spnt); 
}

/*------------------------------------*/

double calctau(double tau)
{
   double k,taugran; 

#define TIMERES  (.1)
#define MINTAU  (TIMERES)
#define TAUGRAN (.632120558)  /* (1 - 1/e) */

 if (tau < MINTAU) tau =  MINTAU;
 taugran = TIMERES * pow(TAUGRAN,TIMERES/tau);  /* fix error for small tau*/
 k = taugran / tau;
 if (k > 1.0) k = 1.0;			      /* limit k to reasonable values */
 else if (k<=0.0) k = .0001;
 return k;
}

/*------------------------------------*/
recpar rectypes[NUMREC];

void initrec(void)

/* initialize receptor transduction cascade constants */

{
    recpar *rpnt;
    double timinc;
    double cabuf, cycgbuf, loopgain;
    int pigm;

    timinc = .001;

    rpnt = &rectypes[0];		/* set up rod cascade constants */
    rpnt->ctype = ROD;
    pigm = 0;
    rpnt->pigm = pigm;
    rpnt->pathl = pigmlen[pigm];	/* path length through o.s. */

    cabuf  = .2;
    cycgbuf = .5;
    loopgain = .2;

    rpnt->pdebase = .02;
    rpnt->kdgcyc = .010;

    rpnt->rhod = 0.0;
    rpnt->gpr1 = 0.0;
    rpnt->gpr2 = 0.0;
    rpnt->gpr3 = 0.0;
    rpnt->pde  = rpnt->pdebase;
    rpnt->cycg = 1.0092;
    rpnt->cond = 1.0;
    rpnt->totcond = 20.0;
    rpnt->ca   = 0.32707;
    rpnt->cax  = .15763;

    rpnt->rhodgain = timinc * 2140.;
    rpnt->ggain1   = timinc * 20.0;
    rpnt->ggain2   = timinc * 20.0;
    rpnt->ggain3   = timinc * 20.0;
    rpnt->ggain4   = timinc * 80.0;
    rpnt->gcygain  = timinc * 7.0 / cycgbuf * loopgain;
    rpnt->pdegain  = timinc * 40.0 / cycgbuf;
    rpnt->condgain = timinc * 160.0 * loopgain;
    rpnt->gca      = timinc * 69.0 / cabuf * loopgain;
    rpnt->cxgain   = timinc * 20.0 * loopgain;

    rpnt->dec1 = 1 - (timinc * 180);
    rpnt->dec2 = 1 - (timinc * 180);
    rpnt->dec3 = 1 - (timinc * 180);
    rpnt->dec4 = 1 - (timinc * 180);
    rpnt->pdedec = 1 - (timinc * 5);
    rpnt->dec5 = 1 - (timinc * 30);
    rpnt->capump =    timinc * 50 / cabuf;
    rpnt->dec6 = 1 - (timinc * 10);

					/* voltage activation of dark curr */
					/* assume linear, but offset */
    rpnt->vrev  =  .72;			/* 0.1 activation / -80 mv */

    coneset(1);
    coneset(2);
    coneset(3);
}

/*------------------------------------*/

void coneset(int pigm)
{
    recpar *cpt;
    double timinc;
    double cabuf, cycgbuf, loopgain;

    timinc = .001;

    cpt = &rectypes[pigm];		/* set up cone cascade constants */
    cpt->ctype = CONE;
    cpt->pigm = pigm;
    cpt->pathl = pigmlen[pigm];	/* path length through o.s. */
 
    cabuf  = .1;
    cycgbuf = .02;
    loopgain = .70;

    cpt->pdebase = .01;
    cpt->kdgcyc  = .02;

    cpt->rhod = 0.0;
    cpt->gpr1 = 0.0;
    cpt->gpr2 = 0.0;
    cpt->gpr3 = 0.0;
    cpt->pde  = cpt->pdebase;
    cpt->cycg = 1.4454;
    cpt->cond = 1.0;
    cpt->totcond = 20.0;
    cpt->ca   = 0.48447;
    cpt->cax  = .12463;

    cpt->rhodgain = timinc * 50.0;
    cpt->ggain1   = timinc * 20.0;
    cpt->ggain2   = timinc * 20.0;
    cpt->ggain3   = timinc * 20.0;
    cpt->ggain3   = timinc * 20.0;
    cpt->ggain4   = timinc * 20.0;
    cpt->gcygain  = timinc * .6 / cycgbuf * loopgain;
    cpt->pdegain  = timinc * 40.0 / cycgbuf;
    cpt->condgain = timinc * 80.0 * loopgain;
    cpt->gca      = timinc * 26.2 / cabuf * loopgain;
    cpt->cxgain   = timinc * 8. * loopgain;

    cpt->dec1 = 1 - (timinc * 200);
    cpt->dec2 = 1 - (timinc * 200);
    cpt->dec3 = 1 - (timinc * 200);
    cpt->dec4 = 1 - (timinc * 200);
    cpt->pdedec = 1 - (timinc * 50);
    cpt->dec5 = 1 - (timinc * 100);
    cpt->capump =    timinc * 50 / cabuf;
    cpt->dec6 = 1 - (timinc * 30);

					/* voltage activation of dark curr */
					/* assume linear, but offset */
    cpt->vrev  =  -0.008;		/* gated reversal pot */
 }

/*------------------------------------*/

recep *makr(photrec *epnt, int ctype)

/* make a dummy receptor and link it to the list. */

{
    recep *rpnt;
 
  if ((rpnt=(recep *)emalloc(sizeof(recep))) == NULL) {
     fprintf (stderr,"no space left for receptor %d\n",cumrec+1);
      return (NULL);  
  }
  rpnt->next = NULL;
  if (!recpnt) recpnt = rpnt;	  	/* save head if first recep */
  if (recend)
    recend->next = rpnt;
  recend = rpnt;

  rpnt->ctype = ctype;			/* set channel to dummy type */
  rpnt->xloc  = epnt->xpos;		/* x loc of receptor */
  rpnt->yloc  = epnt->ypos;		/* y loc of receptor */
  rpnt->recnm1 = epnt->node1a;		/* receptor number from node */
  rpnt->recnm2 = epnt->node1b;
  rpnt->recnm3 = epnt->node1c;
  cumrec++;	 			/* increment total receptors */
  return (rpnt); 
}

/*------------------------------------*/

recep *makrecep(photrec *epnt, comp *cpnt)
               
/* make a new photoreceptor and link it to the list. */

{
    int pigm,filt,pnois;
    recep *rpnt;
    recpar *cpt;
    double maxcond, dia, pathl, attf, timec;
 
  if ((rpnt=(recep *)emalloc(sizeof(recep))) == NULL) {
     fprintf (stderr,"no space left for receptor %d\n",cumrec+1);
      return (NULL);  
  }
  rpnt->next = NULL;
  if (!recpnt) recpnt = rpnt;  	/* save head if first recep */
  if (recend)
    recend->next = rpnt;
  recend = rpnt;

  rpnt->comp1 = cpnt; 			/* no presynaptic cell */
  maklst(&cpnt->clst,(conn*)rpnt);	/* compartment assoc with recep */
  rpnt->ctype = epnt->ctype;		/* set channel type: ROD or CONE */
  rpnt->xloc  = epnt->xpos;		/* x loc of receptor */
  rpnt->yloc  = epnt->ypos;		/* y loc of receptor */

  if (epnt->ctype == ROD) {

    if ((maxcond=epnt->maxcond)==NULLNOD) maxcond=dmaxrod;/* max conductance*/
    if ((dia=epnt->dia)==NULLNOD) dia = 1.674; /* 2.2 um2 area */
    epnt->dia = dia;
    rpnt->area = dia * dia * MPI / 4;	/* photon collecting area of rod */
    rpnt->maxcond = maxcond;		/* maximum conductance of channel */
    rpnt->consts  = &rectypes[0];	/* rod constants */
    pigm = 0;				/* rod pigment type */
    if ((attf=epnt->attf)==NULLNOD) attf = .9;   /* misc attenuation factor */
   }

 else {				/* cone */

    if ((maxcond=epnt->maxcond)==NULLNOD) maxcond=dmaxcon; /* max conductance */
    if ((dia=epnt->dia)==NULLNOD) dia = 1.674; /* 2.2 um2 area */
    epnt->dia = dia;
    rpnt->area = dia * dia * MPI / 4;	/* photon collecting area of cone */
    rpnt->maxcond = maxcond;		/* maximum conductance of channel */
    if ((pigm=(int)(epnt->pigm))==NULLNOD) pigm = 1; /* cone default pigment */
    rpnt->consts  = &rectypes[pigm];	/* cone constants */
    if ((attf=epnt->attf)==NULLNOD) attf = .9;   /* misc attenuation factor */
   }

  cpt = rpnt->consts;			/* pointer to receptor constants */
  if ((pathl=epnt->pathl)==NULLNOD) pathl=cpt->pathl; /*default path length*/
  rpnt->pathl = pathl;
  if ((filt=epnt->filt)==NULLNOD) filt = 0; 	/* filter over pigment */
  rpnt->filt = filt;
  if ((timec=epnt->timec1)==NULLNOD) timec = 1.0; /* time speed factor */
  if (timec > 1.0) timec = 1.0;
  if (timec <= 0.0) timec = 0.001;
  rpnt->timec = 1 / timec;
  if ((pnois=epnt->photnoise)==NULLNOD) pnois = 0;	/* photon noise */
  rpnt->pnois = pnois;
  rpnt->attf = attf;
  restorec(rpnt);			/* set receptor constants */
  rpnt->recnm1 = epnt->node1a;		/* receptor number from node */
  rpnt->recnm2 = epnt->node1b;
  rpnt->recnm3 = epnt->node1c;
  cumrec++;	 			/* increment total receptors */
  return (rpnt); 
}

/*------------------------------------*/

void saverec(recep *rpnt)
                

/* Save a photoreceptor's kinetic states for a later restore */

{
    recpar *rp,*cpt;
    char *emalloc();

  cpt = rpnt->consts;			/* pointer to receptor constants */
  if (cpt== &rectypes[cpt->pigm]) { /* if orig estimates, make space */

    if ((rp=(recpar *)emalloc(sizeof(recpar))) == NULL) {
       fprintf (stderr,"no space left for receptor save\n");
       return;  
    }
    *rp = *cpt;			/* copy the old constants */
  }
  else rp = cpt;

  rp->rhod = rpnt->rhod;                /* set the new values */
  rp->gpr1 = rpnt->gpr1;
  rp->gpr2 = rpnt->gpr2;
  rp->gpr3 = rpnt->gpr3;
  rp->pde  = rpnt->pde;
  rp->cycg = rpnt->cycg;
  rp->cond = rpnt->cond;
  rp->ca   = rpnt->ca;
  rp->cax  = rpnt->cax;
  rpnt->consts = rp;			/* pointer to receptor constants */
}

/*------------------------------------*/

void restorec(recep *rpnt)
                 

/* restore photoreceptor kinetic states */

{
    recpar *cpt;

  cpt = rpnt->consts;			/* pointer to receptor constants */
  rpnt->rhod = cpt->rhod;               /* initialize receptor constants */
  rpnt->gpr1 = cpt->gpr1;
  rpnt->gpr2 = cpt->gpr2;
  rpnt->gpr3 = cpt->gpr3;
  rpnt->pde  = cpt->pde;
  rpnt->cycg = cpt->cycg;
  rpnt->cond = cpt->cond;
  rpnt->ca   = cpt->ca;
  rpnt->cax  = cpt->cax;
}

/*------------------------------------*/

recep *modrecep(photrec *pepnt)
               
/* modify a photoreceptor */

{
    recep *rpnt;
    double maxcond, dia, attf, pathl;
    int filt, pnois;
    elem *epnt;
 
    for (epnt=elempnt; epnt; epnt=epnt->next) {
      if (epnt->elnum == pepnt->modif) break;
    }
    if (epnt==NULL) {
       fprintf (stderr,"modrec: can't find element %d\n",pepnt->modif);
       return (NULL);  
    }
    rpnt = (recep *)epnt->lptr;	/* get pointer to recept from elem */
    if (rpnt==NULL) return NULL; 
    if (rpnt->ctype != ROD && rpnt->ctype != CONE) {
     fprintf (stderr,"modrec: element %d is not a photoreceptor\n",pepnt->modif);
       return (NULL);  
    }
					/* don't want to change type or loc: */

/*  rpnt->ctype = pepnt->ctype;		/* set channel type: ROD or CONE */
/*  rpnt->xloc  = pepnt->xpos;		/* x loc of receptor */
/*  rpnt->yloc  = pepnt->ypos;		/* y loc of receptor */

  if ((maxcond=pepnt->maxcond)!=NULLNOD)	/* maximum conductance */
     rpnt->maxcond = maxcond;	/* maximum conductance of channel */
  if ((dia=pepnt->dia)!=NULLNOD) 	
     rpnt->area = dia * dia * MPI / 4; /* photon collecting area of rod */
  if ((attf=pepnt->attf)!=NULLNOD)   	/* misc attenuation factor */
     rpnt->attf = attf;
  if ((pathl=pepnt->pathl)!=NULLNOD) 	/* path length */
      rpnt->pathl = pathl;
  if ((filt=pepnt->filt)!=NULLNOD)	/* filter over pigment */
      rpnt->filt = filt;
  if ((pnois=pepnt->photnoise)!=NULLNOD)/* photon noise */
      rpnt->pnois = pnois;

  if (pepnt->save) {			/* save equilibrium kinetic states */
     saverec(rpnt);
  }
  else 
    if (pepnt->restore) {		/* restore kinetic states */
     restorec(rpnt);
  }

  return (rpnt);
}

/*------------------------------------*/

comp *makcomp(elem *epnt, double rm, double cap, double vrest, double vrev)

/* make a new compartment and link it to the compartment list. */

{
    comp *cpnt;
    int ctype;
 
#ifdef DEBUG
 if (debug & 1 && debugz & 16)
	 fprintf (stderr,"comp %d in branch %d\n",cumcomp+1,epnt->elnum);
#endif

  if ((cpnt=(comp *)emalloc(sizeof(comp))) == NULL) {
     fprintf (stderr,"no space left for comp %d in branch %d\n",
				cumcomp+1,epnt->elnum);
     return (NULL);  
  }
  cpnt->next = NULL;			/* make new comp be end of list */
  if (!compnt) compnt = cpnt;  		/* save head if first synap */
  if (compend)
    compend->next = cpnt;
  cpnt->last = compend;			/* pointer to last compartment */
  compend = cpnt;
  cpnt->nodlst = NULL;			/* pointer to list of nodes for comp */
  cpnt->capnt = NULL;			/* pointer to possible ca comp */

  ctype = epnt->ctype;
  cpnt->ctype 	= ctype;
  cpnt->rm 	= rm;
  cpnt->cap 	= cap;
  cpnt->num 	= cumcomp;
  cpnt->miscfl 	= 0;		/* clear misc flags (IEXT,VEXT,VBAT,CA, etc. */
  cpnt->extv 	= 0.0;
  cpnt->exti 	= 0.0;
  switch (ctype) {		/* neural elements that control vrest, vrev */
   				/*   if they are alone in compartment */
   case CABLE:
   case SPHERE:
   case LOAD: 
   case HH: 
   case NA: 
   case CA: 
   case K: 
	if (vrest == NULLNOD) cpnt->v = vcl;
	else	            cpnt->v = vrest;
	cpnt->vest = cpnt->v;
	if (vrev  == NULLNOD) cpnt->vrev = vcl;
	else		    cpnt->vrev = vrev;
	break;

   default:			/* other elements can make comps, but don't */
	break;			/*    control vrest, vrev. */
  }

  cpnt->clst 	= NULL;
  cpnt->capnt 	= NULL;
  cumcomp++; 			/* increment total */

  return (cpnt); 
}

/*------------------------------------*/

conn *makconn(comp *comp1, comp *comp2, double s, int type)

/* make a new connection betw compartments and link
    it to both compartments' conn lists. */
/* connect comp1 to comp2 with conductance s; */

{
    conn *cpnt;

#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"makconn: comp %d comp %d\n",comp1->num, comp2->num);
#endif

    if ((cpnt=(conn *)emalloc(sizeof(conn))) == NULL) {
      fprintf (stderr,"no space left for conn %d\n",cumconn+1);
      return (NULL);  
    }
					/* add gj or op amp to synapse list */
					/*  so it can be made time-varying */ 
/*    spnt->next = NULL;
    if (!synpnt) synpnt = cpnt;  	/* save head if first synap */
/*    if (synend)
      synend->next = cpnt;
    synend = cpnt;
*/
    cpnt->next = NULL;			/* zero list pointer */
    if (!connpnt) connpnt = cpnt;	/* increment connection list */
    else if (connend) connend->next = cpnt;
    cpnt->last = connend;		/* pointer to last connection */
    connend = cpnt;

    maklst(&comp1->clst, cpnt);
    maklst(&comp2->clst, cpnt);
    cpnt->comp1 = comp1; 
    cpnt->comp2 = comp2; 
    cpnt->conduct = s;			/* conductance of connection */
    cpnt->ctype = type;
    cumconn++; 				/* increment total */
    return (cpnt); 
}

/*------------------------------------*/

conn *modconn(elem *cepnt, double s)
               
/* modify a connection betw compartments */

{
    conn *cpnt;
    elem *epnt;

#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"modconn: elem %d\n",cepnt->elnum);
#endif

  for (epnt=elempnt; epnt; epnt=epnt->next) {
    if (epnt->elnum == cepnt->modif) break;
  }
  if (epnt==NULL) {
     fprintf (stderr,"modconn: can't find element %d\n",cepnt->modif);
     return (NULL);  
  }
  cpnt = (conn *)epnt->lptr;	/* get pointer to recept from elem */
  if (cpnt==NULL) return NULL; 
  /* if (cpnt->ctype != GJ && cpnt->ctype != RESISTOR) {
     fprintf (stderr,"modconn: element %d is not a gj %d\n",cepnt->modif);
     return (NULL);  
  } */

    cpnt->conduct = s;			/* new conductance of connection */
    return (cpnt); 
}

/*------------------------------------*/

dbuf *makdelbuf(comp *comp1, comp *comp2, double delay, int type)
                       
/* make a delayed buffer connection betw compartments and link
    it to both compartments' conn lists. */
/* connect comp1 to comp2 with conductance s; */

{
    dbuf *cpnt;
    int i,idelay,v;
    short int *dpnt;
    double vf;

#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"makconn: comp %d comp %d\n",comp1->num, comp2->num);
#endif

    if ((cpnt=(dbuf *)emalloc(sizeof(dbuf))) == NULL) {
      fprintf (stderr,"no space left for conn %d\n",cumconn+1);
      return (NULL);  
    }
#define MAXDELAY 10000
					/* delay in msec, */
   idelay = (int)(delay * 10);		/*  time steps of 100 usec */
   if (idelay < 0) {
      idelay = 0;
      fprintf (stderr,"Error: negative delay in buffer, setting to zero.\n");
   }   
   else if (idelay > 10000) {
      fprintf (stderr,"Warning: possible error; excessive delay %d in buffer\n", 
				idelay);
      if (idelay > MAXDELAY) idelay = MAXDELAY;
   }

   if (idelay)  {
     if ((dpnt=(short int *)emalloc(idelay * sizeof(short int))) == NULL) {
       fprintf (stderr,"no space left for delay buffer %d\n",cumconn+1);
       return (NULL);  
     }
   }
   else dpnt = NULL;
					/* add delay buffer to synapse list */
					/*  so it can be made time-varying */ 
    cpnt->next = NULL;
    if (!synpnt) synpnt = (synap *)cpnt; /* save head if first synap */
    if (synend)
      synend->next = (synap *)cpnt;
    synend = (synap *)cpnt;

    maklst(&comp1->clst, (conn*)cpnt);
    maklst(&comp2->clst, (conn*)cpnt);
    cpnt->comp1 = comp1; 
    cpnt->comp2 = comp2; 
    cpnt->conduct = 0;			/* conductance of connection */
    cpnt->delay = idelay;		/* conductance of connection */
    cpnt->ctype = type;
    cpnt->delbuf = dpnt;		/* pointer to circular buffer */
    cpnt->delpnt = dpnt;		/* input/output pointer */

    vf = comp1->v / DELCONV;		/* convert voltage */
    if (vf > 0) 
         v = (int)(vf+0.5); 		/* input voltage */
    else
         v = (int)(vf-0.5); 		/* input voltage */
    if (v>MAXSHORT) v = MAXSHORT;	/* no overflow of short int */
    else if (v<MINSHORT) v = MINSHORT;
    if (dpnt)
      for (i=0; i<idelay; i++) 
        *dpnt++ = v;			/* set buffer to constant voltage */
    cumconn++; 				/* increment total */
    return (cpnt); 
}

/*------------------------------------*/

chan *makchan(attrib *apnt, comp *comp1, int type)
                 
/* make a new channel at a compartment and link
   it to the compartment's conn list. */

{
    int i,ns,chansiz,stype;
    sschan *chpnt;
    chan *addchan(comp *pnt, chan *chpnt);
    attrib *npnt;
    double voff,vrev,maxcond,taum,tauh,taun;
    static int maktabfl = 0;
    static stconc *conc;
    double alph,bet,k1,k2,d1,d2,ca;

#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"makchan: type %d %d\n",apnt->ctype,apnt->stype);
#endif
    			/* first, figure out how much space to allocate */

  if (apnt->stype == NULLNOD) stype = 0;	/* channel sub-type */
  else			      stype = apnt->stype;

  if (stype < 0) stype = 0;
  if (stype > NCHANTYP) stype = NCHANTYP;

  switch (type) {
    case NA:
       switch (stype) {
	case 0:					/* Hodgkin-Huxley */
  	 chansiz = sizeof(hhchan);
	 break;

	case 1:					/* sequential state */
         ns = (natypes[stype].numstate - MINSTATE);    /* state concs */
         chansiz = sizeof(sschan) + ns*sizeof(stconc);
	 break;

	default:
	if (natypes[stype].numstate==0)
  	   chansiz = sizeof(hhchan);
	else {
          ns = (natypes[stype].numstate - MINSTATE);    /* state concs */
          chansiz = sizeof(sschan) + ns*sizeof(stconc);
	}
	break;

       } /* switch (stype) */
       break;	/* case NA */
 
    case K:
       switch (stype) {

	case 0:
	 chansiz = sizeof(hhchan);
         break;

	case 1:
         ns = (ktypes[stype].numstate - MINSTATE);     /* state concs */
         chansiz = sizeof(sschan) + ns*sizeof(stconc);
         break;

	case 2:					/* calcium-sens potassium */
         chansiz = sizeof(kcachan);
         break;

	case 3:					/* type A potassium */
	 chansiz = sizeof(hhchan);
         break;

	default:
         if (ktypes[stype].numstate==0)    		/* HH */
	    chansiz = sizeof(hhchan);
	 else {
           ns = (ktypes[stype].numstate - MINSTATE);    /* sequential state */
           chansiz = sizeof(sschan) + ns*sizeof(stconc);
	 }
         break;

       }  /* switch (stype) */
       break;

    }  /* switch (type) */

    /* Note that for a sequential-state channel with more states
       than MINSTATE (4) the allocation below can be larger than
       the size of the state's struct definition.  This allows a
       variable number of states to be handled.  The security of 
       this scheme depends on the fact that "conc" inside the 
       "sschan" structure is the last field and can be expanded 
       without running into other fields or structures.
    */

    if ((chpnt=(sschan *)emalloc(chansiz)) == NULL) {
      fprintf (stderr,"no space left for chan %d\n",cumchan+1);
      return (NULL);  
    } 

    if (!maktabfl) {
	maktabfl = 1;
	maktables(timinc);
    }
    chpnt->ctype = type;
    chpnt->stype = stype;
    switch (type) {

    case NA:
       if ((maxcond=apnt->maxcond)==NULLNOD) maxcond = dmaxna;
       if (apnt->vrev == NULLNOD) vrev = vna;
       else 			  vrev = apnt->vrev;
       if (apnt->taum == NULLNOD) taum = dnataum;
       else 		  	  taum = apnt->taum;
       if (apnt->tauh == NULLNOD) tauh = dnatauh;
       else 		  	  tauh = apnt->tauh;
       if (apnt->thresh == NULLNOD) voff = dnathr;
       else 		  	    voff = apnt->thresh;
       chpnt->vthr = voff - NATHR; 
       chpnt->consts  = &natypes[stype];	/* NA constants */
       narate(comp1->v,stype);
       switch (stype) {
        case 0:			/* set up equilibrium rate constants */
           ((hhchan *)chpnt)->m    = alpham / (alpham + betam);
           ((hhchan *)chpnt)->h    = alphah / (alphah + betah);
    	   ((hhchan *)chpnt)->maxcond  = maxcond;
    	   ((hhchan *)chpnt)->vrev = vrev;
       	   ((hhchan *)chpnt)->taum = taum;
       	   ((hhchan *)chpnt)->tauh = tauh;
	  break;

	case 1:
          chpnt->comp1 = comp1;			/* set loc of voltage sens. */
          chpnt->comp2 = comp1;
          chpnt->numstate = natypes[stype].numstate;
          if (chpnt->numstate <= 0) chpnt->numstate=1;
	  for (i=0; i<chpnt->numstate; i++) {	/* set initial concs */
	      chpnt->conc[i].cval = 0.0;
	      chpnt->conc[i].cest = 0.0;
	  }
	  chpnt->conc[0].cest = 1;
    	  chpnt->maxcond  = maxcond;
    	  chpnt->vrev = vrev;
          chpnt->taum = taum;
          chpnt->tauh = tauh;
	  dochani(chpnt,1e-8);
	  break;

	default:			   /* any new channel type */
         if (natypes[stype].numstate==0) {    		/* HH */
           ((hhchan *)chpnt)->m    = alpham / (alpham + betam);
           ((hhchan *)chpnt)->h    = alphah / (alphah + betah);
    	   ((hhchan *)chpnt)->maxcond  = maxcond;
    	   ((hhchan *)chpnt)->vrev = vrev;
       	   ((hhchan *)chpnt)->taum = taum;
       	   ((hhchan *)chpnt)->tauh = tauh;
	 }
	 else {					/* sequential state */
          chpnt->comp1 = comp1;			/* set loc of voltage sens. */
          chpnt->comp2 = comp1;
          chpnt->numstate = natypes[stype].numstate;
          if (chpnt->numstate <= 0) chpnt->numstate=1;
	  for (i=0; i<chpnt->numstate; i++) {	/* set initial concs */
	      chpnt->conc[i].cval = 0.0;
	      chpnt->conc[i].cest = 0.0;
	  }
	  chpnt->conc[0].cest = 1;
    	  chpnt->maxcond  = maxcond;
    	  chpnt->vrev = vrev;
          chpnt->taum = taum;
          chpnt->tauh = tauh;
	  dochani(chpnt,1e-8);
	 }
         break;
       }
       break;

    case K:
       if ((maxcond=apnt->maxcond)==NULLNOD) maxcond = dmaxk; 
       if (apnt->vrev == NULLNOD) vrev = vk;
       else 			  vrev = apnt->vrev;
       if (apnt->taum == NULLNOD) taun = dktau;
       else 		          taun = apnt->taum;
       if (apnt->thresh == NULLNOD) voff = dkthr;
       else 		  	    voff = apnt->thresh;
       chpnt->vthr = voff - KTHR; 
       chpnt->consts  = &ktypes[stype];		/* K constants */
       krate(comp1->v,stype);
       switch (stype) {
        case 0:				/* set up equilibrium rate constants */
           ((hhchan *)chpnt)->m = alphan / (alphan + betan);
           ((hhchan *)chpnt)->h = 0;
    	   ((hhchan *)chpnt)->maxcond  = maxcond;
    	   ((hhchan *)chpnt)->vrev = vrev;
       	   ((hhchan *)chpnt)->taum = taun;
	  break;

	case 1:
          chpnt->comp1 = comp1;			/* set loc of voltage sens. */
	  chpnt->comp2 = comp1;
          chpnt->numstate = ktypes[stype].numstate;
          if (chpnt->numstate <= 0) chpnt->numstate=1;
	  for (i=0; i<chpnt->numstate; i++) {	/* set initial concs */
	      chpnt->conc[i].cval = 0.0;
	      chpnt->conc[i].cest = 0.0;
	  }
	  chpnt->conc[0].cest = 1.0;
    	  chpnt->maxcond  = maxcond;
    	  chpnt->vrev = vrev;
          chpnt->taum = taun;
	  dochani(chpnt,1e-8);
	  break;

	case 2:				/* Ca sensitive K channel */
          if ((taun=apnt->taum)==NULLNOD) taun = dkcatau;
          if ((d1=apnt->d1)==NULLNOD) d1 = dd1; 
          if ((d2=apnt->d2)==NULLNOD) d2 = dd2; 
          if ((k1=apnt->k1)==NULLNOD) k1 = dk1; 
          if ((k2=apnt->k2)==NULLNOD) k2 = dk2; 

	  if (comp1->capnt) {				/* found calcium comp*/
	     ca = comp1->capnt->cais[1];
	     ((kcachan *)chpnt)->initfl=1;		/* initialized OK */
	  }
	  else {		/* calcium not set up yet, must do later */
	     ca = dcai;
	     ((kcachan *)chpnt)->initfl=0;	/* not properly initialized */
	  }
	  alph = akcacalc((comp1->v-chpnt->vthr),ca,taun,d1,k1);
	  bet  = bkcacalc((comp1->v-chpnt->vthr),ca,taun,d2,k2);
	  ((kcachan *)chpnt)->m = alph / (alph + bet);
          ((kcachan *)chpnt)->h = 0;
	  ((kcachan *)chpnt)->maxcond  = maxcond;
	  ((kcachan *)chpnt)->vrev = vrev;
	  ((kcachan *)chpnt)->taum = taun;
	  ((kcachan *)chpnt)->d1 = d1;
	  ((kcachan *)chpnt)->d2 = d2;
	  ((kcachan *)chpnt)->k1 = k1;
 	  ((kcachan *)chpnt)->k2 = k2;
	  break;

        case 3:				/* type A deactivating potassium */
           ((hhchan *)chpnt)->m = alphan / (alphan + betan);
           ((hhchan *)chpnt)->h = alphad / (alphad + betad);
    	   ((hhchan *)chpnt)->maxcond  = maxcond;
    	   ((hhchan *)chpnt)->vrev = vrev;
       	   ((hhchan *)chpnt)->taum = taun;
	  break;

	default:
         if (ktypes[stype].numstate==0) {    		/* HH */
           ((hhchan *)chpnt)->m = alphan / (alphan + betan);
           ((hhchan *)chpnt)->h = 0;
    	   ((hhchan *)chpnt)->maxcond  = maxcond;
    	   ((hhchan *)chpnt)->vrev = vrev;
       	   ((hhchan *)chpnt)->taum = taun;
	 }
	 else {					/* sequential-state */
          chpnt->comp1 = comp1;			/* set loc of voltage sens. */
	  chpnt->comp2 = comp1;
          chpnt->numstate = ktypes[stype].numstate;
          if (chpnt->numstate <= 0) chpnt->numstate=1;
	  for (i=0; i<chpnt->numstate; i++) {	/* set initial concs */
	      chpnt->conc[i].cval = 0.0;
	      chpnt->conc[i].cest = 0.0;
	  }
	  chpnt->conc[0].cest = 1.0;
    	  chpnt->maxcond  = maxcond;
    	  chpnt->vrev = vrev;
          chpnt->taum = taun;
	  dochani(chpnt,1e-8);
	 }
	 break;

       } /* switch (stype) */
       break;
    }
    chpnt->csites = 0;
    for (npnt=apnt->attpnt; npnt; npnt=npnt->attpnt) /* set up noise params */
      switch (npnt->ctype) {

 	case CHNOISE:
        chpnt->cq = 1.0;
	if (apnt->density==NULLNOD) 	/* use defaults if not defined */ 
	    chpnt->csites = (int)(dscn);	
	else
	    chpnt->csites = (int)(apnt->density);	
	break;
    }
    cumchan++;
					/* add chan to comp's connections */
    chpnt = (sschan*)addchan(comp1, (chan *)chpnt);
					/*  but possibly erase new channel   */
					/*  and return pointer to old one.   */
#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"makchan end\n");
#endif
    return ((chan*)chpnt); 
}

/*------------------------------------*/

chan *addchan(comp *pnt, chan *chpnt)
               
/* Add a channel to a compartment's connections. */
/* If a channel of same type already exists,     */
/*   don't make another one, just add to the existing */
/*   channel conductance. */

{
   conlst *lpnt;
   chan *cpnt;
   int found;

  if (!pnt || !chpnt) return NULL;
  for (found=0,lpnt=pnt->clst; lpnt && !found; lpnt=lpnt->next) {

     if (!(cpnt=(chan*)lpnt->conpnt)) {
	fprintf(stderr,"addchan: missing connection...\n");
 	break;
     }

     switch (chpnt->ctype) {

     case NA:
     case K:
       if (cpnt->ctype==chpnt->ctype)
	 switch (chpnt->stype) {

        case 0:
        case 1:
        case 2:
	default:
     	    if (cpnt->stype==chpnt->stype) {
	      found = 1;
	    } 
          break;
       }  /* switch (chpnt->stype) */
     default:
            break;
     }  /* switch (chpnt->ctype) */
  }   /* for (found=0,lpnt;;) */

  if (found) {
       cpnt->maxcond += chpnt->maxcond;  /* add the conductance */
       free (chpnt);	/* no need for new (or moved) channel conn */
       cumchan--;	/* decrement total chans if new chan */
       return cpnt;	/* return pointer to existing channel */
  } 
  else {		/* use new channel, add it to list */	
     maklst(&pnt->clst, (conn*)chpnt);	/* add to comp's connections */
     chpnt->comp1 = pnt; 
     return chpnt;		/* return pointer to new channel */
  }
}

/*------------------------------------*/

chan *makca(elem *epnt, cattrib *apnt, comp *comp1, double area)
    	           
/* Make a new calcium channel, and if it doesn't have one,
   make a new calcium compartment, too.  
   Area calibrated in cm2.
*/

{
    chan *chpnt,*addchan(comp *pnt, chan *chpnt);
    cacomp *capnt;
    int i,stype,chansiz,ns;
    static int maktabfl;
    double cao, cai, cabnd, dia, r, dr, x, vrev, vthr;
    double vmax, pkm, ekm, kex;
    double tau,thresh,maxcond,ovol,ivol,core,dca,len;

#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"makca: comp %d\n",comp1->num);
#endif

    			/* first, figure out how much space to allocate */

  if (apnt->stype == NULLNOD) stype = 0;	/* channel sub-type */
  else			      stype = apnt->stype;

  if (stype == 0) {  /* simple H-H */
	chansiz = sizeof(hhchan);
  } 
  else {				/* channel defined by sequen states */
    if (stype < 0) stype = 0;
    if (stype > NCHANTYP) stype = NCHANTYP;
    switch (stype) {
    case CA:
       ns = (catypes[stype].numstate - MINSTATE);    /* state concs */
       chansiz = sizeof(chan) + ns*sizeof(stconc);
       break;
    }  /* switch (type) */
  }  /* else */

  if ((chpnt=(chan *)emalloc(chansiz)) == NULL) {
      fprintf (stderr,"no space left for chan %d\n",cumchan+1);
      return (NULL);  
  } 
  chpnt->next = NULL;
  chpnt->ctype = apnt->ctype;
  chpnt->stype = stype;
  chpnt->comp1 = comp1;

			/* beginning of ca comp stuff */

  if (!(capnt=comp1->capnt)) {			/* make new ca compartment */
    if ((capnt=(cacomp *)emalloc(sizeof(cacomp))) == NULL) {
      fprintf (stderr,"no space left for calcium comp %d\n",cumcacomp+1);
      return (NULL);  
    }
    cumcacomp++;
    capnt->next = NULL;
    capnt->comp1 = comp1;		/* set ca comp pntr to voltage comp */
    comp1->capnt = capnt;		/* set voltage comp pntr to ca comp */

    cao=apnt->cao;
    cai=apnt->cai;
    vrev=apnt->vrev;
						/* If calcium not determined */
    if (cao==NULLNOD || cai==NULLNOD) {		/*  must hunt for info */
      if (vrev!=NULLNOD) {			/* Find ca conc from vrev */
        if (cao==NULLNOD) {
          cao = exp(vrev * F2R / ktemp) * cai; 
        }
        else
        if (cai==NULLNOD) {
          cai = cao / exp(vrev * F2R / ktemp); 
        }
      }
      else {  /* if (vrev==NULLNOD) */	      /* vrev unknown, use defaults */
        if (cao==NULLNOD) {
          cao = dcao;
        }
        if (cai==NULLNOD) {
          cai = dcai;
        }
      }
    } 
    if (cai==0.0) cai = 1e-9;		/* make cai nonzero */

    /* Try to get best est. for radius and area. */

					/* diameter in microns */
    if   ((epnt->ctype==SPHERE) && ((dia=((sphere *)epnt)->dia)!=NULLNOD)) {
         r = dia * 0.5 * 1e-5;		/* radius in dm (decimeters) */
         area = (4.0 * MPI * r*r);	/* shell surface (dm2) */
    }
    else if ((epnt->ctype==CABLE)  && ((dia=((cable *)epnt)->dia) !=NULLNOD)) {
         r = dia * 0.5 * 1e-5;		/* radius in dm (decimeters) */
	 len = 2 * r;			/* need a better est of comp length */
         area = (2.0 * MPI * r*len);	/* shell surface (dm2) */
    }
    else {				/* find radius from area (capacitance) */
       if (area) {			/* in this case area is cm2 */
          area *= 1e-2;
          r = sqrt (area / (4.0*MPI)); /* radius in dm (decimeters) */
       }
       else {
          r = CACOMPRAD * 1e-5;		/* (calib in dm) make it 5 um radius */
          area = (4.0 * MPI * r*r);	/* shell surface (dm2) */
       }
    }
    dr = CASHD*0.1;			/* thickness of shell in dm */
    dca = DCa * 1e-2;			/* Diffusion const for Ca in dm2/sec*/

    switch (epnt->ctype) {

    case CA:
    case SPHERE:
    default:			 /* shell factors for diffusion eqn. */

	capnt->casf0 = dr/(dca*timinc*4*MPI*r*r);	/* 1st shell */
	capnt->casfn = dca * timinc/(dr*dr); 		/* norm shells */
	core = r - (NUMCASH-1) * dr;
	capnt->casfc = dca * timinc *3.0/(dr*core); 	/* D*t*SA/(dr*vol) */
							/*  for core shell */
	break;

    case CABLE:			/* must fix this for cylindrical shells */

	capnt->casf0 = dr/(dca*timinc*4*MPI*r*r);	/* 1st shell */
	capnt->casfn = dca * timinc/(dr*dr);	 	/* norm shells */
	core = r - (NUMCASH-1) * dr;
	capnt->casfc = dca * timinc *3.0/(dr*core); 	/* D*t*SA/(dr*vol) */
							/*  for core shell */
	break;
    }


    if ((cabnd=apnt->cabnd)==NULLNOD) cabnd=dcabnd; /* ratio of bnd to free */
    capnt->cabnd = 1.0 / (cabnd+1);		/* multiplier is one more */
    capnt->cao  = cao;
    for (i=0; i<NUMCASH; i++) {
       capnt->cais[i]  = cai;
    }
    capnt->cai  = cai;				/* save orig value */
    if (apnt->pump) {
      if ((pkm=apnt->pkm)==NULLNOD) pkm = dcapkm;    /* Km for ca pump */
      if ((vmax=apnt->vmax)==NULLNOD) vmax = dcavmax; /* Vmax for ca pump */
      capnt->pkm   = pkm;
      capnt->vmax = vmax * area;
    }
    else capnt->vmax = 0;

    if (apnt->exch) {
      if ((ekm=apnt->ekm)==NULLNOD) ekm = dcaekm; /* Km for ca exchngr */
      if ((kex=apnt->kex)==NULLNOD) kex = dcakex; /* Exch rate for ca exch */
      capnt->ekm  = ekm;
      capnt->kex  = kex  * area;
    }
    else capnt->kex = 0;
  
  } 		/* if (!comp1->capnt)  */

  else {	/* if ca comp already exists */
    cao = capnt->cao;
    cai = capnt->cais[1];
  }

  chpnt->comp2 = (comp *)capnt;	/* pointer from ca chan to ca comp */
  maklst(&capnt->clst, (conn*)chpnt);	/* add to ca comp's list of cachans */

  vrev = R2F * ktemp * log(cao/cai);   /* Nernst equation */
  capnt->vrev = vrev;

			/*  end of ca comp stuff */

  if (!maktabfl) {
	maktabfl = 1;
	maktables(timinc);
  }
  if ((maxcond=apnt->maxcond)==NULLNOD) maxcond = dmaxca;
  if ((tau=apnt->taum) == NULLNOD) tau = dcatau;
  if ((vthr=apnt->thresh) == NULLNOD) vthr = dcathr;
  chpnt->vthr = vthr - CATHR; 
  chpnt->consts  = &catypes[stype];	/* CA constants */
  carate(comp1->v,stype);

  switch (stype) {
   case 0:			/* set up equilibrium rate constants */
      ((hhchan *)chpnt)->m    = alphac / (alphac + betac);
      ((hhchan *)chpnt)->maxcond  = maxcond;
      ((hhchan *)chpnt)->vrev = vrev;
      ((hhchan *)chpnt)->taum = tau;
      break;
   case 1:
   case 2:
      break;
  }

  cumchan++;
  chpnt = addchan(comp1, chpnt); /* add channel to comp's conns */
					/*  but possibly erase new channel   */
					/*  and return pointer to old one.   */
#ifdef DEBUG
 if (debug & 1 && debugz & 16)
   fprintf (stderr,"makca end.\n");
#endif

  return chpnt;
}

/*------------------------------------*/

int findrecep(int num1, int num2, int num3, recep **rpnt, char *str)
{
   recep *pnt;

  for (pnt=recpnt; pnt; pnt=(recep *)pnt->next) {
    if ((pnt->recnm1 == num1) && (pnt->recnm2 == num2) && (pnt->recnm3==num3)) {
	*rpnt = pnt;
	return 1;
    }
  }
  if (str) {
     if (num3 != NULLNOD)
  	 fprintf(stderr,"\n%s: can't find recep %d %d %d\n",str,num1,num2,num3);
     else if (num2 != NULLNOD)
  	 fprintf(stderr,"\n%s: can't find recep %d %d\n",str,num1,num2);
     else 
  	 fprintf(stderr,"\n%s: can't find recep %d\n",str,num1);
   }
  return 0; 
}

/*------------------------------------*/

recstim *makvstim(double start, int nodnm1, int nodnm2, int nodnm3, 
			double value, char *action)
                        
/* Make a new clamp stimulus and link it to the list. 
   A clamp stimulus looks just like a receptor stimulus, but
   has different meaning when running the simulation. 
*/

{
    recstim *rspnt;
 
  if ((rspnt=(recstim *)emalloc(sizeof(recstim))) == NULL) {
     fprintf (stderr,"no space left for clamp stim %d\n", cumrecst+1);
     return (NULL);  
  }
  rspnt->next = NULL;
  if (!recspnt) recspnt = rspnt; 	/* save head if first synap */
  if (recsend)
    recsend->next = rspnt;
  recsend = rspnt;

  rspnt->ctype = *action;
  rspnt->time = start;
  rspnt->recnm1 = nodnm1;
  rspnt->recnm2 = nodnm2;
  rspnt->recnm3 = nodnm3;
  rspnt->val = value;
  cumrecst++; 			/* increment total */
  return (rspnt); 
}

/*------------------------------------*/

recstim *makrstim(double start, int recnm1, int recnm2, int recnm3, 
		double inten, double wavel, char *action)
                              
/* make a new receptor stimulus and link it to the list. */

{
    recstim *rspnt;

  if ((rspnt=(recstim *)emalloc(sizeof(recstim))) == NULL) {
     fprintf (stderr,"no space left for recstim %d\n", cumrecst+1);
     return (NULL);  
  }
  rspnt->next = NULL;
  if (!recspnt) recspnt = rspnt; 	/* save head if first synap */
  if (recsend)
    recsend->next = rspnt;
  recsend = rspnt;

  rspnt->ctype = *action;
  rspnt->time = start;
  rspnt->recnm1 = recnm1;
  rspnt->recnm2 = recnm2;
  rspnt->recnm3 = recnm3;
  rspnt->val = inten;
  rspnt->wavel = wavel;
  cumrecst++; 			/* increment total */
  return (rspnt); 
}

/*------------------------------------*/

void delrstim(recstim *rpnt)

/* delete the current receptor stimulus. */

{
 
  if (!rpnt)  return;
  free(rpnt);			 	/* delete current stimulus */
  cumrecst--; 				/* decrement total */
  return; 
}

/*------------------------------------*/

void maklst(conlst **head, conn *cpnt)

/* add an entry to a connection list */

{
    conlst *lpnt,*tmp,*oldtmp;

  if (!head) { 
     fprintf (stderr,"maklst: incorrect list pointer\n");
     return;  
  }
  if ((lpnt=(conlst *)emalloc(sizeof(conlst))) == NULL) {
     fprintf (stderr,"maklst: no space left for conlist\n");
     return;  
  }
  if (! *head) *head = lpnt;	/* save head if first one */
  else {
    for (oldtmp=tmp= *head; tmp; tmp=tmp->next)  /* find end of list */
       oldtmp = tmp;			   /*  and get pointer to last */
    if (oldtmp) oldtmp->next = lpnt;
  }
 
  lpnt->next = NULL;
  lpnt->conpnt = cpnt;		/* set pointer to connection */
}

/*------------------------------------*/

void dellst(conlst **head, conlst *lpnt)

/* delete an entry from a connection list */

{
    conlst *tmp,*oldtmp;
    int found;

  if (!head) { 
     fprintf (stderr,"lst: incorrect list pointer\n");
     return;  
  }
  if (! *head) return;			/* empty list: nothing to do */
  else if (! lpnt) return;
  else {
    for (found=0,oldtmp=tmp= *head; tmp; oldtmp=tmp,tmp=tmp->next) { 
       if (tmp==lpnt) {
          found = 1;
          break;
       }
    }
    if (found) {
      oldtmp->next = lpnt->next;
      if (lpnt== *head) *head = NULL;
      free (lpnt);
    }
  }    /* else */
}

/*------------------------------------*/

