/* Module Ncinit in Program NC */

/* Translates neuron cable description into
   the equivalent compartmental model. */

/* Assume a dendrite is made up of small isopotential
compartments which are modeled as electrical nodes.
Each node is connected to its neighbors through axial 
resistances, and has conduction pathways to ground
through membrane resistance and capacitance.
Additional conduction pathways are allowed for any node. 

Unless otherwise noted, all length values are calibrated
in meters, all voltages are in volts, all currents are
in amperes, and capacitance is in farads.

	Oct 87			R.G. Smith

*/

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>

#ifdef __cplusplus
}
#endif

#include "adef.h"
#include "nc.h"
#include "y.tab.h"
#include "ncelem.h"
#include "ncomp.h"
#include "ncsub.h"
#include "control.h"

#define CMMSCL  1e-2			/* scale from cm to meters */
#define CMMSCL2 1e-4			/* scale from cm2 to meters2 */
#define UMMSCAL 1e-6			/* scale from um to meters */
#define CMUMSCL  1e4			/* scale from cm to microns */
#define CMUMSCL2 1e8			/* scale from cm2 to microns2 */

#define DEBUG

extern int cumelem,cumnode;
extern int cumcomp;			/* total number of compartments */
extern int cumconn;			/* total number of connections */
extern int cumchan;			/* total number of channels */
extern int cumcacomp;			/* total number of Ca comps */
extern elem *elempnt;			/* pointer to element list */
extern elem *elemend;			/* pointer to end of element list */
extern elem *oepnt;			/* pointer to last element */
extern comp *compnt;			/* pointer to compartment list */
extern comp *compend;			/* pointer to end of compartment list */
extern conn *connpnt;			/* pointer to connection list */
extern conn *connend;			/* pointer to end of connection list */
extern node *nodepnt;			/* pointer to node list */
extern synap *synpnt;			/* pointer to synapse list */

#ifdef __cplusplus
extern "C" {
#endif

double exp(double);
double sqrt(double);
void free (...);

#ifdef __cplusplus
}
#endif

comp *makcomp(elem *epnt, double rm, double cap, double vrest, double vrev);
char *findsym(int);
char *nodstr(int a, int b, int c);
chan *makchan(attrib *apnt, comp *comp1, int type);
chan *makca(elem *epnt, cattrib *apnt, comp *comp1, double area);
synap *maksynap(synapse *epnt, comp *comp1, comp *comp2);
recep *makrecep(photrec *epnt, comp *cpnt);
load *makload(loadelem *epnt, comp *comp1);
conn *makconn(comp *comp1, comp *comp2, double s, int type);
double dist3d(node *n1, node *n2);

void execerror (char*, char*);
void calcnod (double Ri, double Rm, double Cap, double d, double clen, \
		double *rri, double *rrm, double *rcap);
void maklst(conlst **head, conn *cpnt);
void chandens(elem *epnt, comp *cpnt, double area);
void modload(loadelem *lepnt);
void modconn(elem *epnt, double conduct);
synap *modsynap(synapse *sepnt);
recep *modrecep(photrec *pepnt);
void checkcomp(elem *epnt, comp **pnt1, comp **pnt2);
dbuf *makdelbuf(comp *comp1, comp *comp2, double delay, int type);
void freelst (conlst *clst);
void freendp(node *nodpt, elem *epnt); 
void initsyn(void);
void setsynv(synap *spnt, double v);
chan *addchan(comp *pnt, chan *chpnt);
void prelem(elem *epnt);
comp *othercomp (comp *pnt,conn *cpnt);

/*------------------------------------*/

void initcomp(void)

/* Add new compartments to the model */
/* or modify existing compartments. */
/* If element is to be saved for future reference, */
/* store its low-level structure in epnt->lpnt. */
/* If element is actually a modification of an existing */
/* element, then don't remake it, but call the */
/* modification routine. */
   
{
    int i,measrcomp,numcomp,typ;
    static double ri,rm,cri,crm,lambda,cap,lcap,complen,len,dia;
    static double Ri,Rm,cplam,dist;
    elem *epnt,*tepnt;
    attrib *apnt,*tapnt;
    node  *npnt;
    comp  *cpnt,*cpntn,*cpntold,*cpntlast;
    synap *tsynap;
    recep *trecep;
    conn *tconn;

    cable *cepnt;
    sphere *sepnt;
    loadelem *lepnt;
    capac *cappnt;
    gapjunc *gjepnt;
    hhchan *hhepnt;
    sschan *ssepnt;

if (debug & 1) fprintf (stderr,"initcomp\n");

 oepnt=NULL;
 for (epnt=elempnt; epnt; ) {			/* for all elements */

  typ = epnt->ctype; 

/*  if (epnt->modif) 
	fprintf (stderr,"modify elem: num %d typ %d '%s'\n",
		epnt->modif,epnt->ctype,findsym(epnt->ctype)); /* */

  if (debug & 1 && debugz & 4) {

        prelem(epnt);
/*	fprintf (stderr,"elem: num %d typ %d '%s'\n",
		epnt->elnum,epnt->ctype,findsym(epnt->ctype)); /* */
  }

  switch (typ) {

  case CABLE:					/* initial calculations */
  {
     float cvrev, cvrest;

  cepnt = (cable *)epnt;
				/* Don't allow modification of cable */
  if (epnt->modif) { }		/* because that would change number of */	
				/* compartments. Actually, it's do-able... */
				/*  Just find old voltages, erase the old comps */
				/*  and make new ones. But no time now...*/
  else {
   if (epnt->lptr==NULL) { 		/* if not made (not saved) yet */
    if ((len=cepnt->length)==NULLNOD) {    /* and if length not set */
	if (!epnt->nodp1 && !epnt->nodp2) break;
	dist = dist3d(epnt->nodp1,epnt->nodp2); /* calc length from locs */
	if (dist > 1e-6) len = dist;
	else execerror ("Initcomp: ","missing length for cable");
    } 
    if ((Ri=cepnt->Ri)<=0) Ri = dri;	/* use default values if nec. */
    if ((Rm=cepnt->Rm)<=0) Rm = drm;
    if ((cvrev=cepnt->vrev)   == NULLNOD) cvrev  = vcl;
    if ((cvrest=cepnt->vrest) == NULLNOD) cvrest = vcl;
    if ((cplam = cepnt->cplam)==NULLNOD) cplam = complam;

    dia = cepnt->dia;
    ri = Ri * CMUMSCL / (MPI * dia * dia / 4);	/* calc ri (ohm/m) */
    rm = Rm * CMUMSCL2 / (MPI * dia);		/* calc rm (ohm*m) */
    lambda = sqrt (rm / ri); 

    complen = lambda * cplam;			/* compartment length */
    if (complen==0) complen = .001;
    measrcomp = (int)(len / complen) + 1;	/* number of comps in cable */
    complen = len / measrcomp;
    numcomp = measrcomp + 1;

    if (debug & 1 && debugz & 4)
	fprintf (stderr,"cable num %d Ri %g Rm %g lam %g cplam %g complen %g\n",
		epnt->elnum,cepnt->Ri,cepnt->Rm,
		lambda,cplam,complen);

    if ((lcap=cepnt->Cm)==NULLNOD) lcap = dcap;
    if (lcap==0) lcap = 1e-6;
    calcnod (Ri,Rm,lcap,cepnt->dia,complen,&cri,&crm,&cap);

    cpntold=NULL;
    for (i=0; i<numcomp; i++) {		/* make comps for one cable seg */

      if (i==0) {			/* first end (beginning) */
	  				/* first, find start node */
	  if (! (npnt=epnt->nodp1)) {
	    fprintf (stderr,"initcomp: elem %d: can't find node 1: %d %d %d\n",
			epnt->elnum,epnt->node1a,epnt->node1b,epnt->node1c);
 	    execerror("Error","");
		break;
	  }
	  cpntn = npnt->comptr;		/* check for compartment at start */
	  if (cpntn) {			/* node already has compartment   */
	    if (cpntn->ctype != SPHERE) {
	      cpntn->vrev = cvrev;      /* cable controls node reversal v */
	      cpntn->v = cvrest;        /* cable controls node initial v  */
	    }
	    cpntn->rm  += 0.5/crm;	/* add to node conductance        */
            cpntn->cap += cap/2;	/* membrane capacitance           */
	    cpntlast = cpntn;		/* save node comp pntr for connection */
	  }
	  else {			/* this is first comp for node */
					/* make the new compartment */
	    cpnt = makcomp(epnt,0.5/crm,cap/2,cvrest,cvrev);
	    cpnt->extvi = cplam;	/* save complam for condense() below */
	    npnt->comptr = cpnt;	/* save comp pntr for node */ 
            maklst(&cpnt->nodlst,(conn*)npnt);	/* save node pntr for comp */
            cpntlast = cpnt;		/* last pointer for next connection */
	  }
	 if (epnt->attpnt)
	      chandens(epnt,cpntlast,cap*0.5/lcap); /* volt-sens chan */
      }
      else if(i==(numcomp-1)) {		/* second end */
	 				/* find far end node */
	  if (! (npnt=epnt->nodp2)) {
	     fprintf (stderr,"initcomp: elem %d: can't find node 2: %d %d %d\n",
			epnt->elnum,epnt->node2a,epnt->node2b,epnt->node2c);
 	     execerror("Error","");
  	     break;
	  }
	  cpntn = npnt->comptr;		/* check for compartment at end */
	  if (cpntn) {			/* node already has compartment */
	    tconn = makconn(cpntn,cpntold,
				1/cri,AXIALRES);/* axial resis to last comp */
	    if (cpntn->ctype != SPHERE) {
	      cpntn->vrev = cvrev;      /*cable controls node rev */
	      cpntn->v = cvrest;        /*cable controls node init */
	    }
	    cpntn->rm  += 0.5/crm;	/* add conductance */
            cpntn->cap += cap/2;	/* membrane capacitance         */
	    cpnt = cpntn;
	  }
	  else {			/* this is first comp for node */
					/* make the new compartment */
	    cpnt = makcomp(epnt,0.5/crm,cap/2,cvrest,cvrev);
	    cpnt->extvi = cplam;	/* save complam for condense() below */
	    tconn = makconn(cpnt,cpntold,
			1/cri,AXIALRES); /* axial resis to last comp */
	    npnt->comptr = cpnt;	/* save comp pntr for node */ 
            maklst(&cpnt->nodlst,(conn*)npnt);/* save node pntr for comp */
	  }
	 if (epnt->attpnt)
	   chandens(epnt,cpnt,cap*0.5/lcap); /* if volt-sens chan */

					/* if there are only 2 comps, */
    	if (epnt->saved && numcomp==2) 	/* save the conn to remember cable */ 
	     epnt->lptr = (elem *)tconn; 	     /* save low-lev ptr */
		
      }
      else {	/* i>0 && i<numcomp-1 */   /* this is middle of branch */
					   /* make the new compartment */
	cpnt = makcomp(epnt,1.0/crm,cap,cvrest,cvrev);
	cpnt->extvi = cplam;		/* save complam for condense() below */
	tconn = makconn(cpnt,cpntold,
			1/cri,AXIALRES); /* axial resis to last comp */
	if (epnt->attpnt)
		 chandens(epnt,cpnt,cap/lcap); /* voltage-sens chans */
	cpntlast = cpnt;
					/* if there are more than 2 comps, */
    	if (epnt->saved && numcomp>2) 	/* save first conn to remember cable */
    	 	  if (i==1) epnt->lptr = (elem *)tconn;   /* save low-lev ptr */
      }
      cpntold = cpntlast;		/* save the current pointer */

      }   /* for ( ; i<numcomp; ) */
    }	/* if (lptr==NULL) */
   }   /* else if not (modif) */
  }    /* case CABLE: */
   break;

     case SPHERE:
	   {
              float svrev, svrest;

	   sepnt = (sphere *) epnt;
    	   if ((svrev=sepnt->vrev)  == NULLNOD)  svrev  = vcl;
    	   if ((svrest=sepnt->vrest) == NULLNOD) svrest = vcl;
	   if ((Rm=sepnt->Rm) <= 0) Rm = drm;
  	   dia = sepnt->dia * UMMSCAL;
 	   rm = Rm * CMMSCL2 / (MPI * dia * dia);
    	   if ((lcap=sepnt->Cm)==NULLNOD) lcap = dcap;	/* lcap is Cm */
 	   cap = lcap * MPI * dia * dia / CMMSCL2;  

	  if (! (npnt=epnt->nodp1)) {
	    fprintf (stderr,"initcomp: sphere %d: can't find node %d %d %d\n",
			epnt->elnum,epnt->node1a,epnt->node1b,epnt->node1c);
 	    execerror("Error","");
		break;
	  }
	  cpnt = npnt->comptr;		/* check for existing compartment */
	  if (cpnt) {			/* node already has compartment */
	    if (typ == SPHERE) {
	      cpnt->vrev  = svrev;      /* sphere controls node reversal v */
	      cpnt->v = svrest;         /* sphere controls node initial v */
	    }
	    cpnt->rm  += 1/rm;		/* add conductance 		*/
            cpnt->cap += cap;		/* membrane capacitance         */
	  }
	  else {
	    cpnt = makcomp(epnt,1/rm,cap,svrest,svrev);
	    npnt->comptr = cpnt;	/* save comp pntr for node 	*/ 
            maklst(&cpnt->nodlst,(conn *)npnt);	/* save node pntr for comp */
	  }
	  if (epnt->attpnt)
		 chandens(epnt,cpnt,cpnt->cap/lcap); /* voltage-sens chans */
	  }
	  break;

      case LOAD:
           {
	       float lvrev, lvrest;

	   lepnt = (loadelem*)epnt;
	   cap = SMALLCAP;
	   if ((lvrev=lepnt->vrev)   == NULLNOD) lvrev = 0;
	   if ((lvrest=lepnt->vrest) == NULLNOD) lvrest = 0;
  	   if (epnt->modif) {
	       modload(lepnt);			/* modify old load */
	   }
	   else {
		load *tload;

	    if (epnt->lptr==NULL) { 			/* check lowlev ptr */
	      if (! (npnt=epnt->nodp1)) {
	       fprintf (stderr,"initcomp: load %d: can't find node %d %d %d\n",
			epnt->elnum,epnt->node1a,epnt->node1b,epnt->node1c);
 	        execerror("Error","");
		break;
	      }
	      cpnt = npnt->comptr;	/* check for existing compartment */
	      if (!cpnt) {		/* node doesn't have compartment */
	        cpnt = makcomp(epnt,0.0,cap,lvrest,lvrev);
	        npnt->comptr = cpnt;	       /* save comp pntr for node  */ 
                maklst(&cpnt->nodlst,(conn *)npnt); /* save node pntr for comp*/
	      }
	      tload = makload((loadelem*)epnt,cpnt); 
	      if (epnt->saved) epnt->lptr = (elem *)tload; /* save synap ptr */
	     }
	    }
	   }
	   break;

      case GNDCAP:
	  {
		float capvrest;

	   cappnt = (capac *)epnt;
	   rm = LARGERES;
	   cap = cappnt->c;
	   if ((capvrest=cappnt->vrest) == NULLNOD) capvrest = 0;
           if (! (npnt=epnt->nodp1)) {
	     fprintf (stderr,"initcomp: sphere %d: can't find node %d %d %d\n",
			epnt->elnum,epnt->node1a,epnt->node1b,epnt->node1c);
 	     execerror("Error","");
		break;
	   }
	  cpnt = npnt->comptr;		/* check for existing compartment */
	  if (cpnt) {			/* node already has compartment */
	   if (cappnt->vrest != NULLNOD) /* gndcap controls node initial v */
	          cpnt->v = capvrest;    
	    cpnt->rm  += 1/rm;		/* add conductance 		*/
            cpnt->cap += cap;		/* membrane capacitance         */
	  }
	  else {
	    cpnt = makcomp(epnt,1/rm,cap,capvrest,(double)NULLNOD);
	    npnt->comptr = cpnt;	/* save comp pntr for node 	*/ 
            maklst(&cpnt->nodlst,(conn *)npnt);	/* save node pntr for comp */
	  }
	  if (epnt->attpnt)
		 chandens(epnt,cpnt,cpnt->cap/lcap); /* voltage-sens chans */
          }
	 break;

      case RESISTOR:

	 ri = ((resistor *)epnt)->z;
	 if (ri == 0.0) ri = 1.0;
	 if (epnt->modif) {
		modconn(epnt,1/ri);			/* modify old resist */
	 }
	 else {
	  if (epnt->lptr==NULL) { 			/* check lowlev ptr */
	   checkcomp(epnt,&cpntn,&cpntlast);
	   tconn=makconn(cpntn,cpntlast,1/ri,typ); /* axial resis to comp */
	   if (epnt->saved) epnt->lptr = (elem *)tconn; /* save synap ptr */
	  }
	 }
 	 break;

      case CAP:
	 cappnt = (capac *)epnt;	 
	 checkcomp(epnt,&cpntn,&cpntlast);
	 makconn(cpntn,cpntlast,cappnt->c,typ); /* series capac */
 	 break;

      case BATT:

	 checkcomp(epnt,&cpntn,&cpntlast);
	 makconn(cpntn,cpntlast,((batt *)epnt)->v,typ); /* series batt */
 	 break;

      case BUF:			/* voltage buffer (voltage follower) */

	{
	 dbuf *tdpnt;

	 checkcomp(epnt,&cpntn,&cpntlast);
	 tdpnt = makdelbuf(cpntn,cpntlast,((vbuf *)epnt)->delay,typ);
	 cpntlast->miscfl |= VEXT | VBAT | VBUF;  /* set comp flags */
 	 break;
	}

      case GJ:			/* gap junction */

	{
	  double specres;

	  gjepnt = (gapjunc *)epnt;
	  if ((specres = gjepnt->specres) < 1e-20) specres = drg;
	  ri = specres / gjepnt->area;	/* spec res / gj area */
	  if (ri == 0.0) ri = 1.0;
	}

	 if (epnt->modif) {
		modconn(epnt,1.0/ri);			/* modify old gj */
	 }
	 else {
	  if (epnt->lptr==NULL) { 			/* check lowlev ptr */
	   checkcomp(epnt,&cpntn,&cpntlast);
	   tconn=makconn(cpntn,cpntlast,1/ri,typ); /* axial resis to last comp*/
	   if (epnt->saved) epnt->lptr = (elem *)tconn; /* save synap ptr */
	  }
	 }
 	 break;

      case CHAN:		/* calcium channel with compartment */
	  if (epnt->attpnt) apnt = epnt->attpnt;
          else { 
		fprintf (stderr,"initcomp chan: can't find chan attrib\n");
 	        execerror("Error","");
		break;
	  }
	  if (! (npnt=epnt->nodp1)) {
		fprintf (stderr,"initcomp chan: can't find node %d %d %d\n",
			epnt->node1a,epnt->node1b,epnt->node1c);
 	        execerror("Error","");
		break;
	  }				/* Check back at node, to */
	  cpnt = npnt->comptr;		/* look for existing compartment */
	  if (!cpnt) {
	    cap = 0.0;
    	    cpnt = makcomp(epnt,1.0/LARGERES,SMALLCAP,NULLNOD,apnt->vrev);
	    npnt->comptr = cpnt;	/* save comp pntr for node 	*/ 
            maklst(&cpnt->nodlst,(conn *)npnt);	/* save node pntr for comp */
	    cpnt->ctype = apnt->ctype;	/* set chan type only if new */
	  }
	  else {
	    cap = cpnt->cap;
	  }
          for ( ; apnt; apnt=apnt->attpnt) {
	    chan *chpnt;
	    int chtyp;
	    double area,density;

				/* When "density" specified, we need "area". */

	     if ((cap<=SMALLCAP) && 
                  (apnt->maxcond==NULLNOD) && (apnt->density!=NULLNOD)) {
                   double r;

                r = CACOMPRAD;
                area = 4.0 * MPI * r * r;
                printf ("# nc: initcomp: compartment for channel has not been defined.\n");
                printf ("# Using diameter of %g um for compartment.",CACOMPRAD*2); 
             }
             else
	       area = cap/lcap;	     /* Area used when comp already specified */
				     /*  calibrated in cm2 */

             chtyp = apnt->ctype;
	     switch (chtyp) {

	      case HH:

	       if (apnt->maxcond == NULLNOD) 
		   apnt->maxcond = dmaxna; 
	       chpnt = makchan(apnt,cpnt,NA); /* both na, k chans in one comp */
	       if (epnt->saved) apnt->lptr = (elem *)chpnt; /* save chan ptr */
               chpnt = makchan(apnt,cpnt,K);
	       chpnt->maxcond *= 0.2;	/* make k conductance 1/5 na cond */
	       break;
	      case NA:
	         if (apnt->maxcond == NULLNOD) { 
	  	    if (apnt->density==NULLNOD) apnt->maxcond=dmaxna;
	            else apnt->maxcond = apnt->density * area;
		 }
	         chpnt=makchan(apnt,cpnt,chtyp);/* single channel for attrib */
		 break;
	      case K:
	         if (apnt->maxcond == NULLNOD) {
	  	    if (apnt->density == NULLNOD) apnt->maxcond=dmaxk;
	            else apnt->maxcond = apnt->density * area;
		 }
	         chpnt=makchan(apnt,cpnt,chtyp);/* single channel for attrib */
		 break;
	      case CA:
	         if (cap<=SMALLCAP) {
		    double r;

                 r = CACOMPRAD;
                 area = 4.0 * MPI * r * r; 
                 printf 
              ("# nc: initcomp: compartment for ca chan has not been defined.\n");
                 printf ("# Using diameter of %g um for compartment.",CACOMPRAD*2); 
                 }
	         if (apnt->maxcond == NULLNOD) {
	  	    if (apnt->density == NULLNOD) apnt->maxcond=dmaxca;
	            else apnt->maxcond = apnt->density * area;
		 }
	         chpnt=makca(epnt,(cattrib*)apnt,cpnt,area); /* single channel */
		 break;

	      }  /* switch */

	       if (epnt->saved) apnt->lptr = (elem *)chpnt; /* save chan ptr */
         }  /* for (;apnt;) */

 	 break;

      case SYNAPSE:

	 if (epnt->modif) {
	    modsynap((synapse*)epnt);		/* modify old synapse */
	 }
	 else {
	  if (epnt->lptr==NULL) { 		/* check lowlev ptr */
	    checkcomp(epnt,&cpntn,&cpntlast); 
	    tsynap = maksynap((synapse*)epnt,cpntn,cpntlast); 
	    if (epnt->saved) epnt->lptr = (elem *)tsynap; /* save synap ptr */
	  }
	 }
	 break;

      case ROD:
      case CONE:

	 if (epnt->modif) {
	    modrecep((photrec*)epnt);		/* modify old recep */
	 }
	 else {
	  if (epnt->lptr==NULL) { 			/* check lowlev ptr */
	     if (! (npnt=epnt->nodp1)) {
	  	fprintf (stderr,"check recep node: can't find node %s\n",
			nodstr(epnt->node1a,epnt->node1b,epnt->node1c));
 	        execerror("Error","");
		break;
	     }
  	    cpnt = npnt->comptr;		/* check for comp at node */
  	    if (!cpnt) {			/* new node for comp */
    	      cpnt = makcomp(epnt,1/LARGERES,SMALLCAP,NULLNOD,NULLNOD);
    	      npnt->comptr = cpnt;		/* save comp pntr for node */ 
              maklst(&cpnt->nodlst,(conn *)npnt);/* save node pntr for comp */
  	    }
	    trecep = makrecep((photrec*)epnt,cpnt);
	    if (epnt->saved) epnt->lptr = (elem *)trecep; /* save recep ptr */
	  }
	 }
	 break;
 
      case GNDBATT:

	 rm = 1e6;			/* small leakage current */
	 cap = 1.0;			/* one farad for battery cap */
	  if (! (npnt=epnt->nodp1)) {
		fprintf (stderr,"initcomp battery: can't find node %s\n",
			nodstr(epnt->node1a,epnt->node1b,epnt->node1c));
 	        execerror("Error","");
		break;
	  }
	  cpnt = npnt->comptr;		/* check for existing compartment */
	  if (cpnt) {			/* node already has compartment */
	   cpnt->rm += 1/rm;		/* add conductance 		*/
	   cpnt->cap += cap;
	  }
	  else {
	    cpnt = makcomp(epnt,1/rm,cap,NULLNOD,NULLNOD);
	    npnt->comptr = cpnt;	/* save comp pntr for node 	*/ 
            maklst(&cpnt->nodlst,(conn*)npnt);	/* save node pntr for comp */
	  }
	  cpnt->ctype = epnt->ctype;	
	  cpnt->extv  = ((batt *)epnt)->v;/* set up compartment as battery */
	  cpnt->v     = ((batt *)epnt)->v;
	  cpnt->miscfl |= VEXT | VBAT;	/* set comp flags as vclamp and batt */
 
	 break;

      case ELEMENT:
	 break;

  }  /* switch */ 
 
  if (epnt->saved) {			/* if elem is saved or will be */
					/*  displayed, don't erase */
    oepnt = epnt;	 		/* remember this one */
    epnt = epnt->next; 			/* skip this one */
  }
  else {				/* erase all elems after translation */
    for (apnt=epnt->attpnt; apnt; ) {	/* free element's attributes first */
        tapnt = apnt;
        apnt = apnt->attpnt;
	free (tapnt);
    }
    tepnt = epnt;			/* the elem being erased */
    freendp(epnt->nodp1,epnt); 		/* free the nodpnt to this elem */
    freendp(epnt->nodp2,epnt); 		/* free the nodpnt to this elem */
    epnt = epnt->next; 			/* the next one */
    if (oepnt)
	oepnt->next = epnt;		/* patch the last saved one */
    if (elempnt==tepnt)			/* is elempnt the one being erased? */
        elempnt = epnt;			/*  yes, then incr elempnt */
    if (elemend==tepnt)			/* is elemend the one being erased? */
        elemend = oepnt;		/*  yes, then decr elemend */
    free(tepnt);			/* erase the element */
    if (elemend) elemend->next = NULL;	/* clear end of list */
  }

 } /* for (epnt;; ) */

if (debug & 1) fprintf (stderr,"initcomp end\n");
}

/*-------------------------------------*/

void chandens(elem *epnt, comp *cpnt, double area)
              
/* Add voltage sensitive channels to a compartment.
   If channel "maxcond" is not specified, calculate
   channel max conductance by multiplying channel
   density by compartment membrane area.  Surface area
   is calibrated in cm2.
*/

{
    chan *chpnt;
    attrib *apnt;

  for (apnt=epnt->attpnt; apnt; apnt=apnt->attpnt) {
	   chan *chpnt2;

      switch (apnt->ctype) {
      
	case HH: 
	   if (apnt->density == NULLNOD) apnt->density = dnadens;
	   apnt->maxcond = apnt->density * area;
           chpnt = makchan(apnt,cpnt,NA);  /* na and k channels in one comp */
           chpnt2 = makchan(apnt,cpnt,K);
	   chpnt2->maxcond = chpnt->maxcond*0.2;  /* k cond = 1/5 na cond */
      	   break;
	case NA: 
	  if (apnt->density == NULLNOD) apnt->density = dnadens;
	  apnt->maxcond = apnt->density * area;
          chpnt = makchan(apnt,cpnt,apnt->ctype); 
	  break;
	case K: 
	  if (apnt->density == NULLNOD) apnt->density = dkdens;
	  apnt->maxcond = apnt->density * area;
          chpnt = makchan(apnt,cpnt,apnt->ctype); 
	  break;
        case CA:
	  if (apnt->density == NULLNOD) apnt->density = dcadens;
	  apnt->maxcond = apnt->density * area;
          chpnt = makca(epnt,(cattrib*)apnt,cpnt,area); 
	  break;

      }	    /* switch (ctype) */

     if (epnt->saved) apnt->lptr = (elem *)chpnt; /* save chan ptr */
		
  }  /* for (apnt= ; ; ) */
} 

/*------------------------------------*/

void checkcomp(elem *epnt, comp **pnt1, comp **pnt2)
               
/* find the compartments that the synapse connects to,
   and make new ones if necessary.
*/

{
    node  *npnt;
    comp  *cpnt,*cpntn,*cpntl;

if (debug & 2 && debugz & 4) fprintf (stderr,"checkcomp \n");

 				/* find first node */
  if (! (npnt=epnt->nodp1)) {
	fprintf (stderr,"Checkcomp: elem '%s', ",findsym(epnt->ctype));
	fprintf (stderr,"can't find node %s\n",
		nodstr(epnt->node1a,epnt->node1b,epnt->node1c));
        execerror("Error: incorrect node spec.","");
	return;
  }
  cpntn = npnt->comptr;		/* check for compartment at end */
  if (cpntn) {			/* node already has compartment */
           cpntl = cpntn;
  }
  else {			/* this is first comp for node */
    cpnt = makcomp(epnt,1/LARGERES,SMALLCAP,NULLNOD,NULLNOD);
    npnt->comptr = cpnt;	/* save comp pntr for node */ 
    maklst(&cpnt->nodlst,(conn*)npnt);	/* save node pntr for comp */
    cpntl = cpnt;
  }

 				/* next, find second node */
  if (! (npnt=epnt->nodp2)) {
	fprintf (stderr,"checkcomp: elem '%s',\n",findsym(epnt->ctype));
	fprintf (stderr,"      Found first node: %s,\n",
		nodstr(epnt->node1a,epnt->node1b,epnt->node1c));
	fprintf (stderr,"can't find second node: %s\n",
		nodstr(epnt->node2a,epnt->node2b,epnt->node2c));
        execerror("Error: incorrect node spec.","");
	return;
  }
  cpntn = npnt->comptr;		 /* check for compartment at end */
  if (!cpntn) {			 /* this is first comp for node */
    cpnt = makcomp(epnt,1/LARGERES,SMALLCAP,NULLNOD,NULLNOD);
    npnt->comptr = cpnt;	 /* save comp pntr for node */ 
    maklst(&cpnt->nodlst,(conn*)npnt);	/* save node pntr for comp */
    cpntn = cpnt;
  }
  *pnt1 = cpntl;		 /* return the compartment pointers */
  *pnt2 = cpntn;

  if (debug & 2 && debugz & 4) fprintf (stderr,"checkcomp end\n");
}

/*------------------------------------*/

char *nodstr (int a, int b, int c)
             

/* make string for printing node in readable format */

{
   static char nodst[48] = {0};

 if (a == NULLNOD) a = NULND;
 if (b == NULLNOD) b = NULND;
 if (c == NULLNOD) c = NULND;
 if (a>=0) {
    if (b>=0) {
       if (c>=0)
         sprintf (nodst,"%3d %3d %3d",a,b,c);
       else
         sprintf (nodst,"%3d %3d",a,b);
    }
    else sprintf (nodst,"%3d",a);
 }
 else sprintf (nodst,"%3d %3d %3d",a,b,c);
 return (nodst);
}

/*------------------------------------*/

void compcon(void)

/* calculate the total unvarying conductance for all compartments. */

{
    static double tcond;
    extern double timinc;
    comp *cpnt;
    conn *conptr;
    conlst *lpnt;

if (debug & 1) fprintf (stderr,"compcon\n");

 for (cpnt=compnt; cpnt; cpnt=cpnt->next)
  {
   for (tcond=0.,lpnt=cpnt->clst; lpnt; lpnt=lpnt->next) 
    {					/* calc total conductance through */
     conptr = lpnt->conpnt;		/*   all connections       */
     if (! conptr) break;
     switch (conptr->ctype)
      {
	case AXIALRES:
	case RESISTOR:
     	  tcond += conptr->conduct;		/* neighbor's axial resist */
		    break;
	case LOAD:				/* load */
     	  tcond += ((load *)conptr)->conduct;	/* conductance of load */
		    break;

	case BATT:				/* series battery */
     	  tcond += BATTCOND;			/* battery conductance */
		    break;

	case CAP:
     	  tcond += conptr->conduct/timinc;	/* capacitor reactance */
		    break;
	case GJ:
	case ROD:
	case CONE:
	case SYNAPSE:
	default:
		    break;
      }
    }
   cpnt->tcond = tcond + cpnt->rm;		/* total constant conductance */
   if (cpnt->miscfl & VBAT) 			/* if comp is battery */
     cpnt->tcond += BATTCOND;			/* battery conductance */
   if (implicit || euler)			/* fully implicit */
     cpnt->k = timinc / cpnt->cap;		/* constant for node */
   else						/* crank-Nicholson */
     cpnt->k = timinc / (2. * cpnt->cap);	/* constant for node */
   cpnt->relax = relax;				/* set initial over-relax */
   cpnt->vest = cpnt->v;			/* set initial volt est */

  }
  initsyn();					/* initialize all synapses */

if (debug & 1) fprintf (stderr,"compcon end\n");
}

/*------------------------------------*/

void initsyn(void)

/* initialize all synaptic connections' */
/*  internal activity according to the presynaptic */
/*  voltage */

{
   synap *spnt;

 for (spnt=synpnt; spnt; spnt= (synap*)spnt->next)
   setsynv(spnt, spnt->comp1->v);
}

/*------------------------------------*/

void calcnod (double Ri, double Rm, double Cap, double d, double clen, \
		double *rri, double *rrm, double *rcap)
                                           

/* Calculate the conductance and capacitance for an incremental
node.  Ri, Rm and Cap (on input) are scaled to centimeters.
clen is length of compartments.
*/

{
  double a;
  double ri;			/* normalized to ohms / meter */
  double rm;			/* normalized to ohms * meter */
  double cap;			/* normalized to uF / meter */

 clen *= UMMSCAL;		/* normalize comp len to meters */
 a = d * UMMSCAL / 2;
 ri = Ri * CMMSCL / (MPI * a * a);
 rm = Rm * CMMSCL2 /  (2 * MPI * a);
 cap = Cap * 2 * MPI * a  / CMMSCL2;
 *rrm = rm / clen;
 *rri = ri * clen;
 *rcap = cap * clen;

/*
if (not clear)
 {
  fprintf (stderr,"a %g Ri %g Rm %g Cap %g clen %g\n",a,Ri,Rm,Cap,clen);
  fprintf (stderr,"a %g ri %g rm %g cap %g\n",a,ri,rm,cap);
  fprintf (stderr,"rri %g rrm %g rcap %g\n",*rri,*rrm,*rcap);
 }
*/

}

/*------------------------------------*/

void condense(void)

/*   This algorithm condenses closely coupled compartments
   with their neighbors.  This function is useful because
   the algorithm of "initcomp()" (which generates compartments
   from the element list) leaves a minimum of 2 compartments
   per cable segment, even if the 2 compartments are closer
   (smaller ri/rm) than specified by "complam".

     The algorithm: For all compartments, check whether 
   (for each of their axial resistive connections)
   sqrt(rm/ri) > lamcrit.  If it isn't (i.e compartment
   is more closely coupled than criterion), then collapse the
   compartment into its neighbor.  Add gm and cm, and add
   ri in proportion to the gm's of the two compartments.
   Move any attributes and synapses to the neighbor, and
   erase the connection between the two compartments.  Also,
   the node that points to the compartment is redirected to
   the neighbor as well.  This operation, of course, may
   leave the neighbor compartment closely coupled to another
   neighbor so that it also needs to be condensed.  Since
   the neighbor may already have been passed over, the whole
   process needs to be repeated.  Therefore, do this condensation
   operation on all compartments, and go back to the start of
   the compartment list to iterate until no more changes.

   Since all compartments in a cable (except ends) are always
   made with proper size rm/ri, we only need to check those 
   compartments associated with nodes.  Therefore, the main 
   loop of the algorithm is done for nodes, not compartments.
   Thus it is fairly easy to move the nodes' compartment pointers
   from the old erased compartment to the new compartment.
*/

{
    int change,done,n,oldcomp;
    comp *cpnt,*ocpnt,*tcpnt;
    conn *conpnt;
    node *npnt,*tnpnt;
    static conlst *lpnt,*nlpnt,*olpnt,*tlpnt;
    double ri,rm,orm,lam,cond,crit,varycrit;
    double ocond,tcond,minlam,ominlam,maxcplam;

#ifdef DEBUG
 if (debug & 1 && debugz && 8) fprintf (stderr,"condense\n");
#endif

maxcplam= 0;
minlam=1000;
if (lamcrit==0) return;			/* no condense on zero lamcrit */
if (lamcrit<0) lamcrit = - lamcrit;
					/* condense lowest lambda comps first */
					/* by varying crit from .2 to 1 */
for (varycrit=.2; varycrit<=1; varycrit+= 0.2) {

 crit = lamcrit * varycrit;		/* variable criterion for lambda */
#ifdef DEBUG
    if ((debug & 2) && (debugz & 8))
     fprintf(stderr,"condense: lambda crit %g...\n", crit);
#endif
 for (change=1; change; ) {	/* iterate last crit while changes made */
  change = 0;
  ominlam = minlam;
  minlam=1000;
  for (npnt=nodepnt; npnt; npnt=npnt->next) {	/* for all nodes */
    cpnt = npnt->comptr;			/* get node's compartment */
    if (!cpnt) {
  fprintf (stderr,"condense: node %s has no compartment. continuing...\n",
			  nodstr(npnt->nodenm1,npnt->nodenm2,npnt->nodenm3));
        continue;			/* skip null node */
    }
    oldcomp = 0;
    for (done=0,lpnt=cpnt->clst; lpnt && !done; lpnt=lpnt->next) {
       if (!(conpnt=lpnt->conpnt)) continue;
       if (conpnt->ctype==AXIALRES) {	/* check all axial connections */

     if (conpnt->comp1==cpnt) ocpnt = conpnt->comp2;  /* find other comp */
     else                     ocpnt = conpnt->comp1;

    rm  =  cpnt->rm;			/* save comp rm */
    orm = ocpnt->rm;			/* save neighbor's comp rm */
    ri = conpnt->conduct;
    if (ri==0) ri = 1e-30;
    tcond = orm/(rm+orm) * 1/ri;	/* this compartment's share of ri */
    ocond =  rm/(rm+orm) * 1/ri;	/* other compartment's share of ri */
    lam = sqrt (rm/ri);
    ominlam = minlam; 
    if (lam < minlam) minlam = lam;	/* remember minimum lambda */
    if (cpnt->extvi > maxcplam) maxcplam = cpnt->extvi;	/* maximum complam */

#ifdef DEBUG
    if ((debug & 2) && (debugz & 8))
    fprintf(stderr,"node %s, conn from comp %d to %d lam %-4.2g cplam %g\n",
			nodstr(npnt->nodenm1,npnt->nodenm2,npnt->nodenm3),
			cpnt->num,ocpnt->num,lam,cpnt->extvi); /* */
#endif
    if (cpnt==ocpnt) {
#ifdef DEBUG
    if ((debug & 2) && (debugz & 8))
    fprintf(stderr,"node %s, conn from comp %d to %d is in tight loop ****\n",
			nodstr(npnt->nodenm1,npnt->nodenm2,npnt->nodenm3),
			cpnt->num,ocpnt->num); /* */
#endif
        lpnt->conpnt = NULL;	 /* remember we're not erasing conlst */
        free (conpnt);		 /* erase connection from comp to itself */
	continue;
    }
    if (lam < cpnt->extvi * crit) {		/* if this comp is too close */
						/* proceed with condensation */
#ifdef DEBUG
    if ((debug & 2) && (debugz & 8))
     fprintf(stderr,"condensing comp %d lam %-4.2g into comp %d\n",
			cpnt->num,lam,ocpnt->num); /* */
#endif
       if (varycrit > .99) change = 1;		/* only repeat last crit */
       ocpnt->rm = ocpnt->rm + rm;		/* condense rm("gm"), cap */
       ocpnt->cap = ocpnt->cap + cpnt->cap;
       minlam = ominlam;			/* set minlam to prev. value */
						/* because comp is condensed */

		/* At this point, we've decided to condense a compartment */
		/*  with its neighbor. The first order of business is to  */
		/*  move all the compartment's connections (except the one */
		/*  already connecting it with the neighbor we're condensing */
		/*  to connect to the compartment we're condensing into.  */
		/*  Once we've moved the connections, we can delete the   */
		/*  old connection pointers */

	for (n=0,olpnt=tlpnt=NULL,nlpnt=cpnt->clst;
		 nlpnt;
		 nlpnt=nlpnt->next, (tlpnt?free(tlpnt),1:0), tlpnt=NULL) {
	   if (!(conpnt=nlpnt->conpnt)) continue;
           switch (conpnt->ctype) {

             case AXIALRES:		
		if (nlpnt==lpnt) {	/* don't want to move orig conn */
	          if (olpnt) olpnt->next = nlpnt->next;  
		  else cpnt->clst = nlpnt->next;
		  tlpnt = nlpnt;	/* remember this one to free later */
		}
		/* It is unlikely that there might be 2 axial conns */
		/* between the 2 compartments being condensed,  */
		/* but it is possible. */
		/* In this case, there should no big problem because */
		/* this loop will move the second connection's */
		/* pointer so the connection points from the comp */
		/* to the same comp. This will, of course, forget */
		/* about the second conn. But if such a conn is */
		/* important, then one should not condense at all. */ 

		else {				/* move other axial res */
     	          if (conpnt->comp1==cpnt) conpnt->comp1 = ocpnt;
     	          else                     conpnt->comp2 = ocpnt;
		  n++;				/* count up other axial conns */
		}
	       break;

	     case GJ:
	     case RESISTOR:
     	       if (conpnt->comp1==cpnt) conpnt->comp1 = ocpnt;
     	       else                     conpnt->comp2 = ocpnt;
	       n++;			/* count up resistive conns */
	       break;
 
	     case NA:
	     case K:		/* add channel to other comp, then delete */

	       if (olpnt) olpnt->next = nlpnt->next;  
	       else cpnt->clst = nlpnt->next;
	       tlpnt = nlpnt;		/* free conn pointer later (above) */
	       if (!addchan (ocpnt,(chan*)conpnt)) {/* add chan to other comp */
		  fprintf (stderr,"can't add chan to other comp.\n");
	        }
		break;

	     case CA:		/* mostly same as "NA" */
	       if (olpnt) olpnt->next = nlpnt->next;  
	       else cpnt->clst = nlpnt->next;
	       tlpnt = nlpnt;		/* free conn pointer later (above) */
	       if (!addchan (ocpnt,(chan*)conpnt)) {/* add chan to other comp */
		  fprintf (stderr,"can't add chan to other comp.\n");
	        }

		   /* Special calcium stuff: */

	       if (ocpnt->capnt) {	/* use "other" ca comp */
		  maklst(&ocpnt->capnt->clst, conpnt);  /* add to ca comp lst*/
		  if (cpnt->capnt) {	/* on first CA, free its ca comp */
			freelst (cpnt->capnt->clst);
			free (cpnt->capnt); 
			cumcacomp--;
		  }
	       }
	       else {			/* use "this" ca comp */
		  ocpnt->capnt = cpnt->capnt;  /* move cacomp pointer */
		  ocpnt->capnt->comp1 = ocpnt; /* ca comp pntr to v comp */
	       }
	       conpnt->comp2 = (comp *)ocpnt->capnt;
	       break;

	     case SYNAPSE:
     	       if (((synap *)conpnt)->comp1==cpnt)
				 ((synap *)conpnt)->comp1 = ocpnt;
     	       else              ((synap *)conpnt)->comp2 = ocpnt;
		break;

	     case ROD:
	     case CONE:
     	       if (((recep *)conpnt)->comp1==cpnt)
				 ((recep *)conpnt)->comp1 = ocpnt;
		break;

	     case LOAD:
     	       if (((load *)conpnt)->comp1==cpnt)
				 ((load *)conpnt)->comp1 = ocpnt;
		break;

             case BATT:
	     case CAP:
     	       if (conpnt->comp1==cpnt) conpnt->comp1 = ocpnt;
     	       else                     conpnt->comp2 = ocpnt;
	       break;

	     case BUF:
     	       if (((dbuf *)conpnt)->comp1==cpnt)
				 ((dbuf *)conpnt)->comp1 = ocpnt;
	       else		 ((dbuf *)conpnt)->comp2 = ocpnt;
	       break;

	     default:
	       fprintf (stderr,"condense: unknown connection type %d comp %d\n",
						conpnt->ctype,cpnt->num);
	       break;
	   }
	   if (nlpnt!=tlpnt)            /* if not erased */
	      olpnt = nlpnt;		/* remember previous one in list */
	}

		/* now add comp's share of ri to its other axial connections */
		/* partition ri evenly between other conns */

	if (n)			/* if there are any other axial conns */
	for (olpnt=NULL,nlpnt=cpnt->clst; nlpnt; nlpnt=nlpnt->next) {
	   if (!(conpnt=nlpnt->conpnt)) continue;
           switch (conpnt->ctype) {
             case AXIALRES:			/* add ri to other axials */
             case GJ:				/* and even to gj's */
		if (ri==0.0) break;
		if ((cond=conpnt->conduct)==0.0) break;
		conpnt->conduct = 1 / (1/cond + tcond/n);
	   }
	}
			/* move the pointer list */ 
			/* this time, look at "other" compartment (ocpnt) */

       for (n=0,tlpnt=olpnt=NULL,nlpnt=ocpnt->clst;
			 nlpnt;
			 nlpnt=nlpnt->next, (tlpnt?free(tlpnt),1:0),tlpnt=NULL){
	   if (!(conpnt=nlpnt->conpnt)) continue;
	   if (conpnt->comp1==ocpnt) tcpnt = conpnt->comp2;
	   else                      tcpnt = conpnt->comp1;
           switch (conpnt->ctype) {
             case AXIALRES:
	       if (tcpnt==cpnt) {     		/* if this is right one */

		/* patch the connection list pointers */
		/*  and erase the connection */

      		  if (conpnt->last) conpnt->last->next = conpnt->next;
      		  else connpnt = conpnt->next;
      		  if (conpnt->next) conpnt->next->last = conpnt->last;
      		  else connend = conpnt->last;
	          free (conpnt);		/* erase the connection */
		  cumconn--;

	          if (olpnt) olpnt->next = nlpnt->next;
		  else ocpnt->clst = nlpnt->next;  /* erase conn pointer */
		  tlpnt = nlpnt;
	       }
	       else n++;			/* count other axial conn's */
	       break;

	     case RESISTOR:
	     case GJ:
		n++;
		break;
	    }
	    if (nlpnt!=tlpnt)                   /* if not erased */
	        olpnt = nlpnt;			/* remember previous one */
 	}

	  /* now add "other" comp's share of ri to its other axial conns */
	  /* partition ri evenly between other conns */

	if (n) 				/* if neighbor has any axial conns */
	 for (nlpnt=ocpnt->clst; nlpnt; nlpnt=nlpnt->next) {
	   if (!(conpnt=nlpnt->conpnt)) continue;
           switch (conpnt->ctype) {
             case AXIALRES:		/* add extra ri to neighbor's conns */
             case GJ:			/*  and even to gap junctions */
			if (ri==0.0) break;
			if ((cond=conpnt->conduct)==0.0) break;
			conpnt->conduct = 1 / (1/cond + ocond/n);
			break;
	   } /* switch */
	 }  /* for (nlpnt) */

			/* find the end of the neighbor's conn list again */

	for (olpnt=NULL,nlpnt=ocpnt->clst; nlpnt; nlpnt=nlpnt->next) 
		olpnt = nlpnt;

		/* now swap the conn list from the erased compartment */
		/*  to the new (i.e. neighbor) compartment */

	if (olpnt) olpnt->next = cpnt->clst;
	else ocpnt->clst = cpnt->clst;
	
		/* move all node compartment pointers from old comp to new */

      for (nlpnt=cpnt->nodlst; nlpnt; nlpnt=nlpnt->next) {
	  tnpnt = (node *)nlpnt->conpnt;
	  if (tnpnt->comptr == cpnt) tnpnt->comptr = ocpnt;
      }
		/* find the end of the neighbor's node list */

	for (olpnt=NULL,nlpnt=ocpnt->nodlst; nlpnt; nlpnt=nlpnt->next) 
		olpnt = nlpnt;

		/* now swap the node list from the erased compartment */
		/*  to the new (i.e. neighbor) compartment */

	if (olpnt) olpnt->next = cpnt->nodlst;
	else ocpnt->nodlst = cpnt->nodlst;

		/* finally, patch the compartment list pointers */
		/*  and erase the compartment */

      if (cpnt->last) cpnt->last->next = cpnt->next;  /* patch pointers */
      else compnt = cpnt->next;
      if (cpnt->next) cpnt->next->last = cpnt->last;
      else compend = cpnt->last;
 
      free (cpnt);			/* erase the compartment */
      cumcomp--;
      done = 1;		/* stop when a comp and its conns have been erased */

      }	      /* if (lam;;) */
     }	  /* if (conpnt->ctype==AXIALRES) */
    if (done) break;	/* go on to check next compartment */

    oldcomp = cpnt->num;
   }   /*  for (done=0,lpnt;; ) */
  }	  /* for (npnt=nodepnt;;) */
#ifdef DEBUG
    if ((debug & 2) && (debugz & 8))
     if (change) fprintf(stderr,"condense: starting over...\n");
#endif
 }      /* for (change=0;; ) */

 if (minlam > (maxcplam*lamcrit)) break;	/* stop when obviously done */
}   /* for (crit=0.2;;) */

#ifdef DEBUG
  if (debug & 1 && debugz & 8) fprintf (stderr,"condense end\n");
#endif

}     /* condense */


/*------------------------------------*/

comp *othercomp (comp *pnt,conn *cpnt)

/* Return pointer to other comp that connection points to. 
   For reliability, we should modify code in "condense()" above 
    to use this.
*/

{
  if (!pnt || !cpnt) {
      fprintf (stderr,"Othercomp: invalid pointer\n");
      return NULL;
  }
  if (cpnt->comp1==pnt) return cpnt->comp2;
  else return cpnt->comp1; 
}

