/* Segment ncdisp in Program nc */

/* draws anatomy of neuronal circuits */

/*      Nov 90                  R.G. Smith */

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <math.h>
#include "stdplt.h"

#ifdef __cplusplus
}
#endif

#include "ncelem.h"
#include "ncomp.h"
#include "ncsub.h"
#include "adef.h"
#include "nc.h"
#include "y.tab.h"
#include "control.h"
#include "colors.h"
#include "gprim.h"

#ifndef PI
#define PI	3.14159265358979323846264
#endif

extern double ncmat[4][4];		/* scale, rotation matrix for display */
extern double ncmatc[4][4];		/* scale, rotation matrix for camera */
double xyscal=1.0;			/* scale for drawing pictures 0 -> 1 */
double zscal=0;				/* z scale for drawing pictures */
extern double linewidth;		/* set by ncmain */
extern int disp_ray;			/* =1 -> render 3D ray-tracing */

#define DEBUG

extern elem *elempnt;			/* pointer to element list */
extern node *nodepnt;			/* pointer to node list */
extern recstim *recspnt;		/* pointer to recstim list */
extern recstim *recsend;		/* pointer to end of recstim list */
extern recep *recpnt;			/* pointer to receptor list */

extern int ncerror;                     /* error flag set by "warning()" */

#ifdef __cplusplus
extern "C" {
#endif

void free(...);

#ifdef __cplusplus
}
#endif

char *findsym(int num);
double ncabs(double x);
void ncdraw (elem *epnt, int dcolor, double dscale, int hide);
void purge (void);
void drnode(node *npnt, double dscale, int color, double (*mat)[4]);
recep *makr(photrec *epnt, int ctype);
int readstim(double stoptime);
void execerror(char *s, char*t);
int findrecep (int n1, int n2, int n3, recep **rpnt, char *s);
void delrstim(recstim *pnt);
void drsphere (double x, double y, double z, double dia, double dscale,
	int color, int hide, double (*mat)[4], int fill);
void drcable (double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	double dia, double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4]);
void drsynap (double x1, double y1, double z1, 
	double x2, double y2, double z2, synapse *epnt,
	double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4]);
void drconn (double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	int ctype, double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4]);
void drload (double x, double y, double z, int ctype, 
	double n1dia, double n2dia, int color, int hide, double (*mat)[4]);
void drphotrec (double x, double y, double z, 
	     photrec *epnt, double dscale,
	     double n1dia,double n2dia,
	     int color, int hide, double (*mat)[4]);
void raysphere (double x, double y, double z, double dia, double dscale, 
	int color, int hide, double (*mat)[4], int fill);
void raycable (double x1, double y1, double z1, 
		double x2, double y2, double z2, 
		double dia, double dscale, double n1dia, double n2dia, 
		int color, int hide, double (*mat)[4]);
void raysynap (double x1, double y1, double z1, 
	double x2, double y2, double z2, synapse *epnt,
	double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4]);
void rayconn (double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	int ctype, double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4]);
void rayload (double x, double y, double z, int ctype, 
	double n1dia, double n2dia, int color, int hide, double (*mat)[4]);
void rayphotrec (double x, double y, double z, 
		     photrec *epnt, double dscale,
		     double n1dia,double n2dia,
		     int color, int hide, double (*mat)[4]);
void transf (double x,double y,double z,
	      double *tx,double *ty,double *tz, double (*mat)[4]); 
void ppurge();
void phot_dr (int type, int pigm, int color, double dist, double dia, int hide);
void synapse_dr (int color, double vrev, double length, double dia, int hide);
void gapjunc_dr (int color, double length, double dia, int hide);
double callfunc(Symbol *funcp, int npar, double par1, double par2);
void callfunc8(Symbol *funcp, int npar, double par1, double par2, 
			double par3, double par4, double par5, 
			double par6, double par7, double par8);
int loccheck (double x, double y, double z);	/* check if valid location */
void raycalib(double x, double y, double len, double size, int color);

/*------------------------------------*/

/* Definitions for user-defined "draw icon" functions". */

static Symbol *phot_drpnt;
static void (*phd)(int,int,int,double,double,int);
static Symbol *syn_drpnt;
static void (*syd)(int,double,double,double,int);
static Symbol *gapj_drpnt;
static void (*gjd)(int,double,double,int);

/*------------------------------------*/

void ncdisp(int n1a, int n1b, int n1c, int elemtype, int exceptype, 
	int na, int nb, int nc, int color, double dscale, int hide, int excl)

/* Draw neural circuit elements that connect to 
   node1. If any of the node numbers == NULLNOD,
   then ignore this dimension.  If "excl"=1, display
   only elements that connect exclusively to given node.
   If exceptype is set or if the "except" node number is
   set then include appropriate exceptions.
*/

{
   elem *epnt;
   int match,nomatch,exceptf;

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		 fprintf (stderr,"ncdisp n1a %d n1b %d n1c %d\n",n1a,n1b,n1c);
#endif

/* fprintf (stderr,"ncdisp n1a %d n1b %d n1c %d excl %d\n",
					n1a,n1b,n1c,excl);/* */

   exceptf = (na!=NULLNOD || nb!=NULLNOD || nc!=NULLNOD);

   for (epnt=elempnt; epnt; epnt=epnt->next) {	/* search all elems */

#ifdef DEBUG
  if ((debug & 2) && (debugz & 2))
		fprintf (stderr,"ncdisp %d start...",epnt->elnum);
#endif

     if (elemtype  && epnt->ctype!=elemtype) continue;
     match = 1;
     if (n1a != NULLNOD) 			/* check node 1 first */	
        if (epnt->node1a!=n1a) match=0;
     if (n1b != NULLNOD)
        if (epnt->node1b!=n1b) match=0;
     if (n1c != NULLNOD)
        if (epnt->node1c!=n1c) match=0;

     if ((!excl && !match) || (excl && (epnt->node2a!=NULLNOD
				    ||  epnt->node2b!=NULLNOD
				    ||  epnt->node2c!=NULLNOD))) {
	 if (!excl) match = 1;				 /* else check node 2 */
         if (n1a != NULLNOD)
            if (epnt->node2a!=n1a) match=0;
         if (n1b != NULLNOD)
            if (epnt->node2b!=n1b) match=0;
         if (n1c != NULLNOD)
            if (epnt->node2c!=n1c) match=0;
     } 

     if (!match) continue;
     nomatch = 0;
     if (exceptf) {
       nomatch = 1;
       if (na!=NULLNOD && epnt->node1a!=na) nomatch=0;     
       if (nb!=NULLNOD && epnt->node1b!=nb) nomatch=0;     
       if (nc!=NULLNOD && epnt->node1c!=nc) nomatch=0;     

       if (!nomatch) {
       nomatch = 1;
         if (na!=NULLNOD && epnt->node2a!=na) nomatch=0;     
         if (nb!=NULLNOD && epnt->node2b!=nb) nomatch=0;     
         if (nc!=NULLNOD && epnt->node2c!=nc) nomatch=0;     
	}
       if (nomatch) 
           if (exceptype && epnt->ctype!=exceptype) nomatch=0;
      }
      else if (exceptype && epnt->ctype==exceptype) nomatch=1;

     if (nomatch) continue;
     ncdraw(epnt,color,dscale,hide);
   }

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr,"ncdisp end\n");
#endif
}

/*------------------------------------*/

void ncdispc(int n1a, int n1b, int n1c, 
	int n2a, int n2b, int n2c, int elemtype, int exceptype, 
	int na, int nb, int nc, int color, double dscale, int hide)
                                                                        
/* Draw anatomy of neural circuit that connects to 
   both node1 and node2. If any of the node
   numbers == NULLNOD, then ignore this dimension.
 */

{
   elem *epnt;
   int n1pri, n2pri, compat,exceptf;
   int n1_used, match, nomatch;
   int qn1a, qn1b, qn1c, qn2a, qn2b, qn2c;		/* nodes from query */

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		 fprintf (stderr,"ncdispc n1a %d n1b %d n1c %d\n",n1a,n1b,n1c);
#endif

/* fprintf (stderr,"ncdisp n1a %d n1b %d n1c %d n2a %d n2b %d n2c %d\n",
				n1a,n1b,n1c,n2a,n2b,n2c); /* */
  n1pri = 0;
  if (n1a!=NULLNOD) n1pri++;
  if (n1b!=NULLNOD) n1pri++;
  if (n1c!=NULLNOD) n1pri++;

  n2pri = 0;
  if (n2a!=NULLNOD) n2pri++;
  if (n2b!=NULLNOD) n2pri++;
  if (n2c!=NULLNOD) n2pri++;

  compat = 1;				/* check if nodes are same */
  if      (n1a != NULLNOD && n2a != NULLNOD && n1a != n2a) compat = 0;
  else if (n1b != NULLNOD && n2b != NULLNOD && n1b != n2b) compat = 0;
  else if (n1c != NULLNOD && n2c != NULLNOD && n1c != n2c) compat = 0;

  qn1a = n1a;
  qn1b = n1b;
  qn1c = n1c;
  qn2a = n2a;
  qn2b = n2b;
  qn2c = n2c;
  if (compat) 				/* if nodes are partially same */
    if (n1pri < n2pri) {		/* reverse order */
      qn1a = n2a;
      qn1b = n2b;
      qn1c = n2c;
      qn2a = n1a;
      qn2b = n1b;
      qn2c = n1c;
    }

   exceptf = (na!=NULLNOD || nb!=NULLNOD || nc!=NULLNOD);

   for (epnt=elempnt; epnt; epnt=epnt->next) {	/* search all elems */

#ifdef DEBUG
  if ((debug & 2) && (debugz & 2))
		fprintf (stderr,"ncdispc %d start...",epnt->elnum);
#endif

     if (elemtype && epnt->ctype!=elemtype) continue;
     n1_used = 0;
     match = 1;
     if (qn1a != NULLNOD) 			/* check node 1 first */	
        if (epnt->node1a!=qn1a) match=0;
     if (qn1b != NULLNOD)
        if (epnt->node1b!=qn1b) match=0;
     if (qn1c != NULLNOD)
        if (epnt->node1c!=qn1c) match=0;

     if (match) n1_used = 1;
     else {					/* else check node 2 */
	 match = 1;
         if (qn1a != NULLNOD)
            if (epnt->node2a!=qn1a) match=0;
         if (qn1b != NULLNOD)
            if (epnt->node2b!=qn1b) match=0;
         if (qn1c != NULLNOD)
            if (epnt->node2c!=qn1c) match=0;
     }

     if (match) {				/* if query node 1 matches */
      if (!n1_used) {				/*   if node 1 not used yet */
        if (qn2a != NULLNOD) 			/* does qn2 match node 1 ? */	
           if (epnt->node1a!=qn2a) match=0;
        if (qn2b != NULLNOD)
           if (epnt->node1b!=qn2b) match=0;
        if (qn2c != NULLNOD)
           if (epnt->node1c!=qn2c) match=0;
      }
      else {				     /* else if node 2 not used yet */
        if (qn2a != NULLNOD)	     		/* does qn2 match node 2 ? */	
           if (epnt->node2a!=qn2a) match=0;
        if (qn2b != NULLNOD)
           if (epnt->node2b!=qn2b) match=0;
        if (qn2c != NULLNOD)
           if (epnt->node2c!=qn2c) match=0;
      }   /* else (n1_used) */
     }   /* if (match) */

     if (!match) continue;
     nomatch = 0;
     if (exceptf) {
       nomatch = 1;
       if (na!=NULLNOD && epnt->node1a!=na) nomatch=0;     
       if (nb!=NULLNOD && epnt->node1b!=nb) nomatch=0;     
       if (nc!=NULLNOD && epnt->node1c!=nc) nomatch=0;     

       if (!nomatch) {
       nomatch = 1;
         if (na!=NULLNOD && epnt->node2a!=na) nomatch=0;     
         if (nb!=NULLNOD && epnt->node2b!=nb) nomatch=0;     
         if (nc!=NULLNOD && epnt->node2c!=nc) nomatch=0;     
	}
       if (nomatch) 
           if (exceptype && epnt->ctype!=exceptype) nomatch=0;
     }
     else if (exceptype && epnt->ctype==exceptype) nomatch=1;

     if (nomatch) continue;
     ncdraw(epnt,color,dscale,hide);
   }

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr,"ncdispc end\n");
#endif
}

/*------------------------------------*/

void ncdispn(int n1a, int n1b, int n1c, 
	int n2a, int n2b, int n2c, int elemtype, int exceptype, 
	int na, int nb, int nc, int color, double dscale, int hide, int excl)

/* Draw neural circuit elements that connect to
   the range of nodes from node1 to node2.
   If any of the node numbers == NULLNOD,
   then ignore this dimension. 
   If "excl"=1, display only elements that connect 
   exclusively to given node.
*/

{
   int swap;
   int qn1a, qn1b, qn1c, qn2a, qn2b, qn2c;		/* nodes from query */
   int exceptf;
   elem *epnt;
   int match,nomatch;

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr,"ncdispn start...\n");
#endif

  if      (n1a > n2a) swap = 1;
  else if (n1b > n2b) swap = 1;
  else if (n1c > n2c) swap = 1;
  else                swap = 0;

  qn1a = n1a;
  qn1b = n1b;
  qn1c = n1c;
  qn2a = n2a;
  qn2b = n2b;
  qn2c = n2c;
  if (swap) {			/* make qn1 = min, qn2 = max */
      qn1a = n2a;
      qn1b = n2b;
      qn1c = n2c;
      qn2a = n1a;
      qn2b = n1b;
      qn2c = n1c;
  }

   exceptf = (na!=NULLNOD || nb!=NULLNOD || nc!=NULLNOD);

   for (epnt=elempnt; epnt; epnt=epnt->next) {	/* search all elems */

     if (elemtype && epnt->ctype!=elemtype) continue;
     match = 1;
     if (qn1a != NULLNOD) {
        if (epnt->node1a<qn1a) match=0;
	if (qn2a != NULLNOD)
           if (epnt->node1a>qn2a) match=0;
     }
     if (qn1b != NULLNOD) {
        if (epnt->node1b<qn1b) match=0;
	if (qn2b != NULLNOD)
           if (epnt->node1b>qn2b) match=0;
     }
     if (qn1c != NULLNOD) {
        if (epnt->node1c<qn1c) match=0;
	if (qn2c != NULLNOD)
           if (epnt->node1c>qn2c) match=0;
     }

     if ((!excl && !match) || (excl && (epnt->node2a!=NULLNOD
				    ||  epnt->node2b!=NULLNOD
				    ||  epnt->node2c!=NULLNOD))) {
       if (!excl) match = 1;
       if (qn1a != NULLNOD) {
          if (epnt->node2a<qn1a) match=0;
          if (qn2a != NULLNOD) 
             if (epnt->node2a>qn2a) match=0;
       }
       if (qn1b != NULLNOD) {
          if (epnt->node2b<qn1b) match=0;
          if (qn2b != NULLNOD)
              if (epnt->node2b>qn2b) match=0;
       }
       if (qn1c != NULLNOD) {
          if (epnt->node2c<qn1c) match=0;
          if (qn2c != NULLNOD)
             if (epnt->node2c>qn2c) match=0;
       }
     }		/* if (!match) */

     if (!match) continue;
     nomatch = 0;
     if (exceptf) {
       nomatch = 1;
       if (na!=NULLNOD && epnt->node1a!=na) nomatch=0;     
       if (nb!=NULLNOD && epnt->node1b!=nb) nomatch=0;     
       if (nc!=NULLNOD && epnt->node1c!=nc) nomatch=0;     

       if (!nomatch) {
       nomatch = 1;
         if (na!=NULLNOD && epnt->node2a!=na) nomatch=0;     
         if (nb!=NULLNOD && epnt->node2b!=nb) nomatch=0;     
         if (nc!=NULLNOD && epnt->node2c!=nc) nomatch=0;     
	}
       if (nomatch) 
           if (exceptype && epnt->ctype!=exceptype) nomatch=0;
     }
     else if (exceptype && epnt->ctype==exceptype) nomatch=1;

     if (nomatch) continue;
     ncdraw(epnt,color,dscale,hide);
   }

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr," end\n");
#endif
}

/*------------------------------------*/

void ncdispe (int elemnum, int color, double dscale, int hide)

/* Draw a neural element, given its element number.
*/

{
   elem *epnt;

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr,"ncdispe start...\n");
#endif


   for (epnt=elempnt; epnt; epnt=epnt->next) {	/* search all elems */
      if (epnt->elnum == elemnum) break;
      if (epnt==NULL) {
         fprintf (stderr,"ncdispe: can't find element %d\n",elemnum);
         return;  
      }
      ncdraw(epnt,color,dscale,hide);
      if (!epnt) break;
   }

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr," end\n");
#endif
}
/*------------------------------------*/

void ncdrnod (int n1a, int n1b, int n1c, 
	int elemtype, int exceptype, int na, int nb, int nc, 
	int color, double dscale)
                                                       
/* Draw neural circuit elements that connect to 
   both node1 and node2. If any of the node
   numbers == NULLNOD, then ignore this dimension.
   If "excl"=1, display only elements that connect 
   exclusively to given node.
 */

{
   node *npnt;
   int match,nomatch;

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		 fprintf (stderr,"ncdrnod n1a %d n1b %d n1c %d\n",n1a,n1b,n1c);
#endif

/* fprintf (stderr,"ncdrnod n1a %d n1b %d n1c %d\n", n1a,n1b,n1c);/* */

 for (npnt=nodepnt; npnt; npnt=npnt->next) {    /* search all nodes */

#ifdef DEBUG
  if ((debug & 2) && (debugz & 2))
		fprintf (stderr,"ncdrnod %d start...",npnt->nodenm1);
#endif
     match = 1;
     if (n1a != NULLNOD) 			/* check node 1 first */	
        if (npnt->nodenm1!=n1a) match=0;
     if (n1b != NULLNOD)
        if (npnt->nodenm2!=n1b) match=0;
     if (n1c != NULLNOD)
        if (npnt->nodenm3!=n1c) match=0;

     if (!match) continue;
     nomatch = 0;
     if (na!=NULLNOD || nb!=NULLNOD || nc!=NULLNOD) {
       nomatch = 1;
       if (na!=NULLNOD && npnt->nodenm1!=na) nomatch=0;     
       if (nb!=NULLNOD && npnt->nodenm2!=nb) nomatch=0;     
       if (nc!=NULLNOD && npnt->nodenm3!=nc) nomatch=0;     

       if (nomatch) 
           if (exceptype && npnt->ctype!=exceptype) nomatch=0;
      }
      else if (exceptype && npnt->ctype==exceptype) nomatch=1;

     if (nomatch) continue;
     drnode(npnt,dscale,color,ncmat);
   }

#ifdef DEBUG
  if ((debug & 1) && (debugz & 2))
		fprintf (stderr,"ncdrnod end\n");
#endif
}

/*------------------------------------*/

void dispstim(double stime, double dia)
{
#define RTIMERND   0.00001		/* down-round value */
#define DSTIME     0.001		/* time incr for reading in stims */
#define NUMCOLS 8			/* number of colors for stim disp. */

   recstim *rspnt,*rslast,*rsnext;
   recep *rpnt,*rlast,*rnext;
   elem *epnt;
   static double start,stop;
   double dlookup(int pigm, double wavel);
   double max_flux, flux_val,norm_val;
   double x,y,tdia,width,dist;
   int i,color_val,col_indx,fill=0;
   static int color_lut[NUMCOLS]=
		{BLUE,GREEN,CYAN,MAGENTA,BROWN,YELLOW,RED,WHITE};

#ifdef DEBUG
  if (debug & 1)  fprintf (stderr,"dispstim\n");
#endif

 for (epnt=elempnt; epnt; epnt=epnt->next) {  	  /* check all elements */
    if (epnt->ctype == ROD || epnt->ctype == CONE) {
     makr((photrec*)epnt, PHOTREC);		/* make a dummy photoreceptor */
    }
  }

	/* now read in the stimulus events in increments of DSTIME: */

for (start=0.0; start<=stime; start+=DSTIME) {
   stop = start + DSTIME;

   if (!readstim(stop)) {  /* read new receptor inputs */
         fprintf (stderr,"dispstim: invalid stimulus at time %g\n",stime);
	 execerror ("Missing stimulus: "," stopping... ");
   }   

	/* check new stimulus events and possibly use and erase them */

  for (rslast=rspnt=recspnt; rspnt; ) {

/*  fprintf (stderr,"stim num %d type '%c' time %g rtime %g start %g stop %g\n",
              rspnt->recnm1, rspnt->ctype,rspnt->time, stime,start,stop);  /* */
 
   if ((rspnt->time >= start) && (rspnt->time <stop)) {
      switch (rspnt->ctype) {

       case 'a':                /* absolute */
          if (findrecep (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
					&rpnt, "dispstim")) {
             rpnt->flux = rspnt->val;
	  }
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"abs stim %d %g %g\n",
                                        rspnt->recnm1,rspnt->val,rspnt->wavel);
#endif
         break;

     case 'd':                /* delta */
          if (findrecep (rspnt->recnm1, rspnt->recnm2, rspnt->recnm3,
                                        &rpnt, "dispstim")) {
             rpnt->flux += rspnt->val;
          }
#ifdef DEBUG
          if (debug & 1)  fprintf (stderr,"del stim %d %g %g\n",
                                        rspnt->recnm1,rspnt->val,rspnt->wavel);
#endif
         break;
      } /* switch */	

		/* now delete the stimulus event if we've used it: */

      rsnext = rspnt->next;            /* delete and patch pointers */
      if (rspnt == recspnt) recspnt = rsnext;
      else rslast->next = rsnext;
      if (recsend==rspnt) {
	 if (recspnt==NULL) recsend=NULL;
         else recsend = rslast;
      }
      delrstim(rspnt);                /* delete stimulus after use */
      rspnt = rsnext;
   }        /* if */

   else {   /* no stimulus here */
     rslast = rspnt;
     rspnt = rspnt->next;
   }
  }  /* for (rslast=rspnt;;) */
  
}  /* for (start=;;) */

   for (max_flux=0,rpnt=recpnt; rpnt; rpnt=(recep*)rpnt->next){
   	flux_val = rpnt->flux;
	if (flux_val > max_flux)
		max_flux = flux_val;
   }
   if (max_flux == 0)  max_flux = 1;
   for (rpnt=recpnt; rpnt; rpnt=(recep*)rpnt->next){
	norm_val = rpnt->flux / max_flux;
	col_indx = (int)(norm_val * (NUMCOLS-1) + 0.5);
	color_val = color_lut[col_indx];
        if (! loccheck (rpnt->xloc,rpnt->yloc,0.0)) {/* check if valid loc */
	  execerror ("nc: dispstim: invalid photoreceptor location","stopping.");
        }
 	drsphere (rpnt->xloc,rpnt->yloc,0.0,dia,1.0,color_val,0,ncmat,fill); 
 	
   }
		/* now delete the dummy photoreceptors we made above */

   for (rlast=rpnt=recpnt; rpnt; ) {
	rnext = (recep*)rpnt->next;
	if (rpnt->ctype==PHOTREC) {		/* delete dummy photorec */
		if (rpnt==recpnt) recpnt = rnext;
		else rlast->next = rnext;
		free (rpnt);	
	}	
	else rlast = rpnt;
	rpnt = rnext;
   }

#define COLBARLEN 0.40
#define XBAR  0.30
#define YBAR  0.95

  tdia = 0.02;
  dist = COLBARLEN/NUMCOLS;
  x = XBAR; y = YBAR;

   gframe ("Col_bar");
   gorigin (x,y);
   for (width= -tdia/2.0; width<tdia/2.0; width+= linewidth) {
     gmove (0, width);
     for (i=0; i<NUMCOLS; i++){
	gpen(color_lut[i]);
        gdraw ((i+1)*dist, width);
     }
  }
  gmove(-0.1,-0.002);
  gtext("MIN");
  gmove(COLBARLEN+0.05,-0.002);
  gtext("MAX");
  ppurge();
  
  gframe ("..");
  grmframe ("Col_bar");
}
/*------------------------------------*/

int vcolor (elem *epnt, int dcolor)

/* find the color of an element from its voltage. */

{
    double range,v;
    node *npnt;
    comp *cpnt;
    int color,vfound;

  vfound = 0;
  range = plmax - plmin;
  range = abs(range);
  if (npnt=epnt->nodp1)
    if (cpnt=npnt->comptr) {
       v = cpnt->v;
       vfound = 1;
     }
  if (!vfound) v = plmin;

  color = (int) ((v-plmin) / range * NCOLOR);
  color = limit (color,NCOLOR,0);
  return color;
}

/*------------------------------------*/

void ncdraw (elem *epnt, int ecolor, double dscale, int hide)

/* draw a neural element on the screen */

{
   node *np1,*np2;
   conlst *lpnt;
   elem *nepnt;
   double n1dia,n2dia;
   double x1,y1,z1,x2,y2,z2;
   int dcolor;

#ifdef DEBUG
  if ((debug & 2) && (debugz & 2))
		fprintf (stderr,"ncdraw start...\n");
#endif

  if (!epnt) return;

  np1 = epnt->nodp1;
  np2 = epnt->nodp2;

#ifdef DEBUG
  if ((debug & 2) && (debugz & 2)) {
	fprintf (stderr,"ncdraw element type: '%s' ",findsym(epnt->ctype));
	fprintf (stderr,"dia %g dscale %g color %d\n",((cable *)epnt)->dia,
			dscale,ecolor);
  }
#endif

  if (!np1) {
#ifdef DEBUG
  if ((debug & 2) && (debugz & 2))
		fprintf (stderr,"ncdraw: returning, nodp1 not set\n");
#endif
     return;
     }

	/* find diameter of any spheres connected to node: */

  if (np1) for (n1dia=0.0,lpnt=np1->elemlst; lpnt; lpnt=lpnt->next) {
      nepnt = ((elem *)lpnt->conpnt);
      if (nepnt->ctype==SPHERE) {
        if (n1dia<((sphere*)epnt)->dia) n1dia = ((sphere*)nepnt)->dia;
      } 
    }
  if (np2) for (n2dia=0.0,lpnt=np2->elemlst; lpnt; lpnt=lpnt->next) {
      nepnt = ((elem *)lpnt->conpnt);
      if (nepnt->ctype==SPHERE) {
        if (n2dia<((sphere*)epnt)->dia) n2dia = ((sphere*)nepnt)->dia;
      } 
    }

  if (np1) { x1 = np1->xloc; y1 = np1->yloc; z1 = np1->zloc; }
  if (np2) { x2 = np2->xloc; y2 = np2->yloc; z2 = np2->zloc; }

	if (!loccheck(x1,y1,0.0))       		
	printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);

  if (ecolor >= VCOLOR) dcolor=vcolor(epnt,ecolor);/* set color from voltage */
  else dcolor = ecolor;

 if (disp_ray) {

  switch (epnt->ctype) {

   case CABLE:	if (!loccheck(x2,y2,0.0)) 
		printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);
   		raycable (x1,y1,z1,x2,y2,z2, ((cable*)epnt)->dia, dscale,
			n1dia,n2dia,dcolor,hide,ncmatc);
		break;

   case SPHERE:	raysphere (x1,y1,z1, ((sphere*)epnt)->dia, dscale,
			dcolor,hide,ncmatc,1);
		break;

   case SYNAPSE: if (!loccheck(x2,y2,0.0)) 
       		printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);
   		raysynap (x1,y1,z1,x2,y2,z2, (synapse *)epnt, dscale,n1dia,n2dia,
			dcolor,hide,ncmatc);
		 break;

   case RESISTOR:
   case AXIALRES:
   case GJ:	if (!loccheck(x2,y2,0.0)) 
       		printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);
   		rayconn (x1,y1,z1,x2,y2,z2,epnt->ctype,dscale,
			n1dia,n2dia,dcolor,hide,ncmatc);
		 break;

   case ROD:	 
   case CONE:   rayphotrec (x1,y1,z1,
		   (photrec *)epnt,dscale,n1dia,n2dia,dcolor,hide,ncmatc);
		 break;

   case GNDCAP:
   case LOAD:	 rayload (x1,y1,z1,
			  epnt->ctype,n1dia,n2dia,dcolor,hide,ncmatc);
		break;
   }
  }  
 else /* display 2D */

  switch (epnt->ctype) {

   case CABLE:	if (!loccheck(x2,y2,0.0)) 
		printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);
   		drcable (x1,y1,z1,x2,y2,z2, ((cable*)epnt)->dia, dscale,
			n1dia,n2dia,dcolor,hide,ncmat);
		break;

   case SPHERE:	drsphere (x1,y1,z1, ((sphere*)epnt)->dia, dscale,
			dcolor,hide,ncmat,1);
		break;

   case SYNAPSE: if (!loccheck(x2,y2,0.0)) 
       		printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);
   		drsynap (x1,y1,z1,x2,y2,z2, (synapse *)epnt, dscale,n1dia,n2dia,
			dcolor,hide,ncmat);
		 break;

   case RESISTOR:
   case AXIALRES:
   case GJ:	if (!loccheck(x2,y2,0.0)) 
       		printf ("# Invalid display location for %s # %d\n",
				findsym(epnt->ctype), epnt->elnum);
   		drconn (x1,y1,z1,x2,y2,z2,epnt->ctype,dscale,
			n1dia,n2dia,dcolor,hide,ncmat);
		 break;

   case ROD:	 
   case CONE:   drphotrec (x1,y1,z1,
		   (photrec *)epnt,dscale,n1dia,n2dia,dcolor,hide,ncmat);
		 break;

   case GNDCAP:
   case LOAD:	 drload (x1,y1,z1,
			  epnt->ctype,n1dia,n2dia,dcolor,hide,ncmat);
		break;
  }

  ppurge();		/* flush the output stream */

#ifdef DEBUG
  if ((debug & 2) && (debugz & 2))
		fprintf (stderr," ncdraw end\n");
#endif
}

/*--------------------------------------*/

void ppurge (void)

{
  fflush (stdout);
}

/*--------------------------------------*/

void drcomp(double xloc, double yloc, double zloc, double dia)

/* draw a compartment */

{
  drsphere (xloc,yloc,zloc,dia,1.0,BLUE,0,ncmat,0);
  ppurge();
}

/*--------------------------------------*/

void drnode(node *npnt, double dscale, int color, double (*mat)[4])

/* display node numbers in 3D location */

{
    double x,y,z,tx,ty,tz,xoffs,charsiz;
    static char numbuf[20];

  x = npnt->xloc;
  y = npnt->yloc;
  z = npnt->zloc;
  if (z==NULLNOD) z = 0.0;
  transf (x,y,z,&tx,&ty,&tz,mat); 
  if (color) gpen (color);
  else       gpen (WHITE);
  if (npnt->nodenm3 != NULLNOD)
      sprintf (numbuf,"%d %d %d",npnt->nodenm1,npnt->nodenm2,npnt->nodenm3);
  else if (npnt->nodenm2 != NULLNOD)
      sprintf (numbuf,"%d %d",npnt->nodenm1,npnt->nodenm2);
  else
      sprintf (numbuf,"%d",npnt->nodenm1);
  if (dscale<0) dscale = -dscale;
  charsiz = .015 * dscale;
  xoffs = strlen(numbuf) / 2.0;		/* find center of number */
  xoffs *= charsiz * .7;
  gmove (tx-xoffs, ty - charsiz*1.1);	/* label node numbers */
  gcwidth (charsiz);
  gtext (numbuf);
  ppurge();
}

/*--------------------------------------*/

void drcable (double x1, double y1, double z1, 
		double x2, double y2, double z2, 
		double dia, double dscale, double n1dia, double n2dia, 
		int color, int hide, double (*mat)[4])
                
/*
   Draw a cable element in 2D projection of 3D rotation.  First, 
transform the element's 2 end points by a 3D rotation matrix, and
then draw the 2D projection of the element in a subframe, in a 
standard position (0,0) and orientation (horizontal).  The subframe is 
translated ("origin()") to one endpoint of the element,
and rotated ("rotate()") by an amount equal to the original
element's 3D orientation angle projected into 2D. 

*/

{
    double tx1,ty1,tz1, tx2,ty2,tz2;
    double width,tdia;
    double dx,dy,dist,theta,r1,r2;
    int fill;

  if (z1==NULLNOD) z1 = 0.0;
  if (z2==NULLNOD) z2 = 0.0;
  if (dscale<0) dscale = -dscale;
  tdia = dia * dscale * xyscal;
  r1 = n1dia * xyscal / 2.0;
  r2 = n2dia * xyscal / 2.0;
  transf (x1,y1,z1,&tx1,&ty1,&tz1,mat);
  transf (x2,y2,z2,&tx2,&ty2,&tz2,mat);
  dy = ty2-ty1;
  dx = tx2-tx1;
  dist = sqrt(dx*dx + dy*dy);
  if (ncabs(dx)<1e-20) dx = .0001;
  theta = atan(dy/dx);
  if (dx<0) theta += PI;
  gframe ("cable");
  gorigin (tx1,ty1);
  grotate (theta/DEG);  
  if (color!=NULLNOD) gpen (color);
  else       gpen (GREEN);
  if (hide) {
    if (r1+r2 < dist) {
      gmove (r1,     -tdia/2.0);
      gdraw (r1,      tdia/2.0);
      gdraw (dist-r2, tdia/2.0);
      gdraw (dist-r2,-tdia/2.0);
      gdraw (r1,     -tdia/2.0);
    } 
  }
  else {
     double x1,y1,x2,y2;

   if (r1+r2 < dist) {          /* if cable not inside sphere */
     width = tdia / 2.0;
     fill = 1;
     x1 = r1;
     y1 = -width;
     x2 = dist-r2;
     y2 = width;
     grect(x1,y1,x2,y1,x2,y2,x1,y2,fill);
   }
  }
  gframe ("..");
  grmframe ("cable");
/*
  fprintf (stderr,"x1 %g y1 %g x2 %g y2 %g theta %g\n",tx1,ty1,tx2,ty2,theta);
*/
}

/*--------------------------------------*/

void drsphere (double x, double y, double z, double dia, double dscale, 
	int color, int hide, double (*mat)[4],int fill)
{
    double tx,ty,tz;

  if (dscale<0) dscale = -dscale;
  if (z==NULLNOD) z = 0.0;
  transf (x,y,z,&tx,&ty,&tz,mat); 
  if (color!=NULLNOD) gpen (color);
  else       gpen (BLUE);
  gmove (tx, ty);
  gcirc (dia*dscale*xyscal/2.0,fill);
}

/*--------------------------------------*/

/*  Don't use this subroutine, assume graphics driver
    has its own circle and rectangle routines. */

/*
void ncdrcirc (double x, double y, double rad, int fill)
{ 
   double theta, chord;
   double plXpos, plYpos, sin(double t), cos(double t);
   
#define rmove(dx,dy)	gmove(plXpos + (dx), plYpos + (dy))
#define rpmove(dr,t)	rmove((dr)*cos((double)t), (dr)*sin((double)t))
#define rdraw(dx,dy)	gdraw(plXpos + (dx), plYpos + (dy))
#define rpdraw(dr,t)	rdraw((dr)*cos((double)t), (dr)*sin((double)t))

 gmove (x,y);
 plXpos = x;
 plYpos = y;
 rpmove (rad,-PI/2);
 if      (rad > .2) chord = .05;
 else if (rad > .1) chord = .1;
 else               chord = .2;
 rpmove (rad*chord/2,-PI);
 for (theta=0.0; theta < PI*2; theta += chord)
   rpdraw (rad*chord,theta);

}
*/

/*--------------------------------------*/

void drsynap (double x1, double y1, double z1, 
	double x2, double y2, double z2, synapse* epnt,
	double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4])
{
    double dx,dy,tx1,ty1,tz1, tx2,ty2,tz2,theta,dist;

  if (color==NULLNOD) color=RED;
  if (z1==NULLNOD) z1 = 0.0;
  if (z2==NULLNOD) z2 = 0.0;
  transf (x1,y1,z1,&tx1,&ty1,&tz1,mat);
  transf (x2,y2,z2,&tx2,&ty2,&tz2,mat);
  dy = ty2-ty1;
  dx = tx2-tx1;
  dist = sqrt(dx*dx + dy*dy) / xyscal;
  if (ncabs(dx)<1e-8) dx = .0001;
  theta = atan(dy/dx);
  if (dx<0) theta += PI;
  gframe ("syn");
  gorigin (tx1,ty1);
  gsize (xyscal);
  grotate (theta/DEG);  
  gmove (0,0);
  (*syd)(color, epnt->vrev, dscale, dist, hide);
  gframe ("..");
  grmframe ("syn");
}

/*--------------------------------------*/

void syn_draw (int color, double vrev, double dscale, double length, int hide)
{
    int fill=0;
    double dia;

  dia = 1.0 * dscale;				/* draw 1 um circle with line */
  if (dia < 0) dia = -dia;
  if (vrev < -.04) color = RED;
  else             color = CYAN;
  gpen (color);
  gmove (length/2.0,0.0);
  gcirc (dia/2.0,fill);
  gmove (0,0);
  gdraw (length,0);
}

/*--------------------------------------*/

void drcconn (double x1, double y1, double z1, double x2, double y2, double z2)
{
   drconn (x1,y1,z1,x2,y2,z2, AXIALRES, 1.0, 0.0, 0.0, YELLOW, 0, ncmat);
}

/*--------------------------------------*/

void drconn (double x1, double y1, double z1, 
	double x2, double y2, double z2, 
	int ctype, double dscale, double n1dia, double n2dia, 
	int color, int hide, double (*mat)[4])
{
    double dx,dy, tx1,ty1,tz1, tx2,ty2,tz2, dist, theta;

  if (z1==NULLNOD) z1 = 0.0;
  if (z2==NULLNOD) z2 = 0.0;
  if (color==NULLNOD) color = YELLOW;
  transf (x1,y1,z1,&tx1,&ty1,&tz1,mat);
  transf (x2,y2,z2,&tx2,&ty2,&tz2,mat);
  dy = ty2-ty1;
  dx = tx2-tx1;
  dist = sqrt(dx*dx + dy*dy) / xyscal;
  if (ncabs(dx)<1e-8) dx = .0001;
  theta = atan(dy/dx);
  if (dx<0) theta += PI;
  gframe ("gj");
  gorigin (tx1,ty1);
  gsize (xyscal);
  grotate (theta/DEG);  
  gmove (0,0);

  switch (ctype){

  case GJ: (*gjd) (color, dist, dscale, hide);
    break;

  case RESISTOR:
  case AXIALRES:
    gmove (tx1,ty1);
    gdraw (tx2,ty2);
    break;
  } 
  gframe ("..");
  grmframe ("gj");
}

/*--------------------------------------*/

void  gj_draw (int color, double length, double dscale, int hide)
{
   double dia;
   int fill;

 fill = 0;
 gpen (color);
 if (dscale>=0) {
   dia = 1.0;
   gmove (length/2.0, 0);
   gcirc (dia/2.0,fill);
   gmove (0,0);
   gdraw (length,0);
 }
 else {
   dia = 1.0 * -dscale;
   gmove (length/2, 0);
   gcirc (dia/2.0,fill);
 }
}

/*--------------------------------------*/

void drphotrec (double x, double y, double z, 
		     photrec *epnt, double dscale,
		     double n1dia,double n2dia,
		     int color, int hide, double (*mat)[4])

/* see "drcable()" above for explanation of
   frame, origin() and rotation.  The screen
   rotation in this case is derived from translating
   two endpoints of a line drawn along the z axis.
   The subframe "photrec" is rotated in relation to
   the root frame, so that photoreceptors are always
   drawn (created) along the horizontal (X) axis.
 */
 
{
    double tx1,ty1,tz1,tx2,ty2,tz2;
    double dia,len,dist,dx,dy,theta;
    int ctype,pigm;
    static int color_trans[NUMREC] = {MAGENTA,RED,GREEN,BLUE};

  ctype = epnt->ctype;
  if (epnt->dia==NULLNOD) dia = 2 * dscale;
  else dia = epnt->dia*dscale;
  if (dia < 0) dia = -dia;
  if (color==NULLNOD) {
      if ((pigm=(int)(epnt->pigm)) == NULLNOD)
         pigm = (ctype==CONE ? 1 : 0);
      pigm = limit(pigm,0,NUMREC);
      color= color_trans[pigm];
  }
  len = 1.0;				/* length of photorec in um */
  if (dia<=0) dia = 2.0;
  if (z==NULLNOD) z = 0.0;
  transf (x,y,z,&tx1,&ty1,&tz1,mat);
  transf (x,y,z+len,&tx2,&ty2,&tz2,mat);
  dy = ty2-ty1;
  dx = tx2-tx1;
  dist = sqrt(dx*dx + dy*dy) / xyscal;
  if (ncabs(dx)<1e-8) dx = .0001;
  theta = atan(dy/dx);
  if (dx<0) theta += PI;
  gframe ("photorec");
  gorigin (tx1,ty1);
  gsize (xyscal);
  grotate (theta/DEG);  
  gmove (0,0);
  (*phd)(ctype,pigm,color,dia,dist,hide);
  gframe ("..");
  grmframe ("photorec");
}

/*--------------------------------------*/

void phot_draw(int ctype, int pigm, int color, 
		double dia, double foreshorten, int hide)

/* Default subroutine to draw a photoreceptor */
/* Given that the pen has already been set at the origin (0,0),
   draw the photoreceptor's icon.  Icon is oriented horizontally
   to the right (zero degrees).
*/

{
  double width,dist;

  dist = foreshorten * 10.0;		/* photoreceptor extends 10 um */
  gpen (color);
  switch (ctype) {

  case ROD:
     width = dia*.75;
     if (hide) {
        gdraw (dist*.25,0);
        gmove (dist*.25,-width*.5);
        gdraw (dist*.25,width*.5);
        gdraw (dist,width*.5);
        gdraw (dist,-width*.5);
        gdraw (dist*.25,-width*.5);
     }
     else {
          int fill=1;
          double x1,y1,x2,y2;

       gdraw (dist*.25,0);
       x1 = dist*.25;
       y1 = -width*.5;
       x2 = dist;
       y2 = width*.5;
       grect(x1,y1,x2,y1,x2,y2,x1,y2,fill);
     }
     break;

  case CONE:
	width = dia;
	gdraw (dist*.25,0);
	gmove (dist*.25,width/2.);
	gdraw (dist*.25,-width/2.);
	gdraw (dist, 0.0);
	gdraw (dist*.25,width/2.);
     break;

  }	/* switch */
}

/*--------------------------------------*/

void drload (double x, double y, double z, int ctype, 
	double n1dia, double n2dia, int color, int hide, double (*mat)[4])
                              
/* see "drphotrec()" above for explanation of
   frame, origin() and rotation. */
 
{
    double tx1,ty1,tz1,tx2,ty2,tz2;
    double len,dist,dx,dy,theta;

  len = 10.;				/* length of load in um */
  if (z==NULLNOD) z = 0.0;
  transf (x,y,z,&tx1,&ty1,&tz1,mat);
  transf (x,y,z+len,&tx2,&ty2,&tz2,mat);
  dy = ty2-ty1;
  dx = tx2-tx1;
  dist = sqrt(dx*dx + dy*dy);
  if (ncabs(dx)<1e-8) dx = .0001;
  theta = atan(dy/dx);
  if (dx<0) theta += PI;
  gframe ("load");
  gorigin (tx1,ty1);
  grotate (theta/DEG);  
  if (color!=NULLNOD) gpen (color);
  else       gpen (BLUE);
  switch (ctype) {

  case LOAD:
        gmove (0, 0);
        gdraw (dist*.25,0);
        gmove (dist*.25,-dist*.15);
        gdraw (dist*.25,dist*.15);
        gdraw (dist*.75,dist*.15);
        gdraw (dist*.75,-dist*.15);
        gdraw (dist*.25,-dist*.15);
        gmove (dist*.75,0);
        gdraw (dist, 0);
     break;

  case GNDCAP:
     break;

  }	/* switch */
	
  gframe ("..");
  grmframe ("load");

}

/*--------------------------------------*/

void drcalib(double x, double y, double len, double size, int color)
{
    double dist;
    static char numbuf[20];

 if (disp_ray) raycalib (x, y, len, size, color);
  else {
   dist = len * xyscal;
   if (color!=NULLNOD) gpen(color);
   else       gpen(CYAN);
   gmove (x,y);
   gdraw (x-dist,y);
   gmove (x-dist/2-.03,y-.03);
   sprintf (numbuf,"%g um",len);  
   gtext (numbuf);  
   gpurge();
  }
}

/*--------------------------------------*/

int loccheck (double x, double y, double z)

/* check whether point is at valid location */

{
   int locok;

  locok = 1;
  if (x==NULLNOD) locok = 0;
  if (y==NULLNOD) locok = 0;
  if (z==NULLNOD) locok = 0;
  return (locok);
}

/*--------------------------------------*/

void set_icons(void)

{
  Symbol *f,*lookup(char *name);

  if ((f=lookup("phot_dr"))==NULL) phd = phot_draw;
  else {
    phot_drpnt = f;	/* set pointer to user function */
    phd = phot_dr;      /* set pointer to call user function */
  }
  if ((f=lookup("synapse_dr"))==NULL) syd = syn_draw;
  else {
    syn_drpnt = f;	/* set pointer to user function */
    syd = synapse_dr;      /* set pointer to call user function */
  }
  if ((f=lookup("gapjunc_dr"))==NULL) gjd = gj_draw;
  else {
    gapj_drpnt = f;	/* set pointer to user function */
    gjd = gapjunc_dr;      /* set pointer to call user function */
  }
}

/*--------------------------------------*/

void phot_dr (int type, int pigm, int color, double dia, 
		double foreshorten, int hide)

/* Call a user-defined subroutine for drawing a photoreceptor icon. */

{
   double x=0;

   callfunc8 (phot_drpnt,5,(double)pigm, (double)color, dia, foreshorten,
   		(double)hide, x,x,x);
}

/*--------------------------------------*/

void synapse_dr (int color, double vrev, double length, double dia, int hide)

/* Call a user-defined subroutine for drawing a photoreceptor icon. */

{
   double x=0;

   callfunc8 (syn_drpnt,5, (double)color, vrev, length, dia, (double)hide,
			x,x,x);
}

/*--------------------------------------*/

void gapjunc_dr (int color, double length, double dscale, int hide)

/* Call a user-defined subroutine for drawing a photoreceptor icon. */

{
   double x=0;

   callfunc8 (gapj_drpnt,4,(double)color, length, dscale,  (double)hide,
		x,x,x,x);
}

/*--------------------------------------*/
