/* modcode.c */

/* This module contains all subroutines called by "nc.y"
   that create neural element data structures.  This
   module is compiled for use by both "nc" and "stim".
   For the "stim" version, conditional compilation statements
   remove or modify some of the subroutines so that they
   do only things associated with making and controlling
   stimuli.  Thus, both "stim" and "nc" interpret the same
   language but "stim" ignores all the "compartment" stuff and
   deals only with neural elements and stimuli.  After
   compilation, the "stim" version is "mv"ed to "stimcode.o.

     An old version of stimcode exists, however this is not
   currently used because of the difficulty of modifying it
   "in parallel" with modifications to "modcode.c".  */

#include "nc.h"
#include "y.tab.h"
#include "adef.h"
#include "ncelem.h"
#include "ncomp.h"
#include "ncsub.h"
#include "stim.h"

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>

#ifdef __cplusplus
}
#endif

#define DEBUG

#ifndef XSTIM
#define XMOD			/* define XMOD when not XSTIM */
#endif

/* Normally stim doesn't need to know about neural elements.  
   However in a complex circuit, it is possible that the arrangement
   of photoreceptors and their stimuli need to be determined from
   information only available from some already-defined neural
   elements.  In this case, comment out the #define XSTIMM below,
   and recompile "stim".
*/

#ifdef XSTIM
#define XSTIMM		/* define XSTIMM when stim should ignore */
#endif				/*    neural elements for speed */
		
/* variables for stim */

int recepnode = 0;                      /* receptor node from conn() */
int recepn2 = 0;
int recepn3 = 0;
int stimfflg = 0;		/* =1 -> stimulus comes from file */


/* interpreter innards: */

#define push(d) *stackp++ = (d)		/*  */
#define popm()  *--stackp		/* function pop still needed */

/* #define popm() pop()			/*  */

#define NSTACK 256

extern char *progname;
extern char *runfile;
extern int vidmode;
extern int debug;
extern int debugz;
extern int setdebug;
extern int setdebgz;
extern int setdisp;
extern int disp;
extern int disp_ray;
extern int stimhlp;			/* in stimsub.c */
extern int nocond;			/* no condense, in ncsub */
extern int stimelem;		/* = 1 => stim understands neural elements */

extern int stopping;
extern int returning;
extern int breaking;
extern int continuing;

extern  datum   *stackp;

extern	Inst	*progp;		/* next free spot for code generation */
extern	Inst	*pc;		/* program counter during execution */
extern	Inst	*progbase;	/* start of curent subprogram */

elem *elpnt;			/* current element pointer */

int curelnum=0;			/* current element being read in */
extern node *nodepnt;		/* pointer to node list */
extern recep *recpnt;		/* pointer to receptor list */
extern recnod *reclist;		/* pointer to receptor stimulus list */
extern elem *elempnt;		/* pointer to current element */
extern int cumelem;
extern int cumnode;
extern int cumrec;

extern double xtime;		/* time variable set in "ncsub" (control.h) */
extern Symbol *timeptr;		/* pointer to "xtime" for setting time */

#ifdef XMOD		/* variables for modcode */

#define STIMCODE 0

extern plotfr plotnod[PLOTNODSIZ];  /* holds nodes to be recorded */
extern int numplots;		/* number of plots to be displayed */
extern int runyet;

			/* functions for modcode */

recstim *makvstim(double start, int nodnm1, int nodnm2, int nodnm3, 
			double value, char *action);
recstim *makrstim(double start, int recnm1, int recnm2, int recnm3, 
		double inten, double wavel, char *action);
int inrect(double x, double y, double maxx, double minx, 
			double maxy, double miny);
int incirc(double x, double y, double circx, 
			double circy, double rad);

void dcomp(int n1a, int n1b, int n1c, int elemtype, int exceptype, 
	int na, int nb, int nc, int color, double dscale, int hide, int excl);
void ncdrnod (int n1a, int n1b, int n1c, int elemtype, int exceptype, 
	int na, int nb, int nc, int color, double dscale);
void ncdisp(int n1a, int n1b, int n1c, int elemtype, int exceptype, 
	int na, int nb, int nc, int color, double dscale, int hide, int excl);
void ncdispc(int n1a, int n1b, int n1c, int n2a, int n2b, int n2c, 
	int elemtype, int exceptype, int na, int nb, int nc, 
	int color, double dscale, int hide);
void ncdispn(int n1a, int n1b, int n1c, int n2a, int n2b, int n2c, 
	int elemtype, int exceptype, int na, int nb, int nc, 
	int color, double dscale, int hide, int excl);
void ncdispe (int elemnum, int color, double dscale, int hide);
void drcalib (double x, double y, double length, double size, int color);
void setrot (double xrot, double yrot, double zrot, 
	double xcent, double ycent, double zcent, 
	double rxcent, double rycent, double rzcent, double scal);
void dispstim(double stime,double dscale);
void set_icons(void);
void initray (double xrot, double yrot, double zrot,
        double xcent, double ycent, double zcent,
        double rxcent, double rycent, double rzcent, double scal);

#else			/* variables for stimcode */

#define STIMCODE 1

plotfr plotnod[PLOTNODSIZ];  	/* holds nodes to be recorded */
int numplots;			/* number of plots to be displayed */
int runyet;
extern int blursize;		/* radius of blur array from "stimsub.c" */

			/* functions for stimcode */
void stcomment (void);
void makblur(double rad, double scale);
void findarrsiz(double *xmax,double *xmin,double *ymax,double *ymin);
int initc(int num, int size);
void recback(int array, double inten, double wavel);
void abslist(double ratio, double time);
void makrect(int arr, double width, double length, double xoff, double yoff, 
		double scale, double inten, double wavel);
void recpat(int array, double xoff, double yoff, double scale);
void stimlist(double delratio, double time);
void makspot(int arr, double dia, double xoff, double yoff, 
		double scale, double inten, double wavel);
void maksine(int arr, double period, double xmin, double xmax, 
		double ymin, double ymax, double toff, double orient,
		double xoff, double yoff, double scale, 
		double inten, double contrast, double wavel);

#ifdef __cplusplus
extern "C" {
#endif

void mktemp(char *f);

#ifdef __cplusplus
}
#endif

#endif				/* stimcode */


int stimcode=STIMCODE;		/* = 1 -> says we're running "stim" */
double xrot,yrot,zrot;
double dxcent=LARGENUM,dycent=LARGENUM,dzcent=LARGENUM;
double rxcent=0,rycent=0,rzcent=0;
double dsize = 200;

char *stfile=0;			/* stimulus file for plotinit */

FILE *stimin=0;			/* stimulus file */
FILE *stimout=stdout;           /* stimulus output file, used only by stim */
static char stimfile[80]={0};

#ifdef __cplusplus
extern "C" {
#endif

double sqrt(double);
double fabs(double);
double sin(double);
void free(...);
#include "gr.h"

#ifdef __cplusplus
}
#endif

node *maknod(short int nodea, short int nodeb, short int nodec);
elem *makelem(int etype, elem *epnt);
elem *tmpelem();
double *darr2();
double setvar(char *str);
double mindist(double dist1, double dist2, double dist3);
char *prnode(int n1, int n2, int n3);
void execerror(char *s, char *t);
void ncleanup();
Symbol *getvar(double **val, short int **typ, short int **vtyp);
void checknod(elem *elpnt);
void checkelemnode(elem *elpnt);
node *findnode(short int node1a, short int node1b, short int node1c, char *s);
void setnodlst(node *npnt, elem *cpnt);
void unsetnodlst(node *npnt, elem *cpnt);
char *prnum(char *adj, double csiz, char *fmt, double v1, double v2, double v3);
int gausnn (double mean, double stdev, double density, double ms, 
	double framex, double framey, double xcent, double ycent, 
	int numcells, float **xarr, float **yarr, 
	int filout, int textfl, int printfl);
double *darr2(Symbol *sp, int narg);
void mplot (double y, double x, int n, int i);
void varcopy();
void plotinit(int numplots);
void plotrst(int numplots);
void plotpen (int val, int i);
void plotchar (int val,int lines, int i);
void plotcsiz (double val, int numplots);
void findconnect(void);
void initcomp();
void condense();
void execute(Inst *pc);
int forexec(Inst *pc);
void maklst(conlst **head, conn *cpnt);
void dellst(conlst **head, conlst *lpnt);
void initchan();
void actrun(double time);

/*------------------------------------------*/

/* Always remember to check a new subroutine added to "modcode.c" for
   its effect onr "stim". The stim subroutines should be called in exactly 
   the same way as their modcode counterparts (exception: stimulus subroutines),
   and they should always leave the stack (popm) and program counter
   (pc) correct.  However unless XSTIMM is not defined, they will
   ignore all neural elements to save time and memory.
*/ 

/*------------------------------------------*/

void eramod(void)

/* erase all of the model stuff, to allow
   more space to run something else. 
*/

{
  ncleanup();				/* erase all lists, reset pointers */
}

/*------------------------------------------*/

int checktyp(datum d)
{
  switch(d.vtype) {
	case NUMBER:
		return 1;
		break;
	case LITCHAR:
		return 2;
		break;
	default:
	case STRING:
		execerror ("wrong type: must have number",0);
		return 0;
		break;
  }
}

/*------------------------------------------*/

int checkstr(datum d)
{
  switch(d.vtype) {
	case NUMBER:
	case LITCHAR:
	default:
		execerror ("wrong type: must have string",0);
		return 0;
		break;
	case STRING:
		return 1;
		break;
  }
}

/*------------------------------------------*/

attrib *chanattr(elem *elpnt) 
               

/* set up attributes for a channel,
   and link them with the parent element.
*/

{
   attrib *apnt,*pnt;

  attrib *makattr(void);

#ifndef XSTIMM
  apnt = makattr();			/* make new attribute */
  if (!elpnt->attpnt) {			/* if element has no attributes */
     elpnt->attpnt = apnt;		/* add directly to element */
  }
  else {				/* else add to element's attr list */
     for (pnt=elpnt->attpnt; pnt->attpnt; pnt=pnt->attpnt) ;
     pnt->attpnt = apnt;
  } 
  return (apnt);
#else
  return (0);
#endif
}

/*------------------------------------------*/

cattrib *cachanattr(elem *elpnt) 
               
/* set up attributes for a channel,
   and link them with the parent element.
*/

{
   attrib *pnt;
   cattrib *apnt;

  cattrib *makcattr(void);

#ifndef XSTIMM
  apnt = makcattr();			/* make new attribute */
  if (!elpnt->attpnt) {			/* if element has no attributes */
     elpnt->attpnt = (attrib *)apnt;	/* add directly to element */
  }
  else {				/* else add to element's attr list */
     for (pnt=elpnt->attpnt; pnt->attpnt; pnt=pnt->attpnt) ;
     pnt->attpnt = (attrib *)apnt;
  } 
  apnt->stype   = NULLNOD;
  apnt->vrev    = NULLNOD;
  apnt->thresh  = NULLNOD;
  apnt->maxcond = NULLNOD;
  apnt->density = NULLNOD;
  apnt->kex     = NULLNOD;
  apnt->vmax    = NULLNOD;
  apnt->ekm     = NULLNOD;
  apnt->pkm     = NULLNOD;
  apnt->cabnd   = NULLNOD;
  apnt->pump    = 0;
  apnt->exch    = 0;
  return (apnt);
#else
  return (0);
#endif
}

/*------------------------------------------*/

void xmod(void)

/* modify an element */

{
  datum d1;
  Symbol *param;
  double *val;
  short *typ,*vtyp;

  param  = (Symbol *)*pc++;
  switch (param->type)
   {
    case MODIFY: if (checktyp(d1 = popm()))
#ifndef XSTIMM
    		    elpnt = tmpelem();			/* Make new element; */
  		    elpnt->modif = (int)(d1.val);	/* mark it modified. */
#endif
		 break;
    case ENAME:  getvar(&val,&typ,&vtyp);	
#ifndef XSTIMM
		 *val = cumelem;		/* save element number in var */
		 if (*typ==UNDEF) *typ = VAR;
		 *vtyp = NUMBER;
  		 elpnt->saved = 1;
#endif
		 break;
   }
}

/*------------------------------------------*/

void xcable(void)
{
  datum d1;
  Symbol *param;
  cable *epnt;

  param  = (Symbol *)*pc++;
  checktyp(d1 = popm());
  epnt = (cable *)makelem(CABLE,elpnt);		/* make new cable */
  elpnt = (elem *)epnt;				/* for membtyp() below */

  switch (param->type)
   {
    case LENGTH: epnt->length = d1.val; 
		break;
    case DIA:    epnt->dia    = d1.val;
		break;
    case CPLAM:  epnt->cplam = d1.val;
		break;
    
   }
/*  printf ("cable #%d %s %6.3f\n",elpnt->elnum,param->name,d1.val);  /* */
}

/*------------------------------------------*/

void xsphere(void)
{
  datum d1;
  Symbol *param;
  sphere *epnt;

  param = (Symbol *)*pc++;
  checktyp(d1 = popm());
  epnt = (sphere *)makelem(SPHERE,elpnt);	/* make new sphere */
  elpnt = (elem *)epnt;				/* for membtyp() below */

   switch (param->type)
    {
	case 0:	epnt->dia = d1.val;
		break;
	case DIA: epnt->dia = d1.val;
		break;
	case RADIUS: epnt->dia = d1.val * 2;
		break;
    }
/*  printf ("sphere %s %g\n",param->name,d1.val); */
}

/*------------------------------------------*/

void xcachan(void)
{
  datum d1;
  Symbol *param;
  static cattrib *apnt; 
  int npops;
  static int pump=0;

  param  = (Symbol *)*pc++;
  elpnt = makelem(CHAN,elpnt);			/* make dummy elem "chan"  */
  switch (param->type)
   {
    case CA:
    case CAPUMP:
    case CAEXCH:
    	 	 npops=0; 
		 break;
    case TYPE:  
    case VREV:  
    case TAUC:
    case CAO:
    case CAI:
    case KEX:
    case VMAX:
    case KM:
    case CBOUND:
    case THRESH:
    case MAXCOND:
    case DENSITY:npops=1;
		 break;
   }
 
  if (npops==1) checktyp(d1 = popm());

  switch (param->type)
   {
    case CA:    apnt = cachanattr(elpnt); 	 /* link chan attribs to elem */
		if (apnt) apnt->ctype = param->type;
		break;
    case CAPUMP:apnt->pump = 1; pump = param->type;
		break;
    case CAEXCH:apnt->exch = 1; pump = param->type;
		break;
    case TYPE:  if (apnt) apnt->stype = (int)(d1.val);
		break;
    case VREV:  if (apnt) apnt->vrev = d1.val;
		break;
    case TAUC:  if (apnt) apnt->taum = d1.val;
		break;
    case THRESH: if (apnt) apnt->thresh  = d1.val;
		break;
    case MAXCOND: if (apnt) apnt->maxcond = d1.val;
		break;
    case DENSITY: if (apnt) apnt->density = d1.val;
		break;
    case CAO: 	  if (apnt) apnt->cao = d1.val;
		break;
    case CAI: 	  if (apnt) apnt->cai = d1.val;
		break;
    case KEX: 	  if (apnt) apnt->kex   = d1.val;
		break;
    case VMAX: 	  if (apnt) apnt->vmax  = d1.val;
		break;
    case KM: 	  if (apnt) {
		    if (pump==CAPUMP) apnt->pkm = d1.val;
		    else              apnt->ekm = d1.val;
		  }
		break;
    case CBOUND:  if (apnt) apnt->cabnd = d1.val;
		break;
   }

/* fprintf (stderr,"ca chan %s %g\n",param->name,d1.val); /* */ 
}

/*------------------------------------------*/

void xchan(void)
{
  datum d1;
  Symbol *param;
  static attrib *apnt; 
  int npops;

  param  = (Symbol *)*pc++;
  elpnt = makelem(CHAN,elpnt);		/* make dummy elem "chan"  */
  switch (param->type)
   {
    case K:
    case NA:
    case HH:     npops=0; 
		 break;
    case TYPE:  
    case VREV:  
    case TAUM:
    case TAUH:
    case TAUN:
    case D1:
    case D2:
    case K1:
    case K2:
    case THRESH:
    case MAXCOND:
    case DENSITY:npops=1;
		 break;
   }
 
  if (npops==1) checktyp(d1 = popm());

  switch (param->type)
   {
    case K:
    case NA:
    case HH:    apnt = chanattr(elpnt); 	 /* link attribs to elem */
		if (apnt) apnt->ctype = param->type;
		break;
    case TYPE:  if (apnt) apnt->stype = (int)(d1.val);
		break;
    case VREV:  if (apnt) apnt->vrev = d1.val;
		break;
    case TAUM: if (apnt) apnt->taum = d1.val;
		break;
    case TAUH: if (apnt) apnt->tauh = d1.val;
		break;
    case TAUN: if (apnt) apnt->taum = d1.val;
		break;
    case THRESH: if (apnt) apnt->thresh  = d1.val;
		break;
    case MAXCOND: if (apnt) apnt->maxcond = d1.val;
		break;
    case DENSITY: if (apnt) apnt->density = d1.val;
		break;
    case D1: if (apnt) apnt->d1 = d1.val;
		break;
    case D2: if (apnt) apnt->d2 = d1.val;
		break;
    case K1: if (apnt) apnt->k1 = d1.val;
		break;
    case K2: if (apnt) apnt->k2 = d1.val;
		break;
   }

/* fprintf (stderr,"chan %s %g\n",param->name,d1.val); /* */ 
}

/*------------------------------------------*/

void noise(void)
{
  datum d1;
  Symbol *param;
  static attrib *apnt; 
  int npops;

  param  = (Symbol *)*pc++;
  switch (param->type)
   {
    case VESNOISE:
    case CHNOISE: npops = 0;
		  break;
    case N:	 
    case DUR:	
    case VSIZE:	  npops = 1; 
		 break;
   }

  if (npops==1) checktyp(d1 = popm());

  switch (param->type)
   {
    case VESNOISE:				 /* quantal vesicle noise */
		 apnt = chanattr(elpnt); 	 /* link attributes to elem */
		 if (apnt) apnt->ctype = param->type;
		 break;

    case CHNOISE:				 /* quantal channel noise */
		 apnt = chanattr(elpnt); 	 /* link attributes to elem */
		 if (apnt) apnt->ctype = param->type;
		 break;

    case N:	 if (apnt) apnt->density = d1.val;
		 break;

    case VSIZE:	 if (apnt && apnt->ctype==VESNOISE)
			 apnt->thresh = d1.val;
		 break;

    case DUR:	 if (apnt && apnt->ctype==CHNOISE)
			 apnt->thresh = d1.val;
		 break;

   }
}

/*------------------------------------------*/

void xsynapse(void)
{
  datum d1;
  Symbol *param;
  int npops;
  synapse *epnt;

  param  = (Symbol *)*pc++;
  epnt = (synapse *)makelem(SYNAPSE,elpnt);	/* make new synapse */
  elpnt = (elem *)epnt;

  switch (param->type)
   {
    case OPEN:  
    case CLOSE: 
    case LINEAR:npops=0;
		break;
    case VREV:  
    case THRESH:
    case NFILT1:
    case TIMEC1:
    case NFILT2:
    case TIMEC2:
    case TFALL2:
    case NFILT3:
    case TIMEC3:
    case TFALL3:
    case IGAIN: 
    case MAXCOND:
    case KD:	
    case EXPON:
    case DYAD:	npops = 1;
		break;
   }

  if (npops==1) checktyp(d1 = popm());

  switch (param->type)
   {
    case OPEN:  epnt->ntact = param->type; 
		break;
    case CLOSE: epnt->ntact = param->type; 
		break;
    case VREV:  epnt->vrev  = d1.val;
		break;
    case THRESH: epnt->thresh= d1.val;
		break;
    case NFILT1: epnt->nfilt1= d1.val;
		break;
    case TIMEC1: epnt->timec1= d1.val;
		break;
    case NFILT2: epnt->nfilt2 = d1.val;
		break;
    case TIMEC2: epnt->timec2 = d1.val;
		break;
    case TFALL2: epnt->tfall2 = d1.val;
		break;
    case NFILT3: epnt->nfilt3 = d1.val;
		break;
    case TIMEC3: epnt->timec3 = d1.val;
		break;
    case TFALL3: epnt->tfall3 = d1.val;
		break;
    case IGAIN: epnt->igain   = d1.val;
		break;
    case MAXCOND: epnt->maxcond = d1.val;
		break;
    case KD:	epnt->kd = d1.val;
		break;
    case LINEAR:epnt->curve = LINEAR;
    		epnt->expon = 0;
		break;
    case EXPON:	epnt->curve = EXPON;
    		epnt->expon = d1.val;
		break;
    case DYAD:	epnt->curve = DYAD;
    		epnt->expon = d1.val;
		break;
   }
/*fprintf(stderr,"synapse %s %g curv %g\n",
		param->name,d1.val,elpnt->curve); /* */ 
}

/*------------------------------------------*/

void xgj(void)
{
  datum d1;
  gapjunc *epnt;

  checktyp(d1 = popm());
  epnt = (gapjunc *)makelem(GJ,elpnt);	/* make new gap junction */
  epnt->area = d1.val;		/* area of gap junction */
  elpnt = (elem *)epnt;			/* for membtyp() below */
}

void rload(void)
{
  datum d1;
  loadelem *epnt;

  checktyp(d1 = popm());
  epnt = (loadelem *)makelem(LOAD,elpnt);	/* make new load */
  epnt->z = d1.val;
}

void rcap(void)
{
  datum d1;
  capac *epnt;

  checktyp(d1 = popm());
  epnt = (capac *)makelem(CAP,elpnt);		/* make new cap */
  epnt->c = d1.val;
}

void xgcap(void)
{
  datum d1;
  capac *epnt;
 
  checktyp(d1 = popm());
  epnt = (capac *)makelem(GNDCAP,elpnt);	/* make new gnd cap */
  epnt->c = d1.val;
}

void xresistor(void)
{
  datum d1;
  resistor *epnt;

  checktyp(d1 = popm());
  epnt = (resistor *)makelem(RESISTOR,elpnt);	/* make new resistor */
  epnt->z = d1.val;
}

void rbatt(void)
{
  datum d1;
  batt *epnt;

  checktyp(d1 = popm());
  epnt = (batt *)makelem(BATT,elpnt);		/* make new batt */
  epnt->v = d1.val;
}

void xgbatt(void)
{
  datum d1;
  batt *epnt;

  checktyp(d1 = popm());
  epnt = (batt *)makelem(GNDBATT,elpnt);	/* make new gndbatt */
  epnt->v = d1.val;
}

void membtyp(void)
{
  datum d1;
  Symbol *param;
  int ctype;

  checktyp(d1 = popm());
  param  = (Symbol *)*pc++;
  ctype = elpnt->ctype;
  switch (param->type)
   {
    case RM: switch (ctype) {
		case CABLE:  ((cable *)elpnt)->Rm = d1.val; break;
		case SPHERE: ((sphere *)elpnt)->Rm = d1.val; break;
	     } break;
    case RI: switch (ctype) {
		case CABLE:  ((cable *)elpnt)->Ri = d1.val; break;
	     } break;
    case RG: switch (ctype) {		/* for gj only */
		case GJ:  ((gapjunc *)elpnt)->specres = d1.val; break;
	     } break;
    case CM: switch (ctype) {
		case CABLE:  ((cable *)elpnt)->Cm = d1.val; break;
		case SPHERE: ((sphere *)elpnt)->Cm = d1.val; break;
	     } break;
    case VREST: switch (ctype) {
		case CABLE:  ((cable *)elpnt)->vrest = d1.val; break;
		case SPHERE: ((sphere *)elpnt)->vrest = d1.val; break;
		case GNDCAP:   ((capac *)elpnt)->vrest = d1.val; break;
		case LOAD:   ((loadelem *)elpnt)->vrest = d1.val; break;
	     } break;
    case VREV: switch (ctype) {
		case CABLE:  ((cable *)elpnt)->vrev = d1.val; break;
		case SPHERE: ((sphere *)elpnt)->vrev = d1.val; break;
		case SYNAPSE: ((synapse *)elpnt)->vrev = d1.val; break;
		case LOAD: ((loadelem *)elpnt)->vrev = d1.val; break;
	     } break;
    default: break;
   }
}

void xvbuf(void)
{
  vbuf *epnt;

  epnt = (vbuf *)makelem(BUF,elpnt);	/* make new gndbatt */
  elpnt = (elem *)epnt;			/* for membtyp() below */
  epnt->delay = 0;			/* no delay */
}

void xvbufd(void)
{
   datum d1;
   vbuf *epnt;

  checktyp(d1 = popm());
  ((vbuf *)elpnt)->delay = d1.val;	/* delay in msec */
}

/*------------------------------------------*/

smaxmin (double val)

{ 
   short int ival;

  val = limit(val, MAXSHORT, MINSHORT);
  ival = (short int) val;

  return (ival);
}


/*------------------------------------------*/

void conn1(void)
{
  datum d1,d2,d3;
  int narg;

  narg = (int)*pc++;
  if (narg > 2) d3 = popm();	/* get node number */
  else d3.val = NULLNOD;
  if (narg > 1) d2 = popm();
  else d2.val = NULLNOD;
  checktyp(d1 = popm());
  elpnt = tmpelem();		/* make new element */
  elpnt->node1a = smaxmin(d1.val);/* set node number */ 
  elpnt->node1b = smaxmin(d2.val);
  elpnt->node1c = smaxmin(d3.val);
  elpnt->node2a = NULLNOD;
  elpnt->node2b = NULLNOD;
  elpnt->node2c = NULLNOD;
  checknod(elpnt);		/* make node associated with element */
}

void conn1l(void)
{
  datum d1,d2,d3;
  int narg;
  node *npnt;

  narg = (int)*pc++;		/* get loc off stack first */
  if (narg > 1) {
    if (narg > 2) 
         d3 = popm();
    else d3.val = NULLNOD;
    d2 = popm();	
  }
  else d2.val = NULLNOD;
  checktyp(d1 = popm());
  conn1();

#ifndef XSTIMM
  checknod(elpnt);		/* make node associated with element */
  if (! (npnt=elpnt->nodp1)) {
	fprintf (stderr,"conn1L: can't find node %s\n",
		prnode(elpnt->node1a,elpnt->node1b,elpnt->node1c));
	return;
  }
  npnt->xloc = d1.val;
  npnt->yloc = d2.val;
  npnt->zloc = d3.val;
#endif
}

double dist3d(node *n1, node *n2)
{
   double dist,xdist,ydist,zdist;

  if (!n1 || !n2) return 0.0;
  if (n1->xloc == NULLNOD) {
   fprintf(stderr,"\ndist3d: location of node %s has not been defined.\n",
		 prnode(n1->nodenm1,n1->nodenm2,n1->nodenm3));
   execerror ("Missing location: ","stopping...");
  }  
  if (n2->xloc == NULLNOD) {
   fprintf(stderr,"\ndist3d: location of node %s has not been defined.\n",
		 prnode(n2->nodenm1,n2->nodenm2,n2->nodenm3));
   execerror ("Missing location: ","stopping...");
  }  
  xdist = n1->xloc - n2->xloc;
  ydist = n1->yloc - n2->yloc;
  zdist = n1->zloc - n2->zloc;
  dist = sqrt (xdist*xdist + ydist*ydist + zdist*zdist);
  return (dist);
}

void conn1m(void)

/* "at node: elem offset=frac put node" */

/* make a new node along a cable element */

{
  datum d1,d2,d3,d4,d5;
  int elnum,narg,found;
  short int nod1a,nod1b,nod1c;
  node *npnt,*npnt2,*npnt3;
  elem *epnt;
  attrib *apnt,*napnt;
  conlst *lpnt,*olpnt;
  double xdist,ydist,zdist,distfrac,len;
  attrib *makattr(void);

  narg = (int)*pc++;			/* get number of new node */
  if (narg > 2) d3 = popm();
  else d3.val = NULLNOD;
  if (narg > 1) d2 = popm();
  else d2.val = NULLNOD;
  checktyp(d1 = popm());

  nod1a=smaxmin(d1.val);
  nod1b=smaxmin(d2.val);
  nod1c=smaxmin(d3.val);

  checktyp(d4 = popm());	/* get fractional distance */
  checktyp(d5 = popm());	/* get cable element # rel to parent */

  conn1();		/* this gets parent node number off stack, */
			/*  makes new element, and connects parent node */
			/*  at one end but does not set type of element, */
			/*  nor its loc */

#ifndef XSTIMM
  if (!(npnt=findnode(elpnt->node1a,elpnt->node1b,elpnt->node1c,"Atmake"))) {
     fprintf (stderr,"Atmake: can't find node %s\n",
			 prnode(elpnt->node1a,elpnt->node1b,elpnt->node1c));
    execerror ("Missing parent node: ","stopping...");
  }

  elnum = (int)(d5.val); 			/* find element */
  for (found=0,lpnt=npnt->elemlst; lpnt && !found; lpnt=lpnt->next) {
    if (!(epnt=(elem *)lpnt->conpnt)) continue;
    if (epnt->elnum==elnum) found=1;
  }
      
  if (!found) {
     fprintf (stderr,"At node: can't find elem %d connected to node %s\n",
		elnum,prnode(elpnt->node1a,elpnt->node1b,elpnt->node1c));
     execerror ("Missing element: ","stopping...");
   }
				/* find the other end of element */

/* fprintf (stderr,"conn1m elem %d elnum %d  n1 %d n2 %d\n",
		epnt->elnum, elnum,epnt->nodp1,epnt->nodp2); /* */

  if ((npnt2=epnt->nodp1)==npnt) npnt2 = epnt->nodp2;
  if (!npnt2) {
      fprintf (stderr,"Atmake: can't find second node for elem %d\n", elnum);
      execerror("Missing node: ","stopping..."); 
  }
				/* stop if element is not a cable */
  if (epnt->ctype != CABLE) {
      fprintf (stderr,"At node: element %d is not a cable\n", elnum);
      execerror("Can't make node:","stopping..."); 
  } 

  elpnt->ctype = CABLE;		/* set new element to cable */

			/* Next, find or make the new node,  */
			/* connect it properly to the new cable, */
			/*  and reconnect old cable to the new node: */ 
  if (!(npnt3=findnode(nod1a,nod1b,nod1c,0))) {  /* use old node if it exists */
     setnodlst((npnt3=maknod(nod1a,nod1b,nod1c)),elpnt); /* new node, pointer */ 
  }
  elpnt->nodp2 = npnt3;		/* set pointer from new cable to new node */
  elpnt->node2a = nod1a;
  elpnt->node2b = nod1b;
  elpnt->node2c = nod1c;
  elpnt->nodp1 = npnt;		/* set pointer from new cable to old node */
  unsetnodlst(npnt,epnt);	/* del pointer from parent node to old cable */
  setnodlst(npnt,elpnt);	/* set pointer from parent node to new cable */
  setnodlst(npnt3,epnt);	/* set pointer from new node to old cable */
  
  if (epnt->nodp1==npnt) {		/* set node pointer for old cable */
	epnt->nodp1 = npnt3;		/*  to point to "new" node (npnt3) */
	epnt->node1a = nod1a;
	epnt->node1b = nod1b;
	epnt->node1c = nod1c;
  }
  else {
	epnt->nodp2 = npnt3;
	epnt->node2a = nod1a;
	epnt->node2b = nod1b;
	epnt->node2c = nod1c;
  }

  		/* Erase connection from node to old cable */
		/* First, find the old element: */
  epnt=NULL;
  olpnt=NULL;
  for (lpnt=npnt->elemlst; lpnt; lpnt=lpnt->next) {
    if (!(epnt=(elem *)lpnt->conpnt)) continue;
    if (epnt->elnum == elnum) {
      if (olpnt) olpnt->next = lpnt->next; 	/* erase conn pointer */
      else npnt->elemlst = lpnt->next; 
      break;
    }
    olpnt=lpnt;
  }
  if (lpnt) free (lpnt);

  distfrac = d4.val;
  if (distfrac > 1.0) distfrac = 1.0;
  if (distfrac < 0.0) distfrac = 0.0;

  if ((len=((cable*)epnt)->length)==NULLNOD)
     len = dist3d(npnt,npnt2);
  ((cable *)elpnt)->length = len * distfrac;	  /*set length of new cable */
  ((cable *)epnt)->length  = len * (1 - distfrac); /*set new length of old cabl*/

  ((cable *)elpnt)->dia   = ((cable *)epnt)->dia;  /* make same dia and cplam */
  ((cable *)elpnt)->cplam = ((cable *)epnt)->cplam;
  ((cable *)elpnt)->Rm    = ((cable *)epnt)->Rm;
  ((cable *)elpnt)->Ri    = ((cable *)epnt)->Ri;
  ((cable *)elpnt)->Cm    = ((cable *)epnt)->Cm;
  ((cable *)elpnt)->vrev  = ((cable *)epnt)->vrev;
  ((cable *)elpnt)->vrest = ((cable *)epnt)->vrest;

 				/* copy attributes from old cable to new */ 
for (apnt=epnt->attpnt; apnt; apnt=apnt->attpnt) {
   napnt = chanattr(elpnt);			/* alloc new attribute */
   napnt->ctype   = apnt->ctype;
   napnt->stype   = apnt->stype;
   napnt->vrev    = apnt->vrev;
   napnt->thresh  = apnt->thresh;
   napnt->maxcond = apnt->maxcond;
   napnt->density = apnt->density;
   napnt->taum    = apnt->taum;
   napnt->tauh    = apnt->tauh;
   napnt->d1      = apnt->d1;
   napnt->d2      = apnt->d2;
   napnt->k1      = apnt->k1;
   napnt->k2      = apnt->k2;
} 
  				/* Finally, calculate location of new node */
  xdist = npnt2->xloc - npnt->xloc;
  ydist = npnt2->yloc - npnt->yloc;
  zdist = npnt2->zloc - npnt->zloc;

  npnt3->xloc = npnt->xloc + xdist * distfrac;
  npnt3->yloc = npnt->yloc + ydist * distfrac;
  npnt3->zloc = npnt->zloc + zdist * distfrac;

#endif     /* XSTIMM */
}

void conn2s(void)
{
  datum d1,d2,d3;
  int narg;

  narg = (int)*pc++;
  if (narg > 2) d3 = popm();
  else d3.val = NULLNOD;
  if (narg > 1) d2 = popm();
  else d2.val = NULLNOD;
  checktyp(d1 = popm());
  elpnt->node1a = smaxmin(d1.val);
  elpnt->node1b = smaxmin(d2.val);
  elpnt->node1c = smaxmin(d3.val);
  checknod(elpnt);		/* make nodes associated with element */
}

void conn2sl(void)
{
  datum d1,d2,d3;
  int narg;
  node *npnt;

  narg = (int)*pc++;	/* get loc off stack first */
  if (narg > 1) {
    if (narg > 2) 
         d3 = popm();
    else d3.val = NULLNOD;
    d2 = popm();	/* get location of node */
  }
  else d2.val = NULLNOD;
  checktyp(d1 = popm());
  conn2s();
#ifndef XSTIMM
  checknod(elpnt);		/* make nodes associated with element */
  if (! (npnt=elpnt->nodp1)) {
	fprintf (stderr,"conn2s: can't find node %s\n",
		prnode(elpnt->node1a,elpnt->node1b,elpnt->node1c));
	return;
  }
  npnt->xloc = d1.val;
  npnt->yloc = d2.val;
  npnt->zloc = d3.val;
#endif
}

void conn2d(void)
{
  datum d1,d2,d3;
  int narg;

  narg = (int)*pc++;
  if (narg > 2) d3 = popm();
  else d3.val = NULLNOD;
  if (narg > 1) d2 = popm();
  else d2.val = NULLNOD;
  checktyp(d1 = popm());
  elpnt = tmpelem();			/* make new element */
  elpnt->node2a = (int)(d1.val);
  elpnt->node2b = (int)(d2.val);
  elpnt->node2c = (int)(d3.val);
}

void conn2dl(void)
{
  datum d1,d2,d3;
  int narg;
  node *npnt;

  narg = (int)*pc++;
  if (narg > 1) {
    if (narg > 2) 
         d3 = popm();
    else d3.val = NULLNOD;
    d2 = popm();	/* get location of node */
  }
  else d2.val = NULLNOD;
  checktyp(d1 = popm());
  conn2d();
#ifndef XSTIMM
  checknod(elpnt);		/* make nodes associated with element */
  if (! (npnt=elpnt->nodp2)) {
	fprintf (stderr,"conn2dl: can't find node %s\n",
		prnode(elpnt->node2a,elpnt->node2b,elpnt->node2c));
	return;
  }
  npnt->xloc = d1.val;
  npnt->yloc = d2.val;
  npnt->zloc = d3.val;
#endif
}

void efield(void)

/* push elem field on stack */

{
    datum d1,d2;
    Symbol *field;
    elem *epnt;
    int elnum;

 field = (Symbol *)*pc++;
 checktyp(d1 = popm());
 d2.val = 0;
#ifndef XSTIMM
 elnum = (int)(d1.val);
 for (epnt=elempnt; epnt; epnt=epnt->next) 		/* find element */
   if (epnt->elnum==elnum) break;

 if (!epnt) { 
   fprintf (stderr,"Elemfield: can't find elem %d\n",elnum);
   execerror ("Missing element; ","stopping...");
 }
 else
  switch (field->type) {
   case TYPE:
	d2.val = epnt->ctype;
	break;
   case LENGTH:
	if (epnt->ctype != CABLE) {
	  d2.val = 0.0;
	  break;
	}
	if (((cable*)epnt)->length != NULLNOD) {
	  d2.val = ((cable*)epnt)->length;
	}
	else {
	  checknod(epnt);
	  d2.val = dist3d (epnt->nodp1,epnt->nodp2);
	}
	break;
   case NDIST:
	checknod(epnt);
	d2.val = dist3d(epnt->nodp1,epnt->nodp2);
	break;
   case DIA:
	d2.val = ((cable *)epnt)->dia;
	break;
   case RM:
	d2.val = ((cable *)epnt)->Rm;
	break;
   case RI:
	d2.val = ((cable *)epnt)->Ri;
	break;
   case CPLAM:
	d2.val = ((cable *)epnt)->cplam;	/* only for cable */
	break;
   case NODE1A:
	d2.val = epnt->node1a;
	break;
   case NODE1B:
	d2.val = epnt->node1b;
	break;
   case NODE1C:
	d2.val = epnt->node1c;
	break;
   case NODE2A:
	d2.val = epnt->node2a;
	break;
   case NODE2B:
	d2.val = epnt->node2b;
	break;
   case NODE2C:
	d2.val = epnt->node2c;
	break;
  }
#endif
 d2.vtype = NUMBER;
 push(d2);
}


void nfield(void)

/* push node field on stack */

{
    datum d1,d2,d3,d4,d5;
    Symbol *field;
    int i, elnum, nod1, nod2, nod3, narg;
    node *npnt;
    conlst *lpnt;
    elem *epnt;

 field = (Symbol *)*pc++;
 if (field == (Symbol *)0) d4 = popm();			/* rel elem number */
 narg = (int) *pc++;
 if (narg > 2) d3 = popm();				/* get node */
 else d3.val = NULLNOD;
 if (narg > 1) d2 = popm();				/* get node */
 else d2.val = NULLNOD;
 if (narg) checktyp(d1 = popm());
 else d1.val = 0.0;
#ifndef XSTIMM
 nod1 = (int)(d1.val);
 nod2 = (int)(d2.val);
 nod3 = (int)(d3.val);
 if (!(npnt=findnode(nod1,nod2,nod3,"Nfield"))) {
   execerror ("Missing node; ","stopping...");
   return; 
 }
 if (field == (Symbol *)0) {			/* "node -> elemnum" */
   elnum = (int)(d4.val);
   for (lpnt=npnt->elemlst,i=1; lpnt && i<elnum; i++)	/* find the element */
        lpnt=lpnt->next;
    if (!lpnt || !(epnt=(elem *)lpnt->conpnt)) {
     fprintf (stderr,"Nodefield: can't find elem %d relative to node %s\n",
				elnum,prnode(nod1,nod2,nod3));
/*     execerror ("Missing element; ","stopping..."); */
     d5.val = -1;			
   }
   else d5.val = epnt->elnum;				/* element's number */
 }
 else {
  switch (field->type) {
   case NUMCONN:
        for (lpnt=npnt->elemlst,i=0; lpnt; i++)		/* how many elems */
          lpnt=lpnt->next; 
	d5.val = i;
	break;
   case XLOC:
	d5.val = npnt->xloc;
	break;
   case YLOC:
	d5.val = npnt->yloc;
	break;
   case ZLOC:
	d5.val = npnt->zloc;
	break;
  }
 }
#endif
 d5.vtype = NUMBER;
 push(d5);
}


double mindist (double dist1, double dist2, double dist3)

/* find minimum distance between a point and
   a line segment. If the point is off the end
   of the line segment, return the distance to
   that end.
*/

{
    double mdist,dist12,dist22,dist32,tmp1,tmp;

				/* check to see if node is off the end: */
   if (dist1 > dist2 && (dist1*dist1 - dist2*dist2) > dist3*dist3)
		 mdist = dist2;
   else
   if (dist1 < dist2 && (dist2*dist2 - dist1*dist1) > dist3*dist3)
		 mdist = dist1;

				/* formula for distance between a point
				/* and a line segment in three dimensions: */
   else {
      if (dist3==0.0) dist3 = 1e-20;
      dist12 = dist1 * dist1;
      dist22 = dist2 * dist2;
      dist32 = dist3 * dist3;
      tmp1 = (dist32 + dist12 - dist22) / (2 * dist3);
      if ((tmp=dist12-tmp1*tmp1) <= 0) mdist = 0;
      else mdist = sqrt (tmp);
   }
  return mdist;
}

void edist(void)

/* return distance between a node and an element */

{
    datum d1a,d1b,d1c,d2;
    int nod1a, nod1b, nod1c, narg, elnum;
    double dist,dist1,dist2,dist3;
    node *npnt,*np1,*np2;
    elem *epnt;

 checktyp(d2 = popm());
 elnum = (int)(d2.val);
 dist = 0;

 narg = (int) *pc++;
 if (narg > 2) d1c = popm();				/* get node */
 else d1c.val = NULLNOD;
 if (narg > 1) d1b = popm();
 else d1b.val = NULLNOD;
 checktyp(d1a = popm());
#ifndef XSTIMM
 nod1a = (int)(d1a.val);
 nod1b = (int)(d1b.val);
 nod1c = (int)(d1c.val);

 if (!(npnt=findnode(nod1a,nod1b,nod1c,"Edist"))) return; 

 for (epnt=elempnt; epnt; epnt=epnt->next) 	/* search all elems */
    if (epnt->elnum == elnum) break;

if (epnt==NULL) {
         fprintf (stderr,"Edist: can't find element %d\n",elnum);
	 execerror ("Missing element: "," stopping... ");
         return;  
 }
 np1 = epnt->nodp1;
 np2 = epnt->nodp2;
 if (!np1) {
         fprintf (stderr,"Edist: can't find node for element %d\n",elnum);
         return;  
 }

 dist1 = dist3d (npnt,np1);

 switch (epnt->ctype) {

#define max(x, y)	(((x) < (y)) ? (y) : (x))

  case CABLE: 
 
    if (!np2) {
         fprintf (stderr,"Edist: can't find node for element %d\n",elnum);
         return;  
    }

   dist2 = dist3d (npnt,np2);
   dist3 = dist3d (np1,np2);
   dist = mindist (dist1,dist2,dist3) - ((cable*)epnt)->dia*0.5;
   dist = max(dist,0);			/* no negative distances */
   break;

  case SPHERE: 
    dist = dist1 - ((sphere*)epnt)->dia*0.5;
    dist = max(dist,0);			/* no negative distances */
    break;

  default:				/* element is not cable */
    dist = dist1;
    break;
  }

#endif
  d1a.val = dist;
  d1a.vtype = NUMBER;
  push(d1a);
}

void efrac(void)

/* return point in an element that is the closest
   to a node */

{
    datum d1a,d1b,d1c,d2;
    int nod1a, nod1b, nod1c, narg, elnum;
    double frac,dist,dist1,dist2,dist3;
    node *npnt,*np1,*np2;
    elem *epnt;

 checktyp(d2 = popm());
 elnum = (int)(d2.val);
 frac = 0;

 narg = (int) *pc++;
 if (narg > 2) d1c = popm();				/* get node */
 else d1c.val = NULLNOD;
 if (narg > 1) d1b = popm();
 else d1b.val = NULLNOD;
 checktyp(d1a = popm());
#ifndef XSTIMM
 nod1a = (int)(d1a.val);
 nod1b = (int)(d1b.val);
 nod1c = (int)(d1c.val);

 if (!(npnt=findnode(nod1a,nod1b,nod1c,"Fdist"))) return; 

 for (epnt=elempnt; epnt; epnt=epnt->next) 	/* search all elems */
    if (epnt->elnum == elnum) break;

if (epnt==NULL) {
         fprintf (stderr,"Fdist: can't find element %d\n",elnum);
	 execerror ("Missing element: "," stopping... ");
         return;  
 }
 np1 = epnt->nodp1;
 np2 = epnt->nodp2;
 if (!np1) {
         fprintf (stderr,"Fdist: can't find node for element %d\n",elnum);
         return;  
 }

 dist1 = dist3d (npnt,np1);

 if (epnt->ctype==CABLE) {
 
    if (!np2) {
         fprintf (stderr,"Edist: can't find node for element %d\n",elnum);
         return;  
    }

   dist2 = dist3d (npnt,np2);
   dist3 = dist3d (np1,np2);
   if (dist3 == 0.0) dist3 = 1e-20;
   dist = mindist (dist1,dist2,dist3);
   if      (dist == dist1) frac = 0.0;
   else if (dist == dist2) frac = 1.0;
   else frac = sqrt (dist1*dist1 - dist*dist) / dist3;
  }

  else {				/* element is not cable */
    frac = 0.0;
  }
#endif
  d1a.val = frac;
  d1a.vtype = NUMBER;
  push(d1a);
}

void ndist(void)

/* return distance between two nodes */

{
    datum d1a,d1b,d1c,d2a,d2b,d2c;
    int nod1a, nod1b, nod1c, narg;
    int nod2a, nod2b, nod2c;
    node *npnt1,*npnt2;

 narg = (int) *pc++;
 if (narg > 2) d1c = popm();				/* get second node */
 else d1c.val = NULLNOD;
 if (narg > 1) d1b = popm();
 else d1b.val = NULLNOD;
 checktyp(d1a = popm());
 nod1a = (int)(d1a.val);
 nod1b = (int)(d1b.val);
 nod1c = (int)(d1c.val);

 narg = (int) *pc++;
 if (narg > 2) d2c = popm();				/* get first node */
 else d2c.val = NULLNOD;
 if (narg > 1) d2b = popm();
 else d2b.val = NULLNOD;
 d2a = popm();
 nod2a = (int)(d2a.val);
 nod2b = (int)(d2b.val);
 nod2c = (int)(d2c.val);

#ifndef XSTIMM
 if (!(npnt1=findnode(nod1a,nod1b,nod1c,"Ndist"))) return; 
 if (!(npnt2=findnode(nod2a,nod2b,nod2c,"Ndist"))) return; 

 d1a.val = dist3d (npnt1,npnt2);
#else
 d1a.val = 0;
#endif
 push(d1a);
}

void xgausnn(void)

/* Make random array of cells with defined regularity. */

{
  int i;
  Symbol *param,*var;
  datum d1,d2,d3;
  static int narg,numcells=0,numpts=0;
  static double mean=0,stdev=0,density=0,reg=0;
  static double xframe=0,yframe=0,xcenter=0,ycenter=0;
  static float *x=0,*y=0;
  double *dpnt;


  param = (Symbol *)*pc++;
  switch (param->type) {

    case MEAN:  checktyp(d1 = popm());
		mean = d1.val;
		break;

    case STDEV: checktyp(d1 = popm());
		stdev = d1.val;
		break;

    case DENSITY: checktyp(d1 = popm());
		density = d1.val;
		break;

    case REG:   checktyp(d1 = popm());
		reg = d1.val;
		break;

    case SIZE:  narg = (int) *pc++;
                if (narg > 2) d3 = popm();
                else d3.val = 0;
                if (narg > 1) d2 = popm();
                else d2.val = 0;
 		checktyp(d1 = popm());
		xframe = d1.val;
		yframe = d2.val;
		break;

    case CENTER: narg = (int) *pc++;
                if (narg > 2) d3 = popm();
                else d3.val = 0;
                if (narg > 1) d2 = popm();
                else d2.val = 0;
 		checktyp(d1 = popm());
		xcenter = d1.val;
		ycenter = d2.val;
		break;

    case GAUSNN:var = (Symbol *)(*pc++);          /* array to be created */
#ifndef XSTIMM
  		if (setdisp>=0) disp = setdisp;	/* "-d n" overrides "disp=n" */
 		vidmode = (int)setvar("vidmode");
    		numpts=gausnn(mean,stdev,density,reg,xframe,yframe,
			xcenter,ycenter,numcells,&x,&y,0,!(disp||vidmode),0);

     		d1.val = numpts;	/* set up nc array dimensions */
     		d1.vtype = NUMBER;
     		d2.val = 2;
     		d2.vtype = NUMBER;
		push(d1);
		push(d2);
		dpnt = darr2(var,2);	     /* allocate space for array */
		for (i=0; i<numpts; i++) {
		  *dpnt++ = x[i];	     /* make x and y values be */
		  *dpnt++ = y[i];	     /*  successive values in array */
		}
		var->vtype = NUMBER;
		mean = stdev = density = reg = 0.0;
		xframe = yframe = xcenter = ycenter = 0;
		numcells = 0;
		if (x) free (x);	/* erase the subroutine's arrays */
		if (y) free (y);
#else
     		d1.val = 0;	
     		d1.vtype = NUMBER;
#endif
		push (d1); 	/* finally, return n */
		break; 
  } /* switch */
}


void xrecept(void)

/* read in data from rod or cone statement:

		1	x position
		2	y position  (optional)
*/

{
  datum d1,d2,d3;
  int narg;
  Symbol *param;
  node *npnt;
  double xpos,ypos,zpos;
  photrec *epnt;

#ifdef XSTIM
  recnod *maksnod(),*rpnt;
#endif

  param = (Symbol *)*pc++;			/* receptor type */
  narg  = (int)*pc++;
  if (narg > 2) d3 = popm();			/* get z loc */
  else d3.val = NULLNOD;
  if (narg > 1) d2 = popm();			/* get y loc */
  else d2.val = NULLNOD;
  if (narg > 0) checktyp(d1 = popm());		/* get x loc */
  else d1.val = NULLNOD;
  
  zpos  = d3.val;				/* z val */
  ypos 	= d2.val;				/* y val */
  xpos  = d1.val;				/* x val */

  epnt = (photrec *)makelem(param->type,elpnt); /* make new photoreceptor */
  elpnt = (elem *)epnt;				/* save for recparm() below */

  if (narg > 2) 
     epnt->zpos  = zpos;			/* z val */
  if (narg > 1) 
     epnt->ypos  = ypos;			/* y val */
  epnt->xpos     = xpos;			/* x val */

						/* no z val in struct recept */
  if (!epnt->modif) {				/* never modify node */
#ifndef XSTIMM
    if (! (npnt=epnt->nodp1)) {
	fprintf (stderr,"recept: can't find node %s\n",
		prnode(epnt->node1a,epnt->node1b,epnt->node1c));
	return;
    }

     if (xpos!=NULLNOD)
        npnt->xloc = xpos;			/* By default, place node at */
     if (ypos!=NULLNOD)
        npnt->yloc = ypos;			/*  same loc. as photorecep */
     if (zpos!=NULLNOD)
        npnt->zloc = zpos;
#endif

#ifdef XSTIM
  rpnt = maksnod();
  rpnt->recnm1 = elpnt->node1a;     /* node numbers are defined by conn() */
  rpnt->recnm2 = elpnt->node1b;
  rpnt->recnm3 = elpnt->node1c;
  if (xpos!=NULLNOD) rpnt->xpos = xpos;
  else               rpnt->xpos = 0;
  if (ypos!=NULLNOD) rpnt->ypos = ypos;
  else               rpnt->ypos = 0;
#endif
  }   /* if (!elpnt->modif) */
}

void xrecparm(void)

/* get parameters for rods and cones */

{
  datum d1;
  Symbol *param;
  int npops;
  photrec * ppnt;

  param  = (Symbol *)*pc++;
  
  switch (param->type) {
    case DIA:   
    case MAXCOND: 
    case PIGM:  
    case PATHL: 
    case ATTF:  
    case FILT:  
    case TIMEC1:npops = 1;
		break;
    case SAVE:  
    case RESTORE:
    case PHOTNOISE: 
		npops = 0;
		break;
   }
  
  if (npops==1) checktyp(d1 = popm());

#ifndef XSTIMM

		/* Here, elpnt is a photrec, set by recept() above */

  ppnt = (photrec *)elpnt; 
  switch (param->type) {
    case DIA:   ppnt->dia = d1.val;
		break;
    case MAXCOND: ppnt->maxcond = d1.val; 
		break;
    case PIGM:  ppnt->pigm  = d1.val; 
		break;
    case PATHL: ppnt->pathl = d1.val; 
		break;
    case ATTF:  ppnt->attf = d1.val; 
		break;
    case FILT:  ppnt->filt = (int)d1.val; 
		break;
    case TIMEC1:ppnt->timec1 = d1.val; 
		break;
    case SAVE:  ppnt->save = 1; 
		break;
    case RESTORE:
		ppnt->restore = 1; 
		break;
    case PHOTNOISE:
		ppnt->photnoise = 1; 
		break;
   }
#endif 		/* XSTIMM */
}

/********************************************************/

#define RTIMINC .001

#ifdef XSTIM
					/* this part for "stim", not "nc" */
void xstim()

/* read in data from stimulus statement:
*/

{
  datum d1,d2,d3,d4;
  Symbol *param;
  int stimarr, backarr;
  int narg, spec;
  FILE *ftemp,*fopen();

static double start=0.0;
static double dur=0.1;
static double wavel=1.0;
static double inten=100.0;
static double backgr=0.0;
static double blurrad=0.0,tblur=0.0;
static double speriod = 20;
static double contrast = 0.5;
static double tfreq = 0.0;
static double sscale = 1.0;
static double ymins = 0.0;
static double ymaxs = 0.0;
static double orient = 0.0;

static int stype;
/* static int rnum1,rnum2,rnum3;        /* not used for "stim" */
static double xstm=0, ystm=0;
static double xcent=0, ycent=0;		/* center coords for stim array */
static int centerfl=0;                  /* center specified */
static double xmax,xmin,ymax,ymin;	/* borders of of photrec array */
static double xrange, yrange;		/* size of photrec array */
static int arrsize;			/* size of stimulus array */
static int arraymade=0;			/* stimulus array already allocated */
static int stimflag=0;                  /* start stimulus */
static int blurfl=0;                    /* blur array done */

extern double stimdia;                     /* defined in "stimsub.c" */
extern double stimydist;                   /* defined in "stimsub.c" */

  param  = (Symbol *)*pc++;
  
  switch (param->type)
   {
    case BAR: 
    case SPOT:  checktyp(d1 = popm());
                stimdia   = d1.val;
                stype = param->type; 
                break;
    case SINE:  checktyp(d1 = popm());
                speriod   = d1.val;
                stype = param->type; 
                break;
    case RECT:
		narg = (int)*pc++;
  		if (narg > 1) d2 = popm();
		else d2.val = 1.0;
 		checktyp(d1 = popm());
		stimdia = d1.val;
		stimydist = d2.val;
		stype = param->type; 
		break;
    case NODE:
                narg = (int)*pc++;
                if (narg > 2) d3 = popm();
                else d3.val = NULLNOD;
                if (narg > 1) d2 = popm();
                else d2.val = NULLNOD;
 		checktyp(d1 = popm());
    /*          rnum1 = d1.val;                 /* get node number */
    /*          rnum2 = d2.val;
                rnum3 = d3.val;
    */
                break;
    case ROD: 
    case CONE:  narg = (int) *pc++;
                if (narg > 2) d3 = popm();
                else d3.val = NULLNOD;
                if (narg > 1) d2 = popm();
                else d2.val = NULLNOD;
 		checktyp(d1 = popm());
    /*          rnum1     = d1.val;
                rnum2     = d2.val;
                rnum3     = d3.val;
    */
                stype = param->type; 
                break;
    case LOC:   narg  = (int)*pc++;
                if (narg > 0) {
                  if (narg > 1) {
                    if (narg > 2) {
                      if (narg > 3) {
		          d4 = popm();
                          ymaxs = d4.val;
                      }
		      d3 = popm();
                      ymins = d3.val;
                    }
                    d2 = popm();                /* read in y loc */
                    ystm = d2.val;
                  }
		  else ystm = 0;
 		checktyp(d1 = popm());
                xstm = d1.val;
                }
                break;
    case CENTER:narg  = (int)*pc++;
                if (narg > 0) {
                  if (narg > 1) {
                    if (narg > 2) 
                       d3 = popm();
                    d2 = popm();                /* read in y center */
                    ycent = d2.val;
                  }
		  else ycent = 0;
 		checktyp(d1 = popm());
                xcent = d1.val;
		centerfl = 1;
                }
                break;
    case SSCALE: checktyp(d1 = popm());
                sscale   = d1.val;
                break;
    case START: checktyp(d1 = popm());
                start = d1.val;
                stimflag = 1;
                break;
    case DUR:   checktyp(d1 = popm());
                dur   = d1.val;
                break;
    case CONTRAST: checktyp(d1 = popm());
                contrast   = d1.val;
                break;
    case TFREQ: checktyp(d1 = popm());
                tfreq = d1.val;
                break;
    case ORIENT: checktyp(d1 = popm());
                orient = d1.val;
                break;
    case WAVEL: spec = (int)*pc++;
		switch (spec) {
		  case 0: 
 		    checktyp(d1 = popm());
		    wavel = d1.val;
		    break;
		  case SUN:
                    wavel = 0;
                    break;
		  case XENON:
                    wavel = 1;
                    break;
		}
		break;
    case INTEN: checktyp(d1 = popm());
                inten = d1.val;
                break;

    case BACKGR: checktyp(d1 = popm());
                backgr = d1.val;
                backarr = 1;
                stype = param->type; 
                break;

    case VCLAMP:
    case CCLAMP: checktyp(d1 = popm());
                inten = d1.val;
                stype = param->type; 
                break;

    case BLUR:  checktyp(d1 = popm());		 /* get blur diameter */
                tblur = d1.val/2;		 /* divide dia by 2 = radius */
		if (blurfl) {
		   if (blurrad!=tblur) blurfl = 0;  /* remake blur function */
                }
		blurrad = tblur;
                break;
    case SFILE: if (*stimfile) fclose (stimout);
                *stimfile = 0;
		if (!checkstr(d1=popm())) break; 	 /* get filename */
                strcpy (stimfile,d1.str);  		/* copy filename */
                if (stimhlp) fprintf (stderr,"stimfil '%s'\n",stimfile);
                if (*stimfile) {
                        stimfflg = 1; 
                        if ((ftemp=fopen(stimfile,"w")) == NULL) {
                          fprintf (stderr,"%s: can't open file '%s'\n",
						progname,stimfile);
                          break;
                        }
                        else stimout = ftemp;
                }
                else {
                        stimfflg = 0;
                        stimout = stdout;
                }
		stcomment();			/* print comment in stim file */
                break;

    case STIM:  
        stimarr = 0;
        backarr = 1;
					/* find photoreceptor array size */

#ifdef XSTIM
  varcopy();
  if (setdebug) debug = setdebug;
  if (setdebgz) debugz = setdebgz;
#endif

     switch (stype) {
       case BAR:
       case RECT:
       case SPOT:
       case SINE:
       case BACKGR:
        if (!blurfl) {
             if (blurrad>0.0) {
	       makblur(blurrad,sscale); /* make a blurring function */
               blurfl = 1;
	       fprintf (stimout,"## blur     array size %d\n",blursize*2);/* */
	     }
        }
	if (!arraymade) {
	   findarrsiz(&xmax,&xmin,&ymax,&ymin);  /* find maxmin of recepts */
	   if (!centerfl) {
	     xcent = (xmax+xmin) * 0.5;
	     ycent = (ymax+ymin) * 0.5;
	   }
	   xrange = xmax-xmin;
	   yrange = ymax-ymin;
	   if (sscale <=0) sscale = 1.0;
	   arrsize = (int)(max(xrange,yrange) / sscale);
	   if (blurrad>0) arrsize += blursize*2;

	   fprintf (stimout,"## stimulus array size %d\n",arrsize);	/* */

	   if (initc(2,arrsize)<=0) {		/* make stimulus to fit */
	       fprintf (stderr,"Stimulus array is too big = %d\n",arrsize);
	       fprintf (stderr,"Blur array dia %d\n",blursize*2);
	       fprintf (stderr,"Stimulus xmax %g xmin %g ymax %g ymin %g\n",
			xmax,xmin,ymax,ymin);
    	       execerror ("can't continue"," stopping...");
	   }
	   arraymade = 1;
	}
	 break;
       default:
	  break;
      } /* switch stype */

        switch (stype) {

        case 0:
        default: break;

/* Vclamp, cclamp, rod and cone are done in xstim() for "nc" below.
   They are not performed by (the program) "stim" because they don't
   require blur, and sometimes need to be done in addition to a
   predefined, blurred stimulus.
*/

        case VCLAMP:
/*              vclist(start,    rnum1,rnum2,rnum3,inten,wavel,"vcl");
                vclist(start+dur,rnum1,rnum2,rnum3,inten,wavel,"eff");
*/              break;

        case CCLAMP:
/*              vclist(start,    rnum1,rnum2,rnum3,inten,wavel,"ccl");
                vclist(start+dur,rnum1,rnum2,rnum3,inten,wavel,"iof");
*/              break;

        case ROD:
        case CONE: 				/* flash one receptor */
/*              vclist(start,    rnum1,rnum2,rnum3, inten,wavel,"del");
                vclist(start+dur,rnum1,rnum2,rnum3,-inten,wavel,"del"); 
*/              break;


        case BACKGR:
                  recback(stimarr,0.0,wavel);   /* zero recept stim inten */
                  recback(backarr,backgr,wavel); /* set recept backgr inten */
                  abslist(0.0, start);          /* make action list */
                  break;
        case BAR:
		makrect(stimarr, stimdia, (double)CONVSIZE,
			xstm-xcent,ystm-ycent,sscale,inten,wavel);
                if (stimflag) {
                  stimflag = 0;
                  recpat(stimarr,xcent,ycent,sscale);  /* set recept inten */
                  stimlist(1.0, start);         /* make an action list */
                  stimlist(-1.0, start+dur);
                  recback(stimarr,0.0,1.0);   /* zero recept stim inten */
                  recback(backarr,backgr,1.0);/* set recept backgr inten */
                }
                break;

        case RECT:
		makrect(stimarr, stimdia, stimydist,xstm-xcent,ystm-ycent,
							sscale,inten,wavel);
                if (stimflag) {
                  stimflag = 0;
                  recpat(stimarr,xcent,ycent,sscale);  /* set recept inten */
                  stimlist(1.0, start);         /* make an action list */
                  stimlist(-1.0, start+dur);
                  recback(stimarr,0.0,1.0);   /* zero recept stim inten */
                  recback(backarr,backgr,1.0);/* set recept backgr inten */
                }
                break;

        case SPOT:
                makspot(stimarr, stimdia,xstm-xcent,ystm-ycent,
						sscale,inten,wavel);
                if (stimflag) {
                  stimflag = 0;
                  recpat(stimarr,xcent,ycent,sscale);  /* set recept inten */
                  stimlist(1.0, start); 	/* make an action list */
                  stimlist(-1.0, start+dur);
                  recback(stimarr,0.0,1.0);     /* zero recept stim inten */
                  recback(backarr,backgr,1.0);  /* set recept backgr inten */
                }
                break;

        case SINE:
					/* xstm is xmin; ystm is xmax. */
          if (speriod <= 0.0) speriod = 1;   /* spatial period from user */
          if (tfreq) {		/* drifting grating */
                 double timres,tperiod,tincr,stime;

            timres = .001;
            tperiod = 1.0 / tfreq;	/* temporal period */
	    tincr = tperiod * timres;
	    if (tincr < RTIMINC) tincr = RTIMINC; /* time incr = recep timinc */
            for (stime=start; stime<(start+dur-tincr); stime += tincr) {
              maksine(stimarr,speriod,xstm,ystm,ymins,ymaxs,
				(stime-start)/tperiod,orient,
				-xcent,-ycent,sscale,inten,contrast,wavel); 
              recpat(stimarr,xcent,ycent,sscale);  /* set recept inten */
              stimlist(1.0, stime); 	    /* make an action list */
              stimlist(-1.0, stime+tincr);
              recback(stimarr,0.0,1.0);     /* zero recept stim inten */
              recback(backarr,backgr,1.0);  /* set recept backgr inten */

            }   /* for (stime= ; ; ) */
          }   /* if (drate) */
          else {		/* static grating */
              maksine(stimarr,speriod,xstm,ystm,ymins,ymaxs,0.0,orient,
				-xcent,-ycent,sscale,inten,contrast,wavel); 

              if (stimflag) {			/* if start val is given */
                 stimflag = 0;
                 recpat(stimarr,xcent,ycent,sscale);  /* set recept inten */
                 stimlist(1.0, start); 	/* make an action list */
                 stimlist(-1.0, start+dur);
                 recback(stimarr,0.0,1.0);     /* zero recept stim inten */
                 recback(backarr,backgr,1.0);  /* set recept backgr inten */
              }
          } /* else */
	  break;

        }       /* switch (stype) */

	stype = 0;		/* reset stype for next stim statement */
        break;  /* case STIM */

  }     /* switch */
}

/*------------------------------------------------------*/

void findarrsiz(double *xmax,double *xmin,double *ymax,double *ymin)

/* Find max, min for the array of photoreceptors
    described by the user's program.
*/

{
  recnod *npt;
  double xma,xmi,yma,ymi;

  xma = yma = 0.0;
  xmi = ymi = 1e10;
  for (npt=reclist; npt; npt=npt->next) {
    xma  = max(xma,npt->xpos);
    xmi  = min(xmi,npt->xpos);
    yma  = max(yma,npt->ypos);
    ymi  = min(ymi,npt->ypos);
  }
  *xmax = xma;
  *xmin = xmi;
  *ymax = yma;
  *ymin = ymi;
}


#endif 			/* end of xstim for generating stimulus file */

/********************************************************/

#ifdef XMOD		 /* this part for nc (modcode.o): */

void xstim(void)

/* read in data from stimulus statement:
*/

{
  datum d1,d2,d3;
  Symbol *param;
  int narg, spec;
  double xpos, ypos;
/*  static int dia, ydist, xloc, yloc, halfx, halfy, radius, siney; */
  static double dia, ydist, xloc, yloc, halfx, halfy, radius, siney;
  static double start,dur;
  static double inten,sinten,backgr,xsine,theta,sincr;
  static int stype,rnum1,rnum2,rnum3,filuse=0;
  static char *filename;
  static double wavel = 1.0;
  static double speriod = 20;
  static double contrast = 0.5;
  static double tfreq = 0.0;
  static double sscale = 1.0;
  static char stimfile[40]={0};
  photrec *ept;
  recep *rpnt;
  FILE *ftemp,*fopen(const char *, const char *);

  param  = (Symbol *)*pc++;

  switch (param->type)
   {
    case RECT:  narg = (int)*pc++;
  		if (narg > 1) d2 = popm();
		else d2.val = 1.0;
 		checktyp(d1 = popm());
		dia   = d1.val;
		ydist = d2.val;
		stype = param->type; 
		break;
    case BAR: 
    case SPOT:  checktyp(d1 = popm());
		dia   = d1.val;
		stype = param->type; 
		break;
    case SINE:  checktyp(d1 = popm());
		speriod  = d1.val;
		stype = param->type; 
		break;
    case NODE:
		narg = (int)*pc++;
  		if (narg > 2) d3 = popm();
		else d3.val = NULLNOD;
  		if (narg > 1) d2 = popm();
		else d2.val = NULLNOD;
 		checktyp(d1 = popm());
		rnum1 = (int)d1.val;			/* get node number */
		rnum2 = (int)d2.val;
		rnum3 = (int)d3.val;
		break;
    case ROD: 
    case CONE:
		narg = (int)*pc++;
  		if (narg > 2) d3 = popm();
		else d3.val = NULLNOD;
  		if (narg > 1) d2 = popm();
		else d2.val = NULLNOD;
 		checktyp(d1 = popm());
		rnum1 = (int)d1.val;			/* get recep number */
	        rnum2 = (int)d2.val;
	        rnum3 = (int)d3.val;
		stype = param->type; 
		break;
    case LOC:	narg  = (int)*pc++;
  		if (narg > 0) {
  		  if (narg > 1) {
  		    if (narg > 2) 
     	 	        d3 = popm();
     	 	    d2 = popm();		/* read in y loc */
		    yloc = d2.val;
  		  }
                else yloc = 0;
 		checktyp(d1 = popm());   	/* x loc */
		xloc = d1.val;
		}
		break;
    case CENTER: narg  = (int)*pc++;
  		if (narg > 0) {
  		  if (narg > 1) {
  		    if (narg > 2) 
     	 	       d3 = popm();
     	 	    d2 = popm();		/* read in y loc */
  		  }
 		checktyp(d1 = popm());	 	/* x loc */
		}
		break;
    case START:	checktyp(d1 = popm());
		start = d1.val;
		break;
    case DUR:	checktyp(d1 = popm());
		dur   = d1.val;
		break;
    case CONTRAST: checktyp(d1 = popm());
                contrast   = d1.val;
                break;
    case TFREQ:	checktyp(d1 = popm());
		tfreq  = d1.val;
		break;
    case WAVEL: spec = (int)*pc++;	
		switch (spec) {
		  case 0:
 		    checktyp(d1 = popm());
		    wavel = d1.val;
		    break;
		  case SUN:
		    wavel = 0;
		    break;
		  case XENON:
		    wavel = 1;
		  break;
		}
		break;
    case SSCALE:checktyp(d1 = popm());
		sscale = d1.val;
		break;
    case INTEN:	checktyp(d1 = popm());
		inten = d1.val;
		break;
    case BACKGR: checktyp(d1 = popm());
		backgr = d1.val;
		stype = param->type; 
		break;
    case VCLAMP:
    case CCLAMP: checktyp(d1 = popm());
		inten = d1.val;			/* get node number */
		stype = param->type; 
		break;
    case BLUR:	checktyp(d1 = popm());
		/* blurrad = d1.val; */
		break;
    case SFILE:	if (!checkstr(d1=popm())) break; 	 /* get filename */
		filename = d1.str;
		if (stimfflg && *stimfile) fclose(stimin); /* close previous */
                strcpy (stimfile,filename);  		/* copy filename */
		stfile = stimfile;			/* for plotinit */
		if (! *filename) { filuse = 0; break; }
		if ((ftemp=fopen(stimfile,"r")) == NULL) {
			fprintf (stderr,"%s: can't open file '%s'\n",
					progname,stimfile);
			filuse = 0;
			break;
		}
		else {
		   stimin = ftemp;
		   filuse = 1;
		}
		break;

    case STIM:	 			/* real time stimulus without blur */

	switch (stype) {

	case VCLAMP:
		makvstim(start,    rnum1,rnum2,rnum3,inten,"vcl");
		makvstim(start+dur,rnum1,rnum2,rnum3,inten,"eff");
		break;

	case CCLAMP:
		makvstim(start,    rnum1,rnum2,rnum3,inten,"ccl");
		makvstim(start+dur,rnum1,rnum2,rnum3,inten,"iof");
		break;

	case ROD:			/* add to inten on specific receptor */
	case CONE: 
		makrstim(start+dur,rnum1,rnum2,rnum3,-inten,wavel,"del");
		makrstim(start,    rnum1,rnum2,rnum3, inten,wavel,"del");
		break;	

       }

      if (filuse) stimfflg = 1;
      else {

        stimfflg = 0;

	switch (stype) {

	case VCLAMP:			/* don't make these stim twice */ 
	case CCLAMP:
	case ROD:
	case CONE: 
		  break;

	case BACKGR: 					/* abs inten */

	       			/* look for existing photoreceptors: */
		if (runyet) {
	  	 for (rpnt=recpnt; rpnt; rpnt=(recep*)rpnt->next) {
		  if (rpnt->ctype == ROD || rpnt->ctype == CONE)
		    makrstim (start,rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						backgr,wavel,"abs");
		  }
                 }
				/* look for new rod, code elements, */
			        /*  even if run yet: */

	  	 for (ept=(photrec*)elempnt; ept; ept=(photrec*)ept->next) {
		  if (ept->ctype == ROD || ept->ctype == CONE)
		    makrstim (start,ept->node1a,ept->node1b,ept->node1c,
						backgr,wavel,"abs");
		 }
		break;


	case BAR:			/* stim for possibly all receptors */
	  halfx = dia / 2;
	  halfy = 1000;
	  if (runyet) {
	   for (rpnt=recpnt; rpnt; rpnt=(recep*)rpnt->next) {
	    if (rpnt->ctype == ROD || rpnt->ctype == CONE) 
	    if (inrect(rpnt->xloc,rpnt->yloc, xloc+halfx, xloc-halfx,
			yloc+halfy, yloc-halfy)) {
		makrstim(start,    rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						 inten,wavel,"del");
		makrstim(start+dur,rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						-inten,wavel,"del");
	     }
	   }
	  }
	  else {
	   for (ept=(photrec*)elempnt; ept; ept=(photrec*)ept->next) {
	    if (ept->ctype == ROD || ept->ctype == CONE) 
	    if (inrect(ept->xpos,ept->ypos,
			xloc+halfx, xloc-halfx,
			yloc+halfy, yloc-halfy)) {
		makrstim(start,    ept->node1a,ept->node1b,ept->node1c,
						 inten,wavel,"del");
		makrstim(start+dur,ept->node1a,ept->node1b,ept->node1c,
						-inten,wavel,"del");
	    }
	   }
	  }
	  break;

	case SINE:			/* stim for possibly all receptors */
					/* xloc is xmin; yloc is xmax. */
	  halfy = 1000;
          siney = 0;
          if (speriod <= 0.0) speriod = 1;   /* spatial period from user */
          if (tfreq) {		/* drifting grating */

                 double timres,tperiod,tincr,stime;

	    timres = .001;
            tperiod = 1.0 / tfreq;	/* temporal period */
            tincr = timres * tperiod;	/* time incr = time res * t period */
	    if (tincr < RTIMINC) tincr = RTIMINC; /* time incr = recep timinc */
            for (stime=start; stime<(start+dur-tincr); stime += tincr) {
	      for (ept=(photrec*)elempnt; ept; ept=(photrec*)ept->next) 
	        if (ept->ctype == ROD || ept->ctype == CONE) {
		   xpos = ept->xpos;
		   ypos = ept->ypos;

				/* if the photorec is under grating */

	           if (inrect(xpos,ypos,yloc,xloc,1000.0,-1000.0)) {
		     theta = (xpos-xloc)/speriod + (stime-start)/tperiod;
                     sinten = inten * (1 + contrast * sin(2*MPI*theta));
		     makrstim(stime,      ept->node1a,ept->node1b,ept->node1c,
					         sinten,wavel,"del");
		     makrstim(stime+tincr,ept->node1a,ept->node1b,ept->node1c,
						-sinten,wavel,"del");
		   } /* if (inrect()) */
                  } /* if (ept->ctype==ROD... */
            }   /* for (stime= ; ; ) */
          }   /* if (tfreq) */
          else {		/* static grating */
	    for (ept=(photrec*)elempnt; ept; ept=(photrec*)ept->next) {
	      if (ept->ctype == ROD || ept->ctype == CONE) {
		 xpos = ept->xpos;
		 ypos = ept->ypos;
	 /*        fprintf  (stderr,"%g %g %g %g %g %g\n",
				xpos,ypos,yloc,xloc,1000.0,-1000.0); /* */
	         if (inrect(xpos,ypos,yloc,xloc,1000.0,-1000.0)) {
	           theta = (xpos-xloc) / speriod;
                   sinten = inten * (1 + contrast * sin(2*MPI*theta));
		   makrstim(start,    ept->node1a,ept->node1b,ept->node1c,
						 sinten,wavel,"del");
		   makrstim(start+dur,ept->node1a,ept->node1b,ept->node1c,
						-sinten,wavel,"del");
	         } /* if (inrect()) */
               }  /* if (ept->ctype==ROD... */
	     }   /* for (ept= ; ; ) */
          } /* else */
	  break;

	case RECT:			/* stim for possibly all receptors */
	  halfx = dia / 2;
	  halfx = ydist / 2;
          if (runyet) {
	  for (rpnt=(recep*)recpnt; rpnt; rpnt=(recep*)rpnt->next) {
	    if (rpnt->ctype == ROD || rpnt->ctype == CONE) 
	    if (inrect(rpnt->xloc,rpnt->yloc, xloc+halfx, xloc-halfx,
			yloc+halfy, yloc-halfy)) {
		makrstim(start,    rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						 inten,wavel,"del");
		makrstim(start+dur,rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						-inten,wavel,"del");
	    }
	   }
          }
	  else {
	   for (ept=(photrec*)elempnt; ept; ept=(photrec*)ept->next) {
	    if (ept->ctype == ROD || ept->ctype == CONE) 
	    if (inrect(ept->xpos,ept->ypos, xloc+halfx, xloc-halfx,
			yloc+halfy, yloc-halfy)) {
		makrstim(start,    ept->node1a,ept->node1b,ept->node1c,
						 inten,wavel,"del");
		makrstim(start+dur,ept->node1a,ept->node1b,ept->node1c,
						-inten,wavel,"del");
	    }
	  }
	 }
	  break;

	case SPOT: 
	  radius = dia / 2;
	  if (runyet) {			/* look for existing photoreceptors */
	   for (rpnt=(recep*)recpnt; rpnt; rpnt=(recep*)rpnt->next) {
	    if (rpnt->ctype == ROD || rpnt->ctype == CONE) {
	    if (incirc(rpnt->xloc,rpnt->yloc, xloc, yloc,radius)) {
		makrstim(start,    rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						inten,wavel,"del");
		makrstim(start+dur,rpnt->recnm1,rpnt->recnm2,rpnt->recnm3,
						-inten,wavel,"del");
	     }
	    }
	   }
	  }
	  else {
	   for (ept=(photrec*)elempnt; ept; ept=(photrec*)ept->next) {
	    if (ept->ctype == ROD || ept->ctype == CONE) {
	    if (incirc(ept->xpos,ept->ypos, xloc, yloc,radius)) {
		makrstim(start,    ept->node1a,ept->node1b,ept->node1c,
						inten,wavel,"del");
		makrstim(start+dur,ept->node1a,ept->node1b,ept->node1c,
						-inten,wavel,"del");
	    }
	   }
 	  }
	 }
	  break;
	 } 	/* switch (stype) */
	}	/* else  (!filuse) */

	stype = 0;		/* reset stype for next stim statement */

	break;	/* case STIM */


  }	/* switch  (param->type) */
}

#endif

/*----------------------------------------------------*/

void sortstimfile(void)

/* Sort the stimulus file after all stimulus events 
   have been put into it.
*/

{
#ifdef XSTIM
   static char ftemplate[] = "/usr/tmp/stimXXXXXX";
   static char stemp[30],tbuf[120];
   
  strcpy (stemp, ftemplate);
  mktemp (stemp);
  sprintf (tbuf,"sort +0 -1 +0n -1 +1n -2 +2n -3 +3n -4 %s > %s && mv %s %s",
					stimfile,stemp,stemp,stimfile);
  system (tbuf);
#endif
}

/*----------------------------------------------------*/

void incrpl(void)

/* Increment number of plots.
   Used between V[] expressions in plot statement,
    and also at end of plot statement.
*/

{
#ifndef XSTIMM
 if (++numplots < PLOTNODSIZ)
     plotnod[numplots].cnod1 = 0; 	/* zero at end */ 
#endif
}

/*----------------------------------------------------*/

void vplot(void)

/* read in data from "plot V[]"  with default max min
*/

{
  datum d1,d2,d3;
  int narg;
  Symbol *param;

 param  = (Symbol *)*pc++;
 narg = (int)*pc++;


 if (narg > 2) d3 = popm();
 else d3.val = NULLNOD;
 if (narg > 1) d2 = popm();
 else d2.val = NULLNOD;
 if (narg > 0) checktyp(d1 = popm());
 else d1.val = NULLNOD;
  
#ifndef XSTIMM
 if (narg>0) {
   plotnod[numplots].cnod1 = (int)d1.val;
   plotnod[numplots].cnod2 = (int)d2.val;
   plotnod[numplots].cnod3 = (int)d3.val;
 }
 switch (param->type) {
   case V:
	plotnod[numplots].pmod  = VREC;
	break;
   case I:
	plotnod[numplots].pmod  = IREC;
	break;
   case L:
	plotnod[numplots].pmod  = LREC;
	break;
   case FA0: case FA1: case FA2: case FA3: case FA4: case FA9:
   case FB0: case FB1: case FB2: case FB3: case FB4:
   case FC0: case FC1: case FC2: case FC3: case FC4:
   case G0: case G1: case G2: case G3: case G4: case G5: case G6:
   case G7:
   case G8:
	plotnod[numplots].pmod  = NRECA0 + (param->type - FA0);
	break;
   case S:
 	plotnod[numplots].spnt  = (char *)d1.sym;
	plotnod[numplots].pmod  = SREC;
	break;
   case CABLE:
	plotnod[numplots].pmod  = CABLE;
 	plotnod[numplots].cnod2 = (int)(d2.val * CABLFRAC);
	break;
 }
#endif
}

/*------------------------------------------------------*/

void vplotm(void)

/* read in max, min data from "plot V() max min" statement
*/

{
  datum d1,d2;

 checktyp(d2 = popm());
 checktyp(d1 = popm());
 
#ifndef XSTIMM
 plotnod[numplots].pymax  = d1.val;
 plotnod[numplots].pymin  = d2.val;
#endif
}

/*------------------------------------------------------*/

void xrecord(void)

/* allow user to record from nodes directly */

{
    datum d1,d2,d3;
    Symbol *param;
    int nod1, nod2, nod3, recmod, narg;
    double record(int cnod1, int cnod2, int cnod3, int pmod);
#ifdef XMOD
    double reccable(int elemnum, double fracdist);
#endif

 param  = (Symbol *)*pc++;
 narg = (int) *pc++;

 switch (param->type) {
   case V:
	recmod = VREC;
	break;
   case I:
	recmod = IREC;
	break;
   case L:
	recmod = LREC;
	break;
   case FA0: case FA1: case FA2: case FA3: case FA4: case FA9:
   case FB0: case FB1: case FB2: case FB3: case FB4:
   case FC0: case FC1: case FC2: case FC3: case FC4:
   case G0: case G1: case G2: case G3: case G4: case G5: case G6: case G7:
   case G8:
	recmod = NRECA0 + (param->type - FA0);
	break;
   case CABLE:
	recmod = CABLE;
	break;
 }
   if (narg > 2) d3 = popm();				/* get node */
   else d3.val = NULLNOD;
   if (narg > 1) d2 = popm();
   else d2.val = NULLNOD;
   checktyp(d1 = popm());
   if (recmod==CABLE) {
#ifdef XMOD
	double fracdist;
	int elemnum;

     fracdist = d2.val;
     elemnum = (int)d1.val;
     d1.val = reccable(elemnum,fracdist);
#else
     d1.val = 0.0;
#endif
   }
   else {
#ifndef XSTIMM
     nod1 = (int)d1.val;
     nod2 = (int)d2.val;
     nod3 = (int)d3.val;
     d1.val = record (nod1,nod2,nod3,recmod);
#else
     d1.val = 0.0;
#endif
   }
   d1.vtype = NUMBER;
   push(d1);
}

/*------------------------------------------------------*/

void grph(void)

/* make a graph */
{
  datum d1,d2;
  Symbol *param;
  static int narg,i,n,npops;
  static double xval,yval[PLOTNODSIZ];
  double setvar(char *str);

 param  = (Symbol *)*pc++;
 narg = (int) *pc++;				/*  (x,y1,y2,...,yn) */
 if (param == 0) {				/* get arguments for graph */
 	n = narg - 1;
	if (n >= PLOTNODSIZ) n=PLOTNODSIZ-1;
	if (n > numplots) n = numplots;
	for (i=n-1; i>=0; i-- ) {
	   d2 = popm();
	   yval[i] = d2.val;
	}	
 	checktyp(d1 = popm());
	xval = d1.val;
#ifndef XSTIMM
	for (i=0; i<n; i++) {
	   mplot (yval[i],xval,n,i);
	}
#endif
 }
 else {
 vidmode = (int)setvar("vidmode");
 switch (param->type) {
   case X:			/* bug: can't use for indiv separate graphs */
   case Y:    npops = 2;
	      break;
   case SIZE: npops = 1;
	      break;
   case INIT:
   case RESTART:
	      npops=0;
	      break;
   case PEN:
   case CHAR:
   case CCHAR:
	      npops = 0;
	      break;
 }

 if (npops==2) checktyp(d2 = popm());
 if (npops>=1) checktyp(d1 = popm());

#ifndef XSTIMM
 switch (param->type) {
   case X:			/* bug: can't use for indiv separate graphs */
	numplots = 0;		/* number of plots for this graph */
 	plotnod[numplots].pxmax  = d1.val;
 	plotnod[numplots].pxmin  = d2.val;
 	plotnod[numplots].xrange = d1.val - d2.val;
	break;

   case Y:
 	plotnod[numplots].pymax  = d1.val;
 	plotnod[numplots].pymin  = d2.val;
 	plotnod[numplots].yrange = d1.val - d2.val;
	plotnod[numplots++].pmod  = GREC;
	break;

  case INIT:
	varcopy();
	plotinit(numplots);
	break;

  case RESTART:
	varcopy();
	plotrst(numplots);
	break;

  case PEN:
	if (narg>0) { 			/* If on line by itself, */
	  for (i=narg; i>0; i-- ) {	/* set plot colors before graph init */
 	   checktyp(d1 = popm()); 	/* Can also be used w/graph*/ 
	   plotpen ((int)d1.val,i-1);
	  }	
	}
	else {				/* If after "max min" then set color */
 	   checktyp(d1 = popm());	/*  for this plot only */
	   plotpen ((int)d1.val,numplots);
	}	    
        break;

  case CHAR:
	if (narg>0) {			/* If on line by itself, */
	  for (i=narg; i>0; i-- ) {	/*  set plot chars before graph init */
 	   checktyp(d1 = popm()); 	/*(Can also be used w/graphing)*/
	   plotchar ((int)d1.val,LINES,i-1);
	  }	
	}
	else {				/* If after "max min" then set char */
 	   checktyp(d1 = popm());	/*  for this plot only */
	   plotchar ((int)d1.val,LINES,numplots);
	}	    
	break;

  case CCHAR:
	if (narg>0) {			/* If on line by itself, */
	  for (i=narg; i>0; i-- ) {	/*  set plot chars before graph init */
 	   checktyp(d1 = popm());	/*(Can also be used w/graphing)*/
	   plotchar ((int)d1.val,NOLINES,i-1);
	  }	
	}
	else {				/* If after "max min" then set char */
 	   checktyp(d1 = popm());	/*  for this plot only */
	   plotchar ((int)d1.val,NOLINES,numplots);
	}	    
	break;
  case SIZE:
	   plotcsiz (d1.val,numplots);
	break;

  }  /* switch */
#endif		/* ! XSTIMM */

 } /* else */
}

/*------------------------------------------------------*/

void dispnod(void)
{
  datum d1,d2,d3;
  static int narg1,narg2;
  static int n1a,n1b,n1c,n2a,n2b,n2c;
  static int na,nb,nc;
  Symbol *param,nulparm;
  static int disptype,dcolor = NULLNOD, excl=0, hide=0;
  static double cline=0,dscale=1.0,stime=0;
  static int elemtype=0;
  static int exceptype=0,except=0;
  static double xcalib=.95,ycalib=.05;

  param = (Symbol *)*pc++;
  if (param==0) {
  	nulparm.type = CONNECT;
	param = &nulparm;
  }
#ifdef XMOD
  varcopy();
  if (setdebug) debug = setdebug;
  if (setdebgz) debugz = setdebgz;
  if (!runyet || elempnt) {
	findconnect();
  }
#endif
  switch (param->type) {

  case ONLY:  excl = 1;
	      break;

  case MATCHING:
  case ELEMENT:
  case RANGE:
  case CONNECT:
 
  disptype = param->type; 
  narg2 = (int)*pc++;
  if (narg2) {
    if (narg2 > 2) d3 = popm();		/* args for node2 are first */
    else d3.val = NULLNOD;		/*  because they're on top of stack */
    if (narg2 > 1) d2 = popm();
    else d2.val = NULLNOD;
    checktyp(d1 = popm());
    n2a = (int)d1.val;
    n2b = (int)d2.val;
    n2c = (int)d3.val;
    if (n2a<0) n2a = NULLNOD;
    if (n2b<0) n2b = NULLNOD;
    if (n2c<0) n2c = NULLNOD;
  }
  else n2a=n2b=n2c=NULLNOD;

  narg1 = (int)*pc++;
  if (narg1) {
    if (narg1 > 2) d3 = popm();
    else d3.val = NULLNOD;
    if (narg1 > 1) d2 = popm();
    else d2.val = NULLNOD;
    checktyp(d1 = popm());
    if (except) {
       na = (int)d1.val;
       nb = (int)d2.val;
       nc = (int)d3.val;
       if (na<0) na = NULLNOD;
       if (nb<0) nb = NULLNOD;
       if (nc<0) nc = NULLNOD;
    }
    else {
       n1a = (int)d1.val;
       n1b = (int)d2.val;
       n1c = (int)d3.val;
       if (n1a<0) n1a = NULLNOD;
       if (n1b<0) n1b = NULLNOD;
       if (n1c<0) n1c = NULLNOD;
       na= nb= nc= NULLNOD;		/*  clear except node */
    }
  }
  else {				/* if node not specified: */
    if (except) na= nb= nc= NULLNOD;	/*  clear except node */
    else n1a=n1b=n1c=NULLNOD;   	/*  clear n1 when not except */
  }

  break;

  case COMPS:
  case CABLE:
  case SPHERE:
  case SYNAPSE:
  case CHAN:
  case ROD:
  case CONE:
  case GJ:
  case LOAD:
  case RESISTOR:
  case CAP:
  case GNDCAP:
  case BATT:
  case GNDBATT:
  case BUF:
  case NODE:
    if (except) exceptype = param->type;
    else         elemtype = param->type;
    break;

  case EXCEPT:  except = 1;
	      break;

  case XROT:				/* get rotation for display */
    checktyp(d1 = popm());
    xrot = -d1.val;
    break;

  case YROT:
    checktyp(d1 = popm());
    yrot = d1.val;
    break;

  case ZROT:
    checktyp(d1 = popm());
    zrot = d1.val;
    break;

  case SIZE:				/* get size of display window */
    checktyp(d1 = popm());
    dsize = d1.val;
    break;

  case DSCALE:				/* get display scale */
    checktyp(d1 = popm());
    dscale = d1.val;
    break;

  case COLOR:				/* get color */
    checktyp(d1 = popm());
    dcolor = (int)d1.val;
    break;

  case CALIBLIN:			/* get calib line */
    narg1 = (int)*pc++;
    if (narg1) {
      if (narg1 > 2) d3 = popm();		/* get possible "loc" */
      if (narg1 > 1) d2 = popm();
      checktyp(d1 = popm());
      xcalib = d1.val; 
      ycalib = d2.val; 
    }
    checktyp(d1 = popm());
    cline = d1.val;
    break;

  case STIM:				/* get stimulus time  */
    checktyp(d1 = popm());
    stime = d1.val;
    break;

  case HIDE:				/* make picture with hidden lines */
    hide = 1;
    break;

  case CENTER:
    narg1 = (int)*pc++;
    if (narg1) {
      if (narg1 > 2) d3 = popm();
      else d3.val = 0;
      if (narg1 > 1) d2 = popm();
      else d2.val = 0;
      checktyp(d1 = popm());
    }
    dxcent = -d1.val;
    dycent = -d2.val;
    dzcent = -d3.val;
    break;

  case RMOVE:
    narg1 = (int)*pc++;
    if (narg1) {
      if (narg1 > 2) d3 = popm();
      else d3.val = 0;
      if (narg1 > 1) d2 = popm();
      else d2.val = 0;
      checktyp(d1 = popm());
    }
    rxcent = d1.val;
    rycent = d2.val;
    rzcent = d3.val;
    break;
 
  case DISPLAY:
	if (!disptype) return;

#ifdef XMOD

   setrot(xrot,yrot,zrot,dxcent,dycent,dzcent,rxcent,rycent,rzcent,dsize); 

   if (disp_ray) {		/* move camera, not objects */
     initray(xrot,yrot,zrot,dxcent,dycent,dzcent,
			rxcent,rycent,rzcent,dsize); /* initialize ray-tracer */
   }

   set_icons();				/* set user-defined neural elem icons */

   if (disp & DSTIM) 
    if (disptype == STIM)
     if (stime){			/* display the stimulus */
	dispstim(stime,dscale*2.0);
        stime=0;
      }

   if (disp & DISP) {		/* if "disp" variable or "-d 1" has been set */
				/* first, translate, rotate, and scale: */
    if (hide) hidstart();

    else if (elemtype==COMPS) {
      initcomp();         /* make compartments */
      if (debug & 256) nocond = 1;
      if (!nocond) condense();
      switch(disptype) {
       case MATCHING:
       case CONNECT: if (disp & (DCOMP|DCONN))
			 dcomp (n1a,n1b,n1c,elemtype,exceptype,
					na,nb,nc,dcolor,dscale,hide,excl);

/*		 else  dcomp2 (n1a,n1b,n1c,n2a,n2b,n2c,elemtype,exceptype,
					na,nb,nc,dcolor,dscale,hide); */
		 break;

      }  /* switch */
    }  /* elemtype==COMPS) */

    else	/* elemtype != COMP */
    if (elemtype==NODE) {
        ncdrnod (n1a,n1b,n1c,elemtype,exceptype,na,nb,nc,dcolor,dscale);
    }  

    else	/* elemtype is CABLE, SPHERE, SYNAPSE, etc. */
    switch (disptype) {

     case MATCHING:
     case CONNECT: if (narg2==0) ncdisp (n1a,n1b,n1c,elemtype,exceptype,
					na,nb,nc,dcolor,dscale,hide,excl);
		 else   ncdispc (n1a,n1b,n1c,n2a,n2b,n2c,elemtype,exceptype,
					na,nb,nc,dcolor,dscale,hide);
		 break;
     case RANGE:    ncdispn (n1a,n1b,n1c,n2a,n2b,n2c,elemtype,exceptype,
					na,nb,nc,dcolor,dscale,hide,excl);
		 break;
     case ELEMENT: ncdispe (n1a,dcolor,dscale,hide);
		 break; 
    }
    if (disp) { 
      if (cline) drcalib (xcalib,ycalib,cline,dsize,dcolor);
      cline=0;
    }

    disptype = 0;
    elemtype = 0;
    except = exceptype = 0;
    rxcent=rycent=rzcent=0;
    na=nb=nc=NULLNOD;
    dscale = 1.0;
    dcolor = 0;
    excl = 0;
    if (hide) hidstop();
   }		/* if (disp & DISP) */

#endif		/* XMOD */

  break;     /* case DISPLAY */
 } 	   /* switch (param) */
}


/*------------------------------------------------------*/

#define NUMDIM 3			/* number of node dimensions */

void foreacode(void)

/* execute a set of statements for one of three conditions (options):

    1) For every element, or each element of a certain type (e.g. "cable")

    2) For every element, or each element of a certain type (e.g. "cable")
         that matches a node specification.  

    3) for each node that matches a node specification.
*/
         
{
	datum d;
	Inst *savepc = pc;
	int i,narg,match,efilt;
	int arg[NUMDIM];
	int val[NUMDIM];
        double *varp[NUMDIM];
	double *elemvar;
	short *typ,*vtyp;
        node *npnt;
        Symbol *eparam;
	elem *epnt;

    eparam = (Symbol *)*pc++;		/* get element type */ 
    efilt = 0;
    if (eparam) switch (eparam->type) {

     case ELEMENT:
			efilt = 1;
			break;
     case CABLE:
     case SPHERE:
     case SYNAPSE:
     case CHAN:
     case ROD:
     case CONE:
     case GJ:
     case LOAD:
     case RESISTOR:
     case CAP:
     case GNDCAP:
     case BATT:
     case GNDBATT:
     case BUF:
			efilt = 2;
			break;	
    }

    if (efilt) {
       getvar(&elemvar,&typ,&vtyp);  /* var pointer for element */
       if (*typ==UNDEF) *typ = VAR;
       *vtyp = NUMBER;
    }
    
    narg = (int)*pc++;			/* number of snode args to expect */

   if (narg) {
    if (narg>NUMDIM) narg = NUMDIM;
    for (i=0; i<NUMDIM; i++) {
       varp[i] = NULL;
       val[i]  = NULLNOD;
       arg[i]  = (int)*pc++;
    }
#ifndef XSTIMM
    execute (savepc+7);			/* evaluate node descriptors */
#endif

   for (i=narg-1; i>=0; i--) {
         if (arg[i]) {
	    getvar(&varp[i],&typ,&vtyp);/*var pointer for node dim */
            if (*typ==UNDEF) *typ = VAR;
            *vtyp = NUMBER;
	 }
	 else {
           checktyp(d = popm()); 	/*else get value of node dim */
	   val[i] = (int)d.val;
         }
   }
  }   /* if (narg) */


#ifndef XSTIMM
 if (efilt) {
   for (epnt=elempnt; epnt; epnt=epnt->next) {	/* search all elems */
     if (efilt==2) if (epnt->ctype!=eparam->type) continue;
     *elemvar = epnt->elnum;
     if (narg) {
       match = 1;
       npnt = epnt->nodp1;		/* test first node */
       if (npnt) 
       for (match=1,i=0; i<narg; i++) {
	if (!arg[i]) 			/* if dimension was given a value */
	  switch (i) {
	   case 0: if (val[i] != npnt->nodenm1) match = 0; break; 
	   case 1: if (val[i] != npnt->nodenm2) match = 0; break; 
	   case 2: if (val[i] != npnt->nodenm3) match = 0; break; 
	  }	
       }
      if (!match) {
       match = 1;
        npnt = epnt->nodp2;		/* test second node */
        if (npnt) 
        for (match=1,i=0; i<narg; i++) {
	 if (!arg[i]) 			/* if dimension was given a value */
	   switch (i) {
	    case 0: if (val[i] != npnt->nodenm1) match = 0; break; 
	    case 1: if (val[i] != npnt->nodenm2) match = 0; break; 
	    case 2: if (val[i] != npnt->nodenm3) match = 0; break; 
	   }	
        }
        if (!match) continue;
       }  /* if (!match) */

       for (i=0; i<narg; i++) {
	if (arg[i])			/* If dimension was given blank var, */
	  switch (i) {			/*   set variable for this iteration */
	  case 0: *varp[i] = npnt->nodenm1; break; 
	  case 1: *varp[i] = npnt->nodenm2; break; 
	  case 2: *varp[i] = npnt->nodenm3; break; 
	  }	
       }
     }    /* if (narg) */

    if (!forexec(savepc+5)) break;

   }  /* for (epnt;;) */
 }  /* if (efilt) */



 else 		/* no element, just node to match */
   for (npnt=nodepnt; npnt; npnt=npnt->next) {	/* search all nodes */
      for (match=1,i=0; i<narg; i++) {
	if (!arg[i]) 			/* if dimension was given a value */
	  switch (i) {
	   case 0: if (val[i] != npnt->nodenm1) match = 0; break; 
	   case 1: if (val[i] != npnt->nodenm2) match = 0; break; 
	   case 2: if (val[i] != npnt->nodenm3) match = 0; break; 
	  }	
     }
     if (!match) continue;

     for (i=0; i<narg; i++) {
	if (arg[i])			/* If dimension was given blank var, */
	  switch (i) {			/*   set variable for this iteration */
	  case 0: *varp[i] = npnt->nodenm1; break; 
	  case 1: *varp[i] = npnt->nodenm2; break; 
	  case 2: *varp[i] = npnt->nodenm3; break; 
	  }	
     }

    if (!forexec(savepc+5)) break;

    }  /* for (npnt;;) */

#endif
  if (!returning)
	pc = *((Inst **)(savepc+6));	/* next stmt */
}

/*------------------------------------------------------*/

int forexec(Inst *body)
{
  execute(*((Inst **)(body)));		/* body of element, node loop */
  if (stopping) {
	if (returning)
		return(0);
	stopping = 0;
	if (breaking) {
		breaking = 0;
		return (0);
	}
		continuing = 0;
   }
   return (1);
}

/*------------------------------------------------------*/

void findconnect(void)

/* Find and/or make all nodes that connect to all cables.
 */

{
   elem *epnt;

#ifndef XSTIMM
 for (epnt=elempnt; epnt; epnt=epnt->next)   /* check nodes for element*/
   {
     checkelemnode(epnt);
   }
#endif

/*
if (debug & 1)
 for (npnt=nodepnt; npnt; npnt=npnt->next)
  fprintf (stderr,"node %d %d conn %d\n",npnt->nodenm1,npnt->nodenm2, 
		(npnt->elemlst ? npnt->elemlst->num : NULLNOD));

 for (epnt=elempnt; epnt; epnt=epnt->next)
  fprintf (stderr,"element %d node1 %d %d %d node2 %d %d %d\n",
		epnt->elnum,epnt->node1a,epnt->node1b,epnt->node1c,
			    epnt->node2a,epnt->node2b);

	/* */
}

/*------------------------------------------------------*/

void checknod (elem *epnt)

/* Find the nodes that an element connects to;
   set up the element's node pointers.
   If node doesn't exist, create a new node.
   Don't set node's element pointer because element
   has not been fully defined yet.
*/

{
   node  *npnt,*newnod;
   int found;

#ifndef XSTIMM
    if (!epnt) return;
    found = 0;
    if (epnt->node1a != NULLNOD) {
      if ((npnt=findnode(epnt->node1a,epnt->node1b,epnt->node1c,NULL))) {
	   epnt->nodp1 = npnt;			/* set node pointer */
/*	   setnodlst(npnt,epnt); 		/* set elem pointer */
	   found = 1;
      }
      if (! found) {				/* if not, then make it */
 	    newnod=maknod(epnt->node1a,epnt->node1b,epnt->node1c);
/* 	    setnodlst(newnod=maknod(
			epnt->node1a,epnt->node1b,epnt->node1c),epnt); /* */
	    epnt->nodp1 = newnod;		/* set node pointer */
      }
    }	/* if (epnt->node1a != NULLNOD) */

    found = 0;
    if (epnt->node2a == NULLNOD) return;	/* ignore if no second node */
      if ((npnt=findnode(epnt->node2a,epnt->node2b,epnt->node2c,NULL))) {
	   epnt->nodp2 = npnt;			/* set node pointer */
/*	   setnodlst(npnt,epnt);	 	/* set elem pointer */
	   found = 1;
      }
    if (! found) {				/* if not, then make it */
 	    newnod=maknod(epnt->node2a,epnt->node2b,epnt->node2c);
/* 	    setnodlst(newnod=maknod(
			epnt->node2a,epnt->node2b,epnt->node2c),epnt); /* */
	    epnt->nodp2 = newnod;		/* set node pointer */
    }

/* fprintf (stderr,"checknod el %d nod1 %d nod2 %d\n",
			epnt->elnum,epnt->node1a,epnt->node2a); /* */

/* fprintf (stderr,"checknod elem %d  n1 %d n2 %d\n",
		epnt->elnum, epnt->nodp1,epnt->nodp2); /* */

#endif		/* XSTIMM */
}

/*------------------------------------------------------*/

void checkelemnode (elem *epnt)

/* 
   Find the nodes that connect to an element;
   for each node, set up the element pointer in the node's list,
   and set up the element's node pointers.
   If node doesn't exist, create a new node.

*/

{
   node  *npnt,*newnod;
   int found;

#ifndef XSTIMM
    if (!epnt) return;
    found = 0;
    if (epnt->node1a != NULLNOD) {
      if ((npnt=findnode(epnt->node1a,epnt->node1b,epnt->node1c,NULL))) {
	   epnt->nodp1 = npnt;			/* set node pointer */
	   setnodlst(npnt,epnt); 		/* set elem pointer */
	   found = 1;
      }
      if (! found) {				/* if not, then make it */
 	    setnodlst(newnod=maknod(
			epnt->node1a,epnt->node1b,epnt->node1c),epnt);
	    epnt->nodp1 = newnod;		/* set node pointer */
      }
    }	/* if (epnt->node1a != NULLNOD) */

    found = 0;
    if (epnt->node2a == NULLNOD) return;	/* ignore if no second node */
      if ((npnt=findnode(epnt->node2a,epnt->node2b,epnt->node2c,NULL))) {
	   epnt->nodp2 = npnt;			/* set node pointer */
	   setnodlst(npnt,epnt);	 	/* set elem pointer */
	   found = 1;
      }
    if (! found) {				/* if not, then make it */
 	    setnodlst(newnod=maknod(
			epnt->node2a,epnt->node2b,epnt->node2c),epnt);
	    epnt->nodp2 = newnod;		/* set node pointer */
    }

/* fprintf (stderr,"checknod el %d nod1 %d nod2 %d\n",
			epnt->elnum,epnt->node1a,epnt->node2a); /* */

/* fprintf (stderr,"checknod elem %d  n1 %d n2 %d\n",
		epnt->elnum, epnt->nodp1,epnt->nodp2); /* */

#endif		/* XSTIMM */
}

/*------------------------------------------------------*/

void setnodlst(node *npnt, elem *cpnt)
              
/* Add an element to a node's element list.  */

{
   int found;
   conlst *lpnt;

#ifndef XSTIMM
  if (!npnt) {
	fprintf (stderr,"setnodlst: invalid node pointer\n");
        return;
  }
  found = 0;
  for (lpnt=npnt->elemlst; lpnt; lpnt=lpnt->next) {
    if (lpnt->conpnt == (conn*)cpnt) found = 1;
  }

  if (! found) 
    maklst(&npnt->elemlst,(conn *)cpnt);
#endif
}

/*------------------------------------------------------*/

void unsetnodlst(node *npnt, elem *cpnt)
              
/* delete an element from a node's element list */

{
   int found;
   conlst *lpnt;

#ifndef XSTIMM
  if (!npnt) {
	fprintf (stderr,"unsetnodlst: invalid node pointer\n");
        return;
  }
  found = 0;
  for (lpnt=npnt->elemlst; lpnt; lpnt=lpnt->next) {
    if (lpnt->conpnt == (conn*)cpnt) {
      found = 1;
      break;
    }
  }

  if (found) dellst(&npnt->elemlst,lpnt);
#endif
}

/*------------------------------------------------*/
    
void modrun(void)
{
  datum d1;
  Symbol *mtyp;

if (debug & 1) fprintf (stderr,"modrun start\n");

  mtyp = (Symbol *)*pc++;
  varcopy();

  if (setdebug) debug  = setdebug;	/* "-y n" overrides "debug=n" */
  if (setdebgz) debugz = setdebgz;	/* "-z n" overrides "debugz=n" */

  switch (mtyp->type)           /* do pops */
    {
     case RUN:  break;
     case STEP:  checktyp(d1=popm());
                break;
    }

#ifdef XMOD
  if (!runyet) {
        initchan();			/* set up sequential-state chans */
  }
  if (!runyet || elempnt) {
	findconnect();
  }
  switch (mtyp->type)
    {

     case RUN:  actrun(0.0);
		runyet = 0;
		break;

     case STEP: actrun(d1.val);
		break;
    
    }
#else
    xtime += d1.val;         /* if XSTIM, set the time correctly, at least */
    timeptr->val = xtime;
#endif
}

/*------------------------------------------*/

char *prnode (int n1, int n2, int n3) {

   static char nbuf[40] = {0};
   static char undefd[] = "undefined";
   int test;

   nbuf[0] = 0;
   test = 0;
   if (n1 != NULLNOD) test |= 1;
   if (n2 != NULLNOD) test |= 2;
   if (n3 != NULLNOD) test |= 4;

   switch (test) {

     default:
     case 0: 	sprintf (nbuf,"%s",undefd);
		break;
     case 1: 	sprintf (nbuf,"[%d]",n1);
		break;
     case 3: 	sprintf (nbuf,"[%d][%d]",n1,n2);
		break;
     case 7: 	sprintf (nbuf,"[%d][%d][%d]",n1,n2,n3);
		break;
   }
   return (nbuf);
}

/*------------------------------------------*/

