/* segment initchan in program nc */

/* sets up channel parameters */

#include <stdio.h>
#include "nc.h"
#include "y.tab.h"
#include "ncomp.h"
#include "ncsub.h"
#include "control.h"

double alpham, betam, alphah, betah;	/* used by runcomp() in "ncomp.c" */
double alphan, betan, alphad, betad;
double alphac, betac;

double m1,m2,h1,h2;			/* used by runcomp() in "ncomp.c" */
double n1,n2,d1,d2;
double c1,c2;
double kca1,kca2;

static int nchantyp=NCHANTYP;		/* number of chan types to set up */

chanpar natypes[NCHANTYP]={0};		/* sequential-state channel params */
chanpar ktypes[NCHANTYP]={0};
chanpar catypes[NCHANTYP]={0};

typedef struct CHANTABLE {
	float am [RATESIZ+1];	 	/* Lookup tables for rate constants. */
	float bm [RATESIZ+1];    	/* Used for all channel types. */
	float ah [RATESIZ+1];    
	float bh [RATESIZ+1];    
	float an [RATESIZ+1];    
	float bn [RATESIZ+1];    
	float ad [RATESIZ+1];    
	float bd [RATESIZ+1];    
	float ac [RATESIZ+1];    
	float bc [RATESIZ+1];    
	float m1tab [RATESIZ+1];	/* Lookup tables for direct comp. */
	float m2tab [RATESIZ+1];	/*  of m,h, or n only */
	float h1tab [RATESIZ+1];
	float h2tab [RATESIZ+1];    
	float n1tab [RATESIZ+1];    
	float n2tab [RATESIZ+1];    
	float d1tab [RATESIZ+1];    
	float d2tab [RATESIZ+1];    
	float c1tab [RATESIZ+1];	/* calcium (not implemented yet) */ 
	float c2tab [RATESIZ+1];    
} chantable;

chantable ratetab[NCHANTYP] = {0};	/* Rate tables for all chan types */
					/* Uses 4096 doubles per type */

static Symbol *ampnt,*bmpnt,*ahpnt,*bhpnt;  /* user funcs */
static Symbol *anpnt,*bnpnt,*adpnt,*bdpnt;
static Symbol *acpnt,*bcpnt;

#define EXPSIZ 1000
#define EXPEXPSIZ 200

double exptab [EXPSIZ+1]={0};    
double expexptab [EXPEXPSIZ+1]={0};    

#define LOG3 1.098612289

#ifdef __cplusplus
extern "C" {
#endif

double exp(double);

#ifdef __cplusplus
}
#endif

double ncabs(double x);
double callfunc(Symbol *funcp, int npar, double par1, double par2);
double akcacalc(double v, double ca, double tau, double d1, double k1);
double bkcacalc(double v, double ca, double tau, double d2, double k2);
double qfac(void);
void makna1();
void makna2();
void makk1();
void makexp();

/*----------------------------------------*/
 
void initchan(void)

/* This subroutine sets up all sequential-state channel tables
    at the beginning of a "run".  To set up additional types,
    copy the "makna1()" or "makk1()" routine and modify it, then
    call it from this routine.  Make sure to set "nchantyp"
    correctly.
*/

{
   int i;

 for (i=0; i<NCHANTYP; i++) {
   natypes[i].ctype = NA;
   natypes[i].numstate = 0;
   ktypes[i].ctype = K;
   ktypes[i].numstate = 0;
   catypes[i].ctype = CA;
   catypes[i].numstate = 0;
 }

 makna1();			/* set up Na sequential states */
 makna2();
 makk1();			/* set up K sequential states */

 makexp();			/* set up exp() table for synapses */
}

/*----------------------------------------*/

void makexp(void)
/* set up exponential function lookup table.  */
{
   int i;
   double exp(double);

 for (i=0; i<EXPSIZ+1; i++)
   exptab[i] = exp((double)i/EXPSIZ); 
 for (i=0; i<EXPEXPSIZ+1; i++)
   expexptab[i] = exp((double)(i-EXPEXPSIZ/2)); 
}

/*----------------------------------------*/

double lkexp(double val)
              

/* lookup an exp() value in the range -100 to 100. */

/* On some machines, it's faster to use this 
    interpolation scheme than use the exp() 
    routine supplied with the library.
*/

{
  register int i,r;
  double rr,x,exp(double);
  
  if      (val > 100.0) val = 100.0; 
  else if (val <-100.0) val = -100.0; 
  i = (int)val;				/* integer for lookup in expexptab[] */
  if (val<0) i--;
  rr = (val - i) * EXPSIZ;		/* remainder for lookup in exptab[] */
  r = (int)(rr + 0.5);
  rr -=  r;
  x = exptab[r] * (1.0+(1.0/EXPSIZ)*rr); /* use linear interpolation */
  return x*expexptab[i+EXPEXPSIZ/2]; 
}

/*----------------------------------------*/

double amcalc(double v, int type)

/* Calculate am as a function of mv.
   Type O and 1 are calculated exactly as in HH (1952) paper.
*/

{
   double am,x,y;

  switch (type) {
  case 0:
  case 1:
  default:
/*    y = 0.1 * (v+25.); 			/* the old way */
    y = -0.1 * (v+40.);				/* the modern way */
    x = exp (y) - 1.;
    if (ncabs(x) > 1e-5)			/* singularity at v = -40 mv */
       am = y / x;
    else
       am = 1;
    break;
  }
  return am;
}

/*----------------------------------------*/

/* Vo is used here to define the resting potential
   that Hodgkin and Huxley (1952) used as a base 
   for their rate constant functions.  Vo is only
   used to normalize the modern definition of 
   membrane potential to their definition.
   "Vo" should not be modified as it refers only to
   the antiquated definition of membrane voltage: */

#define Vo (-65.)

double bmcalc(double v, int type)

/* Calculate bm as a function of mv,
   exactly as in HH (1952) paper.
*/
{
  double val=0.0;

  switch (type) {
  case 0:
  case 1:
  default:
/*    val =  4 * exp (v / 18.);		/* the old way */

    val =  4 * exp ((v-Vo) / -18.);		/* the modern way */
    break;
  }
 return val;
}

/*----------------------------------------*/

double ancalc(double v, int type)
           
/* Type 0 and 1 are calculated as a function of mv,
      exactly as in HH (1952) paper.
*/

{
   double an,x,y;

  switch (type) {
  case 0:
  case 1:
  default:
/*    y = 0.1 * (v+10.);			/* the old way */

    y = -0.1 * (v+55.);				/* the modern way */
    x = exp (y) - 1.;
    if (ncabs(x) > 1e-5)			/* singularity at v = -55 mv */
       an = 0.1 * y / x;
    else
       an = 0.1;
    break;

  case 3:					/* type A potassium chan */
    y = -0.1 * (v+90.);				/* the modern way */
    x = exp (y) - 1.;
    if (ncabs(x) > 1e-5)			/* singularity at v = -90 mv */
       an = 0.01 * y / x;
    else
       an = 0.01;
    break;
  }
  return an;
}

/*----------------------------------------*/

double bncalc(double v, int type)

/* Calculate bm as a function of mv,
   exactly as in HH (1952) paper.
*/

{
   double val= 0.0;

  switch (type) {
  case 0:
  case 1:
  default:
/*    val = 0.125 * exp (v / 80.);		/* the old way */

    val = 0.125 * exp ((v-Vo) / -80.);	/* the modern way */
    break;
  case 3:
    val =  0.4 * exp ((v+30) / -18.);		/* type A potassium chan */
    break;
  }
 return val;
}

/*----------------------------------------*/

double akcacalc(double v, double ca, double tau, double d1, double k1)

/* Alpha for calcium-activated potassium channel,
      from Hines, 1989 and Moczydlowski and Latorre (1983).

   Does not include effect of timestep and temperature.
   Return units of per ms.
*/

{
   double alpha,t,vm;

    vm = v * 1000.0;
    if (ca==0.0) ca = 1e-8;
    t = dkcatau / tau;
    alpha = t / (1.0 + k1/ca * exp (d1* -vm/10.));
    return alpha;
}

/*----------------------------------------*/

double bkcacalc(double v, double ca, double tau, double d2, double k2)

/* Beta for calcium-activated potassium channel,
      from Hines, 1989 and Moczydlowski and Latorre (1983).

   Does not include effect of timestep and temperature.
   Return units of per ms.
*/

{
   double beta,t,vm;

    vm = v * 1000.0;
    if (ca==0.0) ca = 1e-8;
    t = dkcatau / tau;
    beta = t / (1.0 + ca / (k2 * exp (d2* -vm/10.)));
    return beta;
}

/*----------------------------------------*/

double ahcalc(double v, int type)
           
/* Calculate ah as a function of mv,
   exactly as in HH (1952) paper.
*/
{
   double val=0.0;

  switch (type) {
  case 0:
  case 1:
  default:
/*    val =  0.07 * exp (v / 20.);		/* old way */
    val =  0.07 * exp ((v-Vo) / -20.);	/* modern way */
    break;
  }
 return val;
}

/*----------------------------------------*/

double bhcalc(double v, int type)
           
/* Calculate bh as a function of mv,
   exactly as in HH (1952) paper.
*/
{   double y,val=0.0;

  switch (type) {
  case 0:
  case 1:
  default:
/*    y = 0.1 * (v+30.);			/* old way */
    y = -0.1 * (v+35.);				/* modern way */
    val =  (1.0 / (exp (y) + 1.0));
    break;
  }
 return val;
}

/*----------------------------------------*/

double adcalc(double v, int type)
           
/* Calculate ad as a function of mv.
   This is "h" (inactivation gate) for
   fast inactivating potassium current.
   */
{
   double val=0.0;

  switch (type) {
  case 3:
  default:
    val = (0.01 * exp (-0.05*(v+70)));
    break;
  }
 return val;
}

/*----------------------------------------*/

double bdcalc(double v, int type)

/* Calculate bd as a function of mv.
   This is "h" (inactivation gate) for
   fast inactivating potassium current.
*/

{   double y,val=0.0;

  switch (type) {
  case 3:
  default:
    y = -0.1 * (v+40.);
    val = (0.1 / (exp (y) + 1.0));
    break;
  }
 return val;
}

/*----------------------------------------*/

double accalc(double v, int type)

/* Calculate ac as a function of mv,
*/

{
   double ac,x,y;
 
  switch (type) {
  case 0:
  case 1:
  default:
    y = -0.1 * (v+15.);	
    x = exp (y) - 1.;
    if (ncabs(x) > 1e-5)			/* singularity at v = -15 mv */
       ac = y / x;
    else
       ac = 1;
    break;
  }
    return ac;
}

/*----------------------------------------*/

double bccalc(double v, int type)
           
/* Calculate bc as a function of mv,
*/

{
  double val=0.0;

  switch (type) {
  case 0:
  case 1:
  default:
    return (4 * exp ((v-Vo) / -18.));		/* the modern way */
    break;
  }
 return val;
}

/*----------------------------------------*/

double amfunc (double v, int type)

/* Call a user-defined function for calculating am. */

{
   return (callfunc (ampnt,2,v,(double)type));
}

/*----------------------------------------*/
double bmfunc (double v, int type)
{
   return (callfunc (bmpnt,2,v,(double)type));
}

/*----------------------------------------*/
double ahfunc (double v, int type)
{
   return (callfunc (ahpnt,2,v,(double)type));
}

/*----------------------------------------*/
double bhfunc (double v, int type)
{
   return (callfunc (bhpnt,2,v,(double)type));
}

/*----------------------------------------*/
double anfunc (double v, int type)
{
   return (callfunc (anpnt,2,v,(double)type));
}

/*----------------------------------------*/
double bnfunc (double v, int type)
{
   return (callfunc (bnpnt,2,v,(double)type));
}
/*----------------------------------------*/
double adfunc (double v, int type)
{
   return (callfunc (adpnt,2,v,(double)type));
}

/*----------------------------------------*/
double bdfunc (double v, int type)
{
   return (callfunc (bdpnt,2,v,(double)type));
}

/*----------------------------------------*/
double acfunc (double v, int type)
{
   return (callfunc (acpnt,2,v,(double)type));
}

/*----------------------------------------*/
double bcfunc (double v, int type)
{
   return (callfunc (bcpnt,2,v,(double)type));
}

/*----------------------------------------*/

double qfac (void)

/* Return rate constant multiplier which 
   depends on temperature. */

{
 return timinc * 1000.0 * exp(LOG3 * (tempcel - 6.3) / 10.0); /* Q10 of 3 */
}

/*----------------------------------------*/

void maktables(double timestep)

/* Make table for rate constants am, bm, ah, bh, an, bn, ac, bc.
   Rate constants are taken from Hodgkin and Huxley (1952). 
   Hodgkin-Huxley equations for rate constants are calculated
   from the negative of the difference between the voltage and
   the resting potential, and are calibrated in units of per msec.
   The tables are calculated here from -100 to 100 mv (0 means -100 mv,
   200 means 100 mv), and are calibrated in units of per second.
   A final multiplication by "timinc" calibrates the tables in 
   terms of the time step.  Therefore, the tables need to be
   recalculated every time the time step changes.
*/

{
  static double oldtiminc=0.0;
  double v,tval,exp(double);
  int vn,t;
  chantable *rtp;
  Symbol *f,*lookup();
  double (*amf)(double,int),(*bmf)(double,int);
  double (*ahf)(double,int),(*bhf)(double,int);
  double (*anf)(double,int),(*bnf)(double,int);
  double (*adf)(double,int),(*bdf)(double,int);
  double (*acf)(double,int),(*bcf)(double,int);

  float *am,*bm,*ah,*bh,*an,*bn,*ad,*bd,*ac,*bc;
  float *m1tab,*m2tab,*h1tab,*h2tab,*n1tab,*n2tab,*d1tab,*d2tab;
  float *c1tab,*c2tab;

/* fprintf (stderr,"maktables begin\n"); /* */

  if (!(f=lookup("amfunc"))) amf = amcalc;
  else {
     ampnt = f;
     amf = amfunc;
  }
  if (!(f=lookup("bmfunc")))  bmf = bmcalc;
  else {
     bmpnt = f;
     bmf = bmfunc;
  }
  if (!(f=lookup("ahfunc"))) ahf = ahcalc;
  else {
     ahpnt = f;
     ahf = ahfunc;
  }
  if (!(f=lookup("bhfunc")))  bhf = bhcalc;
  else {
     bhpnt = f;
     bhf = bhfunc;
  }
  if (!(f=lookup("anfunc"))) anf = ancalc;
  else {
     anpnt = f;
     anf = anfunc;
  }
  if (!(f=lookup("bnfunc")))  bnf = bncalc;
  else {
     bnpnt = f;
     bnf = bnfunc;
  }
  if (!(f=lookup("adfunc")))  adf = adcalc;
  else {
     anpnt = f;
     anf = anfunc;
  }
  if (!(f=lookup("bdfunc")))  bdf = bdcalc;
  else {
     bnpnt = f;
     bnf = bnfunc;
  }
  if (!(f=lookup("acfunc")))  acf = accalc;
  else {
     acpnt = f;
     acf = acfunc;
  }
  if (!(f=lookup("bcfunc")))  bcf = bccalc;
  else {
     bcpnt = f;
     bcf = bcfunc;
  }
  
  if (timestep==oldtiminc) return;

 for (t=0; t<nchantyp; t++) {  /* tables for diff chan types */ 

  rtp = &ratetab[t];
  am = rtp->am;
  bm = rtp->bm;
  ah = rtp->ah;
  bh = rtp->bh;
  an = rtp->an;
  bn = rtp->bn;
  ad = rtp->ad;
  bd = rtp->bd;
  ac = rtp->ac;
  bc = rtp->bc;
  m1tab = rtp->m1tab;
  m2tab = rtp->m2tab;
  h1tab = rtp->h1tab;
  h2tab = rtp->h2tab;
  n1tab = rtp->n1tab;
  n2tab = rtp->n2tab;
  d1tab = rtp->d1tab;
  d2tab = rtp->d2tab;
  c1tab = rtp->c1tab;
  c2tab = rtp->c2tab;

  for (vn=0; vn<=RATESIZ; vn++) {   /* make tables from -100 to 100 mV, */
				    /*  i.e. vn==0 -> Vm= -100 mV */

				    /* the old way:  */
/*  v = -vn + (100-65);		    /* v = - mV above -65 mV (as in HH pap)*/

    v = vn - 100;		    /* v = mv, the modern way */ 
    am[vn] = qfac() * (*amf)(v,t);  /* alpha m for sodium */
    bm[vn] = qfac() * (*bmf)(v,t);

    ah[vn] = qfac() * (*ahf)(v,t);  /* alpha h for sodium */
    bh[vn] = qfac() * (*bhf)(v,t);

    an[vn] = qfac() * (*anf)(v,t);  /* alpha n for potassium */
    bn[vn] = qfac() * (*bnf)(v,t);

    ad[vn] = qfac() * (*adf)(v,t);  /* alpha h for type A potassium */
    bd[vn] = qfac() * (*bdf)(v,t);

    ac[vn] = qfac() * (*acf)(v,t);
    bc[vn] = qfac() * (*bcf)(v,t);

    tval  =	  (1.0 - (am[vn] + bm[vn])*0.5) /
    	    	  (1.0 + (am[vn] + bm[vn])*0.5); 

    if (tval>0) {
      m1tab[vn] = tval;				/* C-N implicit */
      m2tab[vn] = am[vn] / 
			(1.0+(am[vn]+rtp->bm[vn])*0.5); 
    }
    else {					/* purely implicit */
      m1tab[vn] = 1.0    / (1.0+am[vn]+bm[vn]);
      m2tab[vn] = am[vn] / (1.0+am[vn]+bm[vn]); 
    }
 

    tval =	  (1.0 - (ah[vn]+bh[vn])*0.5) /
    	    	  (1.0 + (ah[vn]+bh[vn])*0.5); 

    if (tval>0) {				/* C-N implicit */
      h1tab[vn] = tval;
      h2tab[vn] = ah[vn] / (1.0 + (ah[vn]+bh[vn])*0.5); 
    }
    else {
      h1tab[vn] = 1.0    / (1.0+ah[vn]+bh[vn]);	/* implicit */
      h2tab[vn] = ah[vn] / (1.0+ah[vn]+bh[vn]); 
    }


    tval =	  (1.0 - (an[vn]+bn[vn])*0.5) /
    	    	  (1.0 + (an[vn]+bn[vn])*0.5); 
    if (tval>0) {
      n1tab[vn] = tval;
      n2tab[vn] = an[vn] / (1.0 + (an[vn]+bn[vn])*0.5); 
    }
    else {
      n1tab[vn] = 1.0    / (1.0+an[vn]+bn[vn]);	/* implicit */
      n2tab[vn] = an[vn] / (1.0+an[vn]+bn[vn]); 
    }

    tval =	  (1.0 - (ad[vn]+bd[vn])*0.5) /
    	    	  (1.0 + (ad[vn]+bd[vn])*0.5); 

    if (tval>0) {				/* C-N implicit */
      d1tab[vn] = tval;
      d2tab[vn] = ad[vn] / (1.0 + (ad[vn]+bd[vn])*0.5); 
    }
    else {
      d1tab[vn] = 1.0    / (1.0+ad[vn]+bd[vn]);	/* implicit */
      d2tab[vn] = ad[vn] / (1.0+ad[vn]+bd[vn]); 
    }


    tval  =	  (1.0 - (ac[vn]+bc[vn])*0.5) /
    	    	  (1.0 + (ac[vn]+bc[vn])*0.5); 

    if (tval>0) {
      c1tab[vn] = tval;				/* C-N implicit */
      c2tab[vn] = ac[vn] / (1.0+(ac[vn]+bc[vn])*0.5); 
    }
    else {					/* purely implicit */
      c1tab[vn] = 1.0    / (1.0+ac[vn]+bc[vn]);
      c2tab[vn] = ac[vn] / (1.0+ac[vn]+bc[vn]); 
    }
 
/*  fprintf (stderr,"t %d %4d %10.5g %10.5g %10.5g %10.5g %10.5g %10.5g\n",
		t,vn-100,am[vn],bm[vn],ah[vn],bh[vn],an[vn],bn[vn]);  /* */

/* fprintf (stderr,"t %d %4d %10.5g %10.5g %10.5g %10.5g %10.5g %10.5g\n",
		t,vn-100,m1tab[vn],m2tab[vn],h1tab[vn],h2tab[vn],
		n1tab[vn],n2tab[vn]);  /* */

  }  /* for (vn;;) */
 }  /* for (t;;) */

 oldtiminc = timestep;
}

void makna1(void)

/* number states from 0 to n; 
   set numstate = n;
   set numtrans = number of transitions.
   set cond     = conductance of state;

   for each state, set transitions:
    set trans = state to go to on this transition. 
    set trate = function that returns basic rate for transition. 
    set ratemul = multiplier for rate function.
    set ratev  =  pointer to state if different than compartment for cond.
*/

  {
     double alm(double v, int typ),betm(double v, int t);
     double alh(double v, int t),  beth(double v, int t);   /* rate functs */
     chanstate *spnt;
     double k;

   natypes[1].ctype = NA;
   natypes[1].numstate = 6;
   spnt = natypes[1].state;

 /*  k = 1.7; */
   k=1;
   spnt[0].numtrans   = 1;
   spnt[0].cond       = 0;
   spnt[0].trans  [0] = 1;
   spnt[0].trate  [0] = alm;
   spnt[0].ratemul[0] = 3.0 * k;
   spnt[0].ratev  [0] = 0;

   spnt[1].numtrans   = 2;
   spnt[1].cond       = 0;
   spnt[1].trans  [0] = 2;	  		/*   0 <-> 1 <-> 2 <-> 3   */
   spnt[1].trate  [0] = alm;
   spnt[1].ratemul[0] = 2.0 * k;
   spnt[1].ratev  [0] = 0;
   spnt[1].trans  [1] = 0;			/*               |     |   */
   spnt[1].trate  [1] = betm;
   spnt[1].ratemul[1] = 1.0 * k;
   spnt[1].ratev  [1] = 0;

   spnt[2].numtrans   = 3;
   spnt[2].cond       = 0;
   spnt[2].trans  [0] = 3;			/*               4 <-> 5   */
   spnt[2].trate  [0] = alm;
   spnt[2].ratemul[0] = 1.0 * k;
   spnt[2].ratev  [0] = 0;
   spnt[2].trans  [1] = 1;
   spnt[2].trate  [1] = betm;
   spnt[2].ratemul[1] = 2.0 * k;
   spnt[2].ratev  [1] = 0;
   spnt[2].trans  [2] = 4;
   spnt[2].trate  [2] = beth;
   spnt[2].ratemul[2] = 2.0;
   spnt[2].ratev  [2] = 0;

   spnt[3].numtrans   = 2;
   spnt[3].cond       = 1.0;
   spnt[3].trans  [0] = 5;			/* state 3 = the open state */
   spnt[3].trate  [0] = beth;
   spnt[3].ratemul[0] = 2.0;
   spnt[3].ratev  [0] = 0;
   spnt[3].trans  [1] = 2;
   spnt[3].trate  [1] = betm;
   spnt[3].ratemul[1] = 3.0 * k;
   spnt[3].ratev  [1] = 0;

   spnt[4].numtrans   = 2;
   spnt[4].cond       = 0;
   spnt[4].trans  [0] = 5;
   spnt[4].trate  [0] = alm;
   spnt[4].ratemul[0] = 1.0;	/*.001;	/* no return to deactivated state */
   spnt[4].ratev  [0] = 0;
   spnt[4].trans  [1] = 2;
   spnt[4].trate  [1] = alh;
   spnt[4].ratemul[1] = 1.0;
   spnt[4].ratev  [1] = 0;

   spnt[5].numtrans   = 2;
   spnt[5].cond       = 0;
   spnt[5].trans  [0] = 3;			/* state 5 = inactivated state*/
   spnt[5].trate  [0] = alh;
   spnt[5].ratemul[0] = 1.0;
   spnt[5].ratev  [0] = 0;
   spnt[5].trans  [1] = 4;
   spnt[5].trate  [1] = betm;
   spnt[5].ratemul[1] = 3.0 * k;
   spnt[5].ratev  [1] = 0;

}


void makna2(void)

/* number states from 0 to n; 
   set numstate = n;
   set numtrans = number of transitions.
   set cond     = conductance of state;

   for each state, set transitions:
    set trans = state to go to on this transition. 
    set trate = function that returns basic rate for transition. 
    set ratemul = multiplier for rate function.
    set ratev  =  pointer to state if different than compartment for cond.
*/

  {
     double alm(double v, int typ),betm(double v, int t),alh(double v, int t),beth(double v, int t);	   /* rate functions for transitions */
     chanstate *spnt;
     double k;

   natypes[2].ctype = NA;
   natypes[2].numstate = 5;
   spnt = natypes[2].state;

 /*  k = 1.7; */
   k=1;
   spnt[0].numtrans   = 1;
   spnt[0].cond       = 0;
   spnt[0].trans  [0] = 1;
   spnt[0].trate  [0] = alm;
   spnt[0].ratemul[0] = 3.0 * k;
   spnt[0].ratev  [0] = 0;

   spnt[1].numtrans   = 2;
   spnt[1].cond       = 0;
   spnt[1].trans  [0] = 2;	  		/*   0 <-> 1 <-> 2 <-> 3   */
   spnt[1].trate  [0] = alm;
   spnt[1].ratemul[0] = 2.0 * k;
   spnt[1].ratev  [0] = 0;
   spnt[1].trans  [1] = 0;			/*               ^     |   */
   spnt[1].trate  [1] = betm;
   spnt[1].ratemul[1] = 1.0 * k;
   spnt[1].ratev  [1] = 0;

   spnt[2].numtrans   = 3;
   spnt[2].cond       = 0;
   spnt[2].trans  [0] = 3;			/*                     4   */
   spnt[2].trate  [0] = alm;
   spnt[2].ratemul[0] = 1.0 * k;
   spnt[2].ratev  [0] = 0;
   spnt[2].trans  [1] = 1;
   spnt[2].trate  [1] = betm;
   spnt[2].ratemul[1] = 2.0 * k;
   spnt[2].ratev  [1] = 0;
   spnt[2].trans  [2] = 4;
   spnt[2].trate  [2] = beth;
   spnt[2].ratemul[2] = 2.0;
   spnt[2].ratev  [2] = 0;

   spnt[3].numtrans   = 2;
   spnt[3].cond       = 1.0;
   spnt[3].trans  [0] = 4;			/* state 3 = the open state */
   spnt[3].trate  [0] = beth;
   spnt[3].ratemul[0] = 2.0;
   spnt[3].ratev  [0] = 0;
   spnt[3].trans  [1] = 2;
   spnt[3].trate  [1] = betm;
   spnt[3].ratemul[1] = 3.0 * k;
   spnt[3].ratev  [1] = 0;

   spnt[4].numtrans   = 2;		/* the inactivated state */
   spnt[4].cond       = 0;
   spnt[4].trans  [0] = 3;
   spnt[4].trate  [0] = alh;
   spnt[4].ratemul[0] = .010;	/* .001; /* no return to activated state */
   spnt[4].ratev  [0] = 0;
   spnt[4].trans  [1] = 2;
   spnt[4].trate  [1] = alh;
   spnt[4].ratemul[1] = 1.0;
   spnt[4].ratev  [1] = 0;

}


void makk1(void)
{
     double aln(double v, int t),betn(double v, int t);	   	/* rate functions for transitions */
     chanstate *spnt;
     double k;

   ktypes[1].ctype = K;
   ktypes[1].numstate = 5;
   spnt = ktypes[1].state;

   k = 2.0;
   spnt[0].numtrans   = 1;
   spnt[0].cond       = 0;
   spnt[0].trans  [0] = 1;
   spnt[0].trate  [0] = aln;
   spnt[0].ratemul[0] = 4.0*k;
   spnt[0].ratev  [0] = 0;

   spnt[1].numtrans   = 2;
   spnt[1].cond       = 0;
   spnt[1].trans  [0] = 2;	  	/*   0 <-> 1 <-> 2 <-> 3  <-> 4 */
   spnt[1].trate  [0] = aln;
   spnt[1].ratemul[0] = 3.0*k;
   spnt[1].ratev  [0] = 0;
   spnt[1].trans  [1] = 0;
   spnt[1].trate  [1] = betn;
   spnt[1].ratemul[1] = 1.0*k;
   spnt[1].ratev  [1] = 0;

   spnt[2].numtrans   = 2;
   spnt[2].cond       = 0;
   spnt[2].trans  [0] = 3;
   spnt[2].trate  [0] = aln;
   spnt[2].ratemul[0] = 2.0*k;
   spnt[2].ratev  [0] = 0;
   spnt[2].trans  [1] = 1;
   spnt[2].trate  [1] = betn;
   spnt[2].ratemul[1] = 2.0*k;
   spnt[2].ratev  [1] = 0;

   spnt[3].numtrans   = 2;
   spnt[3].cond       = 0;
   spnt[3].trans  [0] = 4;
   spnt[3].trate  [0] = aln;
   spnt[3].ratemul[0] = 1.0*k;
   spnt[3].ratev  [0] = 0;
   spnt[3].trans  [1] = 2;
   spnt[3].trate  [1] = betn;
   spnt[3].ratemul[1] = 3.0*k;
   spnt[3].ratev  [1] = 0;

   spnt[4].numtrans   = 1;			/* state 4 = the open state */
   spnt[4].cond       = 1.0;
   spnt[4].trans  [0] = 3;
   spnt[4].trate  [0] = betn;
   spnt[4].ratemul[0] = 4.0*k;
   spnt[4].ratev  [0] = 0;

}

/*--------------------------------------------*/

void natab(double v, int typ, double taum, double tauh)
                       
/* Used at runtime. Given input voltage in volts, lookup rate
   constants in tables that were constructed as a function
   of millivolts above -100 mv. Do interpolation between
   adjacent table entries.  Tables give second-order
   equation for rate constants.

*/

{
  register int vn,vnn;
  static double x,r,rr,tval,tm,th;
  static double oldv= -1e8;
  chantable *rtp;
  double a,b,ab,ab2;
  float *am,*bm,*ah,*bh;
  float *m1tab,*m2tab,*h1tab,*h2tab;

  if (v==oldv) return;
  rtp = &ratetab[typ];
  m1tab = rtp->m1tab;
  m2tab = rtp->m2tab;
  h1tab = rtp->h1tab;
  h2tab = rtp->h2tab;
  if (ncabs(v) < 1e-20) v = 0.0;
  x = (v * 1000.0) + 100.0;		/* convert to mv + 100 */
  if (x < 0) x = 0;
  if (x >=RATESIZ) x = RATESIZ-1;
  vn = (int)x;				/* table starts at -100 mv */
  vnn = vn + 1;

  r = x - vn;				/* remainder for interpolation */
  rr = 1.0 - r;


  if (taum==dnataum && tauh==dnatauh) {

    m1  =  m1tab[vn]*rr + m1tab[vnn]*r;	/* interpolate m1 table */ 
    m2  =  m2tab[vn]*rr + m2tab[vnn]*r;	/* interpolate m2 table */ 

    h1  =  h1tab[vn]*rr + h1tab[vnn]*r;	/* interpolate h1 table */ 
    h2  =  h2tab[vn]*rr + h2tab[vnn]*r;	/* interpolate h2 table */ 
  }
  else {		/* non-standard tau */
    am = rtp->am; 
    bm = rtp->bm; 
    ah = rtp->ah; 
    bh = rtp->bh; 

    if (taum==0) taum = 1e-4;
    tm = dnataum / taum;

    a = (am[vn]*rr+am[vnn]*r)*tm;
    b = (bm[vn]*rr+bm[vnn]*r)*tm;
    ab  = a+b;
    ab2 = ab*0.5;

    tval  =	  (1.0 - ab2) /
    	    	  (1.0 + ab2); 

    if (tval>0) {
      m1 = tval;				/* C-N implicit */
      m2 = a / (1.0+ab2); 
    }
    else {					/* purely implicit */
      m1 = 1.0 / (1.0+ab);
      m2 = a   / (1.0+ab); 
    }

    if (tauh==0) tauh = 1e-4;
    th = dnatauh / tauh;

    a = (ah[vn]*rr+ah[vnn]*r)*th;
    b = (bh[vn]*rr+bh[vnn]*r)*th;
    ab  = a+b;
    ab2 = ab*0.5;

    tval  =	  (1.0 - ab2) /
    	    	  (1.0 + ab2); 

    if (tval>0) {
      h1 = tval;				/* C-N implicit */
      h2 = a / (1.0 + ab2); 
    }
    else {					/* purely implicit */
      h1 = 1.0 / (1.0+ab);
      h2 = a   / (1.0+ab); 
    }

  }
}

/*--------------------------------------------*/

void catab(double v, int typ, double tauc)
                  
/* Used at runtime. Given input voltage in volts, lookup rate
   constants in tables that were constructed as a function
   of millivolts above -100 mv. Do interpolation between
   adjacent table entries.  Tables give second-order
   equation for rate constants.

*/

{
  register int vn,vnn;
  static double x,r,rr,tval,tc;
  static double oldv= -1e8;
  chantable *rtp;
  float *ac,*bc,*c1tab,*c2tab;
  double a,b,ab,ab2;

  if (v==oldv) return;
  rtp = &ratetab[typ];
  c1tab = rtp->c1tab; 
  c2tab = rtp->c2tab; 
  
  if (ncabs(v) < 1e-20) v = 0.0;
  x = (v * 1000.0) + 100.0;		/* convert to mv + 100 */
  if (x < 0) x = 0;
  if (x >=RATESIZ) x = RATESIZ-1;
  vn = (int)x;				/* table starts at -100 mv */
  vnn = vn + 1;

  r = x - vn;				/* remainder for interpolation */
  rr = 1.0 - r;

/*    ac = rtp->ac; 
    bc = rtp->bc; 
fprintf (stderr,"v %g vn %d ac %g bc %g\n",v,vn,ac[vn],bc[vn]);
*/

  if (tauc==dcatau) {
    c1  =  c1tab[vn]*rr + c1tab[vnn]*r;	/* interpolate c1 table */ 
    c2  =  c2tab[vn]*rr + c2tab[vnn]*r;	/* interpolate c2 table */ 
  }
  else {		/* non-standard tau */
    ac = rtp->ac; 
    bc = rtp->bc; 

    if (tauc==0) tauc = 1e-4;
    tc = dcatau / tauc;

    a = (ac[vn]*rr+ac[vnn]*r)*tc;
    b = (bc[vn]*rr+bc[vnn]*r)*tc;
    ab  = a+b;
    ab2 = ab*0.5;

    tval  =	  (1.0 - ab2) /
    	    	  (1.0 + ab2); 

    if (tval>0) {
      c1 = tval;				/* C-N implicit */
      c2 = a / (1.0 + ab2); 
    }
    else {					/* purely implicit */
      c1 = 1.0 / (1.0+ab);
      c2 = a   / (1.0+ab); 
    }
  }
}

/*--------------------------------------------*/

void ktab(double v, int typ, double taun)

/* Used at runtime. Given input voltage in volts, lookup rate
   constants in tables that were constructed as a function
   of millivolts above -100 mv. Do interpolation between
   adjacent table entries.  Tables give second-order
   equation for rate constants.

*/

{
  register int vn,vnn;
  static double x,r,rr,tn,tval;
  static double oldv= -1e8;
  chantable *rtp;
  float *an,*bn,*ad,*bd;
  float *n1tab,*n2tab,*d1tab,*d2tab;
  double a,b,ab,ab2;


  if (v==oldv) return;
  rtp = &ratetab[typ];
  n1tab = rtp->n1tab; 
  n2tab = rtp->n2tab; 
  d1tab = rtp->d1tab; 
  d2tab = rtp->d2tab; 
  if (ncabs(v) < 1e-20) v = 0.0;
  x = (v * 1000.0) + 100.0;		/* convert to mv + 100 */
  if (x < 0) x = 0;
  if (x >=RATESIZ) x = RATESIZ-1;
  vn = (int)x;				/* table starts at -100 mv */
  vnn = vn + 1;
  r = x - vn;				/* remainder for interpolation */
  rr = 1.0 - r;

  if (taun==dktau) { 
    n1  =  n1tab[vn]*rr + n1tab[vnn]*r;	/* interpolate n1 table */ 
    n2  =  n2tab[vn]*rr + n2tab[vnn]*r;	/* interpolate n2 table */ 

    d1  =  d1tab[vn]*rr + d1tab[vnn]*r;	/* interpolate d1 table */ 
    d2  =  d2tab[vn]*rr + d2tab[vnn]*r;	/* interpolate d2 table */ 
  }
  else {
    an = rtp->an; 
    bn = rtp->bn; 

    ad = rtp->ad; 
    bd = rtp->bd; 

    if (taun==0) taun = 1e-4;
    tn = dktau / taun;

    a = (an[vn]*rr+an[vnn]*r)*tn;
    b = (bn[vn]*rr+bn[vnn]*r)*tn;
    ab  = a+b;
    ab2 = ab*0.5;

    tval  =	  (1.0 - ab2) /
    	    	  (1.0 + ab2); 

    if (tval>0) {
      n1 = tval;				/* C-N implicit */
      n2 = a / (1.0+ab2); 
    }
    else {					/* purely implicit */
      n1 = 1.0 / (1.0+ab);
      n2 = a   / (1.0+ab); 
    }

    a = (ad[vn]*rr+ad[vnn]*r)*tn;
    b = (bd[vn]*rr+bd[vnn]*r)*tn;
    ab  = a+b;
    ab2 = ab*0.5;

    tval  =	  (1.0 - ab2) /
    	    	  (1.0 + ab2); 

    if (tval>0) {
      d1 = tval;				/* C-N implicit */
      d2 = a / (1.0+ab2); 
    }
    else {					/* purely implicit */
      d1 = 1.0 / (1.0+ab);
      d2 = a   / (1.0+ab); 
    }

  } 
}

/*--------------------------------------------*/

void kcatab(double v, double ca, int typ, double taun, 
		double d1, double d2, double k1, double k2)
{
   double a,b,ab,ab2,tval;

    if (taun==0) taun = 1e-4;
    a = qfac() * akcacalc(v,ca,taun,d1,k1);
    b = qfac() * bkcacalc(v,ca,taun,d2,k2);
    ab  = a+b;
    ab2 = ab*0.5;

    tval = (1.0 - ab2) /
    	   (1.0 + ab2); 

    if (tval>0) {
      kca1 = tval;				/* C-N implicit */
      kca2 = a / (1.0+ab2); 
    }
    else {					/* purely implicit */
      kca1 = 1.0 / (1.0+ab);
      kca2 = a   / (1.0+ab); 
    }
}
/*--------------------------------------------*/

void narate(double v, int typ)
             
/* Given input voltage in volts, lookup alpha and beta rate
   constants in tables that were constructed as a function
   of millivolts above -100 mv. Do interpolation between
   adjacent table entries.
*/

{
  int vn;
  double x,r;
  static double oldv= -1e8;
  chantable *rtp;
  float *am,*bm,*ah,*bh;

  if (v==oldv) return;
  rtp = &ratetab[typ];
  am = rtp->am; 
  bm = rtp->bm; 
  ah = rtp->ah; 
  bh = rtp->bh; 
  if (ncabs(v) < 1e-20) v = 0.0;
  x = (v * 1000.0) + 100.0;		/* convert to mv + 100 */
  if (x < 0) x = 0;
  if (x > RATESIZ) x = RATESIZ-1;
  vn = (int)x;				/* table starts at -100 mv */
  r = x - vn;				/* remainder for interpolation */
  alpham = am[vn];			
  alpham += (am[vn+1] - alpham) * r;        /* interpolate alpha m table */ 

  betam =  bm[vn];			
  betam  += (bm[vn+1] - betam) * r;         /* interpolate beta m table */ 

  alphah = ah[vn];			
  alphah += (ah[vn+1] - alphah) * r;    

  betah =  bh[vn];			
  betah  += (bh[vn+1] - betah) * r;    

  oldv = v;
/* fprintf (stderr,"v %g vn %d am %g bm %g\n",v,vn,alpham,betam); /* */
}

/*--------------------------------------------*/

void carate(double v, int typ)

/* Given input voltage in volts, lookup alpha and beta rate
   constants in tables that were constructed as a function
   of millivolts above -100 mv. Do interpolation between
   adjacent table entries.
*/

{
  int vn;
  double x,r;
  static double oldv= -1e8;
  chantable *rtp;
  float *ac,*bc;

  if (v==oldv) return;
  rtp = &ratetab[typ];
  ac = rtp->ac; 
  bc = rtp->bc; 
  if (ncabs(v) < 1e-20) v = 0.0;
  x = (v * 1000.0) + 100.0;		/* convert to mv + 100 */
  if (x < 0) x = 0;
  if (x > RATESIZ) x = RATESIZ-1;
  vn = (int)x;				/* table starts at -100 mv */
  r = x - vn;				/* remainder for interpolation */
  alphac = ac[vn];			
  alphac += (ac[vn+1] - alphac) * r;        /* interpolate alpha c table */ 

  betac =  bc[vn];			
  betac  += (bc[vn+1] - betac) * r;         /* interpolate beta c table */ 

  oldv = v;
/* fprintf (stderr,"v %g vn %d ac %g bc %g\n",v,vn,alphac,betac); /* */
}


/*------------------------------------*/

void krate(double v, int typ)

{
  int vn;
  double x,r;
  static double oldv;
  chantable *rtp;
  float *an,*bn,*ad,*bd;

  if (v==oldv) return;
  rtp = &ratetab[typ];
  an = rtp->an; 
  bn = rtp->bn; 
  ad = rtp->ad; 
  bd = rtp->bd; 
  if (ncabs(v) < 1e-20) v = 0.0;
  x = v * 1000.0 + 100.0;		/* convert to mv */
  if (x < 0) x = 0;
  if (x > RATESIZ) x = RATESIZ-1;
  vn = (int)x;				/* table starts at -100 mv */
  r = x - vn;				/* remainder for interpolation */
  alphan = an[vn];			
  alphan += (an[vn+1] - alphan) * r;        /* interpolate alpha n table */ 

  betan =  bn[vn];			
  betan  += (bn[vn+1] - betan) * r;         /* interpolate beta n table */ 

  alphad = ad[vn];			
  alphad += (ad[vn+1] - alphad) * r;        /* interpolate alpha d table */ 

  betad =  bd[vn];			
  betad  += (bd[vn+1] - betad) * r;         /* interpolate beta d table */ 

  oldv = v;
}

/*--------------------------------------------*/

			   /* rate functions for transitions */
double alm(double v, int typ)
{
 narate(v,typ);
 return alpham;
}

double betm(double v, int t)
{
 narate(v,t);
 return betam;
}

double alh(double v, int t)
{
 narate(v,t);
 return alphah;
}

double beth(double v, int t)
{
 narate(v,t);
 return betah;
}

double aln(double v, int t)
{
 krate(v,t);
 return alphan;
}

double betn(double v, int t)
{
 krate(v,t);
 return betan;
}

double ald(double v, int t)
{
 krate(v,t);
 return alphad;
}

double betd(double v, int t)
{
 krate(v,t);
 return betad;
}

double alc(double v, int t)
{
 carate(v,t);
 return alphac;
}

double betc(double v, int t)
{
 carate(v,t);
 return betac;
}

