/* code.c */

#include "nc.h"
#include "y.tab.h"
#include "adef.h"

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
/* #include <ctype.h> */
#include <math.h>
#include "stdplt.h"

#ifdef __cplusplus
}
#endif

#define push(d) *stackp++ = (d)		/*  */
#define popm()  *--stackp		/* function pop still needed */

/* #define CODEBUG			/*  */
/*#define popm() pop()			/*  */

#ifndef MAXSIZE
# define MAXSIZE 8191				/* max size of double array */
# define NSTACK 400
# define NPROG  2000
#else						/* larger machine */
# define NSTACK 1000
# if MAXSIZE < 65536
#  define NPROG  4096
# else
#  define NPROG 16384
# endif
#endif


extern char *progname;
extern char *runfile;

static datum stack[NSTACK];	/* the stack */
datum *stackp=0;		/* next free spot on stack */

Inst	prog[NPROG];	/* the machine */
Inst	*progp;		/* next free spot for code generation */
Inst	*pc;		/* program counter during execution */
Fpa	*fpa;		/* pointer to function with arguments */
Inst	*progbase = prog; /* start of curent subprogram */
int	returning;	/* 1 if return stmt seen */
int	continuing;	/* 1 if continue stmt seen */
int	breaking;	/* 1 if break stmt seen */
int	stopping;	/* 1 if return, continue, or break stmt seen */

typedef struct Frame {	/* proc/func call stack frame */
	Symbol	*sp;	/* symbol table entry */
	Inst	*retpc;	/* where to resume after return */
	datum	*argn;	/* n-th argument on stack */
	int	nargs;	/* number of arguments */
	int	nlocal;	/* number of local variables */
} Frame;

#define NFRAME 100
Frame	pframe[NFRAME];
Frame	*fp;		/* frame pointer */

static char undef[] = {"undefined variable"};
static char evnon[] = {"attempt to evaluate non-variable"};
static char asnon[] = {"assignment to non-variable"};
static char uninit[] = {"variable not initialized"};
static char diftyp[] = {"variables are different type"};
static char notstr[] = {"variable not a string"};
static char ncreat[] = {"array not created yet"};

#ifdef __cplusplus
extern "C" {
#endif

double atof(const char *);
void strcat (char *s1, char *s2);
char *strtok(char *, const char *);
void free(...);

#ifdef __cplusplus
}
#endif

#include "gprim.h"

double *garray(Symbol *sp);
char *emalloc(unsigned int n);
double *darr2(Symbol *sp, int narg);
void execerror (char *s, char *t);
void constpush(void);
void execute (Inst *pc);
void gtext(char *s);
void fft (double *real, double *imag, int size, int type);
int yyparse();
int checkstr(datum d);
void onintr(void);

/*-----------------------------------------------------------*/

void initcode(void) {
	progbase = prog;
	progp = progbase;
	stackp = stack;
	fp = pframe;
	returning = 0;
	continuing = 0;
	breaking = 0;
	stopping = 0;
}

void resetcode(void) {
	progp = progbase;
	stackp = stack;
	fp = pframe;
	returning = 0;
	continuing = 0;
	breaking = 0;
	stopping = 0;
}

/*push(d)
	datum d;
{
	if (stackp >= &stack[NSTACK])
		execerror ("stack too deep", (char *)0);
	*stackp++ = d;
} */


void expop(void)
{
	if (stackp == stack)
		execerror ("stack underflow", (char *)0);
	--stackp;
	return;
}

datum pop(void)
{
	if (stackp == stack)
		execerror ("stack underflow", (char *)0);
	return *(--stackp);
}

void constpush(void)
{
	datum d;

#ifdef CODEBUG
  fprintf (stderr,"constpush\n");
#endif

	d.type = CONST; 
	d.vtype = ((Symbol *)*pc)->type;	/* since literal, vtype=type */
	switch (d.vtype) {
	  case NUMBER:
	  case LITCHAR:
		d.val = ((Symbol *)*pc)->val;
		break;
	  case STRING:
		d.str = ((Symbol *)*pc)->name;
		break;
	}

	pc++;
	push (d);
}

void varpush(void)
{
	datum d1,d2;
	int narg;

#ifdef CODEBUG
  fprintf (stderr,"varpush\n");
#endif
	d1.sym  = (Symbol *)(*pc++);
	narg = (int) d1.sym;	
	if (narg < 100 && narg>0) {	/* if var is arg to a function */
		push(d1);
	}
	else {
	  switch (d1.sym->type) {
	  case UNDEF:
	  case CONST:
	  case VAR:
	     d1.vtype = d1.sym->vtype;    /* set NUMBER, STRING, etc.*/
	     push(d1);
	     break;
	  case ARRAY:
	      if (d1.sym->val==LARGENUM) {
		execerror (d1.sym->name,ncreat);
	        return;
	      }
	      d2.sym = (Symbol *)(*pc++); /* don't convert to double */
	      push(d2);			    /* number of dimensions */
	      d1.vtype = d1.sym->vtype;   /* set NUMBER, STRING, etc.*/
	      push(d1); 
	      break;
	default: execerror ("Assignment to non-variable", d1.sym->name);
	      break;
	  }
	}
}

void breakcode(void)
{
#ifdef CODEBUG
  fprintf (stderr,"breakcode\n");
#endif
	breaking = stopping = 1;
}

void contcode(void)
{
#ifdef CODEBUG
  fprintf (stderr,"contcode\n");
#endif
	continuing = stopping  = 1;
}

int forval(datum d)
{
   int val;

   switch (d.vtype) {
     case NUMBER:
     case LITCHAR:
	val = (int)(d.val);
	break;
     case STRING:
	val = strlen(d.str);
	break;
   }
   return val;
}

void forcode(void)
{
	datum d;
	Inst *savepc = pc;
	int val;

#ifdef CODEBUG
  fprintf (stderr,"forcode\n");
#endif
	execute (savepc+4);	/* init */
	d = popm();
	execute(*((Inst **)(savepc))); /* condition */
	d = popm();
	while (val=forval(d)) {
		execute(*((Inst **)(savepc+2))); /* body */
		if (stopping) {
			if (returning)
				break;
			stopping = 0;
			if (breaking) {
				breaking = 0;
				break;
			}
			continuing = 0;
		}
		execute(*((Inst **)(savepc+1))); /* end of loop expr */
		d = popm();
		execute(*((Inst **)(savepc))); /* condition */
		d = popm();
	}
	if (!returning)
		pc = *((Inst **)(savepc+3));	/* next stmt */
}

void whilecode(void)
{
	datum d;
	Inst *savepc = pc;
	int val;

#ifdef CODEBUG
  fprintf (stderr,"whilecode\n");
#endif
	execute (savepc+2);	/* condition */
	d = popm();
	while (val=forval(d)) {
		execute(*((Inst **)(savepc)));	/* body */
		if (stopping) {
			if (returning)
				break;
			stopping = 0;
			if (breaking) {
				breaking = 0;
				break;
			}
			continuing = 0;
		}
		execute(savepc+2);	/* condition */
		d = popm();
	}
	if (!returning)
		pc = *((Inst **)(savepc+1));	/* next stmt */
}

void ifcode(void)
{
	datum d;
	Inst *savepc = pc;	/* then part */
	int val;

#ifdef CODEBUG
  fprintf (stderr,"ifcode\n");
#endif
	execute(savepc+3);	/* condition */
	d = popm();
	if (val=forval(d))
		execute(*((Inst **)(savepc)));
	else if (*((Inst **)(savepc+1))) /* else part? */
		execute(*((Inst **)(savepc+1)));
	if (!returning)
		pc = *((Inst **)(savepc+2)); /* next stmt */
}

void define(Symbol *sp, int narg)	/* put func/proc in symbol table */
         
{
#ifdef CODEBUG
  fprintf (stderr,"define\n");
#endif
	sp->defn = progbase;	/* start of code */
	sp->vtype = narg;	/* number of arguments declared */
	progbase = progp;	/* next code starts here */
}

void local(void)	/* add # of local variables to arg num in stack frame */

{
    int i,nargs;
    datum d1;

#ifdef CODEBUG
  fprintf (stderr,"local\n");
#endif
	d1.val = LARGENUM;		/* set to uninitialized value */
	d1.vtype = 0;			/* reset value type */
	nargs = (int)*pc++;
	fp->nargs += nargs;
	fp->nlocal += nargs;
	for (i=0; i<nargs; i++) 	/* increment stack for local vars */
	    push(d1);
	fp->argn = stackp - 1;		/* last local var */
}

void call(void)		/* call a function */
{
	Symbol *sp = (Symbol *)pc[0]; /* symbol table entry */
					/* for function */
#ifdef CODEBUG
  fprintf (stderr,"call\n");
#endif
	if (fp++ >= &pframe[NFRAME-1])
		execerror(sp->name, ": call nested too deeply");
	fp->sp = sp;
	fp->nargs = (int)pc[1];		/* # of args passed from call */
	fp->nlocal = 0;			/* reset # of local variables */
	fp->retpc = pc + 2;
	fp->argn = stackp - 1;	/* last argument (including local variables) */
	if (sp->vtype != fp->nargs) 
		execerror(sp->name, ": wrong number of arguments");
	execute(sp->defn);
	returning = stopping = 0;
}

double callfunc(Symbol *funcp, int npar, double par1, double par2)
{
    datum d1,d2;

#ifdef CODEBUG
  fprintf (stderr,"callfunc\n");
#endif
   d1.val = par1; 		/* simulate a call from interpreter */
   d1.vtype = NUMBER; 
   d2.val = par2; 
   d2.vtype = NUMBER; 
   if (npar>0) push (d1);
   if (npar>1) push (d2);
   if (fp++ >= &pframe[NFRAME-1])
		execerror(funcp->name, ": call nested too deeply");
   fp->sp = funcp;
   fp->nargs = npar; 
   fp->retpc = pc;
   fp->argn = stackp - 1;	/* last argument (including local variables) */
   if (funcp->vtype != fp->nargs) 
		execerror(funcp->name, ": wrong number of arguments");
   execute(funcp->defn);
   returning = stopping = 0;
   d1 = popm(); 
   return (d1.val);
}

void callfunc8 (Symbol *funcp, int npar, double par1, double par2,
			double par3, double par4, double par5,
			double par6, double par7, double par8)

/* simulate a call with 8 arguments and no return */

{
    datum d[8];
    int i;

#ifdef CODEBUG
  fprintf (stderr,"callfunc\n");
#endif
   d[0].val = par1; 		/* simulate a call from interpreter */
   d[1].val = par2;
   d[2].val = par3;
   d[3].val = par4;
   d[4].val = par5;
   d[5].val = par6;
   d[6].val = par7;
   d[7].val = par8;
   npar = limit(npar,0,8);
   for (i=0; i<npar; i++) {
     d[i].vtype = NUMBER; 
     push(d[i]);
   }
   if (fp++ >= &pframe[NFRAME-1])
		execerror(funcp->name, ": call nested too deeply");
   fp->sp = funcp;
   fp->nargs = npar; 
   fp->retpc = pc;
   fp->argn = stackp - 1;	/* last argument (including local variables) */
   if (funcp->vtype != fp->nargs) 
		execerror(funcp->name, ": wrong number of arguments");
   execute(funcp->defn);
   returning = stopping = 0;
}

void ret(void)		/* common return from func or proc */
{
	int i;
#ifdef CODEBUG
  fprintf (stderr,"ret\n");
#endif
	for (i=0; i<fp->nargs; i++)
		pop();  /* pop arguments */
	pc = (Inst *)fp->retpc;
	--fp;
	returning = stopping = 1;
}

void funcret(void)	/* return from a function */
{
	datum d;
#ifdef CODEBUG
  fprintf (stderr,"funcret\n");
#endif
	if (fp->sp->type == PROCEDURE)
		execerror(fp->sp->name, "(proc) returns value");
	d = popm();	/* preserve function return value */
	ret();
	push(d);
}

void procret(void)	/* return from a procedure */
{
#ifdef CODEBUG
  fprintf (stderr,"procret\n");
#endif
	if (fp->sp->type == FUNCTION)
		execerror(fp->sp->name,"func returns no value");
	ret();
}

void xexit(void)

/* exit program completely from any stack location. 
*/

{
  onintr();
}

void getarg(int narg, double **val, short int **vtyp)	
			/* return pointer to argument */
	          
	             
	              
{
	if (narg > fp->nargs)
		execerror(fp->sp->name, ": not passed enough arguments");
	*val  = &fp->argn[narg - fp->nargs].val;
	*vtyp = &fp->argn[narg - fp->nargs].vtype;
}

void bltin(void)

/* call a built-in function, with as many as five arguments */
/*  Arguments may be either NUMBER or STRING. */

{
	int i,narg;
	datum d[5];

#ifdef CODEBUG
  fprintf (stderr,"bltin\n");
#endif
	narg = (int)*pc++;			/* get number of arguments */
	for (i=0; i<narg; i++)
	   d[narg-i-1] = popm();
	fpa = (Fpa *)pc++;

	switch (narg) {
	case 0: d[0] = (*fpa)();
		break;
	case 1: d[0] = (*fpa)(d[0]);
		break;
	case 2: d[0] = (*fpa)(d[0],d[1]);
		break;
	default:if (narg>5) {
		    narg = 5;
		    execerror ("more than 5 arguments.",0);
		}
		d[0] = (*fpa)(d[0],d[1],d[2],d[3],d[4]);
		break;
	}
	push(d[0]);
}

Symbol *getvar(double **val, short int **typ, short int **vtyp)

/* Get pointer to variable */
                      
{
	datum d;
	int narg;
	static short int atyp;
	static Symbol locvar;

	d = popm();			/* get pointer to variable */
	narg = (int)d.sym;
	if (narg > 0 && narg < 100) {	/* if var is arg to a function */
		getarg(narg,val,vtyp);	
		atyp = ARG;
		*typ = &atyp;
		switch (**vtyp) {
		 default:
		 case NUMBER:
	           if (**val==LARGENUM) {	/* return error if uninitialized */
		      static char ebuf[20];
		      int argn;

		   if ((argn=narg-(fp->nargs-fp->nlocal)) <=0)
		     sprintf (ebuf,"proc %s: #%d argument",fp->sp->name,narg);
		   else
		     sprintf (ebuf,"proc %s: #%d local",fp->sp->name,argn);
           	   locvar.name=ebuf;	/* fake a proper symbol and name */
		   return &locvar;
		  }
		  break;
		 case STRING: break;
	      }
	}
	else {
	  *typ  = &d.sym->type;
	  *vtyp = &d.sym->vtype;
	  switch (d.sym->type) {
	  case UNDEF:
	  case CONST:
	  case VAR:   *val = &d.sym->val;
		      break;
	  case ARRAY:  
		       *val = garray(d.sym); 
		      break;
	  default: execerror(evnon, d.sym->name);
		      break;
	  }
	}
	switch (**vtyp) {	/* return var of variable if uninitialized */
	default:
	case NUMBER: 
	  if (**val==LARGENUM) return (d.sym);
	  break;
	case STRING: 	
	  if (*val==NULL) return (d.sym);
	  break;
	}
     return ((Symbol *)0); 
}

void evalvar(void)		/* evaluate variable on stack */
{
	datum d1;
	double *val;
	short *typ,*vtyp;
	Symbol *s;

#ifdef CODEBUG
  fprintf (stderr,"evalvar\n");
#endif
	if (s=getvar(&val,&typ,&vtyp))
           execerror (s->name, uninit);
	d1.type  = *typ;
	d1.vtype = *vtyp;	     /* set value type: NUMBER, STRING, etc. */
 /*      if (d1.type==CONST) d1.vtype = NUMBER;	/* convert CONST to NUMBER */
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
		d1.val  = *val;
		break;
	  case STRING:
		d1.str  = *(char **)val;
		break;
	}
	push(d1);
}

void popone(void)
{
	datum d1,d2;

#ifdef CODEBUG
  fprintf (stderr,"popone\n");
#endif
	d1 = popm();
	d2 = popm();
	push(d1);
}

void add(void)
{
	datum d1,d2;
	int len1,len2;
	char *strp;

#ifdef CODEBUG
  fprintf (stderr,"add\n");
#endif
	d2 = popm();
	d1 = popm();
	if (d1.vtype != d2.vtype) {
		execerror (diftyp,0);
	}
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
		d1.val += d2.val;
		break;
	  case STRING:
		len1 = strlen(d1.str);
		len2 = strlen(d2.str);
		strp = emalloc(len1+len2+1);
		strcpy (strp,d1.str);
		strcat (strp,d2.str);
		d1.str = strp;
		break;
	}
	push(d1);
}

void postinc(void)
{
	datum d1;
	double *p;
	short *typ,*vtyp;
	Symbol *s;

#ifdef CODEBUG
  fprintf (stderr,"postinc\n");
#endif
	if (s=getvar(&p,&typ,&vtyp))
           execerror (s->name, uninit);
	d1.val = *p;
	d1.vtype = NUMBER;
	push(d1);
/*	(*p)++;   */
	*p += 1; 
}

void postdec(void)
{
	datum d1;
	double *p;
	short *typ,*vtyp;
	Symbol *s;

#ifdef CODEBUG
  fprintf (stderr,"postdec\n");
#endif
	if (s=getvar(&p,&typ,&vtyp))
           execerror (s->name, uninit);
	d1.val = *p;
	d1.vtype = NUMBER;
	push(d1);
/*	(*p)--; */
	*p -= 1;
}

void preinc(void)
{
	datum d1;
	double *p;
	short *typ,*vtyp;
	Symbol *s;

#ifdef CODEBUG
  fprintf (stderr,"preinc\n");
#endif
	if (s=getvar(&p,&typ,&vtyp))
           execerror (s->name, uninit);
/*	(*p)++; */
	*p += 1;
	d1.val = *p;
	d1.vtype = NUMBER;
	push(d1);
}

void predec(void)
{
	datum d1;
	double *p;
	short *typ,*vtyp;
	Symbol *s;

#ifdef CODEBUG
  fprintf (stderr,"predec\n");
#endif
	if (s=getvar(&p,&typ,&vtyp))
           execerror (s->name, uninit);
/*	(*p)--; */
	*p -= 1;
	d1.val = *p;
	d1.vtype = NUMBER;
	push(d1);
}

void sub(void)
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"sub\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val -= d2.val;
	push(d1);
}

void mul(void)
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"mul\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val *= d2.val;
	push(d1);
}

void xdiv(void)
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"xdiv\n");
#endif
	d2 = popm();
	if (d2.val == 0.0)
		execerror("division by zero", (char *)0);
	d1 = popm();
	d1.val /= d2.val;
	push(d1);
}

void varexp(double **p1, double *p2)
                     

/* Check for valid variable and
return variable pointer and expression value */

{
	datum d1;
	double *p;
	short *typ,*vtyp;
	Symbol *s;

	d1 = popm();
	if (s=getvar(&p,&typ,&vtyp))
           execerror (s->name, uninit);
	*p1 = p;
	*p2 = d1.val;
}

void addeq(void)
{
	double *var,expr;
	datum d1,d2;
	int len1,len2;
	char *strp;

#ifdef CODEBUG
  fprintf (stderr,"addeq\n");
#endif
	d2 = *(stackp-1);
	d1.vtype = (stackp-2)->sym->vtype;
	if (d1.vtype == 0) d1.vtype = d2.vtype;
	if (d1.vtype != d2.vtype) {
/*		fprintf (stderr,"d1 %d  d2 %d\n",d1.vtype,d2.vtype); */
		execerror (diftyp,0);
	}
	varexp (&var,&expr);
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
        	*var += expr;
		d1.val = *var;
		break;
	  case STRING:
		len1 = strlen(*(char **)var);
		len2 = strlen(d2.str);
		strp = emalloc(len1+len2+1);
		strcpy (strp,*(char **)var);
		strcat (strp,d2.str);
		*(char **)var = strp;
		d1.str = strp;
		break;
	}
	push(d1);
}

void subeq(void)
{
	double *var,expr;
	datum d;

#ifdef CODEBUG
  fprintf (stderr,"subeq\n");
#endif
	varexp (&var,&expr);
        *var -= expr;
	d.val = *var;
	d.vtype = NUMBER;
	push(d);
}

void muleq(void)
{
	double *var,expr;
	datum d;

#ifdef CODEBUG
  fprintf (stderr,"muleq\n");
#endif
	varexp (&var,&expr);
        *var *= expr;
	d.val = *var;
	d.vtype = NUMBER;
	push(d);
}

void diveq(void)
{
	double *var,expr;
	datum d;

#ifdef CODEBUG
  fprintf (stderr,"diveq\n");
#endif
	varexp (&var,&expr);
        *var /= expr;
	d.val = *var;
	d.vtype = NUMBER;
	push(d);
}

void negate(void) {
	datum d;
#ifdef CODEBUG
  fprintf (stderr,"negate\n");
#endif
	d = popm();
	d.val = -d.val;
	push(d);
}

void gt(void)
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"gt\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = (double)(d1.val > d2.val);
	push(d1);
}

void lt(void) 
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"lt\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = (double)(d1.val < d2.val);
	push(d1);
}

void ge(void) 
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"ge\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = (double)(d1.val >= d2.val);
	push(d1);
}
void le(void) 
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"le\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = (double)(d1.val <= d2.val);
	push(d1);
}
void eq(void)
{
	datum d1,d2;
	int cmp;

#ifdef CODEBUG
  fprintf (stderr,"eq\n");
#endif
 	d2 = popm();
	d1 = popm();
	if (d1.vtype != d2.vtype) {
		execerror (diftyp,0);
	}
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
		d1.val = (double)(d1.val == d2.val);
		break;
	  case STRING:
		cmp = strcmp(d1.str,d2.str);
		d1.val = (double)(cmp==0);
		break;
	}
	d1.vtype = NUMBER;
	push(d1);
}
void ne(void)
{
	datum d1,d2;
	int cmp;

#ifdef CODEBUG
  fprintf (stderr,"ne\n");
#endif
	d2 = popm();
	d1 = popm();
	if (d1.vtype != d2.vtype) {
		execerror (diftyp,0);
	}
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
		d1.val = (double)(d1.val != d2.val);
		break;
	  case STRING:
		cmp = strcmp(d1.str,d2.str);
		d1.val = (double)(cmp!=0);
		break;
	}
	d1.vtype = NUMBER;
	push(d1);
}

void xand(void)
{
	datum d1,d2;
	int val1,val2;

#ifdef CODEBUG
  fprintf (stderr,"xand\n");
#endif
	d2 = popm();
	d1 = popm();
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
		val1 = (int)(d1.val != 0.0);
		break;
	  case STRING:
		val1 = (strlen(d1.str) != 0);
		break;
	}
	switch (d2.vtype) {
	  case NUMBER:
	  case LITCHAR:
		val2 = (int)(d2.val != 0.0);
		break;
	  case STRING:
		val2 = (strlen(d2.str) != 0);
		break;
	}
	d1.val = (double)(val1 && val2);
	d1.vtype = NUMBER;
	push(d1);
}

void bitand(void)
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"bitand\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = (double)((int)d1.val &  (int)d2.val);
	push(d1);
}

void bitor(void)
{
	datum d1,d2;
#ifdef CODEBUG
  fprintf (stderr,"bitor\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = (double)((int)d1.val |  (int)d2.val);
	push(d1);
}
void orx(void)
{
	datum d1,d2;
	int val1,val2;

#ifdef CODEBUG
  fprintf (stderr,"orx\n");
#endif
	d2 = popm();
	d1 = popm();
	switch (d1.vtype) {
	  case NUMBER:
	  case LITCHAR:
		val1 = (int)(d1.val != 0.0);
		break;
	  case STRING:
		val1 = (strlen(d1.str) != 0);
		break;
	}
	switch (d2.vtype) {
	  case NUMBER:
	  case LITCHAR:
		val2 = (int)(d2.val != 0.0);
		break;
	  case STRING:
		val2 = (strlen(d2.str) != 0);
		break;
	}
	d1.val = (double)(val1 || val2);
	d1.vtype = NUMBER;
	push(d1);
}
void xnot(void)
{
	datum d;
	int cmp;

#ifdef CODEBUG
  fprintf (stderr,"xnot\n");
#endif
	d = popm();
	switch (d.vtype) {
	  case NUMBER:
	  case LITCHAR:
		d.val = (double)(d.val == 0.0);
		break;
	  case STRING:
		cmp = strlen(d.str);
		d.val = (double)(cmp==0);
		break;
	}
	d.vtype = NUMBER;
	push(d);
}

void power(void)
{
	datum d1,d2;
	extern double xpow();
#ifdef CODEBUG
  fprintf (stderr,"power\n");
#endif
	d2 = popm();
	d1 = popm();
	d1.val = pow(d1.val, d2.val);
	push(d1);
}

void assign(void)
{
	datum d1,*d;
	double *p;
	short *typ,*vtyp;

#ifdef CODEBUG
  fprintf (stderr,"assign\n");
#endif
	d1 = popm();			/* get expression to be assigned */
	d = stackp-1;			/* get address of variable   */
	getvar(&p,&typ,&vtyp);		/* get variable to set value */

	if (*vtyp == 0 ||		/* check to make sure same type */
	    *vtyp == d1.vtype) {
	  *vtyp = d1.vtype;		/* assign value's type */
	  switch (d1.vtype) {
	    case NUMBER:
	    case LITCHAR:
		*p = d1.val;			/* assign value */
		break;
	    case STRING:
		*(char **)p = d1.str;		/* assign string value */
		break;
	  }		
	  if (*typ==UNDEF) *typ=VAR;		/* make sure it's a variable*/

	}  /* if */

	else {
	   execerror("assignment of incorrect type for ",d->sym->name);
	}
	push(d1);
}

void print(void)	/* pop top value from stack, print it */
{
	datum d;

#ifdef CODEBUG
  fprintf (stderr,"print\n");
#endif
	d = popm();
	switch (d.vtype) {
	  case 0:
	  case NUMBER:
		printf ("\t%.8g\n",d.val);
		break;
	  case LITCHAR:
		printf ("\t%c\n",d.val);
		break;
	  case STRING:
		printf ("\t%s\n",d.str);
		break;
	}
}

void prexpr(void)	/* print value of expression: no tab */
{
	datum d;
#ifdef CODEBUG
  fprintf (stderr,"prexpr\n");
#endif
	d = popm();
	switch (d.vtype) {
	  case 0:
	  case NUMBER:
		printf("%.8g ", d.val);
		break;
	  case LITCHAR:
		printf("%c ", (char)d.val);
		break;
	  case STRING:
		printf("%s ", d.str);
		break;
	}
}

void crlf(void)
{
#ifdef CODEBUG
  fprintf (stderr,"crlf\n");
#endif
	printf("\n");
}

/*---------------------------------------------------------*/

#define MAXARGS 10

void pscanf(void) 			/* scanf statement */

/* Unfinished, needs work on assigning variables after they've been
   read into local array.  For strings, need to malloc new space.
*/

{
#define STRLENGTH 100
	static datum  d1;
	static double *v[MAXARGS];
	static short int *vvtyp[MAXARGS];
	static char  s[MAXARGS][STRLENGTH],*cpnt;
	int i,scase,indx,narg;
        short int *typ;

#ifdef CODEBUG
  fprintf (stderr,"pscanf\n");
#endif
	narg = (int) *pc++;
	if (narg>MAXARGS) 
	   execerror("scanf: more than 10 arguments","");
	for (i=0; i<narg; i++) {
	   indx = MAXARGS-1-i;
	   getvar(&v[indx],&typ,&vvtyp[indx]);	/* pop var off stack */ 
	}					/*  save pointer to value */
	for (i=0; i<narg; i++) {
            v[i] = v[MAXARGS-narg+i];
            vvtyp[i] = vvtyp[MAXARGS-narg+i];
	}
	for (; i<MAXARGS; i++) {		/* zero the remaining ptrs */
            v[i] = NULL;
            vvtyp[i] = NULL;
	}
	d1 = popm();				/* get format string */
	if (d1.vtype!=STRING)
	   execerror("scanf: string required for format","");
	for (scase=i=0; i<narg; i++) {
	   switch (*vvtyp[i]) {
	    case 0:  *vvtyp[i] = NUMBER;
		     break;
	    case NUMBER:
	    case LITCHAR:
		break;
	    case STRING:
	        scase += 1 << i;
		break;
	   }
	}
	cpnt = d1.str;
	/* fprintf (stderr,"scase %d\n",scase);   /* */
	switch (scase) {
 	case 0:
	  scanf(cpnt,v[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 1:
	  scanf(cpnt,s[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 2:
	  scanf(cpnt,v[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 3:
	  scanf(cpnt,s[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 4:
	  scanf(cpnt,v[0],v[1],s[2],v[3]);
	  break;
 	case 5:
	  scanf(cpnt,s[0],v[1],s[2],v[3]);
	  break;
 	case 6:
	  scanf(cpnt,v[0],s[1],s[2],v[3]);
	  break;
 	case 7:
	  scanf(cpnt,s[0],s[1],s[2],v[3]);
	  break;
	default:
	  break;
	}
					/* assign the new values */
	for (i=0; i<narg; i++) {
	   double *val; short *typ,*vtyp; int len;

	   switch (*vvtyp[i]) {
	    case 0:
	    case NUMBER:
	    case LITCHAR:
		break;
	    case STRING:
		if ((len=strlen(s[i]))>0) {
		  *((char **)v[i]) = emalloc (len+1);
		  strcpy(*((char **)v[i]),s[i]);
		}
		break;
	   }
	}

}

/*---------------------------------------------------------*/

void pprintf(void) 			/* printf statement */
{
	datum  d1,d[MAXARGS];
	double v[MAXARGS];
	char  *s[MAXARGS],*cpnt;
	int i,scase,indx,narg;

#ifdef CODEBUG
  fprintf (stderr,"pprintf\n");
#endif
	narg = (int) *pc++;
	if (narg>MAXARGS) 
	   execerror("printf: more than 10 arguments","");
	for (i=0; i<narg; i++)
	   d[MAXARGS-1-i] = popm();
	d1 = popm();
	if (d1.vtype!=STRING)
	   execerror("printf: string required for format","");
	for (scase=i=0; i<narg; i++) {
	   indx = MAXARGS-narg+i;
	   switch (d[indx].vtype) {
	    case 0:
	    case NUMBER:
	    case LITCHAR:
	   	v[i] = d[indx].val;	
		break;
	    case STRING:
	   	s[i] = d[indx].str;	
	        scase += 1 << i;
		break;
	   }
	}
	cpnt = d1.str;
	/* fprintf (stderr,"scase %d\n",scase);   /* */
	switch (scase) {
 	case 0:
	  printf(cpnt,v[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 1:
	  printf(cpnt,s[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 2:
	  printf(cpnt,v[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 3:
	  printf(cpnt,s[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 4:
	  printf(cpnt,v[0],v[1],s[2],v[3]);
	  break;
 	case 5:
	  printf(cpnt,s[0],v[1],s[2],v[3]);
	  break;
 	case 6:
	  printf(cpnt,v[0],s[1],s[2],v[3]);
	  break;
 	case 7:
	  printf(cpnt,s[0],s[1],s[2],v[3]);
	  break;
	default:
	  break;
	}
	fflush(stdout);
}

/*---------------------------------------------------------*/

void dprintf(void) 			/* fprintf (stderr,"") statement */
{
	datum  d1,d[MAXARGS];
	double v[MAXARGS];
	char  *s[MAXARGS],*cpnt;
	int i,scase,indx,narg;

#ifdef CODEBUG
  fprintf (stderr,"dprintf\n");
#endif
	narg = (int) *pc++;
	if (narg>MAXARGS) 
	   execerror("fprintf: more than 10 arguments","");
	for (i=0; i<narg; i++)
	   d[MAXARGS-1-i] = popm();
	d1 = popm();
	if (d1.vtype!=STRING)
	   execerror("fprintf: string required for format","");
	for (scase=i=0; i<narg; i++) {
	   indx = MAXARGS-narg+i;
	   switch (d[indx].vtype) {
	    case 0:
	    case NUMBER:
	    case LITCHAR:
	   	v[i] = d[indx].val;	
		break;
	    case STRING:
	   	s[i] = d[indx].str;	
	        scase += 1 << i;
		break;
	   }
	}
	cpnt = d1.str;
	/* fprintf (stderr,"scase %d\n",scase);   /* */
	switch (scase) {
 	case 0:
	fprintf(stderr,cpnt,v[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 1:
	fprintf(stderr,cpnt,s[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 2:
	fprintf(stderr,cpnt,v[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 3:
	fprintf(stderr,cpnt,s[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 4:
	  fprintf(stderr,cpnt,v[0],v[1],s[2],v[3]);
	  break;
 	case 5:
	  fprintf(stderr,cpnt,s[0],v[1],s[2],v[3]);
	  break;
 	case 6:
	  fprintf(stderr,cpnt,v[0],s[1],s[2],v[3]);
	  break;
 	case 7:
	  fprintf(stderr,cpnt,s[0],s[1],s[2],v[3]);
	  break;
	}
	fflush(stderr);

}

/*---------------------------------------------------------*/

void txtf(void) 			/* textf statement */
{
	datum  d1,d[MAXARGS];
	double v[MAXARGS];
	char  *s[MAXARGS];
	char  *cpnt,cbuf[120];
	int i,scase,indx,narg;

#ifdef CODEBUG
  fprintf (stderr,"txtf\n");
#endif
	narg = (int) *pc++;
	if (narg>MAXARGS) 
	   execerror("textf: more than 10 arguments","");
	for (i=0; i<narg; i++)
	   d[MAXARGS-1-i] = popm();
	d1 = popm();
	if (d1.vtype!=STRING)
	   execerror("textf: string required for format","");
	cpnt = d1.str;
	for (scase=i=0; i<narg; i++) {
	   indx = MAXARGS-narg+i;
	   switch (d[indx].vtype) {
	    case 0:
	    case NUMBER:
	    case LITCHAR:
	   	v[i] = d[indx].val;	
		break;
	    case STRING:
	   	s[i] = d[indx].str;	
	        scase += 1 << i;
		break;
	   }
	}
	/* fprintf (stderr,"scase %d\n",scase);   /* */
	switch (scase) {
 	case 0:
	sprintf(cbuf,cpnt,v[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 1:
	sprintf(cbuf,cpnt,s[0],v[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 2:
	sprintf(cbuf,cpnt,v[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 3:
	sprintf(cbuf,cpnt,s[0],s[1],v[2],v[3],v[4],v[5],v[6],v[7],v[8],v[9]);
	  break;
 	case 4:
	  sprintf(cbuf,cpnt,v[0],v[1],s[2],v[3]);
	  break;
 	case 5:
	  sprintf(cbuf,cpnt,s[0],v[1],s[2],v[3]);
	  break;
 	case 6:
	  sprintf(cbuf,cpnt,v[0],s[1],s[2],v[3]);
	  break;
 	case 7:
	  sprintf(cbuf,cpnt,s[0],s[1],s[2],v[3]);
	  break;
	}
 gtext(cbuf);

}

/*---------------------------------------------------------*/

int garrsiz(Symbol *apnt)
                
 
/*  find absolute size of array */

{
   int i,*dpnt,ndim,size;

   dpnt = apnt->arrp;			/* pointer to array */
   dpnt += sizeof(int *) / sizeof(int);	/* skip over pointer */
   ndim = *dpnt++;
   for (size=1,i=0; i<ndim; i++) {
	size *= *dpnt++;		/* multiply dims for size */
   }
   return size;
}

void darray(void)	/* allocate space for array */

/* Allocate memory space for array and
 its descriptor block, and set descriptor block.
	
	Desciptor block:

	0	pointer to start of array
	1	number of dimensions in array
	2	dimension 1
	3	dimension 2
		etc.
*/

#define MAXDIM  20				/* maximum # of dimensions */

{
	int narg;
	Symbol *sp;
	double *dpnt;

#ifdef CODEBUG
  fprintf (stderr,"darray\n");
#endif
	sp = (Symbol *)(*pc++);			/* pointer to array symbol */
	narg = (int) *pc++;			/* number of args */
	dpnt = darr2(sp,narg);
}

double *darr2(Symbol *sp, int narg)
{
	int i,*dpnt,**tdpnt,darr[MAXDIM],x;
	datum d;
	unsigned size, dsize;

	if (sp->val!=LARGENUM) {
		execerror(sp->name,"array already defined");
		return(NULL);
	}
	size = (narg + 1) * sizeof(int) + sizeof (char *);
	dpnt = sp->arrp = (int *)emalloc(size);/* allocate space for limits */
	if (!dpnt) {
		execerror("no space for array ",sp->name);
		return(NULL);
	}
	tdpnt = (int **)dpnt;			/* save pointer to array */
	dpnt += sizeof(int *) / sizeof(int);	/* skip over pointer */
	*dpnt++ = narg;
	size = 1;
	for (i=0; i<narg; i++) {
	  d = popm();				/* get dimensions from stack */
	  x = (int)(d.val);			/* this line fixes bug */
	  darr[(MAXDIM-1)-i] = x;		/* save backwards in temp arr */
	  size *= x;				/* */
	}
	for (i=0; i<narg; i++) {
	  *dpnt++ = darr[MAXDIM-narg+i];	/* put first dim first */
	}
	if (size == 0) {
		execerror("array dim is zero",sp->name);
		return(NULL);
	}
	if (size < 0) {
		execerror("array dim is negative",sp->name);
		return(NULL);
	}
	else if (size > MAXSIZE) {
		execerror("array too large",sp->name);
		return(NULL);
	}
	dsize = size * sizeof(double);		
/*	fprintf(stderr,"size %u\n",dsize); 		/* */
	*tdpnt = (int *)emalloc(dsize);		/* allocate space for array */
	if (! *tdpnt) {
		execerror("run out of space for array ",sp->name);
		return (NULL);
	}
	for (i=0; i<size; i++) 
	  *(((double *)*tdpnt)+i) = LARGENUM;	/* make array uninitialized */
	sp->type = ARRAY;
	return ((double *)*tdpnt);
}

double
*garray(Symbol *sp)	/* return a pointer to an array value */
               
{
	datum d;
	int i,narg,darr[MAXDIM],ndim;
	int *dpnt,*tdpnt,*index,offset,x;
	double *parr;

	d = popm();				/* get number of indexes */
	narg = (int)d.sym;
	index = darr+MAXDIM;
	for (i=0; i<narg; i++) {
	  d = popm();				/* get indexes from stack */
	  x = (int)(d.val);
	  *(--index) = x;			/* save backwards in temp arr*/
	}
	parr = *(double **)sp->arrp;		/* get pointer to array */
	if (!parr) {
	   execerror("array space not allocated for",sp->name);
	   return ((double *)NULL);
	}

	dpnt = sp->arrp + sizeof(int *) / sizeof(int);  /* skip array pntr */
	tdpnt = dpnt;				/* save ptr to num of dims */
	ndim = *dpnt++;				/* get # of dimensions */
	if (narg != ndim) {
	   execerror("array indices different than dimensions of",
					sp->name);
	   return ((double *)NULL);
	}
	index = darr+MAXDIM-narg;		/* pointer to first index */
	for (i=0; i<ndim; i++) {
	  if (*index++ >= *dpnt++)
            execerror("illegal index size for",sp->name);
	}
	dpnt = tdpnt+2;				/* start with second dim */
	offset = 0;
	index = darr+MAXDIM-narg;		/* pointer to first index */
	for (i=1; i<ndim; i++) {
	  offset += *index++;			/* add the index */
	  offset *= *dpnt++;			/* mult by next lower dim */
	}
	offset += *index;			/* add the final index */
	parr += offset;				/* calculate array pointer */
	return (parr);
}

void dofft(void) /* fourier transform on an array */
		 /* array must be:   arr[length][2]
		    where arr is the array name;
		    and length is the length of the array.
		    arr[0] is transformed into the amplitude fft, and
	 	    arr[1] is transformed into the phase fft.
		*/
{
	int *arrp,*dpnt,size1;
	double *realpt, *imagpt;
	Symbol *sp,*param;

#ifdef CODEBUG
  fprintf (stderr,"dofft\n");
#endif
	param = (Symbol *)(*pc++);		/* pointer to fft type symbol */
	sp = (Symbol *)(*pc++);			/* pointer to array symbol */
	arrp = sp->arrp;		/* get pointer to address of array */
	if (!arrp) {
	   execerror("fft: array space not allocated for",sp->name);
	   return;
	}
	realpt = *(double **)arrp;
	dpnt   = arrp + sizeof(int *) / sizeof(int);  /* skip array pntr */
	size1 = *(dpnt+2);				/* get second dim */
	imagpt = realpt + size1;
	fft (realpt,imagpt,size1,param->type);
}

void erarr(void)			/* erase array */

{
	Symbol *sp;
	
#ifdef CODEBUG
  fprintf (stderr,"erarr\n");
#endif
	sp = (Symbol *)(*pc++);			/* pointer to array symbol */
	switch (sp->type) {
	case  ARRAY:
		if (sp->val==LARGENUM) {
		   sp->type = UNDEF;
		   execerror (sp->name,ncreat);
		   return;
		}
		free((char *)*(sp->arrp));  /* erase array */
	 	free(sp->arrp);		  /* erase array descriptor */
	case VAR:    
		sp->val = LARGENUM;
		sp->type = UNDEF;
		break;
	default:
		break;
	}
}

void varread(void)	/* read into variable */
{
	datum d1,d2;
	double *val;
	short *typ,*vtyp;
	char vbuf[120];
	int len;

#ifdef CODEBUG
  fprintf (stderr,"varread\n");
#endif
	d2 = *(stackp-1);
	getvar(&val,&typ,&vtyp);
	switch (*vtyp) {
	  case 0:
	  case NUMBER:
	  default:
		switch (fscanf(stdin, "%lf", val)) {
		case EOF:
			d1.val = *val = 0.0;
			break;
		case 0:
			execerror("non-number read into", d2.sym->name);
			break;
		default:
			d1.val = 1.0;
			break;
		}
        	*vtyp = NUMBER;
		break;

	  case STRING:
		vbuf[0]=0;
		switch (fscanf(stdin, "%s",vbuf)) {
		case EOF:
			d1.val = *val = 0;
			break;
		case 0:
			execerror("non-string read into", d2.sym->name);
			break;
		default:
			d1.val = 1.0;
			if ((len=strlen(vbuf))>0) {
			  *((char **)val) = emalloc (len+1);
			  strcpy(*((char **)val),vbuf);
			}
			break;
		}
        	*vtyp = STRING;
		break;
	}
	/* *typ = VAR; */
	d1.vtype = NUMBER;
	push(d1);
}

void xfread(void)

/* define and read array from file */
/* Can be 1 or 2 dimensional; 
   Number of lines defines first dimension;
   Number of elements on first line (first row) defines second dimension;
   Variables as well as numbers are allowed;
   but these variables must already exist in symbol table;
   Comment lines start with "#".
*/

#define STRSIZ 120

{
    char *filnam;
    FILE *ftemp;
    Symbol *var,*s;
    double strtod(const char *, char **);
    int i,j,narg,nwid,nlong;
    datum d1,d2,d3;
    char *cp,*np,str[STRSIZ],*strp,*strtok(char *, const char *),*fgets(char *, int, FILE *);
    double *dim1,*dim2,*dpnt,tmp;
    short *typ,*vtyp1,*vtyp2;

#ifdef CODEBUG
  fprintf (stderr,"xfread\n");
#endif
     var = (Symbol *)(*pc++);		/* array to be created */
     narg = (int) *pc++;		/* number of var's (dimensions) */
     if (narg==2) getvar(&dim2,&typ,&vtyp2); /* get pointer to second dim */
     getvar(&dim1,&typ,&vtyp1);		/* get pointer to first dim */

     if (!checkstr(d1=popm())) {
	pc+=2;
	return; 
     }
     filnam = d1.str;		/* file to be read */
     if ((ftemp=fopen(filnam,"r")) == NULL) {
	fprintf (stderr,"fread: can't open file %s\n",filnam);
	return;
     }

    while ((strp=fgets(str,STRSIZ,ftemp)) && *str== '#'); /* get first line */
    if (!strp) return;

     cp = (char *)NULL;
     for (np=str,nwid=0; np!=cp; nwid++) {	/* how many numbers on line */
        cp = np;
	cp = strtok(np," \t,\n");		/* count the separators */
	np = (char *)NULL;
/*	tmp = strtod(cp,&np);  */
     }
     nwid--;
						/* how many lines in file */
    for (nlong=1; ; nlong++) {
      while ((strp=fgets(str,STRSIZ,ftemp)) && *str=='#');/* get next line */
      if (!strp) break;
    }

     rewind(ftemp);				/* go back to beginning */

     d1.val = nlong;
     d2.val = nwid;
     d1.type = d2.type = CONST;
     d1.vtype = d2.vtype = NUMBER;

     if (nwid > 1 && nlong > 1) {
	push(d1);
	push(d2);
	dpnt=darr2(var,2);
     }
     else {
	if (nlong > 1) push(d1);
	else 	       push(d2);
	dpnt=darr2(var,1);
     } 

   for (i=0; i<nlong; i++) {
     while ((strp=fgets (str,STRSIZ,ftemp)) && *str=='#');   /* get next line */
     if (!strp) return;
     np = str;
     for (j=0; j<nwid; j++) {
       if (nwid > 1 && nlong > 1) {
	  d3.sym = (Symbol *)2;		/* set up array indices */
	  d1.val = i;
	  d2.val = j;
	  push(d1);
	  push(d2);
	  push(d3);
       }
       else {
	  d3.sym = (Symbol *)1;
	  if (nlong>1) d1.val = i;
	  else         d1.val = j;
	  push(d1);
	  push(d3);
       } 
        cp = strtok(np," ,\t\n");
	if (cp==NULL) {
	    fprintf (stderr,"Error in format of file while reading array.\n");
	}
	np = (char *)NULL;	
	if (isalpha(*cp)) {
	    s = lookup(cp); 
	    if (s) *garray(var) = s->val;
	} 
       else        *garray(var) = atof(cp);

     }   /* for j */
   }     /* for i */
  *dim1 = nlong;
  *vtyp1 = var->vtype = NUMBER;
  if(narg==2) {
	 *dim2 = nwid;
         *vtyp2 = NUMBER;
  }
}

void xfwrite(void)

/* write array to file.
 
   Array can be 1 or 2 dimensional; 
   Size of first dimension defines number of lines;
   Size of second dimension defines number of columns;
*/

#define STRSIZ 120

{
    char *filnam;
    FILE *ftemp;
    Symbol *var;
    double *vpnt;
    int i,j;
    int dims,dim1,dim2,**apnt,*dpnt;
    datum d;

#ifdef CODEBUG
  fprintf (stderr,"xfwrite\n");
#endif
  if (!checkstr(d=popm())) {
	 pc+=2;
         return;
  }
  filnam = (char *)(*pc++);		/* file to be written */
  var = (Symbol *)(*pc++);		/* array */
  apnt = (int **)var->arrp;		/* pointer to pointer to array */
  dpnt = (int *)(apnt + 1);		/* pointer to number of dims */
  dims = *dpnt;				/* number of dimensions */
  dim1 = *(dpnt+1);			/* size of dimensions */
  dim2 = *(dpnt+2);
  vpnt = (double *)*apnt;

  if (strcmp(filnam,"stdout")==0) 
	ftemp = stdout;
  else if ((ftemp=fopen(filnam,"w")) == NULL) {
    fprintf (stderr,"fwrite: can't open file %s\n",filnam);
    return;
  }

  switch (dims) {

  case 2:
   for (i=0; i<dim1; i++) {
     for (j=0; j<dim2; j++,vpnt++) {
       if (j==(dim2-1))
          fprintf (ftemp,"%g",*vpnt);
       else
          fprintf (ftemp,"%g ",*vpnt);
     }
     fprintf (ftemp,"\n");
   }
   break;

   case 1:
     for (i=0; i<dim1; i++) {
          fprintf (ftemp,"%g\n",*vpnt++);
     }
    break;

    default:
    break;

  }  /* switch dims */

  fclose(ftemp);
}

void gplot(void)

/* low-level plotting commands */

{
   int narg,fill;
   Symbol *param;
   datum d1,d2;
   double x,y;
   double x1,y1,x2,y2,x3,y3,x4,y4;
   char *str=0;

#ifdef CODEBUG
  fprintf (stderr,"gplot\n");
#endif
 param  = (Symbol *)*pc++;
 narg = (int) *pc++;
 if (param->type==GRECT) {
     d1 = popm(); fill = (int)(d1.val);
     d1 = popm(); y4 = d1.val;
     d1 = popm(); x4 = d1.val;
     d1 = popm(); y3 = d1.val;
     d1 = popm(); x3 = d1.val;
     d1 = popm(); y2 = d1.val;
     d1 = popm(); x2 = d1.val;
     d1 = popm(); y1 = d1.val;
     d1 = popm(); x1 = d1.val;
  }
 else
  if (narg > 1) {
   d2 = popm();
   y = d2.val;
 }	
 else y = 0;
 d1 = popm();
 switch (d1.vtype) {
   default:
   case 0:
   case NUMBER:
    x   = d1.val; break;
   case STRING:
    str = d1.str; break;
 }
 switch (param->type) {


   case GMOVE:  gmove (x,y);
		break;

   case GDRAW:  gdraw (x,y);
		break;

   case GRMOVE: grmove (x,y);
		break;

   case GRDRAW: grdraw (x,y);
		break;

   case GPEN:   gpen ((int) x);
		break;

   case GROT:   grotate (x);
		break;

   case GCROT:  gcrotate (x);
		break;

   case GORIG:  gorigin (x,y);
		break;

   case GFRAME:	if (str) gframe (str);
		break;

   case GCWID:	gcwidth (x);
		break;

   case GSIZ:	gsize (x);
		break;

   case GDASH:	gdash ((int)x);
		break;

   case GCIRC:	gcirc (x,(int)y);		/* radius, fill */
		break;

   case GRECT:	grect (x1,y1,x2,y2,x3,y3,x4,y4,fill);
		break;
	}
}

Inst *code(Inst f)	/* install one instruction or operand */
	       
{
	Inst *oprogp = progp;

	if (progp >= &prog[NPROG])
		execerror("program too big", (char *)0);
	*progp++ = f;
	return oprogp;
}

void execute(Inst *p)
{
	Inst func;

	for (pc = p; *pc != STOP && !stopping; ) {
/*	fprintf (stderr,"%d %d\n",(int)*pc,(int)expop);  /* debug pc */
                func = *pc++;   /* inside func, pc must point to "arg"  */
                ((Inst )(*func))();
/*              (*(*pc++))();   /* old way, doesn't always work */
				/*  because pc is incr after func call */
	}

}

void edit(void)
{
    char sysbuf[40];
    datum d1;

#ifdef CODEBUG
  fprintf (stderr,"edit\n");
#endif
        if (!checkstr(d1=popm())) return;
	sprintf(sysbuf,"vi %s", d1.str);
	system (sysbuf);
}

void runfil(char *rfil)
{
    extern FILE *fin;

   if (rfil)
     {
	if ((fin=fopen(rfil,"r")) == NULL)
	   {
	     fprintf(stderr,"%s: can't open %s\n", progname, rfil);
	     return;
	   }
	for (initcode(); yyparse(); resetcode())
		execute(progbase);
	fclose(fin);
     }
}

/*------------------------------------*/

