/* Segment MVID in programs MONT, MPICT and FSTOR */

/* Common video screen graphics and text subroutines,
   used to drive screen in four modes. */

/* These routines support text and graphics on the screen.
   Each graphics device has its own set of drivers 
   in an independent module. For those boards not supporting
   text in graphics mode, a software-generated text font is
   supported.

   When a program is started, a call to "txtmode()" establishes
   the screen mode by setting up an array of function calls to
   the proper drivers.  The "txtmode" routine may be called
   at any time to change text modes.

   Designed for use with Enhanced Graphics
    or Monochrome Graphics or both, or Hercules board.

   OVERLAY  mode is text overlaid on graphics screen.
	    This mode can't scroll screen up when a line feed
	    occurrs on bottom line, so only a screenful may be
	    displayed.

   MONO2ND  mode is text written directly to secondary monochrome
	    board so it doesn't disturb graphics on Enhanced board.
	    Text scrolling is supported. Needs a second video monitor.

   GRAPHICS mode is text written onto the graphics screen
            in a font generated by software.

   STDTXT   mode is text written to stderr so it goes through the
	    operating system to the console screen.

*/

/*  Latest mod  17-Jun-86                R.G.Smith       */

#include <stdio.h>
#include "mdef.h"
#include "colors.h"
#include "vdef.h"
#include "grdef.h"

/* video modes: */
/*  Originally, "TOVERLAY" mode meant text overlaid */
/*   on the graphics screen, for those graphics drivers */
/*   that supported this mode. */

/*  I found that it is easier (when transporting this system) */
/*   simply to draw text in GRAPHICS mode using a vector font;
/*   hence "TGRAPHICS" mode.   R.G. Smith   Nov, 1987 */

/* #define TOVERLAY			/* define these symbols for overlay */
/* #define TMONO2ND 			/*   and secondary monochrome output */
#define TGRAPHICS			/* text font drawn by software */

/* Optional drivers: */

/*  Define any one of these if you want to draw graphics on */
/*    any of these devices */
/*   Make sure to include the "mprint... file in makefile. */
  
/* #define MONO 			/* include system mono drivers */
/* #define THERC			/* include Hercules mono graphics */
/* #define TEGA				/* include EGA graphics */
/* #define TVGA				/* include VGA graphics */
/* #define TTX4014			/* Tektronix 4014 drivers */
#define TXWIN				/* X-windows graphics */
#define TPOSTSCRIPT			/* Postscript driver */
#define TCOLORPS			/* Color Postscript driver */
#define TDASHPS				/* Dashed Postscript driver */
/*#define TTIFF				/* TIFF format driver */
/* #define TLASER			/* HP laserjet II graphics */

#define  CLR  0				/* the text functions */
#define  XY   1				/* move alpha text to X=col, Y=row */
#define  XYV  2				/* move alpha text to graphics coord */ 
#define  PR   3				/* print string */
#define  COL  4				/* change color */
#define  DOT  5				/* write dot */
#define  MOVE  6			/* move graphics */
#define  DRAW  7			/* draw graphics */
#define  XSIZE 8			/* get X size of screen */
#define  YSIZE 9			/* get Y size of screen */
#define  XSCAL 10			/* set X size of screen */
#define  YSCAL 11			/* set Y size of screen */
#define  DUMP  12			/* dump graphics to printer */
#define  DRCIRC 13			/* draw circle */
#define  DRRECT 14			/* draw rectangle */
#define  DRTRI  15			/* draw triangle */
#define  FILL   16			/* fill circle or rectangle */
#define  NFUNCS 17			/* number of functions */

#define  OV    0			/* three different text modes */
#define  SECND 1
#define	 GR    2
#define  STD   3
#define  NMODES 4

static int  (*vidarr [NFUNCS])(); 
static int Xpos=0, Ypos=0;

extern int scrntyp;			/* type of graphics screen */

#define LINEWIDTH (STDSIZ/1000.)	/* width of line drawn on device */
static int linewidth=LINEWIDTH;


#ifdef TOVERLAY

#ifdef TEGA

extern teinit();		/* EGA funcs, "mprinte.c" */
extern teexit();
extern clrce();
extern xycone();
extern xyvide();
extern putce();				/* put dot matrix char onto screen */
extern putse(); 			/* put dot matrix string */
extern setecol(); 
extern movee();
extern drawe();
extern edot();
extern texsize();
extern teysize();
extern xscalee();
extern yscalee();

#endif /* TEGA */

#ifdef MONO

extern tminit();			/* Primary Mono funcs */
extern tmexit();			/* originally supported by VENIX */
extern clrcm();				/*  through library calls */
extern xyconm();
extern xyvidm();
extern putcm();
extern putsm(); 
extern setmcol(); 
extern move();				/* in "libtpc.a" */
extern cont();				/* in "libtpc.a" */
extern point();				/* in "libtpc.a" */
extern tmxsize();
extern tmysize();

#endif /* MONO */

#endif			/* TOVERLAY mode */

#ifdef TGRAPHICS

#ifdef TXWIN

extern txinit(void);		/* XWIN funcs, "mprintx.c" */
extern txexit(void);
extern xerase(void);
extern xyconx(int x, int y);
extern xyvidx(int x, int y);
/* extern putc(ch,stream); 	/*  */
/* extern puts(const char *);   /*  */
extern setxcol(int pen); 
extern movex(int x, int y);
extern drawx(int x, int y);
extern xcircle(int x, int y, int radius, int fill);
extern xrect(int x1, int y1, int x2, int y2, int x3, int y3, 
			int x4, int y4, int fill);
extern xdot(void);
extern txxsize(void);
extern txysize(void);
extern xscalex(double size);
extern yscalex(double size);
#endif

#ifdef TVGA

extern tvinit();		/* VGA funcs, "mprintv.c" */
extern tvexit();
extern clrcv();
extern xyconv();
extern xyvidv();
extern putcv();
extern putsv(); 
extern setvcol(); 
extern movev();
extern drawv();
extern vdot();
extern tvxsize();
extern tvysize();
extern xscalev();
extern yscalev();

#endif 

#ifdef TEGA

extern teinit();		/* Enhanced Graphics funcs, "mprinte.c" */
extern teexit();
extern clrce();
extern xycon();
extern xyvid();
extern putce();
extern putse(); 
extern setecol(); 
extern movee();
extern drawe();
extern edot();
extern texsize();
extern teysize();
extern xscalee();
extern yscalee();

#endif

#ifdef MONO

extern tminit();		/* Primary Mono funcs, "mprintm.c" */
extern tmexit();
extern clrcm();
extern xyconm();
extern xyvidm();
extern putcm();
extern putsm(); 
extern setmcol(); 
extern move();				/* in "libtpc.a" */
extern cont();				/* in "libtpc.a" */
extern point();				/* in "libtpc.a" */
extern tmxsize();
extern tmysize();

#endif


#ifdef THERC

extern thinit();		/* Hercules card functions, "mprinth.c" */
extern thexit();
extern clrch();
extern xyconh();
extern xyvidh();
extern sethcol();
extern moveh();
extern drawh();
extern hdot();
extern thxsize();
extern thysize();
extern xscalh();
extern yscalh();

#endif

#ifdef TLASER

extern tlinit();		/* Raster PS functions, "mprintl.c" */
extern tlexit();
extern clrcl();
extern xyconl();
extern xyvidl();
extern setlcol();
extern movel();
extern drawl();
extern ldot();
extern tlxsize();
extern tlysize();
extern xscall();
extern yscall();
extern grdump();

#endif

#ifdef TTIFF

extern tfinit();		/* Raster PS functions, "mprintl.c" */
extern tfexit();
extern clrcf();
extern xyconf();
extern xyvidf();
extern setfcol();
extern movef();
extern drawf();
extern fdot();
extern tfxsize();
extern tfysize();
extern xscalf();
extern yscalf();
extern grdumpf();

#endif

#ifdef TPOSTSCRIPT

extern tainit(void);                /* Postscript functions, "mprinta.c" */
extern taexit(void);
extern setacol(int pen);
extern movea(int ix, int iy);
extern drawa(int ix, int iy);
extern tadrcirc(int ix, int iy, int irad, int fill);
extern tadrrect(int x1, int y1, int x2, int y2, 
                int x3, int y3, int x4, int y4, int fill);
extern tadrtri (int x1, int y1, int x2, int y2, 
                int x3, int y3, int fill);
extern tafill(int fill);
extern taxsize(void);
extern taysize(void);
extern putstra(char *str);

#endif

#ifdef TCOLORPS

extern tcinit(void);                /* Postscript functions, "mprintc.c" */
extern tcexit(void);
extern setccol(int pen);
extern movec(int ix, int iy);
extern drawc(int ix, int iy);
extern tcdrcirc(int ix, int iy, int irad, int fill);
extern tcdrrect(int x1, int y1, int x2, int y2, 
                int x3, int y3, int x4, int y4, int fill);
extern tcdrtri (int x1, int y1, int x2, int y2, 
                int x3, int y3, int fill);
extern tcfill(int fill);
extern tcxsize(void);
extern tcysize(void);
extern putstrc(char *str);

#endif

#ifdef TDASHPS

extern tbinit(void);                /* Postscript functions, "mprintb.c" */
extern tbexit(void);
extern setbcol(int pen);
extern moveb(int ix, int iy);
extern drawb(int ix, int iy);
extern tbdrcirc(int ix, int iy, int irad, int fill);
extern tbdrrect(int ilx, int ily, int iux, int iuy, int fill);
extern tbxsize(void);
extern tbysize(void);
extern putstrb(char *str);

#endif


#ifdef TTX4014

extern clrctx(void);
extern settxcol(int color);
extern movetx(int x, int y);
extern drawtx(int x, int y);
extern xyvidtx(int x, int y);
extern xyvidvtx(int x, int y);
extern ttxsize(void);
extern ttysize(void);

#endif

#endif		/* TGRAPHICS mode */


#ifdef TMONO2ND
extern int silent;			/* defined in mpict.c */

extern tmsinit();			 
extern tmsexit();
extern clrcms();
extern xymscur();
extern putsms();
extern setmscol(); 
#endif


extern putsym(char *str);

/* extern fputs(const char *, FILE *);		/* C library functions */
/* extern fprintf(); */

extern putsc(char *str);
extern putch(char ch);
/********************************************/

txtmode(int mode)
{
   static int i,oldmode=0,oldm;

if (!mode) return oldmode;		/* return with previously set mode */

#ifdef TOVERLAY
 if (mode & OVERLAY)  			/* drivers in mprinte.c */
   {
#ifdef TVGA
    if (scrntyp & VGA)
      {
       tvinit(); 
       vidarr[CLR]  = clrcv;
       vidarr[XY]   = xyconv;
       vidarr[XYV]  = xyvidv;
       vidarr[PR]   = putsv; 
       vidarr[COL]  = setvcol; 
       vidarr[DOT]  = vdot; 
       vidarr[MOVE] = movev; 
       vidarr[DRAW] = drawv; 
       vidarr[XSIZE] = tvxsize; 
       vidarr[YSIZE] = tvysize; 
       vidarr[XSCAL] = xscalev; 
       vidarr[YSCAL] = yscalev; 
       vidarr[DUMP]  = NULL; 
       vidarr[DRCIRC]  = NULL; 
       vidarr[DRRECT]  = NULL; 
       vidarr[DRTRI]  = NULL; 
       vidarr[FILL]  = NULL; 
     }
#endif

#ifdef TEGA
    else if (scrntyp & ENHANCED)
      {
       teinit(); 
       vidarr[CLR]  = clrce;
       vidarr[XY]   = xycone;
       vidarr[XYV]  = xyvide;
       vidarr[PR]   = putse; 
       vidarr[COL]  = setecol; 
       vidarr[DOT]  = edot; 
       vidarr[MOVE] = movee; 
       vidarr[DRAW] = drawe; 
       vidarr[XSIZE] = texsize; 
       vidarr[YSIZE] = teysize; 
       vidarr[XSCAL] = xscalee; 
       vidarr[YSCAL] = yscalee; 
       vidarr[DUMP]  = NULL; 
       vidarr[DRCIRC]  = NULL; 
       vidarr[DRRECT]  = NULL; 
       vidarr[DRTRI]  = NULL; 
       vidarr[FILL]  = NULL; 
     }
#endif

#ifdef MONO
    else if (scrntyp & MONOCHR)
     {
        tminit();
        vidarr[CLR]  = clrcm;
        vidarr[XY]   = xyconm;
        vidarr[XYV]  = xyvidm;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = point;
        vidarr[COL]  = setmcol; 
        vidarr[MOVE] = move;
        vidarr[DRAW] = cont;
        vidarr[XSIZE] = tmxsize;
        vidarr[YSIZE] = tmysize;
        vidarr[XSCAL] = NULL; 
        vidarr[YSCAL] = NULL; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif

   }
 else if (oldmode & OVERLAY)
   {
#ifdef TVGA
     else if (scrntyp & VGA) tvexit();
#endif
#ifdef TEGA
     else if (scrntyp & ENHANCED) teexit();
#endif
#ifdef MONO
     else if (scrntyp & MONOCHR)  tmexit();
#endif
   }

#endif			/* ifdef OVERLAY */

#ifdef TMONO2ND 
 if (mode & MONO2ND)			/* drivers in mprintm.c */
   {
     tmsinit(); 
     vidarr[CLR]  = clrcms;
     vidarr[XY]   = xymscur;
     vidarr[XYV]  = NULL;
     vidarr[PR]   = putsms; 
     vidarr[COL]  = setmscol; 
     vidarr[DOT]  = NULL; 
     vidarr[MOVE] = NULL; 
     vidarr[DRAW] = NULL; 
     vidarr[XSIZE] = NULL; 
     vidarr[YSIZE] = NULL; 
     vidarr[XSCAL] = NULL; 
     vidarr[YSCAL] = NULL; 
     vidarr[DUMP]  = NULL; 
     vidarr[DRCIRC]  = NULL; 
     vidarr[DRRECT]  = NULL; 
     vidarr[DRTRI]  = NULL; 
     vidarr[FILL]  = NULL; 
   }
 else if (oldmode & MONO2ND)
    tmsexit();
#endif 

#ifdef TGRAPHICS
 if (mode & GRAPHICS)  			/* drivers in mprinte and mprinth */
   {
#ifdef TXWIN
    if (scrntyp & XWIN)
      {
        txinit(); 
        vidarr[CLR]  = xerase;
        vidarr[XY]   = xyconx;
        vidarr[XYV]  = xyvidx;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = xdot; 
        vidarr[COL]  = setxcol; 
        vidarr[MOVE] = movex; 
	vidarr[DRAW] = drawx; 
        vidarr[XSIZE] = txxsize; 
        vidarr[YSIZE] = txysize; 
        vidarr[XSCAL] = xscalex; 
        vidarr[YSCAL] = yscalex; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = xcircle;
        vidarr[DRRECT]  = xrect; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
      }
#endif

#ifdef TVGA
    if (scrntyp & VGA)
      {
        tvinit(); 
        vidarr[CLR]  = clrcv;
        vidarr[XY]   = xyconv;
        vidarr[XYV]  = xyvidv;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = vdot; 
        vidarr[COL]  = setvcol; 
        vidarr[MOVE] = movev; 
	vidarr[DRAW] = drawv; 
        vidarr[XSIZE] = tvxsize; 
        vidarr[YSIZE] = tvysize; 
        vidarr[XSCAL] = xscalev; 
        vidarr[YSCAL] = yscalev; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
      }
#endif
#ifdef TEGA
    else if (scrntyp & ENHANCED)
      {
        teinit(); 
        vidarr[CLR]  = clrce;
        vidarr[XY]   = xycon;
        vidarr[XYV]  = xyvid;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = edot; 
        vidarr[COL]  = setecol; 
        vidarr[MOVE] = movee; 
        vidarr[DRAW] = drawe; 
        vidarr[XSIZE] = texsize; 
        vidarr[YSIZE] = teysize; 
        vidarr[XSCAL] = xscalee; 
        vidarr[YSCAL] = yscalee; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
      }
#endif
#ifdef THERC
    else if (scrntyp & HERCULES)
     {
        thinit();
        vidarr[CLR]  = clrch;
        vidarr[XY]   = xyconh;
        vidarr[XYV]  = xyvidh;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = hdot; 
        vidarr[COL]  = sethcol; 
        vidarr[MOVE] = moveh; 
        vidarr[DRAW] = drawh; 
        vidarr[XSIZE] = thxsize; 
        vidarr[YSIZE] = thysize; 
        vidarr[XSCAL] = xscalh; 
        vidarr[YSCAL] = yscalh; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif
#ifdef TLASER
    else if (scrntyp & LASERJET)
     {
        tlinit();
        vidarr[CLR]  = clrcl;
        vidarr[XY]   = xyconl;
        vidarr[XYV]  = xyvidl;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = ldot; 
        vidarr[COL]  = setlcol; 
        vidarr[MOVE] = movel; 
        vidarr[DRAW] = drawl; 
        vidarr[XSIZE] = tlxsize; 
        vidarr[YSIZE] = tlysize; 
        vidarr[XSCAL] = xscall; 
        vidarr[YSCAL] = yscall; 
        vidarr[DUMP]  = grdump; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif
#ifdef TTIFF
    else if (scrntyp & TIFF)
     {
        tfinit();
        vidarr[CLR]  = clrcf;
        vidarr[XY]   = xyconf;
        vidarr[XYV]  = xyvidf;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = fdot; 
        vidarr[COL]  = setfcol; 
        vidarr[MOVE] = movef; 
        vidarr[DRAW] = drawf; 
        vidarr[XSIZE] = tfxsize; 
        vidarr[YSIZE] = tfysize; 
        vidarr[XSCAL] = xscalf; 
        vidarr[YSCAL] = yscalf; 
        vidarr[DUMP]  = grdumpf; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif
#ifdef TPOSTSCRIPT
    if (scrntyp & POSTSCRIPT)
 {
        tainit();
        vidarr[CLR]  = NULL;
        vidarr[XY]   = NULL;
        vidarr[XYV]  = NULL;
        vidarr[PR]   = putstra;
        vidarr[DOT]  = NULL;
        vidarr[COL]  = setacol;
        vidarr[MOVE] = movea;
        vidarr[DRAW] = drawa;
        vidarr[XSIZE] = taxsize;
        vidarr[YSIZE] = taysize;
        vidarr[XSCAL] = NULL;
        vidarr[YSCAL] = NULL;
        vidarr[DUMP]  = NULL;
        vidarr[DRCIRC]  = tadrcirc;
        vidarr[DRRECT]  = tadrrect; 
        vidarr[DRTRI]  = tadrtri; 
        vidarr[FILL]  = tafill; 
     }
#endif
#ifdef TCOLORPS
    if (scrntyp & COLORPS)
 {
        tcinit();
        vidarr[CLR]  = NULL;
        vidarr[XY]   = NULL;
        vidarr[XYV]  = NULL;
        vidarr[PR]   = putstrc;
        vidarr[DOT]  = NULL;
        vidarr[COL]  = setccol;
        vidarr[MOVE] = movec;
        vidarr[DRAW] = drawc;
        vidarr[XSIZE] = tcxsize;
        vidarr[YSIZE] = tcysize;
        vidarr[XSCAL] = NULL;
        vidarr[YSCAL] = NULL;
        vidarr[DUMP]  = NULL;
        vidarr[DRCIRC]  = tcdrcirc;
        vidarr[DRRECT]  = tcdrrect; 
        vidarr[DRTRI]  = tcdrtri; 
        vidarr[FILL]  = tcfill; 
     }
#endif

#ifdef TDASHPS
    if (scrntyp & DASHPS)
 {
        tbinit();
        vidarr[CLR]  = NULL;
        vidarr[XY]   = NULL;
        vidarr[XYV]  = NULL;
        vidarr[PR]   = putstrb;
        vidarr[DOT]  = NULL;
        vidarr[COL]  = setbcol;
        vidarr[MOVE] = moveb;
        vidarr[DRAW] = drawb;
        vidarr[XSIZE] = tbxsize;
        vidarr[YSIZE] = tbysize;
        vidarr[XSCAL] = NULL;
        vidarr[YSCAL] = NULL;
        vidarr[DUMP]  = NULL;
        vidarr[DRCIRC]  = tbdrcirc;
        vidarr[DRRECT]  = tbdrrect; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif

#ifdef TTX4014
    else if (scrntyp & TX4014)
     {
        ttxinit();
        vidarr[CLR]  = clrctx;
        vidarr[XY]   = xyvidtx;
        vidarr[XYV]  = xyvidvtx;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = NULL; 
        vidarr[COL]  = settxcol; 
        vidarr[MOVE] = movetx; 
        vidarr[DRAW] = drawtx; 
        vidarr[XSIZE] = ttxsize; 
        vidarr[YSIZE] = ttysize; 
        vidarr[XSCAL] = NULL; 
        vidarr[YSCAL] = NULL; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif
#ifdef MONO
    else if (scrntyp & MONOCHR)
     {
        tminit();
        vidarr[CLR]  = clrcm;
        vidarr[XY]   = xyconm;
        vidarr[XYV]  = xyvidm;
        vidarr[PR]   = putsym; 
        vidarr[DOT]  = point;
        vidarr[COL]  = setmcol; 
        vidarr[MOVE] = move;
        vidarr[DRAW] = cont;
        vidarr[XSIZE] = tmxsize; 
        vidarr[YSIZE] = tmysize; 
        vidarr[XSCAL] = NULL; 
        vidarr[YSCAL] = NULL; 
        vidarr[DUMP]  = NULL; 
        vidarr[DRCIRC]  = NULL; 
        vidarr[DRRECT]  = NULL; 
        vidarr[DRTRI]  = NULL; 
        vidarr[FILL]  = NULL; 
     }
#endif
   end

 else if (oldmode & GRAPHICS)
  {
#ifdef TXWIN
    if      (scrntyp & XWIN) txexit();
#endif
#ifdef THERC
    if      (scrntyp & HERCULES) thexit();
#endif
#ifdef TTX4014
    if (scrntyp & TX4014)   ttxexit();
#endif
#ifdef TVGA
    if (scrntyp & VGA) tvexit();
#endif
#ifdef TEGA
    if (scrntyp & ENHANCED) teexit();
#endif
#ifdef MONO
    if (scrntyp & MONOCHR)  tmexit();
#endif
#ifdef TPOSTSCRIPT
    if (scrntyp & POSTSCRIPT) taexit();
#endif
#ifdef TCOLORPS
    if (scrntyp & COLORPS) tcexit();
#endif
#ifdef TDASHPS
    if (scrntyp & DASHPS) tbexit();
#endif
  }
#endif	/* TGRAPHICS */

 if (mode & STDTXT)			/* drivers in C library */
   {
     vidarr[CLR]  = NULL;
     vidarr[XY]   = NULL;
     vidarr[XYV]  = NULL;
     vidarr[PR]   = putsc;
     vidarr[DOT]  = NULL;
     vidarr[COL]  = NULL;
     vidarr[MOVE] = NULL; 
     vidarr[DRAW] = NULL; 
     vidarr[XSIZE] = NULL; 
     vidarr[YSIZE] = NULL; 
     vidarr[XSCAL] = NULL; 
     vidarr[YSCAL] = NULL; 
     vidarr[DUMP]  = NULL; 
     vidarr[DRCIRC]  = NULL; 
     vidarr[DRRECT]  = NULL; 
     vidarr[DRTRI]  = NULL; 
     vidarr[FILL]  = NULL; 
   }

 oldm = oldmode;
 oldmode = mode;
 return (oldm);
}

/********************************************/

txtclr(void)
{
  int (*fpnt)(); 

  fpnt=vidarr[CLR];
  if (fpnt) (*fpnt)(); 
}

/********************************************/
  
txtxy(int x, int y)
           

/* set the text location to (X,Y) in cols, rows. */

{
  int (*fpnt)(); 

  fpnt=vidarr[XY];
  if (fpnt) (*fpnt)(x,y); 
}

/********************************************/
  
txtxyv(int x, int y)
           

/* set the text location to (X,Y) in graphics coords. */

{
  int (*fpnt)(); 

  fpnt=vidarr[XYV];
  if (fpnt) (*fpnt)(x,y); 
}

/********************************************/
  
txtstr(char *str)
{
  int (*fpnt)(); 

  fpnt=vidarr[PR];
  if (fpnt) (*fpnt)(str); 
}

/********************************************/

tprintf(FILE *stream, TEXT *str, int arg1, int arg2, int arg3, int arg4, int arg5, int arg6)
                
             
                                     

/* Print str on the screen like printf.
   If stream is not stderr, it's a file:
   print on stream as given.
*/

{
  int (*fpnt)(); 
  static TEXT outstr[120];

  if (stream == stderr) {
    sprintf (outstr,str,arg1,arg2,arg3,arg4,arg5,arg6);
    fpnt=vidarr[PR];
    if (fpnt) (*fpnt)(outstr);
  }
  else
   fprintf (stream,str,arg1,arg2,arg3,arg4,arg5,arg6);	/* output to file */
}

/********************************************/

int txtcolr (int color)
{
  int (*fpnt)(); 

  fpnt=vidarr[COL];
  if (fpnt) return ((*fpnt)(color));
}

/********************************************/

tdot(int x, int y, int color)
{
  int (*fpnt)(); 

  fpnt=vidarr[DOT];
  if (fpnt) (*fpnt)(x,y);
  Xpos = x;
  Ypos = y;

}

/********************************************/

tmove(int x, int y)
{
  int (*fpnt)(); 

  Xpos = x;
  Ypos = y;
  fpnt=vidarr[MOVE];
  if (fpnt) (*fpnt)(x,y);
}

/********************************************/

tdraw(int x, int y)
{
  int (*fpnt)(); 

  fpnt=vidarr[DRAW];
  if (fpnt) (*fpnt)(x,y);       /* First try to draw line. */
  else {                        /*  If that doesn't work, */
    fpnt=vidarr[DOT];
    if (fpnt) {                 /*  then try to draw line of dots */
       vlinseg (Xpos,Ypos,x,y,fpnt);
    }
  }
  Xpos = x;
  Ypos = y;
}

/**************************************************/

vlinseg (int x1, int y1, int x2, int y2, int (*fpnt) (/* ??? */))
                   
                    /* fpnt is point subroutine */
   
/* Bresenham's Algorithm;
   see Newman and Sproul, 2nd Ed.  for explanation.

 */

{
        static int deltax,deltay,dx,dy;
        static int incr1,incr2,d,xend,yend;
        static int xinc,yinc,xp,yp;

 deltax = x2 - x1;
 deltay = y2 - y1;
 dx = abs(deltax); dy = abs(deltay);

 if (dy <= dx)
   {
     if (deltax < 0)
       {
         yinc = ((deltay < 0) ? 1 : -1); 
         xp = x2; yp = y2; xend = x1;
       }
     else   /* deltax >= 0 */
       {
         yinc = ((deltay < 0) ? -1 : 1);
         xp = x1; yp = y1; xend = x2;
       }
     (*fpnt)(xp,yp);
     d = (dy << 1) - dx;
     incr1 = dy << 1;
     incr2 = (dy - dx) << 1;
     while (xp < xend)
       {
         xp++;
         if (d < 0) d += incr1;
         else     { yp += yinc; d += incr2; }
         (*fpnt)(xp,yp);
       }
    } 
  else   /* dy > dx */
    {
      if (deltay < 0)
        {
          xinc = ((deltax < 0) ? 1 : -1);
          xp = x2; yp = y2; yend = y1;
        }
      else
        {
          xinc = ((deltax < 0) ? -1 : 1);
          xp = x1; yp = y1; yend = y2;
        }
      (*fpnt)(xp,yp);
      d = (dx << 1) - dy;
      incr1 = dx << 1;
      incr2 = (dx - dy) << 1;
      while (yp < yend)
       {
         yp++;
         if (d < 0) d += incr1;
         else     { xp += xinc; d += incr2; }
         (*fpnt)(xp,yp);
       }
   }  /* else */
}

/********************************************/

putch(char ch)
{
 putc(ch,stderr);		/* putc() is a macro defined in stdio.h */
}

/********************************************/

putsc(char *str)
{
 fputs(str,stderr);
}

/********************************************/

TEXT *getstm (TEXT *buf, int n, FILE *stream)
{
   TEXT *getln(char *s, int n);

#ifdef TMONO2ND
  if (! silent and stream == stdin) return(getln(buf,n));	/* */
#else
  if (stream == stdin) return(getln(buf,n));			/* */
#endif
  else
    return (fgets (buf,n,stream));
}  

/********************************************/

int txsize(void)
{
  int (*fpnt)(); 

  fpnt=vidarr[XSIZE];
  if (fpnt) return ((*fpnt)());
}

/********************************************/

int tysize(void)
{
  int (*fpnt)(); 

  fpnt=vidarr[YSIZE];
  if (fpnt) return ((*fpnt)());
}

/********************************************/

double txscale(double size)
{
  double (*fpnt)(); 

  fpnt=(double (*)())vidarr[XSCAL];
  if (fpnt) return ((*fpnt)(size));
}

/********************************************/

double tyscale(double size)
{
  double (*fpnt)(); 

  fpnt=(double (*)())vidarr[YSCAL];
  if (fpnt) return ((*fpnt)(size));
}

/********************************************/

void tfill(int fill) 
{
  void (*fpnt)(); 

  fpnt=(void (*)())vidarr[FILL];
  if (fpnt) (*fpnt)(fill);
}
/********************************************/

tdrcirc(int x, int y, int rad, int fill)
                     

/* Draw circle at (x,y) with radius "rad".  If circle is to be
   filled (fill=1), then check if device can fill.
   If not, then fill circle with radial lines.
*/

{
  void (*fpnt)(); 
  double sin(double t), cos(double t);

#define PI	3.14159265358979323844	/* pi */
#define rmove(dx,dy)	tmove(Xpos + (int)(dx), Ypos + (int)(dy))
#define rpmove(dr,t)	rmove((dr)*cos((double)t), (dr)*sin((double)t))
#define rdraw(dx,dy)	tdraw(Xpos + (int)(dx), Ypos + (int)(dy))
#define rpdraw(dr,t)	rdraw((dr)*cos((double)t), (dr)*sin((double)t))

  fpnt=(void (*)())vidarr[DRCIRC];
  if (fpnt) (*fpnt)(x,y,rad,fill);
  else {
      double chord, theta;

     tmove (x,y);
     rpmove (rad,-PI/2);
     if      (rad > .2*STDSIZ) chord = .05;
     else if (rad > .1*STDSIZ) chord = .1;
     else                      chord = .2;
     rpmove (rad*chord/2,-PI);
     for (theta=0.0; theta < PI*2; theta += chord) {
       rpdraw (rad*chord,theta);
     }
     tmove (x,y);
     if (fill) {
       fpnt=(void (*)())vidarr[FILL];
       if (fpnt) (*fpnt)(fill);		/* if device has built-in fill */
       else {	
	 chord = ((double)linewidth) / rad;
         for (theta=0.0; theta < PI*2; theta += chord) {
           tmove (x,y);
           rpdraw (rad,theta);
         }
       }
     }
   }
}
/********************************************/

tdrrect(int x1, int y1, int x2, int y2, int x3, int y3, 
			int x4, int y4, int fill)

/* Draw rectangle that is oriented at any angle.
   If it is to be filled (fill=1) then draw lines inside.

   To fill the inside, find all the points along one
   end, using "filrect()" below, and make a series of
   moves and draws along the corrrect angle.
   Since we only want to write to each pixel on the
   hardare only once (or at most twice), we skip along
   the end line in increments of the hardware scale.

*/

/*          (x4,y4)                     (x3,y3)
               .
               .
            (x1,y1)                     (x2,y2)
*/

{
  void (*fpnt)(...); 

  fpnt=(void (*)())vidarr[DRRECT];
  if (fpnt) (*fpnt)(x1,y1,x2,y2,x3,y3,x4,y4,fill);
  else {

    tmove (x1,y1);
    tdraw (x2,y2);
    tdraw (x3,y3);
    tdraw (x4,y4);
    tdraw (x1,y1);
    if (fill) {
      fpnt=(void (*)())vidarr[FILL];
      if (fpnt) (*fpnt)(fill);          /* if device has built-in fill */
      else {
          double scale,temp;            /* scale from dev-ind to hardware */
          double (*sfpnt)(); 

         sfpnt=(double (*)())vidarr[XSCAL];
         if (sfpnt) scale=(*sfpnt)(1.0);   /* query for hardware scale */
         if (scale<=0) scale = 1; 
         if (sfpnt) temp=(*sfpnt)(tysize()/scale);  /* set scale back */ 
         filrect (x1,y1,x4,y4,x2,y2,(int)scale/3);  /* fill it */
      }
    }
  }
}

/********************************************/

filrect (int x1, int y1, int x2, int y2, int x3, int y3, int scale)
   
/* Bresenham's Algorithm;
   see Newman and Sproul, 2nd Ed.  for explanation.
  
   Starting along a line from (x1,y1) to (x2,y2), draw a slanted line
   parallel to (x2,y2) -> (x3,y3).

 */

{
        static int deltax,deltay,dx,dy;
        static int incr1,incr2,d,xend,yend;
        static int xinc,yinc,xp,yp;

 deltax = x2 - x1;
 deltay = y2 - y1;
 dx = abs(deltax); dy = abs(deltay);

 if (dy <= dx)
   {
     if (deltax < 0)
       {
         yinc = ((deltay < 0) ? 1 : -1)*scale; 
         xp = x2; yp = y2; xend = x1;
       }
     else   /* deltax >= 0 */
       {
         yinc = ((deltay < 0) ? -1 : 1)*scale;
         xp = x1; yp = y1; xend = x2;
       }
     drawslant(xp,yp,x1,y1,x3,y3);
     d = (dy << 1) - dx;
     incr1 = dy << 1;
     incr2 = (dy - dx) << 1;
     while (xp < xend)
       {
         xp+=scale;
         if (d < 0) d += incr1;
         else     { yp += yinc; d += incr2; }
         drawslant(xp,yp,x1,y1,x3,y3);
       }
    } 
  else   /* dy > dx */
    {
      if (deltay < 0)
        {
          xinc = ((deltax < 0) ? 1 : -1)*scale;
          xp = x2; yp = y2; yend = y1;
        }
      else
        {
          xinc = ((deltax < 0) ? -1 : 1)*scale;
          xp = x1; yp = y1; yend = y2;
        }
      drawslant(xp,yp,x1,y1,x3,y3);
      d = (dx << 1) - dy;
      incr1 = dx << 1;
      incr2 = (dx - dy) << 1;
      while (yp < yend)
       {
         yp+=scale;
         if (d < 0) d += incr1;
         else     { xp += xinc; d += incr2; }
         drawslant(xp,yp,x1,y1,x3,y3);
       }
   }  /* else */

}

/********************************************/

drawslant(int xp, int yp, int x1, int y1, int x2, int y2)
                        

/* Starting at (x,y), draw line parallel to (x1,y1) -> (x2,y2). */

{
  int dx, dy;

  dx = xp - x1;
  dy = yp - y1;
  tmove (xp,yp);
  tdraw (x2+dx,y2+dy);
} 


/********************************************/

tdrtri(int x1, int y1, int x2, int y2, int x3, int y3, int fill)

{
  void (*fpnt)(...); 

  fpnt=(void (*)())vidarr[DRTRI];
  if (fpnt) (*fpnt)(x1,y1,x2,y2,x3,y3,fill);
  else {

    tmove (x1,y1);
    tdraw (x2,y2);
    tdraw (x3,y3);
    tdraw (x1,y1);
    if (fill) {
      fpnt=(void (*)())vidarr[FILL];
      if (fpnt) (*fpnt)(fill);          /* if device has built-in fill */
      else {
          double scale,temp;            /* scale from dev-ind to hardware */
          double (*sfpnt)(); 

         sfpnt=(double (*)())vidarr[XSCAL];
         if (sfpnt) scale=(*sfpnt)(1.0);   /* query for hardware scale */
         if (scale<=0) scale = 1; 
         if (sfpnt) temp=(*sfpnt)(tysize()/scale);  /* set scale back */ 
       /*  filtri (x1,y1,x4,y4,x2,y2,(int)scale/3);  /* fill it */
      }
    }
  }
}

/********************************************/

tgrdump(void)
{
  double (*fpnt)(); 

  fpnt=(double (*)())vidarr[DUMP];
  if (fpnt) (*fpnt)();
}

/********************************************/

/* This function added by Ahsu 2/6/91 to free vid.c from machine
dependency 
*/
txtevent(int parm)
{
#ifdef TXWIN
dealwithwindow(parm);
#endif
}


