/* math.c for "nc" */

#include <errno.h>
#include "nc.h"
#include "y.tab.h"

#define MPI 3.14159265358979323846264

extern int errno;
double errcheck(double d, char *s);

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
void free(...);
char *strtok(char *, const char *);
#include <math.h>

#ifdef __cplusplus
}
#endif

double drand();
void execerror(char *s, char *t);
char *emalloc(unsigned int n);
void fftstp (double *zinr, double *zini, int after, int now, 
		int before, double *zoutr, double *zouti);
void four1(double *data, int nn, int isign);

/*-------------------------------------------------------*/

datum ncasin(datum x)
{
	x.u.val = errcheck(asin(x.u.val), "asine");
	x.vtype = NUMBER;
	return x;
}

datum ncatan(datum x)
{
	x.u.val = errcheck(atan(x.u.val), "atan");
	x.vtype = NUMBER;
	return x;
}

datum nclog(datum x)
{
	x.u.val = errcheck(log(x.u.val), "log");
	x.vtype = NUMBER;
	return x;
}

datum nclog10(datum x)
{
	x.u.val=errcheck(log10(x.u.val), "log10");
	x.vtype = NUMBER;
	return x;
}

datum ncsqrt(datum x)
{
	x.u.val=errcheck(sqrt(x.u.val), "sqrt");
	x.vtype = NUMBER;
	return x;
}
datum ncexp(datum x)
{
	x.u.val=errcheck(exp(x.u.val), "exp");
	x.vtype = NUMBER;
	return x;
}
datum ncsin(datum x)
{
	x.u.val=errcheck(sin(x.u.val), "sine");
	x.vtype = NUMBER;
	return x;
}
datum nccos(datum x)
{
	x.u.val=errcheck(cos(x.u.val), "cos");
	x.vtype = NUMBER;
	return x;
}
datum nctan(datum x)
{
	x.u.val=errcheck(tan(x.u.val), "tan");
	x.vtype = NUMBER;
	return x;
}
datum ncpow(datum x, datum y)
{
	x.u.val=errcheck(pow(x.u.val,y.u.val), "exponentiation");
	x.vtype = NUMBER;
	return x;
}
datum ncrand(void)
{
	datum d;

	d.u.val=errcheck(drand(), "random number");
	d.vtype = NUMBER;
	return d;
}
datum ncinteger(datum x)
{
	x.u.val=(double)(long)x.u.val;
	x.vtype = NUMBER;
	return x;
}
datum ncfabs(datum a)
{
	double x;
	x = a.u.val;
	a.u.val=(double)((x)<0 ? -(x) : (x));
	a.vtype = NUMBER;
	return a;
}

double ncabs(double x)
{
	return (double)((x)<0 ? -(x) : (x));
}

double errcheck(double d, char *s)		/* check result of library call */
	         
	        
{
	if (errno == EDOM) {
		errno = 0;
		execerror(s, "argument out of domain");
	} else if (errno == ERANGE) {
		errno = 0;
		execerror(s, "result out of range");
	}
	return d;
}

/****************************************/

void fft(double *real, double *imag, int asiz, int param)

/* Fourier transform of data in "real" and "imag", stored as 
   alternating real and imaginary values in array.  Internal 
   "four()" FFT routine has Fortran index usage and ignores 
   data[0].  First value must be in data[1].  Size of array 
   passed to "four()" must be power of 2, so allocate more 
   memory if necessary. 
*/

{
   int i,j,isign,arrsiz,fdatsiz;
   double *fdata=0,re,im,scale;

 if (!asiz) return;
 for (i=1,arrsiz=1; i<20; i++ ) {    /* find next higher power of 2 */
      arrsiz <<= 1;
      if (arrsiz >= asiz) break;
 }

 fdatsiz = (arrsiz+1) * 2;     		/* allocate space for next power of 2 */
 if ((fdata=(double*)emalloc(fdatsiz*sizeof(double)))==NULL) {
  fprintf (stderr,"fft: unable to allocate internal array of size %d.\n",
						fdatsiz);
  execerror ("Memory allocation error.",0);
 }
 else {
       double sum,mean;
 
    if (param!=IFFT) {                        /* if not inverse transform */
      for (i=0; i<asiz; i++) {          /* find mean of real part */
         sum = real[i];
      }
    mean = sum / asiz;
    }
    else mean = 0.0;
    for (i=j=0; i<arrsiz; i++,j+=2) { /* copy  data into new array */
       if (i<asiz) {
         fdata[j] = real[i];
         fdata[j+1] = imag[i];
       }
       else {
         fdata[j] = mean;		/* zero out the rest */
         fdata[j+1] = 0.0;
       }
     }
  }

/* for (i=0; i<arrsiz; i++) 
     printf("%10g %10g\n",real[i],imag[i]);       /* */

/* fprintf (stderr,"Running fourier transform of orig %d size %d.\n",
			asiz,arrsiz); /* */

 if (param==IFFT) isign = -1; 		/* inverse FFT */
 else             isign =  1;

 four1(fdata-1,arrsiz,isign); 

  switch (param) {

   case FFT: 
	for (i=j=0; i<asiz; i++,j+=2) { /* copy transformed data into orig array */
         real[i] = fdata[j]; 
         imag[i] = fdata[j+1];
        }
	break;
   case IFFT: scale = 1.0/arrsiz;
    	for (i=j=0; i<asiz; i++,j+=2) { /* copy transformed data into orig array */
         real[i] = fdata[j] * scale;
         imag[i] = fdata[j+1] * scale;
        }
	break;
   case PFFT: 
	for (i=j=0; i<asiz; i++,j+=2) { /* copy transformed data into orig array */
          re = fdata[j];
          im = fdata[j+1];
          real[i] = sqrt (re*re + im*im);
          imag[i] = 0.0;
        }
     break;
    case ACOV: 
        for (i=j=0; i<arrsiz; i++,j+=2) { /* copy transf data into orig array */
          re = fdata[j];
          im = fdata[j+1];
          fdata[j] = (re*re + im*im); /* mul by complex conjugate */
          fdata[j+1] = 0;
        }
        four1(fdata-1,arrsiz,-1);     /* reverse transform the product */
        scale = 1.0/arrsiz;
      for (i=j=0; i<asiz; i++,j+=2) { /* copy transf data into orig array */
          real[i] = fdata[j] * scale;
          imag[i] = fdata[j+1] * scale;
        }
     break;

  }
 if (fdata) free (fdata);
}

/****************************************/

#define SWAP(a,b) tempr=(a);(a)=(b);(b)=tempr

void four1(double *data, int nn, int isign)

/* Replaces data by its discrete Fourier transform,
   if isign is input as 1; or replaces data by nn
   times its inverse discrete Fourier transform,
   if isign is input as -1.  data is a complex array
   of length nn, input as a real array data[1..2*nn].
   nn MUST be an integer power of 2 (this is not checked for!).
   Data alternates real and imaginary in data array.
   This routine ignores data[0] because it imitates FORTRAN indexing.
*/

{
   int n, mmax,m,j,istep,i;
   double wtemp,wr,wpr,wpi,wi,theta;    /* double precis for trig recurrences */
   double tempr, tempi;

  n = nn << 1;
  j = 1;
  for (i=1; i<n; i+=2) {                /* the bit-reversal section */
    if (j>i) {
        SWAP(data[j],data[i]);          /* exchange the two complex numbers */
        SWAP(data[j+1],data[i+1]);
    }
    m = n >> 1;
    while (m >= 2 && j > m) {
        j -= m;
        m >>= 1;
    }
    j += m;
  }
  mmax = 2;             /* Here begins the Danielson-Lanczos section */
  while (n>mmax) {      /* outer loop executed log2 nn times */
    istep = mmax << 1;
    theta = isign * (6.28318530717959 / mmax);    /* init for trig recurrence */
    wtemp = sin(0.5 * theta);
    wpr = -2.0 * wtemp * wtemp;
    wpi = sin(theta);
    wr = 1.0;
    wi = 0.0;
    for (m=1; m<mmax; m+=2) {           /* the two nested loops */
        for (i=m; i<=n; i+= istep) {
           j = i+mmax;                  /* the Danielson-Lanczos formula: */
           tempr = wr*data[j] - wi*data[j+1];
           tempi = wr*data[j+1] + wi*data[j];
           data[j]   = data[i]   - tempr;
           data[j+1] = data[i+1] - tempi;
           data[i] += tempr;
           data[i+1] += tempi;
        }                               /* trigonometric recurrence */
        wr = (wtemp=wr)*wpr - wi*wpi + wr;
        wi = wi*wpr + wtemp*wpi + wi;
    }
    mmax = istep;
  }
}

/****************************************/

double mulreal(double a, double b, double c, double d);
double mulimag(double a, double b, double c, double d);

void dfft(double *z1real, double *z1imag, int n, int param)

/* This routine, while it can transform an array of any size,
   does not do an inverse FFT.  Therefore it is not currently used.  */

/* On input, z1 and z2 are complex n-vectors,
   n is length of z1 and z2,
   inzee indicates whether z1 or z2 is to be transformed.

   Both z1 and z2 are used as work arrays.
   
   On output, z1 or z2 contains the desired transform
   in the correct order. 
   Inzee indicates whether z1 or z2 contains the transform.
   inzee = 1  => z1 contains transform
   inzee = 2  => z2 contains transform

   Method:  

     The integer n is divided into its prime factors (up to
   a point).  For each such factor p, the p-transform of 
   appropriate p-subvectors of z1 (or z2) is calculated
   in fftstp and stored in a suitable way in z2 (or z1).

   Adapted from fortran routine in:

	 Conte and de Boor,
	 'Elementary Numerical Analysis' 1980
		Chapter 6, "Fast Fourier Transforms", p 283
*/

#define FFTSIZ 2048
 
{
    int i;
/*  static float z1real[FFTSIZ] = 0;    */
/*  static float z1imag[FFTSIZ] = 0;	*/
/*  static double z2real[FFTSIZ] = 0;	/* */
/*  static double z2imag[FFTSIZ] = 0;	/* */
    double *z2real, *z2imag, real,imag; 
    static int prime[] = {2,3,5,7,11,13,17,19,23,29,31,37};
    int inzee,after,before,next,nextmx,now;
    char *emalloc();

 inzee = 1;
 nextmx = 12;
/* n = readin(z1real,z1imag); 		/* read first list */

 z2real = (double *)emalloc(sizeof(double)*n);
 z2imag = (double *)emalloc(sizeof(double)*n);
 if (!z1real || !z2real) {
	fprintf (stderr,"fft: can't allocate work space...\n");
	return;
 }

 after = 1;
 before = n;
 next = 1;
 while (before != 1) {
   if ((before / prime[next-1]) * prime[next-1] < before) {
    next++;
     if (next <= nextmx) continue;
     else {
	now = before;
	before = 1;
     }
   }
   else {
     now = prime[next-1];
     before = before / prime[next-1];
   }

   if (inzee == 1)
      fftstp(z1real, z1imag, after, now, before, z2real, z2imag);
   else
      fftstp(z2real, z2imag, after, now, before, z1real, z1imag);
   inzee = 3 - inzee;
   if (before != 1) after *= now;

 }    /* while (before ) */

if (inzee==1) {
   if (param==FFT) {
/*    for (i=0; i<n; i++) 
     printf("%10g %10g\n",z1real[i],z1imag[i]); /* */
   }
   else if (param==PFFT) {
     for (i=0; i<n; i++) {
       real = z1real[i];
       imag = z1imag[i];
       z1real[i] = sqrt (real*real + imag*imag);
       z1imag[i] = 0.0;
/*     printf("%10g %10g\n",z1real[i],z1imag[i]); /* */
    }
  } 	/*  else (param==PFFT) */
}	/* if (inzee==1) */
else {
   if (param==FFT) {
    for (i=0; i<n; i++) {
/*     printf("%10g %10g\n",z2real[i],z2imag[i]); /* */
     z1real[i] = z2real[i];
     z1imag[i] = z2imag[i];
    }
   }
   else if (param==PFFT) {
    for (i=0; i<n; i++) {
       real = z2real[i];
       imag = z2imag[i];
       z1real[i] = sqrt (real*real + imag*imag);
       z1imag[i] = 0.0;
     }
   }
}
 free (z2real);
 free (z2imag);

}

/****************************************/

void fftstp (double *zinr, double *zini, int after, int now, 
			int before, double *zoutr, double *zouti)
{
    int ia,ib,in,j;
    static double angle;
    static double argreal,argimag;
    static double omegreal,omegimag;
    static double valreal,valimag;
    static double tvalreal,targreal;
    int pnt;

 angle = 2 * MPI / (now * after);
 omegreal = cos(angle);
 omegimag = -sin(angle); 
 argreal  = 1.0;
 argimag  = 0.0;
 for (j=0; j<now; j++) {
   for (ia=0; ia<after; ia++) {
     for (ib=0; ib<before; ib++) {
       pnt = ((now-1)*before+ib)*after+ia;
       valreal = *(zinr+pnt);
       valimag = *(zini+pnt);
       for (in=now-2; in>=0; in--)  {
	 pnt = (in*before+ib)*after+ia;
	 tvalreal = mulreal(valreal,valimag,argreal,argimag);
	 tvalreal = tvalreal + *(zinr+pnt);
	 valimag = mulimag(valreal,valimag,argreal,argimag);
	 valimag = valimag + *(zini+pnt);
	 valreal = tvalreal;
      }
      pnt = (ib*now+j)*after+ia;
      *(zoutr+pnt) = valreal;
      *(zouti+pnt) = valimag;
    }
    targreal = mulreal(argreal,argimag,omegreal,omegimag);
    argimag = mulimag(argreal,argimag,omegreal,omegimag);
    argreal = targreal;
  }
 }
}

/****************************************/

double mulreal(double a, double b, double c, double d)
{
 return (a*c - b*d);
}

/****************************************/

double mulimag(double a, double b, double c, double d)
{
 return (b*c + a*d);
}

/****************************************/

static char notstr[] = {"variable not a string"};
   
datum xstrlen(datum d)
{
	switch (d.vtype) {
	  case 0:
	  case NUMBER:
		execerror (notstr,0);
		break;
	  case LITCHAR:
		d.u.val = 1;
		break;
	  case STRING:
		d.u.val = strlen (d.u.str);
		break;
	}
	d.vtype = NUMBER;
	return d;
}

datum xstrtok(datum d1, datum d2)
{
	char *strtok(char *, const char *);

	if (d1.vtype!=STRING && d1.u.val!=0.0) execerror (notstr,0);
 	if (d2.vtype!=STRING) execerror ("separator is not a string",0); 
	d1.u.str = strtok(d1.u.str, d2.u.str);
	d1.vtype = STRING;
	return d1;
}


