/*
 * addrcmac.c - MEX-file implementation of CMAC address generation function
 *
 * Matlab form is addrs = addrcmac(ip,iprange,c,width,memsize)
 * where addrs is a c-dimensional vector of addresses,
 * ip is an input vector, iprange is the range of values for 
 * each element of ip, width is the generalisation width and
 * memsize is the range of addresses returned.
 *
 * written 26 August 1993 by Donald Reay
 *
 * using MetaWare High C/C++ version 3.1
 *
 * Power Electronics Group
 * Department of Computing and Electrical Engineering
 * Heriot-Watt University
 * Edinburgh UK
 *
 * e-mail dsr@cee.hw.ac.uk
 *
 */

#include <math.h>
#include "cmex.h"

extern  void addrcmac(double *, double *, int, double *, int, double *, long);

#define NO_CMAC_INARGS 5 
#define NO_CMAC_OUTARGS 1

#define IP_IN prhs[0]
#define IPRANGE_IN prhs[1]
#define C_IN prhs[2]
#define WIDTH_IN prhs[3]
#define MEMSIZE_IN prhs[4]
#define ADDRS_OUT plhs[0]

#define HASH 12345

/*
 * first part of MEX-file source is MATLAB interface
 */

void mexFunction(int nlhs, Matrix *plhs[], int nrhs, Matrix *prhs[])

{
  /* check for proper number of arguments */

  if (nrhs != NO_CMAC_INARGS)
    mexErrMsgTxt("ADDRCMAC requires five input arguments.");
  else if (nlhs != NO_CMAC_OUTARGS)
    mexErrMsgTxt("ADDRCMAC requires one output argument.");

  /* check the dimensions of ip.  ip should be a vector */

  if ((int)(mxGetN(IP_IN)) != 1)
    mexErrMsgTxt("ADDRCMAC requires that IP be a vector.");

  /* Create a matrix for the return argument */

  ADDRS_OUT = mxCreateFull((int)(*mxGetPr(C_IN)), 1, REAL);

  /* do the actual computations in a subroutine */

  addrcmac(mxGetPr(ADDRS_OUT), mxGetPr(IP_IN), (int)(mxGetM(IP_IN)),
	   mxGetPr(IPRANGE_IN), (int)(*mxGetPr(C_IN)),
	   mxGetPr(WIDTH_IN), (long)(*mxGetPr(MEMSIZE_IN)));

} /* mexFunction */

/*
 * second part of MEX-file source is C function that does the work 
 */

void addrcmac(double *addrs, double *ip, int ipdim, double *iprange,
	      int c, double *width, long memsize)

{

  int i, j;
  int quantisation;
  double shift, address, offset, ofs;

  /* compute network parameters */

  quantisation = (int)((*iprange)/(*width));
  offset = (*width)/c;
  ofs = 0.0;

  /* compute addresses */

  for (i=0 ; i<c ; i++)
  {
    address = 0; 
    shift = 1.0;
    for (j=0 ; j<ipdim ; j++)
    {
      address += (((int)((*(ip+j) + ofs)/(*width))) % quantisation)*shift;
      shift *= quantisation;
    }
    address += shift*i;

  /* check for hashing */

    if (memsize != 0)
      *(addrs+i) = (long)(log(address+1)*HASH) % memsize;
    else
      *(addrs+i) = (long)(address);  

  /* add one to address for compatibility with MATLAB indexing */

    *(addrs+i) += 1;
    ofs += offset;
  }

} /* addrcmac */
