
/* 
 *	Kohonen Self-Organizing Map routines
 */

#include <math.h>
#include <X11/Intrinsic.h>
#include "map.h"

extern	double	randm(), normal();

int	best_match();
void	init_state(), learn(), proportional_lr(), radius_alpha_lr(), get_distribution(),
	ttd_radius_alpha_lr(), ttd_proportional_lr(), infold_lr(), ttd_infold_lr(), sigma_alpha_lr(),
	ttd_sigma_alpha_lr();

double	alpha		= 0.8,
	radius		= 3.0,
	sigma		= 3.0,
	alpha_decay	= 1.0,
	zspace		= 0.14;
int	xstate		= 10,
	ystate		= 10,
	zstate		= 1,
	fault_column	= 0,
	distribution	= RECTANGLE,
	array		= RECTANGULAR,
	learning_rule	= RADIUS_ALPHA,
	dimension	= DIMENSION2,
	faults,
	neurons;
int	stx[MAXN],
	sty[MAXN],
	stz[MAXN],
	fault_loc[MAXN];

double	neuron[MAXN][DIM];

int
best_match(x, y, z)
double	x, y, z;
{
	int	best, i;
	double	min_dist, dist, dx, dy, dz;

	min_dist = 9.9;
	for (i=0; i<neurons; i++) {
		if (fault_loc[i] == 0) {
			dx = x - neuron[i][0];
			dy = y - neuron[i][1];
			if (dimension == DIMENSION3) {
				dz = z - neuron[i][2];
				dist = dx * dx + dy * dy + dz * dz;
			} else
				dist = dx * dx + dy * dy;
			if (dist < min_dist) {
				best = i;
				min_dist = dist;
			}
		}
	}
	return best;
}

void
init_state()
{
	int	x, y, z;

	faults = 0;
	fault_column = 0;
	neurons = 0;
	for (z=0; z<zstate; z++)
		for (y=0; y<ystate; y++)
			for (x=0; x<xstate; x++) {
				stx[neurons] = x;
				sty[neurons] = y;
				stz[neurons] = z;
				neuron[neurons][0] = randm() * 0.5 + 0.25;
				neuron[neurons][1] = randm() * 0.5 + 0.25;
				switch (dimension) {
				case DIMENSION2:
					neuron[neurons][2] = 0.5;
					break;
				case DIMENSION3:
					neuron[neurons][2] = randm() * 0.5 + 0.25;
					break;
				}
				fault_loc[neurons] = 0;
				neurons++;
			}
}

void
get_distribution(x, y, z)
double	*x, *y, *z;
{
	double	nx, xo, yo, zo, sigmoid;

	if (distribution == GAUSSIAN) {
		sigmoid = 0.2;
		xo = 0.5;
		yo = 0.5;
		zo = 0.5;
	} else if (distribution == CLUSTERED) {
		sigmoid = 0.07;
		xo = randm();
		if (xo < 0.5)
			xo = 0.25;
		else
			xo = 0.75;
		yo = randm();
		if (yo < 0.5)
			yo = 0.25;
		else
			yo = 0.75;
		if (zo < 0.5)
			zo = 0.25;
		else
			zo = 0.75;
	} else {
		*x = randm();
		*y = randm();
		*z = randm() * zspace + 0.5 - zspace * 0.5;
	}
	switch (distribution) {
		case CIRCLE:
			nx = fabs(*x - 0.5);
			nx = 2.0 * sqrt(0.25 - nx * nx);
			*y *= nx;
			*y += (1.0 - nx) * 0.5;
			*z *= nx;
			*z += (1.0 - nx) * 0.5;
			break;
		case TRIANGLE:
			*y *= (1.0 - *x);
			*y += *x * 0.5;
			*z *= (1.0 - *x);
			*z += *x * 0.5;
			break;
		case GAUSSIAN:
		case CLUSTERED:
			do {
				*x = normal() * sigmoid + xo;
			} while ((*x <= 0.0) || (*x >= 1.0));
			do {
				*y = normal() * sigmoid + yo;
			} while ((*y <= 0.0) || (*y >= 1.0));
			do {
				*z = normal() * sigmoid + zo;
			} while ((*z <= 0.0) || (*z >= 1.0));
			*z = *z * zspace + 0.5 - zspace * 0.5;
			break;
	}
	if (dimension == DIMENSION2)
		*z = 0.5;
}

void
radius_alpha_lr(x, y, best)
double	x, y;
int	best;
{
	int	i, j, n, yb, xe, ye;

	if (radius < 0.99) {
		neuron[best][0] += alpha * (x - neuron[best][0]);
		neuron[best][1] += alpha * (y - neuron[best][1]);
	} else switch (array) {
		case RECTANGULAR:
			i = stx[best]-(int)radius;
			yb = sty[best]-(int)radius;
			xe = stx[best]+(int)radius;
			ye = sty[best]+(int)radius;
			if (i < 0)
				i = 0;
			if (yb < 0)
				yb = 0;
			if (xe >= xstate)
				xe = xstate - 1;
			if (ye >= ystate)
				ye = ystate - 1;
			for (; i<=xe; i++)
				for (j=yb; j<=ye; j++) {
					n = j * ystate + i;
					neuron[n][0] += alpha * (x - neuron[n][0]);
					neuron[n][1] += alpha * (y - neuron[n][1]);
				}
			break;
		case RECT_4:
			i = stx[best]-(int)radius;
			xe = stx[best]+(int)radius;
			if (i < 0)
				i = 0;
			if (xe >= xstate)
				xe = xstate - 1;
			for (; i<=xe; i++) {
				yb = sty[best] - (int)radius + abs(i - stx[best]);
				ye = sty[best] + (int)radius - abs(i - stx[best]);
				if (yb < 0)
					yb = 0;
				if (ye >= ystate)
					ye = ystate - 1;
				for (j=yb; j<=ye; j++) {
					n = j * ystate + i;
					neuron[n][0] += alpha * (x - neuron[n][0]);
					neuron[n][1] += alpha * (y - neuron[n][1]);
				}
			}
			break;
		case LINEAR:
			i = best - (int)radius;
			if (i < 0)
				i = 0;
			xe = best + (int)radius;
			if (xe >= neurons)
				xe = neurons - 1;
			for (; i<=xe; i++) {
				neuron[i][0] += alpha * (x - neuron[i][0]);
				neuron[i][1] += alpha * (y - neuron[i][1]);
			}
			break;
		case HEXAGONAL:
			j = sty[best]-(int)radius;
			ye = sty[best]+(int)radius;
			if (j < 0)
				j = 0;
			if (ye >= ystate)
				ye = ystate - 1;
			for (; j<=ye; j++) {
				if (sty[best] % 2)
					i = stx[best]-(int)radius+(abs(sty[best]-j)+1)/2;
				else
					i = stx[best]-(int)radius+abs(sty[best]-j)/2;
				xe = i+2*(int)radius-abs(sty[best]-j);
				if (i < 0)
					i = 0;
				if (xe >= xstate)
					xe = xstate - 1;
				for (; i<=xe; i++) {
					n = j * ystate + i;
					neuron[n][0] += alpha * (x - neuron[n][0]);
					neuron[n][1] += alpha * (y - neuron[n][1]);
				}
			}
			break;
	}
}

void
ttd_radius_alpha_lr(x, y, z, best)
double	x, y, z;
int	best;
{
	int	i, j, k, n, yb, zb, xe, ye, ze;

	if (radius < 0.99) {
		neuron[best][0] += alpha * (x - neuron[best][0]);
		neuron[best][1] += alpha * (y - neuron[best][1]);
		neuron[best][2] += alpha * (z - neuron[best][2]);
	} else switch (array) {
		case RECTANGULAR:
			i = stx[best]-(int)radius;
			yb = sty[best]-(int)radius;
			zb = stz[best]-(int)radius;
			xe = stx[best]+(int)radius;
			ye = sty[best]+(int)radius;
			ze = stz[best]+(int)radius;
			if (i < 0)
				i = 0;
			if (yb < 0)
				yb = 0;
			if (zb < 0)
				zb = 0;
			if (xe >= xstate)
				xe = xstate - 1;
			if (ye >= ystate)
				ye = ystate - 1;
			if (ze >= zstate)
				ze = zstate - 1;
			for (; i<=xe; i++)
				for (j=yb; j<=ye; j++)
					for (k=zb; k<=ze; k++) {
						n = i + ystate * (j + xstate * k);
						neuron[n][0] += alpha * (x - neuron[n][0]);
						neuron[n][1] += alpha * (y - neuron[n][1]);
						neuron[n][2] += alpha * (z - neuron[n][2]);
					}
			break;
		case LINEAR:
			i = best - (int)radius;
			if (i < 0)
				i = 0;
			xe = best + (int)radius;
			if (xe >= neurons)
				xe = neurons - 1;
			for (; i<=xe; i++) {
				neuron[i][0] += alpha * (x - neuron[i][0]);
				neuron[i][1] += alpha * (y - neuron[i][1]);
				neuron[i][2] += alpha * (z - neuron[i][2]);
			}
			break;
	}
}

void
proportional_lr(x, y, best)
double	x, y;
int	best;
{
	double	dx, dy, db, dn;
	int	i, j, n, yb, xe, ye;

	dx = x - neuron[best][0];
	dy = y - neuron[best][1];
	db = sqrt(dx * dx + dy * dy);
	switch (array) {
		case RECTANGULAR:
			i = stx[best]-(int)radius;
			yb = sty[best]-(int)radius;
			xe = stx[best]+(int)radius;
			ye = sty[best]+(int)radius;
			if (i < 0)
				i = 0;
			if (yb < 0)
				yb = 0;
			if (xe >= xstate)
				xe = xstate - 1;
			if (ye >= ystate)
				ye = ystate - 1;
			for (; i<=xe; i++)
				for (j=yb; j<=ye; j++) {
					n = j * ystate + i;
					dx = x - neuron[n][0];
					dy = y - neuron[n][1];
					dn = sqrt(dx * dx + dy * dy);
					neuron[n][0] += db / dn * alpha * dx;
					neuron[n][1] += db / dn * alpha * dy;
				}
			break;
		case LINEAR:
			n = best - (int)radius;
			if (n < 0)
				n = 0;
			xe = best + (int)radius;
			if (xe >= neurons)
				xe = neurons - 1;
			for (; n<=xe; n++) {
				dx = x - neuron[n][0];
				dy = y - neuron[n][1];
				dn = sqrt(dx * dx + dy * dy);
				neuron[n][0] += db / dn * dx;
				neuron[n][1] += db / dn * dy;
			}
			break;
		case HEXAGONAL:
			j = sty[best]-(int)radius;
			ye = sty[best]+(int)radius;
			if (j < 0)
				j = 0;
			if (ye >= ystate)
				ye = ystate - 1;
			for (; j<=ye; j++) {
				if (sty[best] % 2)
					i = stx[best]-(int)radius+(abs(sty[best]-j)+1)/2;
				else
					i = stx[best]-(int)radius+abs(sty[best]-j)/2;
				xe = i+2*(int)radius-abs(sty[best]-j);
				if (i < 0)
					i = 0;
				if (xe >= xstate)
					xe = xstate - 1;
				for (; i<=xe; i++) {
					n = j * ystate + i;
					dx = x - neuron[n][0];
					dy = y - neuron[n][1];
					dn = sqrt(dx * dx + dy * dy);
					neuron[n][0] += db / dn * dx;
					neuron[n][1] += db / dn * dy;
				}
			}
			break;
	}

}

void
ttd_proportional_lr(x, y, z, best)
double	x, y, z;
int	best;
{
	double	dx, dy, dz, db, dn;
	int	i, j, k, n, yb, zb, xe, ye, ze;

	dx = x - neuron[best][0];
	dy = y - neuron[best][1];
	dz = z - neuron[best][2];
	db = sqrt(dx * dx + dy * dy + dz * dz);
	switch (array) {
		case RECTANGULAR:
			i = stx[best]-(int)radius;
			yb = sty[best]-(int)radius;
			zb = stz[best]-(int)radius;
			xe = stx[best]+(int)radius;
			ye = sty[best]+(int)radius;
			ze = stz[best]+(int)radius;
			if (i < 0)
				i = 0;
			if (yb < 0)
				yb = 0;
			if (xe >= xstate)
				xe = xstate - 1;
			if (ye >= ystate)
				ye = ystate - 1;
			if (ye >= ystate)
				ye = ystate - 1;
			if (ze >= zstate)
				ze = zstate - 1;
			for (; i<=xe; i++)
				for (j=yb; j<=ye; j++)
					for (k=zb; k<=ze; k++) {
						n = i + ystate * (j + xstate * k);
						dx = x - neuron[n][0];
						dy = y - neuron[n][1];
						dz = z - neuron[n][2];
						dn = sqrt(dx * dx + dy * dy + dz * dz);
						neuron[n][0] += db / dn * alpha * dx;
						neuron[n][1] += db / dn * alpha * dy;
						neuron[n][2] += db / dn * alpha * dz;
					}
			break;
	}
}

void
infold_lr(x, y)
double	x, y;
{
	int	c;

	do {
		c = randm() * neurons;
	} while (fault_loc[c]);
	neuron[c][0] = x;
	neuron[c][1] = y;
}

void
ttd_infold_lr(x, y, z)
double	x, y, z;
{
	int	c;

	do {
		c = randm() * neurons;
	} while (fault_loc[c]);
	neuron[c][0] = x;
	neuron[c][1] = y;
	neuron[c][2] = z;
}

void
sigma_alpha_lr(x, y, best)
double	x, y;
int	best;
{
	double	rad, xxx, ss, ss2, dis;
	int	n, xx, yy;
	int	nn, xb, xe, yb, ye;

	ss = sigma * sigma;
	ss2 = 1.6 * sigma;
	ss2 *= ss2;
	switch (array) {
	case RECTANGULAR:
		for (n=0; n<neurons; n++) {
			xx = stx[best] - stx[n];
			yy = sty[best] - sty[n];
			rad = alpha * exp(-(double)(xx * xx + yy * yy) / ss);
			neuron[n][0] += rad * (x - neuron[n][0]);
			neuron[n][1] += rad * (y - neuron[n][1]);
		}
		break;
	case LINEAR:
		for (n=0; n<neurons; n++) {
			xx = abs(best - n);
			rad = alpha * exp(-(double)(xx * xx) / ss);
			neuron[n][0] += rad * (x - neuron[n][0]);
			neuron[n][1] += rad * (y - neuron[n][1]);
		}
		break;
	case HEXAGONAL:
		for (n=0; n<neurons; n++) {
			xx = stx[best] - stx[n];
			yy = sty[best] - sty[n];
			dis = xx * xx + yy * yy;
			rad = alpha * (1.2 * exp(-dis / ss) - 0.2 * exp(-dis / ss2));
			neuron[n][0] += rad * (x - neuron[n][0]);
			neuron[n][1] += rad * (y - neuron[n][1]);
		}
		break;
	case RING:
		for (n=0; n<neurons; n++) {
			xx = abs(best - n);
			if (xx > neurons / 2)
				xx = neurons - xx;
			rad = alpha * exp(-(double)(xx * xx) / ss);
			neuron[n][0] += rad * (x - neuron[n][0]);
			neuron[n][1] += rad * (y - neuron[n][1]);
		}
		break;
	}
}

void
ttd_sigma_alpha_lr(x, y, z, best)
double	x, y, z;
int	best;
{
	double	rad;
	int	n, xx, yy, zz;

	switch (array) {
	case RECTANGULAR:
		for (n=0; n<neurons; n++) {
			xx = stx[best] - stx[n];
			yy = sty[best] - sty[n];
			zz = stz[best] - stz[n];
			rad = alpha * exp(-(double)(xx * xx + yy * yy + zz * zz) / (sigma * sigma));
			neuron[n][0] += rad * (x - neuron[n][0]);
			neuron[n][1] += rad * (y - neuron[n][1]);
			neuron[n][2] += rad * (z - neuron[n][2]);
		}
		break;
	}
}

void
learn()
{
	double	x, y, z;
	int	best;

	get_distribution(&x, &y, &z);
	if (learning_rule != INFOLD)
		best = best_match(x, y, z);
	switch (dimension) {
	case DIMENSION2:
		switch (learning_rule) {
		case RADIUS_ALPHA:
			radius_alpha_lr(x, y, best);
			break;
		case PROPORTIONAL:
			proportional_lr(x, y, best);
			break;
		case INFOLD:
			infold_lr(x, y);
			break;
		case SIGMA_ALPHA:
			sigma_alpha_lr(x, y, best);
			break;
		}
		break;
	case DIMENSION3:
		switch (learning_rule) {
		case RADIUS_ALPHA:
			ttd_radius_alpha_lr(x, y, z, best);
			break;
		case PROPORTIONAL:
			ttd_proportional_lr(x, y, z, best);
			break;
		case INFOLD:
			ttd_infold_lr(x, y, z);
			break;
		case SIGMA_ALPHA:
			ttd_sigma_alpha_lr(x, y, z, best);
			break;
		}
		break;
	}
}
