//**************************************************************************
//
// program:	forth.cpp
// purpose:	programmable front-end for annsim
// version:	3.1
// author:	BJ McDonald
// date:	5/9/1991
//
//**************************************************************************


#include <alloc.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <conio.h>
#include <stdio.h>

#include "forth.h"


//
// Methods definition follow ...
//

Forth::Forth()
{
	line="";							// no line
	toke[0]=0;							// no token
	errcnd=0;							// no error
	loading=0;							// no loading
//
// Other classes will clean themselves up ...
//
}

Forth::~Forth(){}


void Forth::ExecLine(char *InStr)
{
	token tstream[LINETOKEN];			// This is where we'll put the tokens

	line=InStr;							// Point at the line to process
	errcnd=0;							// reset any errors
	ctoken=tstream;						// work with the stream of tokens

	Tokenize();							// tokenize the input stream
	ctoken=tstream;						// start from beginning again
	if(!errcnd)
		Interpret();					// interpret it

	if(!errcnd&&!loading) {				// signal if no error
		clreol();
		puts(" ok");
	}

	else if(loading) cputs(".");		// get dotty

	else {
		if(p.error()>0)
			cprintf("%s\r\n",p.errstring());
		else if(r.error()>0)
			cprintf("%s\r\n",p.errstring());
	}
}


//
// Add a builtin word to list
//

void Forth::AddWord(char *nme, int delim, void (*func)())
{
	words.Addword(nme,delim,func);
}



//
// error exit routine - prints an error string
//


void Forth::Bomb(char *errstr)
{
	cputs(errstr);
	cputs("\r\n");
	(token *)ctoken=(token *)NULL;		// do not process the line further
	errcnd=1;							// signal error condition
}



//
// error exit routine - somewhat more complex
//

void Forth::Error(char *str1, char *str2)
{
	char buffer[MAXBUFFER];

	sprintf(buffer,str1,str2);
	Bomb(buffer);
}



//
// get the string token from the input string - 'line'
//

void Forth::GetNextWord(void)
{
	int i=0;	// counter

	for(;isspace(*line)&&*line!='\0';line++);	// spool past white space
	if(*line==NULL) { toke[0]=NULL; return; }
	for(;!isspace(*line)&&*line!=NULL;toke[i++]=tolower(*line++));
	toke[i]=NULL;								// terminate this with a null
	for(;isspace(*line)&&*line!='\0';line++);   // spool space white space
}



//
// Get the next token from the input 'line' up to terminator: 'term'
//

char *Forth::GetTok(char term)
{
	char *thees;						// the returned token

	while(*line==' '||*line=='\t') *line++;
	thees=line;							// we are at something
	while(*line!='\0'&&*line!='\n'&&*line!=term)
		line++;

	if(*line=='\n'||*line==term)
		*line++='\0';

	return (char *)thees;
}


//
// if current token is a number - push on pstack, else error
//

long Forth::Number(char *name)
{
	int i, flag=0;

	for(i=0;name[i]!=NULL;i++) {		// do for each character in token
		if(i==0&&name[i]=='-') i++;		// test for unary minus and digits
		if(!isdigit(name[i])) flag=1;	//  - set the flag if not number
	}

	if(flag) {
		printf("\n%s ?\n",name);	   	// it ain't no number
		*line=NULL;						// stop processing now!
		errcnd=1;						// signal error condition
		return 0;
	} else
		return atol(name);				// its a number - ppush it
}



//
// pack the string into the tokens
//

int Forth::Pack(char terminal)
{
	char *pck;			// temporary string to hold the tokens
	char *str;			// pointer to ctoken for strcpy
	int j,i,num;		// number of characters in the tokens

	ctoken++;			// move to the next token ( packed character ptr )
	pck=GetTok(terminal);	// get the next character from the line
	i=strlen(pck)+1;	// the number of characters received
	num=i/4+(i%4>0);	// calculate the number of tokens to allocate
	str=(char *)ctoken;	// point to ctoken
	for(j=0;j<num;j++){
		ctoken->num=0;	// clear it ( for safeties sake)
		ctoken++;		// skip past that many tokens
	}

	strcpy(str,pck);	// copy in the tokens
	return num;								// tell caller number packed
}


//
// remove the string from the next tokens
//

char *Forth::Unpack()
{
	char *outbuf;			// buffer to hold the string
	int num;				// number of successive tokens given pc's
	int j;

	num=ctoken->id.n;		// get the number
	ctoken++;				// start unpacking from the next token
	outbuf=(char *)ctoken;  // point at start of packed chars stream
	for(j=0;j<num-1;j++)    // don't skip past the end
		ctoken++;			// skip past that many tokens

	return outbuf;			// return the string - REMEMBER to free it !!!

}



//
// process the input line and tokenize
//

void Forth::Tokenize()
{
	token *ttoken;			// used to hold the place in token stream
	long num;				// used to insert numbers into the token stream
	long numvals[20];		// hold numbers before token stream insertion
	int loop,numindex;		// loop counter
	int wnum;				// word number


	numindex=0;				// no numbers received

	while(*line!=0) {		// continue until nothing left to process
		ctoken->num=0;		// clear it to zero
		GetNextWord();		// get the next word from the input stream
		word *sword=words.Findword(toke);	// Get a pointer to the word
		if(sword!=(word *)NULL) {			// OK so its definitely a word

			if(numindex>0) { 				// we have been getting numbers
				ctoken->id.n=numindex;		// the tokens following
				ctoken->id.z=0;				// doesn't apply here
				ctoken->id.x=0;				// signal tokens following numbers
				ctoken++;					// move to next token

				for(loop=0;loop<numindex;loop++) {
					ctoken->num=numvals[loop];
					ctoken++;
				}

				numindex=0;					// ready to get more numbers
				ctoken->num=0;				// clear to zero
			}

			char dlm=sword->delim;			// capture the delim char
			if (dlm!=0) {					// has this word got a delimiter?

				ttoken=ctoken;				// hold the position
				num=Pack(dlm);				// pack the next num tokens

				ttoken->id.n=num;			// the next num tokens are pc's
				ttoken->id.x=sword->type;	// the type of token
				ttoken->id.z=sword->code;	// save the word's code

			} else {

				ctoken->id.n=0;				// no following data tokens
				ctoken->id.x=sword->type;	// the type of token
				ctoken->id.z=sword->code;	// save the word's code
				ctoken++;					// move to the next token

			}
		} else {
			num=Number(toke);				// the number
			if(errcnd) return;				// if error bomb out
			else numvals[numindex++]=num;	// add to the holding array
		}
	}

	if(numindex>0) {		 				// we have been getting numbers

		ctoken->id.n=numindex;				// the tokens following
		ctoken->id.z=0;						// doesn't apply here
		ctoken->id.x=0;						// signal tokens following numbers
		ctoken++;							// move to next token

		for(loop=0;loop<numindex;loop++) {
			ctoken->num=numvals[loop];
			ctoken++;
		}
	}

	ctoken->num=0;							// clear it to zero
	ctoken->id.x=255;						// this is the last of the tokens
	ctoken++;
}



//
// handle the if-[else]-then statements
//

void Forth::IfStat(void)
{
	int cond;					// condition value on ptos
	token *condtoken;			// pointer to conditional buffer
	token *cbuffer;				// hold line for execution
	int loop;					// loop counter
	int num;					// number of text,number tokens to copy
	int nest=0;                 // nesting level


	cond=p.pop();						// get condition from ptos

	if((cbuffer=new token[LINETOKEN])==(token *)NULL) {
		Bomb("could not allocate cbuffer");
		return;
	}

	int welse=words.Searchword("else");	// generic mappings for else
	int wif  =words.Searchword("if");	// if &
	int wthen=words.Searchword("then");	// then

	condtoken=cbuffer;					// start at beginning
	ctoken++;							// move past the 'if'

	do {
		if (ctoken->id.x==255) {		// have we reached end of tokens
			condtoken->id.z=wthen;		// yes - substitute code for then
			condtoken->id.x=2;			// no post word processing
			condtoken->id.n=0;			// no post tokens related
			condtoken++;				// move to the next condtoken
			nest=-1;					// and signal end of processing
			cond=0;						// no more copying
		}

		else

		if (ctoken->id.z==welse&&nest==0) {// is it an else statement
			cond=!cond; 				// handle else clause
			ctoken++;					// move to next token
		}


		if (ctoken->id.z==wif) 			// we have hit an embedded if
			nest++;						// increase the nesting level

		if (ctoken->id.z==wthen)		// we have hit a then
			nest--;						// lower the nesting level

		if (cond&&nest>-1) {			// condition true

			num=ctoken->id.n;			// how many tokens to copy
			*condtoken++=*ctoken++;		// move to first token with text
			for(loop=0;loop<num;loop++)
				*condtoken++=*ctoken++; // ok now do the copy

		}
		else {

			num=ctoken->id.n;			// how many to skip ?
			ctoken++;					// move to first data token
			for(loop=0;loop<num;loop++)
				ctoken++;				// move past data token(s)
		}


	} while(nest>-1);					// append until 'then' found

	condtoken->num=0;           		// clear to zero
	condtoken->id.z=wthen;				// add a then statement
	condtoken->id.x=2;					// type of token

	condtoken++;						// move to next token
	condtoken->num=0;           		// clear to zero
	condtoken->id.x=255;				// signal termination

	r.push((long)cbuffer);				// save cbuffer for deleting
	r.push((long)ctoken);				// save pointer to after 'then'
	ctoken=cbuffer;						// re-direct line pointer to clause
	ctoken--;							// rewind ctoken for interpret
}



//
// Handle a 'then' word
//

void Forth::ThenStat(void) {			// then
	ctoken=(token *)r.pop();
	ctoken--;				            // move back a step
	delete (token *)r.pop();	        // delete memory
}



//
// if the next token is a valid word, return its address on tos
//

void Forth::Tick(){p.push((long)words.Findword(line));}



//
// if next token is word - execute, if number - put on stack else strike out
//

void Forth::Interpret()
{
	long loop;  						// loop counter
	word *sword;						// pointer to word as token
	int num;							// number of times to copy numbers

	while(ctoken->id.x!=255) {       	// while the input line not finished

		errcnd+=(p.error()+r.error());	// check out error
		if(errcnd) return;				// an error has happened

		if(ctoken->id.x==0) {			// its a number
			num=ctoken->id.n;           // set up number of copies
			for(loop=0;loop<num;loop++){
				ctoken++;				// move to next token
				p.push(ctoken->num);	// ppush number to stack
			}
			ctoken++;					// move to the next token
		} else {
			sword=words.Findword(ctoken->id.z);
			if(sword->delim!=0)           	// is this word pwp or nwp
				line=Unpack();
			switch(sword->type) {
				case(builtin):				// built-in word
					(sword->funcode)();		// do the word
					break;
				case(defined):				// user-defined word
					r.push((long )ctoken);	// push token ptr onto the rstack
					ctoken=sword->tokens;
					ctoken--;
					break;
				default:					// variable etc
					p.push((long)sword);    // ppush word addr to stack
					break;
			}
			errcnd+=(p.error()+r.error());	// check out error
			if(errcnd) return;				// an error has happened
			ctoken++;						// move to next token
		}
	}
}



//
// adds a word to the end of the word list
//

void Forth::AddWord(int code)
{
	if (code==40) {					// : definition ??

		token *dtoken;				// token pointer in dstream
		token *defstr;				// once we have the actual number of tokens
		int  ntok=0;				// number of actual tokens needed
		int  loop;					// loop'er for skipping text,number tokens
		int  i,flag=0;

		dtoken=++ctoken;

		do {
			if(dtoken->id.x!=255) {

				if(dtoken->id.z==39) flag=1;	// ';' found

				i=dtoken->id.n;					// number of tokens to copy

				for(loop=0;loop<i+1;loop++) {
					ntok++;
					dtoken++;
				}
			}
			else {								// NO ';' then
				dtoken->id.z=39;				// make one
				dtoken->id.x=0;
				dtoken->id.n=0;
				ntok++;
				flag=1;             			// signal stop
				dtoken++;						// and end processing
			}
		} while(flag!=1&&ntok<LINETOKEN);


		dtoken->num=0;								// clear dtoken
		dtoken->id.x=255;							// set as last token
		ntok++;										// finalize all

		if((defstr=new token[ntok])==(token *)NULL) {
			Error("Could not create definition memory for %s",line);
			return;
		}

		for(loop=0;loop<ntok;loop++)               	// copy from max sized array
			defstr[loop]=*ctoken++;					// to allocated size array

		words.Addword(line,0,defstr,ntok);		// add new word

		ctoken--;
		ctoken--;								// rewind back two ctokens

	} else {

		word *newword=NULL;
		char *name=NULL;

		if((newword=new word)==(word *)NULL) {
			Error("Could not allocate word struct memory for %s",line);
			return;
		}

		if ((name=new char[strlen(line)+2])==(char *)NULL) {
			delete newword;
			Error("Could not allocate word name memory for %s",line);
			return;
		}

		strcpy(name,line);

		newword->name=name;
		newword->delim='';						// no delimiter

		if (((long *)newword->data=new long(0))==(long *)NULL) {
			delete newword;
			Error("Could not allocate data memory for %s",line);
			return;
		}

		if (code>41)
			newword->data=p.pop();				// insert constant value

		switch(code) {
			case(41):newword->type=variable;	// variable
				break;
			case(45):newword->type=constant;	// constant
				break;
		}

		words.Addword(newword);					// Add it to the list
	}
}


//
// Handle an exit condition ';'
//

void Forth::ExitStat(void) { ctoken=(token *)r.pop();}


//
// Handle a 'do' loop statement
//

void Forth::DoStat(void) {						// do
	r.push((long)ctoken);						// save loop ptr
	long index=p.pop();							// get index from pstack
	long limit=p.pop();							// get limit from pstack
	r.push(limit);								// push limit on rstack
	r.push(index);								// push limit on rstack
}


//
// Handle a 'loop' statement
//

void Forth::LoopStat(void) {					// loop
	long index=r.pop();							// pop index from rstack
	long limit=r.pop();							// pop limit from rstack
	long back=r.pop();							// pop loop ptr from rstack
	if(index<limit) {   						// loop done ?
		r.push(back);       					// NO - put everything back
		r.push(limit);
		r.push(index+1);						// incrementing the index
		ctoken=(token *)back;					// and restart
	};
}


//
// Handle a 'loop+' statement
//

void Forth::LoopPStat(void) {					// +loop
	long index=r.pop();							// pop index from rstack
	long limit=r.pop();							// pop limit from rstack
	long back=r.pop();							// pop loop ptr from rstack
	if(index<limit) {   						// loop done ?
		r.push(back);        					// NO - put everything back
		r.push(limit);
		r.push(index+p.pop());					// incr'ing index by ptos
		ctoken=(token *)back;					// and restart
	};
}



//
// Handle a 'begin' statement
//

void Forth::BeginStat(void)  {p.push((long)ctoken);}	// begin



//
// Handle an 'until' statement
//

void Forth::UntilStat(void) {					// until
	long back=r.pop();							// Get begin ptr from stack
	if(!p.pop()){								// if ptos=0 loop again
		r.push(back);							// put begin ptr to stack
		ctoken=(token *)back;					// re-direct token ptr
	};
}



//
// Handle a 'load' and run from disk file
//

void Forth::LoadStat(void)
{
	FILE	*FilePtr;							// remember what file is open
	char 	buffer[FILBUFFER];					// line by line execution
	char	IdentStr[50];						// used to check ascii/tokens
	int		num1;								// identifier number

	if((FilePtr=fopen(line,"r"))==NULL)
		Error("Could not open file: %s",line);

	else {

		fgets(buffer,FILBUFFER,FilePtr);			// may be identifier
		sscanf(buffer,"%s %d",IdentStr,&num1);		// load in Identifier

		if(!strcmp(IdentStr,":TSTREAM")&&num1==101){// is it a token stream
			puts("Token loading not yet implemented");
		} else {									// It's ascii
			loading=1;								// set loading flag
			r.push((long)ctoken);					// save the old token ptr
			ExecLine(buffer);						// execute
			do {
				fgets(buffer,FILBUFFER,FilePtr);    // get input line by line
				ExecLine(buffer);                   // and execute it
			} while(!feof(FilePtr));				// do until eof is hit

			ctoken=(token *)r.pop();				// restore old line ptr
			loading=0;
		}
		fclose(FilePtr);							// close up shop
	}

}



//
// Handle a 'save'.  Saves the data to disk.
//

void Forth::SaveStat(void)
{
	puts("Token saving not yet implemented");
}




//
// Handle a 'list' statement
//

void Forth::ListStat(void) {           				// list
	for(word *lword=words.Tail();
		lword!=NULL;lword=lword->prev)
			puts(lword->name);
}



//
// Handle a 'forget' statement
//

void Forth::ForgetWord()
{
	words.Forgetword(line);
}


//
// Handle a 'delim' statement
//

void Forth::DelimStat(void) {					// delim

	Forth::Tick();								// line holds name of word

	word *fword=(word *)p.pop();				// get address of word

	if(fword==(struct word *)NULL)				// non-existant word
		Error("Word: %s is non-existant ",line);
	else
		fword->delim=(char)p.pop();
}

int Forth::ReadError(void){	return errcnd;}

char *Forth::ReadLine(void){ return line;}

void Forth::SetData(void) {
	word *sword=(word *)p.pop();			// pointer to data on ptos
	sword->data=p.pop();						// set data
}

void Forth::ReadData(void) {
	word *sword=(word *)p.pop();			// pointer on ptos
	p.push(sword->data);						// place data on ptos
}

void Forth::ForgetSince(void) {
	words.Forgetsince(line);
}

