// ************************************************************************
//
// This is Scott Fahlman's quickprop program translated from Common Lisp
// into C by Terry Regier at the University of California, Berkeley.
// Netmail address: regier@cogsci.berkeley.edu
// This version is Quickprop 1 from September, 1988.
//
// An example of network setup data is included at the end of this file.
//
// The algorithm and some test results are described in Fahlman's paper
// "Faster-Learning Variations on Back-Propagation: An Empirical Study"
// in Proceedings of 1988 Connectionist Models Summer School, published
// by Morgan Kaufmann.
//
// Note: the parameter called "mu" in the paper is called "max-factor" here.
//
// ************************************************************************
//
// The C version of quikprop has itself been translated into a class
// to be incorporated into C++ programmes by Bruce McDonald at the
// University of Natal, Durban.  Mail address is:
// mcdonald@daisy.ee.und.ac.za.
//
// class and methods version >>> 0.1 <<<<
//
// Version		Comments
// 0.1			Setting up listing using C++ comments, removing unnecessary
//				braces, understanding each software component.  Not yet
//				compiled.
// 1.0          Everything works OK ( I think )
// 2.0			Setting up for digital approximations by the addition of
//				new class threshold
//
// ************************************************************************
//

#include "threshld.h"
#include "matrix.h"

class NeuralNet {

//
// general variables
//
	int		Epoch;				// Current epoch number .
	float	WeightRange;		// Random-init weights in range [-WR,+WR] .
	float	SigmoidPrimeOffset;	// Add to sigmoid-prime to kill flat spots .
	float	Epsilon;			// For grad descent if last step was (almost) 0 .
	float	Momentum;			// Normal old momentum term .
	float	ModeSwitchThreshold;// Inside thresh, do grad descent; outside, jump. .
	float	MaxFactor;			// Don't jump more than this times last step .
	float	Decay;				// Weight decay .
	int		Step;				// Turned to 1 after each pause, briefly .
	float	TotalError;			// Total output error for one epoch .
	float	ScoreThreshold;		// This close to desired value => bit is correct .
	int		TotalErrorBits;		// Total # bits in epoch that were wrong .
	int		DidGradient;		// Total # patterns that did gradient descent .
//
// Operations flags
//
	int		FlagNet;			// 1 => Network structures alloc'd else none
	int		FlagTest;			// 1 => Test data matrices alloc'd else none
	int		FlagTrain;			// 1 => Training data matrices alloc'd
	int		HyperErr;			// 1 => use atanh error function .
	int		SplitEpsilon;		// 1 => divide epsilon by fan-in before use .
	int		SinglePass;			// 1 => Pause after forward/backward cycle .
	int		SingleEpoch;		// 1 => Pause after each training epoch .
	int		Restart;			// 1 => restart when max epochs reached .
	int		KeepScore;			// 1 => accumulate error score for each epoch .
	int		DigitSim;			// 1 => Limited resolution-digital simulation.
//
// These define the net's nodes
//
	int		Nunits;				// Total number of units in net .
	int		Ninputs;			// Number of input units .
	int		FirstHidden;		// Index of 1st hidden unit .
	int		Nhidden;			// Number of hidden units .
	int		FirstOutput;		// Index of 1st output unit .
	int		Noutputs;			// Number of output units .
//
// Array and matrix stuff that needs to be allocated
//
	float far *Outputs;			// Final output value for each unit
	float far *ErrorSums;		// Total error activation for each unit
	float far *Errors;			// Final error value for each unit
	int	  far *Nconnections;	// # of INCOMING connections per unit
	int	  far **Connections;	// C[i][j] lists jth unit projecting to unit i
	Matrix 	Weights;			// W[i][j] holds weight of C[i][j].
	Matrix 	DeltaWeights;		// Change between previous weight and current one
	Matrix 	Slopes;				// Accumulated slope value for each position
	Matrix 	PrevSlopes;			// Similarly, for the last position visited

	int		NTrainingPatterns;	// !! Not in Lisp version.  Needed here.
	int		NTestPatterns;		// !! Not in Lisp version.  Needed here.
	Matrix 	TrainingInputs;
	Matrix 	TrainingOutputs;
	Matrix 	TestInputs;
	Matrix 	TestOutputs;

	float far *tinputs;			// Input vector to be tested. .

	Threshold thresh;			// Synapse unit for threshold functioning

public:
	NeuralNet();					// The constructor
	~NeuralNet();					// The Destructor

	void	InitInternals(void);	// set up the internal variables
	void	BuildNet(int,int,int);	// Alloc memory to build the net
	void	BuildTrains(void);		// Alloc memory for training matrix
	void	BuildTests(void);		// Alloc memory for test matrix
	void	FreeNet(void);			// Free up the network
	void	FreeTrains(void);		// Free up the test data matrix
	void	FreeTests(void);		// Free up the training data matrix
	void	LoadNet(char *);		// Load a net with specified name
	void	SaveNet(char *);		// Save a net with specified name
	void	LoadWeights(char *);	// Load a set of weights from disk
	void	SaveWeights(char *);	// Save a set of weights to disk
	void	LoadTrains(char *);		// Load training patterns
	void	LoadTests(char *);		// Load Test patterns
	void	ConnectNode(int,int);	// connect a node to another
	void	ConnectLayer(int,int,int,int);
	void	InitWeights(void);		// initialize weights
	float	RandWeight(float);		// Return a weight in range
	void	ClearSlopes(void);		// Clear all slopes
	float	Sigmoid(float);			// Sigmoid function
	float	SigmoidPrime(float);	// Derivative of sigmoid function
	float	ErrFun(float,float);	// Return the error
	void	Forward(float *);		// Do a forward pass
	void	Backward(float *);		// Do a backward pass
	void	UpdateWeights(void);	// Update all the weights
	void	TrainOnce(void);		// Train the network once
	void	Train(int);				// Train the network n times
	void 	Test(void);				// Test the Network
	int		NumUnits(void) 			// Return the number of units in net
			{return Nunits;}
	int		NumInput(void)			// Return the number of input units
			{return Ninputs;}
	int		NumHidden(void)			// Return the number of hidden units
			{return Nhidden;}
	int		NumOutput(void)			// Return the number of output units
			{return Noutputs;}
//
//	Set The Internals
//
	void	SetWeightRange(float n) 	{WeightRange=n;}
	void	SetSigPriOffset(float n)	{SigmoidPrimeOffset=n;}
	void	SetEpsilon(float n)			{Epsilon=n;}
	void	SetMaxFactor(float n)		{MaxFactor=n;}
	void	SetDecay(float n)			{Decay=n;}
	void	SetSplitEpsilon(int n)		{SplitEpsilon=n;}
//
//	Read the internals
//
	float	ReadTotalError(void)		{return TotalError;}
	int		ReadDidGradient(void)		{return DidGradient;}
	int		ReadTotalErrorBits(void)	{return TotalErrorBits;}
	float	ReadErrOf(int aNode)		{return Errors[aNode];}
	float	ReadOutputOf(int aNode)		{return Outputs[aNode];}
	float	ReadErrSumOf(int aNode)		{return ErrorSums[aNode];}
	int		ReadNConnsOf(int aNode)		{return Nconnections[aNode];}
//
//	Matrix reads
//

	int		ReadConnBetween(int N1, int N2)
										{return Connections[N1][N2];}
	float	ReadWeightBetween(int N1, int N2)
										{return Weights.data[N1][N2];}
	float	ReadDWeightBetween(int N1, int N2)
										{return DeltaWeights.data[N1][N2];}
	float	ReadSlopeBetween(int N1, int N2)
										{return Slopes.data[N1][N2];}
	float	ReadPSlopeBetween(int N1, int N2)
										{return PrevSlopes.data[N1][N2];}
//
// Matrix writes
//
	void	SetConnBetween(int N1, int N2, int val)
										{Connections[N1][N2]=val;}
	void	SetWeightBetween(int N1, int N2, float val)
										{Weights.data[N1][N2]=val;}
	void	SetDWeightBetween(int N1, int N2, float val)
										{DeltaWeights.data[N1][N2]=val;}
	void	SetSlopeBetween(int N1, int N2,float val)
										{Slopes.data[N1][N2]=val;}
	void	SetPSlopeBetween(int N1, int N2, float val)
										{PrevSlopes.data[N1][N2]=val;}
};

