// ************************************************************************
//
// This is Scott Fahlman's quickprop program translated from Common Lisp
// into C by Terry Regier at the University of California, Berkeley.
// Netmail address: regier@cogsci.berkeley.edu
// This version is Quickprop 1 from September, 1988.
//
// An example of network setup data is included at the end of this file.
//
// The algorithm and some test results are described in Fahlman's paper
// "Faster-Learning Variations on Back-Propagation: An Empirical Study"
// in Proceedings of 1988 Connectionist Models Summer School, published
// by Morgan Kaufmann.
//
// Note: the parameter called "mu" in the paper is called "max-factor" here.
//
// ************************************************************************
//
// The C version of quikprop has itself been translated into a class
// to be incorporated into C++ programmes by Bruce McDonald at the
// University of Natal, Durban.  Mail address is:
// mcdonald@daisy.ee.und.ac.za.
//
// class and methods version >>> 0.1 <<<<
//
// Version		Comments
// 0.1			Setting up listing using C++ comments, removing unnecessary
//				braces, understanding each software component.  Not yet
//				compiled.
// 1.0			Kinda working - still lots of debugging to go ..
// 1.2			Works OK.  Still testing ...
// 2.0			Works fine - adding digital synapse implementation
//
// ************************************************************************
//

#include "ann.h"
#include <stdio.h>
#include <math.h>
#include <stdlib.h>


//
//  Get and initialize a network.
//

NeuralNet::NeuralNet()
{
	NeuralNet::InitInternals();
}


//
// Release all memory to the OS
//

NeuralNet::~NeuralNet()
{
	NeuralNet::FreeNet();
	NeuralNet::FreeTrains();
	NeuralNet::FreeTests();
}


//
// Load a network configuration from a diskfile
//

void NeuralNet::LoadNet(char *fname)
{
	FILE 	*infile;
	char	junk[10][80];
	char	stringjunk[80];
	char	realfname[80];
	char	c;
	int		temp[10], i, j, connect, flag;
//
// Get numbers of input, hidden, and output units. .
//
	sprintf (realfname, "%s.net", fname );
	if((infile = fopen ( realfname, "r" ))==NULL)
		fprintf(stderr,"Could not open file: %s\n",realfname);
	else {

		c = 'c';           // Discard leading comments .
		while (c != '#')
			fscanf (infile, "%c", &c);
//
// Get numbers of inputs, hidden units, and output units .
//
		fscanf (infile, "%s %d %s %d %s %d",
			junk[0], &temp[0], junk[1], &temp[1], junk[2], &temp[2]);

		FreeNet();
		FreeTrains();
		FreeTests();
		InitInternals();
		BuildNet( temp[0], temp[1], temp[2] );
//
// Connect layers. .
//
		fscanf (infile, "%s %d", junk[0], &connect);

		for (i=0; i<connect; i++) {         // Reading ConnectLayer lines .
			fscanf (infile, "%d %d %d %d",
				&temp[0], &temp[1], &temp[2], &temp[3]);
			ConnectLayer ( temp[0], temp[1], temp[2], temp[3] );
		}
//
// Read in number of training patterns, then patterns themselves .
//
		fscanf (infile, "%s %d", junk[0], &NTrainingPatterns);
//
// Alloate the Training inputs matrix
//
		NeuralNet::BuildTrains();
//
// Read in the data items
//
		if (FlagTrain)
			for (i=0; i<NTrainingPatterns; i++) {
				for (j=0; j<Ninputs; j++)
					fscanf (infile, "%f", &TrainingInputs.data[i][j]);
				for (j=0; j<Noutputs; j++)
					fscanf (infile, "%f", &TrainingOutputs.data[i][j]);
			}
//
// Read in number of test patterns, then patterns themselves .
//
		fscanf (infile, "%s %d", junk[0], &NTestPatterns);
//
// Allocate the Test inputs matrix
//
		NeuralNet::BuildTests();
//
// Read in the data items
//
		if (FlagTest)
			for (i=0; i<NTestPatterns; i++) {
				for (j=0; j<Ninputs; j++)
					fscanf (infile, "%f", &TestInputs.data[i][j]);
				for (j=0; j<Noutputs; j++)
					fscanf (infile, "%f", &TestOutputs.data[i][j]);
			}
	fclose(infile);
	}
}


//
// Load a set of training values from a disk file
//

void NeuralNet::LoadTrains(char *fname)
{
	FILE 	*infile;
	char	realfname[80],junk[20];
	int		i, j, flag;
//
// Get numbers of input, hidden, and output units. .
//
	sprintf (realfname, "%s.trn", fname );
	if((infile = fopen ( realfname, "r" ))==NULL)
		fprintf(stderr,"Could not open file: %s\n",realfname);
	else {
//
// Free all previous alloc'd memory
//
		NeuralNet::FreeTrains();
//
// Read in number of training patterns, then patterns themselves .
//
		fscanf (infile, "%s %d", junk[0], &NTrainingPatterns);

		NeuralNet::BuildTrains();

		if(FlagTrain) {
			for (i=0; i<NTrainingPatterns; i++) {
				for (j=0; j<Ninputs; j++)
					fscanf (infile, "%f", &TrainingInputs.data[i][j]);
				for (j=0; j<Noutputs; j++)
					fscanf (infile, "%f", &TrainingOutputs.data[i][j]);
			}
		}
	}
	fclose(infile);
}


//
// Load a set of test values from a disk file
//

void NeuralNet::LoadTests(char *fname)
{
	FILE 	*infile;
	char	realfname[80],junk[20];
	int		i, j, flag;

	sprintf (realfname, "%s.tst", fname );
	if((infile = fopen ( realfname, "r" ))==NULL)
		fprintf(stderr,"Could not open file: %s\n",realfname);
	else {
//
// Forget the previous defined tests
//
		NeuralNet::FreeTests();
//
// Read in number of test patterns, then patterns themselves .
//
		fscanf (infile, "%s %d", junk[0], &NTestPatterns);

		NeuralNet::BuildTests();

		if (FlagTest) {
			for (i=0; i<NTestPatterns; i++) {
				for (j=0; j<Ninputs; j++)
					fscanf (infile, "%f", &TestInputs.data[i][j]);
				for (j=0; j<Noutputs; j++)
					fscanf (infile, "%f", &TestOutputs.data[i][j]);
			}
		}
	}
	fclose(infile);
}


//
// Save a network configuration to a disk file
//

void NeuralNet::SaveNet(char *nme)
{
	puts("Not yet implemented");
	puts(nme);
}


//
// Connect one unit to another unit
//

void NeuralNet::ConnectNode(int n1, int n2)
{
	NeuralNet::ConnectLayer(n1,n1,n2,n2);
}

//
//  Dump weights in the specified file.
//

void NeuralNet::SaveWeights(char *fname)
{
	FILE  *outfile;
	int  i, j;
	char realfname[80];
//
// Dump weights .
//
	sprintf (realfname, "%s.wts", fname);
	if((outfile=fopen(realfname,"w"))==NULL)
		fprintf(stderr,"Error opening file: %s\n",realfname);
	else {
		for (i=0; i<Nunits; i++)
			for (j=0; j<Nunits; j++) {
				if ( Weights.data[i][j] != 0.0 )
					fprintf (outfile, "%d %d %f ", i, j, Weights.data[i][j]);
				else
					fprintf (outfile, "%d %d %f ", -1, -1, -1.0);
					// Signal EOF .
			}
		fclose (outfile);
	}
}


//
//  Get weights from the specified file.
//

void NeuralNet::LoadWeights(char *fname)
{
	FILE  *infile;
	int  i, j;
	float **inweight;
	char realfname[80];
//
// Work out filename, open it up and Get weights .
//
	sprintf (realfname, "%s.wts", fname);
	if((infile=fopen(realfname,"r"))==NULL)
		fprintf(stderr,"Error opening file: %s\n",realfname);
	else {
		for (i=0; i<Nunits; i++)				// Do the columns
			for (j=0; j<Nunits; j++)			// and the rows
				fscanf (infile, "%d %d %f", &i, &j, &Weights.data[i][j]);
		fclose (infile);
	}
}


//
// Initialise the private member variables
//

void NeuralNet::InitInternals(void)
{
	Epoch = 0;
	WeightRange = 1.0;
	SigmoidPrimeOffset = 0.1;
	HyperErr = 1;
	SplitEpsilon = 1;
	Epsilon = 1.1;
	Momentum = 0.3;
	ModeSwitchThreshold = 0.0;
	MaxFactor = 1.75;
	Decay = -0.001;
	SinglePass = 0;
	SingleEpoch = 0;
	Step = 0;
	Restart = 1;
	KeepScore = 0;
	TotalError = 0.0;
	ScoreThreshold = 0.4;
	TotalErrorBits = 0;
	FlagNet=0;
	FlagTrain=0;
	FlagTest=0;
}


//
// Set up the connectivity matrix
//

void NeuralNet::BuildNet (int ninputs, int nhidden, int noutputs)
{
	int i, flag =0;
//
// Calculate the unit indices
//
	Nunits      = 1 + ninputs + nhidden + noutputs;
	Ninputs     = ninputs;
	FirstHidden = 1 + ninputs;
	Nhidden     = nhidden;
	FirstOutput = 1 + ninputs + nhidden;
	Noutputs    = noutputs;

//
// Allocate the memory
//

	if ((Outputs=new float[Nunits])==NULL) flag=1;
	if ((ErrorSums=new float[Nunits])==NULL) flag=1;
	if ((Errors=new float[Nunits])==NULL) flag=1;
	if ((Nconnections=new int[Nunits])==NULL) flag=1;
//
// Allocate the connections matrix
//
	if ((Connections=new int*[Nunits])!=NULL)
		for(i=0;i<Nunits;i++)
			Connections[i]=new int[Nunits];
	else
		flag=1;
//
// Allocate the weights matrix
//
	Weights.Make(Nunits);

//
// Allocate the delta weights matrix
//
	DeltaWeights.Make(Nunits);

//
// Allocate the slopes matrix
//
	Slopes.Make(Nunits);

//
// Allocate the previous slopes matrix
//
	PrevSlopes.Make(Nunits);

	for (i=0; i<=Nunits; i++) {
		Outputs[i] = 0.0;
		ErrorSums[i] = 0.0;
		Errors[i] = 0.0;
		Nconnections[i] = 0;
	}

	Outputs[0] = 1.0;        // The bias unit .

	if(flag) return;		 // junk
}


//
// Build the test patterns matrix
//

void NeuralNet::BuildTests(void)
{
	int i;

	if (NTestPatterns==0) return;

	TestInputs.Make(NTestPatterns,Ninputs);
//
// Allocate the test outputs matrix
//
	TestOutputs.Make(NTestPatterns,Noutputs);

	tinputs=new float[Ninputs];

	FlagTest=1;
}


//
// Build the training pattern matrix
//

void NeuralNet::BuildTrains(void)
{
	int i;

	if (NTrainingPatterns==0) return;

	TrainingInputs.Make(NTrainingPatterns,Ninputs);
//
// Allocate the training outputs matrix
//
	TrainingOutputs.Make(NTrainingPatterns,Noutputs);

	FlagTrain=1;
}


//
// Free net memory allocated by this class
//

void NeuralNet::FreeNet(void)
{
	if (FlagNet==0) return;

	delete [Nunits]Outputs;
	delete [Nunits]ErrorSums;
	delete [Nunits]Errors;
	delete [Nunits]Nconnections;
	delete [Nunits]tinputs;

	for (int i=0;i<Nunits;i++)
		delete [Nunits]Connections[i];

	delete [Nunits]Connections;

	Weights.Free();
	DeltaWeights.Free();
	Slopes.Free();
	PrevSlopes.Free();


	FlagNet=0;
	Nunits=0;
}


//
// Free Training matrix memory held by this class
//

void NeuralNet::FreeTrains(void)
{

	if (FlagTrain==0) return;

	TrainingInputs.Free();
	TrainingOutputs.Free();

	FlagTrain=0;					// training matrices de-alloc'd
}


//
// Free Test matrix memory held by this class
//

void NeuralNet::FreeTests(void)
{
	if (FlagTest==0) return;

	TestInputs.Free();
	TestOutputs.Free();

	FlagTest=0;						// testing matrices de-alloc'd
}


//
// Return a float between -range and +range.
//

float NeuralNet::RandWeight (float range)
{
	 return ( (float) (range * (rand()%1000 / 500.0)) - range );
}



//
//  Build a connection from every unit in range1 to every unit in range2.
//  Also add a connection from the bias unit (unit 0) to every unit in
//  range2.  Set up random weights on links.
//

void NeuralNet::ConnectLayer (int start1, int end1, int start2, int end2)
{
	int n, i, j, k;

	Epoch = 0;
	n = 2 + (end1 - start1);   //  Include the bias unit, too.  .

	for (i=start2; i<=end2; i++) {
		Nconnections[i] = n;
		Connections[i][0] = 0;
		Weights.data[i][0]=RandWeight(WeightRange);
		DeltaWeights.data[i][0]=0.0;
		Slopes.data[i][0]=0.0;
		PrevSlopes.data[i][0]=0.0;

		k = 1;
		for (j=start1; j<=end1; j++) {
			Connections[i][k]=j;
			Weights.data[i][k]=RandWeight(WeightRange);
			DeltaWeights.data[i][k]=0.0;
			Slopes.data[i][k]=0.0;
			PrevSlopes.data[i][k]=0.0;
			k++;
		}
	}
}



//
//  For each connection, select a random initial weight between WeightRange
//  and its negative.  Clear delta and previous delta values.
//

void NeuralNet::InitWeights()
{
	int i, j;

	for (i=0; i<Nunits; i++)
		for (j=0; j<Nconnections[i]; j++) {
			Weights.data[i][j]=RandWeight(WeightRange);
			DeltaWeights.data[i][j]=0.0;
			Slopes.data[i][j]=0.0;
			PrevSlopes.data[i][j]=0.0;
		}
}


//
//  Save the current slope values as PrevSlopes, and "clear" all current
//  slopes (actually set to corresponding weight, decayed a bit).
//

void NeuralNet::ClearSlopes()
{
	int i, j;

	for (i=FirstHidden; i<Nunits; i++)
		for (j=0; j<Nconnections[i]; j++) {
			PrevSlopes.data[i][j]=Slopes.data[i][j];
			Slopes.data[i][j]=Decay * Weights.data[i][j];
		}
}


//
//  Squashing function to condition the node inputs
//

float NeuralNet::Sigmoid (float activation)
{
	return thresh.T(activation);
}


//
//  Derivative of the squashing function
//

float NeuralNet::SigmoidPrime (float output)
{
	return ( SigmoidPrimeOffset + (output * (1.0 - output)) );
}


//
//  Compute the error for one output unit.
//  HyperErr==0 => use squared error.
//  HyperErr==1 => use atanh.
//

float NeuralNet::ErrFun (float desired, float actual)
{
	float dif;

	dif = desired - actual;

	if ( KeepScore ) {
		TotalError += dif*dif;
		if ( fabs(dif) >= ScoreThreshold )
			TotalErrorBits++;
	}

	if ( HyperErr == 0 )	// Not using atanh for error.
		if ((-0.1 < dif) && (dif < 0.1))	// check tolerance
			return (0.0);   				// below tolerance
		else
			return (dif);					// above tolerance
	else					// Using atanh for error .
		if ( dif < -.9999999 )
			return (-17.0);
		else
			if ( dif > .9999999 )
				return (17.0);
			else
				return ( log ( (1.0+dif) / (1.0-dif) ) );
}



//
//  This is it, ya Habaayib (? - BJM):  the forward pass in BP.
//

void NeuralNet::Forward (float *input)
{
	int i, j;
	float sum;
//
// Load in the input vector
//
	for (i=0; i<Ninputs; i++)
		Outputs[i+1] = input[i];

//
// For each unit, collect incoming activation and pass through sigmoid. .
//
	for (j=FirstHidden; j<Nunits; j++) {
		sum = 0.0;
		for (i=0; i<Nconnections[j]; i++)
			sum += ( Outputs[ Connections[j][i] ] * Weights.data[j][i] );

		Outputs[j] = Sigmoid(sum);
	}
}


//
//  Goal is a vector of desired values for the output units.  Propagate
//  the error back through the net, accumulating weight deltas.
//

void NeuralNet::Backward (float *goal)
{
	int i, j, cix;     // cix is "connection index"

//
// Compute error sums for output and other nodes
//
	for (i=0; i<Nunits; i++)		// Set the ErrorSum initial value
		if (i>=FirstOutput)			// output units set to the difference
			ErrorSums[i] = ErrFun( goal[i-FirstOutput], Outputs[i]);
		else
			ErrorSums[i] = 0.0;		// other units set to zero.
//
// Back-propagate from the last unit in the net.  When we reach a
// given unit in loop, the error from all later units will have
// been collected in the variable ErrorSums.  The final error value
// for the node is given by the variable Errors.
//
	for (j=Nunits-1; j>=FirstHidden; j--) {
		Errors[j] = SigmoidPrime(Outputs[j]) * ErrorSums[j];
		for (i=0; i<Nconnections[j]; i++) {
			cix = Connections[j][i];
//
// cix is the i th node connected to node j and it is this node which has
// its ErrorSum variable increased by the error value of the current node
// multiplied by the connection weighting between them.
//
			ErrorSums[cix] += Errors[j] * Weights.data[j][i];

			Slopes.data[j][i] += Errors[j] * Outputs[cix];
		}
	}
}


//
//  Update all weights in the network as a function of each weight's current
//  slope, previous slope, and the size of the last jump.
//

void NeuralNet::UpdateWeights()
{
	int i, j;				// unit counters
	float next_step;		// the amount to add to the current weight value
	float shrink_factor;	//

	shrink_factor = MaxFactor / ( 1.0 + MaxFactor );

	for (j=FirstHidden; j<Nunits; j++)
		for (i=0; i<Nconnections[j]; i++) {
			next_step = 0.0;

			if ( DeltaWeights.data[j][i] > ModeSwitchThreshold ) {
				// Last step was significant +ive..... .

				if ( Slopes.data[j][i] > 0.0 ) // Add in epsilon if +ive slope .
					next_step += (SplitEpsilon ?
						( (Epsilon * Slopes.data[j][i]) / Nconnections[j] ) :
						( Epsilon * Slopes.data[j][i]));
					// If slope > (or close to) prev slope, take max size step. .
				if ( Slopes.data[j][i] > (shrink_factor * PrevSlopes.data[j][i]) )
					next_step += ( MaxFactor * DeltaWeights.data[j][i]);
				else        //  Use quadratic estimate .
					next_step += ( (Slopes.data[j][i]/(PrevSlopes.data[j][i]-
						Slopes.data[j][i]))*DeltaWeights.data[j][i]);
			} else

			if ( DeltaWeights.data[j][i] < -ModeSwitchThreshold ) {
			// Last step was significant -ive.... .
				if ( Slopes.data[j][i] < 0.0 ) // Add in epsilon if -ive slope .
					next_step += (SplitEpsilon ?
						( (Epsilon * Slopes.data[j][i]) / Nconnections[j] ) :
						( Epsilon * Slopes.data[j][i] ));
					// If slope < (or close to) prev slope, take max size step. .
					if ( Slopes.data[j][i] < (shrink_factor * PrevSlopes.data[j][i]))
						next_step += ( MaxFactor * DeltaWeights.data[j][i]);
							else        //  Use quadratic estimate .
						next_step += ((Slopes.data[j][i]/(PrevSlopes.data[j][i]
							-Slopes.data[j][i]))	* DeltaWeights.data[j][i]);
			} else       // Normal gradient descent, complete with momentum .
			{
				DidGradient++;
				next_step += ((SplitEpsilon ?
				( (Epsilon * Slopes.data[j][i]) / Nconnections[j] ) :
				( Epsilon * Slopes.data[j][i]))
				+ (Momentum * DeltaWeights.data[j][i]));
			}
//
// Set delta weight, and adjust the weight itself. .
//
		DeltaWeights.data[j][i]=next_step;
		Weights.data[j][i] += next_step;
	}
}


//
//  Perform forward and back propagation once for each pattern in the
//  training set, collecting deltas.  Then burn in the weights.
//

void NeuralNet::TrainOnce()
{
	int i;

	ClearSlopes();

	for (i=0; i<NTrainingPatterns; i++) {
		Forward ( TrainingInputs.data[i] );
		Backward ( TrainingOutputs.data[i] );
	}

	UpdateWeights();
	Epoch++;
}


//
//  Train the network for the specified number of epochs, printing out
//  performance stats every 10 epochs.
//

void NeuralNet::Train ( int times )
{
	int i, report;

	report = 10;

	for (i=0; i<times; i++) {
		if ( Epoch % report == 0 ) {    // Time to report status .
			DidGradient = 0;
			KeepScore = 1;
			TotalError = 0.0;
			TotalErrorBits = 0;
			TrainOnce();
			printf ("Epoch %d:  %d Bits Wrong, Total Error = %f, DidGradient = %d.\n",
				(Epoch - 1), TotalErrorBits, TotalError, DidGradient);
			KeepScore = 0;
		} else
		TrainOnce();
	}
}


void NeuralNet::Test()
{
	int 	i;
	char	str[20];

	tinputs[0] = 1.0;          // Initial nonzero value .

	while (1) {
		printf ("Enter the %d input values [enter 'q' to stop]: ",
			Ninputs);
		for (i=0; i<Ninputs; i++) {
			scanf ("%s", str);
			if (str[0]=='q') return;
			else tinputs[i]=atof(str);
		}

		Forward(tinputs);
		printf ("Output is: ");
		for (i=0; i<Noutputs; i++)
			printf ("%f  ", Outputs[FirstOutput+i]);
		printf ("\n");
	}
}
