;;;-*- Mode: Lisp; Syntax: COMMON-LISP; Package: TL; -*-
;;; ****************************************************************
;;; Top-Level Listener and Command Interpreter *********************
;;; ****************************************************************
;;;
;;; This file implements utilities for "top level" Lisp interaction.
;;; It has been tested in MCL 2.0, Franz 3.1/ 4.1 (SGI, NeXT), 
;;; AKCL 615 (NeXT), Franz CL/PC and CLISP.
;;;
;;; tl:top-level
;;; A generic, portable lisp listener and command interpreter that
;;; supports lisp evaluation and non lisp command dispatching from
;;; the same read-eval-print loop. It accepts input from any string
;;; input stream, so forms can be processed from files and strings
;;; in addition to normal interactive input from a terminal.  You
;;; create your own interpreters or applications by defining new
;;; command tables and passing them to Top-Level. A sample app 
;;; called tl:tl is defined at the end of this file which 
;;; implements a few simple commands for general Lisp interaction,
;;; such as Package, Cload, Args, etc.  See the main comments for
;;; tl:top-level for more information.
;;;
;;; ****************************************************************
;;; tl:eval-from-file
;;; Calls top-level on the contents of a specified file, which may
;;; contain both lisp and command expressions.
;;;
;;; tl:eval-from-string
;;; Just like eval-from-file except input is read from a string.
;;;
;;; tl:ask-user
;;; A general purpose user-query facility with all sorts of bells
;;; and whistles, and will work in conjunction with top-level to
;;; process non-interactive input from files or strings.
;;;
;;; tl:edit-object
;;; Supports interactive clos instance editing by defining a command
;;; table appropriate for a supplied object's slots and invoking
;;; Top-Level to implement the slot editing command loop.
;;;
;;; TL is part of the Common Music (CM) music composition language,
;;; but is generic enough and portable enough to be of use in other
;;; applications.  CM itself is available by anonymous ftp from
;;; ccrma-ftp.stanford.edu [36.49.0.93] and ftp.zkm.de
;;; [192.101.28.17] in the files pub/cm.tar.Z.  For further
;;; information, send email to Rick Taube, <hkt@zkm.de> or
;;; <hkt@ccrma.stanford.edu>.
;;;
;;; Change history:
;;;
;;; 23-Jun-93:  
;;;   1. top-level and ask-user now support input and output to
;;;      files or streams, and optionally echo the input read.
;;;   2. new functions tl:eval-from-file and tl:eval-from-string 
;;;      process top-level input from files and strings.
;;;   3. new variable *command-delimiting* decides if multiple
;;;      line command input is allowed using curly brackets {} as
;;;      command delimiters.
;;;   4. new variable *prompt-newline* decides if prompting is
;;;      prefaced with a #\newline in addition to a #\freshline.
;;;   5. new :output and :echo-input keywords for top-level and
;;;      ask-user
;;;   6. output functions now check for null output stream
;;;   7. ask-user now includes a fresh-line directive in its prompt
;;;   8. ported to Franz Cl/PC
;;;        

#-(or cltl2 lispworks loop)
(require ':loop)

#+(or cltl2 lispworks clisp)
(defpackage :tl
  (:use #-clisp :common-lisp #+clisp :lisp :tl))

(in-package :tl)

(defparameter *top-level-prompt* "> ")
(defparameter *top-level-running* nil)
(defparameter *top-level-commands* nil)
#+mcl (defvar *top-level-stack* nil)
(defparameter *prompt-newline* t)

;;;
;;; Fixup error handling differences
;;;

#+(and excl (not cltl2)) 
(import '(excl::with-simple-restart excl::find-restart excl::invoke-restart))	    

#+kcl
(progn
  (defun find-restart (x) (and *top-level-running* x))
  (defun invoke-restart (x) (throw x nil))
  (defmacro with-simple-restart (options &body body)
    `(catch ',(car options) ,@ body)))

#+aclpc
(progn
  (defun find-restart (x) (and *top-level-running* x))
  (defun invoke-restart (x) (allegro:unwind-stack nil 'throw x))
  (defmacro with-simple-restart (options &body body)
    (let ((x (gensym)) (y (gensym)) (z (gensym)))
     `(multiple-value-bind (,x ,y ,z) (allegro:trap-exits (progn ,@body))
        (when (eq ,y 'throw)
          (unless (eq ,z ',(car options))
            (invoke-restart ,z))))))
  (defun tl-error-handler (continue-format error-format args)
    ;; this handler is installed by top-level.
    (case (allegro:ask-error-action continue-format error-format args)
      (:debug (allegro:debugger nil))
      ((:abort :continue)))
    (allegro:unwind-stack nil 'error ':top-level-return)))

#+mcl
(eval-when (load eval)
  ;; make Command-. return to our Top-Level. 
  ;; there is probably a right way to do this...
  (ccl:advise ccl::interactive-abort
              (if (and (= ccl::*break-level* 1)
                       (find-restart ':top-level-return)) 
                (invoke-restart (find-restart ':top-level-return)))
              :when :before :name :abort-under-top-level))

;;;
;;; tl returns from error breaks. It can also be invoked via a
;;; 'break command' in ports that implement them:
;;; Franz       :TL
;;; KCL         :TL  (but doesnt appear in akcl's help listing)
;;; MCL         Command-.
;;; CLISP       TL
;;;

(defun tl ()
  (let ((tl (find-restart ':top-level-return)))
    (if tl (invoke-restart tl)
      (format t "Can't return: Top-level not currently running!"))))

#+excl (top-level:alias ("tl" 1) () (tl))
#+kcl (setf (get :tl 'system::break-command) 'tl)
#+clisp  ;; i have no shame.
(if (constantp 'sys::commands1)
    (locally (declare (special sys::commands1))
      (unless (string-equal (caar (last sys::commands1)) "TL")
        (let ((new (append sys::commands1 (list (cons "TL" #'tl)))))
          (EVAL-WHEN (COMPILE)
            (SYSTEM::C-PROCLAIM-CONSTANT 'SYS::COMMANDS1 ()))
          (SYSTEM::%PROCLAIM-CONSTANT 'SYS::COMMANDS1 new))))
  (unless (fboundp '.oldcommands1.)
    (setf (symbol-function '.oldcommands1.)
       (symbol-function 'sys::commands1))
    (setf (symbol-function 'sys::commands1) 
       #'(lambda ()
           (append (funcall (symbol-function '.oldcommands1.))
                   (list (cons "TL" #'tl)))))))
;;;
;;; exports and imports
;;;

(export '(top-level tl defcommand find-command top-level-quit top-level-help
          *top-level-prompt* *top-level-commands* *command-delimiting*
          *prompt-newline* show-documentation next-whitespace-start
          next-token-start string-forms string-tokens nullstringp trim-line
          ask-user comment eval-from-string eval-from-file))

#+(or clos pcl mcl aclpc)
(eval-when (compile load eval)
  (let ((package #+pcl :PCL 
                 #+(or (and excl cltl2) lispworks (and clisp (not pcl))) :CLOS
                 #+mcl :CCL
                 #+aclpc :COMMON-LISP)
        (names '("DEFGENERIC" "DEFMETHOD" "PRINT-OBJECT" "STANDARD-OBJECT"
                 #-(and clisp (not pcl)) "SLOT-DEFINITION-NAME" 
                 #-mcl "CLASS-SLOTS" 
                 "CLASS-OF" "SLOT-BOUNDP" "SLOT-VALUE" "SLOT-MAKUNBOUND")))
    (import (mapcar #'(lambda (s) (find-symbol s package)) names))
    #+mcl (defun class-slots (class) (ccl:class-instance-slots class))
    #+(and clisp (not pcl))
    (defun slot-definition-name (slot) (clos::slotdef-name slot))
    (export '(edit-object))))

;;;
;;; Defcommand installs a command into a command table given a command name,
;;; command function (symbol or function object), optional short help string,
;;; optional command table and help file, ie:
;;; (defcommand "HI" #'(lambda (s) (format t "hi: ~a~%" s) "Say hello.")
;;;

(defstruct (command (:type list)) name function help help-file)

(defun cmd-name (cmd)
  (let ((name (car cmd)))
    (format nil "~A" name)))

(defmacro defcommand (name function &optional (help "No help avaiable.")
					      (table '*top-level-commands*)
                                    &key help-file)
  (let ((old (gensym))
 	(tbl (gensym)))
   `(eval-when (load eval)
      (let ((,tbl ,table))
        (let ((,old (find ,name ,tbl :test #'string-equal :key #'car)))
          (when ,old 
            (setf ,tbl (remove ,old ,tbl :test #'eq))))
        (push (make-command :name ,name 
                            :function ,(if (symbolp function)
                                           `(quote ,function)
                                         function)
                            :help ,help 
                            :help-file ,(if (eq help-file t) name help-file))
              ,tbl)
        (setf ,table (sort ,tbl #'string-lessp :key #'car))
        t))))

;;;
;;; fUn WiTh StRiNgS
;;;

(defun nullstringp (string)
  (string= string ""))

(defun trim-line (stream &optional (bag '(#\space #\tab)))
  (let ((line (read-line stream nil ':eof)))
    #+aclpc
    (when (and (eq line ':eof)
               (eq (allegro:stream-device stream) 'top::toploop-pane))
      (setf line ""))
    (if (eq line ':eof) 
        ':eof
      (string-trim bag line))))

(defun string-forms (string &optional (start 0) end)
  (let (form)
    (loop do (multiple-value-setq (form start)
               (read-from-string  string nil ':string-forms-eof 
                                  :start start :end end))
          until (eq form ':string-forms-eof) 
          collect form)))

(defun string-tokens (string &optional (delimiters '(#\Space)))
  (loop with len = (length string)
        for beg = 0 then (1+ end) until (> beg len)
        for end = (or (position-if #'(lambda (char) (member char delimiters))
                                   string :start beg)
                      len)
        unless (= beg end)
        collect (subseq string beg end)))

(defun next-whitespace-start (string &key (start 0) end from-end
                                          (whitespace '(#\Space #\Tab)))
  (position-if #'(lambda (x) (find x whitespace :test #'char=))
               string :start start :end end :from-end from-end))

(defun next-token-start (string &key (start 0) end from-end
                                     (whitespace '(#\Space #\Tab)))
  (position-if-not #'(lambda (x) (find x whitespace :test #'char=))
                   string :start start :end end :from-end from-end))

(defmacro format? (stream &rest args)
  (let ((s (gensym)))
    `(let ((,s ,stream))
       (when (streamp ,s) (format ,s ,@args)))))

;;;
;;; Top-Level is a read/eval/print and command dispatch loop. Standard 
;;; dispatching macro forms, such as ( # ' and so on are read via READ, 
;;; and are taken as forms to evaluate. A line of input that does not start
;;; with a dispatching form is READ-LINEed and processed as a (possible) 
;;; command.  Since it is not possible to distinguish between
;;; a line of input that is a mistyped command name and a line of input that
;;; is a symbol we want to evaluate, we take Symbolics' convention of
;;; an initial comma in a line of input as signifiying explicit evaluation:
;;;
;;; > ,foo
;;; The symbol FOO has no value.
;;; > foo
;;; "FOO" is not a command.
;;;
;;; A single command may span more than one line of input if the global
;;; varible *command-delimiting* is true (the default) and the multi line
;;; command input is delimited by {}. If command delimiting is turned off, a
;;; command must occupy a single line of input.
;;;

(defvar *echo-input* nil)               ; echo input to output stream

(defun top-level (&key (stream *standard-input*)
                       (commands *top-level-commands*)
                       (output *standard-output*)
                       (prompt *top-level-prompt*)
                       (banner nil bp)
                       (clear-input t)
                       (echo-input *echo-input*))
  (let ((*top-level-running* t)
        (*standard-input* stream)
        (*standard-output* output)
        (*top-level-commands* commands) 
        (*top-level-prompt* prompt)
        (*echo-input* echo-input)
	#+mcl (fresh-prompt t)   ; needed for 'odoc' apple event
        #+aclpc (allegro:*error-hook* #'tl-error-handler) ; use our handler
        input type)
    (when clear-input (clear-input stream))
    (if banner 
        (format? output banner)
      (unless bp (print-default-banner)))
    #+mcl                               ; update mcl's package display
    (progn 
      (ccl:set-window-package ccl:*top-listener* (package-name *package*))
      (ccl:mini-buffer-update ccl:*top-listener*))
    (with-simple-restart (:top-level-exit "Exit Top-Level.")
      (catch :top-level-quit
        (loop do
          (with-simple-restart (:top-level-return "Return to Top-Level.")
            #-mcl (print-prompt output prompt)
            #+mcl                       ; yield cycles during idle
            (progn
              (if fresh-prompt          ; be system 7 'odoc' friendly
                  (print-prompt output prompt)
                (setf fresh-prompt t))
              (setf ccl:*idle* t)
              (ccl:set-mini-buffer ccl:*top-listener* "Idle")
              (setf input nil)
              (if (interactive-stream-p stream)       ; files are different!
                  (loop until (or (listen stream) 
                                  (setf input (ccl:get-next-queued-form)))
                        do (ccl:event-dispatch t))
                (progn 
                  (setf input (ccl:get-next-queued-form))
                  (ccl:event-dispatch t)))
              (setf ccl:*idle* nil)
              (ccl:set-mini-buffer ccl:*top-listener* "Busy")
              (if input 
                  (flet ((tl-eval (inp) (eval inp) (setf type ':empty))) 
                    (if (listp input)
                      ;; files are opened by a call to #'ccl::ed
                        (if (eq (first input) 'ccl::ed)
                            (progn 
                              (setf fresh-prompt nil)
                              (tl-eval input))
                          (progn
                            (format? output "~&")
                              ;; :read-input signals top-level command input
                            (if (eq (first input) :read-input)
                                (multiple-value-setq (input type)
                                  (funcall #'read-input (second input)
                                           commands))
                              (tl-eval input))))
                      (tl-eval input)))
                (progn
                  (multiple-value-setq (input type)
                    (read-input stream commands))
                  (unless echo-input 
                    (format? output "~&")))))   ; may complete sans newline
            #-mcl
            (multiple-value-setq (input type) 
              (read-input stream commands))
            #+aclpc (fresh-line output)
            (ecase type
              (:form       
               ;; eval lisp forms, but check for simple typos.
               (loop while input
                     for f = (pop input)
                     do
                     (if (and (symbolp f) (not (boundp f)))
                       (format? output "The symbol ~S has no value.~%" f)
                       (progn 
                         #-(or cltl2 lispworks)
			 (when (and *echo-input* 
				    (streamp output))
			       (pprint f output)) 
                         #+(or cltl2 lispworks)
			 (when *echo-input*
			       (format? output "~@[~(~:w~)~]" f))
                       (let ((values (multiple-value-list (eval f))))
                         (setf +++ ++ ++ + + f)
                         (setf *** ** ** * * (car values))
                         #-(or cltl2 lispworks clisp)
			 (progn 
			   (dolist (v values) 
				   (when (streamp output) 
					 (pprint v output)))
			   (format? output "~&"))
                         #+(or cltl2 lispworks clisp)
			 (format? output "~{~&~:w~}" values))))
                     ;; print top level prompt between each evaled form
                     (when input 
                       (print-prompt output prompt))))
              (:command
               (when *echo-input* 
                 (format? output "~A ~A~%" 
                         (first (first input)) (second input)))
               (funcall (second (first input)) (second input)))
              (:error 
               (format? output input))     
              (:empty nil)
              (:eof
               (throw :top-level-quit nil)))))))
    (values)))

;;;
;;; These basic top-level Help and Quit command functions are exported.
;;;

(defun top-level-help (str)
  (declare  (ignore str))
  (fresh-line t)
  (loop for c in *top-level-commands*
        for n = (cmd-name c)
        collect n into names
        maximize (length n) into tab
        finally 
        (loop for nam in names
              for cmd in *top-level-commands*
              do
              (format? *standard-output* "~A~VT~A~%" 
                       nam (+ 1 tab 8) (third cmd)))))

(defun top-level-quit (str)
  (declare (ignore str))
  #-aclpc (throw :top-level-quit t)
  #+aclpc (invoke-restart ':top-level-quit)
  )

;;;
;;; two wrappers for non-interactive input processing
;;;

(defun eval-from-file (file &rest args)
  (if (probe-file file)
      (with-open-file (f file)
        (apply #'top-level :stream f :banner nil args))
    (format? *standard-output* "~S is not a file." file)))

(defun eval-from-string (string &rest args)
  (with-input-from-string (str string)
    (apply #'top-level :stream str :clear-input nil
          :banner nil args)))

;;;
;;; the reader and support routines
;;;

(defvar *command-delimiting* t)    ; if t {} delimit multi-line command input 

(defun read-input (stream commands)
  (let ((? (peek-char nil stream nil ':eof nil))
        #+aclpc (fix nil))

    ;; hack around aclpc returning :eof on empty line of input for terminal
    ;; streams and #\linefeeds in file streams.
    #+aclpc 
    (if (eq (allegro:stream-device stream) 'cg:text)
        (loop while (and (characterp ?) (char= ? #\linefeed))
              do (read-char stream)
                 (setf ? (peek-char nil stream nil ':eof nil)))
      (when (eq ? ':eof) (setf ? #\newline fix t)))
    (cond ((eq ? ':eof)                         ; reached end-of-file
           (values nil ':eof))
          ((char= ? #\newline)                  ; empty line of input
           #-aclpc (read-char stream)
           #+aclpc (unless fix (read-char stream))
           (values nil :empty))
          ((read-character?                     ; read lisp form
             (setf ? (peek-char t stream nil ':eof nil)))
           (when (char= ? #\,) 
             (read-char stream))                ; flush a comma 
           (let ((form (read stream nil ':eof nil)))
              #+(or mcl akcl clisp)
                (when (and (listen stream)  ; mcl may leave crlf in stream.
                           (char= (peek-char nil stream) #\newline))
                      (read-char stream))
              (values (list form) ':form)))  
          (t                                    ; possible command input
           (let (str len)
             (if (and *command-delimiting*      ; read multi line command input
                      (char= ? #\{))
                 (multiple-value-setq (str len) (read-curly-bracket stream))
               (progn (setf str (read-line stream nil ':eof nil))
                      (setf str (string-trim '(#\space #\tab) str))            
                      (setf len (length str))))
             (if (= len 0)                    ; just spaces
                 (values nil ':empty))
               (multiple-value-bind (cmd arg bad)
                        (find-command str commands 0 len)
                      (if cmd 
                          (values (list cmd (subseq str arg len)) ':command)
                        (if bad
                            (values (format nil 
                             "\"~:@(~A~)\" matches more than one command.~%" 
                                           (subseq str 0 arg))
                                  ':error)
                          (let ((const? (string-constantp str)))
                            (if const?
                                (values (list const?) ':form)
                              (values (format nil
                               "\"~:@(~A~)\" is not a command."
                                              (subseq str 0 arg))
                                      ':error)))))))))))

(defun read-character? (char)
  (if (char= char #\,)                  ; we handle comma ourselves.
      T
    #+excl (not (eq (get-macro-character char) #'excl::read-token))
    #-excl (get-macro-character char)))

(defun read-curly-bracket (stream)
  ;; parse multi line command input between {}
  (loop with nullchar = (code-char 0)
        with last = (read-char stream t)     ; the first {
        for  char = (read-char stream t)
        until (char= char #\})
        when (and (char= last #\{)
                  (member char '(#\space #\tab #\newline) :test #'char=))
        do (setf char nullchar)
        when (char= char #\newline)
        do (loop until (peek-char t stream nil ':eof nil)
                 finally (setf char #\space))
        when (char= char #\{)
        do (error "Bogus left bracket found.")
        unless (char= char nullchar)
        collect char into chars and do (setf last char)
        finally (when (peek-char #\newline stream nil nil nil)
                  (read-char stream nil))
                (return
                  (let* ((len (length chars))
                         (str (make-string len)))
                    (loop for i below len
                          do (setf (aref str i) (pop chars)))
                    (values str len)))))

(defun find-command (str commands &optional (start 0) length)
  (unless length (setf length (length str)))
  (when (char= (elt str start) #\:)        ; allow commands to start with colon
    (incf start))
  (let* ((end (or (next-whitespace-start str :start start :end length)
                 length))
         (arg (if (< end length)
                (next-token-start str :start end)
                length)) 
         pos match)
    (dolist (c commands)
      (setf pos (mismatch str (car c) :start1 start :end1 end 
                          :test #'char-equal))
      (if (not pos)
        (return-from find-command (values c arg nil))
        (when (= pos end)
          (push c match))))
    (if (= (length match) 1)
      (values (car match) arg nil)
      (values nil end match))))

(defun string-constantp (string) 
  ;; yuck!
  (let (value)
    (and (every #'(lambda (x)
                    (or (alphanumericp x)
                        (not (member x '(#\space #\( #\) #\tab #\, 
                                         #\' #\" #\; #\` #\return)
                                     :test #'char=))))
                string)
         (constantp (setf value (read-from-string string)))
         value)))

;;;
;;; show-documentation displays long documention help. Help text is kept
;;; in the file specified by pathname and printed to stream.  
;;;

(defun show-documentation (pathname &key (stream t) (indent 0))
  (declare (optimize (speed 3)(safety 0)))
  (with-open-file (f pathname)
    (let (line flag)
      (loop do (setf line (read-line f nil ':eof))
            until (eq line ':eof)
            do (dotimes (i indent) (write-char #\Space stream))
            (write-line (the string line) stream))
      flag))
  t)

(defun print-default-banner ()
  (let ((help (or (find-command "?" *top-level-commands*)
                  (find-command "HELP" *top-level-commands*))))
    (when help
      (format? *standard-output* "~%Type ~A for help.~%" 
               (cmd-name help)))))

;;;
;;; Prompts can be strings, functions, or structures with their own
;;; print function. This fn would show the current package as the prompt:
;;; #'(lambda (s) (format? s "~:[~&~;~%~]~A" 
;;;                        *prompt-newline* (package-name *package*))))
;;;

(defun print-prompt (stream prompt)
  (if (stringp prompt)
      (format? stream "~:[~&~;~%~]~A" *prompt-newline* prompt)
    (if (functionp prompt)
        (funcall prompt stream)
      (format? stream "~:[~&~;~%~]~S" *prompt-newline* prompt)))
  #+mcl
  (let ((l (ccl::current-listener)))
    (ccl::set-mark (ccl::listener-prompt-mark l) 
                   (ccl::listener-eof-mark l))))

;;;
;;; Ask-User is a general function that supports user querying.
;;; Its got a trillion keyword arguments and I expect that most people
;;; will do what I do -- create wrapper functions that implement specific 
;;; query operations based on ask-user, ie. ask-file, ask-number, 
;;; ask-type-or-token, etc. Look in stella/ask.lisp for examples.
;;;

(defun ask-user (&key (stream *standard-input*) (prompt "> ")
                      dont-read (input "") predicate predicate-is-filter
                      default (default-value nil dvp) default-prompt
                      (default-input-match "") check-default
                      (null-ok nil nullokp) junk-allowed
                      (error nil errorp) (help nil helpp)
                      abort-ok (abort-token '^) (abort-value ':aborted)
                      (help-token '?) error-return
                      (output *standard-output*))
  ;; coerce user supplied t to actual input/output streams.
  (when (eq stream t) (setf stream *standard-input*)) 
  (when (eq output t) (setf output *standard-output*)) 
  (when (null input) (setf input ""))
  (when default
    (unless dvp (setf default-value default))
    (unless default-prompt 
      (setf default-prompt (format nil "(<cr>=~A)" default-value))))
  (unless nullokp (setf null-ok default))
  (unless (or error errorp) (setf error "~s is not valid input.~%"))
  (unless (or help helpp) 
    (setf help (format nil "No help available.~@[ Type ~A to abort~].~%" 
                       (and abort-ok abort-token))))
  (when help-token (setf help-token (string help-token)))
  (when abort-token (setf abort-token (string abort-token)))
  (let ((more t) (first t)(pos 0) (len 0) value check)
    (loop while more
          do
	  (setf value nil)
	  (when (nullstringp input)
            (when (or (not first) (eq null-ok :ask))	    
	      (format? output "~&~a~@[~a ~]" prompt default-prompt)
	      (setf input (trim-line stream))
              ;; this hack should be removed when clisp fixes its io bug
              #+clisp (clear-output output)
              (when *echo-input* (format output "~A~&" input))))
          (when (eq input ':eof)
            (return-from ask-user (or (and abort-ok abort-value) input)))
	  (setf len (length input)) 
	  (if (or (nullstringp input)
	          (string-equal input default-input-match))
            (if default
              (setf value default-value more nil 
                    check check-default pos len)
              (if null-ok (setf value nil more nil) (setf check nil)))
            (setf check t))
	  (when check
	    (unless value
	      (if dont-read 
                (setf value input pos len)
                (multiple-value-setq (value pos) (read-from-string input))))
            (if (and abort-ok (string-equal input abort-token))
              (setf more nil value abort-value input ""  pos 0 len 0)
	      (if (and help (string-equal input help-token))
                (progn 
                  (if (stringp help)
                    (format? output help)
                    (funcall help stream))
                  (setf input ""))
		(if predicate
                  (multiple-value-bind (? ??) (funcall predicate value)
                    (cond ((or ? ??)
                           (setf more nil)
                           (when predicate-is-filter (setf value ?)))
                          ;(error-return (setf more nil value ':error pos len))
                          (t 
                           (when error 
                             (if (stringp error)
                               (format output error value)
                               (funcall error value stream)))
                           (if error-return
                             (setf more nil value ':error pos len)
                             (progn
                               (setf input "" pos 0 len 0)
                               (when (eq value default) 
                                 (setf default nil null-ok nil more t
                                       default-prompt nil 
                                       default-input-match "")))))))
	          (setf more nil))))
            (unless (or (= pos len) junk-allowed)
              (format? output "~a contains extra junk.~%" input)
              (setf input "" more t)))
	  (setf first nil))
    (values value input pos)))

;;;
;;; Edit-Object supports interactive slot editing using a top-level command 
;;; shell whose command table is constructed dynamically from slot
;;; definitions of the instance to edit. By default, commands will be made 
;;; for all the slots of the object. To specify a subset, pass either the 
;;; list of slot names to allow using the :slots keyword argument, or 
;;; else the list of slot names to block (ignore) by using the :ignore-slots
;;; argument. Any slot that is editable may have its command behavior 
;;; specified or customized by supplying a specification in the optional 
;;; slot  command specification list passed in the :slot-commands argument.
;;; A slot that does not have an associated entry in this list will receive
;;; one automatically by the system.  Each entry in the :slot-commands list
;;; is of the form: (slot &key command printer parser help command-name)
;;; where the name of the slot is required and the the rest of the
;;; keyword arguments are optional. If :parser is supplied, it becomes
;;; the edit input value parsing function for the system supplied command
;;; function.  The :parser function takes a single value, the user's
;;; input string, and must return either nil (failure) or the value it 
;;; parsed from the input string. If :printer is supplied, the system
;;; command function will use this whenever it prints the value of the slot.
;;; If :command is specified, it overrides the system command building
;;; altogether and everything (printing, parsing, setting slot values, etc.)
;;; must then be handled by this function. If a :command function is supplied,
;;; it is passed three values: the object, the slot being set, and
;;; the edit input string.  The :command-name argument may be used to specify
;;; a command name different than the actual name of the slot.
;;; In addition to :slot-commands, the :help-command, :quit-command and
;;; :show-command arguments provide command specs for basic functionality,
;;; and by default appear in the command table as: Help, Quit and Show.
;;; The Help and Quit commands are obvious and behave exactly as their
;;; counterparts in the basic tl interpreter; Show prints a specified
;;; set of slots (or all the slots if no argument is given) and their
;;; and values in a formatted table.
;;;

#+(or pcl clos mcl aclpc)
(progn

(defgeneric edit-object (object &key stream prompt))

(defmethod edit-object ((object standard-object)
                        &key (stream *standard-input*) (prompt "Edit: ") 
                             (help-command '?) (quit-command 'quit) 
                             (show-command 'show)
                             (object-printer #'print-object)
                             (slots t) (ignore-slots nil) 
                             slot-commands (command-sorter #'string-lessp)
                             (output *standard-output*))
  (let ((slot-names (mapcar #'slot-definition-name 
                      (class-slots (class-of object))))
        (commands '()))
    (when (and (functionp slots)
               (or (fboundp slots) (not (symbolp slots))))
      (setf slots (funcall slots object)))
    (unless (or (eq slots t) (eq slots ':all))
      (setf slot-names
        (loop for name in slot-names
              when (find name slots) collect name)))
    (when ignore-slots
      (when (and (functionp ignore-slots)
                 (or (not (symbolp ignore-slots)) (fboundp ignore-slots)))
        (setf ignore-slots (funcall ignore-slots object)))
      (if (or (eq ignore-slots t) (eq ignore-slots ':all))
          (setf slot-names nil)
        (setf slot-names 
          (loop for name in slot-names
                unless (find name ignore-slots) collect name))))

    ;; build the command table out from the union of slot-names, commands
    ;; passed in, and help, quit, and show. an entry in slot-commands
    ;; overrides any is occurance in slot-names.
    (flet ((make-edit-cmd (slot &key (parser #'parse-input-value)
                                (printer #'print-slot-and-value)
                                (help "Set value of named slot.")
                                command command-name)
             (list (string (or command-name slot))
                   (make-edit-fn object slot command parser printer stream 
                                 output)
                   help)))
      (setf commands
        (loop for slot in slot-names
              for entry = (or (find slot slot-commands :key #'car) slot)
              if (consp entry) 
              collect (apply #'make-edit-cmd entry)
              else collect (make-edit-cmd entry)))
      ;; add quit help and show commands
      (push (if (atom quit-command)
                (list (string quit-command) #'top-level-quit "Quit Edit.")
              (apply #'make-edit-cmd quit-command))
            commands)
      (push (if (atom help-command)
                (list (string help-command) #'top-level-help 
                      "Show this help.")
              (apply #'make-edit-cmd help-command))
            commands)
      (push (if (atom show-command)
                (list (string show-command)
                      #'(lambda (s)
                          (let ((slots (string-forms s)))
                            (if slots
                                (dolist (s slots)
                                  (unless (find s slot-names)
                                    (format
                                      t "~&~A is not a legal slot name.~%" s)
                                    (setf slots nil)))                               
                              (setf slots slot-names))
                            (when slots
                              (print-slots object :slots slots))))
                      "Show slots and values.")
              (apply #'make-edit-cmd show-command))
            commands))

    ;; sort the command table
    (setf commands (sort commands command-sorter :key #'car))

    ;; call the edit loop
    (when output
      (format output "~&Editing ")
      (funcall object-printer object output))
    (top-level :prompt prompt :stream stream :commands commands 
               :output output)))

(defun parse-input-value (string)
  ;; default command processer selectively evaluates user input.
  ;; constants and bound vars are evaled, as are lists with fboundp
  ;; first element. anything else is immediate data.
  (let ((form (read-from-string string)))
    (if (constantp form) (values (eval form) t)
      (if (symbolp form)
          (if (boundp form) (values (eval form) t) (values form t))
        (if (consp form)
            (if (and (symbolp (car form)) (fboundp (car form)))
                (values (eval form) t)
               (values form t))
           (values form t))))))

(defun make-edit-fn (instance slot-name command parser printer stream
                     output)
     ;; create a closure to implement the command for a specified slot.
     ;; if user specified a command function, pass the instance slot
     ;; string input to it. otherwise build the basic edit command
     (if command
       #'(lambda (input) (funcall command instance slot-name input))       
       #'(lambda (input)
           (let (value)
             (when (string= input "")
               (funcall printer instance slot-name output))
             (setf value
                   (ask-user :prompt "new value: "
                             :stream stream :dont-read t :predicate parser
                             :input input :output output :default 'unchanged
                             :default-prompt "(<cr>=Unchanged) "
                             :predicate-is-filter t :null-ok ':ask 
                             :abort-ok t))
             (unless (and (symbolp value)
                          (or (eq value ':aborted)
                              (string-equal (symbol-name value) "UNCHANGED")))
               (if (and (symbolp value)
                        (string-equal (symbol-name value) "UNSET"))
                 (slot-makunbound instance slot-name)
                 (setf (slot-value instance slot-name) value))
               (funcall printer instance slot-name output))))))

(defun print-slot-and-value (instance slot-name output)
     (let ((bound? (slot-boundp instance slot-name))
           (value ':unset))
       (format? output "~&The value of ~A is " slot-name)
       (when bound? (setf value (slot-value instance slot-name)))
       (format? output "~:[~(~A~)~;~S~]~%" bound? value)))
 
(defun print-slots (instance &key (stream *standard-output*) slots)
  (loop for slot in (or slots (mapcar #'slot-definition-name
                                      (class-slots (class-of instance))))
        maximize (length (string slot)) into lengths
        collect slot into show
        finally 
        (loop for slot in show 
              do
          (if (slot-boundp instance slot)
              (format? stream "~&~A~VT~S~%" slot (+ lengths 9)
                       (slot-value instance slot))
            (format? stream "~&~A~VT-unset-~%" slot (+ lengths 9))))))

) ;#+(or pcl clos mcl)

;;;
;;; Andexample of defining and using a command set for top-level.
;;; The varible *lisp-commands* holds a small command set for
;;; common user operations like compile/load, macroexpand, etc. 
;;; To start the listener, type:
;;; (tl:top-level :commands tl::*lisp-commands* :prompt "Lisp> ")
;;; Use (tl) to resume from breaks; Quit leaves the listener.
;;;

(defparameter *lisp-commands* nil "Command table for TL interpreter.")
#+(or excl mcl pcl)
(defcommand "ARGS" args "Print function arguments." *lisp-commands*)
(defcommand "CLOAD" cload "Compile/Load a file." *lisp-commands*)
(defcommand "CF" cf "Compile a file." *lisp-commands*)
(defcommand "CC" cc "Compile code." *lisp-commands*)
(defcommand "LD" ld "Load a file." *lisp-commands*)
(defcommand "PACKAGE" pkg "Switch lisp packages." *lisp-commands*)
(defcommand "?" top-level-help "Show this help." *lisp-commands*)
(defcommand "QUIT" top-level-quit "Quit Top-Level." *lisp-commands*)
(defcommand "MACROEXPAND" ma "Macro expand form." *lisp-commands*)



(defun pkg (&optional pkg)
  (if (or (null pkg) (nullstringp pkg))
      (format? *standard-output*
               "The current package is ~A.~%" (package-name *package*))
    (let* ((name (read-from-string pkg))
           (pkg? (find-package name)))
      (if pkg?
          (progn
            (setf *package* pkg?)
            ;; inform top listener window in mcl.
            #+mcl (progn
                    (ccl:set-window-package ccl:*top-listener*
                                            (package-name pkg?))
                    (ccl:mini-buffer-update ccl:*top-listener*))
            (format? *standard-output*
                    "Package set to ~A.~%" (package-name *package*)))
         (format? *standard-output* "No package named ~A.~%" pkg)))))

#+(or excl mcl pcl)
(defun args (fn)
  (let ((sym (if (stringp fn) (read-from-string fn) fn)))
    (if (and (symbolp sym) (fboundp sym))
        (let* ((argl #+mcl  (ccl:arglist sym)
                     #+excl (excl:arglist sym)
                     #+(and pcl (not excl)) (pcl::function-pretty-arglist sym))
               (args (if argl (princ-to-string argl) "()")))
          (format t "~S: ~A~%" sym args)) 
      (format? *standard-output* "~S is not a lisp function.~%" sym))))

(defun cload (file &key (source-type "lisp")
                        (binary-type #+kcl "o" #-kcl "fasl"))
  ;; mcl logical pathnames have bug
  #+mcl (setf file (translate-logical-pathname (namestring file)))
  ;; kcl pathnames have pathname caching bug
  #+kcl (setf file (namestring file))
  (let ((source (make-pathname :defaults file 
                               :type (or (pathname-type file)
			                 source-type)))
        (binary (make-pathname :defaults file :type binary-type))
	(compiled? nil)
	(loaded? nil))
    ;; apparently kcl also needs its working directory set or it can't
    ;; find its .o file!
  (if (probe-file source)
      (progn
        (setf source (truename source))
        (when (or (not (probe-file binary))
                  (< (file-write-date binary)
                     (file-write-date source)))
          (compile-file source :output-file binary)
	  (setf compiled? source)))
    (warn "Source file ~A not found." (namestring source)))
  (if (probe-file binary)
      (progn (setf binary (truename binary))
             (load binary) ; :verbose T
	     (setf loaded? binary))
      (warn "Binary file ~A not found." (namestring binary)))
  (values loaded? compiled?)))

(defun cf (file)
  #+mcl (setf file (translate-logical-pathname (namestring file)))
  (unless (pathname-type file)
    (let ((source (make-pathname :defaults file :type "lisp")))
      (when (probe-file source)
        (setf file source))))
  (if (probe-file file)
      (progn 
        (compile-file file))
    (progn (warn "File ~A not found." (namestring file)) nil)))

(defun cc (&optional spec)
  (if (or (null spec) (nullstringp spec))
      (setf spec *)
    (when (stringp spec)
      (setf spec (read-from-string spec))))
  (when (and (symbolp spec) (fboundp spec) )
     (unless (compiled-function-p (symbol-function spec))
       (format? *standard-output* ";;; Compiling ~A~&" spec)
       (compile spec))))

(defun ld (file)
  #+mcl (setf file (translate-logical-pathname (namestring file)))
  (unless (pathname-type file)
    (let ((source (make-pathname :defaults file :type "lisp"))
          (binary (make-pathname :defaults file :type #+kcl "o" 
	                                              #-kcl "fasl")))
      (if (probe-file binary)
        (setf file binary)
        (if (probe-file source)
          (setf file source)))
      (when (and (probe-file binary)
                 (probe-file source)
                 (< (file-write-date binary)
                    (file-write-date source)))
        (warn "Loading binary ~S but source is newer."
              (namestring (truename binary))))))
  (if (probe-file file)
      (progn 
             (load file) ; :verbose T
	     file)
    (progn (warn "File ~A not found." (namestring file))
           nil)))

(defun ma (input)
  (let ((form (read-from-string input)))
    (if (not form)
        nil
      (progn (pprint (setf form (macroexpand form)))
             (terpri)
             form))))

(pushnew ':tl *features*)

